/*
   Copyright (C) 2006-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "rendererbase.h"

#include "i18n.h"
#include "rendereffect.h"
#include "project.h"

#include <QPainter>
#include <QPaintDevice>
#include <QMatrix>
#include <QFontMetrics>

#include <QMessageBox>

#include <math.h>
#include <iostream>


namespace KoverArtist
{


RendererBase::RendererBase(Project* aProject)
:mProject(aProject)
,scaleFactorX(1.0)
,scaleFactorY(1.0)
,mFrontWidth(0)
,mBackWidth(0)
,mWidth(0)
,mHeight(0)
,mPixFront()
,mPixBack()
,mPixEff()
,mPaint()
,mPaintEff()
,mImgBG()
,mBBEff(0, 0, 0, 0)
,mTextList()
,mREff(0)
{
}


RendererBase::~RendererBase()
{
   deleteQueuedTexts();
   delete mREff;
}


void RendererBase::beginEffect(const Effect& aEff)
{
   mPixEff = QPixmap(mWidth, mHeight);
   mPixEff.fill(QColor(0, 0, 0, 0));

   mPaintEff.begin(&mPixEff);
   mPaintEff.setPen(Qt::white);
   //mPaintEff.setWorldMatrix(mPaint.worldMatrix());

   mREff = RenderEffect::create(this, aEff, mWidth, mHeight);
   if (mREff) mREff->setup(mPaintEff, mPixEff);

   mBBEff.setRect(0, 0, 0, 0);
}


void RendererBase::endEffect()
{
   mPaintEff.end();

   if (mREff && !mBBEff.isNull())
   {
      mREff->setBoundingBox(mBBEff);
      mREff->render(mPixEff);
      const QRect& bb = mREff->boundingBox();
      int x=bb.x(), y=bb.y();
      mImgBG.setImage(x, y, *mREff, x, y, bb.width(), bb.height());
      delete mREff;
      mREff = 0;
   }
}


void RendererBase::beginSurface(QPaintDevice* aDev, const QMatrix& aMatrix,
                                int aWidth, int aHeight)
{
   mWidth = aWidth;
   mHeight = aHeight;
   mImgBG = RenderImage(0, 0);

   if (mPaint.isActive())
   {
      mPaint.restore();
      mPaint.save();
   }
   else
   {
      mPaint.begin(aDev);
      mPaint.save();
   }
   mPaint.setWorldMatrix(aMatrix, false);
}


void RendererBase::endSurface()
{
   // Draw background
   if (!mImgBG.isNull())
      mPaint.drawImage(0, 0, mImgBG);

   // Draw queued texts
   renderQueuedTexts(mPaint);
}


void RendererBase::setSurfaceBackground(const QPixmap& aPix)
{
   mImgBG = aPix.toImage();
}


void RendererBase::finish()
{
   if (mPaint.isActive())
   {
      mPaint.restore();
      mPaint.end();
   }
}


void RendererBase::renderText(const QString& aText, QRect aRect, int aAlign,
                              const QFont& aFont, const QColor& aColor,
                              int aAngle)
{
   Qt::AlignmentFlag af = (Qt::AlignmentFlag) aAlign;

   if (aAngle)
   {
      QMatrix m;
      m.rotate(aAngle);
      mBBEff |= m.mapRect(aRect);
   }
   else mBBEff |= aRect;

   RenderText *rt = new RenderText(this, aFont, aColor, aRect, af, aText, aAngle);
   mTextList.append(rt);

   mPaintEff.save();
   mPaintEff.setFont(aFont);
   if (aAngle) mPaintEff.rotate(aAngle);
   mPaintEff.drawText(aRect, aAlign, aText);
   if (aAngle) mPaintEff.rotate(-aAngle);
   mPaintEff.restore();
}


void RendererBase::deleteQueuedTexts()
{
   while (!mTextList.isEmpty())
      delete mTextList.takeLast();
}


void RendererBase::renderQueuedTexts(QPainter& p)
{
   RenderText* rt;

   while (!mTextList.isEmpty())
   {
      rt = mTextList.takeFirst();
      rt->render(p);
      delete rt;
   }
}


int RendererBase::scaleX(int x) const
{
   return int(x*scaleFactorX);
}


int RendererBase::scaleY(int y) const
{
   return int(y*scaleFactorY);
}


QSize RendererBase::scale(const QSize& sz) const
{
   return QSize(int(sz.width()*scaleFactorX), int(sz.height()*scaleFactorY));
}


QPoint RendererBase::scale(const QPoint& p) const
{
   return QPoint(int(p.x()*scaleFactorX), int(p.y()*scaleFactorY));
}


QRect RendererBase::scale(const QRect& r) const
{
   return QRect(scaleX(r.x()), scaleY(r.y()), scaleX(r.width()), scaleY(r.height()));
}


QPixmap RendererBase::createPixmap(int aWidth, int aHeight,
                                   const KoverArtist::Image& aImg) const
{
   const QPixmap& ipix = aImg.pixmap();
   QRect rect(0, 0, aWidth, aHeight);
   int wpix = ipix.width();
   int hpix = ipix.height();
   int wrec = rect.width();
   int hrec = rect.height();
   int w, h, x, y;
   double f;

   QPixmap pix(aWidth, aHeight);
   if (pix.isNull()) return pix;

   QPainter p(&pix);
   p.fillRect(0, 0, aWidth, aHeight, QBrush(mProject->backgroundColor()));

   switch (aImg.position())
   {
   case KoverArtist::Centered:
      p.drawPixmap((wrec-wpix)>>1, (hrec-hpix)>>1, ipix);
      break;

   case KoverArtist::Tiled:
      p.drawTiledPixmap(rect, ipix);
      break;

   case KoverArtist::CenteredTiled:
      x = wpix - ((wrec%wpix)>>1);
      y = hpix - ((hrec%hpix)>>1);
      p.drawTiledPixmap(rect, ipix, QPoint(x, y));
      break;

   case KoverArtist::CenteredMaxpect:
      f = fmin(double(wrec)/wpix, double(hrec)/hpix);
      w = int(wpix * f);
      h = int(hpix * f);
      rect.setX(rect.x() + ((wrec-w)>>1));
      rect.setY(rect.y() + ((hrec-h)>>1));
      rect.setWidth(w);
      rect.setHeight(h);
      p.drawPixmap(rect, ipix);
      break;

   case KoverArtist::Scaled:
      p.drawPixmap(rect, ipix);
      break;

   case KoverArtist::ScaleCrop:
      p.setClipRect(rect, Qt::ReplaceClip);
      f = fmax(double(wrec)/wpix, double(hrec)/hpix);
      w = int(wpix*f);
      h = int(hpix*f);
      rect.setX(rect.x()+((wrec-w)>>1));
      rect.setY(rect.y()+((hrec-h)>>1));
      rect.setWidth(w);
      rect.setHeight(h);
      p.drawPixmap(rect, ipix);
      break;

   default:
      QMessageBox::critical(0, i18n("KoverArtist Error"),
         i18n("Internal error: invalid\nimage position #%1 %2").arg(int(aImg.position()))
            .arg(Image::imagePositionAsString(aImg.position())));
      break;
   }

   p.end();
   return pix;
}


void RendererBase::preparePixmaps()
{
   const Case& c = mProject->box();
   Image& imgFront = mProject->imgFront();
   Image& imgBack = mProject->imgBack();

   int fh = scaleY(c.front().height());
   int bh = scaleY(c.back().height());

   int fw = c.front().width();
   if (imgFront.onSides()) fw += c.frontLeftSide() + c.frontRightSide();
   fw = scaleX(fw);

   int bw = c.back().width();
   if (imgBack.onSides()) bw += c.backLeftSide() + c.backRightSide();
   bw = scaleX(bw);

   if (mProject->imgFrontWrapAround() && c.hasFront() && c.hasBack())
   {
      int ww = fw + bw;
      int h = fh > bh ? fh : bh;

      QPixmap pix = createPixmap(ww, h, imgFront);

      mPixBack = QPixmap(bw, bh);
      QPainter bp(&mPixBack);
      bp.drawPixmap(0, 0, pix, 0, 0, bw, bh);

      mPixFront = QPixmap(fw, fh);
      QPainter fp(&mPixFront);
      fp.drawPixmap(0, 0, pix, bw, 0, fw, fh);
   }
   else
   {
      mPixFront = createPixmap(fw, fh, imgFront);
      mPixBack = createPixmap(bw, bh, imgBack);
   }
}


} //namespace
