/*
   Copyright (C) 2006-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#ifndef mainwindow_h
#define mainwindow_h

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "project.h"
#include "renderer.h"


#include <kapplication.h>
#include <kmainwindow.h>
#include <kurl.h>


class KColorButton;
class KEditListBox;
class KPrinter;
class KUrl;
class KURLRequester;

class QBoxLayout;
class QComboBox;
class QLabel;
class QLineEdit;
class QMenu;
class QPushButton;
class QTimer;
class QTabBar;
class QTab;
class QTextEdit;

class QCheckBox;
class QSlider;
class QStackedWidget;

class TabProject;
class TabOptions;
class TabDisc;
class TextStyleConfig;
class CddbQuery;


/**
* This class serves as the main window for KoverArtist.  It handles the
* menus, toolbars, and status bars.
*
* @short Main window class
*/
class MainWindow: public KMainWindow
{
   typedef KMainWindow Inherited;
   Q_OBJECT

public:
   /**
   * Default Constructor
   */
   MainWindow();

   /**
   * Default Destructor
   */
   virtual ~MainWindow();

   /** Import the disc d from url. */
   bool discImport(KoverArtist::Disc& d, const KUrl& url);

   /** Export the disc d to url. */
   bool discExport(const KoverArtist::Disc& d, const KUrl& url);

public slots:
   /** Mark project as modified. */
   void modified();

   /** Load the project from the given url. */
   virtual bool load(const KUrl& url);

   /** Save the project to an url. */
   virtual bool save(const KUrl& url);

protected slots:
   void fileNew();
   void fileOpen();
   void fileRevert();
   bool fileSave();
   bool fileSaveAs();
   void filePrint();

   void windowNew();

   void discImport();
   void discImportCddb();
   void discExport();
   void discNew();
   void discDelete();
   void discFront();
   void discBack();
   void discClear();

   void searchCoverImages();

   void changeStatusbar(const QString& text);
   void changeCaption(const QString& text=QString::null);

   void switchToTab(int);

   void redisplay();
   void settings();

   void updatePreview(int msec=-1);
   void updatePreviewFront(int msec=-1);
   void updatePreviewBack(int msec=-1);
   void updatePreviewNow();

   void updatePreviewTimeout();

   void cddbQueryAudioCdDone();

   /** Update the discs tabbar to the number of discs the project has. */
   void updateDiscsTabBar();

protected:
   virtual void dragEnterEvent(QDragEnterEvent *event);
   virtual void dropEvent(QDropEvent *event);

protected:
   /** The window shall be closed. */
   virtual bool queryClose();

   /** An image url was dropped. Let the user select what to do with the image. */
   virtual bool imageDropped(const KUrl& url, const QPoint& globalMousePos);

   /** Test if the given url can be decoded by looking at the url only. */
   virtual bool canDecode(const QString& url) const;

   /** Test if the file is a image by looking at the filename. */
   virtual bool isImageFile(const QString& fname) const;

private:
   /** Update the contents of the widgets. */
   virtual void updateContents();

   /** Update the window title. */
   virtual void updateTitle();

   bool confirmDiscNotEmpty(int discIndex);

   QPushButton* createButton(const char* icon, QWidget* parent,
                             const char* slot, QBoxLayout* box=0) const;
   QLabel* createCaptionLabel(const QString& aText, QWidget* aParent) const;

   /** Setup the actions. */
   virtual void setupActions();

   /** Setup the menubar. */
   virtual void setupMenuBar();

   /** Setup the toolbar. */
   virtual void setupToolBar();

   /** Create an @ref QAction action object. */
   virtual QAction* createAction(const QString& actionName,
                                 const QString& text,
                                 const char* shortcut,
                                 QObject* object, const char* slotName,
                                 const char* iconName = 0);

   void scheduleUpdatePreview(int msec=-1);

private:
   MainWindow(const MainWindow&);
   MainWindow& operator=(const MainWindow&);

private:
   const QString mTitle;
   KoverArtist::Project mProject;
   KoverArtist::Renderer *mRender;
   QPrinter *mPrinter;
   CddbQuery *mCddbQuery;
   QFont mCaptionLblFont;
   int mIdActiveDisc, mNumTabsNoDisc, mCddbDiscId;
   QString mLocalFileName;
   QPixmap mPixPreview;
   bool mUpdPreviewFront, mUpdPreviewBack, mRendering;
   QLabel *mPreview;
   QTimer *mTmrUpdate;

   QTabBar *mTabBar;
   QStackedWidget *mStkSide;
   QWidget *mBaseProject, *mBaseOptions, *mBaseDisc;
   QMenu *mMnuFile, *mMnuDisc;

   int mTabIdProject, mTabIdOptions, mTabIdDisc;
   TabProject *mTabProject;
   TabOptions *mTabOptions;
   TabDisc *mTabDisc;

   QMap<QString,QAction*> mActions;
};

#endif // _KOVERARTIST_H_

