/*	$NetBSD: $ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/vmem.h>

#include <dev/ofw/openfirm.h>

#include <machine/autoconf.h>
#include <sys/bus.h>

#include <arch/powerpc/mpc5200/mpc5200reg.h>
#include <arch/powerpc/mpc5200/mpc5200var.h>
#include <arch/powerpc/mpc5200/sramvar.h>

static int sram_match(device_t, cfdata_t, void *);
static void sram_attach(device_t, device_t, void *);

struct sram_softc {
	device_t		sc_dev;
	bus_space_tag_t		sc_iot;
	bus_space_handle_t	sc_ioh;
	vmem_t *		sc_arena;
};

CFATTACH_DECL_NEW(sram, sizeof(struct sram_softc),
	sram_match, sram_attach, NULL, NULL);

static struct sram_softc *sram_sc;

static int
sram_match(device_t parent, cfdata_t cf, void *aux)
{
	struct confargs *ca = aux;
	char name[32];

	if (strcmp(ca->ca_name, "sram") != 0)
		return 0;

	OF_getprop(ca->ca_node, "compatible", name, sizeof(name));
	if (strcmp(name, "mpc5200-sram") != 0)
		return 0;

	return 1;
}

static void
sram_attach(device_t parent, device_t self, void *aux)
{
	struct sram_softc *sc = device_private(self);
	struct confargs *ca = aux;
	struct reg {
		uint32_t addr;
		uint32_t size;
	} range, *rp = &range;

	aprint_normal("\n");

	sc->sc_dev = self;
	sc->sc_iot = ca->ca_tag;

	if (OF_getprop(ca->ca_node, "reg", rp, sizeof(range))
	    != sizeof(range))
		return;

	if (bus_space_map(sc->sc_iot, rp->addr, rp->size,
			  BUS_SPACE_MAP_LINEAR, &sc->sc_ioh) != 0)
		aprint_normal_dev(self, "map failed\n");
#if 0
	sc->sc_arena = vmem_create(device_xname(sc->sc_dev),
				   (vmem_addr_t) bus_space_vaddr(sc->sc_iot,
								 sc->sc_ioh),
				   rp->size, sizeof(uint32_t), NULL, NULL,
				   NULL, 1, VM_NOSLEEP, IPL_VM);
#endif
	sram_sc = sc;
}

vmem_addr_t
sram_alloc(size_t size)
{
	vmem_addr_t addr;

	if (vmem_alloc(sram_sc->sc_arena, size, VM_BESTFIT | VM_NOSLEEP, &addr) != 0)
		return addr;
	else return 0;
}
