/*	$NetBSD: $ */

/*-
 * Copyright (c) 2009 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include "opt_interrupt.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/kernel.h>

#include <uvm/uvm_extern.h>

#define _POWERPC_BUS_DMA_PRIVATE
#include <machine/pio.h>
#include <sys/bus.h>

#include <dev/ofw/openfirm.h>

#include <machine/autoconf.h>
#include <arch/powerpc/pic/picvar.h>

#include <arch/powerpc/mpc5200/mpc5200reg.h>
#include <arch/powerpc/mpc5200/mpc5200var.h>
#include <arch/powerpc/mpc5200/bestcommvar.h>

static int bestcomm_match(device_t, cfdata_t, void *);
static void bestcomm_attach(device_t, device_t, void *);

struct bestcomm_softc {
	device_t	sc_dev;

	void		*sc_ih;
	int		sc_irq;
};

CFATTACH_DECL_NEW(bestcomm, sizeof(struct bestcomm_softc),
	bestcomm_match, bestcomm_attach, NULL, NULL);

static void bestcomm_enable_irq(struct pic_ops *, int, int);
static void bestcomm_disable_irq(struct pic_ops *, int);
static int  bestcomm_get_irq(struct pic_ops *, int);
static void bestcomm_ack_irq(struct pic_ops *, int);
static void bestcomm_establish_irq(struct pic_ops *, int, int, int);
/*static void bestcomm_intr(void *);*/

struct bestcomm_pic_ops {
	struct pic_ops pic;
	uint32_t enable_mask;
	uint32_t pending_events;
};

struct powerpc_bus_dma_tag bestcomm_bus_dma_tag = {
	0,			/* _bounce_thresh */
	_bus_dmamap_create,
	_bus_dmamap_destroy,
	_bus_dmamap_load,
	_bus_dmamap_load_mbuf,
	_bus_dmamap_load_uio,
	_bus_dmamap_load_raw,
	_bus_dmamap_unload,
	NULL,			/* _dmamap_sync */
	_bus_dmamem_alloc,
	_bus_dmamem_free,
	_bus_dmamem_map,
	_bus_dmamem_unmap,
	_bus_dmamem_mmap
};

#define SDMA_PENDING_REG		((uint32_t)pic->pic_cookie + 0x14)
#define SDMA_MASK_REG			((uint32_t)pic->pic_cookie + 0x18)

#define SDMA_LEVEL_MASK			0xffff0000

static int
bestcomm_match(device_t parent, cfdata_t cf, void *aux)
{
	struct confargs *ca = aux;

	if (strcmp(ca->ca_name, "bestcomm") != 0)
		return 0;

	return 1;
}

static void
bestcomm_attach(device_t parent, device_t self, void *aux)
{
	struct bestcomm_pic_ops *bestcomm_pic;
	struct bestcomm_softc *sc = device_private(self);
	struct confargs *ca = aux;
	struct pic_ops *pic;
	struct reg {
		uint32_t addr;
		uint32_t size;
	} range, *rp = &range;

	sc->sc_dev = self;

	aprint_normal("\n");
	if (OF_getprop(ca->ca_node, "reg", rp, sizeof(range))
	    != sizeof(range))
		return;

	bestcomm_pic = malloc(sizeof(struct bestcomm_pic_ops), M_DEVBUF, M_NOWAIT);
	KASSERT(bestcomm_pic != NULL);
	pic = &bestcomm_pic->pic;

	pic->pic_numintrs = 32;
	pic->pic_cookie = mapiodev(rp->addr, rp->size, false);
	pic->pic_enable_irq = bestcomm_enable_irq;
	pic->pic_reenable_irq = bestcomm_enable_irq;
	pic->pic_disable_irq = bestcomm_disable_irq;
	pic->pic_get_irq = bestcomm_get_irq;
	pic->pic_ack_irq = bestcomm_ack_irq;
	pic->pic_establish_irq = bestcomm_establish_irq;
	pic->pic_finish_setup = NULL;
	strcpy(pic->pic_name, "bestcomm");

	bestcomm_pic->pending_events = 0;
	bestcomm_pic->enable_mask = 0;
	pic_add(pic);

	out32(SDMA_PENDING_REG, 0xffffffff);
	out32(SDMA_MASK_REG, 0xffffffff);

}

static void
bestcomm_enable_irq(struct pic_ops *pic, int irq, int type)
{
	struct bestcomm_pic_ops *bestcomm_pic = (struct bestcomm_pic_ops *)pic;
	uint32_t mask = 1 << (31 - irq);

	bestcomm_pic->enable_mask |= mask;
	out32(SDMA_MASK_REG, ~(bestcomm_pic->enable_mask) & SDMA_LEVEL_MASK);
}

static void
bestcomm_disable_irq(struct pic_ops *pic, int irq)
{
	struct bestcomm_pic_ops *bestcomm_pic = (struct bestcomm_pic_ops *)pic;
	uint32_t mask = 1 << (31 - irq);

	bestcomm_pic->enable_mask &= ~mask;
	out32(SDMA_MASK_REG, ~(bestcomm_pic->enable_mask) & SDMA_LEVEL_MASK);
}

static int
bestcomm_get_irq(struct pic_ops *pic, int mode)
{
	struct bestcomm_pic_ops *bestcomm_pic = (struct bestcomm_pic_ops *)pic;
	int bit, mask, irqs;

	if (bestcomm_pic->pending_events == 0) {
		irqs = in32(SDMA_PENDING_REG);
		bestcomm_pic->pending_events |= irqs & SDMA_LEVEL_MASK;
	}

	if (bestcomm_pic->pending_events == 0)
		return 255;

	else {
		bit = 31 - __builtin_clz(bestcomm_pic->pending_events);
		mask = 1 << bit;
		bestcomm_pic->pending_events &= ~mask;
		return bit;
	}

	/* we should never get here */
	return 255;
}

static void
bestcomm_ack_irq(struct pic_ops *pic, int irq)
{
}

static void
bestcomm_establish_irq(struct pic_ops *pic, int irq, int type, int maxlevel)
{
	aprint_normal("bestcomm_establish_irq: irq %d type %d\n", irq, type);

}

void
bestcomm_init_task(int id, uint32_t *code, size_t len)
{

}
