 /*
  * UAE - The Un*x Amiga Emulator
  *
  * WDC support (A1200/A4000)
  *
  */

#define SINGLE_WDC 1
#define WDC_DEBUG 0

#if WDC_DEBUG
#define DPRINTF(n,x) if (WDC_DEBUG>=n) write_log x
#else
#define DPRINTF(n,x)
#endif

#include "sysconfig.h"
#include "sysdeps.h"
#include <assert.h>

#include "config.h"
#include "options.h"
#include "custom.h"
#include "memory.h"
#include "autoconf.h"
#include "wdc.h"
#include "savestate.h"

#include <sys/endian.h>
#include <sys/types.h>
#include <sys/stat.h>

struct drivedata {
	int cylinders;
	int heads;
	int sectors;
	off_t numsectors;
	int multi_sectors;
	int read_only;

	int isopen;
	int fd;
#define DISKIOBUFSIZE 8192
	uae_u8 *iobuf;
};

typedef	enum wdcstate {
	INIT = 0,
	SELECT,
	READ,
	WRITE,
	READDATA,
	WRITEDATA,
	PREREAD,
	DONEREAD,
	DONE,
	ERROR,
	ABORT,
	CHANGED
} wdcstate_t;

#define GAYLE_INT_IDE     0x80
#define GAYLE_INT_IDEACK0 0x02
#define GAYLE_INT_IDEACK1 0x01

#define NUMDRIVES 2

struct wdc {
	uae_u16 data;
	uae_u8 error;
	uae_u8 seccnt;
	uae_u8 sector;
	uae_u8 sector2;
	uae_u8 cyl_lo;
	uae_u8 cyl_lo2;
	uae_u8 cyl_hi;
	uae_u8 cyl_hi2;
	uae_u8 sdh;
	/* uae_u8 command; */
	uae_u8 status;
	uae_u8 control;
	uae_u8 features;

	int lba48;

	wdcstate_t oldstate;

	wdcstate_t state;
	wdcstate_t prevstate;  /* reload this for each i/o chunk */
	wdcstate_t nextstate;  /* advance when complete */

	/* disk offset and total sector count */
	off_t iooff;
	int iolen;

	/* outstanding number of sectors */
	int iocount;
	/* size of partial I/O operation */
	int iosectors;

	/* size of data in buffer and residual count */
	int iosize;
	int ioresid;
};

struct ide {
	uae_u8 intreq;
	uae_u8 intena;
	uae_u8 intena2;

	/* controller registers and state */
	int channel;
	struct wdc wdc[NUMDRIVES];
	struct drivedata drivedata[NUMDRIVES];
};

static struct ide the_ide;

#define use_channel(n) do { the_ide.channel = (n); } while (0)
#define DRIVEDATA(wdc) (&the_ide.drivedata[the_ide.channel])
#define GAYLE_INTREQ(wdc) (the_ide.intreq)
#define GAYLE_INTENA(wdc) (the_ide.intena)
#define GAYLE_INTENAR(wdc) (the_ide.intena|the_ide.intena2)
#define WDC(i) (the_ide.wdc[i])

#if SINGLE_WDC

#define THE_WDC WDC(0)

#else

#define THE_WDC WDC(the_ide.channel)

#endif

/****************************************************************/

static void cleanup_drive(struct drivedata *d)
{
	if (d->isopen)
		close(d->fd);
	d->fd = -1;
	d->isopen = 0;

	if (d->iobuf)
		free(d->iobuf);
	d->iobuf = NULL;
}

static void setup_drive(struct drivedata *d, char *filename,
			int ro, int heads, int sectors)
{
	int fd;
	struct stat st;
	off_t n;
	void *buf;

	write_log("setup_drive %s\n",filename);

	cleanup_drive(d);

	fd = open(filename, O_RDWR);
	if (fd < 0 || fstat(fd, &st) < 0)
		return;

	buf = malloc(DISKIOBUFSIZE);
	if (buf == NULL) {
		close(fd);
		return;
	}

#if 0
	fcntl(fd, F_SETFL, O_NONBLOCK);
#endif

	d->fd = fd;
	d->iobuf = buf;

	n = st.st_size / 512;
	if (n > 67108864)
		n = 67108864;

	d->numsectors = n;
	d->heads = heads > 0 ? heads : 64;
	d->sectors = sectors > 0 ? sectors : 32;
	d->cylinders = n/(d->heads * d->sectors);

	d->multi_sectors = DISKIOBUFSIZE / 512;

	d->read_only = ro;

	d->isopen = 1;

	write_log("drive %d geometry %d/%d/%d = %d\n",d->fd,
		d->cylinders, d->heads, d->sectors, d->numsectors);
}

static wdcstate_t read_drive(struct drivedata *d, off_t x, int n)
{
	off_t pos = x * 512;
	size_t nbytes = n * 512;
	ssize_t rc;

	if (!d->isopen)
		return ABORT;

DPRINTF(2,("read_drive %d seek %qd\n", d->fd, pos);)
	if (lseek(d->fd, pos, SEEK_SET) == (off_t)-1) {
		perror("lseek");
		return ERROR;
	}

	rc = read(d->fd, d->iobuf, nbytes);
DPRINTF(2,("read_drive rc = %d (fd = %d, bytes = %d, errno = %d)\n",rc,d->fd,nbytes,errno);)

	if (rc == -1 && errno == EAGAIN)
		return READ;
	else if (rc != (ssize_t) nbytes)
		return ERROR;
	return READDATA;
}

static wdcstate_t write_drive(struct drivedata *d, off_t x, int n)
{
	off_t pos = x * 512;
	size_t nbytes = n * 512;
	ssize_t rc;

	if (!d->isopen)
		return ABORT;

	if (d->read_only)
		return ERROR;

DPRINTF(2,("write_drive %d seek %qd\n", d->fd, pos);)
	if (lseek(d->fd, pos, SEEK_SET) == (off_t)-1)
		return ERROR;

	rc = write(d->fd, d->iobuf, nbytes);

	if (rc == -1 && errno == EAGAIN)
		return WRITE;
	else if (rc != (ssize_t) nbytes)
		return ERROR;
	else
		return DONE;
}

static wdcstate_t flush_drive(struct drivedata *d)
{
	int rc;

	if (!d->isopen)
		return ABORT;

DPRINTF(2,("flush_drive %d\n", d->fd);)
	rc = fdatasync(d->fd);
	if (rc == -1)
		return ERROR;
	else
		return DONE;
}

/****************************************************************/

static uae_u64 wdc_get_sector(struct wdc *wdc)
{
    uae_u64 s;
    if (wdc->sdh & 0x40) {
        if (wdc->lba48) {
            s =
                (uae_u64)wdc->cyl_hi2 << 40 |
                (uae_u64)wdc->cyl_lo2 << 32 |
                (uae_u64)wdc->sector2 << 24 |
                (uae_u64)wdc->cyl_hi << 16 |
                (uae_u64)wdc->cyl_lo << 8 |
                (uae_u64)wdc->sector;
DPRINTF(3,("LBA48(%qu)\n",s);)
        } else {
            s =
                (uae_u64)(wdc->sdh & 0xf) << 24 |
                (uae_u64)wdc->cyl_hi << 16 |
                (uae_u64)wdc->cyl_lo << 8 |
                (uae_u64)wdc->sector;
DPRINTF(3,("LBA(%qu)\n",s);)
        }

    } else {
        struct drivedata *d = DRIVEDATA(wdc);
        s =
            (
                ((uae_u64)wdc->cyl_hi << 8 | wdc->cyl_lo) * d->heads +
                (wdc->sdh & 0x0f)
            ) * d->sectors + wdc->sector - 1;
DPRINTF(3,("CHS(%qu,%u,%u) = %qu GEOM(%u,%u)\n",wdc->cyl_hi<<8|wdc->cyl_lo,
 wdc->sdh & 0x0f, s, wdc->sector,d->heads,d->sectors);)

    }
    return s;
}

#if 0
static void wdc_set_sector(struct wdc *wdc, uae_u64 s)
{
    if (wdc->sdh & 0x40) {
        if (wdc->lba48) {
            wdc->sector = s;
            wdc->cyl_lo = s >> 8;
            wdc->cyl_hi = s >> 16;
            wdc->sector2 = s >> 24;
            wdc->cyl_lo2 = s >> 32;
            wdc->cyl_hi2 = s >> 40;
        } else {
            wdc->sdh = (wdc->sdh & 0x40) | (s & 0x0f000000) >> 24;
            wdc->cyl_hi = s >> 16;
            wdc->cyl_lo = s >> 8;
            wdc->sector = s;
        }
    } else {
        unsigned int c, r;
	struct drivedata *d = DRIVEDATA(wdc);

        c = s / (d->heads * d->sectors);
        r = s - c * (d->heads * d->sectors);
        wdc->cyl_hi = c >> 8;
        wdc->cyl_lo = c;
        wdc->sdh = (wdc->sdh & 0xf0) | (r/d->sectors & 0xf);
        wdc->sector = r%d->sectors+1;
    }
}
#endif

static void wdc_interrupt(struct wdc *wdc, int onoff)
{
DPRINTF(3,("INTREQ(%d)\n",onoff);)
	if (onoff) {
		GAYLE_INTREQ(wdc) |= GAYLE_INT_IDE;
		if ((GAYLE_INTENAR(wdc) & GAYLE_INT_IDE) != 0 &&
		    (wdc->control & WDCTL_IDS) == 0) {
			INTREQ_PORTS(0x8001);
		}
	} else {
		/* GAYLE_INTREQ(wdc) &= ~GAYLE_INT_IDE; */
		INTREQ_PORTS(0x0001);
	}
}

static void wdc_get_status(struct wdc *wdc)
{
	wdcstate_t oldstate = wdc->state;
	struct drivedata *d = DRIVEDATA(wdc);

	switch (oldstate) {
	case READ:
DPRINTF(2,("read_drive(%qd,%d)\n",wdc->iooff,wdc->iosectors);)
		wdc->state = read_drive(d, wdc->iooff, wdc->iosectors);
		wdc->iooff += wdc->iosectors;
		break;
	case WRITE:
DPRINTF(2,("write_drive(%qd,%d)\n",wdc->iooff,wdc->iosectors);)
		wdc->state = write_drive(d, wdc->iooff, wdc->iosectors);
		wdc->iooff += wdc->iosectors;
		if (wdc->state == DONE && wdc->iocount > 0)
			wdc->state = WRITEDATA;
		break;
	case PREREAD:
		wdc->state = READDATA;
		break;
	default:
		break;
	}

	if (wdc->state != wdc->oldstate) {
DPRINTF(3,("wdc_get_status = %d %d %d\n", wdc->oldstate, oldstate, wdc->state);)
		wdc->oldstate = wdc->state;
	}

	if (DRIVEDATA(wdc)->isopen) {
		wdc->status |= WDCS_DRDY;
	} else {
		wdc->status &= ~WDCS_DRDY;
	}

	switch (wdc->state) {
	case INIT:
	case PREREAD:
		break;
	case SELECT:
		wdc->status &= ~WDCS_DWF;
		wdc->status &= ~WDCS_BSY;
		wdc->status |= WDCS_DRQ;
		break;
	case READ:
	case WRITE:
		wdc->status &= ~(WDCS_DWF | WDCS_ERR);
		wdc->status |= WDCS_BSY;
		wdc->status &= ~WDCS_DRQ;
		break;
	case WRITEDATA:
	case READDATA:
		wdc->status &= ~(WDCS_DWF | WDCS_ERR);
		wdc->status &= ~WDCS_BSY;
		wdc->status |= WDCS_DRQ;
		if (oldstate != wdc->state)
			wdc_interrupt(wdc, 1);
		break;
	case DONEREAD:
		wdc->status &= ~(WDCS_BSY | WDCS_DRQ);
		wdc->state = INIT;
		break;
	case DONE:
		wdc->status &= ~(WDCS_BSY | WDCS_DRQ);
		wdc->state = INIT;
		wdc_interrupt(wdc, 1);
		break;
	case CHANGED:
		wdc->status &= ~(WDCS_BSY | WDCS_DRQ);
DPRINTF(2,("wdc_get_status signal media change\n");)
		wdc->error = WDCE_MC;
		wdc->state = INIT;
		wdc_interrupt(wdc, 1);
		break;
	case ABORT:
		wdc->status |= WDCS_ERR;
		wdc->status &= ~(WDCS_BSY | WDCS_DRQ);
DPRINTF(2,("wdc_get_status signal abort\n");)
		wdc->error = WDCE_ABRT;
		wdc->state = INIT;
		wdc_interrupt(wdc, 1);
		break;
	case ERROR:
		wdc->status |= WDCS_ERR;
		if (oldstate == WRITE)
			wdc->status |= WDCS_DWF;
		else
			wdc->status &= ~WDCS_DWF;
		wdc->status &= ~(WDCS_BSY | WDCS_DRQ);
DPRINTF(2,("wdc_get_status signal error status %02x\n",wdc->status);)
		wdc->error = WDCE_UNC;
		wdc->state = INIT;
		wdc_interrupt(wdc, 1);
		break;
	}
}

static void wdc_read_data(struct wdc *wdc)
{
	struct drivedata *d = DRIVEDATA(wdc);
	uae_u8 *p;

	switch (wdc->state) {
	case READDATA:
		if (wdc->ioresid > 1) {
			if (d->iobuf) {
				p = &d->iobuf[wdc->iosize - wdc->ioresid];
				wdc->data = p[0] << 8 | p[1];
			}
			wdc->ioresid -= 2;
		} else
			write_log("wdc_read_data: stale data\n");

		if (wdc->ioresid < 2) {
DPRINTF(2,("wdc_read_data @%lld %d %d\n",wdc->iooff,wdc->iocount,wdc->iosectors);)
			if (wdc->iocount >= wdc->iosectors)
				wdc->iocount -= wdc->iosectors;
			else {
				write_log("iocount underflow\n");
				wdc->iocount = 0;
			}
			if (wdc->iocount > 0) {
				if (wdc->iocount < wdc->iosectors) {
					wdc->iosectors = wdc->iocount;
					wdc->iosize = wdc->iosectors * 512;
				}

				wdc->ioresid = wdc->iosize;
				wdc->state   = wdc->prevstate;
			} else {
				wdc->iocount   = 0;
				wdc->ioresid   = 0;
				wdc->state     = wdc->nextstate;
				wdc->nextstate = INIT;
DPRINTF(2,("wdc_read_data -> %d next %d\n",wdc->state,wdc->nextstate);)
			}
			wdc_get_status(wdc);
		}
		break;
	default:
		write_log("wdc_read_data: not in readdata state\n");
		break;
	}
}

static void wdc_write_data(struct wdc *wdc)
{
	struct drivedata *d = DRIVEDATA(wdc);
	uae_u8 *p;

	switch (wdc->state) {
	case WRITEDATA:
		if (wdc->iocount < wdc->iosectors) {
			wdc->iosectors = wdc->iocount;
			wdc->iosize = wdc->iosectors * 512;
			wdc->ioresid = wdc->iosize;
		}

		if (wdc->ioresid > 1) {
			if (d->iobuf) {
				p = &d->iobuf[wdc->iosize - wdc->ioresid];
				p[0] = wdc->data >> 8;
				p[1] = wdc->data;
			}
			wdc->ioresid -= 2;
		}
		if (wdc->ioresid < 2) {
			if (wdc->iocount >= wdc->iosectors)
				wdc->iocount -= wdc->iosectors;
			else {
				write_log("iocount underflow\n");
				wdc->iocount = 0;
			}
			if (wdc->iocount > 0) {
				wdc->ioresid = wdc->iosize;
				wdc->state   = wdc->prevstate;
			} else {
				wdc->iocount   = 0;
				wdc->ioresid   = 0;
				wdc->state     = wdc->nextstate;
				wdc->nextstate = INIT;
			}
			wdc_get_status(wdc);
		}
		break;
	default:
		write_log("wdc_write_data: not in writedata state\n");
		break;
	}
}

/****************************************************************/

static void cmd_invalid(struct wdc *wdc)
{
DPRINTF(1,("cmd_invalid\n");)

	wdc->state     = ABORT;
	wdc->prevstate = ABORT;
	wdc->nextstate = DONE;
}

static void cmd_dummy(struct wdc *wdc)
{
DPRINTF(1,("cmd_dummy\n");)

	wdc->state     = DONE;
	wdc->prevstate = DONE;
	wdc->nextstate = DONE;
}

static void bufset(uae_u16 *b, char *src, size_t n)
{
	size_t i;

	strncpy((char *)b, src, n);

	for (i=0; i<n; i += sizeof(uae_u16)) {
		*b = bswap16(*b);
		++b;
	}
}

static void cmd_identify(struct wdc *wdc)
{
	struct drivedata *d = DRIVEDATA(wdc);
	uae_u16 *b = (uae_u16 *)d->iobuf;
	uae_u32 n = d->cylinders * d->heads * d->sectors;

DPRINTF(1,("cmd_identify\n");)

	if (!d->isopen) {
		wdc->state     = ABORT;
		wdc->prevstate = ABORT;
		wdc->nextstate = DONE;
		return;
	}

	memset(b, 0, 256 * sizeof(*b));
	b[0]   = htole16(0x0040); /* ATA_CFG_FIXED */
	b[1]   = htole16(d->cylinders);
	b[3]   = htole16(d->heads);
	b[4]   = htole16(d->sectors * 512);
	b[5]   = htole16(512);
	b[6]   = htole16(d->sectors);
DPRINTF(1,("CHS %u * %u * %u\n",le16toh(b[1]),le16toh(b[3]),le16toh(b[6]));)

	strncpy((char *)&b[10], "1234567890", 20);
	b[20]  = htole16(3);
	b[21]  = htole16(512); /* cache size */
	b[22]  = htole16(4); /* ecc bytes */

	bufset(&b[23], "1.0", 8);
	bufset(&b[27], "UAE Harddrive", 40);
	if (d->multi_sectors)
		b[47]  = htole16(0x8000 | (d->multi_sectors & 0xff));
	b[48]  = htole16(1); /* dword I/O */
	b[49]  = htole16(1<<11|1<<9|0<<8); /* IORDY, LBA, no DMA */
	b[51]  = htole16(0x200); /* PIO transfer cycle */
	b[52]  = htole16(0x200); /* DMA transfer cycle */
	b[53]  = htole16(1|(1<<1)|(0<<2)); /* GEOM, MODES, UDMA */

	/* ATA */
	b[54]  = htole16(d->cylinders);
	b[55]  = htole16(d->heads);
	b[56]  = htole16(d->sectors);
DPRINTF(1,("geom %u * %u * %u\n",le16toh(b[54]),le16toh(b[55]),le16toh(b[56]));)
	b[57]  = htole16(n);
	b[58]  = htole16(n >> 16);
DPRINTF(1,("oldsize %u << 16 | %u\n",le16toh(b[58]),le16toh(b[57]));)
	if (d->multi_sectors)
		b[59]  = htole16(0x100 | (d->multi_sectors & 0xff));
	b[60]  = htole16(d->numsectors);
	b[61]  = htole16(d->numsectors >> 16);
DPRINTF(1,("newsize %u << 16 | %u\n",le16toh(b[61]),le16toh(b[60]));)

	b[63]  = htole16(0); /* Multiword DMA modes */
	b[64]  = htole16(0); /* Advanced PIO modes  */
	b[65]  = htole16(120); /* minimum DMA cycle times */
	b[66]  = htole16(120); /* recommended DMA cycle times */
	b[67]  = htole16(120); /* minimum PIO cycle times */
	b[68]  = htole16(120); /* minimum PIO cycle times with IORDY */

	b[80]  = htole16(0xf0); /* ATA Major Version numbers */
	b[81]  = htole16(0x16); /* ATA Minor Version number */

	/* cmd_set1 */
	/* NOP */
	b[82]  = htole16(0x4000);

	/* cmd_set2 */
	/* ????|FLUSH_CACHE_EXT|FLUSH_CACHE|ATA_CMD2_LBA48 */
	/* b[83]  = htole16(0x4000 | 0x2000 | 0x1000 | 0x400); */
	b[83]  = htole16(0x400);

	/* cmd_ext */
	/* ??? */
	b[84]  = htole16(0x4000);

	/* enable 82 */
	b[85]  = b[82];

	/* enable 83 */
	b[86]  = b[83];

	/* default ? */
	b[87]  = b[84];

	b[100] = htole16(d->numsectors);
	b[101] = htole16(d->numsectors >> 16);
	b[102] = htole16(d->numsectors >> 32);
	b[103] = htole16(d->numsectors >> 48);

	wdc->iocount   = 1;
	wdc->iosectors = 1;
	wdc->iosize    = 256 * sizeof(*b);
	wdc->ioresid   = wdc->iosize;
	wdc->state     = PREREAD;
	wdc->prevstate = READDATA;
	wdc->nextstate = DONEREAD;
}

static void cmd_drivediags(struct wdc *wdc)
{
	struct drivedata *d = DRIVEDATA(wdc);

DPRINTF(1,("cmd_drivediags\n");)

	if (!d->isopen) {
		wdc->state     = ABORT;
		wdc->prevstate = ABORT;
		wdc->nextstate = DONE;
		return;
	}

	wdc->error  = 0x01; /* Device 0 passed, Device 1 passed or not present */
	wdc->seccnt = 0x00; /* signature bytes, 0x0000 = ATA */
	wdc->sector = 0x00;
	wdc->cyl_lo = 0x00;
	wdc->cyl_hi = 0x00;
	wdc->sdh    = 0x00;

	wdc->state = DONE;
	wdc->prevstate = ABORT;
	wdc->nextstate = INIT;

	wdc_get_status(wdc);
}

static void cmd_setmulti(struct wdc *wdc)
{
	struct drivedata *d = DRIVEDATA(wdc);

DPRINTF(1,("cmd_setmulti %d\n",wdc->seccnt);)

	if (!d->isopen) {
		wdc->state     = ABORT;
		wdc->prevstate = ABORT;
		wdc->nextstate = DONE;
		return;
	}

	if (wdc->seccnt != d->multi_sectors)
		wdc->state = ABORT;
	else
		wdc->state = DONE;

	wdc->prevstate = ABORT;
	wdc->nextstate = INIT;

	wdc_get_status(wdc);
}

static void cmd_read(struct wdc *wdc, int lba48, int mult)
{
	wdc->lba48 = lba48;
	wdc->iooff = wdc_get_sector(wdc);
	wdc->iolen = wdc->seccnt ? wdc->seccnt : 256;

DPRINTF(1,("cmd_read %qd %d %d\n", wdc->iooff, wdc->iolen, mult);)

	if (mult) {
		int sectors = DRIVEDATA(wdc)->multi_sectors;
		if (sectors < 1) sectors = 1;
		if (sectors > DISKIOBUFSIZE/512)
			sectors = DISKIOBUFSIZE/512;
		if (sectors > wdc->iolen)
			sectors = wdc->iolen;

		wdc->iocount   = wdc->iolen;
		wdc->iosectors = sectors;
		wdc->iosize    = 512 * wdc->iosectors;
		wdc->ioresid   = wdc->iosize;
		wdc->state     = READ;
		wdc->prevstate = READ;
		wdc->nextstate = DONE;
	} else {
		wdc->iocount   = wdc->iolen;
		wdc->iosectors = 1;
		wdc->iosize    = 512;
		wdc->ioresid   = wdc->iosize;
		wdc->state     = READ;
		wdc->prevstate = READ;
		wdc->nextstate = DONE;
	}
}

static void cmd_write(struct wdc *wdc, int lba48, int mult)
{
	wdc->lba48 = lba48;
	wdc->iooff = wdc_get_sector(wdc);
	wdc->iolen = wdc->seccnt ? wdc->seccnt : 256;

DPRINTF(1,("cmd_write %qd %d %d\n", wdc->iooff, wdc->iolen, mult);)

	if (mult) {
		int sectors = DRIVEDATA(wdc)->multi_sectors;
		if (sectors < 1) sectors = 1;
		if (sectors > DISKIOBUFSIZE/512)
			sectors = DISKIOBUFSIZE/512;
		if (sectors > wdc->iolen)
			sectors = wdc->iolen;

		wdc->iocount   = wdc->iolen;
		wdc->iosectors = sectors;
		wdc->iosize    = 512 * wdc->iosectors;
		wdc->ioresid   = wdc->iosize;
		wdc->state     = WRITEDATA;
		wdc->prevstate = WRITE;
		wdc->nextstate = WRITE;
	} else {
		wdc->iocount   = wdc->iolen;
		wdc->iosectors = 1;
		wdc->iosize    = 512;
		wdc->ioresid   = wdc->iosize;
		wdc->state     = WRITEDATA;
		wdc->prevstate = WRITE;
		wdc->nextstate = WRITE;
	}
}

static void cmd_flushcache(struct wdc *wdc)
{
	struct drivedata *d = DRIVEDATA(wdc);
	
DPRINTF(1,("cmd_flushcache\n");)
	if (!d->isopen) {
		wdc->state     = ABORT;
		wdc->prevstate = ABORT;
		wdc->nextstate = DONE;
		return;
	}

	wdc->state = flush_drive(d);

	wdc->prevstate = ABORT;
	wdc->nextstate = INIT;

	wdc_get_status(wdc);
}

/****************************************************************/

static void InitWDC(void)
{
DPRINTF(1,("InitWDC()\n");)

	THE_WDC.data    = 0;
	THE_WDC.error   = 0;
	THE_WDC.sdh     = 0;
	THE_WDC.control = 0;

	THE_WDC.oldstate  = INIT;
	THE_WDC.state     = INIT;
	THE_WDC.nextstate = INIT;

	THE_WDC.iocount   = 0;
	THE_WDC.iosectors = 0;
	THE_WDC.iosize    = 0;
	THE_WDC.ioresid   = 0;

	/* wdc_interrupt(&THE_WDC, 0); */
}

static uae_u16 ReadWDC(int r)
{
	uae_u16 w = 0;

	switch (r) {
	case gayle_idebase  + wd_data:
	case gayle_idebase2 + wd_data:
	case gayle_idebase  + wd_data + dport_offset:
	case gayle_idebase2 + wd_data + dport_offset:
		wdc_read_data(&THE_WDC);
		w = THE_WDC.data;
		break;
	case gayle_idebase  + wd_error:
	case gayle_idebase2 + wd_error:
	case gayle_idebase  + wd_error + dport_offset:
	case gayle_idebase2 + wd_error + dport_offset:
		wdc_get_status(&THE_WDC);
		w = THE_WDC.error;
		break;
	case gayle_idebase  + wd_seccnt:
	case gayle_idebase2 + wd_seccnt:
	case gayle_idebase  + wd_seccnt + dport_offset:
	case gayle_idebase2 + wd_seccnt + dport_offset:
		w = THE_WDC.seccnt;
		break;
	case gayle_idebase  + wd_sector:
	case gayle_idebase2 + wd_sector:
	case gayle_idebase  + wd_sector + dport_offset:
	case gayle_idebase2 + wd_sector + dport_offset:
		w = (THE_WDC.control & WDCTL_HOB) ? THE_WDC.sector2 : THE_WDC.sector;
		break;
	case gayle_idebase  + wd_cyl_lo:
	case gayle_idebase2 + wd_cyl_lo:
	case gayle_idebase  + wd_cyl_lo + dport_offset:
	case gayle_idebase2 + wd_cyl_lo + dport_offset:
		w = (THE_WDC.control & WDCTL_HOB) ? THE_WDC.cyl_lo2 : THE_WDC.cyl_lo;
		break;
	case gayle_idebase  + wd_cyl_hi:
	case gayle_idebase2 + wd_cyl_hi:
	case gayle_idebase  + wd_cyl_hi + dport_offset:
	case gayle_idebase2 + wd_cyl_hi + dport_offset:
		w = (THE_WDC.control & WDCTL_HOB) ? THE_WDC.cyl_hi2 : THE_WDC.cyl_hi;
		break;
	case gayle_idebase  + wd_sdh:
	case gayle_idebase2 + wd_sdh:
	case gayle_idebase  + wd_sdh + dport_offset:
	case gayle_idebase2 + wd_sdh + dport_offset:
		w = THE_WDC.sdh;
		break;
	case gayle_idebase  + wd_command:
	case gayle_idebase2 + wd_command:
	case gayle_idebase  + wd_command + dport_offset:
	case gayle_idebase2 + wd_command + dport_offset:
		wdc_get_status(&THE_WDC);
		w = THE_WDC.status;
		GAYLE_INTREQ(THE_WDC) &= ~GAYLE_INT_IDE;
		wdc_interrupt(&THE_WDC, 0);
		break;
	case gayle_idebase  + wd_status:
	case gayle_idebase2 + wd_status:
		if (DRIVEDATA(&THE_WDC)->isopen) {
			wdc_get_status(&THE_WDC);
			w = THE_WDC.status;
DPRINTF(3,("WDC.status = %02x\n",w);)
		} else {
			w = 0;
DPRINTF(3,("WDC.status = null\n");)
		}
		break;
	case gayle_idebase  + wd_features:
	case gayle_idebase2 + wd_features:
		w = 0;
		break;
	case gayle_idebase  + wd_control:
	case gayle_idebase2 + wd_control:
		// w = THE_WDC.control;
		w = THE_WDC.status;
DPRINTF(3,("WDC.status = %02x\n",w);)
		break;
	case gayle_intreq:
	case gayle_intreq2:
		w = GAYLE_INTREQ(&THE_WDC);
DPRINTF(3,("GAYLE_INTREQ = %02x\n",w);)
		break;
	case gayle_intena:
		w = GAYLE_INTENA(&THE_WDC);
DPRINTF(3,("GAYLE_INTENA = %02x\n",w);)
		break;
	default:
		write_log("Unknown ReadWDC(%08x,%04x)\n",r,w);
		break;
	}

	return w;
}

static void WriteWDC(int r, uae_u16 w)
{
	switch (r) {
	case gayle_idebase  + wd_data:
	case gayle_idebase2 + wd_data:
	case gayle_idebase  + wd_data + dport_offset:
	case gayle_idebase2 + wd_data + dport_offset:
		THE_WDC.data = w;
		wdc_write_data(&THE_WDC);
		break;
	case gayle_idebase  + wd_error:
	case gayle_idebase2 + wd_error:
	case gayle_idebase  + wd_error + dport_offset:
	case gayle_idebase2 + wd_error + dport_offset:
		THE_WDC.features = w;
		THE_WDC.control &= ~WDCTL_HOB;
		break;
	case gayle_idebase  + wd_seccnt:
	case gayle_idebase2 + wd_seccnt:
	case gayle_idebase  + wd_seccnt + dport_offset:
	case gayle_idebase2 + wd_seccnt + dport_offset:
		THE_WDC.seccnt = w;
		THE_WDC.control &= ~WDCTL_HOB;
		break;
	case gayle_idebase  + wd_sector:
	case gayle_idebase2 + wd_sector:
	case gayle_idebase  + wd_sector + dport_offset:
	case gayle_idebase2 + wd_sector + dport_offset:
		THE_WDC.sector = w;
		THE_WDC.control &= ~WDCTL_HOB;
		break;
	case gayle_idebase  + wd_cyl_lo:
	case gayle_idebase2 + wd_cyl_lo:
	case gayle_idebase  + wd_cyl_lo + dport_offset:
	case gayle_idebase2 + wd_cyl_lo + dport_offset:
		THE_WDC.cyl_lo2 = THE_WDC.cyl_lo;
		THE_WDC.cyl_lo = w;
		THE_WDC.control &= ~WDCTL_HOB;
		break;
	case gayle_idebase  + wd_cyl_hi:
	case gayle_idebase2 + wd_cyl_hi:
	case gayle_idebase  + wd_cyl_hi + dport_offset:
	case gayle_idebase2 + wd_cyl_hi + dport_offset:
		THE_WDC.cyl_hi2 = THE_WDC.cyl_hi;
		THE_WDC.cyl_hi = w;
		THE_WDC.control &= ~WDCTL_HOB;
		break;
	case gayle_idebase  + wd_sdh:
	case gayle_idebase2 + wd_sdh:
	case gayle_idebase  + wd_sdh + dport_offset:
	case gayle_idebase2 + wd_sdh + dport_offset:
DPRINTF(1,("SELECT %d\n",(w >> 4) & 1);)
		use_channel((w >> 4) & 1);
		THE_WDC.sdh = w;
		wdc_get_status(&THE_WDC);
		break;
	case gayle_idebase  + wd_command:
	case gayle_idebase2 + wd_command:
	case gayle_idebase  + wd_command + dport_offset:
	case gayle_idebase2 + wd_command + dport_offset:
DPRINTF(1,("DISK_COMMAND 0x%04lx\n",w);)
		if ((THE_WDC.status & WDCS_BSY) != 0) {
			write_log("wdc busy, command ignored\n");
			break;
		}

		THE_WDC.status &= ~WDCS_ERR;

		if (!DRIVEDATA(&THE_WDC)->isopen)
			cmd_invalid(&THE_WDC);
		else switch (w & 0xff) {
		case WDCC_READ:
			cmd_read(&THE_WDC, 0, 0);
			break;
		case WDCC_WRITE:
			cmd_write(&THE_WDC, 0, 0);
			break;
		case WDCC_IDP:
			cmd_invalid(&THE_WDC);
			break;
		case WDCC_READMULTI:
			cmd_read(&THE_WDC, 0, 1);
			break;
		case WDCC_WRITEMULTI:
			cmd_write(&THE_WDC, 0, 1);
			break;
		case WDCC_SETMULTI:
			cmd_setmulti(&THE_WDC);
			break;
		case WDCC_RECAL:
			cmd_dummy(&THE_WDC);
			break;
		case WDCC_ACKMC:
			cmd_invalid(&THE_WDC);
			break;
		case WDCC_FLUSHCACHE:
			cmd_flushcache(&THE_WDC);
			break;
		case WDCC_FLUSHCACHE_EXT:
			cmd_flushcache(&THE_WDC);
			break;
		case WDCC_IDENTIFY:
			cmd_identify(&THE_WDC);
			break;
		case WDCC_READ_EXT:
			cmd_read(&THE_WDC, 1, 0);
			break;
		case WDCC_WRITE_EXT:
			cmd_write(&THE_WDC, 1, 0);
			break;
		case WDCC_READMULTI_EXT:
			cmd_read(&THE_WDC, 1, 1);
			break;
		case WDCC_WRITEMULTI_EXT:
			cmd_write(&THE_WDC, 1, 1);
			break;
		case WDCC_EXEC_DRIVE_DIAGS:
			cmd_drivediags(&THE_WDC);
			break;
		case WDCC_NOP:
		default:
			cmd_invalid(&THE_WDC);
			break;
		}
		wdc_get_status(&THE_WDC);
		break;
	case gayle_idebase  + wd_control:
	case gayle_idebase2 + wd_control:
DPRINTF(3,("WD_CONTROL(%02x)\n",w);)
		if (w & WDCTL_RST)
			InitWDC();
		THE_WDC.control = w;
		break;
	case gayle_intena:
DPRINTF(3,("GAYLE_INTENA(%02x)\n",w);)
		GAYLE_INTENA(THE_WDC) = w;
		break;
	case gayle_intreq:
	case gayle_intreq2:
DPRINTF(3,("GAYLE_INTREQ(%02x)\n",w);)
		if ((w & GAYLE_INT_IDE) == 0) {
			GAYLE_INTREQ(THE_WDC) &= w;
			wdc_interrupt(&THE_WDC, 0);
		}
		break;
	default:
		write_log("Unknown WriteWDC(%08x,%04x)\n",r,w);
		break;
	}
}

/****************************************************************/

/* WDC memory access */

static uae_u32 wdc_lget (uaecptr) REGPARAM;
static uae_u32 wdc_wget (uaecptr) REGPARAM;
static uae_u32 wdc_bget (uaecptr) REGPARAM;
static void wdc_lput (uaecptr, uae_u32) REGPARAM;
static void wdc_wput (uaecptr, uae_u32) REGPARAM;
static void wdc_bput (uaecptr, uae_u32) REGPARAM;

addrbank wdc_bank = { 
    wdc_lget, wdc_wget, wdc_bget,
    wdc_lput, wdc_wput, wdc_bput,
    default_xlate, default_check, NULL,
    "wdc"
};

#define regnum(addr) (((unsigned long)(addr & 0xfffc)) >> 2)

uae_u32 REGPARAM2 wdc_lget (uaecptr addr)
{
	uae_u32 value;

	special_mem |= S_READ;
	value = ReadWDC(regnum(addr)) << 16;
	value |= ReadWDC(regnum(addr));

DPRINTF(5,("wdc_lget(%08lx)=%08lx\n",addr,value);)
	return value;
}

uae_u32 REGPARAM2 wdc_wget (uaecptr addr)
{
	uae_u32 value;

	special_mem |= S_READ;
	value = ReadWDC(regnum(addr));

DPRINTF(5,("wdc_wget(%08lx)=%04lx\n",addr,value);)
	return value;
}

uae_u32 REGPARAM2 wdc_bget (uaecptr addr)
{
	uae_u32 value;

	special_mem |= S_READ;
	value = ReadWDC(regnum(addr)) & 0xFF;

DPRINTF(4,("wdc_bget(%08lx)=%02lx\n",addr,value);)
	return value;
}

void REGPARAM2 wdc_lput (uaecptr addr, uae_u32 value)
{
DPRINTF(5,("wdc_lput(%08lx,%08lx)\n",addr,value);)
	special_mem |= S_WRITE;
	WriteWDC(regnum(addr), value >> 16);
	WriteWDC(regnum(addr), value);
}   
        
void REGPARAM2 wdc_wput (uaecptr addr, uae_u32 value)
{
DPRINTF(5,("wdc_wput(%08lx,%04lx)\n",addr,value);)
	special_mem |= S_WRITE;
	WriteWDC(regnum(addr), value);
}   

void REGPARAM2 wdc_bput (uaecptr addr, uae_u32 value)
{
DPRINTF(4,("wdc_bput(%08lx,%02lx)\n",addr,value);)
	special_mem |= S_WRITE;
	WriteWDC(regnum(addr), value);
}

#undef regnum

/****************************************************************/

/* Setup */
uae_u8 *restore_wdc (uae_u8 *src)
{
	int i;

	for (i=0; i<NUMDRIVES; ++i) {
		use_channel(i);
		InitWDC();
	}
	use_channel(0);

	return src;
}

uae_u8 *save_wdc (int *len)
{
	uae_u8 *dstbak,*dst;

	dstbak = dst = malloc(16);

	*len = dst - dstbak;
	return dstbak;
}

void wdc_install(void)
{
	int i;

	if (currprefs.wdc >= 2)
		the_ide.intena2 = GAYLE_INT_IDE;
	else
		the_ide.intena2 = 0;

	for (i=0; i<NUMDRIVES; ++i) {
		use_channel(i);
		InitWDC();
		cleanup_drive(DRIVEDATA(&WDC(i)));
	}
	use_channel(0);
}

static int awu = 0;
void add_wdc_unit (char *str, int ro, int secs, int heads, int reserved, int bs)
{
	if (currprefs.wdc == 0)
		return;

	use_channel(awu);
	setup_drive(DRIVEDATA(&WDC(awu)), str, ro, heads, secs);
	awu = (awu + 1) % NUMDRIVES;
}

