 /*
  * UAE - The Un*x Amiga Emulator
  *
  * VNC interface
  *
  */

#include "sysconfig.h"
#include "sysdeps.h"

#include <rfb/rfb.h>

#include <ctype.h>

#include "config.h"
#include "options.h"
#include "threaddep/thread.h"
#include "uae.h"
#include "memory.h"
#include "xwin.h"
#include "custom.h"
#include "drawing.h"
#include "newcpu.h"
#include "keyboard.h"
#include "keybuf.h"
#include "gui.h"
#include "debug.h"
#include "picasso96.h"

extern int kc_decode(int);

static int width, height;
static rfbScreenInfoPtr chips;
#ifdef PICASSO96
static rfbScreenInfoPtr picasso;
#endif

static int keystate[260];
static void dokey(rfbBool down, rfbKeySym k, rfbClientPtr cl)
{
	int kc;

	kc = kc_decode(k);
	if (kc < 0 || kc >= 260)
		return;

	if (keystate[kc] != down) {
		keystate[kc] = down;
		record_key(kc << 1 | !down);
	}
}

static void doptr(int buttonMask, int x, int y, rfbClientPtr cl)
{
	static int mask[3] = { 1, 2, 4 };
	int i;

	newmousecounters = 0;
	for (i=0; i<3; ++i) {
		if (!buttonstate[i] && (buttonMask & mask[i])) {
			/* buttonpress */
			buttonstate[i] = 1;
		}
		else if (buttonstate[i] && !(buttonMask & mask[i])) {
			/* buttonrelease */
			buttonstate[i] = 0;
		}
	}

	/* motionnotify */
	if (x != lastmx || y != lastmy) {
		lastmx = x;
		lastmy = y;
	}
}

static enum rfbNewClientAction newclient(rfbClientPtr cl)
{
	return RFB_CLIENT_ACCEPT;
	/* return RFB_CLIENT_REFUSE; */
}

/*****************************************************************/

static RETSIGTYPE sigbrkhandler(int foo)
{
	activate_debugger();
}

void setup_brkhandler(void)
{
	struct sigaction sa;
	sa.sa_handler = sigbrkhandler;
	sa.sa_flags = 0;
	sa.sa_flags = SA_RESTART;
	sigemptyset(&sa.sa_mask);
	sigaction(SIGINT, &sa, NULL);
}

void flush_line(int y)
{
	rfbMarkRectAsModified(chips, 0, y, width, 1);
}

void flush_block(int ystart, int ystop)
{
	rfbMarkRectAsModified(chips, 0, ystart, width, ystop+1);
}

void flush_screen(int ystart, int ystop)
{
}

int graphics_setup(void)
{
#if 0
	/* hack to avoid mutex bug in libvncserver */
	rfbProtocolExtension E;
	E.next = NULL;
	rfbRegisterProtocolExtension(&E);
	rfbUnregisterProtocolExtension(&E);
#else
	rfbRegisterTightVNCFileTransferExtension();
#endif

	return 1;
}

int graphics_init(void)
{
	rfbScreenInfoPtr s;
	uint16_t *f;
	static char *pwlist[3];

	rfbLog = rfbErr = write_log;
	rfbLogEnable(1);

	width = currprefs.gfx_width;
	height = currprefs.gfx_height;

	pwlist[0] = currprefs.vnc_password;
	pwlist[1] = currprefs.vnc_viewonly;
	pwlist[2] = NULL;

	f = calloc(height, width*sizeof(uint16_t));
	if (f == NULL)
		return 0;

	s = rfbGetScreen(NULL, 0, width, height, 5, 3, sizeof(uint16_t));
	s->serverFormat.trueColour = TRUE;
	s->frameBuffer = (char *)f;
	s->kbdAddEvent = dokey;
	s->ptrAddEvent = doptr;
	s->newClientHook = newclient;
	s->alwaysShared = TRUE;
	s->authPasswdData = pwlist;
	s->passwordCheck = rfbCheckPasswordByList;
	s->autoPort = TRUE;

	rfbInitServer(s);
	chips = s;

	memset(keystate, 0, sizeof(keystate));

	gfxvidinfo.bufmem = (char *)f;
	gfxvidinfo.linemem = NULL;
	gfxvidinfo.emergmem = NULL;
	gfxvidinfo.rowbytes = width*sizeof(uint16_t);
	gfxvidinfo.pixbytes = sizeof(uint16_t);
	gfxvidinfo.width = width;
	gfxvidinfo.height = height;
	gfxvidinfo.maxblocklines = 256;
	gfxvidinfo.can_double = 0;

	alloc_colors64k(5,5,5,0,5,10);

	{
		int i;
		for (i=0; i<256; ++i)
			xcolors[i] = 0x0101 * xcolors[i];
		gfxvidinfo.can_double = 1;
	}

#ifdef PICASSO96
	if (currprefs.gfxmem_size > 0) {
		s = rfbGetScreen(NULL, 0, width, height, 5, 3, sizeof(uint16_t));
		s->serverFormat.trueColour = TRUE;
		s->frameBuffer = (char *)f;
		s->kbdAddEvent = dokey;
		s->ptrAddEvent = doptr;
		s->newClientHook = newclient;
		s->alwaysShared = TRUE;
		s->authPasswdData = pwlist;
		s->passwordCheck = rfbCheckPasswordByList;
		s->autoPort = TRUE;

		rfbInitServer(s);
		picasso = s;

		picasso_vidinfo.rgbformat = RGBFB_B8G8R8A8;
		picasso_vidinfo.rowbytes = width * sizeof(uint32_t);
		picasso_vidinfo.extra_mem = 0;
	}
#endif

	return 1;
}

void graphics_leave (void)
{
	free(chips->frameBuffer);
	rfbScreenCleanup(chips);
	chips = NULL;
#ifdef PICASSO96
	if (picasso)
		rfbScreenCleanup(picasso);
	picasso = NULL;
#endif

	rfbLogEnable(0);
}

void handle_events (void)
{
	gui_handle_events();

	rfbProcessEvents(chips,0);
#ifdef PICASSO96
	if (picasso)
		rfbProcessEvents(picasso,0);
#endif

	/* XXX why here? */
	if ((keystate[AK_CTRL] || keystate[AK_RCTRL])
	    && keystate[AK_LAMI] && keystate[AK_RAMI])
		uae_reset();
}

int check_prefs_changed_gfx (void)
{
	return 0;
}

int debuggable (void)
{
	return 1;
}

int needmousehack (void)
{
	return 0;
}

void LED (int on)
{
}

int lockscr (void)
{
    return 1;
}

void unlockscr (void)
{
}

void target_save_options (FILE *f, struct uae_prefs *p)
{
}

int target_parse_option (struct uae_prefs *p, char *option, char *value)
{
	return 0;
}

/*****************************************************************/

#ifdef PICASSO96
uae_u8 *gfx_lock_picasso(void)
{
	return NULL;
}

void gfx_unlock_picasso()
{
}

void gfx_set_picasso_state(int on)
{
	write_log("picasso: %d\n",on);
}

void gfx_set_picasso_modeinfo(int w, int h, int depth, int rgbfmt)
{
	picasso_vidinfo.width = w;
	picasso_vidinfo.height = h;
	picasso_vidinfo.depth = depth;

	if ((1<<rgbfmt) & RGBMASK_8BIT)
		picasso_vidinfo.pixbytes = 1;
	else if ((1<<rgbfmt) & (RGBMASK_15BIT|RGBMASK_16BIT))
		picasso_vidinfo.pixbytes = 2;
	else if ((1<<rgbfmt) & RGBMASK_24BIT)
		picasso_vidinfo.pixbytes = 3;
	else
		picasso_vidinfo.pixbytes = 4;
}

void DX_SetPalette(int start, int count)
{
	write_log("DX_SetPalette(%d,%d)\n",start,count);
}

int DX_BitsPerCannon (void)
{
	return 8;
}

void DX_Invalidate(int first, int last)
{
	write_log("DX_Invalidate\n");
}

int DX_FillResolutions (uae_u16 *ppixel_format)
{
	static int xy[] = {
		320,200, 320,240, 320,256, 320,400,
		640,350, 640,480, 640,512, 800,600,
		1024,768, 1152,864, 1280,1024
	};
	static int z[] = {
		8,15,16,24,32
	};
#define ac(x) ((int)(sizeof(x)/sizeof(x[0])))
	int i,k,n;

	*ppixel_format = RGBFF_B8G8R8A8;

	n = 0;
	for (i=0; i<ac(xy); i+=2) {
		for (k=0; k<ac(z); ++k) {
			DisplayModes[n].res.width = xy[i];
			DisplayModes[n].res.height = xy[i+1];
			DisplayModes[n].depth = z[k];
			DisplayModes[n].refresh = 75;
			++n;
		}
	}

	return n;
}

#endif /* PICASSO96 */
