 /*
  * UAE - The Un*x Amiga Emulator
  *
  *  Serial Line Emulation
  *
  * (c) 1996, 1997 Stefan Reinauer <stepan@linux.de>
  * (c) 1997 Christian Schmitt <schmitt@freiburg.linux.de>
  *
  */

#include "sysconfig.h"
#include "sysdeps.h"

#include "config.h"
#include "options.h"
#include "uae.h"
#include "memory.h"
#include "custom.h"
#include "cia.h"
#include "newcpu.h"
#include "cia.h"
#include "console.h"

#include <termios.h>
#include <unistd.h>
#include <sys/ioctl.h>

#ifndef O_NONBLOCK
#define O_NONBLOCK O_NDELAY
#endif

#define MODEMTEST   1 /* 0 or 1 */

void serial_open (void);
void serial_close (void);
void serial_init (void);
void serial_exit (void);

void serial_dtr_on (void);
void serial_dtr_off (void);

void serial_flush_buffer (void);
static int serial_read (char *buffer);

int serial_readstatus (void);
uae_u16 serial_writestatus (int, int);

uae_u16 SERDATR (void);

int  SERDATS (void);
void  SERPER (uae_u16 w);
void  SERDAT (uae_u16 w);

static char inbuf[1024], outbuf[1024];
static size_t inptr, inlast, outlast;

int doreadser=0;
int serstat=-1;
static int carrier=0;
static int serdev=0;
static int dsr=0;
static int dtr=0;
static int isbaeh=0;

static int sdin = -1, sdout = -1;

static struct termios tios0, tios;

uae_u16 serper=0, serdat = 0;

void SERPER (uae_u16 w)
{
	int baud=0, pspeed;

	if (!currprefs.use_serial)
		return;

	if (serper == w)  /* don't set baudrate if it's already ok */
		return;
	serper=w;

	if (w&0x8000)
		write_log ("SERPER: 9bit transmission not implemented.\n");

	/* These values should be calculated by the current
	 * color clock value (NTSC/PAL). But this solution is
	 * easy and it works.
	 */
	switch (w & 0x7fff) {
	case 0x2e9b:
	case 0x2e14: baud=300; pspeed=B300; break;
	case 0x170a:
	case 0x0b85: baud=1200; pspeed=B1200; break;
	case 0x05c2:
	case 0x05b9: baud=2400; pspeed=B2400; break;
	case 0x02e9:
	case 0x02e1: baud=4800; pspeed=B4800; break;
	case 0x0174:
	case 0x0173:
	case 0x0170: baud=9600; pspeed=B9600; break;
	case 0x00b9:
	case 0x00b8: baud=19200; pspeed=B19200; break;
	case 0x005c:
	case 0x005d: baud=38400; pspeed=B38400; break;
	case 0x003d: baud=57600; pspeed=B57600; break;
	case 0x001e: baud=115200; pspeed=B115200; break;
	case 0x000f: baud=230400; pspeed=B230400; break;
	default:
		write_log ("SERPER: unsupported baudrate (0x%04x) %d\n",
			w&0x7fff,
			(unsigned int)(3579546.471/(double)((w&0x7fff)+1)));
		return;
	}

	if (serdev != 1)
		return;

	/* set serial input speed */
	if (cfsetispeed (&tios, pspeed) < 0) {
		write_log ("SERPER: CFSETISPEED (%d bps) failed\n", baud);
		return;
	}
	/* set serial output speed */
	if (cfsetospeed (&tios, pspeed) < 0) {
		write_log ("SERPER: CFSETOSPEED (%d bps) failed\n", baud);
		return;
	}

	if (tcsetattr (sdin, TCSADRAIN, &tios) < 0) {
		write_log ("SERPER: TCSETATTR failed\n");
		return;
	}
}

/* Not (fully) implemented yet:
 *
 *  -  Something's wrong with the Interrupts.
 *     (NComm works, TERM does not. TERM switches to a
 *     blind mode after a connect and wait's for the end
 *     of an asynchronous read before switching blind
 *     mode off again. It never gets there on UAE :-< )
 *
 *  -  RTS/CTS handshake, this is not really neccessary,
 *     because you can use RTS/CTS "outside" without
 *     passing it through to the emulated Amiga
 *
 *  -  ADCON-Register ($9e write, $10 read) Bit 11 (UARTBRK)
 *     (see "Amiga Intern", pg 246)
 */

void SERDAT (uae_u16 w)
{
	unsigned char z;

	z = w & 0xff;

	if (currprefs.serial_demand && !dtr) {
		if (!isbaeh) {
			write_log("SERDAT: Baeh.. Your software needs SERIAL_ALWAYS to work properly.\n");
			isbaeh=1;
		}
		return;
	}

	outbuf[outlast++] = z;
	if (outlast == sizeof outbuf)
		serial_flush_buffer();

	serdat |= 0x2000; /* Set TBE in the SERDATR ... */
	intreq |= 1;      /* ... and in INTREQ register */
}

static int serpoll = 0;

static
int serial_poll(void)
{
	unsigned char z;
	int rc;

	if (serdat & 0x4000) {
		intreq |= 0x0800; /* signal RBF */
		return 1;
	}

	rc = serial_read ((char *)&z);
	if (rc == 0)
		return 0;

	serdat &= ~0x4000;

	serdat = (serdat & ~0x8000) | (serdat & 0x4000) << 1; /* set OVRUN */
	serdat |= 0x4000; /* set RBF status */

	if (rc == 1)
		serdat |= 0x0300; /* both stop bits */
	else
		serdat &= ~0x0300; /* break condition */

	serdat = (serdat & 0xff00) | z;

	intreq |= 0x0800; /* signal RBF */
	return 1;
}

uae_u16 SERDATR (void)
{
	if (outlast > 0)
		serial_flush_buffer();

	if (serdat & 0x4000) {
		serpoll = 0;
	} else {
		/* try to idle the CPU when detecting
		 * a busy-waiting console read loop */
		serpoll++;
		if (serpoll >= 50) {
			set_special(SPCFLAG_SLEEP);
			serpoll = 0;
		}
	}

	(void) serial_poll();

	return serdat;
}

/*
 * polled each hsync until no char remains
 * reactivated on vsync
 */
int SERDATS (void)
{
	unsigned char z;

	/* this is no longer the console read loop */
	serpoll = 0;

	if (serdev == 0)
		return 0;

	if (outlast > 0)
		serial_flush_buffer();

	return serial_poll();
}

void serial_dtr_on(void)
{
	dtr=1;
	if (currprefs.serial_demand)
		serial_open();
}

void serial_dtr_off(void)
{
	dtr=0;
	if (currprefs.serial_demand)
		serial_close();
}

static void serial_input (void)
{
	int rc;

	/* reset buffer when there are no unread characters */
	if (inptr != 0 && inptr >= inlast)
		inptr = inlast = 0;

	/* fill buffer if there is space */
	if (inlast < sizeof inbuf) {
		if (serdev == 2)
			rc = aux_read(inbuf+inlast, sizeof inbuf-inlast);
		else
			rc = read(sdin, inbuf+inlast, sizeof inbuf-inlast);
		if (rc > 0)
			inlast += rc;
	}

	if (inlast - inptr >= 2 &&
	    inbuf[inlast-2] == '\033' &&
	    inbuf[inlast-1] == '(') {
		activate_debugger();

		/* kill input buffer */
		inptr = inlast = 0;
	}
}

static int serial_read (char *buffer)
{
	static time_t tout = 0;
	time_t now;

	if (serdev == 0)
		return 0;

	serial_input();

	/* if buffer contains unread characters */
	if (inptr < inlast) {
		/* does it start with ESC ? */
		if (inbuf[inptr] == '\033') {
			/* is there another character buffered? */
			if (inptr+1 < inlast) {
				if (inbuf[inptr+1] == 'B') {
					/* signal a break condition */
					*buffer = '\0';
					inptr += 2;
					return 2;
				}
			} else {
				time(&now);
				/* initialize timeout */
				if (tout == 0)
					tout = now + 3;
				/* timeout not expired ? */
				if (now <= tout)
					return 0;
				/* rearm timeout */
				tout = 0;
			}
		}

		/* read next character from buffer */
		*buffer = inbuf[inptr];
		inptr++;
		return 1;
	}

	return 0;
}

void serial_flush_buffer(void)
{
	int rc;

	if (serdev > 0) {
		if (serdev == 2) {
			rc = aux_write(outbuf, outlast);
			outlast = 0;
		} else {
			rc = write(sdout, outbuf, outlast);
		}
	} else {
		outlast = 0;
		inptr = 0;
		inlast = 0;
	}
}

int serial_readstatus(void)
{
	int status = 0;
	int bit;

	switch (serdev) {
	case 1:
		(void) ioctl (sdin, TIOCMGET, &status);

		bit = (status & TIOCM_CAR) != 0;
		if (carrier != bit) {
			carrier = bit;
			CIA_cd(carrier);
		}
		bit = (status & TIOCM_DSR) != 0;
		if (dsr != bit) {
			dsr = bit;
			CIA_dsr(dsr);
		}
		break;
	case 2:
		if (carrier == 0) {
			carrier = 1;
			CIA_cd(carrier);
		}
		if (dsr == 0) {
			dsr = 1;
			CIA_dsr(dsr);
		}
		status = 1;
		break;
	}

	return status;
}

uae_u16 serial_writestatus (int old, int nw)
{
	if ((old ^ nw) & 0x80) {
		if (nw & 0x80)
			serial_dtr_off();
		else
			serial_dtr_on();
	}

	if (serdev != 2) {
	if ((old ^ nw) & 0x40)
		write_log ("RTS %s.\n", (nw & 0x40) ? "set" : "cleared");
	if ((old ^ nw) & 0x10)
		write_log ("RTS %s.\n", (nw & 0x10) ? "set" : "cleared");
	}

	return nw;
}

void serial_open(void)
{
	int newserdev;

	if (serdev > 0)
		return;

	if (currprefs.sername[0] == '\0') {
		sdin = -1;
		sdout = -1;
		newserdev = 2;
		write_log ("Serial: Using stdio\n");
		aux_use();
	} else {
		sdin = open(currprefs.sername, O_RDWR|O_NONBLOCK|O_BINARY, 0);
		sdout = sdin;
		if (sdin == -1) {
			write_log ("Error: Could not open Device %s\n", currprefs.sername);
			return;
		}
		newserdev = 1;
		write_log ("Serial: Using %s\n",currprefs.sername);

		/* Initialize Serial tty */
		tcgetattr(sdin, &tios0);
		tios = tios0;

		cfmakeraw (&tios);
#ifndef MODEMTEST
		tios.c_cflag &= ~CRTSCTS; /* Disable RTS/CTS */
#else
		tios.c_cflag |= CRTSCTS; /* Enabled for testing modems */
#endif
	}

	serdev = newserdev;
}

void serial_close (void)
{
	if (serdev == 0)
		return;

	if (serdev == 1)
		close(sdin);

	sdin = -1;
	sdout = -1;
	serdev = 0;
}

void serial_init (void)
{
	serdat = 0x2000;

	if (!currprefs.use_serial)
		return;

	if (!currprefs.serial_demand)
		serial_open ();
}

void serial_exit (void)
{
	serial_close();
	dtr = 0;
}

void serial_tick (void)
{
	(void) serial_input();
}
