#include "sysconfig.h"
#include "sysdeps.h"

#include "config.h"
#include "options.h"
#include "events.h"
#include "uae.h"
#include "memory.h"
#include "custom.h"
#include "newcpu.h"
#include "autoconf.h"
#include "ersatz.h"
#include "debug.h"
#include "compiler.h"
#include "gui.h"
#include "savestate.h"

#define DBG_MMU_VERBOSE	0
#define DBG_MMU_LOGGING 0
#define DBG_MMU_SANITY	0

static struct mmu_atc_line atc[64];
static int atc_rand = 0;
static int atc_last_hit = -1;
static int mmu_exception = 0;

int mmu_berr = 0; /* disable bus errors for debugger */

struct xlt {
	uae_u32 page_frame;
	uaecptr paddr;
	unsigned write;
	addrbank *ab;
} __attribute ((aligned(32)));

#define PF_INVALID 1

#define XLT_LINES 4096 /* direct mapped, 16MB */

static struct xlt xlt_inst[2];
static struct xlt xlt_data[2][XLT_LINES];
static unsigned xlt_lines[2] = { XLT_LINES, XLT_LINES };

static struct xlt *xlt_inst_ptr = &xlt_inst[0];
static struct xlt *xlt_data_ptr = &xlt_data[0][0];
static unsigned *xlt_lines_ptr = &xlt_lines[0];


/***************************************************************************/

static void mmu_dump_ttr(const char * label, uae_u32 ttr)
{
	uae_u32 from_addr, to_addr;

	from_addr = ttr & MMU_TTR_LOGICAL_BASE;
	to_addr = (ttr & MMU_TTR_LOGICAL_MASK) << 8;
	
	write_log("%s: [%08lx] %08lx - %08lx enabled=%d supervisor=%d wp=%d cm=%02d\n",
			label, ttr,
			from_addr, to_addr,
			ttr & MMU_TTR_BIT_ENABLED ? 1 : 0,
			(ttr & (MMU_TTR_BIT_SFIELD_ENABLED | MMU_TTR_BIT_SFIELD_SUPER)) >> MMU_TTR_SFIELD_SHIFT,
			ttr & MMU_TTR_BIT_WRITE_PROTECT ? 1 : 0,
			(ttr & MMU_TTR_CACHE_MASK) >> MMU_TTR_CACHE_SHIFT
		  );
}

static void mmu_dump_table(const char * label, uaecptr root_ptr)
{
	const int ROOT_TABLE_SIZE = 128,
		PTR_TABLE_SIZE = 128,
		PAGE_TABLE_SIZE = 64,
		ROOT_INDEX_SHIFT = 25,
		PTR_INDEX_SHIFT = 18,
		PAGE_INDEX_SHIFT = 12;
	int root_idx, ptr_idx, page_idx;
	uae_u32 root_des, ptr_des, page_des;
	uaecptr ptr_des_addr, page_addr,
		root_log, ptr_log, page_log;
		
	write_log("%s: root=%lx\n", label, root_ptr);
	
	for (root_idx = 0; root_idx < ROOT_TABLE_SIZE; root_idx++)	{
		root_des = phys_get_long(root_ptr + root_idx);

		if ((root_des & 2) == 0)
			continue;	/* invalid */
		
		write_log("ROOT: %03d U=%d W=%d UDT=%02d\n", root_idx,
				root_des & 8 ? 1 : 0,
				root_des & 4 ? 1 : 0,
				root_des & 3
			  );

		root_log = root_idx << 25;
		
		ptr_des_addr = root_des & MMU_ROOT_PTR_ADDR_MASK;
		
		for (ptr_idx = 0; ptr_idx < PTR_TABLE_SIZE; ptr_idx++)	{
			struct {
				uaecptr	log, phys;
				int start_idx, n_pages;	/* number of pages covered by this entry */
				uae_u32 match;
			} page_info[PAGE_TABLE_SIZE];
			int n_pages_used;

			ptr_des = phys_get_long(ptr_des_addr + ptr_idx * 4);
			ptr_log = root_log | (ptr_idx << 18);

			if ((ptr_des & 2) == 0)
				continue; /* invalid */

			page_addr = ptr_des & regs.ptr_page_mask;

			n_pages_used = -1;
			for (page_idx = 0; page_idx < (regs.pgi_shift == 12 ? 64 : 32); page_idx++)	{
				
				page_des = phys_get_long(page_addr + page_idx * 4);
				page_log = ptr_log | (page_idx << regs.pgi_shift);

				switch (page_des & 3)	{
					case 0: /* invalid */
						continue;
					case 1: case 3: /* resident */
					case 2: /* indirect */
						if (n_pages_used == -1 || page_info[n_pages_used].match != page_des)	{
							/* use the next entry */
							n_pages_used++;

							page_info[n_pages_used].match = page_des;
							page_info[n_pages_used].n_pages = 1;
							page_info[n_pages_used].start_idx = page_idx;
							page_info[n_pages_used].log = page_log;
						}
						else	{
							page_info[n_pages_used].n_pages++;
						}
						break;
				}
			}

			if (n_pages_used == -1)
				continue;

			write_log(" PTR: %03d U=%d W=%d UDT=%02d\n", ptr_idx,
				ptr_des & 8 ? 1 : 0,
				ptr_des & 4 ? 1 : 0,
				ptr_des & 3
			  );


			for (page_idx = 0; page_idx <= n_pages_used; page_idx++)	{
				page_des = page_info[page_idx].match;

				if ((page_des & MMU_PDT_MASK) == 2)	{
					write_log("  PAGE: %03d-%03d log=%08lx INDIRECT --> addr=%08lx\n",
							page_info[page_idx].start_idx,
							page_info[page_idx].start_idx + page_info[page_idx].n_pages - 1,
							page_info[page_idx].log,
							page_des & MMU_PAGE_INDIRECT_MASK
						  );

				}
				else	{
					write_log("  PAGE: %03d-%03d log=%08lx addr=%08lx UR=%02d G=%d U1/0=%d S=%d CM=%d M=%d U=%d W=%d\n",
							page_info[page_idx].start_idx,
							page_info[page_idx].start_idx + page_info[page_idx].n_pages - 1,
							page_info[page_idx].log,
							page_des & regs.page_addr_mask,
							(page_des & regs.page_ur_mask) >> MMU_PAGE_UR_SHIFT,
							page_des & MMU_DES_GLOBAL ? 1 : 0,
							(page_des & MMU_TTR_UX_MASK) >> MMU_TTR_UX_SHIFT,
							page_des & MMU_DES_SUPER ? 1 : 0,
							(page_des & MMU_TTR_CACHE_MASK) >> MMU_TTR_CACHE_SHIFT,
							page_des & MMU_DES_MODIFIED ? 1 : 0,
							page_des & MMU_DES_USED ? 1 : 0,
							page_des & MMU_DES_WP ? 1 : 0
						  );
				}
			}
		}
		
	}
}

void mmu_dump_atc(void)
{
	int i;
	for (i = 0; i < 64; i++)	{
		if (!atc[i].v)
			continue;
		write_log("ATC[%02d] G=%d S=%d M=%d W=%d R=%d FC2=%d log=%08x --> phys=%08x\n",
				i,
				atc[i].g,
				atc[i].s,
				atc[i].m,
				atc[i].w,
				atc[i].r,
				atc[i].fc2,
				atc[i].log,
				atc[i].phys
				);
	}
}

void mmu_dump_tables(void)
{
	if (currprefs.cpu_level != 4)	{
		write_log("This CPU has no MMU hardware\n");
		return;
	}
	write_log("URP: %08x   SRP: %08x  MMUSR: %x  TC: %x\n", regs.urp, regs.srp, regs.mmusr, regs.tc);
	mmu_dump_ttr("DTT0", regs.dtt0);	
	mmu_dump_ttr("DTT1", regs.dtt1);	
	mmu_dump_ttr("ITT0", regs.itt0);	
	mmu_dump_ttr("ITT1", regs.itt1);	
	mmu_dump_atc();
	mmu_dump_table("SRP", regs.srp);
	mmu_dump_table("URP", regs.urp);
}

static void phys_dump_mem (uaecptr addr, int lines)
{
	for (;lines--;) {
		int i;
		write_log ("%08lx ", addr);
		for (i = 0; i < 16; i++) {
			write_log ("%04x ", phys_get_word(addr)); addr += 2;
		}
		write_log ("\n");
	}
}

/***************************************************************************/

/* set translation cache pointers depending on supervisor mode */
static
void mmu_set_xlt(unsigned super)
{
	xlt_inst_ptr = &xlt_inst[super];
	xlt_data_ptr = &xlt_data[super][0];
	xlt_lines_ptr = &xlt_lines[super];
}

/* check if an address matches a ttr */
static inline REGPARAM2
mmu_rc_t mmu_match_ttr(uae_u32 ttr, uae_u8 msb, int write)
{
	uae_u8 match, mask;

	if ((ttr & MMU_TTR_BIT_ENABLED) == 0)
		return NO_MATCH;

	match = (ttr & MMU_TTR_LOGICAL_BASE) >> 24;
	mask = (ttr & MMU_TTR_LOGICAL_MASK) >> 16;

	if ((msb & ~mask) != match)
		return NO_MATCH;

	if ((ttr & MMU_TTR_BIT_SFIELD_ENABLED) == 0)	{
		if ((ttr & MMU_TTR_BIT_SFIELD_SUPER) && !regs.s)
			return NO_MATCH;
		if ((ttr & MMU_TTR_BIT_SFIELD_SUPER) == 0 && regs.s)
			return NO_MATCH;
	}

	regs.mmusr_test = MMU_MMUSR_T | MMU_MMUSR_R;

	if ((ttr & MMU_TTR_BIT_WRITE_PROTECT) && write)
		return NO_WRITE;

	return OK_MATCH;
}

static inline REGPARAM2
mmu_rc_t mmu_tlb_translate(uae_u32 page_frame, uae_u32 offset,
			unsigned super, unsigned write, uaecptr *paddrp,
			int *atc_indexp)
{
	int i, atc_sel, atc_index, wp;
	uaecptr atc_hit_addr;

	atc_sel = *atc_indexp;
	for (i = 0; i<4; i++)	{
		atc_index = atc_sel * 4 + i;
		if (atc[atc_index].log == page_frame
		    && atc[atc_index].fc2 == super
		    && atc[atc_index].v)
			break;
	}
	if (i == 4) {
#if 1
		/* search invalid atc slot */
		for (i = 0; i < 4; i++)	{
			if (!atc[atc_sel * 4 + i].v) {
				atc_index = atc_sel * 4 + i;
				break;
			}
		}
		/* or else random choice */
		if (atc_index == -1)
			atc_index = atc_sel * 4 + atc_rand;
#else
		atc_index = atc_sel * 4;
		if (atc[atc_index].v)
			atc_index += atc_rand;
#endif

		*atc_indexp = atc_index;
		return NO_MATCH;
	}

	*atc_indexp = atc_index;

	if (!atc[atc_index].r)
		return NO_READ;

	wp = atc[atc_index].w;

	atc_hit_addr = atc[atc_index].phys | offset;
	regs.mmusr_test |= atc[atc_index].g ? MMU_MMUSR_G : 0;
	regs.mmusr_test |= atc[atc_index].s ? MMU_MMUSR_S : 0;
	regs.mmusr_test |= atc[atc_index].m ? MMU_MMUSR_M : 0;
	regs.mmusr_test |= atc[atc_index].w ? MMU_MMUSR_W : 0;
	regs.mmusr_test |= atc[atc_index].r ? MMU_MMUSR_R : 0;
	regs.mmusr_test |= atc[atc_index].phys & MMU_MMUSR_ADDR_MASK;

	if (atc[atc_index].s && !super)
		return NO_SUPER;

	if (write) {
		if (wp)
			return NO_WRITE;
		if (!atc[atc_index].w)
			return OK_UPDATE;
	}

	*paddrp = atc_hit_addr;
	return OK_MATCH;
}

static inline REGPARAM2
mmu_rc_t mmu_table_search(uaecptr theaddr,
			uae_u32 page_frame, uae_u32 page_offset,
			unsigned super, unsigned write,
			uaecptr *paddrp, int atc_index)
{
	uae_u32 root_ptr, root_des_addr, root_des;
	uae_u32 ptr_des_addr, ptr_des;
	uae_u32 page_des_addr, page_des;
	uae_u32 ri, pi, pgi;
	uae_u32 wp;
	int i;

	ri  = (theaddr & 0xfe000000) >> 25;
	pi  = (theaddr & 0x01fc0000) >> 18;
	pgi = (theaddr & regs.pgi_mask) >> regs.pgi_shift;

	/* TWE: flag as being in table search */
	regs.mmu_fslw |= FSLW_TWE;

	/* root descriptor */
	root_ptr = super ? regs.srp : regs.urp;
	root_des_addr = (root_ptr & MMU_ROOT_PTR_ADDR_MASK) | (ri << 2);
	root_des = phys_get_long(root_des_addr);

	switch (root_des & MMU_UDT_MASK) {
	case 0x0:
	case 0x1:
		regs.mmu_fslw |= FSLW_PTA;
		goto non_resident_atc;
	}
	
	wp = root_des & MMU_DES_WP;
	if (!wp && (root_des & MMU_DES_USED) == 0) {
		root_des |= MMU_DES_USED;
		phys_put_long(root_des_addr, root_des);
	}

	ptr_des_addr = (root_des & MMU_ROOT_PTR_ADDR_MASK) | (pi << 2);
	ptr_des = phys_get_long(ptr_des_addr);
	
	switch (ptr_des & MMU_UDT_MASK) {
	case 0x0:
	case 0x1:
		regs.mmu_fslw |= FSLW_PTB;
		goto non_resident_atc;
	}

	wp |= ptr_des & MMU_DES_WP;
	if (!wp && (ptr_des & MMU_DES_USED) == 0) {
		ptr_des |= MMU_DES_USED;
		phys_put_long(ptr_des_addr, ptr_des);
	}

	page_des_addr = (ptr_des & regs.ptr_page_mask) | (pgi << 2);
	
	do {
		page_des = phys_get_long(page_des_addr);
		i = page_des & MMU_PDT_MASK;
		if (i == 0) {
			regs.mmu_fslw |= FSLW_PF;
			goto non_resident_atc;
		} else if (i == 2) {
			/* indirect */
			if (regs.mmu_fslw & FSLW_IL)
				goto non_resident_atc;
			page_des_addr = page_des & MMU_PAGE_INDIRECT_MASK;

			/* in case a fault occurs later, tag it as indirect */
			regs.mmu_fslw |= FSLW_IL;
		}
	} while (i != 1 && i != 3);

	wp |= page_des & MMU_DES_WP;
	if (!wp) {
		int modify = 0;
		if ((page_des & MMU_DES_USED) == 0) {
			page_des |= MMU_DES_USED;
			modify = 1;
		}
		/* set the modified bit */
		if (write && (page_des & MMU_DES_MODIFIED) == 0) {
			page_des |= MMU_DES_MODIFIED;
			modify = 1;
		}
		if (modify)
			phys_put_long(page_des_addr, page_des);
	}

	atc[atc_index].log = page_frame;
	atc[atc_index].v = 1;
	atc[atc_index].g = (page_des & MMU_DES_GLOBAL) ? 1 : 0;
	atc[atc_index].s = (page_des & MMU_DES_SUPER) ? 1 : 0;
	atc[atc_index].m = (page_des & MMU_DES_MODIFIED) ? 1 : 0;
	atc[atc_index].w = wp ? 1 : 0;
	atc[atc_index].r = 1;
	atc[atc_index].fc2 = super;
	atc[atc_index].phys = page_des & regs.page_addr_mask;

	if (!super && atc[atc_index].s) {
		regs.mmu_fslw |= FSLW_SP;
		return NO_SUPER;
	}
	if (wp && write) {
		regs.mmu_fslw |= FSLW_WP;
		return NO_WRITE;
	}

	*paddrp = atc[atc_index].phys | page_offset;

	return OK_MATCH;

non_resident_atc:

	atc[atc_index].log = page_frame;
	atc[atc_index].phys = 0;
	atc[atc_index].v = 0;
	atc[atc_index].r = 0;

	return NO_MATCH;
}

/***************************************************************************/

static void xlt_flush_all(void)
{
	int i, n;

	/* set page frame value that never matches */

	n = xlt_lines[0];
	for (i=0; i<n; ++i)
		xlt_data[0][i].page_frame = PF_INVALID;
	n = xlt_lines[1];
	for (i=0; i<n; ++i)
		xlt_data[1][i].page_frame = PF_INVALID;

	xlt_inst[0].page_frame = PF_INVALID;
	xlt_inst[1].page_frame = PF_INVALID;

	xlt_lines[0] = 0;
	xlt_lines[1] = 0;
}

static void xlt_flush_addr(uae_u32 page_frame, unsigned super)
{
	unsigned fx;

	fx = (page_frame >> 12) & (XLT_LINES - 1);

	if (xlt_data[super][fx].page_frame == page_frame)
		xlt_data[super][fx].page_frame = PF_INVALID;
}

void mmu_init(void)
{
	xlt_flush_all();
	regs.mmu_enabled = 0;
}

static inline REGPARAM2
mmu_rc_t mmu_translate_test_inst (
	uaecptr theaddr, unsigned super, uaecptr *paddrp, addrbank **abp)
{
	mmu_rc_t rc;
	uae_u32 ri, pi, pgi, page_frame, page_offset;
	int atc_index;
	struct xlt *xlt;
	uae_u8 msb;
	uaecptr paddr;

	page_frame  = theaddr & regs.frame_mask;
	page_offset = theaddr & regs.offset_mask;

	/* one level caching */
	xlt = xlt_inst_ptr;
	if (page_frame == xlt->page_frame) {
		*paddrp = xlt->paddr | page_offset;
		*abp    = xlt->ab;
		return OK_MATCH;
	}

	/* randomize */
	atc_rand = (atc_rand + 1) % 4;

	*paddrp = theaddr;

	/* TTRs work independent of MMU enable */
	msb = (theaddr & MMU_TTR_LOGICAL_BASE) >> 24;
	rc = mmu_match_ttr(regs.itt0, msb, 0);
	if (rc == NO_MATCH)
		rc = mmu_match_ttr(regs.itt1, msb, 0);
	if (rc != NO_MATCH)
		goto hit;

	if (!regs.mmu_enabled) {
		*abp = &get_mem_bank(theaddr);
		return OK_MATCH;
	}

	regs.mmusr_test = 0;

	/* First check the TLB */
	atc_index   = (theaddr & regs.sel_mask) >> regs.sel_shift;
	rc = mmu_tlb_translate(page_frame, page_offset, super, 0,
					paddrp, &atc_index);
	if (rc != NO_MATCH && rc != OK_UPDATE)
		goto hit;

	/* Otherwise consult the page tables */
	rc = mmu_table_search(theaddr, page_frame, page_offset, super, 0,
					paddrp, atc_index);
	if (rc != NO_MATCH)
		goto hit;

	/* record possible page fault in mmusr register */
	*abp = NULL;
	regs.mmusr_test = MMU_MMUSR_B;
	return rc;

hit:
	paddr = *paddrp & regs.frame_mask;
	*abp = &get_mem_bank(paddr);
	if (rc == OK_MATCH) {
		xlt->paddr = paddr;
		xlt->ab = *abp;
		xlt->page_frame = page_frame;
		xlt->write = 0;
	}
	return rc;
}

static inline REGPARAM2
mmu_rc_t mmu_translate_test_data (
	uaecptr theaddr, unsigned super, unsigned write, uaecptr *paddrp, addrbank **abp)
{
	mmu_rc_t rc;
	uae_u32 ri, pi, pgi, page_frame, page_offset;
	int atc_index;
	struct xlt *xlt;
	unsigned fx;
	uae_u8 msb;
	uaecptr paddr;

	page_frame  = theaddr & regs.frame_mask;
	page_offset = theaddr & regs.offset_mask;

	/* one level caching */
	fx = (page_frame >> 12) & (XLT_LINES - 1);
	xlt = &xlt_data_ptr[fx];
	if ((!write || xlt->write) && page_frame == xlt->page_frame) {
		*paddrp = xlt->paddr | page_offset;
		*abp    = xlt->ab;
		return OK_MATCH;
	}

	/* randomize */
	atc_rand = (atc_rand + 1) % 4;

	*paddrp = theaddr;

	/* TTRs work independent of MMU enable */
	msb = (theaddr & MMU_TTR_LOGICAL_BASE) >> 24;
	rc = mmu_match_ttr(regs.dtt0, msb, write);
	if (rc == NO_MATCH)
		rc = mmu_match_ttr(regs.dtt1, msb, write);
	if (rc != NO_MATCH)
		goto hit;

	if (!regs.mmu_enabled) {
		*abp = &get_mem_bank(theaddr);
		return OK_MATCH;
	}

	regs.mmusr_test = 0;

	/* First check the TLB */
	atc_index   = (theaddr & regs.sel_mask) >> regs.sel_shift;
	rc = mmu_tlb_translate(page_frame, page_offset, super, write,
					paddrp, &atc_index);
	if (rc != NO_MATCH && rc != OK_UPDATE)
		goto hit;

	/* Otherwise consult the page tables */
	rc = mmu_table_search(theaddr, page_frame, page_offset, super, write,
					paddrp, atc_index);
	if (rc != NO_MATCH)
		goto hit;

	/* record possible page fault in mmusr register */
	*abp = NULL;
	regs.mmusr_test = MMU_MMUSR_B;
	return rc;

hit:
	paddr = *paddrp & regs.frame_mask;
	*abp = &get_mem_bank(paddr);
	if (rc == OK_MATCH) {
		xlt->paddr = paddr;
		xlt->ab = *abp;
		xlt->page_frame = page_frame;
		xlt->write = write;

		if (fx >= *xlt_lines_ptr)
			*xlt_lines_ptr = fx+1;
	}
	return rc;
}

static
void mmu_bus_error(uaecptr theaddr, unsigned super, unsigned datamode, unsigned write, unsigned size)
{
	uae_u8 fc = (super ? 4 : 0) | (datamode ? 1 : 2);

	if (mmu_exception > 0) {
		write_log("BUS_ERROR IN EXCEPTION fc=%d w=%d log=%08x size=%d\n",fc,write,theaddr,size);
		debug();
		abort();
	}

	if (mmu_berr == 0) {
		write_log("BUS_ERROR IN DEBUGGER fc=%d w=%d log=%08x size=%d\n",fc,write,theaddr,size);
		debug();
		return;
	}

#if 0
	mmu_change_fc();
#endif

	regs.mmu_ssw  = 0;
	regs.mmu_fslw = 0;

	/* SSW */
	regs.mmu_ssw |= SSW4_ATC;
	if (!write)
	 	regs.mmu_ssw |= SSW4_RW;
	regs.mmu_ssw |= (size << SSW4_SZSHFT) & SSW4_SZMASK;
	regs.mmu_ssw |= fc & SSW4_TMMASK;

	/* FSLW */
	regs.mmu_fslw |= write ? FSLW_RW_W : FSLW_RW_R;
	if (fc & 1)
		regs.mmu_fslw |= FSLW_IO;
#if 0
	if (regs.t0)
		regs.mmu_fslw |= FSLW_TT0;
	if (regs.t1)
		regs.mmu_fslw |= FSLW_TT1;
#endif

	/* Special access errors */
	if (regs.movem) {
		/* Access error during movem */
		regs.movem = 0;
		regs.mmu_ssw |= SSW4_CM;
	}
	if (regs.fpp_movem) {
		/* Access error during fpp movem */
		regs.fpp_movem = 0;
		regs.mmu_ssw |= SSW4_CP;
	}
	if (regs.move16) {
		/* Access error during move16 */
		regs.move16 = 0;
		regs.wb2s |= 0x08;
	}
	
	regs.mmu_fault_addr = theaddr;

	/*
	 * a write error is emulated in the bus error handler and the
	 * program continues with the next instruction.
	 * a read error restarts the program with the instruction that
	 * caused the bus error.
	 * The exception is the movem instruction that needs to be
	 * restarted on write errors. The CM causes the movem instruction
	 * to use the saved calculated effective address which the
	 * Exception saved on the stack
	 */
	++mmu_exception;
	if (write && (regs.mmu_ssw & (SSW4_CP|SSW4_CU|SSW4_CT|SSW4_CM)) == 0)
		Exception(2, m68k_getpc());
	else
		Exception(2, regs.restart);
	--mmu_exception;
	longjmp(m68k_exception, 0);

	/* NOTREACHED */
	return;
}

static inline
void mmu_translate_inst(
	uaecptr theaddr, unsigned size,
	uae_u32 *paddrp, addrbank **abp)
{
	mmu_rc_t rc;

	rc = mmu_translate_test_inst(theaddr, regs.s, paddrp, abp);
	if (rc == OK_MATCH)
		return;

	mmu_bus_error(theaddr, regs.s, 0, 0, size);

	/* NOTREACHED */
	*abp = &dummy_bank;
	*paddrp = 0;
	return;
}

static inline
void mmu_translate_data(
	uaecptr theaddr, unsigned write, unsigned size,
	uae_u32 *paddrp, addrbank **abp)
{
	mmu_rc_t rc;

	rc = mmu_translate_test_data(theaddr, regs.s, write, paddrp, abp);
	if (rc == OK_MATCH)
		return;

	mmu_bus_error(theaddr, regs.s, 1, write, size);

	/* NOTREACHED */
	*abp = &dummy_bank;
	*paddrp = 0;
	return;
}

static inline
void mmu_translate_fc(
	uaecptr theaddr, unsigned write, unsigned fc, unsigned size,
	uae_u32 *paddrp, addrbank **abp)
{
	mmu_rc_t rc;
	unsigned s;
	unsigned datamode = fc & 1;
	unsigned super = fc & 4 ? 1 : 0;

	mmu_set_xlt(super);
	if (datamode)
		rc = mmu_translate_test_data(theaddr, super, write, paddrp, abp);
	else
		rc = mmu_translate_test_inst(theaddr, super, paddrp, abp);
	mmu_set_xlt(regs.s);
	if (rc == OK_MATCH)
		return;

	mmu_bus_error(theaddr, super, datamode, write, size);

	/* NOTREACHED */
	*abp = &dummy_bank;
	*paddrp = 0;
	return;
}


/***************************************************************************/

void mmu_set_tc(uae_u16 tc)
{
	regs.mmu_enabled = tc & 0x8000 ? 1 : 0;
	regs.mmu_pagesize = tc & 0x4000 ? MMU_PAGE_8KB : MMU_PAGE_4KB;

	switch (regs.mmu_pagesize) {
	case MMU_PAGE_8KB:
		regs.pgi_mask       = 0x0003e000;
		regs.pgi_shift      = 13;
		regs.sel_mask       = 0x0001e000;
		regs.sel_shift      = 13;
		regs.frame_mask     = 0xffffe000;
		regs.ptr_page_mask  = MMU_PTR_PAGE_ADDR_MASK_8;
		regs.page_addr_mask = MMU_PAGE_ADDR_MASK_8;
		regs.page_ur_mask   = MMU_PAGE_UR_MASK_8;
		break;
	case MMU_PAGE_4KB:
		regs.pgi_mask       = 0x0003f000;
		regs.pgi_shift      = 12;
		regs.sel_mask       = 0x0000f000;
		regs.sel_shift      = 12;
		regs.frame_mask     = 0xfffff000;
		regs.ptr_page_mask  = MMU_PTR_PAGE_ADDR_MASK_4;
		regs.page_addr_mask = MMU_PAGE_ADDR_MASK_4;
		regs.page_ur_mask   = MMU_PAGE_UR_MASK_4;
		break;
	}
	regs.offset_mask = ~regs.frame_mask;

	memset(atc, 0, sizeof(atc));
	xlt_flush_all();
	mmu_change_fc();
	m68k_setpc(m68k_getpc());

	write_log("MMU: enabled=%d page=%d\n", regs.mmu_enabled, regs.mmu_pagesize);
}

void mmu_set_root_pointer(int regno, uae_u32 val)
{
	switch (regno) {
	case 0x806:
		regs.urp = val;
		break;
	case 0x807:
		regs.srp = val;
		break;
	}
}

void mmu_set_ttr(int regno, uae_u32 val)
{
	switch (regno) { 
	case TTR_I0:   
		regs.itt0 = val;
		break;
	case TTR_I1:
		regs.itt1 = val;
		break;
	case TTR_D0:
		regs.dtt0 = val;
		break;
	case TTR_D1:
		regs.dtt1 = val;
		break;
	}       
}


void mmu_make_transparent_region(uaecptr baseaddr, uae_u32 size, int fc)
{
	uae_u32 val;
	int r;

	val = (baseaddr & MMU_TTR_LOGICAL_BASE)
	    | ((baseaddr + size - 1) & MMU_TTR_LOGICAL_BASE) >> 8
	    | MMU_TTR_BIT_ENABLED;

	if (fc & 1) {
		if ((regs.dtt0 & MMU_TTR_BIT_ENABLED) == 0)
			r = TTR_D0;
		else if ((regs.dtt1 & MMU_TTR_BIT_ENABLED) == 0)
			r = TTR_D1;
		else
			return;
	} else {
		if ((regs.itt0 & MMU_TTR_BIT_ENABLED) == 0)
			r = TTR_I0;
		else if ((regs.itt1 & MMU_TTR_BIT_ENABLED) == 0)
			r = TTR_I1;
		else
			return;
	}

	mmu_set_ttr(r, val);

	write_log("MMU: map transparent mapping of %08x\n", val);
}

void mmu_op(uae_u32 opcode, uae_u16 extra)
{
	if ((opcode & 0xFE0) == 0x0500) {
		int i, regno, didflush = 0;
		uaecptr addr, page_frame = 0;
		int atc_sel, atc_index = 0;
		unsigned super = 0;
		int op;

		/* PFLUSH */
		regs.mmusr = 0;

		mmu_change_fc();

		op = (opcode & 24) >> 3;

		switch (op) {
		case 0:
		case 1:
			regno = opcode & 7;
			addr = m68k_areg(regs, regno);
			if (regs.mmu_pagesize == MMU_PAGE_8KB) {
				page_frame = addr & 0xffffe000;
				atc_sel = ((addr & 0x1e000) >> 13) & 0xf;
			} else {
				page_frame = addr & 0xfffff000;
				atc_sel = ((addr & 0xf000) >> 12) & 0xf;
			}
			super = (regs.dfc & 4) ? 1 : 0;
			atc_index = atc_sel * 4;
			xlt_flush_addr(page_frame, super);
			break;
		case 2:
		case 3:
			xlt_flush_all();
			break;
		}

		switch (op) {
		case 0:
			/* PFLUSHN (An) flush page entry if not global */
#if DBG_MMU_LOGGING
			write_log ("PFLUSHN (A%d) %08x DFC=%d\n", regno, m68k_areg(regs, regno), regs.dfc);
#endif
			for (i = atc_index; i<atc_index+4; ++i) {
				if (atc[i].v && !atc[i].g
					&& atc[i].log == page_frame
					&& atc[i].fc2 == super)
				{
					atc[i].v = 0;
					didflush++;
				}
			}
			break;
		case 1:
			/* PFLUSH (An) flush page entry */
#if DBG_MMU_LOGGING
			write_log ("PFLUSH (A%d) %08x DFC=%d\n", regno, m68k_areg(regs, regno), regs.dfc);
#endif
			for (i = atc_index; i<atc_index+4; ++i) {
				if (atc[i].v
					&& atc[i].log == page_frame
					&& atc[i].fc2 == super)
				{
					atc[i].v = 0;
					didflush++;
				}
			}
			break;

		case 2:
			/* PFLUSHAN flush all except global */
#if DBG_MMU_LOGGING
			write_log ("PFLUSHAN\n");
#endif
			for (i = 0; i < 64; i++)	{
				if (atc[i].v && !atc[i].g) {
					atc[i].v = 0;
					didflush++;
				}
			}
			break;

		case 3:
			/* PFLUSHA flush all entries */
#if DBG_MMU_LOGGING
			write_log ("PFLUSHA\n");
#endif
			for (i = 0; i < 64; i++)	{
				if (atc[i].v) {
					atc[i].v = 0;
					didflush++;
				}
			}
			break;
		}

#if DBG_MMU_LOGGING
		if (didflush)
			write_log("  -> flushed %d matching entries\n", didflush);
#endif
		
	} else if ((opcode & 0x0FD8) == 0x548) {

		int write, regno, super;
		uaecptr addr;
		addrbank *ab;

		regno = opcode & 7;
		write = opcode & 32;
		super = regs.dfc & 4 ? 1 : 0;

#if 0
		write_log ("PTEST%c (A%d) %08x DFC=%d\n", write ? 'W' : 'R', regno, m68k_areg(regs, regno), regs.dfc);
#endif

		regs.mmusr = 0;
		if (regs.dfc & 1) {
			(void)mmu_translate_test_data(m68k_areg(regs, regno), super, write, &addr, &ab);
		} else {
			(void)mmu_translate_test_inst(m68k_areg(regs, regno), super, &addr, &ab);
		}
		regs.mmusr = regs.mmusr_test;
	} else
		op_illg (opcode);
}


/*************************************************************************/

static inline uae_u32 ablget(addrbank *ab, uaecptr addr) {
        return call_mem_get_func(ab->lget, addr);
}
static inline uae_u16 abwget(addrbank *ab, uaecptr addr) {
        return call_mem_get_func(ab->wget, addr);
}
static inline uae_u8 abbget(addrbank *ab, uaecptr addr) {
        return call_mem_get_func(ab->bget, addr);
}
static inline void ablput(addrbank *ab, uaecptr addr, uae_u32 l) {
        call_mem_put_func(ab->lput, addr, l);
}  
static inline void abwput(addrbank *ab, uaecptr addr, uae_u16 w) {
        call_mem_put_func(ab->wput, addr, w);
} 
static inline void abbput(addrbank *ab, uaecptr addr, uae_u8 b) {
        call_mem_put_func(ab->bput, addr, b);
} 

/* handle memory access that spans pages */

static
uae_u32 REGPARAM2 mmu_get_long_pages(uaecptr addr, unsigned fc)
{
	uaecptr pa1, pa2;
	addrbank *ab1, *ab2;
	uae_u32 l = 0;

	mmu_translate_fc(addr, 0, fc, sz_long, &pa1, &ab1);
	mmu_translate_fc(addr + 4 - (addr & 3), 0, fc, sz_long, &pa2, &ab2);

	switch (addr & 3) {
	case 1:
		l =
			abbget(ab1, pa1) << 24 |
			abwget(ab1, pa1+1) << 8 |
			abbget(ab2, pa2);
		break;
	case 2:
		l =
			abwget(ab1, pa1) << 16 |
			abwget(ab2, pa2);
		break;
	case 3:
		l =
			abbget(ab1, pa1) << 24 |
			abwget(ab2, pa2) << 8 |
			abbget(ab2, pa2+2);
		break;
	}

	return l;
}

static
uae_u16 REGPARAM2 mmu_get_word_pages(uaecptr addr, unsigned fc)
{
	uaecptr pa1, pa2;
	addrbank *ab1, *ab2;
	uae_u16 w = 0;

	mmu_translate_fc(addr, 0, fc, sz_word, &pa1, &ab1);
	mmu_translate_fc(addr + 1, 0, fc, sz_word, &pa2, &ab2);

	w =
		abbget(ab1, pa1) << 8 |
		abbget(ab2, pa2);

	return w;
}

static
void REGPARAM2 mmu_put_long_pages(uaecptr addr, unsigned fc, uae_u32 l)
{
	uaecptr pa1, pa2;
	addrbank *ab1, *ab2;

	mmu_translate_fc(addr, 1, fc, sz_long, &pa1, &ab1);
	mmu_translate_fc(addr + 4 - (addr & 3), 1, fc, sz_long, &pa2, &ab2);

	switch (addr & 3) {
	case 1:
		abbput(ab1, pa1, l >> 24);
		abwput(ab1, pa1, l >> 8);
		abbput(ab2, pa2, l);
		break;
	case 2:
		abwput(ab1, pa1, l >> 16);
		abwput(ab2, pa2, l);
		break;
	case 3:
		abbput(ab1, pa1, l >> 24);
		abwput(ab2, pa2, l >> 8);
		abbput(ab2, pa2+2, l);
		break;
	}
}

static
void REGPARAM2 mmu_put_word_pages(uaecptr addr, unsigned fc, uae_u16 w)
{
	uaecptr pa1, pa2;
	addrbank *ab1, *ab2;

	mmu_translate_fc(addr, 1, fc, sz_word, &pa1, &ab1);
	mmu_translate_fc(addr + 1, 1, fc, sz_word, &pa2, &ab2);

	abbput(ab1, pa1, w >> 8);
	abbput(ab2, pa2, w);
}

/*************************************************************************/

uaecptr iframe = 1;

void REGPARAM2 mmu_change_fc(void)
{
	iframe = 1;
	mmu_set_xlt(regs.s);
}

uae_u32 REGPARAM2 mmu_get_ilong(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	if ((addr & 0xfff) >= 0xffd)
		return mmu_get_long_pages(addr, regs.s ? 6 : 2);
	mmu_translate_inst(addr, sz_long, &pa, &ab);
	return ablget(ab, pa);
}
uae_u16 REGPARAM2 mmu_get_iword(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

#if 0
	/* We never fetch instruction words from odd addresses */
	if ((addr & 0xfff) >= 0xfff)
		return mmu_get_word_pages(addr, regs.s ? 6 : 2);
#endif
	mmu_translate_inst(addr, sz_word, &pa, &ab);
	return abwget(ab, pa);
}
uae_u8 REGPARAM2 mmu_get_ibyte(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	mmu_translate_inst(addr, sz_byte, &pa, &ab);
	return abbget(ab, pa);
}

uae_u8 REGPARAM2 *get_real_iaddress(uaecptr addr)
{
	uaecptr frame, pa;
	addrbank *ab;

	/* this is only called by m68k_setpc(). We cache the
	 * page translation and use a much cheaper translation
	 * as long as we stay within the same page.
	 */

	frame = addr & regs.frame_mask;
	iframe = frame;
	mmu_translate_inst(addr, sz_word, &pa, &ab);
	return ab->xlateaddr(pa);
}

/*************************************************************************/

uae_u32 REGPARAM2 mmu_get_long(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	if ((addr & 0xfff) >= 0xffd)
		return mmu_get_long_pages(addr, regs.s ? 5 : 1);
	mmu_translate_data(addr, 0, sz_long, &pa, &ab);
	return ablget(ab, pa);
}
uae_u16 REGPARAM2 mmu_get_word(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	if ((addr & 0xfff) >= 0xfff)
		return mmu_get_word_pages(addr, regs.s ? 5 : 1);
	mmu_translate_data(addr, 0, sz_word, &pa, &ab);
	return abwget(ab, pa);
}
uae_u8 REGPARAM2 mmu_get_byte(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	mmu_translate_data(addr, 0, sz_byte, &pa, &ab);
	return abbget(ab, pa);
}

void REGPARAM2 mmu_put_long(uaecptr addr, uae_u32 l)
{
	uaecptr pa;
	addrbank *ab;

	regs.wb2a = addr;
	regs.wb2d = l;
	regs.wb2s = WBS_VALID | WBS_SIZE_LONG | (regs.s ? 0x04 : 0x00) | 1;
	if ((addr & 0xfff) >= 0xffd)
		mmu_put_long_pages(addr, regs.s ? 5 : 1, l);
	else {
		mmu_translate_data(addr, 1, sz_long, &pa, &ab);
		ablput(ab, pa, l);
	}
	regs.wb2s = 0;
}
void REGPARAM2 mmu_put_word(uaecptr addr, uae_u16 w)
{
	uaecptr pa;
	addrbank *ab;

	regs.wb2a = addr;
	regs.wb2d = w;
	regs.wb2s = WBS_VALID | WBS_SIZE_WORD | (regs.s ? 0x04 : 0x00) | 1;
	if ((addr & 0xfff) >= 0xfff)
		mmu_put_word_pages(addr, regs.s ? 5 : 1, w);
	else {
		mmu_translate_data(addr, 1, sz_word, &pa, &ab);
		abwput(ab, pa, w);
	}
	regs.wb2s = 0;
}
void REGPARAM2 mmu_put_byte(uaecptr addr, uae_u16 b)
{
	uaecptr pa;
	addrbank *ab;

	regs.wb2a = addr;
	regs.wb2d = b;
	regs.wb2s = WBS_VALID | WBS_SIZE_BYTE | (regs.s ? 0x04 : 0x00) | 1;
	mmu_translate_data(addr, 1, sz_byte, &pa, &ab);
	abbput(ab, pa, b);
	regs.wb2s = 0;
}

uae_u8 REGPARAM2 *get_real_address(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	mmu_translate_data(addr, 0, sz_byte, &pa, &ab);
	return ab->xlateaddr(pa);
}

int REGPARAM2 valid_address(uaecptr addr, uae_u32 size)
{
	uaecptr pa;
	addrbank *ab;

	mmu_translate_data(addr, 0, sz_byte, &pa, &ab);
	return ab->check(pa, size);
}


uae_u32 REGPARAM2 sfc_get_long(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	if ((addr & 0xfff) >= 0xffd)
		return mmu_get_long_pages(addr, regs.sfc);
	mmu_translate_fc(addr, 0, regs.sfc, sz_long, &pa, &ab);
	return ablget(ab, pa);
}
uae_u16 REGPARAM2 sfc_get_word(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	if ((addr & 0xfff) >= 0xfff)
		return mmu_get_word_pages(addr, regs.sfc);
	mmu_translate_fc(addr, 0, regs.sfc, sz_word, &pa, &ab);
	return abwget(ab, pa);
}
uae_u8 REGPARAM2 sfc_get_byte(uaecptr addr)
{
	uaecptr pa;
	addrbank *ab;

	mmu_translate_fc(addr, 0, regs.sfc, sz_byte, &pa, &ab);
	return abbget(ab, pa);
}


void REGPARAM2 dfc_put_long(uaecptr addr, uae_u32 l)
{
	uaecptr pa;
	addrbank *ab;

	regs.wb2a = addr;
	regs.wb2d = l;
	regs.wb2s = WBS_VALID | WBS_SIZE_LONG | regs.dfc;
	if ((addr & 0xfff) >= 0xffd)
		mmu_put_long_pages(addr, regs.dfc, l);
	else {
		mmu_translate_fc(addr, 1, regs.dfc, sz_long, &pa, &ab);
		ablput(ab, pa, l);
	}
	regs.wb2s = 0;
}
void REGPARAM2 dfc_put_word(uaecptr addr, uae_u16 w)
{
	uaecptr pa;
	addrbank *ab;

	regs.wb2a = addr;
	regs.wb2d = w;
	regs.wb2s = WBS_VALID | WBS_SIZE_WORD | regs.dfc;
	if ((addr & 0xfff) >= 0xffd)
		mmu_put_word_pages(addr, regs.dfc, w);
	else {
		mmu_translate_fc(addr, 1, regs.dfc, sz_word, &pa, &ab);
		abwput(ab, pa, w);
	}
	regs.wb2s = 0;
}
void REGPARAM2 dfc_put_byte(uaecptr addr, uae_u16 b)
{
	uaecptr pa;
	addrbank *ab;

	regs.wb2a = addr;
	regs.wb2d = b;
	regs.wb2s = WBS_VALID | WBS_SIZE_BYTE | regs.dfc;
	mmu_translate_fc(addr, 1, regs.dfc, sz_byte, &pa, &ab);
	abbput(ab, pa, b);
	regs.wb2s = 0;
}

