 /* 
  * UAE - The Un*x Amiga Emulator
  * 
  * MC68000 emulation - machine dependent bits
  *
  * Copyright 1996 Bernd Schmidt
  */

#ifndef OPTIMIZED_FLAGS

struct flag_struct {
    unsigned char c;
    unsigned char z;
    unsigned char n;
    unsigned char v; 
    unsigned char x;
};

extern struct flag_struct regflags;

#define GET_ZFLG (regflags.z)
#define GET_NFLG (regflags.n)
#define GET_CFLG (regflags.c)
#define GET_VFLG (regflags.v)
#define GET_XFLG (regflags.x)

static inline
void SET_ZFLG(unsigned int y) {
	regflags.z = y;
}
static inline
void SET_NFLG(unsigned int y) {
	regflags.n = y;
}
static inline
void SET_CFLG(unsigned int y) {
	regflags.c = y;
}
static inline
void SET_VFLG(unsigned int y) {
	regflags.v = y;
}
static inline
void SET_XFLG(unsigned int y) {
	regflags.x = y;
}

static inline
unsigned int GET_CZNV(void) {
	return
		regflags.c
		| regflags.z << 6
		| regflags.n << 7
		| regflags.v << 11;
}
static inline
void IOR_CZNV(unsigned int y) {
	regflags.c |= y & 0x0001 ? 1 : 0;
	regflags.z |= y & 0x0040 ? 1 : 0;
	regflags.n |= y & 0x0080 ? 1 : 0;
	regflags.v |= y & 0x0800 ? 1 : 0;
}
static inline
void SET_CZNV(unsigned int y) {
	regflags.c = y & 0x0001 ? 1 : 0;
	regflags.z = y & 0x0040 ? 1 : 0;
	regflags.n = y & 0x0080 ? 1 : 0;
	regflags.v = y & 0x0800 ? 1 : 0;
}

#define CLEAR_CZNV do { regflags.c = regflags.z = regflags.n = regflags.v = 0; } while (0)
#define COPY_CARRY do { regflags.x = regflags.c; } while (0)

static inline
int cctrue(int cc) {
	switch(cc) {
	case 0: /* T */
		return 1;
	case 1: /* F */
		return 0;
	case 2: /* HI */
		return !regflags.c && !regflags.z;
	case 3: /* LS */
		return regflags.c || regflags.z;
	case 4: /* CC */
		return !regflags.c;
	case 5: /* CS */
		return regflags.c;
	case 6: /* NE */
		return !regflags.z;
	case 7: /* EQ */
		return regflags.z;
	case 8: /* VC */
		return !regflags.v;
	case 9: /* VS */
		return regflags.v;
	case 10: /* PL */
		return !regflags.n;
	case 11: /* MI */
		return regflags.n;
	case 12: /* GE */
		return regflags.n == regflags.v;
	case 13: /* LT */
		return regflags.n != regflags.v;
	case 14: /* GT */
		return !regflags.z && (regflags.n == regflags.v);
	case 15: /* LE */
		return regflags.z || (regflags.n != regflags.v);
	}
	abort();
	return 0;
}

#else

struct flag_struct {
    unsigned int cznv;
    unsigned int x;
};

#define FLAGVAL_Z 0x40
#define FLAGVAL_N 0x80

#ifdef GCC_NOT_BROKEN

#define SET_ZFLG(y) (regflags.cznv = (regflags.cznv & ~0x40) | ((unsigned int)((y) & 1) << 6))
#define SET_CFLG(y) (regflags.cznv = (regflags.cznv & ~1) | ((unsigned int)(y) & 1))
#define SET_VFLG(y) (regflags.cznv = (regflags.cznv & ~0x800) | ((unsigned int)((y) & 1) << 11))
#define SET_NFLG(y) (regflags.cznv = (regflags.cznv & ~0x80) | ((unsigned int)((y) & 1) << 7))
#define SET_XFLG(y) (regflags.x = (y))

#else

static void __SET_ZFLG(int);
static void __SET_CFLG(int);
static void __SET_VFLG(int);
static void __SET_NFLG(int);
static void __SET_XFLG(int);
extern struct flag_struct regflags;

STATIC_INLINE void __SET_ZFLG(int y) {
	regflags.cznv = (regflags.cznv & ~0x40) | (y & 1) << 6;
}
#define SET_ZFLG(y) __SET_ZFLG(y)
STATIC_INLINE void __SET_CFLG(int y) {
	regflags.cznv = (regflags.cznv & ~0x1) | (y & 1);
}
#define SET_CFLG(y) __SET_CFLG(y)
STATIC_INLINE void __SET_VFLG(int y) {
	regflags.cznv = (regflags.cznv & ~0x800) | (y & 1) << 11;
}
#define SET_VFLG(y) __SET_VFLG(y)
STATIC_INLINE void __SET_NFLG(int y) {
	regflags.cznv = (regflags.cznv & ~0x80) | (y & 1) << 7;
}
#define SET_NFLG(y) __SET_NFLG(y)
STATIC_INLINE void __SET_XFLG(int y) {
	regflags.x = y;
}
#define SET_XFLG(y) __SET_XFLG(y)

#endif

#define GET_ZFLG ((regflags.cznv >> 6) & 1)
#define GET_CFLG (regflags.cznv & 1)
#define GET_VFLG ((regflags.cznv >> 11) & 1)
#define GET_NFLG ((regflags.cznv >> 7) & 1)
#define GET_XFLG (regflags.x & 1)

#define CLEAR_CZNV (regflags.cznv = 0)
#define GET_CZNV (regflags.cznv)
#define IOR_CZNV(X) (regflags.cznv |= (X))
#define SET_CZNV(X) (regflags.cznv = (X))

#define COPY_CARRY (regflags.x = regflags.cznv)


extern struct flag_struct regflags __asm__ ("regflags");

static __inline__ int cctrue(int cc)
{
    uae_u32 cznv = regflags.cznv;
    switch(cc){
     case 0: return 1;                       /* T */
     case 1: return 0;                       /* F */
     case 2: return (cznv & 0x41) == 0; /* !GET_CFLG && !GET_ZFLG;  HI */
     case 3: return (cznv & 0x41) != 0; /* GET_CFLG || GET_ZFLG;    LS */
     case 4: return (cznv & 1) == 0;        /* !GET_CFLG;               CC */
     case 5: return (cznv & 1) != 0;           /* GET_CFLG;                CS */
     case 6: return (cznv & 0x40) == 0; /* !GET_ZFLG;               NE */
     case 7: return (cznv & 0x40) != 0; /* GET_ZFLG;                EQ */
     case 8: return (cznv & 0x800) == 0;/* !GET_VFLG;               VC */
     case 9: return (cznv & 0x800) != 0;/* GET_VFLG;                VS */
     case 10:return (cznv & 0x80) == 0; /* !GET_NFLG;               PL */
     case 11:return (cznv & 0x80) != 0; /* GET_NFLG;                MI */
     case 12:return (((cznv << 4) ^ cznv) & 0x800) == 0; /* GET_NFLG == GET_VFLG;             GE */
     case 13:return (((cznv << 4) ^ cznv) & 0x800) != 0;/* GET_NFLG != GET_VFLG;             LT */
     case 14:
	cznv &= 0x8c0;
	return (((cznv << 4) ^ cznv) & 0x840) == 0; /* !GET_ZFLG && (GET_NFLG == GET_VFLG);  GT */
     case 15:
	cznv &= 0x8c0;
	return (((cznv << 4) ^ cznv) & 0x840) != 0; /* GET_ZFLG || (GET_NFLG != GET_VFLG);   LE */
    }
    abort();
    return 0;
}

#define optflag_testl(v) \
  __asm__ __volatile__ ("testl %1,%1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv) : "r" (v) : "cc")

#define optflag_testw(v) \
  __asm__ __volatile__ ("testw %w1,%w1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv) : "r" (v) : "cc")

#define optflag_testb(v) \
  __asm__ __volatile__ ("testb %b1,%b1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv) : "q" (v) : "cc")

#define optflag_addl(v, s, d) do { \
  __asm__ __volatile__ ("addl %k2,%k1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv), "=r" (v) : "rmi" (s), "1" (d) : "cc"); \
    COPY_CARRY; \
    } while (0)
#define optflag_addw(v, s, d) do { \
  __asm__ __volatile__ ("addw %w2,%w1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv), "=r" (v) : "rmi" (s), "1" (d) : "cc"); \
    COPY_CARRY; \
    } while (0)

#define optflag_addb(v, s, d) do { \
  __asm__ __volatile__ ("addb %b2,%b1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv), "=q" (v) : "qmi" (s), "1" (d) : "cc"); \
    COPY_CARRY; \
    } while (0)

#define optflag_subl(v, s, d) do { \
  __asm__ __volatile__ ("subl %k2,%k1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv), "=r" (v) : "rmi" (s), "1" (d) : "cc"); \
    COPY_CARRY; \
    } while (0)

#define optflag_subw(v, s, d) do { \
  __asm__ __volatile__ ("subw %w2,%w1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv), "=r" (v) : "rmi" (s), "1" (d) : "cc"); \
    COPY_CARRY; \
    } while (0)

#define optflag_subb(v, s, d) do { \
  __asm__ __volatile__ ("subb %b2,%b1\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv), "=q" (v) : "qmi" (s), "1" (d) : "cc"); \
    COPY_CARRY; \
    } while (0)

#define optflag_cmpl(s, d) \
  __asm__ __volatile__ ("cmpl %k1,%k2\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv) : "rmi" (s), "r" (d) : "cc")

#define optflag_cmpw(s, d) \
  __asm__ __volatile__ ("cmpw %w1,%w2\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv) : "rmi" (s), "r" (d) : "cc")

#define optflag_cmpb(s, d) \
  __asm__ __volatile__ ("cmpb %b1,%b2\n\t" \
			"pushfl\n\t" \
			"popl %0\n\t" \
			: "=r" (regflags.cznv) : "qmi" (s), "q" (d) : "cc")

#endif

