/*
 * UAE - The U*nix Amiga Emulator
 *
 * A2065 network card
 */

#define WRITE_LOG(x)

#include "sysconfig.h"
#include "sysdeps.h"

#include "config.h"
#include "options.h"
#include "custom.h"
#include "memory.h"
#include "lance.h"

#include <sys/ioctl.h>
#include <sys/uio.h>
#include <sys/endian.h>

#define LMEM 16384

struct lance {
	uae_u16 rap;

	uae_u16 csr0;
	uae_u16 csr1;
	uae_u16 csr2;
	uae_u16 csr3;

	int enable;

	uae_u16 mode;
	uae_u16 padr[3];
	uae_u16 ladr[4];
	int     rdra, rlen;
	int     tdra, tlen;

	int rptr, tptr;
	int attached;
	int devfd;

	char *devname;

	uae_u16 b[LMEM];
};

#define lmem(l,x) (&(l)->b[(x) & (LMEM-1)])

/****************************************************************/

static int open_tap(const char *dev)
{
	int fd;

	fd = open(dev, O_RDWR|O_NONBLOCK, 0);

	if (fd != -1)
		(void)fcntl(fd, F_SETOWN, getpid());

	return fd;
}

static int send_packet(int fd, struct iovec *iov, int iovcnt)
{
	int rc;

	rc = writev(fd, iov, iovcnt);
WRITE_LOG(("send_packet rc=%d, errno=%d\n",rc,errno));
	if (rc == -1 && (errno == EINTR || errno == EAGAIN))
		return -1;
	return 0;
}

static int poll_packet(int fd)
{
	int rc;
	int len;

	rc = ioctl(fd, FIONREAD, &len);
	if (rc == -1) {
		WRITE_LOG(("poll_packet errno=%d\n",errno));
		return 0;
	}

	return len;
}

static int recv_packet(int fd, struct iovec *iov, int iovcnt)
{
	int rc;
	size_t m;

	/* read 4 bytes less as tap doesn't deliver an FCS,
	 * the FCS is ignored anyway, no need to compute a real one
	 */
	m = 4;
	while (iovcnt > 0) {
		if (iov[iovcnt-1].iov_len >= m) {
			iov[iovcnt-1].iov_len -= m;
			break;
		}
		m -= iov[iovcnt-1].iov_len;
		--iovcnt;
	}

	rc = readv(fd, iov, iovcnt);
	if (rc == -1 && (errno == EINTR || errno == EAGAIN))
		return -1;
	if (rc == -1)
		return 0;
	return rc;
}

/****************************************************************/

static void interrupt(struct lance *l)
{
	uae_u16 mask = LE_C0_INTR | LE_C0_INEA;
	uae_u16 all  = (
		LE_C0_BABL |
		LE_C0_MISS |
		LE_C0_MERR |
		LE_C0_RINT |
		LE_C0_TINT |
		LE_C0_IDON );

	if (l->csr0 & all)
		l->csr0 |= LE_C0_INTR;
	else
		l->csr0 &= ~LE_C0_INTR;

	if ((l->csr0 & mask) == mask) {
WRITE_LOG(("interrupt csr0=%04x all=%04x mask=%04x!\n", l->csr0, all, mask));
		INTREQ_PORTS(0x8002);
	} else {
		INTREQ_PORTS(0x0002);
	}
}

static int putbuf(
	struct iovec *v,
	int k,
	int a,
	int n,
	uae_u16 *m,
	int s)
{
	int loc  = (a / sizeof(m[0])) & (LMEM-1);
	int w    = (n+1) / sizeof(m[0]);

	if (loc+w > s) {
		v[k].iov_base = &m[loc];
		v[k].iov_len  = (s - loc) * sizeof(m[0]);
		v[k+1].iov_base = &m[0];
		v[k+1].iov_len  = n - v[k].iov_len;
		k += 2;
	} else {
		v[k].iov_base = &m[loc];
		v[k].iov_len  = n;
		k += 1;
	}
	return k;
}

#define PUTBUF(v,k,a,n,m,s) k = putbuf(v,k,a,n,m,s)

static void read_tmd(struct lance *l, int p, struct letmd *tmd)
{
	uae_u16 w;

	tmd->tmd0 = be16toh(*lmem(l, p));
	w         = be16toh(*lmem(l, p+1));
	tmd->tmd1_hadr = w & 0xff;
	tmd->tmd1_bits = w >> 8;
	tmd->tmd2 = be16toh(*lmem(l, p+2));
	tmd->tmd3 = be16toh(*lmem(l, p+3));
}

static void write_tmd1_bits(struct lance *l, int p, uae_u8 bits)
{
	uae_u16 w;

	w = ((uae_u16)bits) << 8;

	*lmem(l, p+1) = htobe16(w);
}

static void write_tmd3(struct lance *l, int p, uae_u16 w)
{
	*lmem(l, p+3) = htobe16(w);
}

static void read_rmd(struct lance *l, int p, struct lermd *rmd)
{
	uae_u16 w;

	rmd->rmd0 = be16toh(*lmem(l, p));
	w         = be16toh(*lmem(l, p+1));
	rmd->rmd1_hadr = w & 0xff;
	rmd->rmd1_bits = w >> 8;
	rmd->rmd2 = be16toh(*lmem(l, p+2));
	rmd->rmd3 = be16toh(*lmem(l, p+3));
}

static void write_rmd1_bits(struct lance *l, int p, uae_u8 bits)
{
	uae_u16 w;

	w = ((uae_u16)bits) << 8;

	*lmem(l, p+1) = htobe16(w);
}

static void write_rmd3(struct lance *l, int p, uae_u16 w)
{
	*lmem(l, p+3) = htobe16(w & 0xfff);
}


static int transmit(struct lance *l)
{
	struct iovec iov[256];
	struct letmd tmd;
	int optr;
	uae_u16 otmd3;
	int i, k;
	int offset, bytes;

	if (!l->enable || (l->csr0 & LE_C0_TXON) == 0)
		return 0;

	read_tmd(l, l->tptr, &tmd);
WRITE_LOG(("transmit? ptr=%04x bits=%04x\n",l->tptr,tmd.tmd1_bits));
	if ((tmd.tmd1_bits & (LE_T1_OWN|LE_T1_STP)) != (LE_T1_OWN|LE_T1_STP))
		return 0;
	
	for (i=0, k=0; i<8; ++i) {
		optr = l->tptr;

		offset = tmd.tmd1_hadr << 16 | tmd.tmd0;
		bytes  = 4096 - (tmd.tmd2 & 0xfff);
WRITE_LOG(("transmit %04x %d\n",offset,bytes));
		PUTBUF(iov,k,offset,bytes,lmem(l,0),LMEM);

		tmd.tmd1_bits &= ~LE_T1_OWN;
		write_tmd1_bits(l, optr, tmd.tmd1_bits);

		l->tptr += 4;
		if (l->tptr >= l->tdra + l->tlen)
			l->tptr = l->tdra;

		/* was this a stop packet? */
		if (tmd.tmd1_bits & LE_T1_STP) {

WRITE_LOG(("transmit sending\n"));
			send_packet(l->devfd, iov, k);
			l->csr0 |= LE_C0_TINT;
			break;
		}

		/* next descriptor */
		otmd3 = tmd.tmd3;
		read_tmd(l, l->tptr, &tmd);
		if ((tmd.tmd1_bits & LE_T1_OWN) == 0) {
WRITE_LOG(("transmit buffer error\n"));
			/* mark the last buffer with an error */
			write_tmd3(l, optr, otmd3 | LE_T3_BUFF);
			l->csr0 |= LE_C0_TINT;
			break;
		}
	}
WRITE_LOG(("transmit end\n"));

	return 1;
}

static int receive(struct lance *l)
{
	struct iovec iov[256];
	struct lermd rmd;
	int optr;
	uae_u8 ormd1_bits;
	int i, k, n;
	int offset, bytes;

	if (!l->enable || (l->csr0 & LE_C0_RXON) == 0)
		return 0;

	/* poll packet length */
	n = poll_packet(l->devfd);
	if (n == 0)
		return 0;

	/* add space for faked CRC */
	n += 4;

WRITE_LOG(("receive! bytes=%d\n",n));

	read_rmd(l, l->rptr, &rmd);
	if ((rmd.rmd1_bits & LE_R1_OWN) == 0) {
WRITE_LOG(("no receive buffer\n"));
		/* MISS? */
		return 0;
	}

	rmd.rmd1_bits |= LE_R1_STP;
	write_rmd1_bits(l, l->rptr, rmd.rmd1_bits);
	
	for (i=0, k=0; i<8; ++i) {
		optr = l->rptr;

		offset = rmd.rmd1_hadr << 16 | rmd.rmd0;
		bytes  = 4096 - (rmd.rmd2 & 0xfff);
WRITE_LOG(("receive %04x %d\n",offset,bytes));

		if (n < bytes)
			bytes = n;
		n -= bytes;

		PUTBUF(iov,k,offset,bytes,lmem(l,0),LMEM);

		rmd.rmd1_bits &= ~LE_R1_OWN;
		write_rmd1_bits(l, optr, rmd.rmd1_bits);

		l->rptr += 4;
		if (l->rptr >= l->rdra + l->rlen)
			l->rptr = l->rdra;

		if (n == 0) {
			rmd.rmd1_bits |= LE_R1_ENP;
			write_rmd1_bits(l, optr, rmd.rmd1_bits);
			rmd.rmd3 = bytes;
			write_rmd3(l, optr, rmd.rmd3);

			/* fetch data */
WRITE_LOG(("receive reading\n"));
			(void)recv_packet(l->devfd, iov, k);

			/* handle error? */

			l->csr0 |= LE_C0_RINT;
			break;
		}

		/* fetch next descriptor */
		ormd1_bits = rmd.rmd1_bits;
		read_rmd(l, l->rptr, &rmd);
		if ((rmd.rmd1_bits & LE_R1_OWN) == 0) {
WRITE_LOG(("receive buffer error\n"));
			recv_packet(l->devfd, iov, k);
			/* mark a buffer error */
			write_rmd1_bits(l, optr, ormd1_bits | LE_R1_BUFF);
			l->csr0 |= LE_C0_RINT;
			break;
		}
	}
WRITE_LOG(("receive end\n"));

	return 1;
}

/****************************************************************/

static void attach_lance(struct lance *l)
{
	if (l->attached && l->devfd != -1)
		close(l->devfd);

	l->devfd = -1;
	if (l->devname != NULL) {
		l->devfd = open_tap(l->devname);
		if (l->devfd == -1)
			write_log("attach_lance failed: %s\n", l->devname);
	}

	l->attached = 1;
}

static void reset_lance(struct lance *l)
{
	l->rap    = 0;
	l->csr0   = LE_C0_STOP;
	l->csr3   = 0;
	l->enable = 0;

	attach_lance(l);

	interrupt(l);
}

static void init_lance(struct lance *l)
{
	int r = l->csr1 >> 1;

	l->mode = be16toh(*lmem(l,r++)) & 0x80ff;
WRITE_LOG(("init_lance: mode = %04x\n", l->mode));

	l->padr[0] = be16toh(*lmem(l,r++));
	l->padr[1] = be16toh(*lmem(l,r++));
	l->padr[2] = be16toh(*lmem(l,r++));
WRITE_LOG(("init_lance: padr = %04x:%04x:%04x\n",l->padr[0],l->padr[1],l->padr[2]));

	l->ladr[0] = be16toh(*lmem(l,r++));
	l->ladr[1] = be16toh(*lmem(l,r++));
	l->ladr[2] = be16toh(*lmem(l,r++));
	l->ladr[3] = be16toh(*lmem(l,r++));
WRITE_LOG(("init_lance: ladr = %04x:%04x:%04x:%04x\n",l->ladr[0],l->ladr[1],l->ladr[2],l->ladr[3]));

	l->rdra = (be16toh(*lmem(l,r++)) & 0xfff8) >> 1;
	l->rdra |= (be16toh(*lmem(l,r)) & 0x00ff) << 15;
	l->rlen = 4 << ((be16toh(*lmem(l,r)) & 0xe000) >> 13);
	r++;
WRITE_LOG(("init_lance: recv ring = %04x + %d\n", l->rdra, l->rlen));

	l->tdra = (be16toh(*lmem(l,r++)) & 0xfff8) >> 1;
	l->tdra |= (be16toh(*lmem(l,r)) & 0x00ff) << 15;
	l->tlen = 4 << ((be16toh(*lmem(l,r)) & 0xe000) >> 13);
	r++;
WRITE_LOG(("init_lance: send ring = %04x + %d\n", l->tdra, l->tlen));

	l->rptr = l->rdra;
	l->tptr = l->tdra;

	l->csr0 |= LE_C0_IDON;
}

static uae_u16 read_csr0(struct lance *l)
{
	uae_u16 w;

	(void) transmit(l);
	(void) receive(l);
	interrupt(l);

	w = l->csr0;

	if (l->csr0 & (LE_C0_BABL|LE_C0_CERR|LE_C0_MISS|LE_C0_MERR))
		w |= LE_C0_ERR;


	return w;
}

static void write_csr0(struct lance *l, uae_u16 w)
{
WRITE_LOG(("write_csr0: %04x\n",w));
	if (w & LE_C0_STOP) {
WRITE_LOG(("write_csr0: STOP\n"));
		reset_lance(l);
		w &= ~(LE_C0_STRT|LE_C0_INIT);
	}
	if (w & LE_C0_STRT) {
WRITE_LOG(("write_csr0: STRT\n"));
		l->enable = 1;
		l->csr0 &= ~LE_C0_STOP;
		l->csr0 |= LE_C0_RXON|LE_C0_TXON;
	}
	if (w & LE_C0_INIT) {
WRITE_LOG(("write_csr0: INIT\n"));
		l->csr0 &= ~LE_C0_STOP;
		init_lance(l);
	}

	if (w & LE_C0_INEA)
		l->csr0 |= LE_C0_INEA;
	else
		l->csr0 &= ~LE_C0_INEA;

	if (w & (LE_C0_STOP|LE_C0_IDON))
		l->csr0 &= ~LE_C0_IDON;
	if (w & (LE_C0_STOP|LE_C0_TINT))
		l->csr0 &= ~LE_C0_TINT;
	if (w & (LE_C0_STOP|LE_C0_RINT))
		l->csr0 &= ~LE_C0_RINT;
	if (w & (LE_C0_STOP|LE_C0_MERR))
		l->csr0 &= ~LE_C0_MERR;
	if (w & (LE_C0_STOP|LE_C0_MISS))
		l->csr0 &= ~LE_C0_MISS;
	if (w & (LE_C0_STOP|LE_C0_CERR))
		l->csr0 &= ~LE_C0_CERR;
	if (w & (LE_C0_STOP|LE_C0_BABL))
		l->csr0 &= ~LE_C0_BABL;

	(void) transmit(l);
	(void) receive(l);
	interrupt(l);
}

/****************************************************************/

static struct lance the_lance;

static uae_u16 ReadLance(int r)
{
	uae_u16 w = 0;

	switch (r & 0x4001) {
	case 0x0001:
		w = the_lance.rap;
		break;
	case 0x0000:
		switch (the_lance.rap) {
		case 0:
			w = read_csr0(&the_lance);
			break;
		case 1:
			w = the_lance.csr1;
			break;
		case 2:
			w = the_lance.csr2;
			break;
		case 3:
			w = the_lance.csr3;
			break;
		}
WRITE_LOG(("ReadLance(%d) = %04x\n",the_lance.rap,w));
		break;
	default:
		w = be16toh(*lmem(&the_lance,r));
WRITE_LOG(("ReadMem(%04x) = %04x\n",r & 0x1fff,w));
		break;
	}

	return w;
}

static void WriteLance(int r, uae_u16 w)
{
	switch (r & 0x4001) {
	case 0x0001:
		the_lance.rap = w & 0x03;
		break;
	case 0x0000:
WRITE_LOG(("WriteLance(%d) = %04x%s\n",the_lance.rap,w,the_lance.csr0 & LE_C0_STOP ? " STOP" : ""));
		if (the_lance.csr0 & LE_C0_STOP) {
			switch (the_lance.rap) {
			case 0:
				write_csr0(&the_lance, w);
				break;
			case 1:
				the_lance.csr1 = w & 0xfffe;
				break;
			case 2:
				the_lance.csr2 = w & 0xff;
				break;
			case 3:
				the_lance.csr3 = w & 0x07;
				break;
			}
		} else {
			switch (the_lance.rap) {
			case 0:
				write_csr0(&the_lance, w);
				break;
			}
		}
		break;
	default:
WRITE_LOG(("WriteMem(%04x) = %04x\n",r & 0x1fff,w));
		*lmem(&the_lance,r) = htobe16(w);
		break;
	}
}

/****************************************************************/

static uae_u32 lance_lget(uaecptr) REGPARAM;
static uae_u32 lance_wget(uaecptr) REGPARAM;
static uae_u32 lance_bget(uaecptr) REGPARAM;
static void lance_lput(uaecptr, uae_u32) REGPARAM;
static void lance_wput(uaecptr, uae_u32) REGPARAM;
static void lance_bput(uaecptr, uae_u32) REGPARAM;

#define regnum(addr) (((unsigned long)(addr & 0xfffe)) >> 1)

addrbank lance_bank = {
    lance_lget, lance_wget, lance_bget,
    lance_lput, lance_wput, lance_bput,
    default_xlate, default_check, NULL,
    "lance"
};

static uae_u32 REGPARAM2 lance_lget (uaecptr addr)
{
	uae_u32 value;

	value = ReadLance(regnum(addr)) << 16;
	value |= ReadLance(regnum(addr)+1);
	return value;
}

static uae_u32 REGPARAM2 lance_wget (uaecptr addr)
{
	uae_u32 value;

	value = ReadLance(regnum(addr));
	return value;
}

static uae_u32 REGPARAM2 lance_bget (uaecptr addr)
{
	uae_u32 value;

	value = ReadLance(regnum(addr));

	if (addr & 1)
		value = value & 0x00ff;
	else
		value = value >> 8 & 0x00ff;

	return value;
}

static void REGPARAM2 lance_lput (uaecptr addr, uae_u32 value)
{
	WriteLance(regnum(addr), value >> 16);
	WriteLance(regnum(addr)+1, value);
}

static void REGPARAM2 lance_wput (uaecptr addr, uae_u32 value)
{
	WriteLance(regnum(addr), value);
}

static void REGPARAM2 lance_bput (uaecptr addr, uae_u32 value)
{
	uae_u32 w;

	w = ReadLance(regnum(addr));

	if (addr & 1)
		w = (w & 0xff00) | (value & 0x00ff);
	else
		w = (w & 0x00ff) | (value & 0x00ff) << 8;

	WriteLance(regnum(addr), w);
}

/****************************************************************/

/* Setup */

void lance_install(void)  
{
	reset_lance(&the_lance);
}

void add_lance(const char *str)
{
	if (the_lance.devname)
		free(the_lance.devname);
	the_lance.devname = strdup(str);

	attach_lance(&the_lance);
}

int lance_tick(void)
{
	int res = 0;

	/* res += transmit(&the_lance); */
	res += receive(&the_lance);
	interrupt(&the_lance);

	return res;
}

