#ifndef NEWCPU_H
#define NEWCPU_H 1

 /*
  * UAE - The Un*x Amiga Emulator
  *
  * MC68000 emulation
  *
  * Copyright 1995 Bernd Schmidt
  */

#include "readcpu.h"
#include "machdep/m68k.h"
#include <setjmp.h>

extern int areg_byteinc[];
extern int imm8_table[];

extern int movem_index1[256];
extern int movem_index2[256];
extern int movem_next[256];

extern int fpp_movem_index1[256];
extern int fpp_movem_index2[256];
extern int fpp_movem_next[256];

extern int broken_in;

typedef unsigned long cpuop_func (uae_u32) REGPARAM;
typedef char flagtype;

struct cputbl {
    cpuop_func *handler;
    int specific;
    uae_u16 opcode;
};

extern unsigned long op_illg (uae_u32) REGPARAM;

/* You can set this to long double to be more accurate. However, the
   resulting alignment issues will cost a lot of performance in some
   apps */
#define USE_LONG_DOUBLE 0

#if USE_LONG_DOUBLE
typedef long double fptype;
#else
typedef double fptype;
#endif

struct regstruct
{
    uae_u32 regs[16];
    uae_u32 usp,isp,msp;
    uae_u16 sr;
    flagtype t1;
    flagtype t0;
    flagtype s;
    flagtype m;
    flagtype x;
    flagtype stopped;
    int intmask;

    uae_u32 pc;
    uae_u8 *pc_p;
    uae_u8 *pc_oldp;
    addrbank *ibank;

    uae_u32 vbr,sfc,dfc;

    fptype fp[8];
    fptype fp_result;

    uae_u32 fpcr,fpsr,fpiar;
    uae_u32 fpsr_highbyte;

    uae_u32 spcflags;
    uae_u32 kick_mask;

    uae_u32 prefetch_pc;
    uae_u32 prefetch;

    uae_u32 caar, cacr, itt0, itt1, dtt0, dtt1, tc, mmusr, urp, srp;

    uae_u32 mmusr_test;

    int mmu_enabled, mmu_pagesize;
    uae_u32 mmu_fslw, mmu_fault_addr;
    uae_u16 mmu_ssw;
    uae_u32 wb1a, wb2a, wb3a;
    uae_u32 wb1d, wb2d, wb3d;
    uae_u16 wb1s, wb2s, wb3s;

    uae_u32 pgi_mask, sel_mask, frame_mask, offset_mask;
    uae_u32 ptr_page_mask, page_addr_mask, page_ur_mask;
    unsigned pgi_shift, sel_shift;

    int movem, fpp_movem, move16;
    uae_u32 ea;
    uae_u32 restart;

    int areg;
    uae_u32 asave;

    int stack;
};

extern struct regstruct regs;

extern void AccessError (void);
extern void Exception (int, uaecptr);
extern jmp_buf m68k_exception;
extern int in_exception_2;
extern void m68k_dumpstate (FILE *, uaecptr *);
extern void m68k_disasm (FILE *, uaecptr, uaecptr *, int);

#define m68k_dreg(r,num) ((r).regs[(num)])
#define m68k_areg(r,num) (((r).regs + 8)[(num)])

#include "mmu.h"

STATIC_INLINE void set_special (uae_u32 x)
{
    regs.spcflags |= x;
}

STATIC_INLINE void unset_special (uae_u32 x)
{
    regs.spcflags &= ~x;
}

#if 0
STATIC_INLINE uaecptr m68k_getpc (void)
{
#ifdef WITH_PREFETCH
    return regs.pc + ((char *)regs.pc_p - (char *)regs.pc_oldp);
#else
    return regs.pc;
#endif
}

STATIC_INLINE void m68k_incpc(uae_u32 o)
{
#ifdef WITH_PREFETCH
	regs.pc_p += o;
#else
	regs.pc += o;
	regs.pc_p += o;
#endif
}
#else
static inline uaecptr m68k_getpc(void) {
	return regs.pc + ((char *)regs.pc_p - (char *)regs.pc_oldp);
}
static inline void m68k_incpc(uae_u32 o) {
	regs.pc_p += o;
}
#endif

#ifdef USE_COMPILER
extern uae_u32 flush_icache(int opt);
extern void m68k_setpc(uaecptr newpc);
extern void m68k_setpc_fast (uaecptr newpc);
extern void m68k_setpc_bcc (uaecptr newpc);
extern void m68k_setpc_rte (uaecptr newpc);
#else
STATIC_INLINE uae_u32 flush_icache(int opt)
{
	return 0;
}
STATIC_INLINE void m68k_setpc (uaecptr newpc)
{
	regs.pc = regs.restart = newpc;
	
#if 0
#ifdef WITH_PREFETCH
	regs.prefetch_pc = 0;
	regs.pc_p = regs.pc_oldp = 0;
	regs.pc_p = regs.pc_oldp = get_real_iaddress(regs.pc);
#else
	regs.pc_p = get_real_iaddress(regs.pc);
#endif
#else
	regs.pc_p = regs.pc_oldp = 0;
	regs.pc_p = regs.pc_oldp = get_real_iaddress(regs.pc);
#endif

        if (newpc & 1)
		Exception(3, newpc);
}
#define m68k_setpc_fast m68k_setpc
#define m68k_setpc_bcc  m68k_setpc
#define m68k_setpc_rte  m68k_setpc
#endif

extern uaecptr iframe;
static inline uae_u32 get_ibyte(uae_u32 o)
{
	uaecptr addr, frame;

	addr = m68k_getpc()+o+1;
	frame = addr & regs.frame_mask;
	if (frame == iframe)
		return do_get_mem_byte((uae_u8*)(regs.pc_p+o+1));
	return mmu_get_ibyte(addr);
}
static inline uae_u32 get_iword(uae_u32 o)
{
	uaecptr addr, frame;

	addr = m68k_getpc()+o;
	frame = addr & regs.frame_mask;
	if (frame == iframe)
		return do_get_mem_word((uae_u16*)(regs.pc_p+o));
	return mmu_get_iword(addr);
}
static inline uae_u32 get_ilong(uae_u32 o)
{
	uaecptr addr, frame;

	addr = m68k_getpc()+o;
	frame = (addr+2) & regs.frame_mask;
	if (frame == iframe)
		return do_get_mem_long((uae_u32*)(regs.pc_p+o));
	return mmu_get_ilong(addr);
}

#define get_ibyte_1 get_ibyte
#define get_iword_1 get_iword
#define get_ilong_1 get_ilong

#define WBS_VALID     0x0080
#define WBS_SIZE_BYTE 0x0020
#define WBS_SIZE_WORD 0x0040
#define WBS_SIZE_LONG 0x0000

#define WBS_TT_NORMAL 0x0000
#define WBS_TM_UDATA  0x0001
#define WBS_TM_SDATA  0x0005

#ifdef WITH_PREFETCH
STATIC_INLINE void refill_prefetch (uae_u32 currpc, uae_u32 offs)
{
    uae_u32 t = (currpc + offs) & ~3;
    uae_s32 pc_p_offs = t - currpc;
    uae_u8 *ptr = regs.pc_p + pc_p_offs;
    uae_u32 r;

    regs.prefetch_pc = t;
    do_put_mem_long(&regs.prefetch, get_ilong(t));
}

STATIC_INLINE uae_u32 get_ibyte_prefetch (uae_s32 o)
{
    uae_u32 currpc = m68k_getpc ();
    uae_u32 addr = currpc + o + 1;
    uae_u32 offs = addr - regs.prefetch_pc;
    uae_u32 v;
    if (offs > 3) {
	refill_prefetch (currpc, o + 1);
	offs = addr - regs.prefetch_pc;
    }
    v = do_get_mem_byte (((uae_u8 *)&regs.prefetch) + offs);
    if (offs >= 2)
	refill_prefetch (currpc, 4);
    return v;
}

STATIC_INLINE uae_u32 get_iword_prefetch (uae_s32 o)
{
    uae_u32 currpc = m68k_getpc ();
    uae_u32 addr = currpc + o;
    uae_u32 offs = addr - regs.prefetch_pc;
    uae_u32 v;
    if (offs > 3) {
	refill_prefetch (currpc, o);
	offs = addr - regs.prefetch_pc;
    }
    v = do_get_mem_word ((uae_u16 *)(((uae_u8 *)&regs.prefetch) + offs));
    if (offs >= 2)
	refill_prefetch (currpc, 4);
    return v;
}

STATIC_INLINE uae_u32 get_ilong_prefetch (uae_s32 o)
{
    uae_u32 v = get_iword_prefetch (o);
    v <<= 16;
    v |= get_iword_prefetch (o + 2);
    return v;
}

STATIC_INLINE void fill_prefetch_0 (void)
{
}
#define fill_prefetch_2 fill_prefetch_0

#endif

/* These are only used by the 68020/68881 code, and therefore don't
 * need to handle prefetch.  */
STATIC_INLINE uae_u32 next_ibyte (void)
{
    uae_u32 r = get_ibyte (0);
    m68k_incpc (2);
    return r;
}

STATIC_INLINE uae_u32 next_iword (void)
{
    uae_u32 r = get_iword (0);
    m68k_incpc (2);
    return r;
}

STATIC_INLINE uae_u32 next_ilong (void)
{
    uae_u32 r = get_ilong (0);
    m68k_incpc (4);
    return r;
}

STATIC_INLINE void m68k_setstopped (int stop)
{
    regs.stopped = stop;
    /* A traced STOP instruction drops through immediately without
       actually stopping.  */
    if (stop && (regs.spcflags & SPCFLAG_DOTRACE) == 0)
	regs.spcflags |= SPCFLAG_STOP;
}

extern uae_u32 get_disp_ea_020 (uae_u32 base, uae_u32 dp);
extern uae_u32 get_disp_ea_000 (uae_u32 base, uae_u32 dp);

extern uae_s32 ShowEA (FILE *, int reg, amodes mode, wordsizes size, char *buf);

extern void MakeSR (void);
extern void MakeFromSR (void);
extern void SwitchSP (void);
extern void dump_counts (void);
extern int m68k_move2c (int, uae_u32 *);
extern int m68k_movec2 (int, uae_u32 *);
extern void m68k_divl (uae_u32, uae_u32, uae_u16, uaecptr);
extern void m68k_mull (uae_u32, uae_u32, uae_u16);
extern void init_m68k (void);
extern void m68k_go (int);
extern void m68k_reset (void);

extern void mmu_op (uae_u32, uae_u16);
extern void fpp_opp (uae_u32, uae_u16);
extern void fdbcc_opp (uae_u32, uae_u16);
extern void fscc_opp (uae_u32, uae_u16);
extern void ftrapcc_opp (uae_u32,uaecptr);
extern void fbcc_opp (uae_u32, uaecptr, uae_u32);
extern void fsave_opp (uae_u32);
extern void frestore_opp (uae_u32);

/* Opcode of faulting instruction */
extern uae_u16 last_op_for_exception_3;
/* PC at fault time */
extern uaecptr last_addr_for_exception_3;
/* Address that generated the exception */
extern uaecptr last_fault_for_exception_3;

#define CPU_OP_NAME(a) op ## a

/* 68040 */
extern struct cputbl op_smalltbl_0_ff[];
/* 68020 + 68881 */
extern struct cputbl op_smalltbl_1_ff[];
/* 68020 */
extern struct cputbl op_smalltbl_2_ff[];
/* 68010 */
extern struct cputbl op_smalltbl_3_ff[];
/* 68000 */
extern struct cputbl op_smalltbl_4_ff[];
/* 68000 slow but compatible.  */
extern struct cputbl op_smalltbl_5_ff[];

extern cpuop_func *cpufunctbl[65536] ASM_SYM_FOR_FUNC ("cpufunctbl");

#endif
