#ifndef EVENTS_H
#define EVENTS_H 1

 /*
  * UAE - The Un*x Amiga Emulator
  *
  * Events
  * These are best for low-frequency events. Having too many of them,
  * or using them for events that occur too frequently, can cause massive
  * slowdown.
  *
  * Copyright 1995-1998 Bernd Schmidt
  */

extern unsigned long currcycle, nextevent;
extern int in_frame;
extern unsigned long sample_evtime;
extern int frame_started, frame_started_slow;
typedef void (*evfunc)(void);

struct ev
{
    int active;
    unsigned long int evtime, oldcycles;
    evfunc handler;
};

enum {
    ev_hsync, ev_copper, ev_audio, ev_cia, ev_blitter, ev_disk,
    ev_max
};

extern struct ev eventtab[ev_max];

void events_schedule(void);
void handle_active_events(void);
void init_eventtab(void);
void real_check_frame_started(void);
void next_frame(int hz);

static inline unsigned long get_cycles(void)
{
	return currcycle;
}

static inline int check_frame_started(void)
{
	if (!frame_started && ++frame_started_slow >= 50000 && !in_frame)
		real_check_frame_started();
	return frame_started;
}

static inline long int do_cycles (unsigned long cycles_to_add)
{
    if (!in_frame && !check_frame_started())
	return cycles_to_add;

    while (nextevent <= currcycle + cycles_to_add) {
        int i;
        cycles_to_add -= (nextevent - currcycle);
        currcycle = nextevent;

        for (i = 0; i < ev_max; i++) {
	    if (eventtab[i].active && eventtab[i].evtime == currcycle) {
		(*eventtab[i].handler)();
	    }
	}
        events_schedule();
    }

    currcycle += cycles_to_add;
    return 0;
}

#endif
