#include <stdarg.h>
#include <stdio.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/select.h>

#include "console.h"

typedef enum { AUX_NONE, AUX_CONSOLE, AUX_SERIAL } aux_t;

static struct termios tios0, tios;
static aux_t aux = AUX_NONE;
static int outpos = -1;

#define conout stdout
#define conin stdin

static void advance(char *p, int n)
{
	while (n-- > 0) {
		if (*p == '\r' || *p == '\f')
			outpos = 0;
		else if (*p != '\n')  
			outpos++;   
	}       
}

static void async(int fd, int doit)
{
	pid_t pid;
	int flags; 

	pid = getpid();
	flags = fcntl(fd, F_GETFL, 0);
	if (doit) {
		flags |= O_ASYNC | O_NONBLOCK;
	} else {
		flags &= ~(O_ASYNC | O_NONBLOCK);
	}
	if (fcntl(fd, F_SETFL, flags))
		perror("F_SETFL");
	if (fcntl(fd, F_SETOWN, pid))
		perror("F_SETOWN");
}

static void set_aux(aux_t newaux)
{
	if (aux == newaux)
		return;

	switch (newaux) {
	case AUX_NONE:
	case AUX_SERIAL:
		if (aux == AUX_CONSOLE)
			write(fileno(conout),"\033[30m",5);
		tcsetattr(fileno(conin), TCSADRAIN, &tios);
		// if (aux == AUX_CONSOLE)
			async(fileno(conin), 1);
		break;
	case AUX_CONSOLE:
		// if (aux == AUX_SERIAL)
			async(fileno(conin), 0);
		if (outpos > 0)
			write(fileno(conout),"\r\n",2);
		outpos = 0;
		tcsetattr(fileno(conin), TCSADRAIN, &tios0);
		if (aux != AUX_NONE)
			write(fileno(conout),"\033[32m",5);
		break;
	}

	aux = newaux;
}

void con_vprintf(const char *fmt, va_list ap)
{
	set_aux(AUX_CONSOLE);
	vfprintf(conout, fmt, ap);
	fflush(conout);
}

void con_printf(const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	con_vprintf(fmt, ap);
	va_end(ap);
}

char *con_gets(char *buf, int size)
{
	set_aux(AUX_CONSOLE);
	char *res;
	fd_set rset;

	do {
		res = fgets(buf, size, conin);
		if (res == NULL && !feof(conin)) {
			clearerr(conin);
			FD_ZERO(&rset);
			FD_SET(fileno(conin),&rset);
			select(fileno(conin)+1,&rset,NULL,NULL,NULL);
		}
	} while (res == NULL);
	return res;
}

FILE *con_outputstream(void)
{
	set_aux(AUX_CONSOLE);
	return conout;
}

int aux_write(char *buf, int size)
{
	if (size <= 0)
		return 0;

	set_aux(AUX_SERIAL);
	advance(buf, size);

	return write(fileno(conout), buf, size);
}

int aux_read(char *buf, int size)
{
	set_aux(AUX_SERIAL);

	return read(fileno(conin), buf, size);
}

void aux_use(void)
{
	set_aux(AUX_SERIAL);
}

void con_init(void)
{
	tcgetattr(fileno(conin), &tios0);
	tios = tios0;

	cfmakeraw(&tios);
	tios.c_lflag &= ~(ISIG|ICANON|IGNBRK|ECHO|ECHOE);
	tios.c_cflag |= CLOCAL;
	tios.c_iflag &= ~(IXON|IXOFF|IXANY);
	tios.c_oflag &= ~OPOST;
	tios.c_cc[VMIN] = 1;
	tios.c_cc[VTIME] = 0;

	outpos = 0;
}

void con_exit(void)
{
	set_aux(AUX_NONE);
}
