/* 
 * Copyright (c) 2012, 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _MYSQL_PARSER_H_
#define _MYSQL_PARSER_H_

/**
 * C++ interface for the ANTLR based MySQL parser.
 */

#ifdef _WIN32
  #ifdef MYSQL_PARSER_EXPORTS
    #define MYSQL_PARSER_PUBLIC_FUNC __declspec(dllexport)
  #else
    #define MYSQL_PARSER_PUBLIC_FUNC __declspec(dllimport)
  #endif
#else
  #define MYSQL_PARSER_PUBLIC_FUNC
#endif

#include <stack>
#include <set>

#include "MySQLLexer.h"  // The generated lexer.
#include "MySQLParser.h" // The generated parser.

struct MySQLParserErrorInfo
{
  std::string message;
  ANTLR3_UINT32 error;
  ANTLR3_UINT32 token_type;
  ANTLR3_UINT32 length;
  ANTLR3_UINT32 line;
  ANTLR3_UINT32 offset;
};

class MySQLRecognizer;

/**
 * The tree walker allows to easily navigate the AST generated by the recognizer and get all kind
 * of token information (like position, type, content etc.).
 */
class MYSQL_PARSER_PUBLIC_FUNC MySQLRecognizerTreeWalker
{
  friend class MySQLRecognizer; // For protected c-tor.
public:
  MySQLRecognizer *recognizer() { return _recognizer; };
  
  bool next();
  bool next_sibling();
  bool previous();
  bool previous_by_index();
  bool previous_sibling();
  bool up();
  bool advance_to_position(int line, int offset);
  bool advance_to_type(unsigned int type, bool recurse);
  bool skip_token_sequence(unsigned int start_token, ...);
  unsigned int look_ahead(bool recursive);
  
  void reset();
  void push();
  bool pop();
  void remove_tos();
  
  // Properties of the current token.
  bool is_nil();
  bool is_subtree();
  bool is_identifier();
  bool is_keyword();
  bool is_relation();

  // Properties of the given token.
  bool is_identifier(unsigned int type);
  bool is_keyword(unsigned int type);
  bool is_relation(unsigned int type);

  std::string token_text();
  unsigned int token_type();
  unsigned int token_line();
  unsigned int token_start();
  int token_length();
  
protected:
  MySQLRecognizerTreeWalker(MySQLRecognizer *recognizer, pANTLR3_BASE_TREE tree);
  
  pANTLR3_BASE_TREE get_next(pANTLR3_BASE_TREE node, bool recurse);
  pANTLR3_BASE_TREE get_previous(pANTLR3_BASE_TREE node, bool recurse);
  void print_token(pANTLR3_BASE_TREE tree);
private:
  pANTLR3_BASE_TREE _origin;
  pANTLR3_BASE_TREE _tree;
  std::stack<pANTLR3_BASE_TREE> _token_stack;
  std::vector<pANTLR3_BASE_TREE> _token_list; // A list of all tokens in incoming order (no hierarchy).
  MySQLRecognizer *_recognizer;
};

class MYSQL_PARSER_PUBLIC_FUNC MySQLRecognizer
{
public:
  MySQLRecognizer(const char *text, int length, bool is_utf8, long server_version, 
    const std::string &sql_mode, const std::set<std::string> &charsets);
  ~MySQLRecognizer();
  
  std::string dump_tree();
  std::string dump_tree(pANTLR3_BASE_TREE tree, const std::string &indentation);
  
  MySQLRecognizerTreeWalker tree_walker();
  std::vector<MySQLParserErrorInfo> error_info() { return _error_info; };
  unsigned sql_mode();
  bool is_charset(const std::string &s);

  std::string token_text(pANTLR3_BASE_TREE node);

  // Internal routine, called from the error callback.
  void add_error(const std::string &text, ANTLR3_UINT32 error, ANTLR3_UINT32 token, ANTLR3_UINT32 line,
                 ANTLR3_UINT32 offset, ANTLR3_UINT32 length);
protected:
  void parse();
  unsigned parse_sql_mode(const std::string &sql_mode);
  
private:
  const char *_text;
  int _text_length;
  std::vector<MySQLParserErrorInfo> _error_info;
  int _input_encoding;
  RecognitionContext _context;
  std::set<std::string> _charsets; // A list of supported charsets.
  
  pANTLR3_INPUT_STREAM _input;
  pMySQLLexer _lexer;
  pANTLR3_COMMON_TOKEN_STREAM _tokens;
  pMySQLParser _parser;
  MySQLParser_query_return _ast;
};

#endif // _MYSQL_PARSER_H_
