# coding: utf-8
# Copyright (c) 2016, 2021, Oracle and/or its affiliates.  All rights reserved.
# This software is dual-licensed to you under the Universal Permissive License (UPL) 1.0 as shown at https://oss.oracle.com/licenses/upl or Apache License 2.0 as shown at http://www.apache.org/licenses/LICENSE-2.0. You may choose either license.

import oci  # noqa: F401
from oci.util import WAIT_RESOURCE_NOT_FOUND  # noqa: F401


class RoverEntitlementClientCompositeOperations(object):
    """
    This class provides a wrapper around :py:class:`~oci.rover.RoverEntitlementClient` and offers convenience methods
    for operations that would otherwise need to be chained together. For example, instead of performing an action
    on a resource (e.g. launching an instance, creating a load balancer) and then using a waiter to wait for the resource
    to enter a given state, you can call a single method in this class to accomplish the same functionality
    """

    def __init__(self, client, **kwargs):
        """
        Creates a new RoverEntitlementClientCompositeOperations object

        :param RoverEntitlementClient client:
            The service client which will be wrapped by this object
        """
        self.client = client

    def create_rover_entitlement_and_wait_for_state(self, create_rover_entitlement_details, wait_for_states=[], operation_kwargs={}, waiter_kwargs={}):
        """
        Calls :py:func:`~oci.rover.RoverEntitlementClient.create_rover_entitlement` and waits for the :py:class:`~oci.rover.models.RoverEntitlement` acted upon
        to enter the given state(s).

        :param oci.rover.models.CreateRoverEntitlementDetails create_rover_entitlement_details: (required)
            Creates a Rover Device Entitlement

        :param list[str] wait_for_states:
            An array of states to wait on. These should be valid values for :py:attr:`~oci.rover.models.RoverEntitlement.lifecycle_state`

        :param dict operation_kwargs:
            A dictionary of keyword arguments to pass to :py:func:`~oci.rover.RoverEntitlementClient.create_rover_entitlement`

        :param dict waiter_kwargs:
            A dictionary of keyword arguments to pass to the :py:func:`oci.wait_until` function. For example, you could pass ``max_interval_seconds`` or ``max_interval_seconds``
            as dictionary keys to modify how long the waiter function will wait between retries and the maximum amount of time it will wait
        """
        operation_result = self.client.create_rover_entitlement(create_rover_entitlement_details, **operation_kwargs)
        if not wait_for_states:
            return operation_result

        lowered_wait_for_states = [w.lower() for w in wait_for_states]
        wait_for_resource_id = operation_result.data.id

        try:
            waiter_result = oci.wait_until(
                self.client,
                self.client.get_rover_entitlement(wait_for_resource_id),
                evaluate_response=lambda r: getattr(r.data, 'lifecycle_state') and getattr(r.data, 'lifecycle_state').lower() in lowered_wait_for_states,
                **waiter_kwargs
            )
            result_to_return = waiter_result

            return result_to_return
        except Exception as e:
            raise oci.exceptions.CompositeOperationError(partial_results=[operation_result], cause=e)

    def delete_rover_entitlement_and_wait_for_state(self, rover_entitlement_id, wait_for_states=[], operation_kwargs={}, waiter_kwargs={}):
        """
        Calls :py:func:`~oci.rover.RoverEntitlementClient.delete_rover_entitlement` and waits for the :py:class:`~oci.rover.models.RoverEntitlement` acted upon
        to enter the given state(s).

        :param str rover_entitlement_id: (required)
            ID of the rover node or cluster entitlement

        :param list[str] wait_for_states:
            An array of states to wait on. These should be valid values for :py:attr:`~oci.rover.models.RoverEntitlement.lifecycle_state`

        :param dict operation_kwargs:
            A dictionary of keyword arguments to pass to :py:func:`~oci.rover.RoverEntitlementClient.delete_rover_entitlement`

        :param dict waiter_kwargs:
            A dictionary of keyword arguments to pass to the :py:func:`oci.wait_until` function. For example, you could pass ``max_interval_seconds`` or ``max_interval_seconds``
            as dictionary keys to modify how long the waiter function will wait between retries and the maximum amount of time it will wait
        """
        initial_get_result = self.client.get_rover_entitlement(rover_entitlement_id)
        operation_result = None
        try:
            operation_result = self.client.delete_rover_entitlement(rover_entitlement_id, **operation_kwargs)
        except oci.exceptions.ServiceError as e:
            if e.status == 404:
                return WAIT_RESOURCE_NOT_FOUND
            else:
                raise e

        if not wait_for_states:
            return operation_result

        lowered_wait_for_states = [w.lower() for w in wait_for_states]

        try:
            waiter_result = oci.wait_until(
                self.client,
                initial_get_result,
                evaluate_response=lambda r: getattr(r.data, 'lifecycle_state') and getattr(r.data, 'lifecycle_state').lower() in lowered_wait_for_states,
                succeed_on_not_found=True,
                **waiter_kwargs
            )
            result_to_return = waiter_result

            return result_to_return
        except Exception as e:
            raise oci.exceptions.CompositeOperationError(partial_results=[operation_result], cause=e)

    def update_rover_entitlement_and_wait_for_state(self, rover_entitlement_id, update_rover_entitlement_details, wait_for_states=[], operation_kwargs={}, waiter_kwargs={}):
        """
        Calls :py:func:`~oci.rover.RoverEntitlementClient.update_rover_entitlement` and waits for the :py:class:`~oci.rover.models.RoverEntitlement` acted upon
        to enter the given state(s).

        :param str rover_entitlement_id: (required)
            ID of the rover node or cluster entitlement

        :param oci.rover.models.UpdateRoverEntitlementDetails update_rover_entitlement_details: (required)
            The information to be updated.

        :param list[str] wait_for_states:
            An array of states to wait on. These should be valid values for :py:attr:`~oci.rover.models.RoverEntitlement.lifecycle_state`

        :param dict operation_kwargs:
            A dictionary of keyword arguments to pass to :py:func:`~oci.rover.RoverEntitlementClient.update_rover_entitlement`

        :param dict waiter_kwargs:
            A dictionary of keyword arguments to pass to the :py:func:`oci.wait_until` function. For example, you could pass ``max_interval_seconds`` or ``max_interval_seconds``
            as dictionary keys to modify how long the waiter function will wait between retries and the maximum amount of time it will wait
        """
        operation_result = self.client.update_rover_entitlement(rover_entitlement_id, update_rover_entitlement_details, **operation_kwargs)
        if not wait_for_states:
            return operation_result

        lowered_wait_for_states = [w.lower() for w in wait_for_states]
        wait_for_resource_id = operation_result.data.id

        try:
            waiter_result = oci.wait_until(
                self.client,
                self.client.get_rover_entitlement(wait_for_resource_id),
                evaluate_response=lambda r: getattr(r.data, 'lifecycle_state') and getattr(r.data, 'lifecycle_state').lower() in lowered_wait_for_states,
                **waiter_kwargs
            )
            result_to_return = waiter_result

            return result_to_return
        except Exception as e:
            raise oci.exceptions.CompositeOperationError(partial_results=[operation_result], cause=e)
