/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* XXX correct? */
#ifndef MYSQL_SERVER
#define MYSQL_SERVER
#endif

#include "mysql_priv.h"
#include "ha_falcon.h"
#include "StorageConnection.h"
#include "StorageTable.h"
#include "StorageTableShare.h"
#include "StorageHandler.h"
#include "CmdGen.h"
//#include "TimeTest.h"

typedef longlong	int64;

#include "ScaledBinary.h"
#include "BigInt.h"

#if defined(_WIN32) && MYSQL_VERSION_ID < 0x50100
#define IS_SLASH(c)	(c == '/' || c == '\\')
#else
#define IS_SLASH(c)	(c == '/')
#endif

#define FALCON_TEMPORARY	"/falcon_temporary"

//static const int MAX_FALCON_KEY_LENGTH = (MAX_INDEX_KEY_LENGTH * RUN / (RUN - 1));

static const char falcon_hton_name[] = "Falcon";
static const char *falcon_extensions[] = {
	".fts",
	".fl1",
	".fl2",
	NullS
};

static StorageHandler	*storageHandler;

ulonglong		falcon_min_record_memory;
ulonglong		falcon_max_record_memory; // = 20 * 1024 * 1024;
ulonglong		falcon_page_cache_size; // = 2097152;
uint			falcon_page_size; // = 4096;
int				falcon_log_mask;
my_bool			falcon_debug_server;
char*			falcon_log_dir;
FILE			*falcon_log_file;

static handler *falcon_create_handler(handlerton *hton,
                                      TABLE_SHARE *table, MEM_ROOT *mem_root)
{
	return new (mem_root) NfsStorageTable(hton, table);
}

handlerton *falcon_hton;

int NfsStorageTable::falcon_init(void *p)
{
	DBUG_ENTER("falcon_init");
	falcon_hton = (handlerton *)p;

	if (!storageHandler)
		storageHandler = getFalconStorageHandler();
		
	falcon_hton->state = have_falcon;
	falcon_hton->db_type = DB_TYPE_FALCON;
	falcon_hton->savepoint_offset = sizeof(void*);
	falcon_hton->close_connection = NfsStorageTable::closeConnection;
	falcon_hton->savepoint_set = NfsStorageTable::savepointSet;
	falcon_hton->savepoint_rollback = NfsStorageTable::savepointRollback;
	falcon_hton->savepoint_release = NfsStorageTable::savepointRelease;
	falcon_hton->commit = NfsStorageTable::commit;
	falcon_hton->rollback = NfsStorageTable::rollback;
	falcon_hton->create = falcon_create_handler;
	falcon_hton->drop_database  = NfsStorageTable::dropDatabase;
	falcon_hton->panic  = NfsStorageTable::panic;
	falcon_hton->alter_table_flags  = NfsStorageTable::alter_table_flags;

#ifdef XA_ENABLED
	falcon_hton->prepare = NfsStorageTable::prepare;
#endif

	falcon_hton->commit_by_xid = NfsStorageTable::commit_by_xid;
	falcon_hton->rollback_by_xid = NfsStorageTable::rollback_by_xid;

	falcon_hton->alter_tablespace = NfsStorageTable::alter_tablespace;
	//falcon_hton->show_status  = NfsStorageTable::show_status;
	falcon_hton->flags = HTON_NO_FLAGS;

	if (have_falcon != SHOW_OPTION_YES)
		DBUG_RETURN(0); // nothing else to do

	storageHandler->addNfsLogger(-1, NfsStorageTable::logger, NULL);

	if (falcon_debug_server)
		storageHandler->startNfsServer();

	//TimeTest timeTest;
	//timeTest.testScaled(16, 2, 100000);
	
	DBUG_RETURN(0);
}

int falcon_strnxfrm (void *cs, 
					 unsigned char *dst, uint dstlen,
                     const unsigned char *src, uint srclen)
{
	CHARSET_INFO *charset = (CHARSET_INFO*) cs;
	
	return charset->coll->strnxfrm(charset, dst, dstlen, src, srclen);
}

unsigned char falcon_pad_char (void *cs)
{
	return ((CHARSET_INFO*) cs)->pad_char;
}

// Return the actual number of characters in the string
// Note, this is not the number of characters with collatable weight.
uint falcon_strnchrlen(void *cs, const unsigned char *s, uint l)
{
	CHARSET_INFO *charset = (CHARSET_INFO*) cs;

	if (charset->mbmaxlen == 1)
		return l;

	uint chrCount = 0;
	uchar *ch = (uchar *) s;
	uchar *end = ch + l;

	while (ch < end)
		{
		int len = charset->cset->mbcharlen(charset, *ch);
		if (len == 0)
			break;  // invalid character.

		ch += len;
		chrCount++;
		}

	return chrCount;
}

// Determine how many bytes are required to store the output of cs->coll->strnxfrm()
// cs is how the source string is formatted.
// srcLen is the number of bytes in the source string.
// partialKey is the max key buffer size if not zero.
// bufSize is the ultimate maximum destSize.
// If the string is multibyte, strnxfrmlen expects srcLen to be 
// the maximum number of characters this can be.  Falcon wants to send 
// a number that represents the actual number of characters in the string
// so that the call to cs->coll->strnxfrm() will not pad.
uint falcon_strnxfrmlen(void *cs, const unsigned char *s, uint srcLen,
						int partialKey, int bufSize)
{
	CHARSET_INFO *charset = (CHARSET_INFO*) cs;
	uint chrLen = falcon_strnchrlen(cs, s, srcLen);
	int maxChrLen = partialKey ? min(chrLen, partialKey / charset->mbmaxlen) : chrLen;

	return min(charset->coll->strnxfrmlen(charset, maxChrLen * charset->mbmaxlen), (uint) bufSize);
}

// Return the number of bytes used in s to hold a certain number of characters.
// This partialKey is a byte length with charset->mbmaxlen figured in.
uint falcon_strntrunc(void *cs, int partialKey, const unsigned char *s, uint l)
{
	CHARSET_INFO *charset = (CHARSET_INFO*) cs;

	if ((charset->mbmaxlen == 1) || (partialKey == 0))
		return l;

	int charLimit = partialKey / charset->mbmaxlen;
	int charCount = 0;
	uchar *ch = (uchar *) s;
	uchar *end = ch + l;

	while ((ch < end) && charLimit)
		{
		int len = charset->cset->mbcharlen(charset, *ch);
		if (len == 0)
			break;  // invalid character.

		ch += len;
		charLimit--;
		}

	return ch - s;
}

int falcon_strnncoll(void *cs, const unsigned char *s1, uint l1, const unsigned char *s2, uint l2, char flag)
{
	CHARSET_INFO *charset = (CHARSET_INFO*) cs;

	return charset->coll->strnncoll(charset, s1, l1, s2, l2, flag);
}

int (*strnncoll)(struct charset_info_st *, const uchar *, uint, const uchar *, uint, my_bool);

void openFalconLogFile(const char *file)
{
	if (falcon_log_file)
		fclose(falcon_log_file);
	
	falcon_log_file = fopen(file, "a");
}

void closeFalconLogFile()
{
	if (falcon_log_file)
		{
		fclose(falcon_log_file);
		falcon_log_file = NULL;
		}
}

void flushFalconLogFile()
{
	if (falcon_log_file)
		fflush(falcon_log_file);
}

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

#ifndef DIG_PER_DEC1
typedef decimal_digit_t dec1;
typedef longlong      dec2;

#define DIG_PER_DEC1 9
#define DIG_MASK     100000000
#define DIG_BASE     1000000000
#define DIG_MAX      (DIG_BASE-1)
#define DIG_BASE2    ((dec2)DIG_BASE * (dec2)DIG_BASE)
#define ROUND_UP(X)  (((X)+DIG_PER_DEC1-1)/DIG_PER_DEC1)
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

NfsStorageTable::NfsStorageTable(handlerton *hton, st_table_share *table_arg)
  : handler(hton, table_arg) //,
  //storageTable(NULL), storageConnection(NULL), storageShare(NULL),
  //share(table_arg), lastRecord(-1), mySqlThread(NULL), activeBlobs(NULL),
  //freeBlobs(NULL), dbName(NULL)
{
	ref_length = sizeof(lastRecord);
	stats.records = 1000;
	stats.data_file_length = 10000;
	tableLocked = false;
	storageTable = NULL;
	storageConnection = NULL;
	storageShare = NULL;
	share = table_arg;
	lastRecord = -1;
	mySqlThread = NULL;
	activeBlobs = NULL;
	freeBlobs = NULL;
	dbName = NULL;
	
	if (table_arg)
		{
		recordLength = table_arg->reclength;
		
		switch (table_arg->tmp_table)
			{
			case INTERNAL_TMP_TABLE:
				tempTable = true;
				break;
			
			default:
				tempTable = false;
			}
		}
}


NfsStorageTable::~NfsStorageTable(void)
{
	if (activeBlobs)
		freeActiveBlobs();

	for (StorageBlob *blob; (blob = freeBlobs); )
		{
		freeBlobs = blob->next;
		delete blob;
		}

	if (storageTable)
		{
		storageTable->deleteStorageTable();
		storageTable = NULL;
		}

	if (storageConnection)
		{
		storageConnection->release();
		storageConnection = NULL;
		}

	delete [] dbName;
}

bool NfsStorageTable::show_status(handlerton* hton, THD* thd, stat_print_fn* print, enum ha_stat_type stat)
{
	const char *status = storageHandler->getEngineStatus(thd);
	(print)(thd, "", 0, "", 0, status, 40);

	return false;
}

int NfsStorageTable::rnd_init(bool scan)
{
	DBUG_ENTER("NfsStorageTable::rnd_init");
	nextRecord = 0;
	lastRecord = -1;

	DBUG_RETURN(0);
}


int NfsStorageTable::open(const char *name, int mode, uint test_if_locked)
{
	DBUG_ENTER("NfsStorageTable::open");
	int tableType = storageHandler->isTempTable(name);
	
	if (tableType >= 0)
		tempTable = tableType > 0;
	else
		tempTable = (table) ? table->s->tmp_table : 0;
	
	if (!storageTable)
		{
		if (!storageConnection)
			{
			if (!(storageConnection = storageHandler->getStorageConnection(getDbName(name), mySqlThread, OpenDatabase)))
				DBUG_RETURN(HA_ERR_NO_CONNECTION);

			if (mySqlThread)
				{
				((StorageConnection**) (mySqlThread->ha_data))[falcon_hton->slot] = storageConnection;
				storageConnection->addRef();
				}
			}

		storageTable = storageConnection->getStorageTable(name, sizeof(THR_LOCK), tempTable);
		storageTable->localTable = this;
		storageShare = storageTable->share;

		if (!storageShare->initialized)
			{
			storageShare->lock(true);
			
			if (!storageShare->initialized)
				{
				thr_lock_init((THR_LOCK *)storageShare->impure);
				storageShare->setTablePath(name, tempTable);
				storageShare->initialized = true;
				}
			
			// Register any collations used
			
			uint fieldCount = (table) ? table->s->fields : 0;
			
			for (uint n = 0; n < fieldCount; ++n)
				{
				Field *field = table->field[n];
				CHARSET_INFO *charset = field->charset();
				
				if (charset)
					storageShare->registerCollation(charset->name, charset);
				}

			storageShare->unlock();
			}
		}

	int ret = storageTable->open();

	if (ret)
		DBUG_RETURN(error(ret));

	thr_lock_data_init((THR_LOCK *)storageShare->impure, &lockData, NULL);
	setIndexes();

	DBUG_RETURN(0);
}

int NfsStorageTable::close(void)
{
  DBUG_ENTER("NfsStorageTable::close");
  DBUG_RETURN(0);
}

int NfsStorageTable::rnd_next(byte *buf)
{
	DBUG_ENTER("NfsStorageTable::rnd_next");
	statistic_increment(table->in_use->status_var.ha_read_rnd_next_count, &LOCK_status);

	if (activeBlobs)
		freeActiveBlobs();

	lastRecord = storageTable->next(nextRecord);

	if (lastRecord < 0)
		{
		lastRecord = -1;
		table->status = STATUS_NOT_FOUND;
		DBUG_RETURN(HA_ERR_END_OF_FILE);
		}
	else
		table->status = 0;

	decodeRecord(buf);
	nextRecord = lastRecord + 1;

	DBUG_RETURN(0);
}


int NfsStorageTable::rnd_pos(byte *buf, byte *pos)
{
	int recordNumber;
	DBUG_ENTER("NfsStorageTable::rnd_pos");
	statistic_increment(table->in_use->status_var.ha_read_rnd_next_count,
	                    &LOCK_status);

	memcpy(&recordNumber, pos, sizeof(recordNumber));

	if (activeBlobs)
		freeActiveBlobs();

	int ret = storageTable->fetch(recordNumber);

	if (ret)
		{
		table->status = STATUS_NOT_FOUND;
		DBUG_RETURN(error(ret));
		}

	lastRecord = recordNumber;
	decodeRecord(buf);
	table->status = 0;

	DBUG_RETURN(0);
}

void NfsStorageTable::position(const byte *record)
{
  DBUG_ENTER("NfsStorageTable::position");
  memcpy(ref, &lastRecord, sizeof(lastRecord));
  DBUG_VOID_RETURN;
}

int NfsStorageTable::info(uint what)
{
	DBUG_ENTER("NfsStorageTable::info");

	if (what & HA_STATUS_VARIABLE)
		{
		/* XXX Someday I should track this. Someday... */
		}

	if (what & HA_STATUS_AUTO)
		stats.auto_increment_value = storageShare->getSequenceValue(0);

	if (what & HA_STATUS_ERRKEY)
		errkey = indexErrorId;

	DBUG_RETURN(0);
}

uint8 NfsStorageTable::table_cache_type(void)
{
	return HA_CACHE_TBL_TRANSACT;
}

const char *NfsStorageTable::table_type(void) const
{
	DBUG_ENTER("NfsStorageTable::table_type");
	DBUG_RETURN(falcon_hton_name);
}


const char **NfsStorageTable::bas_ext(void) const
{
	DBUG_ENTER("NfsStorageTable::bas_ext");
	DBUG_RETURN(falcon_extensions);
}


ulonglong NfsStorageTable::table_flags(void) const
{
	DBUG_ENTER("NfsStorageTable::table_flags");
	DBUG_RETURN(HA_REC_NOT_IN_SEQ | HA_NULL_IN_KEY | HA_AUTO_PART_KEY |
	            HA_PARTIAL_COLUMN_READ | HA_CAN_GEOMETRY);
}


ulong NfsStorageTable::index_flags(uint idx, uint part, bool all_parts) const
{
	DBUG_ENTER("NfsStorageTable::index_flags");
	DBUG_RETURN(HA_READ_RANGE | HA_KEY_SCAN_NOT_ROR);
}


int NfsStorageTable::create(const char *mySqlName, TABLE *form,
                            HA_CREATE_INFO *info)
{
	DBUG_ENTER("NfsStorageTable::create");
	tempTable = (info->options & HA_LEX_CREATE_TMP_TABLE) ? true : false;
	OpenOption openOption = (tempTable) ? OpenTemporaryDatabase : OpenOrCreateDatabase;

	if (storageTable)
		{
		storageTable->deleteStorageTable();
		storageTable = NULL;
		}

	if (!storageConnection &&
		!(storageConnection = storageHandler->getStorageConnection(getDbName(mySqlName), mySqlThread, openOption)))
		DBUG_RETURN(HA_ERR_NO_CONNECTION);

	storageTable = storageConnection->getStorageTable(mySqlName, sizeof(THR_LOCK), tempTable);
	storageTable->localTable = this;
	storageShare = storageTable->share;
	storageShare->setTablePath(mySqlName, tempTable);

	int ret;
	int64 incrementValue = 0;
	uint n;
	CmdGen gen;
	const char *tableName = storageTable->getName();
	const char *schemaName = storageTable->getSchemaName();
	gen.gen("create table %s.\"%s\" (\n", schemaName, tableName);
	const char *sep = "";
	char nameBuffer[129];

	for (n = 0; n < form->s->fields; ++n)
		{
		Field *field = form->field[n];
		CHARSET_INFO *charset = field->charset();
		
		if (charset)
			storageShare->registerCollation(charset->name, charset);

		storageShare->cleanupFieldName(field->field_name, nameBuffer,
										sizeof(nameBuffer));
		gen.gen("%s  \"%s\" ", sep, nameBuffer);
		genType(field, &gen);

		if (!field->maybe_null())
			gen.gen(" not null");
			
		sep = ",\n";
		}

	if (form->found_next_number_field) // && form->s->next_number_key_offset == 0)
		{
		incrementValue = info->auto_increment_value;

		if (incrementValue == 0)
			incrementValue = 1;
		}

	if (form->s->primary_key < form->s->keys)
		{
		KEY *key = form->key_info + form->s->primary_key;
		gen.gen(",\n  primary key ");
		genKeyFields(key, &gen);
		}

	gen.gen (")");

	DBUG_PRINT("info",("incrementValue = %d", incrementValue));

	if ((ret = storageTable->create(gen.getString(), incrementValue)))
		DBUG_RETURN(error(ret));

	for (n = 0; n < form->s->keys; ++n)
		if (n != form->s->primary_key)
			createIndex(schemaName, tableName, form->key_info + n, n);

	DBUG_RETURN(0);
}

int NfsStorageTable::add_index(TABLE* table_arg, KEY* key_info, uint num_of_keys)
{
	DBUG_ENTER("NfsStorageTable::add_index");
	int ret = createIndex(storageTable->getSchemaName(), storageTable->getName(), key_info, table_arg->s->keys);
	
	DBUG_RETURN(ret);
}

int NfsStorageTable::createIndex(const char *schemaName, const char *tableName,
                                 KEY *key, int indexNumber)
{
	CmdGen gen;
	const char *unique = (key->flags & HA_NOSAME) ? "unique " : "";
	gen.gen("create %sindex \"%s$%d\" on %s.\"%s\" ", unique, tableName,
	        indexNumber, schemaName, tableName);
	genKeyFields(key, &gen);
	const char *sql = gen.getString();

	return storageTable->share->createIndex(storageConnection, key->name, sql);
}

uint NfsStorageTable::alter_table_flags(uint flags)
{
	if (flags & ALTER_DROP_PARTITION)
		return 0;

	return HA_ONLINE_ADD_INDEX | HA_ONLINE_ADD_UNIQUE_INDEX;
}

bool NfsStorageTable::check_if_incompatible_data(HA_CREATE_INFO* create_info, uint table_changes)
{
	if (true || create_info->auto_increment_value != 0)
		return COMPATIBLE_DATA_NO;
		
	return COMPATIBLE_DATA_YES;
}

THR_LOCK_DATA **NfsStorageTable::store_lock(THD *thd, THR_LOCK_DATA **to,
                                            enum thr_lock_type lock_type)
{
	DBUG_ENTER("NfsStorageTable::store_lock");

	if (lock_type != TL_IGNORE && lockData.type == TL_UNLOCK)
		{
		/*
		  Here is where we get into the guts of a row level lock.
		  If TL_UNLOCK is set
		  If we are not doing a LOCK TABLE or DISCARD/IMPORT
		  TABLESPACE, then allow multiple writers
		*/

		if ( (lock_type >= TL_WRITE_CONCURRENT_INSERT && lock_type <= TL_WRITE) &&
		     !thd->in_lock_tables)
			lock_type = TL_WRITE_ALLOW_WRITE;

		/*
		  In queries of type INSERT INTO t1 SELECT ... FROM t2 ...
		  MySQL would use the lock TL_READ_NO_INSERT on t2, and that
		  would conflict with TL_WRITE_ALLOW_WRITE, blocking all inserts
		  to t2. Convert the lock to a normal read lock to allow
		  concurrent inserts to t2.
		*/

		#ifdef XXX_TALK_TO_SERGEI
		if (lock_type == TL_READ_NO_INSERT && !thd->in_lock_tables)
			lock_type = TL_READ;
		#endif


		lockData.type = lock_type;
		}

	*to++ = &lockData;
	DBUG_RETURN(to);
}


int NfsStorageTable::delete_table(const char *tableName)
{
	DBUG_ENTER("NfsStorageTable::delete_table");
	tempTable = storageHandler->isTempTable(tableName) > 0;
	
	if (!storageConnection &&
		!(storageConnection = storageHandler->getStorageConnection(getDbName(tableName), NULL, OpenDatabase)))
			DBUG_RETURN(HA_ERR_NO_CONNECTION);

	if (!storageTable)
		if (!(storageTable = storageConnection->getStorageTable(tableName, sizeof(THR_LOCK), tempTable)))
			DBUG_RETURN(0);

	if (!storageShare && storageTable)
		storageShare = storageTable->share;

	if (storageShare)
		{
		storageShare->lock(true);

		if (storageShare->initialized)
			{
			thr_lock_delete((THR_LOCK*) storageShare->impure);
			storageShare->initialized = false;
			//DBUG_ASSERT(false);
			}

		storageShare->unlock();
		}

	int res = storageTable->deleteTable();
	storageTable->deleteStorageTable();
	storageTable = NULL;

	DBUG_RETURN(error(res));
}

uint NfsStorageTable::max_supported_keys(void) const
{
	DBUG_ENTER("NfsStorageTable::max_supported_keys");
	DBUG_RETURN(MAX_KEY);
}


int NfsStorageTable::write_row(byte *buff)
{
	DBUG_ENTER("NfsStorageTable::write_row");
	statistic_increment(table->in_use->status_var.ha_write_count,&LOCK_status);

	/* If we have a timestamp column, update it to the current time */

	if (table->timestamp_field_type & TIMESTAMP_AUTO_SET_ON_INSERT)
		table->timestamp_field->set_time();

	/*
		If we have an auto_increment column and we are writing a changed row
		or a new row, then update the auto_increment value in the record.
	*/

	if (table->next_number_field && buff == table->record[0])
		{
		update_auto_increment();

		/*
		   If the new value is less than the current highest value, it will be
		   ignored by setSequenceValue().
		*/

		int code = storageShare->setSequenceValue(table->next_number_field->val_int());
		
		if (code)
			DBUG_RETURN(error(code));
		}

	encodeRecord(buff, false);
	lastRecord = storageTable->insert();

	if (lastRecord < 0)
		{
		int code = lastRecord >> StoreErrorIndexShift;
		indexErrorId = (lastRecord & StoreErrorIndexMask) - 1;
		
		DBUG_RETURN(error(code));
		}

	DBUG_RETURN(0);
}


int NfsStorageTable::update_row(const byte* oldData, byte* newData)
{
	DBUG_ENTER("NfsStorageTable::update_row");
	DBUG_ASSERT (lastRecord >= 0);

	statistic_increment(table->in_use->status_var.ha_update_count,&LOCK_status);

	/* If we have a timestamp column, update it to the current time */
	
	if (table->timestamp_field_type & TIMESTAMP_AUTO_SET_ON_UPDATE)
		table->timestamp_field->set_time();

	/* If we have an auto_increment column, update the sequence value.  */

	Field *autoInc = table->found_next_number_field;
	
	if ((autoInc) && bitmap_is_set(table->read_set, autoInc->field_index))
		{
		int code = storageShare->setSequenceValue(autoInc->val_int());
		
		if (code)
			DBUG_RETURN(error(code));
		}

	encodeRecord(newData, true);

	int ret = storageTable->updateRow(lastRecord);

	if (ret)
		{
		int code = ret >> StoreErrorIndexShift;
		indexErrorId = (ret & StoreErrorIndexMask) - 1;
		DBUG_RETURN(error(code));
		}

	DBUG_RETURN(0);
}


int NfsStorageTable::delete_row(const byte* buf)
{
	DBUG_ENTER("NfsStorageTable::delete_row");
	DBUG_ASSERT (lastRecord >= 0);
	statistic_increment(table->in_use->status_var.ha_delete_count,&LOCK_status);

	if (activeBlobs)
		freeActiveBlobs();

	int ret = storageTable->deleteRow(lastRecord);

	if (ret < 0)
		DBUG_RETURN(error(ret));

	lastRecord = -1;

	DBUG_RETURN(0);
}


int NfsStorageTable::commit(handlerton *hton, THD* thd, bool all)
{
	DBUG_ENTER("NfsStorageTable::commit");

	if (all || !(thd->options & (OPTION_NOT_AUTOCOMMIT | OPTION_BEGIN)))
		storageHandler->commit(thd);
	else
		storageHandler->releaseVerb(thd);
	
	DBUG_RETURN(0);
}

int NfsStorageTable::prepare(handlerton* hton, THD* thd, bool all)
{
	DBUG_ENTER("NfsStorageTable::prepare");

	if (all || !(thd->options & (OPTION_NOT_AUTOCOMMIT | OPTION_BEGIN)))
		storageHandler->prepare(thd, sizeof(thd->transaction.xid), (const unsigned char*) &thd->transaction.xid);
	else
		storageHandler->releaseVerb(thd);
	
	DBUG_RETURN(0);
}

int NfsStorageTable::rollback(handlerton *hton, THD *thd, bool all)
{
	DBUG_ENTER("NfsStorageTable::rollback");

	if (all || !(thd->options & (OPTION_NOT_AUTOCOMMIT | OPTION_BEGIN)))
		storageHandler->rollback(thd);
	else
		storageHandler->rollbackVerb(thd);
	
	DBUG_RETURN(0);
}

int NfsStorageTable::commit_by_xid(handlerton* hton, XID* xid)
{
	DBUG_ENTER("NfsStorageTable::commit_by_xid");
	int ret = storageHandler->commitByXID(sizeof(XID), (const unsigned char*) xid);
	DBUG_RETURN(ret);
}

int NfsStorageTable::rollback_by_xid(handlerton* hton, XID* xid)
{
	DBUG_ENTER("NfsStorageTable::rollback_by_xid");
	int ret = storageHandler->rollbackByXID(sizeof(XID), (const unsigned char*) xid);
	DBUG_RETURN(ret);
}

const COND* NfsStorageTable::cond_push(const COND* cond)
{
  DBUG_ENTER("NfsStorageTable::cond_push");
  char buff[256];
  String str(buff,(uint32) sizeof(buff), system_charset_info);
  str.length(0);
  Item *cond_ptr= (COND *)cond;
  cond_ptr->print(&str);
  str.append('\0');
  DBUG_PRINT("NfsStorageTable::cond_push", ("%s", str.ptr()));
  DBUG_RETURN(0);
}

void NfsStorageTable::startTransaction(void)
{
	threadSwitch(table->in_use);

	if (!storageConnection->transactionActive)
		{
		storageConnection->startTransaction(mySqlThread->variables.tx_isolation);
		trans_register_ha(mySqlThread, true, falcon_hton);
		}

	switch (mySqlThread->variables.tx_isolation)
		{
		case ISO_READ_UNCOMMITTED:
			error(StorageWarningReadUncommitted);
			break;
			
		case ISO_SERIALIZABLE:
			error(StorageWarningSerializable);
			break;
		}
}


int NfsStorageTable::savepointSet(handlerton *hton, THD *thd, void *savePoint)
{
	return storageHandler->savepointSet(thd, savePoint);
}


int NfsStorageTable::savepointRollback(handlerton *hton, THD *thd, void *savePoint)
{
	return storageHandler->savepointRollback(thd, savePoint);
}


int NfsStorageTable::savepointRelease(handlerton *hton, THD *thd, void *savePoint)
{
	return storageHandler->savepointRelease(thd, savePoint);
}


int NfsStorageTable::index_read(byte *buf, const byte *keyBytes, uint key_len,
                                enum ha_rkey_function find_flag)
{
	DBUG_ENTER("NfsStorageTable::index_read");
	int ret, which = 0;
	statistic_increment(table->in_use->status_var.ha_read_key_count,
						&LOCK_status);

	// XXX This needs to be revisited
	switch(find_flag) 
		{
		case HA_READ_KEY_EXACT:
			which = UpperBound | LowerBound;
			break;
			
		case HA_READ_KEY_OR_NEXT:
			storageTable->setPartialKey();
			which = LowerBound;
			break;
			
		case HA_READ_AFTER_KEY:     // ???
			if (!storageTable->isKeyNull((const unsigned char*) keyBytes, key_len))
				which = LowerBound;
	        
			break;

		case HA_READ_BEFORE_KEY:    // ???
		case HA_READ_PREFIX_LAST_OR_PREV: // ???
		case HA_READ_PREFIX_LAST:   // ???
		case HA_READ_PREFIX:
		case HA_READ_KEY_OR_PREV:
		default:
			DBUG_RETURN(HA_ERR_UNSUPPORTED);
		}

	const unsigned char *key = (const unsigned char*) keyBytes;
	
	if (which)
		if ((ret = storageTable->setIndexBound(key, key_len, which)))
			DBUG_RETURN(error(ret));

	if ((ret = storageTable->indexScan()))
		DBUG_RETURN(error(ret));

	nextRecord = 0;
	
	for (;;)
		{
		int ret = index_next(buf);
		
		if (ret)
			DBUG_RETURN(ret);
		
		int comparison = storageTable->compareKey(key, key_len);
		
		if ((which & LowerBound) && comparison < 0)
			continue;
		
		if ((which & UpperBound) && comparison > 0)
			continue;
		
		DBUG_RETURN(0);
		}
}


int NfsStorageTable::index_init(uint idx, bool sorted)
{
	DBUG_ENTER("NfsStorageTable::index_init");
	active_index = idx;
	nextRecord = 0;
	haveStartKey = false;
	haveEndKey = false;
	
	if (!storageTable->setIndex(idx))
		DBUG_RETURN(0);

	StorageIndexDesc indexDesc;
	getKeyDesc(table->key_info + idx, &indexDesc);

	if (idx == table->s->primary_key)
		indexDesc.primaryKey = true;

	int ret = storageTable->setIndex(table->s->keys, idx, &indexDesc);

	if (ret)
		DBUG_RETURN(error(ret));

	DBUG_RETURN(0);
}


int NfsStorageTable::index_end(void)
{
	DBUG_ENTER("NfsStorageTable::index_end");
	storageTable->indexEnd();
	DBUG_RETURN(0);
}

ha_rows NfsStorageTable::record_in_range(uint index, key_range *lower,
                                         key_range *upper)
{
	DBUG_ENTER("NfsStorageTable::record_in_range");
	ha_rows ret = ha_rows(); // TODO
	DBUG_RETURN(ret);
}


void NfsStorageTable::getKeyDesc(KEY *keyInfo, StorageIndexDesc *indexInfo)
{
	int numberKeys = keyInfo->key_parts;
	indexInfo->numberSegments = numberKeys;
	indexInfo->name = keyInfo->name;
	indexInfo->unique = (keyInfo->flags & HA_NOSAME);
	indexInfo->primaryKey = false;

	for (int n = 0; n < numberKeys; ++n) 
		{
		StorageSegment *segment = indexInfo->segments + n;
		KEY_PART_INFO *part = keyInfo->key_part + n;
		segment->offset = part->offset;
		segment->length = part->length;
		segment->type = part->field->key_type();
		segment->nullBit = part->null_bit;
		segment->isUnsigned = ((Field_num*) part->field)->unsigned_flag;

		switch (segment->type)
			{
			case HA_KEYTYPE_TEXT:
			case HA_KEYTYPE_VARTEXT1:
			case HA_KEYTYPE_VARTEXT2:
			case HA_KEYTYPE_VARBINARY1:
			case HA_KEYTYPE_VARBINARY2:
				segment->mysql_charset = part->field->charset();
				break;
			
			default:
				segment->mysql_charset = NULL;
			}
		}
}


int NfsStorageTable::rename_table(const char *from, const char *to)
{
	DBUG_ENTER("NfsStorageTable::rename_table");
	tempTable = storageHandler->isTempTable(from) > 0;
	table = 0; // XXX hack?
	int ret = open(from, 0, 0);

	if (ret)
		DBUG_RETURN(ret);

	DBUG_RETURN(storageShare->renameTable(storageConnection, to));
}


double NfsStorageTable::read_time(uint index, uint ranges, ha_rows rows)
{
	DBUG_ENTER("NfsStorageTable::read_time");
	DBUG_RETURN(rows2double(rows / 3));
}


int NfsStorageTable::read_range_first(const key_range *start_key,
                                      const key_range *end_key,
                                      bool eq_range_arg, bool sorted)
{
	DBUG_ENTER("NfsStorageTable::read_range_first");
	storageTable->clearIndexBounds();
	haveStartKey = false;
	haveEndKey = false;
	
	if (start_key && !storageTable->isKeyNull((const unsigned char*) start_key->key, start_key->length))
		{
		haveStartKey = true;
		startKey = *start_key;
		
		if (start_key->flag == HA_READ_KEY_OR_NEXT)
			storageTable->setPartialKey();
			
		int ret = storageTable->setIndexBound((const unsigned char*) start_key->key,
												start_key->length, LowerBound);
		if (ret)
			DBUG_RETURN(ret);
		}

	if (end_key)
		{
		int ret = storageTable->setIndexBound((const unsigned char*) end_key->key,
												end_key->length, UpperBound);
		if (ret)
			DBUG_RETURN(ret);
		
		if (end_key->flag == HA_READ_AFTER_KEY)
			storageTable->setPartialKey();
		}

	storageTable->indexScan();
	nextRecord = 0;
	lastRecord = -1;
	eq_range = eq_range_arg;
	end_range = 0;

	if (end_key)
		{
		haveEndKey = true;
		endKey = *end_key;
		end_range = &save_end_range;
		save_end_range = *end_key;
		key_compare_result_on_equal = ((end_key->flag == HA_READ_BEFORE_KEY) ? 1 :
										(end_key->flag == HA_READ_AFTER_KEY) ? -1 :
										0);
		}
	
	range_key_part = table->key_info[active_index].key_part;
	
	for (;;)
		{
		int result = index_next(table->record[0]);

		if (result)
			{
			if (result == HA_ERR_KEY_NOT_FOUND)
				result = HA_ERR_END_OF_FILE;
			    
			table->status = result;
			DBUG_RETURN(result);
			}
		
		DBUG_RETURN(0);
		}
}


int NfsStorageTable::index_next(byte *buf)
{
	DBUG_ENTER("NfsStorageTable::index_next");
	statistic_increment(table->in_use->status_var.ha_read_next_count, &LOCK_status);

	if (activeBlobs)
		freeActiveBlobs();

	for (;;)
		{
		lastRecord = storageTable->nextIndexed(nextRecord);

		if (lastRecord < 0)
			{
			lastRecord = -1;
			table->status = STATUS_NOT_FOUND;
			
			DBUG_RETURN(HA_ERR_END_OF_FILE);
			}

		nextRecord = lastRecord + 1;
		
		if (haveStartKey)
			{
			int n = storageTable->compareKey((const unsigned char*) startKey.key, startKey.length);
			
			if (n < 0 || (n == 0 && startKey.flag == HA_READ_AFTER_KEY))
				continue;
			}
			
		if (haveEndKey)
			{
			int n = storageTable->compareKey((const unsigned char*) endKey.key, endKey.length);
			
			if (n > 0 || (n == 0 && endKey.flag == HA_READ_BEFORE_KEY))
				continue;
			}
		
		decodeRecord(buf);
		table->status = 0;

		DBUG_RETURN(0);
		}
}


int NfsStorageTable::index_next_same(byte *buf, const byte *key, uint key_len)
{
	DBUG_ENTER("NfsStorageTable::index_next_same");
	statistic_increment(table->in_use->status_var.ha_read_next_count,
						&LOCK_status);
	
	for (;;)
		{
		int ret = index_next(buf);
		
		if (ret)
			DBUG_RETURN(ret);
		
		int comparison = storageTable->compareKey((const unsigned char*) key, key_len);
		
		if (comparison == 0)
			DBUG_RETURN(0);
		}
}


double NfsStorageTable::scan_time(void)
{
	DBUG_ENTER("NfsStorageTable::scan_time");
	DBUG_RETURN(stats.records * 1000);
}


bool NfsStorageTable::threadSwitch(THD* newThread)
{
	if (newThread == mySqlThread)
		return false;

	storageConnection = storageHandler->getStorageConnection(getDbName(table->s->path.str),
	                                        newThread, OpenDatabase);

	if (storageTable)
		storageTable->setConnection(storageConnection);
	else
		storageTable = storageConnection->getStorageTable(storageShare);

	mySqlThread = newThread;

	return true;
}


int NfsStorageTable::threadSwitchError(void)
{
	return 1;
}


int NfsStorageTable::error(int storageError)
{
	DBUG_ENTER("NfsStorageTable::error");

	if (storageError == 0)
		{
		DBUG_PRINT("info", ("returning 0"));
		DBUG_RETURN(0);
		}

	switch (storageError) 
		{
		case StorageErrorDupKey:
			DBUG_PRINT("info", ("StorageErrorDupKey"));
			DBUG_RETURN(HA_ERR_FOUND_DUPP_KEY);

		case StorageErrorDeadlock:
			DBUG_PRINT("info", ("StorageErrorDeadlock"));
			//DBUG_RETURN(HA_ERR_LOCK_DEADLOCK);
			DBUG_RETURN(200 - storageError);

		case StorageErrorRecordNotFound:
			DBUG_PRINT("info", ("StorageErrorRecordNotFound"));
			DBUG_RETURN(HA_ERR_KEY_NOT_FOUND);

		case StorageErrorTableNotFound:
			DBUG_PRINT("info", ("StorageErrorTableNotFound"));
			DBUG_RETURN(HA_ERR_NO_SUCH_TABLE);

		case StorageErrorNoIndex:
			DBUG_PRINT("info", ("StorageErrorNoIndex"));
			DBUG_RETURN(HA_ERR_WRONG_INDEX);

		case StorageErrorBadKey:
			DBUG_PRINT("info", ("StorageErrorBadKey"));
			DBUG_RETURN(HA_ERR_WRONG_INDEX);

		case StorageErrorTableExits:
			DBUG_PRINT("info", ("StorageErrorTableExits"));
			DBUG_RETURN(HA_ERR_TABLE_EXIST);

		case StorageErrorUpdateConflict:
			DBUG_PRINT("info", ("StorageErrorUpdateConflict"));
			DBUG_RETURN(HA_ERR_RECORD_CHANGED);

		case StorageErrorUncommittedUpdates:
			DBUG_PRINT("info", ("StorageErrorUncommittedUpdates"));
			DBUG_RETURN(HA_ERR_TABLE_EXIST);

		case StorageErrorNoSequence:
			DBUG_PRINT("info", ("StorageErrorNoSequence"));

			if (storageConnection)
				storageConnection->setErrorText("no sequenced defined for autoincrement operation");

			DBUG_RETURN(200 - storageError);

		case StorageErrorTruncation:
			DBUG_PRINT("info", ("StorageErrorTruncation"));
			DBUG_RETURN(HA_ERR_TO_BIG_ROW);

		case StorageWarningSerializable:
			push_warning_printf(current_thd, MYSQL_ERROR::WARN_LEVEL_WARN,
			                    ER_CANT_CHANGE_TX_ISOLATION,
			                    "Falcon does not support SERIALIZABLE ISOLATION, using REPEATABLE READ instead.");
			DBUG_RETURN(0);

		case StorageWarningReadUncommitted:
			push_warning_printf(current_thd, MYSQL_ERROR::WARN_LEVEL_WARN,
			                    ER_CANT_CHANGE_TX_ISOLATION,
			                    "Falcon does not support READ UNCOMMITTED ISOLATION, using REPEATABLE READ instead.");
			DBUG_RETURN(0);

		default:
			DBUG_PRINT("info", ("Unknow Falcon Error"));
			DBUG_RETURN(200 - storageError);
		}

	DBUG_RETURN(storageError);
}

int NfsStorageTable::start_stmt(THD *thd, thr_lock_type lock_type)
{
	DBUG_ENTER("NfsStorageTable::start_stmt");
	StorageConnection *storageConn = storageHandler->getStorageConnection(NULL, thd, OpenDatabase);

	if (storageConn)
		{
		if (storageConn->markVerb())
			trans_register_ha(thd, FALSE, falcon_hton);

		storageConn->release();
		}

	DBUG_RETURN(0);
}


int NfsStorageTable::external_lock(THD *thd, int lock_type)
{
	DBUG_ENTER("NfsStorageTable::external_lock");
	threadSwitch(thd);

	if (lock_type == F_UNLCK)
		{
		if (!(thd->options & (OPTION_NOT_AUTOCOMMIT | OPTION_BEGIN)))
			storageConnection->endImplicitTransaction();
		else
			storageConnection->releaseVerb();

		if (storageTable)
			{
			storageTable->clearRecord();
			storageTable->clearBitmap();
			}
		}
	else
		{
		switch (thd->lex->sql_command)
			{
			case SQLCOM_ALTER_TABLE:
			case SQLCOM_DROP_INDEX:
			case SQLCOM_CREATE_INDEX:
				{
				int ret = storageTable->alterCheck();
				
				if (ret)
					DBUG_RETURN(error(ret));
				}
				break;
			
			default:
				break;
			}

		if ((thd->options & (OPTION_NOT_AUTOCOMMIT | OPTION_BEGIN)))
			{
			if (storageConnection->startTransaction(thd->variables.tx_isolation))
				trans_register_ha(thd, true, falcon_hton);
				
			if (storageConnection->markVerb())
				trans_register_ha(thd, false, falcon_hton);
			}
		else
			{
			if (storageConnection->startImplicitTransaction(thd->variables.tx_isolation))
				trans_register_ha(thd, false, falcon_hton);
			}
			
		switch (mySqlThread->variables.tx_isolation)
			{
			case ISO_READ_UNCOMMITTED:
				error(StorageWarningReadUncommitted);
				break;
				
			case ISO_SERIALIZABLE:
				error(StorageWarningSerializable);
				break;
			}
		}

	DBUG_RETURN(0);
}


void NfsStorageTable::get_auto_increment(ulonglong offset, ulonglong increment,
                                         ulonglong nb_desired_values,
                                         ulonglong *first_value,
                                         ulonglong *nb_reserved_values)
{
	DBUG_ENTER("NfsStorageTable::get_auto_increment");
	*first_value = storageShare->getSequenceValue(1);
	*nb_reserved_values = 1;

	DBUG_VOID_RETURN;
}

int NfsStorageTable::reset_auto_increment(ulonglong value)
{
	return handler::reset_auto_increment(value);
}

const char *NfsStorageTable::index_type(uint key_number)
{
  DBUG_ENTER("NfsStorageTable::index_type");
  DBUG_RETURN("BTREE");
}

void NfsStorageTable::dropDatabase(handlerton *hton, char *path)
{
	DBUG_ENTER("NfsStorageTable::dropDatabase");
	char pathname[FN_REFLEN];
	char *q = pathname;

	for (const char *p = path; *p;)
		{
		char c = *p++;
		*q++ = (IS_SLASH(c)) ? '/' : c;
		}

	if (q[-1] == '/')
		--q;

	*q = 0;

	StorageConnection *storageConn = storageHandler->getStorageConnection(pathname, NULL, OpenDatabase);
	
	if (storageConn)
		{
		storageConn->dropDatabase();
		storageHandler->databaseDropped(storageConn);
		storageConn->release();
		}

	DBUG_VOID_RETURN;
}


void NfsStorageTable::freeActiveBlobs(void)
{
	for (StorageBlob *blob; (blob = activeBlobs); )
		{
		activeBlobs = blob->next;
		storageTable->freeBlob(blob);
		blob->next = freeBlobs;
		freeBlobs = blob;
		}
}


void NfsStorageTable::shutdown(handlerton *htons)
{
	storageHandler->shutdownHandler();
}


int NfsStorageTable::panic(handlerton* hton, ha_panic_function flag)
{
	storageHandler->shutdownHandler();
	
	return 0;
}


const char *NfsStorageTable::getDbName(const char *tableName)
{
	if (dbName)
		return dbName;

	const char *slash = NULL;
	const char *p;

	for (p = tableName; *p; p++)
		if (IS_SLASH(*p))
			slash = p;

	if (!slash)
		slash = p;

	int len = slash - tableName + 1;
	
	if (tempTable)
		len += sizeof(FALCON_TEMPORARY);
		
	char *q = new char[len];
	dbName = q;

	for (p = tableName; p < slash; )
		{
		char c = *p++;
		*q++ = (IS_SLASH(c)) ? '/' : c;
		}

	if (tempTable)
		for (p = FALCON_TEMPORARY; *p;)
			*q++ = *p++;
			
	*q = 0;

	return dbName;
}


int NfsStorageTable::closeConnection(handlerton *hton, THD *thd)
{
	DBUG_ENTER("NfsStorageEngine::closeConnection");
	StorageConnection *storageConn = storageHandler->getStorageConnection(NULL, thd, OpenDatabase);

	if (storageConn)
		{
		storageConn->close();
		
		if (storageConn->mySqlThread)
			storageConn->release();	// This is for thd->ha_data[falcon_hton->slot]
			
		storageConn->release();	// This is for storageConn
		}

	thd->ha_data[hton->slot] = NULL;

	DBUG_RETURN(0);
}


int NfsStorageTable::alter_tablespace(handlerton* hton, THD* thd, st_alter_tablespace* ts_info)
{
	DBUG_ENTER("NfsStorageEngine::alter_tablespace");
	int ret;
	
	switch (ts_info->ts_cmd_type)
		{
		case CREATE_TABLESPACE:
			ret = storageHandler->createTablespace(ts_info->tablespace_name, ts_info->data_file_name);
			break;
		
		case DROP_TABLESPACE:
			ret = storageHandler->deleteTablespace(ts_info->tablespace_name);
			break;
		
		default:
			DBUG_RETURN(HA_ADMIN_NOT_IMPLEMENTED);
		}
	
	DBUG_RETURN(0);
}

uint NfsStorageTable::max_supported_key_length(void) const
{
	// Dig deep for the pagesize, but be careful.  Assume 4K page unless proven otherwise.
	//if (storageConnection)
	//	if (storageConnection->database)
	//		if (storageConnection->database->dbb)
	//			if (storageConnection->database->dbb->pageSize == 8K)
	//				return MAX_INDEX_KEY_LENGTH_8k;

	return MAX_INDEX_KEY_LENGTH_4K;
}


uint NfsStorageTable::max_supported_key_part_length(void) const
{
	// Dig deep for the pagesize, but be careful.  Assume 4K page unless proven otherwise.
	//if (storageConnection)
	//	if (storageConnection->database)
	//		if (storageConnection->database->dbb)
	//			if (storageConnection->database->dbb->pageSize == 8K)
	//				return MAX_INDEX_KEY_LENGTH_8K;

	return MAX_INDEX_KEY_LENGTH_4K;
}


void NfsStorageTable::logger(int mask, const char* text, void* arg)
{
	if (mask & falcon_log_mask)
		{
		printf ("%s", text);
		
		if (falcon_log_file)
			fprintf(falcon_log_file, "%s", text);
		}
}


void NfsStorageTable::setIndexes(void)
{
	if (!table || storageShare->haveIndexes())
		return;

	storageShare->lock(true);

	if (!storageShare->haveIndexes())
		{
		StorageIndexDesc indexDesc;

		for (uint n = 0; n < table->s->keys; ++n)
			{
			getKeyDesc(table->key_info + n, &indexDesc);

			if (n == table->s->primary_key)
				indexDesc.primaryKey = true;

			storageTable->setIndex(table->s->keys, n, &indexDesc);
			}
		}

	storageShare->unlock();
}


int NfsStorageTable::genType(Field* field, CmdGen* gen)
{
	const char *type;
	const char *arg = NULL;
	int length = 0;

	switch (field->real_type()) 
	{
	case MYSQL_TYPE_DECIMAL:
	case MYSQL_TYPE_TINY:
	case MYSQL_TYPE_SHORT:
	case MYSQL_TYPE_BIT:
		type = "smallint";
		break;

	case MYSQL_TYPE_INT24:
	case MYSQL_TYPE_LONG:
	case MYSQL_TYPE_YEAR:
		type = "int";
		break;

	case MYSQL_TYPE_FLOAT:
		type = "float";
		break;

	case MYSQL_TYPE_DOUBLE:
		type = "double";
		break;

	case MYSQL_TYPE_TIMESTAMP:
		type = "timestamp";
		break;

	case MYSQL_TYPE_SET:
	case MYSQL_TYPE_LONGLONG:
		type = "bigint";
		break;

	/*
		Falcon's date and time types don't handle invalid dates like MySQL's do,
		so we just use an int for storage
	*/
	
	case MYSQL_TYPE_DATE:
	case MYSQL_TYPE_TIME:
	case MYSQL_TYPE_ENUM:
	case MYSQL_TYPE_NEWDATE:
		type = "int";
		break;

	case MYSQL_TYPE_DATETIME:
		type = "bigint";
		break;

	case MYSQL_TYPE_VARCHAR:
	case MYSQL_TYPE_VAR_STRING:
	case MYSQL_TYPE_STRING:
		{
		CHARSET_INFO *charset = field->charset();

		if (charset)
			{
			arg = charset->name;
			type = "varchar (%d) collation %s";
			}
		else
			type = "varchar (%d)";
			
		length = field->field_length;
		}
		break;

	case MYSQL_TYPE_TINY_BLOB:
	case MYSQL_TYPE_LONG_BLOB:
	case MYSQL_TYPE_BLOB:
	case MYSQL_TYPE_MEDIUM_BLOB:
	case MYSQL_TYPE_GEOMETRY:
		if (field->field_length < 256)
			type = "varchar (256)";
		else
			type = "blob";
		break;

	case MYSQL_TYPE_NEWDECIMAL:
		gen->gen("numeric (%d,%d)", ((Field_new_decimal*) field)->precision,
				((Field_new_decimal*) field)->dec);
		return 0;

	default:
		return -1;
	}

	gen->gen(type, length, arg);

	return 0;
}


void NfsStorageTable::genKeyFields(KEY* key, CmdGen* gen)
{
	const char *sep = "(";
	char nameBuffer[129];

	for (uint n = 0; n < key->key_parts; ++n)
		{
		KEY_PART_INFO *part = key->key_part + n;
		Field *field = part->field;
		storageShare->cleanupFieldName(field->field_name, nameBuffer,
										sizeof(nameBuffer));
		
		if (part->key_part_flag & HA_PART_KEY_SEG)
			gen->gen("%s\"%s\"(%d)", sep, nameBuffer, part->length);
		else
			gen->gen("%s\"%s\"", sep, nameBuffer);
			
		sep = ", ";
		}

	gen->gen(")");
}


void NfsStorageTable::encodeRecord(byte *buf, bool updateFlag)
{
	storageTable->preInsert();
	my_ptrdiff_t ptrDiff = buf - table->record[0];
	my_bitmap_map *old_map = dbug_tmp_use_all_columns(table, table->read_set);

	for (uint n = 0; n < table->s->fields; ++n)
		{
		Field *field = table->field[n];

		if (ptrDiff)
			field->move_field_offset(ptrDiff);

		if (updateFlag && !bitmap_is_set(table->write_set, field->field_index))
			{
			const unsigned char *p = storageTable->getEncoding(n);
			storageTable->dataStream.encodeEncoding(p);
			}
		else if (field->is_null())
			storageTable->dataStream.encodeNull();
		else
			switch (field->real_type()) 
				{
				case MYSQL_TYPE_TINY:
				case MYSQL_TYPE_SHORT:
				case MYSQL_TYPE_INT24:
				case MYSQL_TYPE_LONG:
				case MYSQL_TYPE_LONGLONG:
				case MYSQL_TYPE_YEAR:
				case MYSQL_TYPE_DECIMAL:
				case MYSQL_TYPE_ENUM:
				case MYSQL_TYPE_SET:
				case MYSQL_TYPE_BIT:
					storageTable->dataStream.encodeInt64(field->val_int());
					break;

				case MYSQL_TYPE_NEWDECIMAL:
					{
					int precision = ((Field_new_decimal *)field)->precision;
					int scale = ((Field_new_decimal *)field)->dec;
					
					if (precision < 19)
						{
						int64 value = ScaledBinary::getInt64FromBinaryDecimal(field->ptr,
																			precision,
																			scale);
						storageTable->dataStream.encodeInt64(value, scale);
						}
					else
						{
						BigInt bigInt;
						ScaledBinary::getBigIntFromBinaryDecimal(field->ptr, precision, scale, &bigInt);
						storageTable->dataStream.encodeBigInt(&bigInt);
						}
					}
					break;

				case MYSQL_TYPE_DOUBLE:
				case MYSQL_TYPE_FLOAT:
					storageTable->dataStream.encodeDouble(field->val_real());
					break;

				case MYSQL_TYPE_TIMESTAMP:
					{
					my_bool nullValue;
					int64 value = ((Field_timestamp*) field)->get_timestamp(&nullValue);
					storageTable->dataStream.encodeDate(value * 1000);
					}
					break;

				case MYSQL_TYPE_DATE:
					storageTable->dataStream.encodeInt64(field->val_int());
					break;

				case MYSQL_TYPE_NEWDATE:
					//storageTable->dataStream.encodeInt64(field->val_int());
					storageTable->dataStream.encodeInt64(uint3korr(field->ptr));
					break;

				case MYSQL_TYPE_TIME:
					storageTable->dataStream.encodeInt64(field->val_int());
					break;

				case MYSQL_TYPE_DATETIME:
					storageTable->dataStream.encodeInt64(field->val_int());
					break;

				case MYSQL_TYPE_VARCHAR:
				case MYSQL_TYPE_VAR_STRING:
				case MYSQL_TYPE_STRING:
					{
					String string;
					String buffer;
					field->val_str(&buffer, &string);
					storageTable->dataStream.encodeOpaque(string.length(), string.ptr());
					}
					break;

				case MYSQL_TYPE_TINY_BLOB:
					{
					Field_blob *blob = (Field_blob*) field;
					uint length = blob->get_length();
					char *ptr;
					blob->get_ptr(&ptr);
					storageTable->dataStream.encodeOpaque(length, ptr);
					}
					break;

				case MYSQL_TYPE_LONG_BLOB:
				case MYSQL_TYPE_BLOB:
				case MYSQL_TYPE_MEDIUM_BLOB:
				case MYSQL_TYPE_GEOMETRY:
					{
					Field_blob *blob = (Field_blob*) field;
					uint length = blob->get_length();
					char *ptr;
					blob->get_ptr(&ptr);
					StorageBlob *storageBlob;
					uint32 blobId;

					for (storageBlob = activeBlobs; storageBlob; storageBlob = storageBlob->next)
						if (storageBlob->data == (uchar*) ptr)
							{
							blobId = storageBlob->blobId;
							break;
							}

					if (!storageBlob)
						{
						StorageBlob storageBlob;
						storageBlob.length = length;
						storageBlob.data = (unsigned char *)ptr;
						blobId = storageTable->storeBlob(&storageBlob);
						blob->set_ptr(storageBlob.length, (char *)storageBlob.data);
						}
						
					storageTable->dataStream.encodeBinaryBlob(blobId);
					}
					break;

				default:
					storageTable->dataStream.encodeOpaque(field->field_length, field->ptr);
				}

		if (ptrDiff)
			field->move_field_offset(-ptrDiff);
		}

	dbug_tmp_restore_column_map(table->read_set, old_map);
}

void NfsStorageTable::decodeRecord(byte *buf)
{
	EncodedDataStream *dataStream = &storageTable->dataStream;
	my_ptrdiff_t ptrDiff = buf - table->record[0];
	my_bitmap_map *old_map = dbug_tmp_use_all_columns(table, table->write_set);
	DBUG_ENTER("NfsStorageTable::decodeRecord");

	for (uint n = 0; n < table->s->fields; ++n)
		{
		Field *field = table->field[n];
		dataStream->decode();

		if (ptrDiff)
			field->move_field_offset(ptrDiff);

		if (dataStream->type == edsTypeNull ||
			!bitmap_is_set(table->read_set, field->field_index))
			field->set_null();
		else
			{
			field->set_notnull();

			switch (field->real_type()) 
				{
				case MYSQL_TYPE_TINY:
				case MYSQL_TYPE_SHORT:
				case MYSQL_TYPE_INT24:
				case MYSQL_TYPE_LONG:
				case MYSQL_TYPE_LONGLONG:
				case MYSQL_TYPE_YEAR:
				case MYSQL_TYPE_DECIMAL:
				case MYSQL_TYPE_ENUM:
				case MYSQL_TYPE_SET:
				case MYSQL_TYPE_BIT:
					field->store(dataStream->getInt64(),
								((Field_num*)field)->unsigned_flag);
					break;

				case MYSQL_TYPE_NEWDECIMAL:
					{
					int precision = ((Field_new_decimal*) field)->precision;
					int scale = ((Field_new_decimal*) field)->dec;
					
					if (dataStream->type == edsTypeBigInt)
						ScaledBinary::putBigInt(&dataStream->bigInt, field->ptr, precision, scale);
					else
						{
						int64 value = dataStream->getInt64(scale);
						ScaledBinary::putBinaryDecimal(value, field->ptr, precision, scale);
						}
					}
					break;

				case MYSQL_TYPE_DOUBLE:
				case MYSQL_TYPE_FLOAT:
					field->store(dataStream->value.dbl);
					break;

				case MYSQL_TYPE_TIMESTAMP:
					{
					int value = (int) (dataStream->value.integer64 / 1000);
					longstore(field->ptr, value);
					}
					break;

				case MYSQL_TYPE_DATE:
					field->store(dataStream->getInt64(), false);
					break;

				case MYSQL_TYPE_NEWDATE:
					//field->store(dataStream->getInt64(), false);
					int3store(field->ptr, dataStream->getInt32());
					break;

				case MYSQL_TYPE_TIME:
					field->store(dataStream->getInt64(), false);
					break;

				case MYSQL_TYPE_DATETIME:
					field->store(dataStream->getInt64(), false);
					break;

				case MYSQL_TYPE_VARCHAR:
				case MYSQL_TYPE_VAR_STRING:
				case MYSQL_TYPE_STRING:
					field->store((const char*) dataStream->value.string.data,
								dataStream->value.string.length, field->charset());
					break;

				case MYSQL_TYPE_TINY_BLOB:
					{
					Field_blob *blob = (Field_blob*) field;
					blob->set_ptr(dataStream->value.string.length,
									(char *)dataStream->value.string.data);
					}
					break;

				case MYSQL_TYPE_LONG_BLOB:
				case MYSQL_TYPE_BLOB:
				case MYSQL_TYPE_MEDIUM_BLOB:
				case MYSQL_TYPE_GEOMETRY:
					{
					Field_blob *blob = (Field_blob*) field;
					StorageBlob *storageBlob = freeBlobs;

					if (storageBlob)
						freeBlobs = storageBlob->next;
					else
						storageBlob = new StorageBlob;

					storageBlob->next = activeBlobs;
					activeBlobs = storageBlob;
					storageBlob->blobId = dataStream->value.blobId;
					storageTable->getBlob(storageBlob->blobId, storageBlob);
					blob->set_ptr(storageBlob->length, (char*) storageBlob->data);
					}
					break;

				default:
					{
					uint l = dataStream->value.string.length;

					if (field->field_length < l)
						l = field->field_length;

					memcpy(field->ptr, dataStream->value.string.data, l);
					}
				}
			}

		if (ptrDiff)
			field->move_field_offset(-ptrDiff);
		}
	dbug_tmp_restore_column_map(table->write_set, old_map);

	DBUG_VOID_RETURN;
}


int NfsStorageTable::extra(ha_extra_function operation)
{
	DBUG_ENTER("NfsStorageTable::extra");
	DBUG_RETURN(0);
}

bool NfsStorageTable::get_error_message(int error, String *buf)
{
	if (storageConnection)
		{
		const char *text = storageConnection->getLastErrorString();
		buf->set(text, strlen(text), system_charset_info);
		}

	return false;
}

void NfsStorageTable::unlockTable(void)
{
	if (tableLocked)
		{
		storageShare->unlock();
		tableLocked = false;
		}
}


struct st_mysql_storage_engine falcon_storage_engine=
{ MYSQL_HANDLERTON_INTERFACE_VERSION };

mysql_declare_plugin(falcon)
{
	MYSQL_STORAGE_ENGINE_PLUGIN,
	&falcon_storage_engine,
	falcon_hton_name,
	"MySQL AB",
	"Falcon storage engine",
	PLUGIN_LICENSE_PROPRIETARY,
	NfsStorageTable::falcon_init, /* Plugin Init */
	NULL, /* Plugin Deinit */
	0x0100, /* 1.0 */
	NULL,                       /* status variables             */
	NULL,                       /* system variables                */
	NULL                        /* config options                  */
}

mysql_declare_plugin_end;
