/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// Transaction.cpp: implementation of the Transaction class.
//
//////////////////////////////////////////////////////////////////////

#include <memory.h>
#include "Engine.h"
#include "Transaction.h"
#include "Database.h"
#include "Dbb.h"
#include "Connection.h"
#include "Table.h"
#include "RecordVersion.h"
#include "SQLError.h"
#include "Sync.h"
#include "PageWriter.h"
#include "Table.h"
#include "Interlock.h"
#include "SavePoint.h"
#include "IOx.h"
#include "DeferredIndex.h"
#include "TransactionManager.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Transaction::Transaction(Connection *cnct, TransId seq)
{
	states = NULL;
	statesAllocated = 0;
	initialize(cnct, seq);
}

void Transaction::initialize(Connection* cnct, TransId seq)
{
	connection = cnct;
	isolationLevel = connection->isolationLevel;
	database = connection->database;
	TransactionManager *transactionManager = database->transactionManager;
	systemTransaction = database->systemConnection == connection;
	transactionId = seq;
	records = NULL;
	recordPtr = &records;
	dependencies = 0;
	commitTriggers = false;
	hasUpdates = false;
	writePending = true;
	waitingFor = NULL;
	syncActive.lock(NULL, Exclusive);
	useCount = 1;
	curSavePointId = 0;
	savePoints = NULL;
	freeSavePoints = NULL;
	deferredIndexes = NULL;
	xidLength = 0;
	xid = NULL;
	Transaction *oldest = transactionManager->activeTransactions.first;
	
	while (oldest && !oldest->isActive())
		oldest = oldest->next;

	oldestActive = (oldest) ? oldest->transactionId : transactionId;
	int count = transactionManager->activeTransactions.count;
	
	if (count > statesAllocated)
		{
		delete [] states;
		statesAllocated = count;
		states = new TransState [statesAllocated];
		}
	
	numberStates = 0;
	
	if (count)
		for (Transaction *transaction = transactionManager->activeTransactions.first; transaction; transaction = transaction->next)
			if (transaction->isActive() && !transaction->systemTransaction)
				{
				TransState *state = states + numberStates++;
				state->transaction = transaction;
				state->transactionId = transaction->transactionId;
				state->state = transaction->state;
				INTERLOCKED_INCREMENT(transaction->dependencies);
				}

	state = Active;
}

Transaction::~Transaction()
{
	if (state == Active)
		{
		ASSERT(false);
		syncActive.unlock();
		}

	delete [] states;
	delete [] xid;
	
	for (RecordVersion *record; (record = records);)
		{
		records = record->next;
		record->release();
		}
	
	SavePoint *savePoint;
	
	while ( (savePoint = savePoints) )
		{
		savePoints = savePoint->next;
		delete savePoint;
		}
	
	while ( (savePoint = freeSavePoints) )
		{
		freeSavePoints = savePoint->next;
		delete savePoint;
		}
	
	for (DeferredIndex *deferredIndex; (deferredIndex = deferredIndexes);)
		{
		deferredIndexes = deferredIndex->nextInTransaction;
		deferredIndex->detachTransaction();
		}
}

void Transaction::commit()
{
	TransactionManager *transactionManager = database->transactionManager;

	if (!isActive())
		throw SQLEXCEPTION (RUNTIME_ERROR, "transaction is not active");

	for (DeferredIndex *deferredIndex= deferredIndexes; deferredIndex;  
		 deferredIndex = deferredIndex->nextInTransaction)
		if (deferredIndex->index)
			database->dbb->logIndexUpdates(this, deferredIndex);
		
	database->dbb->logUpdatedRecords(this, records);
	database->pageWriter->waitForWrites(transactionId);
	state = Committed;
	syncActive.unlock();

	for (RecordVersion *record = records; record; record = record->next)
		if (!record->isSuperceded())
			record->table->updateRecord (record);

	if (commitTriggers)
		for (RecordVersion *record = records; record; record = record->next)
			if (!record->isSuperceded())
				record->table->postCommit (this, record);

	releaseDependencies();

	if (hasUpdates)
		database->flushInversion(this);
	else
		writeComplete();

	database->commit(this);
	delete [] xid;
	xid = NULL;
	xidLength = 0;
	
	// If the transaction has updates, it needs to be moved to the committed transaction list.
	// Otherwise, keep it around for eventual re-use
	
	if (hasUpdates)
		{
		Sync syncActiveTransactions(&transactionManager->activeTransactions.syncObject, "Transaction::commit");
		syncActiveTransactions.lock(Exclusive);
		transactionManager->activeTransactions.remove(this);
		syncActiveTransactions.unlock();
		}
	
	// If there's no reason to stick around, just go away
	
	if ((dependencies == 0) && !writePending)
		commitRecords();
	
	// Add ourselves to the list of lingering committed transactions
	
	if (hasUpdates)
		{
		Sync syncCommitted(&transactionManager->committedTransactions.syncObject, "Transaction::commit");
		syncCommitted.lock(Exclusive);
		transactionManager->committedTransactions.append(this);
			
		// And, while we're at it, check for any fully mature transactions to ditch
		
		for (Transaction *transaction, *next = transactionManager->committedTransactions.first; (transaction = next);)
			{
			next = transaction->next;

			if ((transaction->state == Committed) && 
				(transaction->dependencies == 0) && 
				!transaction->writePending)
				{
				transaction->commitRecords();
				transactionManager->committedTransactions.remove(transaction);
				transaction->release();
				}
			}
		}
	else
		state = Available;
}

void Transaction::rollback()
{
	if (!isActive())
		throw SQLEXCEPTION (RUNTIME_ERROR, "transaction is not active");

	TransactionManager *transactionManager = database->transactionManager;
	state = RolledBack;

	// Get rid of old versions forth-with and directly

	while (records)
		{
		RecordVersion *record = records;
		records = record->next;
		record->rollback();
		record->release();
		}

	writePending = false;
	releaseDependencies();
	syncActive.unlock();
	database->rollback(this);
	delete [] xid;
	xid = NULL;
	xidLength = 0;
	
	Sync sync (&transactionManager->activeTransactions.syncObject, "Transaction::rollback");
	sync.lock (Exclusive);

	for (Transaction *trans = transactionManager->activeTransactions.first; trans; trans = trans->next)
		if (trans->isActive())
			trans->expungeTransaction(this);

	transactionManager->activeTransactions.remove(this);
	sync.unlock();

	release();
}


void Transaction::expungeTransaction(Transaction * transaction)
{
	for (TransState *state = states, *end = states + numberStates; state < end; ++state)
		if (state->transaction == transaction)
			{
			state->transaction = NULL;
			break;
			}
}

void Transaction::prepare(int xidLen, const UCHAR *xidPtr)
{
	if (state != Active)
		throw SQLEXCEPTION (RUNTIME_ERROR, "transaction is not active");

	if ( (xidLength = xidLen) )
		{
		xid = new UCHAR[xidLength];
		memcpy(xid, xidPtr, xidLength);
		}
		
	database->pageWriter->waitForWrites(transactionId);
	state = Limbo;
	database->dbb->prepareTransaction(transactionId, xidLength, xid);
}

void Transaction::addRecord(RecordVersion * record)
{
	ASSERT(record->recordNumber >= 0);
	record->addRef();
	record->next = NULL;
	*recordPtr = record;
	recordPtr = &record->next;
	hasUpdates = true;
}

void Transaction::removeRecord(RecordVersion *record)
{
	for (RecordVersion **ptr = &records; *ptr; ptr = &(*ptr)->next)
		if (*ptr == record)
			{
			*ptr = record->next;
			record->next = NULL;
			record->transaction = NULL;

			if (*recordPtr == record)
				recordPtr = ptr;

			record->release();

			return;
			}
}

/***
@brief		Determine if changes by another transaction are visible to this.
@detail		This function is called for Consistent-Read transactions to determine
			if the sent trans was committed before this transaction started.  If not,
			it is invisible to this transaction.
***/
bool Transaction::visible(Transaction * transaction)
{
	// If the transaction is NULL, it is long gone and therefore committed

	if (!transaction)
		return true;

	// If we're the transaction in question, consider us committed

	if (transaction == this)
		return true;

	// If we're the system transaction, just use the state of the other transaction

	if (database->systemConnection->transaction == this)
		return transaction->state == Committed;

	// If the other transaction is not yet committed, the trans is not visible.

	if (transaction->state != Committed)
		return false;

	// The other transaction is committed.  
	// If this is READ_COMMITTED, it is visible.

	if (isolationLevel == TRANSACTION_READ_COMMITTED)
		return true;

	// This is REPEATABLE_READ
	// If the transaction started after we did, consider the transaction active

	if (transaction->transactionId > transactionId)
		return false;

	// If the transaction was active when we started, use it's state at that point

	for (int n = 0; n < numberStates; ++n)
		if (states [n].transactionId == transaction->transactionId)
			return false;

	return true;
}

void Transaction::releaseDependencies()
{
	if (!numberStates)
		return;

	for (TransState *state = states, *end = states + numberStates; state < end; ++state)
		if (state->transaction)
			state->transaction->releaseDependency();

	/***
	numberStates = 0;
	delete [] states;
	states = NULL;
	***/
}

/*
 *  Transaction is fully mature and about to go away.
 *  Fully commit all records
 */

void Transaction::commitRecords()
{
	ASSERT(!writePending);
	
	for (RecordVersion *record; (record = records);)
		{
		ASSERT (record->useCount > 0);
		records = record->next;
		record->commit();
		record->release();
		}
}


/***
@brief		Get the relative state between this transaction and another.
***/

State Transaction::getRelativeState(Transaction *transaction, TransId transId)
{
	// A record may still have the transId even after the trans itself has been deleted.
	
	if (!transaction)
		{
		// If the transaction is no longer around, then this is a previously committed record.

		if (transactionId == transId)
			return Us;

		// For REPEATABLE_READ, if transId was active when this started, the dependency
		// would have kept the transaction pointer around. 

		if ((isolationLevel == TRANSACTION_REPEATABLE_READ) &&
		    (transactionId < transId))
			return CommittedButYounger;
		
		return CommittedAndOlder;
		}

	if (transaction == this)
		return Us;

	if (transaction->isActive())
		{
		// If waiting would cause a deadlock, don't try it

		for (Transaction *trans = transaction->waitingFor; trans; trans = trans->waitingFor)
			if (trans == this)
				return Deadlock;

		// OK, add a reference to the transaction to keep the object around, then wait for it go go away

		transaction->addRef();
		waitingFor = transaction;
		transaction->waitForTransaction();
		waitingFor = NULL;
		State transactionState = (State) transaction->state;
		transaction->release();

		return WasActive;			// caller will need to re-fetch
		}

	if (transaction->state == Committed)
		{
		// Return CommittedAndOlder if the other trans has a lower TransId and 
		// it was committed when we started.
		if (visible (transaction))
			return CommittedAndOlder;

		return CommittedButYounger;
		}

	return (State) transaction->state;
}

void Transaction::dropTable(Table* table)
{
	for (RecordVersion **ptr = &records, *rec; (rec = *ptr);)
		if (rec->table == table)
			{
			*ptr = rec->next;
			rec->release();
			}
		else
			ptr = &rec->next;
}

bool Transaction::hasUncommittedRecords(Table* table)
{
	for (RecordVersion *rec = records; rec; rec = rec->next)
		if (rec->table == table)
			return true;
	
	return false;
}

void Transaction::writeComplete(void)
{
	ASSERT(writePending == true);
	ASSERT(state == Committed);
	
	for (DeferredIndex *deferredIndex; (deferredIndex = deferredIndexes);)
		{
		ASSERT(deferredIndex->transaction == this);
		deferredIndexes = deferredIndex->nextInTransaction;
		deferredIndex->detachTransaction();
		}

	writePending = false;
}

bool Transaction::waitForTransaction(TransId transId)
{
	TransactionManager *transactionManager = database->transactionManager;
	Sync syncActiveTransactions(&transactionManager->activeTransactions.syncObject, "Transaction::waitForTransaction");
	syncActiveTransactions.lock(Shared);
	Transaction *transaction;
	
	// If the transaction is still active, find it

	for (transaction = transactionManager->activeTransactions.first; transaction; transaction = transaction->next)
		if (transaction->transactionId == transId)
			break;
	
	// If the transction is no longer active, see if it committed
	
	if (!transaction)
		return true;
		
	if (transaction->state == Committed)
		return true;

	// If waiting would cause a deadlock, don't try it

	for (Transaction *trans = transaction->waitingFor; trans; trans = trans->waitingFor)
		if (trans == this)
			return true;

	// OK, add a reference to the transaction to keep the object around, then wait for it to go away

	waitingFor = transaction;
	transaction->addRef();
	syncActiveTransactions.unlock();
	transaction->waitForTransaction();
	transaction->release();
	waitingFor = NULL;

	return transaction->state == Committed;
}

void Transaction::releaseDependency(void)
{
	INTERLOCKED_DECREMENT(dependencies);
}

void Transaction::waitForTransaction()
{
	syncActive.lock(NULL, Shared);
	syncActive.unlock();
}

void Transaction::addRef()
{
	INTERLOCKED_INCREMENT(useCount);
}

void Transaction::release()
{
	if (INTERLOCKED_DECREMENT(useCount) == 0)
		delete this;
}

int Transaction::createSavepoint()
{
	SavePoint *savePoint;
	
	if ( (savePoint = freeSavePoints) )
		freeSavePoints = savePoint->next;
	else
		savePoint = new SavePoint;
	
	savePoint->records = recordPtr;
	savePoint->id = ++curSavePointId;
	savePoint->next = savePoints;
	savePoints = savePoint;

	return savePoint->id;
}

void Transaction::releaseSavepoint(int savePointId)
{
	for (SavePoint **ptr = &savePoints, *savePoint; (savePoint = *ptr); ptr = &savePoint->next)
		if (savePoint->id == savePointId)
			{
			int nextLowerSavePointId = (savePoint->next) ? savePoint->next->id : 0;
			*ptr = savePoint->next;
			savePoint->next = freeSavePoints;
			freeSavePoints = savePoint;

			// commit pending record versions to the next pending savepoint
			
			RecordVersion *record = *savePoint->records;
			
			while ( (record) && (record->savePointId == savePointId) )
				{
				record->savePointId = nextLowerSavePointId;
				record->scavenge(transactionId, nextLowerSavePointId);
				record = record->next;
				}

			return;
			}

	//throw SQLError(RUNTIME_ERROR, "invalid savepoint");
}

void Transaction::rollbackSavepoint(int savePointId)
{
	SavePoint *savePoint = savePoints;

	// Be sure the target savepoint is valid before rollong them back.
	
	for (savePoint = savePoints; savePoint; savePoint = savePoint->next)
		if (savePoint->id <= savePointId)
			break;
			
	if ((savePoint) && (savePoint->id != savePointId))
		throw SQLError(RUNTIME_ERROR, "invalid savepoint");

	savePoint = savePoints;
	
	while (savePoint)
		{
		if (savePoint->id < savePointId)
			break;

		// Purge out records from this savepoint

		RecordVersion *record = *savePoint->records;
		recordPtr = savePoint->records;
		*recordPtr = NULL;
		RecordVersion *stack = NULL;

		while (record)
			{
			RecordVersion *rec = record;
			record = rec->next;
			rec->next = stack;
			stack = rec;
			}

		while (stack)
			{
			RecordVersion *rec = stack;
			stack = rec->next;
			rec->rollback();
#ifdef CHECK_RECORD_ACTIVITY
			rec->active = false;
#endif
			rec->release();
			}

		// Move skipped savepoints object to the free list
		// Leave the target savepoint empty, but connected to the transaction.
		
		if (savePoint->id > savePointId)
			{
			savePoints = savePoint->next;
			savePoint->next = freeSavePoints;
			freeSavePoints = savePoint;
			savePoint = savePoints;
			}
		else
			savePoint = savePoint->next;
		}
}

void Transaction::scavengeRecords(int ageGroup)
{
	scavenged = true;
}

void Transaction::add(DeferredIndex* deferredIndex)
{
	deferredIndex->nextInTransaction = deferredIndexes;
	deferredIndexes = deferredIndex;
}

bool Transaction::isXidEqual(int testLength, const UCHAR* test)
{
	if (testLength != xidLength)
		return false;
	
	return memcmp(xid, test, xidLength) == 0;
}
