/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <string.h>
#include <stdio.h>
#include <memory.h>
#include "Engine.h"
#include "StorageHandler.h"
#include "StorageConnection.h"
#include "Connection.h"
#include "SyncObject.h"
#include "Sync.h"
#include "SQLError.h"
#include "StorageDatabase.h"
#include "StorageTableShare.h"
#include "Stream.h"
#include "Configuration.h"
#include "Threads.h"

#define DICTIONARY_NAME			"falcon_dictionary"
#define DICTIONARY_PATH			"falcon_dictionary"
#define DICTIONARY_ACCOUNT		"mysql"
#define DICTIONARY_PW			"mysql"

#define HASH(address,size)				(((UIPTR) address >> 2) % size)

struct StorageSavepoint {
	StorageSavepoint*	next;
	StorageConnection*	storageConnection;
	int					savepoint;
	};

static const char *falconSchema [] = {
	"upgrade table falcon.tablespaces ("
	"    name varchar(128) not null primary	key,"
	"    pathname varchar(1024) not null)",
	
	"upgrade table falcon.tables ("
	"    given_schema_name varchar(128) not null,"
	"    effective_schema_name varchar(128) not null,"
	"    given_table_name varchar(128) not null,"
	"    effective_table_name varchar(128) not null,"
	"    pathname varchar(1024) not null primary key)",
	
	"upgrade unique index on falcon.tables (effective_schema_name, effective_table_name)",
	
	NULL };
				
class Server;
extern Server*	startServer(int port, const char *configFile);

static StorageHandler *storageHandler;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

StorageHandler*	getFalconStorageHandler()
{
	if (!storageHandler)
		storageHandler = new StorageHandler;
	
	return storageHandler;
}


StorageHandler::StorageHandler(void)
{
	memset(connections, 0, sizeof(connections));
	memset(storageDatabases, 0, sizeof(storageDatabases));
	syncObject = new SyncObject;
	hashSyncObject = new SyncObject;
	dictionaryConnection = NULL;
}

StorageHandler::~StorageHandler(void)
{
	delete syncObject;
	delete hashSyncObject;
	
	for (int n = 0; n < databaseHashSize; ++n)
		for (StorageDatabase *storageDatabase; (storageDatabase = storageDatabases[n]);)
			{
			storageDatabases[n] = storageDatabase->collision;
			delete storageDatabase;
			}
}

void StorageHandler::startNfsServer(void)
{
	startServer(0, NULL);
}

void StorageHandler::addNfsLogger(int mask, Logger listener, void* arg)
{
	addLogListener(mask, listener, arg);
}

StorageConnection* StorageHandler::getStorageConnection(const char* path, THD* mySqlThread, OpenOption createFlag)
{
	Sync sync(syncObject, "StorageConnection::getStorageConnection");
	sync.lock(Shared);
	int slot = HASH(mySqlThread, connectionHashSize);
	StorageConnection *storageConnection;
	
	for (storageConnection = connections[slot]; storageConnection; storageConnection = storageConnection->collision)
		if (storageConnection->mySqlThread == mySqlThread && (!path ||storageConnection->path == path))
			{
			storageConnection->addRef();
			
			return storageConnection;
			}

	sync.unlock();
	sync.lock(Exclusive);
	
	for (storageConnection = connections[slot]; storageConnection; storageConnection = storageConnection->collision)
		if (storageConnection->mySqlThread == mySqlThread && (!path || storageConnection->matches(path)))
			{
			storageConnection->addRef();
			
			return storageConnection;
			}
	
	if (!path)
		return NULL;

	storageConnection = new StorageConnection(this, path, mySqlThread);
	bool success = false;
	
	if (createFlag != CreateDatabase && createFlag != OpenTemporaryDatabase)
		try
			{
			storageConnection->connect();
			success = true;
			}
		catch (SQLException& exception)
			{
			fprintf(stderr, "database open failed: %s\n", exception.getText());
            storageConnection->setErrorText(exception.getText());
			
			if (createFlag == OpenDatabase)
				{
				delete storageConnection;
				
				return NULL;
				}
			}
	
	if (!success && createFlag != OpenDatabase)
		try
			{
			storageConnection->create();
			}
		catch (SQLException& exception)
			{
			fprintf(stderr, "database create failed: %s\n", exception.getText());
			delete storageConnection;
			return NULL;
			}
	
	storageConnection->collision = connections[slot];
	connections[slot] = storageConnection;
	
	return storageConnection;
}

int StorageHandler::isTempTable(const char *path)
{
	Sync sync(hashSyncObject, "StorageHandler::getStorageDatabase");
	sync.lock(Shared);
	int slot = JString::hash(path, databaseHashSize);
	StorageDatabase *storageDatabase;
	
	for (int n = 0; n < databaseHashSize; ++n)
		for (storageDatabase = storageDatabases[n]; 
			 storageDatabase; 
			 storageDatabase = storageDatabase->collision)
			{
			Sync syncShares(&storageDatabase->syncObject, "StorageDatabase::isTempTable");
			syncShares.lock(Shared);
			
			for (int n2 = 0; n2 < shareHashSize; ++n2)
				for (StorageTableShare *share = storageDatabase->shares[n2]; share; share = share->collision)
					if (share->pathName == path)
						return share->tempTable;
			}
	
	return -1;
}

void StorageHandler::shutdownHandler(void)
{
	for (int n = 0; n < databaseHashSize; ++n)
		for (StorageDatabase *storageDatabase = storageDatabases[n]; storageDatabase; storageDatabase = storageDatabase->collision)
			storageDatabase->close();
	
	//Connection::shutdownDatabases();
}

void StorageHandler::databaseDropped(StorageConnection* storageConnection)
{
	StorageDatabase *storageDatabase = storageConnection->storageDatabase;
	
	if (storageDatabase)
		{
		Sync syncHash(hashSyncObject, "StorageHandler::dropDatabase");
		int slot = JString::hash(storageDatabase->filename, databaseHashSize);
		syncHash.lock(Exclusive);
		
		for (StorageDatabase **ptr = storageDatabases + slot; *ptr; ptr = &(*ptr)->collision)
			if (*ptr == storageDatabase)
				{
				*ptr = storageDatabase->collision;
				break;
				}

		syncHash.unlock();
		}

	Sync sync(syncObject, "StorageHandler::~dropDatabase");
	sync.lock(Exclusive);

	for (int n = 0; n < connectionHashSize; ++n)
		for (StorageConnection *cnct = connections[n]; cnct; cnct = cnct->collision)
			if (cnct != storageConnection)
				cnct->databaseDropped(storageDatabase);
			
	sync.unlock();
}

void StorageHandler::remove(StorageConnection* storageConnection)
{
	Sync sync(syncObject, "StorageHandler::~remove");
	sync.lock(Exclusive);
	int slot = HASH(storageConnection->mySqlThread, connectionHashSize);
	
	for (StorageConnection **ptr = connections + slot; *ptr; ptr = &(*ptr)->collision)
		if (*ptr == storageConnection)
			{
			*ptr = storageConnection->collision;
			break;
			}
			
}

int StorageHandler::commit(THD* mySqlThread)
{
	Sync sync(syncObject, "StorageHandler::commit");
	sync.lock(Shared);
	int slot = HASH(mySqlThread, connectionHashSize);
	
	for (StorageConnection *connection = connections[slot]; connection; connection = connection->collision)
		if (connection->mySqlThread == mySqlThread)
			{
			int ret =connection->commit();
			
			if (ret)
				return ret;
			}
	
	return 0;
}

int StorageHandler::prepare(THD* mySqlThread, int xidSize, const UCHAR *xid)
{
	Sync sync(syncObject, "StorageHandler::prepare");
	sync.lock(Shared);
	int slot = HASH(mySqlThread, connectionHashSize);
	
	for (StorageConnection *connection = connections[slot]; connection; connection = connection->collision)
		if (connection->mySqlThread == mySqlThread)
			{
			int ret = connection->prepare(xidSize, xid);
			
			if (ret)
				return ret;
			}
	
	return 0;
}

int StorageHandler::rollback(THD* mySqlThread)
{
	Sync sync(syncObject, "StorageHandler::rollback");
	sync.lock(Shared);
	int slot = HASH(mySqlThread, connectionHashSize);
	
	for (StorageConnection *connection = connections[slot]; connection; connection = connection->collision)
		if (connection->mySqlThread == mySqlThread)
			{
			int ret =connection->rollback();
			
			if (ret)
				return ret;
			}
	
	return 0;
}

int StorageHandler::releaseVerb(THD* mySqlThread)
{
	Sync sync(syncObject, "StorageHandler::releaseVerb");
	sync.lock(Shared);
	int slot = HASH(mySqlThread, connectionHashSize);
	
	for (StorageConnection *connection = connections[slot]; connection; connection = connection->collision)
		if (connection->mySqlThread == mySqlThread)
			connection->releaseVerb();
	
	return 0;
}

int StorageHandler::rollbackVerb(THD* mySqlThread)
{
	Sync sync(syncObject, "StorageHandler::rollbackVerb");
	sync.lock(Shared);
	int slot = HASH(mySqlThread, connectionHashSize);
	
	for (StorageConnection *connection = connections[slot]; connection; connection = connection->collision)
		if (connection->mySqlThread == mySqlThread)
			connection->rollbackVerb();
	
	return 0;
}

int StorageHandler::savepointSet(THD* mySqlThread, void* savePoint)
{
	Sync sync(syncObject, "StorageHandler::savepointSet");
	sync.lock(Shared);
	int slot = HASH(mySqlThread, connectionHashSize);
	StorageSavepoint *savepoints = NULL;
	
	for (StorageConnection *connection = connections[slot]; connection; connection = connection->collision)
		if (connection->mySqlThread == mySqlThread)
			{
			StorageSavepoint *savepoint = new StorageSavepoint;
			savepoint->next = savepoints;
			savepoints = savepoint;
			savepoint->storageConnection = connection;
			savepoint->savepoint = connection->savepointSet();
			}
	
	*((void**) savePoint) = savepoints;
	
	return 0;
}

int StorageHandler::savepointRelease(THD* mySqlThread, void* savePoint)
{
	Sync sync(syncObject, "StorageHandler::savepointRelease");
	sync.lock(Shared);
	
	for (StorageSavepoint *savepoints = *(StorageSavepoint**) savePoint, *savepoint; 
		  (savepoint = savepoints);)
		{
		savepoint->storageConnection->savepointRelease(savepoint->savepoint);
		savepoints = savepoint->next;
		delete savepoint;
		}
		
	*((void**) savePoint) = NULL;

	return 0;
}

int StorageHandler::savepointRollback(THD* mySqlThread, void* savePoint)
{
	Sync sync(syncObject, "StorageHandler::savepointRollback");
	sync.lock(Shared);
	
	for (StorageSavepoint *savepoints = *(StorageSavepoint**) savePoint, *savepoint; 
		   (savepoint = savepoints);)
		{
		savepoint->storageConnection->savepointRollback(savepoint->savepoint);
		savepoints = savepoint->next;
		delete savepoint;
		}
	
	*((void**) savePoint) = NULL;

	return 0;
}

StorageDatabase* StorageHandler::getStorageDatabase(const char* dbName, const char* path)
{
	Sync sync(hashSyncObject, "StorageHandler::getStorageDatabase");
	sync.lock(Shared);
	int slot = JString::hash(path, databaseHashSize);
	StorageDatabase *storageDatabase;
	
	for (storageDatabase = storageDatabases[slot]; storageDatabase; storageDatabase = storageDatabase->collision)
		if (storageDatabase->filename == path)
			{
			storageDatabase->addRef();
			return storageDatabase;
			}
	
	sync.unlock();
	sync.lock(Exclusive);
	
	for (storageDatabase = storageDatabases[slot]; storageDatabase; storageDatabase = storageDatabase->collision)
		if (storageDatabase->filename == path)
			{
			storageDatabase->addRef();
			return storageDatabase;
			}
		
	storageDatabase = new StorageDatabase(dbName, path);
	storageDatabase->collision = storageDatabases[slot];
	storageDatabases[slot] = storageDatabase;
	storageDatabase->addRef();
	
	return storageDatabase;
}

void StorageHandler::closeDatabase(const char* path)
{
	Sync sync(hashSyncObject, "StorageHandler::getStorageDatabase");
	int slot = JString::hash(path, databaseHashSize);
	sync.lock(Exclusive);
	
	for (StorageDatabase *storageDatabase, **ptr = storageDatabases + slot; (storageDatabase = *ptr); ptr = &storageDatabase->collision)
		if (storageDatabase->filename == path)
			{
			*ptr = storageDatabase->collision;
			storageDatabase->close();
			storageDatabase->release();
			break;
			}
}

const char* StorageHandler::getEngineStatus(THD* mySqlThread)
{
	StorageConnection *storageConnection = getStorageConnection(NULL, mySqlThread, OpenDatabase);
	
	if (!storageConnection || !storageConnection->connection)
		return NULL;
	
	storageConnection->lastErrorText = storageConnection->connection->analyze(analyzeTables | analyzeSpace);
	
	return storageConnection->lastErrorText;
}

void StorageHandler::releaseText(const char* text)
{
	delete [] text;
}

int StorageHandler::commitByXID(int xidLength, const UCHAR* xid)
{
	return 0;
}

int StorageHandler::rollbackByXID(int xidLength, const UCHAR* xis)
{
	return 0;
}

Connection* StorageHandler::getDictionaryConnection(void)
{
	if (!dictionaryConnection)
		{
		Configuration *configuration = new Configuration(NULL);
		configuration->recordMemoryUpper = 1000000;
		configuration->pageCacheSize = 1024000;
		dictionaryConnection = new Connection(configuration);
		Threads *threads = new Threads(NULL);
		
		try
			{
			dictionaryConnection->openDatabase(DICTIONARY_NAME, DICTIONARY_PATH, DICTIONARY_ACCOUNT, DICTIONARY_PW, NULL, threads);
			}
		catch (SQLException&)
			{
			try
				{
				dictionaryConnection->createDatabase(DICTIONARY_NAME, DICTIONARY_PATH, DICTIONARY_ACCOUNT, DICTIONARY_PW, threads);
				}
			catch (SQLException&)
				{
				dictionaryConnection->close();
				dictionaryConnection = NULL;
				throw;
				}
			}
		}
	
	return dictionaryConnection->clone();
}

int StorageHandler::createTablespace(const char* tableSpaceName, const char* filename)
{
	try
		{
		}
	catch (SQLException& exception)
		{
		}
	
	return 0;
}

int StorageHandler::deleteTablespace(const char* tableSpaceName)
{
	return 0;
}
