/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// RecordLeaf.cpp: implementation of the RecordLeaf class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <memory.h>
#include "Engine.h"
#include "RecordLeaf.h"
#include "RecordGroup.h"
#include "RecordVersion.h"
#include "Table.h"
#include "Sync.h"
#include "Interlock.h"

#ifdef _DEBUG
static char THIS_FILE[]=__FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

RecordLeaf::RecordLeaf()
{
	memset (records, 0, sizeof (records));
}

RecordLeaf::~RecordLeaf()
{
	for (int n = 0; n < RECORD_SLOTS; ++n)
		if (records[n])
			{
#ifdef CHECK_RECORD_ACTIVITY
			for (Record *rec = records[n]; rec; rec = rec->getPriorVersion())
				rec->active = false;
#endif
				
			records[n]->release();
			}
}

Record* RecordLeaf::fetch(int32 id)
{
	if (id >= RECORD_SLOTS)
		return NULL;

	Sync sync(&syncObject, "RecordLeaf::fetch");
	sync.lock(Shared);

	Record *record = records[id];

	if (record)
		record->addRef();

	return record;
}


bool RecordLeaf::store(Record *record, Record *prior, int32 id, RecordSection **parentPtr)
{
	// If this doesn't fit, create a new level above use, then store
	// record in new record group.

	Sync sync(&syncObject, "RecordLeaf::store");
	
	while (id >= RECORD_SLOTS)
		{
		RecordGroup* group = new RecordGroup (RECORD_SLOTS, 0, this);
		
		if (COMPARE_EXCHANGE_POINTER(parentPtr, this, group))
			 return group->store (record, prior, id,parentPtr);
		
		delete group;
		}

	sync.lock(Exclusive);

	if (records[id] != prior)
		return false;
		
	// It fits.  Not problem.

	if (record)
		record->addRef ();

	if (prior)
		{
		prior->active = false;
		prior->release();
		}

	records[id] = record;

	return true;
}

int RecordLeaf::retireRecords (Table *table, int base, int age, TransId oldestActiveTransaction)
{
	int count = 0;

	for (Record **ptr = records, **end = records + RECORD_SLOTS; ptr < end; ++ptr)
		{
		Record *record = *ptr;
		
		if (record)
			{
			if (record->isVersion())
				{
				if (record->scavenge(oldestActiveTransaction) && 
					 ((record->useCount == 1 && record->ageGroup <= age) ||
					  !record->hasRecord()))
					{
					*ptr = NULL;
#ifdef CHECK_RECORD_ACTIVITY
					record->active = false;
#endif
					record->release();
					}
				else
					++count;
				}
			else if (record->ageGroup <= age)
				{
				*ptr = NULL;
#ifdef CHECK_RECORD_ACTIVITY
				record->active = false;
#endif
				record->release();
				}
			else
				++count;
			}
		}

	return count;
}

int RecordLeaf::countActiveRecords()
{
	int count = 0;

	for (Record **ptr = records, **end = records + RECORD_SLOTS; ptr < end; ++ptr)
		if (*ptr)
			++count;

	return count;
}
