/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// MACAddress.cpp: implementation of the MACAddress class.
//
//////////////////////////////////////////////////////////////////////


#ifdef _WIN32
//#include <Afx.h>
#include <Windows.h>
#include <Iphlpapi.h>

#else
#include <unistd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#endif

#include <stdio.h>
#include <string.h>
#include "Engine.h"
#include "MACAddress.h"

#define MAX_ADDRESSES	10

#ifndef _WIN32
static const char *devices [] = { 
	"eth0", 
	"eth1", 
	"eth2",
	"xyzzy",
	NULL
	};
#endif

static int count = MACAddress::getAddresses();
static QUAD macAddresses [MAX_ADDRESSES];
static const char *hexDigits = "0123456789ABCDEF";

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

MACAddress::MACAddress()
{

}

MACAddress::~MACAddress()
{

}

int MACAddress::getAddresses()
{
	int count = 0;

#ifdef _WIN32

	char buffer [4096];
	MIB_IFTABLE	*ifTable = (MIB_IFTABLE*) buffer;
	ULONG tableLength = sizeof (buffer);
	DWORD ret = GetIfTable (ifTable, &tableLength, 0);

	if (ret == 0)
		for (DWORD n = 0; n < ifTable->dwNumEntries && count < MAX_ADDRESSES; ++n)
			{
			int length = ifTable->table[n].dwPhysAddrLen;
			if (length > 0 && length <= 8)
				macAddresses [count++] = getAddress (length, ifTable->table[n].bPhysAddr);
			}
#else
	int fd = socket (PF_INET, SOCK_DGRAM, IPPROTO_IP);

	if (fd < 0)
		return 0;
	
	ifreq request;

	for (const char **device = devices; *device; ++device)
		{
		strncpy (request.ifr_name, *device, IFNAMSIZ);
		if (ioctl (fd, SIOCGIFHWADDR, &request) == 0)
			macAddresses [count++] = getAddress (6, (UCHAR*) request.ifr_hwaddr.sa_data);
		}
	
	close (fd);
#endif

	return count;
}

QUAD MACAddress::getAddress(int length, UCHAR *bytes)
{
	QUAD address = 0;

	for (int n = 0; n < length; ++n)
		address = address << 8 | bytes [n];

	return address;
}

int MACAddress::getAddressCount()
{
	return count;
}

QUAD MACAddress::getAddress(int index)
{
	return (index < 0 || index >= count) ? 0 : macAddresses [index];
}

QUAD MACAddress::getAddress(const char *string)
{
	QUAD address = 0;

	for (const char *p = string; *p;)
		{
		int n = getHexDigit (*p++);
		if (n >= 0)
			{
			int hex = n;
			for (; *p && (n = getHexDigit (*p)) >= 0; ++p)
				hex = hex << 4 | n;
			address = address << 8 | hex;
			}
		}

	return address;
}

int MACAddress::getHexDigit(char c)
{
	if (c >= '0' && c <= '9')
		return c - '0';

	if (c >= 'a' && c <= 'f')
		return 10 + c - 'a';

	if (c >= 'A' && c <= 'F')
		return 10 + c - 'A';

	return -1;
}

bool MACAddress::isAddress(QUAD address)
{
	for (int n = 0; n < count; ++n)
		if (address == macAddresses [n])
			return true;

	return false;
}

JString MACAddress::getAddressString(QUAD address)
{
	char string [32];
	char *p = string;

	for (int shift = 40; shift >= 0; shift -= 8)
		{
		int byte = (int) ((address >> shift) & 0xff);
		*p++ = hexDigits [byte >> 4];
		*p++ = hexDigits [byte & 0xf];
		if (shift)
			*p++ = '-';
		}

	*p = 0;

	return string;
}
