/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// IndexKey.cpp: implementation of the IndexKey class.
//
//////////////////////////////////////////////////////////////////////

#include <memory.h>
#include "Engine.h"
#include "IndexKey.h"
#include "Index.h"

#define SHIFT(n)	((uint64) 1 << n)

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

IndexKey::IndexKey(Index *idx)
{
	index = idx;
	keyLength = 0;
}

IndexKey::IndexKey(IndexKey *indexKey)
{
	index = indexKey->index;
	keyLength = indexKey->keyLength;
	memcpy (key, indexKey->key, keyLength);
}

IndexKey::IndexKey()
{
	index = NULL;
	keyLength = 0;
}

IndexKey::IndexKey(int length, const unsigned char *indexKey)
{
	index = NULL;
	keyLength = length;
	memcpy(key, indexKey, length);
}

IndexKey::~IndexKey()
{

}

void IndexKey::setKey(IndexKey *indexKey)
{
	index = indexKey->index;
	keyLength = indexKey->keyLength;
	memcpy (key, indexKey->key, keyLength);
}


void IndexKey::setKey(int length, const unsigned char* keyPtr)
{
	keyLength = length;
	memcpy(key, keyPtr, keyLength);
}

bool IndexKey::isEqual(IndexKey *indexKey)
{
	if (indexKey->keyLength != keyLength)
		return false;

	return memcmp(key, indexKey->key, keyLength) == 0;
}

void IndexKey::appendNumber(double number)
{
	union {
		double	dbl;
		QUAD	quad;
		unsigned char	chars [8];
		} stuff;

	stuff.dbl = number;
	unsigned char *p = key + keyLength;

	if (stuff.quad >= 0)
		stuff.quad ^= QUAD_CONSTANT(0x8000000000000000);
	else
		stuff.quad ^= QUAD_CONSTANT(0xffffffffffffffff);

	for (unsigned char *q = stuff.chars + 8; q > stuff.chars; )
		*p++ = *--q;

	while (p > key && p [-1] == 0)
		--p;

	keyLength = p - key;
}

void IndexKey::appendRecordNumber(int32 recordNumber)
{
	int len = (recordNumber < SHIFT(8))  ? 2 :
			  (recordNumber < SHIFT(16)) ? 3 :
			  (recordNumber < SHIFT(24)) ? 4 : 5;

	int shift = len * 8 - 12;	
	key[keyLength++] = (len << 4) | ((recordNumber >>shift) & 0x0f);
	
	for (shift -= 8; shift > 0; shift -= 8)
		key[keyLength++] = recordNumber >> shift;
	
	key[keyLength++] = (recordNumber << 4) | len;
	ASSERT(getNumberLength() == len);
	const UCHAR *p = key + keyLength - len;
	int32 value = *p++ & 0x0f;
	
	for (int length = len; length > 2; --length)
		value = (value << 8) | *p++;
	
	value = (value << 4) | (*p >> 4);
	ASSERT(value == recordNumber);
	/***
	for (int n = 24; n >= 0; n -= 8)
		key[keyLength++] = recordNumber >> n;
	***/
}

int IndexKey::checkTail(const unsigned char* endPoint)
{
	for (const UCHAR *p = endPoint, *end = key + keyLength; p < end; ++p)
		if (*p)
			return 1;
	
	return -1;
}

void IndexKey::setKey(int offset, int length, const UCHAR *data)
{
	memcpy(key + offset, data, length);
	length = offset + length;
}

int IndexKey::compare(IndexKey* indexKey)
{
	int l = MIN(keyLength, indexKey->keyLength);
	int ret = memcmp(indexKey->key, key, l);
	
	if (ret)
		return ret;
	
	return indexKey->keyLength - keyLength;
}

int IndexKey::compare(IndexKey* indexKey, int recordNumber1, int recordNumber2)
{
	int ret = compare(indexKey);
	
	if (ret)
		return ret;
	
	return recordNumber1 - recordNumber2;
}
