/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/

/* $Id: header,v 1.17 2006/01/01 13:09:48 sniper Exp $ */
#include "php.h"
#include "mysqlnd.h"
#include "mysqlnd_priv.h"

#define MYSQLND_SILENT
#define MYSQLND_DONT_DUMP_STATS


/* {{{ mysqlnd_palloc_init_cache */
MYSQLND_ZVAL_PCACHE* mysqlnd_palloc_init_cache(unsigned int cache_size)
{
	MYSQLND_ZVAL_PCACHE *ret = calloc(1, sizeof(MYSQLND_ZVAL_PCACHE));
	unsigned int i;
#ifndef MYSQLND_SILENT
	php_printf("[mysqlnd_palloc_init_cache %p]\n", ret);
#endif

	ret->max_items = cache_size;
	ret->free_items = cache_size;
	ret->references = 1;

	/* 1. First initialize the free list part of the structure */
	ret->free_list		= calloc(1, sizeof(mysqlnd_ndzval_list));

	/* One more for empty position of last_added - always 0x0, bounds checking */
	ret->free_list->ptr_line = calloc(ret->max_items + 1, sizeof(mysqlnd_zval *));
	ret->free_list->last_added = ret->free_list->ptr_line + ret->max_items;

	/* 2. Allocate and initialize our zvals and initialize the free list */
	ret->block = calloc(ret->max_items, sizeof(mysqlnd_zval));
	ret->last_in_block = &(ret->block[ret->max_items]);
	for (i = 0; i < ret->max_items; i++) {
		/* 1. Initialize */
		INIT_PZVAL(&(ret->block[i].zv));
		ZVAL_NULL(&(ret->block[i].zv));
		/* Assure it will never be freed before MSHUTDOWN */
		ZVAL_ADDREF(&(ret->block[i].zv));
		/* 2. Add to the free list  */
		*(--ret->free_list->last_added) = &(ret->block[i]);
	}
	

	return ret;
}
/* }}} */


/* {{{ mysqlnd_palloc_get_cache_reference */
MYSQLND_ZVAL_PCACHE* mysqlnd_palloc_get_cache_reference(MYSQLND_ZVAL_PCACHE *cache)
{
	if (cache) {
		cache->references++;
	}
	return cache;
}
/* }}} */


/* {{{ mysqlnd_palloc_free_cache */
/*
  As this call will happen on MSHUTDOWN(), then we don't need to copy the zvals with
  copy_ctor but scrap what they point to with zval_dtor() and then just free our
  pre-allocated block. Precondition is that we ZVAL_NULL() the zvals when we put them
  to the free list after usage. We ZVAL_NULL() them when we allocate them in the 
  constructor of the cache.
*/
static
void mysqlnd_palloc_free_cache(MYSQLND_ZVAL_PCACHE *cache)
{
#ifndef MYSQLND_SILENT
	php_printf("[mysqlnd_palloc_free_cache %p]\n", cache);
#endif
	/* Data in pointed by 'block' was cleaned in RSHUTDOWN */
	free(cache->block);
	free(cache->free_list->ptr_line);
	free(cache->free_list);
	free(cache);
}
/* }}} */


/* {{{ mysqlnd_palloc_free_cache_reference */
void mysqlnd_palloc_free_cache_reference(MYSQLND_ZVAL_PCACHE **cache)
{
#ifndef MYSQLND_SILENT
	php_printf("[mysqlnd_palloc_free_cache_reference %p] refs=%d\n", *cache, (*cache)->references);
#endif
	if (*cache && --(*cache)->references == 0) {
		mysqlnd_palloc_free_cache(*cache);
	}
	*cache = NULL;
}
/* }}} */


/*
  The cache line is a big contiguous array of zval pointers.
  Because the CPU cache will cache starting from an address, and not
  before it, then we have to organize our structure according to this.
  Thus, if 'last_added' is valid pointer (not NULL) then last_added is
  increased. When zval is cached, if there is room, last_added is decreased
  and then the zval pointer will be assigned to it. This means that some
  positions may become hot points and stay in the cache.
  Imagine we have 5 pointers in a line
  1. last_added = list_item->ptr_line + cache->max_items;
  2. get_zval -> *last_added = NULL. Use MAKE_STD_ZVAL
  3. get_zval -> *last_added = NULL. Use MAKE_STD_ZVAL
  4. get_zval -> *last_added = NULL. Use MAKE_STD_ZVAL
    0x0
 	0x0
 	0x0
 	0x0 
 	0x0
	---
    empty_position, always 0x0 <-- last_added

  5. free_zval -> if (free_items++ != max_items) {// we can add more
                    *(--last_added) = zval_ptr;
 				 }
	(memory addresses increase downwards)
    0x0
	0x0
	0x0
	0x0 
	0xA <-- last_added
	---
	0x0

  6. free_zval -> if (free_items++ != max_items) {// we can add more
                   *(--last_added) = zval_ptr;
				 }
    0x0
	0x0
	0x0
	0xB <-- last_added
	0xA 
	---
	0x0

  7. free_zval -> if (free_items++ != max_items) {// we can add more
                   *(--last_added) = zval_ptr;
				 }
    0x0
	0x0
	0xC <-- last_added
	0xB 
	0xA 
	---
	0x0

  8. get_zval -> *last_added != NULL. -> p = *last_added; *last_added++ = NULL;
    0x0
	0x0
	0x0 
	0xB <-- last_added
	0xA 
	---
	0x0

  9. get_zval -> *last_added != NULL. -> p = *last_added; *last_added++ = NULL;
    0x0
	0x0
	0x0 
	0x0 
	0xA <-- last_added
	---
	0x0

  10. get_zval -> *last_added != NULL. -> p = *last_added; *last_added++ = NULL;
    0x0
	0x0
	0x0 
	0x0 
	0x0 
	---
	0x0 <-- last_added

*/


/* {{{ mysqlnd_palloc_get_zval */
void *mysqlnd_palloc_get_zval(MYSQLND_ZVAL_PCACHE * const cache, zend_bool *allocated)
{
	void *ret = NULL;

#ifndef MYSQLND_SILENT
	php_printf("[mysqlnd_palloc_get_zval %p] *last_added=%p free_items=%d\b",
				cache, cache? cache->free_list->last_added:NULL, cache->free_items);
#endif

	if (cache) {
		if ((ret = *cache->free_list->last_added)) {
			*cache->free_list->last_added++ = NULL;
			--cache->free_items;
			*allocated = FALSE;
			++cache->get_hits;
		} else {
			++cache->get_misses;
		}
	}
	if (!ret) {
		/*
		  We allocate a bit more. The user of this function will use it, but at
		  end it will use only the zval part. Because the zval part is first then
		  when freeing the zval part the whole allocated block will be cleaned, not
		  only the zval part (by the Engine when destructing the zval).
		*/
		ALLOC_ZVAL(ret);
		INIT_PZVAL((zval *) ret);
		*allocated = TRUE;
	} else {
		/* This will set the refcount to 1, increase it, to keep the variable */
		INIT_PZVAL(&((mysqlnd_zval *) ret)->zv);	
		ZVAL_ADDREF(&(((mysqlnd_zval *)ret)->zv));
	}

	return ret;
}
/* }}} */


/* {{{ mysqlnd_palloc_zval_ptr_dtor */
void mysqlnd_palloc_zval_ptr_dtor(zval **zv, MYSQLND_ZVAL_PCACHE * const cache, zend_bool ps)
{
#ifndef MYSQLND_SILENT
	php_printf("[mysqlnd_palloc_zval_ptr_dtor %p] *zv=%p ps=%d refc=%d\n", cache, *zv, ps, ZVAL_REFCOUNT(*zv));
#endif
	/* refcount is always > 1, because we call ZVAL_ADDREF(). Thus test refcount > 2 */
	if (!cache || ((char *)*zv < (char *)cache->block || (char *)*zv > (char *)cache->last_in_block)) {
		/* 
		  This zval is not from the cache block.
		  Thus the refcount is -1 than of a zval from the cache, as the
		  cache owns the latter.
		*/
		if (ZVAL_REFCOUNT(*zv) > 1) {
			if (!ps) {
				/*
				  Not a prepared statement, then we have to
				  call copy_ctor and then zval_ptr_dtor()
				*/
				zval_copy_ctor(*zv);
			} else {
				/* For PS we copy stndup() the row buffers, so do nothing here */
			}
		} else if (ps == FALSE) {
			/* Prevent Zend from freeing PS allocated data */
			ZVAL_NULL(*zv);
		}
		zval_ptr_dtor(zv);
		return;
	}

	/* The zval is from our cache */
	if (ZVAL_REFCOUNT(*zv) > 2) {
		++cache->put_misses;
		/*
		  Because the zval is first in mysqlnd_zval structure, then we can do
		  upcasting from zval to mysqlnd_zval here, because we know that this
		  zval is part of our pre-allocated block.

		  Now check whether this zval points to ZE allocated memory our to our
		  buffers. If so, call copy_ctor() which will do estrndup for strings.
		  And nothing for null, int, double.
		*/
		if (((mysqlnd_zval *)*zv)->ze_alloced == FALSE) {
			zval_copy_ctor(*zv);
			((mysqlnd_zval *)*zv)->ze_alloced = TRUE;
		}
		/*
		  This will decrease the counter of the user-level (mysqlnd). When the engine
		  layer (the script) has finished working this this zval, when the variable is
		  no more used, out of scope whatever, then it will try zval_ptr_dtor() but
		  and the refcount will reach 1 and the engine won't try to destruct the
		  memory allocated by us.
		*/
		zval_ptr_dtor(zv);
		
		/*
		  Unfortunately, we can't return this variable to the free_list
		  because it's still used. And this cleaning up will happen at request
		  shutdown :(.
		*/
	} else {
		/* No user reference */
		++cache->put_hits;
		++cache->free_items;
		if (((mysqlnd_zval *)*zv)->ze_alloced == TRUE) {
			zval_dtor(*zv);
			((mysqlnd_zval *)*zv)->ze_alloced = FALSE;	
		}
		ZVAL_DELREF(*zv);	/* Make it 1 */
		ZVAL_NULL(*zv);
		*(--cache->free_list->last_added) = (mysqlnd_zval *)*zv;
	}
}
/* }}} */


/* {{{ mysqlnd_palloc_rinit */
void mysqlnd_palloc_rinit(MYSQLND_ZVAL_PCACHE * const cache)
{
	return;
}
/* }}} */


/* {{{ mysqlnd_palloc_rshutdown */
void mysqlnd_palloc_rshutdown(MYSQLND_ZVAL_PCACHE * const cache)
{
	unsigned int i;
#ifndef MYSQLND_SILENT
	php_printf("[mysqlnd_palloc_rshutdown %p]\n", cache);
#endif
	if (!cache) {
		return;
	}

	for (i = 0; i < cache->max_items; i++) {
		/*
		  For PS, could be for Unicode and normal statements too, when
		  implemented.
		*/
		if (cache->block[i].ze_alloced == TRUE) {
			zval_dtor(&(cache->block[i].zv));
			cache->block[i].ze_alloced = FALSE;
		}
	}
}
/* }}} */

void mysqlnd_palloc_stats(const MYSQLND_ZVAL_PCACHE * const cache, zval *return_value)
{
	if (cache) {
		array_init(return_value);
		add_assoc_long_ex(return_value, "put_hits",		sizeof("put_hits"), 	cache->put_hits);
		add_assoc_long_ex(return_value, "put_misses",	sizeof("put_misses"),	cache->put_misses);
		add_assoc_long_ex(return_value, "get_hits", 	sizeof("get_hits"), 	cache->get_hits);
		add_assoc_long_ex(return_value, "get_misses",	sizeof("get_misses"), 	cache->get_misses);
		add_assoc_long_ex(return_value, "size", 		sizeof("size"), 		cache->max_items);
		add_assoc_long_ex(return_value, "free_items",	sizeof("free_items"),	cache->free_items);
		add_assoc_long_ex(return_value, "references",	sizeof("references"),	cache->references);
	} else {
		ZVAL_NULL(return_value);
	}
}
