/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/

/* $Id: header,v 1.17 2006/01/01 13:09:48 sniper Exp $ */

#ifndef MYSQLND_H
#define MYSQLND_H


#include "portability.h"

#ifdef ZTS
#include "TSRM.h"
#endif

#define MYSQLND_VERSION "mysqlnd 5.0.0-alpha - 50100 - $Revision: 14 $"
#define MYSQLND_VERSION_ID 50100


#ifdef PHP_WIN32
#define MYSQLND_LLU_SPEC "%I64u"
#define MYSQLND_LL_SPEC "%I64d"
#define L64(x) x##i64
typedef unsigned __int64 my_uint64;
typedef __int64 my_int64;
typedef unsigned __int64 mynd_ulonglong;
typedef __int64 mynd_longlong;
#else
#define MYSQLND_LLU_SPEC "%llu"
#define MYSQLND_LL_SPEC "%lld"
#define L64(x) x##LL
typedef unsigned long long my_uint64;
typedef long long my_int64;
typedef unsigned long long mynd_ulonglong;
typedef long long mynd_longlong;
#endif


#define MYSQLND_ERRMSG_SIZE			512
#define MYSQLND_SQLSTATE_LENGTH		5
#define MYSQLND_SQLSTATE_NULL		"00000"

#define MYSQLND_SERVER_QUERY_NO_GOOD_INDEX_USED	16
#define MYSQLND_SERVER_QUERY_NO_INDEX_USED		32

#define MYSQLND_NO_DATA			100
#define MYSQLND_DATA_TRUNCATED	101

#define SHA1_MAX_LENGTH 20 
#define SCRAMBLE_LENGTH 20
#define SCRAMBLE_LENGTH_323 8

#define CLIENT_LONG_PASSWORD	1	/* new more secure passwords */
#define CLIENT_FOUND_ROWS	2	/* Found instead of affected rows */
#define CLIENT_LONG_FLAG	4	/* Get all column flags */
#define CLIENT_CONNECT_WITH_DB	8	/* One can specify db on connect */
#define CLIENT_NO_SCHEMA	16	/* Don't allow database.table.column */
#define CLIENT_COMPRESS		32	/* Can use compression protocol */
#define CLIENT_ODBC		64	/* Odbc client */
#define CLIENT_LOCAL_FILES	128	/* Can use LOAD DATA LOCAL */
#define CLIENT_IGNORE_SPACE	256	/* Ignore spaces before '(' */
#define CLIENT_PROTOCOL_41	512	/* New 4.1 protocol */
#define CLIENT_INTERACTIVE	1024	/* This is an interactive client */
#define CLIENT_SSL              2048	/* Switch to SSL after handshake */
#define CLIENT_IGNORE_SIGPIPE   4096    /* IGNORE sigpipes */
#define CLIENT_TRANSACTIONS	8192	/* Client knows about transactions */
#define CLIENT_RESERVED         16384   /* Old flag for 4.1 protocol  */
#define CLIENT_SECURE_CONNECTION 32768  /* New 4.1 authentication */
#define CLIENT_MULTI_STATEMENTS (1UL << 16) /* Enable/disable multi-stmt support */
#define CLIENT_MULTI_RESULTS    (1UL << 17) /* Enable/disable multi-results */

enum
{
	MYSQLND_FETCH_ASSOC = 1,
	MYSQLND_FETCH_NUM = 2,
	MYSQLND_FETCH_BOTH = 1|2,
};

/* Follow libmysql convention */
typedef enum func_status
{
	PASS = 0,
	FAIL = 1,
} enum_func_status;

typedef enum mysqlnd_query_type
{
	QUERY_UPSERT,
	QUERY_SELECT,
	QUERY_LOAD_LOCAL
} enum_mysqlnd_query_type;

typedef enum mysqlnd_res_type
{
	MYSQLND_RES_NORMAL = 1,
	MYSQLND_RES_PS
} enum_mysqlnd_res_type;

typedef enum mysqlnd_option
{
	MYSQL_OPT_CONNECT_TIMEOUT,
	MYSQL_OPT_COMPRESS,
	MYSQL_OPT_NAMED_PIPE,
	MYSQL_INIT_COMMAND,
	MYSQL_READ_DEFAULT_FILE,
	MYSQL_READ_DEFAULT_GROUP,
	MYSQL_SET_CHARSET_DIR,
	MYSQL_SET_CHARSET_NAME,
	MYSQL_OPT_LOCAL_INFILE,
	MYSQL_OPT_PROTOCOL,
	MYSQL_SHARED_MEMORY_BASE_NAME,
	MYSQL_OPT_READ_TIMEOUT,
	MYSQL_OPT_WRITE_TIMEOUT,
	MYSQL_OPT_USE_RESULT,
	MYSQL_OPT_USE_REMOTE_CONNECTION,
	MYSQL_OPT_USE_EMBEDDED_CONNECTION,
	MYSQL_OPT_GUESS_CONNECTION,
	MYSQL_SET_CLIENT_IP,
	MYSQL_SECURE_AUTH,
	MYSQL_REPORT_DATA_TRUNCATION,
	MYSQL_OPT_RECONNECT,
	MYSQL_OPT_SSL_VERIFY_SERVER_CERT
} enum_mysqlnd_option;


typedef enum mysqlnd_field_types
{
	MYSQL_TYPE_DECIMAL, 
	MYSQL_TYPE_TINY,
	MYSQL_TYPE_SHORT,
	MYSQL_TYPE_LONG,
	MYSQL_TYPE_FLOAT,
	MYSQL_TYPE_DOUBLE,
	MYSQL_TYPE_NULL,
	MYSQL_TYPE_TIMESTAMP,
	MYSQL_TYPE_LONGLONG,
	MYSQL_TYPE_INT24,
	MYSQL_TYPE_DATE,
	MYSQL_TYPE_TIME,
	MYSQL_TYPE_DATETIME,
	MYSQL_TYPE_YEAR,
	MYSQL_TYPE_NEWDATE,
	MYSQL_TYPE_VARCHAR,
	MYSQL_TYPE_BIT,
	MYSQL_TYPE_NEWDECIMAL=246,
	MYSQL_TYPE_ENUM=247,
	MYSQL_TYPE_SET=248,
	MYSQL_TYPE_TINY_BLOB=249,
	MYSQL_TYPE_MEDIUM_BLOB=250,
	MYSQL_TYPE_LONG_BLOB=251,
	MYSQL_TYPE_BLOB=252,
	MYSQL_TYPE_VAR_STRING=253,
	MYSQL_TYPE_STRING=254,
	MYSQL_TYPE_GEOMETRY=255
} enum_mysqlnd_field_types;

/* Please update this if there is a new type after MYSQL_TYPE_GEOMETRY */
#define MYSQL_TYPE_LAST 	MYSQL_TYPE_GEOMETRY


typedef enum mysqlnd_server_option
{
	MYSQL_OPTION_MULTI_STATEMENTS_ON,
	MYSQL_OPTION_MULTI_STATEMENTS_OFF

} enum_mysqlnd_server_option;


#define FIELD_TYPE_DECIMAL     MYSQL_TYPE_DECIMAL
#define FIELD_TYPE_NEWDECIMAL  MYSQL_TYPE_NEWDECIMAL
#define FIELD_TYPE_TINY        MYSQL_TYPE_TINY
#define FIELD_TYPE_SHORT       MYSQL_TYPE_SHORT
#define FIELD_TYPE_LONG        MYSQL_TYPE_LONG
#define FIELD_TYPE_FLOAT       MYSQL_TYPE_FLOAT
#define FIELD_TYPE_DOUBLE      MYSQL_TYPE_DOUBLE
#define FIELD_TYPE_NULL        MYSQL_TYPE_NULL
#define FIELD_TYPE_TIMESTAMP   MYSQL_TYPE_TIMESTAMP
#define FIELD_TYPE_LONGLONG    MYSQL_TYPE_LONGLONG
#define FIELD_TYPE_INT24       MYSQL_TYPE_INT24
#define FIELD_TYPE_DATE        MYSQL_TYPE_DATE
#define FIELD_TYPE_TIME        MYSQL_TYPE_TIME
#define FIELD_TYPE_DATETIME    MYSQL_TYPE_DATETIME
#define FIELD_TYPE_YEAR        MYSQL_TYPE_YEAR
#define FIELD_TYPE_NEWDATE     MYSQL_TYPE_NEWDATE
#define FIELD_TYPE_ENUM        MYSQL_TYPE_ENUM
#define FIELD_TYPE_SET         MYSQL_TYPE_SET
#define FIELD_TYPE_TINY_BLOB   MYSQL_TYPE_TINY_BLOB
#define FIELD_TYPE_MEDIUM_BLOB MYSQL_TYPE_MEDIUM_BLOB
#define FIELD_TYPE_LONG_BLOB   MYSQL_TYPE_LONG_BLOB
#define FIELD_TYPE_BLOB        MYSQL_TYPE_BLOB
#define FIELD_TYPE_VAR_STRING  MYSQL_TYPE_VAR_STRING
#define FIELD_TYPE_STRING      MYSQL_TYPE_STRING
#define FIELD_TYPE_CHAR        MYSQL_TYPE_TINY
#define FIELD_TYPE_INTERVAL    MYSQL_TYPE_ENUM
#define FIELD_TYPE_GEOMETRY    MYSQL_TYPE_GEOMETRY
#define FIELD_TYPE_BIT         MYSQL_TYPE_BIT

#define NOT_NULL_FLAG			    1
#define PRI_KEY_FLAG			    2
#define UNIQUE_KEY_FLAG			    4
#define MULTIPLE_KEY_FLAG		    8
#define BLOB_FLAG				   16
#define UNSIGNED_FLAG			   32
#define ZEROFILL_FLAG			   64
#define BINARY_FLAG				  128	
#define ENUM_FLAG				  256
#define AUTO_INCREMENT_FLAG		  512
#define TIMESTAMP_FLAG			 1024
#define SET_FLAG				 2048
#define NO_DEFAULT_VALUE_FLAG	 4096
#define PART_KEY_FLAG			16384
#define GROUP_FLAG				32768
#define NUM_FLAG				32768

#define IS_PRI_KEY(n)	((n) & PRI_KEY_FLAG)
#define IS_NOT_NULL(n)	((n) & NOT_NULL_FLAG)
#define IS_BLOB(n)		((n) & BLOB_FLAG)
#define IS_NUM(t)		((t) <= FIELD_TYPE_INT24 || (t) == FIELD_TYPE_YEAR || (t) == FIELD_TYPE_NEWDECIMAL)


/*
          /-----> CONN_CLOSE  <---------------\
         |           ^                         \
         |           |                         \
	CONN_READY -> CONN_QUERY_SENT -> CONN_FETCHING_DATA 
	    ^                                      |
		\-------------------------------------/
*/
typedef enum mysqlnd_connection_state
{
	CONN_ALLOCED = 0,
	CONN_READY,
	CONN_QUERY_SENT,
	CONN_FETCHING_DATA,
	CONN_NEXT_RESULT_PENDING,
	CONN_QUIT_SENT, /* object is "destroyed" at this stage */
} enum_mysqlnd_connection_state;


typedef enum mysqlnd_stmt_state {
	MYSQLND_STMT_INITTED = 0,
	MYSQLND_STMT_PREPARED,
	MYSQLND_STMT_EXECUTED,
	MYSQLND_STMT_FETCHED,
	MYSQLND_STMT_USER_FETCHING,
} enum_mysqlnd_stmt_state;


typedef struct _mysqlnd_cmd_buffer {
	zend_uchar		*buffer;
	size_t			length;
} MYSQLND_CMD_BUFFER;


typedef struct _mysqlnd_field {
	char *name;					/* Name of column */
	char *org_name;				/* Original column name, if an alias */
	char *table;				/* Table of column if column was a field */
	char *org_table;			/* Org table name, if table was an alias */
	char *db;					/* Database for table */
	char *catalog;				/* Catalog for table */
	char *def;					/* Default value (set by mysql_list_fields) */
	unsigned long length;		/* Width of column (create length) */
	unsigned long max_length;	/* Max width for selected set */
	unsigned int name_length;
	unsigned int org_name_length;
	unsigned int table_length;
	unsigned int org_table_length;
	unsigned int db_length;
	unsigned int catalog_length;
	unsigned int def_length;
	unsigned int flags;			/* Diverse flags */
	unsigned int decimals;		/* Number of decimals in field */
	unsigned int charsetnr;		/* Character set */
	enum mysqlnd_field_types type;	/* Type of field. See mysql_com.h for types */
	char *root;
} MYSQLND_FIELD;



typedef struct _mysqlnd_upsert_result {
	unsigned int		warning_count;
	unsigned int		server_status;
	unsigned long long	affected_rows;
	unsigned long long	last_insert_id;
} mysqlnd_upsert_status;


typedef struct _mysqlnd_error_info {
	char error[MYSQLND_ERRMSG_SIZE+1];
	char sqlstate[MYSQLND_SQLSTATE_LENGTH + 1];
	unsigned int error_no;
} mysqlnd_error_info;


typedef struct _mysqlnd_zval_cache	MYSQLND_ZVAL_CACHE;
typedef struct _mysqlnd_zval_pcache	MYSQLND_ZVAL_PCACHE;


typedef struct _mysqlnd_infile_info {
	php_stream			*fd;
	int					error_no;
	char				error_msg[MYSQLND_ERRMSG_SIZE + 1];
	const char			*filename;
	zval				*callback;
} MYSQLND_INFILE_INFO;

/* character set information */
typedef struct _mysqlnd_charset
{
	uint	nr;
	char	*name;
	char	*collation;
	uint	char_minlen;
	uint	char_maxlen;
	uint	dangerous_for_escape_backslash;
	uint 	(*mb_charlen)(uint c);	
	uint 	(*mb_valid)(const char *start, const char *end);	
} MYSQLND_CHARSET;

/* local infile handler */
typedef struct _mysqlnd_infile
{
	int		(*local_infile_init)(void **ptr, char *filename, void **userdata TSRMLS_DC);
	int		(*local_infile_read)(void *ptr, char *buf, uint buf_len TSRMLS_DC);
	int		(*local_infile_error)(void *ptr, char *error_msg, uint error_msg_len TSRMLS_DC);
	void	(*local_infile_end)(void *ptr TSRMLS_DC);
	zval	*callback;
	void	*userdata;
} MYSQLND_INFILE;

typedef struct _mysqlnd_option {
	/* timeouts */
	uint		timeout_connect;
	uint		timeout_read;
	uint		timeout_write;

	ulong		flags;

	/* init commands - we need to send them to server directly after connect */
	uint		num_commands;
	char		**init_commands;

	/* configuration file information */
	char 		*cfg_file;
	char		*cfg_section;

	/* SSL information */
	char		*ssl_key;
	char		*ssl_cert;
	char		*ssl_ca;
	char		*ssl_capath;
	char		*ssl_cipher;
	zend_bool	use_ssl;

	/* maximum allowed packet size for communication */
	ulong	max_allowed_packet;
} MYSQLND_OPTION;


typedef struct _mysqlnd_res MYSQLND_RES;
typedef char** 				MYSQLND_ROW;		/* return data as array of strings */


typedef struct mysqlnd_connection {
	php_stream		*stream;
/* Information related */
	char			*host;
	char			*unix_socket;
	char			*user;
	char			*passwd;
	unsigned int	*passwd_len;
	char			*scheme;
	unsigned long long	thread_id;
	char			*server_version;
	char			*host_info;
	char			*scramble;
	MYSQLND_CHARSET *charset;
	MYSQLND_INFILE  infile;
	unsigned int	protocol_version;
	unsigned long	max_packet_size;
	unsigned int	port;
	unsigned long	client_flag;
	unsigned long	server_capabilities;

	int				tmp_int;


	/* For UPSERT queries */
	mysqlnd_upsert_status upsert_status;
	char			*last_message;
	unsigned int	last_message_len;

	/* If error packet, we use these */
	mysqlnd_error_info	error_info;

/* Operation related */
	/* sequence for simple checking of correct packets */
	zend_uchar		packet_no;

	/*
	  To prevent queries during unbuffered fetches. Also to
	  mark the connection as destroyed for garbage collection.
	*/
	enum mysqlnd_connection_state	state;
	enum_mysqlnd_query_type			last_query_type;
	/* Temporary storage between query and (use|store)_result() call */
	MYSQLND_RES						*current_result;

	/*
	  How many result sets reference this connection.
	  It won't be freed until this number reaches 0.
	  The last one, please close the door! :-)
	  The result set objects can determine by inspecting
	  'quit_sent' whether the connection is still valid.
	*/
	unsigned int	references;

	/* Temporal storage for mysql_query */
	unsigned int	field_count;

	/* multi query mode */
	zend_bool		multistatements;
	
	/* persistent connection */
	zend_bool		persistent;

	/* options */
	MYSQLND_OPTION	options;

	/* zval cache */
	MYSQLND_ZVAL_PCACHE	*zval_cache;

	/* cmd buffer */
	MYSQLND_CMD_BUFFER	cmd_buffer;
} MYSQLND;


struct _mysqlnd_res {
	enum_func_status	(*fetch_row)(MYSQLND_RES *result, void *param,
									 unsigned int flags,
									 zend_bool *fetched_anything TSRMLS_DC);
	unsigned long *		(*fetch_lengths)(MYSQLND_RES *result);
	MYSQLND				*conn;
	enum_mysqlnd_res_type type;
	unsigned int		field_count;

	/* For metadata functions */
	MYSQLND_FIELD		*fields;
	unsigned int		current_field;
	unsigned long		*zend_hash_keys;

	/* To be used with store_result() - both normal and PS */
	zval				***data;
	zval				***data_cursor;
	zend_uchar			**row_buffers;


	/* For unbuffered (both normal and PS) */
	zval				**last_row_data;
	zend_uchar			*last_row_buffer;

	/* For old-API myslqnd_fetch_row() */
	MYSQLND_ROW			last_row;

	/*
	  Column lengths of current row - both buffered and unbuffered.
	  For buffered results it duplicates the data found in **data 
	*/
	unsigned long		*lengths;

	/* To be used with use_result() */
	zend_bool			eof_reached;

	/*
	  This is set at once with store_result() or grows with
	  use_result(). To check whether the result set is unbuffered
	  use 'conn', which is non-NULL in this case.
	*/
	mynd_ulonglong		row_count;

	/*
	  For future use in php_mysqlnd_rowp_read() .
	  For one time allocation of the row buffer
	*/
	char			*row_buffer;
	unsigned int 	row_buffer_len;

	void			*row_packet;

	/* zval cache */
	MYSQLND_ZVAL_PCACHE	*zval_cache;
};


enum mysqlnd_stmt_attr
{
	STMT_ATTR_UPDATE_MAX_LENGTH,
	STMT_ATTR_CURSOR_TYPE,
	STMT_ATTR_PREFETCH_ROWS
};

enum myslqnd_cursor_type
{
	CURSOR_TYPE_NO_CURSOR= 0,
	CURSOR_TYPE_READ_ONLY= 1,
	CURSOR_TYPE_FOR_UPDATE= 2,
	CURSOR_TYPE_SCROLLABLE= 4
};

#define DEFAULT_PREFETCH_ROWS (ulong) 1

typedef struct _mysqlnd_param_bind {
	zval		*zv;
	zend_uchar	type;
} MYSQLND_PARAM_BIND;

typedef struct _mysqlnd_result_bind {
	zval		*zv;
	zend_uchar	original_type;
	zend_bool	bound;
} MYSQLND_RESULT_BIND;


typedef struct _mysqlnd_stmt {
	MYSQLND						*conn;
	unsigned long				stmt_id;
	unsigned long				flags;/* cursor is set here */
	enum_mysqlnd_stmt_state		state;
	unsigned int				warning_count;
	MYSQLND_RES					*result;
	unsigned int				field_count;
	unsigned int				param_count;
	unsigned char				send_types_to_server;
	MYSQLND_PARAM_BIND			*param_bind;
	MYSQLND_RESULT_BIND			*result_bind;

	mysqlnd_upsert_status		upsert_status;

	mysqlnd_error_info			error_info;

	zend_bool					update_max_length;
	unsigned long				prefetch_rows;

	MYSQLND_CMD_BUFFER			cmd_buffer;
} MYSQLND_STMT;





typedef unsigned int MYSQLND_FIELD_OFFSET;

void mysqlnd_conn_rsrc_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC);
void mysqlnd_result_rsrc_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC);
void mysqlnd_stmt_rsrc_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC);
MYSQLND_CHARSET *mysqlnd_find_charset_nr(uint charsetno);
MYSQLND_CHARSET *mysqlnd_find_charset_name(char *charsetname);

/* Connect */
MYSQLND *mysqlnd_init(zend_bool persistent);
MYSQLND *mysqlnd_connect(MYSQLND *conn,
						 char *host, char *user,
						 char *passwd, unsigned int passwd_len,
						 char *db, unsigned int db_len,
						 unsigned int port,
						 char *socket,
						 unsigned int mysql_flags,
						 MYSQLND_ZVAL_PCACHE *zval_cache TSRMLS_DC);
enum_func_status mysqlnd_close(MYSQLND *conn TSRMLS_DC);
enum_func_status mysqlnd_change_user(MYSQLND *conn, const char *user, const char *passwd,
									 const char *db TSRMLS_DC);

/* Query */
#define mysqlnd_fetch_row(result)	\
		_mysqlnd_fetch_row((result) TSRMLS_CC ZEND_FILE_LINE_CC)

#define mysqlnd_fetch_into(result, flags, return_value)	\
		_mysqlnd_fetch_into((result), (flags), (return_value) TSRMLS_CC ZEND_FILE_LINE_CC)

#define mysqlnd_fetch_all(result, flags, return_value)	\
		_mysqlnd_fetch_all((result), (flags), (return_value) TSRMLS_CC ZEND_FILE_LINE_CC)


void 				mysqlnd_start_psession(MYSQLND *conn);
void 				mysqlnd_end_psession(MYSQLND *conn);
MYSQLND_RES *		mysqlnd_use_result(MYSQLND *conn TSRMLS_DC);
MYSQLND_RES *		mysqlnd_store_result(MYSQLND *conn TSRMLS_DC);
enum_func_status	mysqlnd_query(MYSQLND *conn, const char *query, unsigned int query_len TSRMLS_DC);
enum_func_status	mysqlnd_fetch_row_ex(MYSQLND_RES *result, zval *row, unsigned int flags, zend_bool *fetched_anything TSRMLS_DC);
enum_func_status	mysqlnd_free_result(MYSQLND_RES *result TSRMLS_DC);
enum_func_status	mysqlnd_next_result(MYSQLND *conn TSRMLS_DC);
zend_bool 			mysqlnd_more_results(const MYSQLND * const conn);
/*
  Use the macros above, not the actual functions. This will save
  you time of writing TSRMLS_DC ZEND_FILE_LINE_DC
*/
void 				_mysqlnd_fetch_into(MYSQLND_RES *result, unsigned int flags, zval *return_value TSRMLS_DC ZEND_FILE_LINE_DC);
MYSQLND_ROW			_mysqlnd_fetch_row(MYSQLND_RES *result TSRMLS_DC ZEND_FILE_LINE_DC);
void 				_mysqlnd_fetch_all(MYSQLND_RES *result, unsigned int flags, zval *return_value TSRMLS_DC ZEND_FILE_LINE_DC);


/* Query metadata */
unsigned int 		mysqlnd_field_count(const MYSQLND * const conn);
MYSQLND_FIELD *		mysqlnd_fetch_field(MYSQLND_RES *result);
MYSQLND_FIELD *		mysqlnd_fetch_field_direct(MYSQLND_RES *result, MYSQLND_FIELD_OFFSET fieldnr);
unsigned int		mysqlnd_num_fields(const MYSQLND_RES * const res);
mynd_ulonglong		mysqlnd_num_rows(const MYSQLND_RES * const res);
MYSQLND_FIELD_OFFSET mysqlnd_field_seek(MYSQLND_RES *result, MYSQLND_FIELD_OFFSET field_offset);
MYSQLND_FIELD_OFFSET mysqlnd_field_tell(MYSQLND_RES *result);
const char *		mysqlnd_field_type_name(enum mysqlnd_field_types field_type);
unsigned long *		mysqlnd_fetch_lengths(MYSQLND_RES *result TSRMLS_DC);
unsigned long long	mysqlnd_insert_id(const MYSQLND * const conn);
unsigned long long	mysqlnd_affected_rows(const MYSQLND * const conn);
unsigned int		mysqlnd_warning_count(const MYSQLND * const conn);
const char *		mysqlnd_info(const MYSQLND * const conn);
enum_func_status	mysqlnd_data_seek(MYSQLND_RES *result, mynd_ulonglong row);


/* Simple commands */
enum_func_status	mysqlnd_autocommit(MYSQLND *conn, zend_bool mode TSRMLS_DC);
enum_func_status	mysqlnd_commit(MYSQLND *conn TSRMLS_DC);
enum_func_status	mysqlnd_rollback(MYSQLND *conn TSRMLS_DC);
enum_func_status 	mysqlnd_dump_debug_info(MYSQLND *conn TSRMLS_DC);
enum_func_status 	mysqlnd_select_db(MYSQLND *conn, const char * const db, unsigned int db_len TSRMLS_DC);
enum_func_status 	mysqlnd_ping(MYSQLND *conn TSRMLS_DC);
enum_func_status 	mysqlnd_kill(MYSQLND *conn, unsigned long pid TSRMLS_DC);
enum_func_status 	mysqlnd_refresh(MYSQLND *conn, unsigned long options TSRMLS_DC);
enum_func_status 	mysqlnd_shutdown(MYSQLND *conn, unsigned long level TSRMLS_DC);
enum_func_status 	mysqlnd_stat(MYSQLND *conn, char **message, unsigned int * message_len TSRMLS_DC);
enum_func_status 	mysqlnd_set_server_option(MYSQLND *conn, enum_mysqlnd_server_option option TSRMLS_DC);
unsigned long		mysqlnd_get_server_version(const MYSQLND * const conn);
const char			*mysqlnd_get_server_info(const MYSQLND * const conn);
const char			*mysqlnd_get_host_info(const MYSQLND * const conn);
unsigned int		mysqlnd_get_proto_info(const MYSQLND * const conn);
mynd_ulonglong		mysqlnd_thread_id(const MYSQLND *const conn);
unsigned int		mysqlnd_get_client_version();
const char 			*mysqlnd_get_client_info();
void				mysqlnd_local_infile_default(MYSQLND *conn);
void 				mysqlnd_set_local_infile_handler(MYSQLND *conn, char *funcname);

/* Errors */
unsigned int		mysqlnd_errno(const MYSQLND * const conn);
const char *		mysqlnd_error(const MYSQLND * const conn);
const char *		mysqlnd_sqlstate(const MYSQLND * const conn);

/* Misc */
ulong				mysqlnd_real_escape_string(MYSQLND *conn, char *newstr, char *escapestr, int escapestr_len TSRMLS_DC);
enum_func_status	mysqlnd_set_charset(MYSQLND *conn, char *charset TSRMLS_DC);

/* PS */
void 				mysqlnd_init_ps();
MYSQLND_STMT * 		mysqlnd_stmt_init(MYSQLND *conn);
enum_func_status	mysqlnd_stmt_prepare(MYSQLND_STMT *stmt, char *query, unsigned int query_len TSRMLS_DC);
enum_func_status	mysqlnd_stmt_execute(MYSQLND_STMT *stmt TSRMLS_DC);
enum_func_status	mysqlnd_stmt_fetch(MYSQLND_STMT *stmt, zend_bool *fetched_anything TSRMLS_DC);
enum_func_status	mysqlnd_stmt_close(MYSQLND_STMT *stmt TSRMLS_DC);
enum_func_status	mysqlnd_stmt_bind_param(MYSQLND_STMT *stmt, MYSQLND_PARAM_BIND *param_bind);
enum_func_status	mysqlnd_stmt_bind_result(MYSQLND_STMT *stmt, MYSQLND_RESULT_BIND *result_bind);
mynd_ulonglong		mysqlnd_stmt_insert_id(const MYSQLND_STMT * const stmt);
mynd_ulonglong		mysqlnd_stmt_affected_rows(const MYSQLND_STMT * const stmt);
unsigned int		mysqlnd_stmt_field_count(const MYSQLND_STMT * const stmt);
unsigned int		mysqlnd_stmt_param_count(const MYSQLND_STMT * const stmt);
unsigned int		mysqlnd_stmt_warning_count(const MYSQLND_STMT * const stmt);
unsigned int		mysqlnd_stmt_errno(const MYSQLND_STMT * const stmt);
const char *		mysqlnd_stmt_error(const MYSQLND_STMT * const stmt);
const char *		mysqlnd_stmt_sqlstate(const MYSQLND_STMT * const stmt);
enum_func_status	mysqlnd_stmt_attr_set(MYSQLND_STMT *stmt, enum mysqlnd_stmt_attr attr_type, unsigned long value);
enum_func_status	mysqlnd_stmt_attr_get(MYSQLND_STMT *stmt, enum mysqlnd_stmt_attr attr_type, void *value);
enum_func_status	mysqlnd_options(MYSQLND *conn, enum_mysqlnd_option option, const char *value);

#if A0
/* Caching zval allocator */
zval * 				mysqlnd_alloc_get_zval(MYSQLND_ZVAL_CACHE * const cache);
void 				mysqlnd_alloc_zval_ptr_dtor(zval **zv, MYSQLND_ZVAL_CACHE * const cache);
MYSQLND_ZVAL_CACHE* mysqlnd_alloc_init_cache();
MYSQLND_ZVAL_CACHE* mysqlnd_alloc_get_cache_reference(MYSQLND_ZVAL_CACHE *cache);
void				mysqlnd_alloc_free_cache_reference(MYSQLND_ZVAL_CACHE **cache);
#endif

/* Persistent caching zval allocator */
typedef struct _mysqlnd_zval mysqlnd_zval;

MYSQLND_ZVAL_PCACHE* mysqlnd_palloc_init_cache(unsigned int cache_size);
MYSQLND_ZVAL_PCACHE* mysqlnd_palloc_get_cache_reference(MYSQLND_ZVAL_PCACHE *cache);
void				mysqlnd_palloc_free_cache_reference(MYSQLND_ZVAL_PCACHE **cache);
void				mysqlnd_palloc_rinit(MYSQLND_ZVAL_PCACHE * const cache);
void				mysqlnd_palloc_rshutdown(MYSQLND_ZVAL_PCACHE * const cache);

void *				mysqlnd_palloc_get_zval(MYSQLND_ZVAL_PCACHE * const cache, zend_bool *allocated);
void				mysqlnd_palloc_zval_ptr_dtor(zval **zv, MYSQLND_ZVAL_PCACHE * const cache, zend_bool ps);
void				mysqlnd_palloc_stats(const MYSQLND_ZVAL_PCACHE * const cache, zval *return_value);
#endif	/* MYSQLND_H */


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
