/*  $Revision: 1.23 $
**
**  Newsgroups and the active file.
*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "configdata.h"
#include "clibrary.h"
#include "nnrpd.h"
#include "mydir.h"
#if ! defined (NO_MMAP_ACTIVE)
#include <sys/mman.h>
#endif /* ! defined (NO_MMAP_ACTIVE) */


/*
**  Newsgroup hashing stuff.  See comments in innd/ng.c.
*/

#define GRP_HASH(Name, p, j)	\
	for (p = Name, j = 0; *p; ) j = (j << 5) + j + *p++
#define GRP_SIZE	8192
#define GRP_BUCKET(j)	&GRPtable[j & (GRP_SIZE - 1)]

typedef struct _GRPHASH {
#if defined (NO_MMAP_ACTIVE)
    int		Size;
    int		Used;
    GROUPENTRY	**Groups;
#else
    GROUPENTRY	*First;
#endif /* defined (NO_MMAP_ACTIVE) */
} GRPHASH;


STATIC GRPHASH		GRPtable[GRP_SIZE];
STATIC GROUPENTRY	*GRPentries;
STATIC int		GRPbuckets;
STATIC int		GRPsize;


/*
**  See if a given newsgroup exists.
*/
GROUPENTRY *
GRPfind(group)
    register char		*group;
{
    register char		*p;
    register unsigned int	j;
    register int		i;
#if ! defined (NO_MMAP_ACTIVE)
    register GROUPENTRY		*gp;
#else
    register GROUPENTRY		**gpp;
#endif /* ! defined (NO_MMAP_ACTIVE) */
    GRPHASH			*htp;
    char			c;

    /* SUPPRESS 6 *//* Over/underflow from plus expression */
    GRP_HASH(group, p, j);
    htp = GRP_BUCKET(j);
#if ! defined (NO_MMAP_ACTIVE)
    for (c = *group, gp = htp->First; gp; gp = gp->Next) {
        if (c == GPNAME(gp)[0] && EQ(group, GPNAME(gp)))
            return gp;
    }
#else
    for (c = *group, gpp = htp->Groups, i = htp->Used; --i >= 0; gpp++)
	if (c == gpp[0]->Name[0] && EQ(group, gpp[0]->Name))
	    return gpp[0];
#endif /* ! defined (NO_MMAP_ACTIVE) */
    return NULL;
}


STATIC void
GRPhash()
{
    register char		*p;
    register int		i;
    register GROUPENTRY		*gp;
    register unsigned int	j;
    register GRPHASH		*htp;

    /* Set up the default hash buckets. */
    GRPbuckets = GRPsize / GRP_SIZE;
    if (GRPbuckets == 0)
	GRPbuckets = 1;
#if ! defined (NO_MMAP_ACTIVE)
    for (i = 0; i < GRP_SIZE; ++i)
	GRPtable[i].First = NULL;
#else
    if (GRPtable[0].Groups)
	for (i = GRP_SIZE, htp = GRPtable; --i >= 0; htp++)
	    htp->Used = 0;
    else
	for (i = GRP_SIZE, htp = GRPtable; --i >= 0; htp++) {
	    htp->Size = GRPbuckets;
	    htp->Groups = NEW(GROUPENTRY*, htp->Size);
	    htp->Used = 0;
	}
#endif /* ! defined (NO_MMAP_ACTIVE) */

    /* Now put all groups into the hash table. */
    for (i = GRPsize, gp = GRPentries; --i >= 0; gp++) {
	/* SUPPRESS 6 *//* Over/underflow from plus expression */
	GRP_HASH(GPNAME(gp), p, j);
	htp = GRP_BUCKET(j);
#if defined (NO_MMAP_ACTIVE)
	if (htp->Used >= htp->Size) {
	    htp->Size += GRPbuckets;
	    RENEW(htp->Groups, GROUPENTRY*, htp->Size);
	}
	htp->Groups[htp->Used++] = gp;
#else
	gp->Next = htp->First;
	htp->First = gp;
	
#endif /* defined (NO_MMAP_ACTIVE) */
    }

    /* Note that we don't sort the buckets. */
}

#if ! defined (NO_MMAP_ACTIVE)
long XRSize = -1;

char *
XReadInFile(const char *fileName, struct stat *pst)
{
    int fd;
    char *ptr = NULL;
    struct stat st;

    if (pst == NULL)
        pst = &st;

    fd = open(fileName, O_RDONLY);
    if (fd >= 0) {
        if (fstat(fd, pst) == 0) {
            ptr = (char *)mmap((caddr_t)NULL, pst->st_size + 1, PROT_READ, MAP_SHARED, fd, 0);
            if (ptr != (char *)-1) {
                XRSize = pst->st_size + 1;
                close(fd);
                /* syslog(L_ERROR, "%s mapped %d", ClientHost, pst->st_size + 1);*/
            } else {
                close(fd);
                return(ReadInFile(fileName, pst));
            }
        } else {
            close(fd);
        }
    }
    return(ptr);
}

void
XDISPOSE(char *active)
{
    if (XRSize >= 0) {
        munmap(active, XRSize);
        XRSize = -1;
    } else {
        DISPOSE(active);
    }
}

char *
GPNAME(GROUPENTRY *gp)
{
    const char *p = strchr(gp->Ptr, ' ');
    static char GPBuf[1024];

    GPBuf[0] = 0;
    if (p && p - gp->Ptr > 0 && p - gp->Ptr < sizeof(GPBuf) - 1) {
        bcopy(gp->Ptr, GPBuf, p - gp->Ptr);
        GPBuf[p - gp->Ptr] = 0;
    }
    return(GPBuf);
}

ARTNUM
GPHIGH(GROUPENTRY *gp)
{
    const char *p = strchr(gp->Ptr, ' ');
    long n = 0;
    if (p)
        n = strtol(p + 1, NULL, 10);
    return(n);
}

ARTNUM
GPLOW(GROUPENTRY *gp)
{
    const char *p = strchr(gp->Ptr, ' ');
    long n = 0;
    if (p) {
        if ((p = strchr(p + 1, ' ')) != NULL)
            n = strtol(p + 1, NULL, 10);
    }
    return(n);
}


char
GPFLAG(GROUPENTRY *gp)
{
    const char *p = strchr(gp->Ptr, ' ');
    char c = 'n';

    if (p) {
        if ((p = strchr(p + 1, ' ')) != NULL) {
            if ((p = strchr(p + 1, ' ')) != NULL) {
                c = p[1];
            }
        }
    }
    return(c);
}


char *
GPALIAS(GROUPENTRY *gp)
{
    const char *p = strchr(gp->Ptr, ' ');
    static char GPBuf[1024];

    GPBuf[0] = 0;

    if (p) {
        if ((p = strchr(p + 1, ' ')) != NULL) {
            if ((p = strchr(p + 1, ' ')) != NULL) {
                if (p[0] == NF_FLAG_ALIAS &&
                    p[1] != '\n' &&
                    strlen(p + 1) < sizeof(GPBuf) - 1
                ) {
                    strcpy(GPBuf, p + 1);
                }
            }
        }
    }
    return(GPBuf);
}
#else
#endif /* ! defined (NO_MMAP_ACTIVE) */

/*
**  Read the active file into memory, sort it, and set the number of
**  newsgroups read in.  Return TRUE if okay, FALSE on error.
*/
BOOL
GetGroupList()
{
    static char			*active;
    static struct stat          active_stat ;
    register char		*p;
    register char		*q;
    register GROUPENTRY		*gp;
    register int		i;

    /* If re-scanning, free previous groups. */
    if (active != NULL) {
        struct stat buf;        /* current active status */

        /*
         * We do not need to reload the active file if it has the
         * same device/inode and it the same length as the last
         * time that we reloaded it.
         *
         * We cannot assume that st_dev, st_ino and st_size are integers
         * so we use memcmp instead of == to test equality.
         */
        if (stat(ACTIVE, &buf) >= 0 &&
            memcmp(&(active_stat.st_dev), &(buf.st_dev),
                   sizeof(active_stat.st_dev)) == 0 &&
            memcmp(&(active_stat.st_ino), &(buf.st_ino),
                   sizeof(active_stat.st_ino)) == 0 &&
            memcmp(&(active_stat.st_size), &(buf.st_size),
                   sizeof(active_stat.st_size)) == 0) {
            
            /*
             * The active file that we have mapped is still the same file
             * with the same size, there is no need to reread it.
             */
            return TRUE;
        }

        /* we are going to reload the active file */
#if ! defined (NO_MMAP_ACTIVE)
	XDISPOSE(active);
#else
	DISPOSE(active);
#endif /* ! defined (NO_MMAP_ACTIVE) */
	DISPOSE(GRPentries);
    } else {
	stat (ACTIVE,&active_stat) ;
    }

    /* Get the new file. */
#if ! defined (NO_MMAP_ACTIVE)
    active = XReadInFile(ACTIVE, (struct stat *)NULL);
#else
    active = ReadInFile(ACTIVE, (struct stat *)NULL);
#endif /* ! defined (NO_MMAP_ACTIVE) */
    if (active == NULL) {
	syslog(L_ERROR, "%s cant read %s %m", ClientHost, ACTIVE);
	return FALSE;
    }

    /* Count lines. */
    for (p = active, i = 0; (p = strchr(p, '\n')) != NULL; p++, i++)
	continue;

    /* Fill in the group array. */
    GRPentries = NEW(GROUPENTRY, i);
    for (i = 0, gp = GRPentries, p = active; *p; i++, gp++, p = q + 1) {
#if ! defined (NO_MMAP_ACTIVE)
	gp->Ptr = p;
#else
	gp->Name = p;
#endif /* ! defined (NO_MMAP_ACTIVE) */
	if ((p = strchr(p, ' ')) == NULL) {
	    syslog(L_ERROR, "%s internal no_space1 \"%.20s...\"",
		ClientHost, GPNAME(gp));
	    return FALSE;
	}
#if ! defined (NO_MMAP_ACTIVE)
        p++ ;
#else
	*p++ = '\0';
#endif /* defined (NO_MMAP_ACTIVE) */

	/* Get the high mark. */
	if ((q = strchr(p, ' ')) == NULL) {
	    syslog(L_ERROR, "%s internal no_space2 \"%.20s...\"",
		ClientHost, GPNAME(gp));
	    return FALSE;
	}
#if ! defined (NO_MMAP_ACTIVE)
        q++ ;
#else
	*q++ = '\0';
	gp->High = atol(p);
#endif /* defined (NO_MMAP_ACTIVE) */

	/* Get the low mark. */
	if ((p = strchr(q, ' ')) == NULL) {
	    syslog(L_ERROR, "%s internal no_space3 \"%.20s...\"",
		ClientHost, GPNAME(gp));
	    return FALSE;
	}
#if ! defined (NO_MMAP_ACTIVE)
        p++ ;
#else
	*p++ = '\0';
	gp->Low = atol(q);
#endif /* defined (NO_MMAP_ACTIVE) */

	/* Kill the newline. */
	if ((q = strchr(p, '\n')) == NULL) {
	    syslog(L_ERROR, "%s internal newline \"%.20s...\"",
		ClientHost, GPNAME(gp));
	    return FALSE;
	}
#if defined (NO_MMAP_ACTIVE)
	*q = '\0';
	gp->Flag = *p;
	gp->Alias = gp->Flag == NF_FLAG_ALIAS ? p + 1 : NULL;
#endif /* defined (NO_MMAP_ACTIVE) */
    }

    GRPsize = i;
    GRPhash();
    return TRUE;
}


/*
**  Sorting predicate to put newsgroup names into numeric order.
*/
STATIC int
ARTcompare(p1, p2)
    CPOINTER p1;
    CPOINTER p2;
{
    ARTNUM	*i1;
    ARTNUM	*i2;

    i1 = CAST(ARTNUM*, p1);
    i2 = CAST(ARTNUM*, p2);
    return *i1 - *i2;
}


/*
**  Fill in ARTnumbers with the numbers of the articles in the current
**  group.
*/
STATIC void
GRPscandir(dir)
    char		*dir;
{
    static char		SPOOL[] = _PATH_SPOOL;
    static int		ARTarraysize;
    register DIRENTRY	*ep;
    register DIR	*dp;
    register char	*p;
    register ARTNUM	i;

    /* Go to the directory. */
    if (chdir(SPOOL) < 0) {
	syslog(L_FATAL, "%s cant cd %s %m", ClientHost, SPOOL);
	ExitWithStats(1);
    }

    if (ARTarraysize == 0) {
	ARTarraysize = 1024;
	ARTnumbers = NEW(ARTNUM, ARTarraysize);
    }

    /* The newsgroup directory might not exist; treat it as empty. */
    ARTsize = 0;
    GRPcount++;
    if (chdir(dir) < 0)
	return;
    dp = opendir(".");
    if (dp == NULL) {
	syslog(L_ERROR, "%s cant opendir %s %m", ClientHost, dir);
	return;
    }

    while ((ep = readdir(dp)) != NULL) {
	/* Get the numeric value of the filename, if it's all digits. */
	for (p = ep->d_name, i = 0; *p; p++) {
	    if (!CTYPE(isdigit, *p))
		break;
	    i = i * 10 + *p - '0';
	}
	if (*p || i == 0)
	    continue;

	if (ARTsize + 1 >= ARTarraysize) {
	    ARTarraysize += 1024;
	    RENEW(ARTnumbers, ARTNUM, ARTarraysize);
	}

	ARTnumbers[ARTsize++] = i;
    }
    (void)closedir(dp);

    ARTcache = NULL;
    qsort((POINTER)ARTnumbers, (SIZE_T)ARTsize, sizeof ARTnumbers[0],
	ARTcompare);
}


/*
**  Change to or list the specified newsgroup.  If invalid, stay in the old
**  group.
*/
FUNCTYPE
CMDgroup(ac, av)
    int			ac;
    char		*av[];
{
    static time_t	last_time;
    static char		NOSUCHGROUP[] = NNTP_NOSUCHGROUP;
    register char	*p;
    register int	i;
    time_t		now;
    char		*grplist[2];
    char		*group;
    char		buff[SPOOLNAMEBUFF];

    if (!PERMcanread) {
	Reply("%s\r\n", NOACCESS);
	return;
    }

    /* Parse arguments. */
    if (ac == 1) {
	if (GRPcount == 0) {
	    Printf("%d No group specified\r\n", NNTP_XGTITLE_BAD);
	    return;
	}
	(void)strncpy(buff, GRPlast, sizeof(buff) - 1);
	buff[sizeof(buff) - 1] = '\0';
	for (p = buff; *p; p++)
	    if (*p == '/')
		*p = '.';
	group = buff;
    }
    else
	group = av[1];
    if (GRPfind(group) == NULL) {
	Reply("%s %s\r\n", NOSUCHGROUP,group);
	return;
    }

    /* If permission is denied, pretend group doesn't exist. */
    if (PERMspecified) {
	grplist[0] = group;
	grplist[1] = NULL;
	if (!PERMmatch(PERMdefault, PERMlist, grplist)) {
	    Reply("%s\r\n", NOSUCHGROUP);
	    return;
	}
    }
    else if (!PERMdefault) {
	Reply("%s\r\n", NOSUCHGROUP);
	return;
    }

    /* Close out any existing article, report group stats. */
    ARTclose();
    ARTindex = 0;
    GRPreport();

    /* Make the group name a directory name. */
    (void)strncpy(buff, group, sizeof(buff) - 1);
    buff[sizeof(buff) - 1] = '\0';
    for (p = buff; *p; p++)
	if (*p == '.')
	    *p = '/';

    /* If we haven't been in the group recently, rescan. */
    (void)time(&now);
    if (!EQ(buff, GRPlast) || now > last_time + NNRP_RESCAN_DELAY) {
	GRPscandir(buff);
	(void)strncpy(GRPlast, buff, sizeof(GRPlast) - 1);
	GRPlast[sizeof(GRPlast) - 1] = '\0';
	last_time = now;
    }

    /* Close down any overview file. */
    OVERclose();

    /* Doing a "group" command? */
    if (caseEQ(av[0], "group")) {
	if (ARTsize == 0)
	    Reply("%d 0 0 0 %s\r\n", NNTP_GROUPOK_VAL, group);
	else
	    Reply("%d %d %ld %ld %s\r\n",
		NNTP_GROUPOK_VAL,
		ARTsize, ARTnumbers[0], ARTnumbers[ARTsize - 1], group);
    }
    else {
	/* Must be doing a "listgroup" command. */
	Reply("%d Article list follows\r\n", NNTP_GROUPOK_VAL);
	for (i = 0; i < ARTsize; i++)
	    Printf("%ld\r\n", ARTnumbers[i]);
	Printf(".\r\n");
    }
}


/*
**  Report on the number of articles read in the group, and clear the count.
*/
void
GRPreport()
{
    register char	*p;
    char		buff[SPOOLNAMEBUFF];

    if (GRPlast[0] && GRParticles != 0) {
	(void)strncpy(buff, GRPlast, sizeof(buff) - 1);
	buff[sizeof(buff) - 1] = '\0';
	for (p = buff; *p; p++)
	    if (*p == '/')
		*p = '.';
	syslog(L_NOTICE, "%s group %s %ld", ClientHost, buff, GRParticles);
	GRParticles = 0;
    }
}


/*
**  Used by ANU-News clients.
*/
FUNCTYPE
CMDxgtitle(ac, av)
    int			ac;
    char		*av[];
{
    register QIOSTATE	*qp;
    register char	*line;
    register char	*p;
    register char	*q;
    char		save;

    /* Parse the arguments. */
    if (ac == 1) {
	if (GRPcount == 0) {
	    Printf("%d No group specified\r\n", NNTP_XGTITLE_BAD);
	    return;
	}
	p = GRPlast;
    }
    else
	p = av[1];

    /* Open the file, get ready to scan. */
    if ((qp = QIOopen(NEWSGROUPS, QIO_BUFFER)) == NULL) {
	syslog(L_ERROR, "%s cant open %s %m", ClientHost, NEWSGROUPS);
	Printf("%d Can't open %s\r\n", NNTP_XGTITLE_BAD, NEWSGROUPS);
	return;
    }
    Printf("%d list follows\r\n", NNTP_XGTITLE_OK);

    /* Print all lines with matching newsgroup name. */
    while ((line = QIOread(qp)) != NULL) {
	for (q = line; *q && !ISWHITE(*q); q++)
	    continue;
	save = *q;
	*q = '\0';
	if (wildmat(line, p)) {
	    *q = save;
	    Printf("%s\r\n", line);
	}
    }

    /* Done. */
    QIOclose(qp);
    Printf(".\r\n");
}
