/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Tambet Ingo <tambet@ximian.com>.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "xst.h"
#include "ui.h"
#include "transfer.h"

/* Common helpers */

static gint
clist_add_row_unique (GtkCList *list, const gchar *text)
{
	gchar *item[2];
	gchar *buf;
	gint   row;

	for (row = 0; gtk_clist_get_text (list, row, 0, &buf); row++) {
		if (buf && (strcmp (buf, text)) == 0)
			return -1;
	}
	
	item[0] = (gchar *)text;
	item[1] = NULL;	
	row = gtk_clist_append (list, item);

	return row;
}

/* Colordepth widget */
/* GtkCList */

#define UI_WIDGET_DEPTH "colordepth_list"

typedef struct {
        gchar *text;
        gint id;
} XstDisplayListItem;

static XstDisplayListItem colordepth_items[] = {
	{ N_("256 colors"),          8 },
	{ N_("32 thousand colors"), 15 },
	{ N_("65 thousand colors"), 16 },
	{ N_("16 million colors"),  24 },
	{ NULL, -1},
};

static const gchar *
depth_to_label (const gchar *depth)
{
	gint val, i;
	
	g_return_val_if_fail (depth != NULL, NULL);
	
	val = atoi (depth);
	for (i = 0; colordepth_items[i].id != -1; i++) {
		if (colordepth_items[i].id == val)
			return colordepth_items[i].text;
	}

	g_warning ("depth_to_label: Unknown depth %s.", depth);
	return NULL;
}

static gchar *
depth_from_label (const gchar *label)
{
	gint i;
	
	g_return_val_if_fail (label != NULL, NULL);
	
	for (i = 0; colordepth_items[i].id != -1; i++) {
		if ((strcmp (colordepth_items[i].text, label)) == 0)
			return g_strdup_printf ("%d", colordepth_items[i].id);
	}

	g_warning ("depth_from_label: Unknown label %s.", label);
	return NULL;
}

static void
depth_selection_changed (GtkCList *list, gint row, gint col, GdkEventButton *event, gpointer data)
{
	gchar   *depth;
	gchar   *res;
	XstTool *tool = XST_TOOL (data);

	xst_dialog_modify (tool->main_dialog);

	res = ui_res_selected (tool);
	depth = gtk_clist_get_row_data (GTK_CLIST (list), row);
	if (depth == NULL)
		g_warning ("depth_selection_changed: Selected row has no data.");
	else {
		populate_resolutions (tool, depth);
		if (res)
			ui_res_select (tool, res);
	}
}

static gint
depth_cmp (GtkCList *list, gconstpointer ptr1, gconstpointer ptr2)
{
	gchar       *text1, *text2;
	gint         n1, n2;
	GtkCListRow *row1 = (GtkCListRow *) ptr1;
	GtkCListRow *row2 = (GtkCListRow *) ptr2;

	text1 = (gchar *)row1->data;
	text2 = (gchar *)row2->data;

	if (!text2)
		return (text1 != NULL);

	if (!text1)
		return -1;

	n1 = atoi (text1);
	n2 = atoi (text2);

	if (n1 == n2)
		return 0;

	if (n1 > n2)
		return 1;

	return -1;
}

void
ui_depth_add (XstTool *tool, const gchar *depth)
{
	GtkWidget       *list;
	XstDialog       *xd;
	const gchar     *label;
	gint             row;
	guint            id;
	static gboolean  first = TRUE;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));
	g_return_if_fail (depth != NULL);

	label = depth_to_label (depth);
	if (label == NULL)
		return;

	xd = tool->main_dialog;
	list = xst_dialog_get_widget (xd, UI_WIDGET_DEPTH);
	if (list == NULL)
		return;

	if (first) {
		first = FALSE;
		gtk_clist_set_compare_func (GTK_CLIST (list), depth_cmp);
		id = gtk_signal_connect (GTK_OBJECT (list), "select-row",
					 GTK_SIGNAL_FUNC (depth_selection_changed),
					 (gpointer) tool);
		gtk_object_set_data (GTK_OBJECT (list), "signal-select-row", GUINT_TO_POINTER (id));
	} else
		id = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (list), "signal-select-row"));

	gtk_signal_handler_block (GTK_OBJECT (list), id);
	
	row = clist_add_row_unique (GTK_CLIST (list), label);
	if (row >= 0) {
		gtk_clist_set_row_data (GTK_CLIST (list), row, (gpointer)depth);
		gtk_clist_sort (GTK_CLIST (list));
	}

	gtk_signal_handler_unblock (GTK_OBJECT (list), id);
}

void
ui_depth_select (XstTool *tool, const gchar *depth)
{
	GtkWidget   *list;
	XstDialog   *xd;
	gchar       *buf;
	gint         row;
	guint        id;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));
	g_return_if_fail (depth != NULL);

	xd = tool->main_dialog;
	list = xst_dialog_get_widget (xd, UI_WIDGET_DEPTH);
	if (list == NULL)
		return;

	id = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (list), "signal-select-row"));
	
	for (row = 0; gtk_clist_get_text (GTK_CLIST (list), row, 0, &buf); row++) {
		buf = gtk_clist_get_row_data (GTK_CLIST (list), row);
		if (buf && (strcmp (buf, depth)) == 0) {
			gtk_signal_handler_block (GTK_OBJECT (list), id);
			gtk_clist_select_row (GTK_CLIST (list), row, 0);
			gtk_signal_handler_unblock (GTK_OBJECT (list), id);
			return;
		}
	}

	g_warning ("ui_depth_select: Can't select depth %s.", depth);
}

gchar *
ui_depth_selected (XstTool *tool)
{
	GtkWidget   *list;
	XstDialog   *xd;
	gchar       *buf;
	gint         row;
	
	g_return_val_if_fail (tool != NULL, NULL);
	g_return_val_if_fail (XST_IS_TOOL (tool), NULL);

	xd = tool->main_dialog;
	list = xst_dialog_get_widget (xd, UI_WIDGET_DEPTH);
	if (list == NULL)
		return NULL;

	if (GTK_CLIST (list)->selection) {
		row = GPOINTER_TO_INT (GTK_CLIST (list)->selection->data);
		buf = gtk_clist_get_row_data (GTK_CLIST (list), row);
		if (buf == NULL)
			g_warning ("ui_depth_selected: Selected row has no data.");
		return buf;
	}

	g_warning ("ui_depth_selected: no selected rows.");
	return NULL;
}

void
ui_depth_freeze (XstTool *tool, gboolean freeze)
{
	GtkWidget *list;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));

	list = xst_dialog_get_widget (tool->main_dialog, UI_WIDGET_DEPTH);
	if (list == NULL)
		return;

	if (freeze)
		gtk_clist_freeze (GTK_CLIST (list));
	else
		gtk_clist_thaw (GTK_CLIST (list));
}

void
ui_depth_clear (XstTool *tool)
{
	GtkWidget *list;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));

	list = xst_dialog_get_widget (tool->main_dialog, UI_WIDGET_DEPTH);
	if (list)
		gtk_clist_clear (GTK_CLIST (list));
}

/* Resolution widget */
/* GtkCList */

#define UI_WIDGET_RESOLUTION "resolution_list"

static void
res_selection_changed (GtkCList *list, gint row, gint col, GdkEventButton *event, gpointer data)
{
	XstDialog *xd = XST_DIALOG (data);
	xst_dialog_modify (xd);
}

static gint
res_cmp (GtkCList *list, gconstpointer ptr1, gconstpointer ptr2)
{
	gchar       *text1, *text2;
	gint         x1, x2;
	gint         y1, y2;
	GtkCListRow *row1 = (GtkCListRow *) ptr1;
	GtkCListRow *row2 = (GtkCListRow *) ptr2;

	text1 = (gchar *)row1->data;
	text2 = (gchar *)row2->data;

	if (!text2)
		return (text1 != NULL);

	if (!text1)
		return -1;

	sscanf (text1, "%dx%d", &x1, &y1);
	sscanf (text2, "%dx%d", &x2, &y2);
	
	if (x1 > x2)
		return 1;
	if (x1 < x2)
		return -1;

	if (y1 > y2)
		return 1;
	if (y1 < y2)
		return -1;

	return 0;
}

void
ui_res_add (XstTool *tool, const gchar *res)
{
	XstDialog       *xd;
	GtkWidget       *list;
	gint             row;
	guint            id;
	static gboolean  first = TRUE;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));
	g_return_if_fail (res != NULL);

	xd = tool->main_dialog;
	list = xst_dialog_get_widget (xd, UI_WIDGET_RESOLUTION);
	if (list == NULL)
		return;

	if (first) {
		first = FALSE;
		gtk_clist_set_compare_func (GTK_CLIST (list), res_cmp);
		id = gtk_signal_connect (GTK_OBJECT (list), "select-row",
					 GTK_SIGNAL_FUNC (res_selection_changed),
					 (gpointer) xd);
		gtk_object_set_data (GTK_OBJECT (list), "signal-select-row", GUINT_TO_POINTER (id));
	} else
		id = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (list), "signal-select-row"));

	gtk_signal_handler_block (GTK_OBJECT (list), id);

	row = clist_add_row_unique (GTK_CLIST (list), res);
	if (row >= 0) {
		gtk_clist_set_row_data (GTK_CLIST (list), row, (gpointer)res);
		gtk_clist_sort (GTK_CLIST (list));
	}

	gtk_signal_handler_unblock (GTK_OBJECT (list), id);
}

void
ui_res_select (XstTool *tool, const gchar *res)
{
	XstDialog *xd;
	GtkWidget *list;
	gint       row;
	guint      id;
	gchar     *buf;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));
	g_return_if_fail (res != NULL);

	xd = tool->main_dialog;
	list = xst_dialog_get_widget (xd, UI_WIDGET_RESOLUTION);
	if (list == NULL)
		return;

	id = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (list), "signal-select-row"));

	for (row = 0; gtk_clist_get_text (GTK_CLIST (list), row, 0, &buf); row++) {
		buf = gtk_clist_get_row_data (GTK_CLIST (list), row);
		if (buf && (strcmp (buf, res)) == 0) {
			gtk_signal_handler_block (GTK_OBJECT (list), id);
			gtk_clist_select_row (GTK_CLIST (list), row, 0);
			gtk_signal_handler_unblock (GTK_OBJECT (list), id);
			return;
		}
	}

	g_warning ("ui_res_select: Can't select resolution %s.", res);
}

gchar *
ui_res_selected (XstTool *tool)
{
	XstDialog *xd;
	GtkWidget *list;
	gchar     *buf;
	gint       row;

	g_return_val_if_fail (tool != NULL, NULL);
	g_return_val_if_fail (XST_IS_TOOL (tool), NULL);

	xd = tool->main_dialog;
	list = xst_dialog_get_widget (xd, UI_WIDGET_RESOLUTION);
	if (list == NULL)
		return NULL;

	if (GTK_CLIST (list)->selection) {
		row = GPOINTER_TO_INT (GTK_CLIST (list)->selection->data);
		buf = gtk_clist_get_row_data (GTK_CLIST (list), row);
		if (buf == NULL)
			g_warning ("ui_res_selected: Selected row has no data.");
		return buf;
	}

	g_warning ("ui_res_selected: no selected rows.");
	return NULL;
}

void
ui_res_freeze (XstTool *tool, gboolean freeze)
{
	GtkWidget *list;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));

	list = xst_dialog_get_widget (tool->main_dialog, UI_WIDGET_RESOLUTION);
	if (list == NULL)
		return;

	if (freeze)
		gtk_clist_freeze (GTK_CLIST (list));
	else
		gtk_clist_thaw (GTK_CLIST (list));
}

void
ui_res_clear (XstTool *tool)
{
	GtkWidget *list;

	g_return_if_fail (tool != NULL);
	g_return_if_fail (XST_IS_TOOL (tool));

	list = xst_dialog_get_widget (tool->main_dialog, UI_WIDGET_RESOLUTION);
	if (list)
		gtk_clist_clear (GTK_CLIST (list));
}
