/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Copyright (C) 2000-2004 Novell, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <camel/camel-provider.h>
#include <camel/camel-session.h>
#include <camel/camel-url.h>
#include "camel-exchange-store.h"
#include "camel-exchange-transport.h"

static guint exchange_url_hash (gconstpointer key);
static gint exchange_url_equal (gconstpointer a, gconstpointer b);

CamelProviderConfEntry exchange_conf_entries[] = {
	/* override the labels/defaults of the standard settings */
	{ CAMEL_PROVIDER_CONF_LABEL, "hostname", NULL,
	  /* i18n: the '_' should appear before the same letter it
	     does in the evolution:mail-config.glade "_Host"
	     translation (or not at all) */
	  N_("Exc_hange Server:") },
	{ CAMEL_PROVIDER_CONF_LABEL, "username", NULL,
	  /* i18n: the '_' should appear before the same letter it
	     does in the evolution:mail-config.glade "User_name"
	     translation (or not at all) */
	  N_("Windows User_name:") },

	/* extra Exchange configuration settings */
	{ CAMEL_PROVIDER_CONF_SECTION_START, "activedirectory", NULL,
	  /* i18n: GAL is an Outlookism, AD is a Windowsism */
	  N_("Global Address List / Active Directory") },
	{ CAMEL_PROVIDER_CONF_ENTRY, "ad_server", NULL,
	  /* i18n: "Global Catalog" is a Windowsism, but it's a
	     technical term and may not have translations? */
	  N_("Global Catalog server name:") },
	{ CAMEL_PROVIDER_CONF_CHECKSPIN, "ad_limit", NULL,
	  N_("Limit number of GAL responses: %s"), "y:1:500:10000" },
	{ CAMEL_PROVIDER_CONF_SECTION_END },
	{ CAMEL_PROVIDER_CONF_SECTION_START, "exchange", NULL,
	  N_("Exchange Server") },
	{ CAMEL_PROVIDER_CONF_ENTRY, "mailbox", NULL,
	  /* i18n: "Mailbox" is an Outlookism. FIXME */
	  N_("Mailbox name") },
	{ CAMEL_PROVIDER_CONF_ENTRY, "owa_path", NULL,
	  /* i18n: "OWA" == "Outlook Web Access". Might not be translated? */
	  N_("OWA path"), "/exchange" },
	{ CAMEL_PROVIDER_CONF_ENTRY, "pf_server", NULL,
	  /* i18n: "Public Folder" is an Outlookism */
	  N_("Public Folder server") },
	{ CAMEL_PROVIDER_CONF_SECTION_END },
	{ CAMEL_PROVIDER_CONF_CHECKBOX, "filter", NULL,
	  /* i18n: copy from evolution:camel-imap-provider.c */
	  N_("Apply filters to new messages in Inbox on this server"), "0" },
	{ CAMEL_PROVIDER_CONF_END }
};

static CamelProvider exchange_provider = {
	"exchange",
	N_("Microsoft Exchange"),

	N_("For handling mail (and other data) on Microsoft Exchange servers"),

	"mail",

	CAMEL_PROVIDER_IS_REMOTE | CAMEL_PROVIDER_IS_SOURCE |
	CAMEL_PROVIDER_IS_STORAGE | CAMEL_PROVIDER_IS_EXTERNAL |
	CAMEL_PROVIDER_SUPPORTS_SSL,

	CAMEL_URL_NEED_USER | CAMEL_URL_NEED_HOST |
	CAMEL_URL_ALLOW_AUTH,

	exchange_conf_entries,

	/* ... */
};

CamelServiceAuthType camel_exchange_ntlm_authtype = {
	/* i18n: "Secure Password Authentication" is an Outlookism */
	N_("Secure Password"),

	/* i18n: "NTLM" probably doesn't translate */
	N_("This option will connect to the Exchange server using "
	   "secure password (NTLM) authentication."),

	"",
	TRUE
};

CamelServiceAuthType camel_exchange_password_authtype = {
	N_("Plaintext Password"),

	N_("This option will connect to the Exchange server using "
	   "standard plaintext password authentication."),

	"Basic",
	TRUE
};

static int
exchange_auto_detect_cb (CamelURL *url, GHashTable **auto_detected,
			 CamelException *ex)
{
	*auto_detected = g_hash_table_new (g_str_hash, g_str_equal);

	g_hash_table_insert (*auto_detected, g_strdup ("mailbox"),
			     g_strdup (url->user));
	g_hash_table_insert (*auto_detected, g_strdup ("pf_server"),
			     g_strdup (url->host));

	return 0;
}

void
camel_provider_module_init (CamelSession *session)
{
	exchange_provider.object_types[CAMEL_PROVIDER_STORE] = camel_exchange_store_get_type ();
	exchange_provider.object_types[CAMEL_PROVIDER_TRANSPORT] = camel_exchange_transport_get_type ();
	exchange_provider.authtypes = g_list_prepend (g_list_prepend (NULL, &camel_exchange_password_authtype), &camel_exchange_ntlm_authtype);
	exchange_provider.url_hash = exchange_url_hash;
	exchange_provider.url_equal = exchange_url_equal;
	exchange_provider.auto_detect = exchange_auto_detect_cb;

	bindtextdomain (GETTEXT_PACKAGE, CONNECTOR_LOCALEDIR);
	exchange_provider.translation_domain = GETTEXT_PACKAGE;

	camel_session_register_provider (session, &exchange_provider);
}

static const char *
exchange_username (const char *user)
{
	const char *p;

	if (user) {
		p = strpbrk (user, "\\/");
		if (p)
			return p + 1;
	}

	return user;
}

static guint
exchange_url_hash (gconstpointer key)
{
	const CamelURL *u = (CamelURL *)key;
	guint hash = 0;

	if (u->user)
		hash ^= g_str_hash (exchange_username (u->user));
	if (u->host)
		hash ^= g_str_hash (u->host);

	return hash;
}

static gboolean
check_equal (const char *s1, const char *s2)
{
	if (!s1)
		return s2 == NULL;
	else if (!s2)
		return FALSE;
	else
		return strcmp (s1, s2) == 0;
}

static gint
exchange_url_equal (gconstpointer a, gconstpointer b)
{
	const CamelURL *u1 = a, *u2 = b;

	return  check_equal (u1->protocol, u2->protocol) &&
		check_equal (exchange_username (u1->user),
			     exchange_username (u2->user)) &&
		check_equal (u1->host, u2->host);
}
