/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include <ctype.h>
#ifdef USE_GNOME
#include <zvt/zvtterm.h>
#endif
#include <time.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "cfgfiles.h"
#include "gtkutil.h"
#include "plugin.h"

extern GSList *sess_list;
extern GdkColor colors[];
extern GdkFont *font_normal;
extern GdkFont *font_bold;
extern GdkFont *dialog_font_normal;
extern GdkFont *dialog_font_bold;
extern GtkStyle *normaltab_style;
extern GtkStyle *redtab_style;
extern GtkStyle *channelwin_style;
extern struct xchatprefs prefs;
extern struct session *current_tab;
extern void check_special_chars (char *);
extern int buf_get_line (char *, char **, int *, int len);
extern void show_and_unfocus (GtkWidget *);
#ifdef USE_PERL
extern int perl_print(char *cmd, struct session *sess, char *b,char *c,char *d,char *e);
#endif

extern char **environ;

GtkWidget *pevent_dialog = NULL, *pevent_dialog_twid,
         *pevent_dialog_entry, *pevent_dialog_sound_entry,
	   *pevent_dialog_list, *pevent_dialog_hlist;

int pevt_build_string (char *input, char **output, int *max_arg);

long 
timecat (char *buf)
{
   time_t timval = time (0);
   char *tim = ctime (&timval) + 10;
   tim[0] = '[';
   tim[9] = ']';
   tim[10] = ' ';
   tim[11] = 0;
   strcat (buf, tim);
   return strlen (buf);
}

unsigned char *
strip_color (unsigned char *text)
{
   int comma, done;
   int j = 0, i = 0, len = strlen (text);
   char *buf = malloc (len + 2);

   while (i < len)
   {
      switch (text[i])
      {
      case 2:
         break;
      case 3:
         comma = FALSE;
         done = FALSE;
         while (i < len && !done)
         {
            i++;
            if (!isdigit (text[i]))
            {
               switch (text[i])
               {
               case ',':
                  if (comma)
                     done = TRUE;
                  else
                     comma = TRUE;
                  break;
               default:
                  done = TRUE;
               }
            }
         }
         i--;
         break;
      default:
         buf[j] = text[i];
         j++;
      }
      i++;
   }
   buf[j] = 0;

   return buf;
}


#ifdef USE_GNOME

/*                0  1  2  3  4  5  6  7   8   9   10  11  12  13  14  15 */
int colconv[] = { 7, 0, 4, 2, 1, 3, 5, 11, 13, 12, 6,  16, 14, 15, 10, 7 };

void 
PrintTextRawZvt (GtkWidget * textwidget, unsigned char *text)
{
   int dotime = FALSE;
   char num[8];
   int reverse = 0, under = 0, bold = 0,
       comma, k, i = 0, j = 0, len = strlen (text);
   unsigned char *newtext = malloc (len + 1024);

   if (prefs.timestamp)
   {
      newtext[0] = 0;
      j = timecat (newtext);
   }
   while (i < len)
   {
      if (dotime && text[i] != 0)
      {
         dotime = FALSE;
         newtext[j] = 0;
         j = timecat (newtext);
      }
      switch (text[i])
      {
      case 3:
         i++;
         if (!isdigit (text[i]))
         {
            newtext[j] = 27;
            j++;
            newtext[j] = '[';
            j++;
            newtext[j] = '0';
            j++;
            newtext[j] = 'm';
            j++;
            i--;
            goto jump2;
         }
         k = 0;
         comma = FALSE;
         while (i < len)
         {
            if (text[i] >= '0' && text[i] <= '9' && k < 2)
            {
               num[k] = text[i];
               k++;
            } else
            {
               int col, mirc;
               num[k] = 0;
               newtext[j] = 27;
               j++;
               newtext[j] = '[';
               j++;
               if (k == 0)
               {
                  newtext[j] = '0';
                  j++;
                  newtext[j] = 'm';
                  j++;
               } else
               {
                  if (comma)
                     col = 40;
                  else
                     col = 30;
                  mirc = atoi (num);
                  mirc = colconv[mirc];
                  if (mirc > 9)
                  {
			   mirc += 50;
                     sprintf ((char *) &newtext[j], "%dm", mirc + col);
                  } else
                  {
                     sprintf ((char *) &newtext[j], "%dm", mirc + col);
                  }
                  j = strlen (newtext);
               }
               switch (text[i])
               {
               case ',':
                  comma = TRUE;
                  break;
               default:
                  goto jump;
               }
               k = 0;
            }
            i++;
         }
         break;
      case '\026':             /* REVERSE */
         if (reverse)
         {
            reverse = FALSE;
            strcpy (&newtext[j], "\033[27m");
         } else
         {
            reverse = TRUE;
            strcpy (&newtext[j], "\033[7m");
         }
         j = strlen (newtext);
         break;
      case '\037':             /* underline */
         if (under)
         {
            under = FALSE;
            strcpy (&newtext[j], "\033[24m");
         } else
         {
            under = TRUE;
            strcpy (&newtext[j], "\033[4m");
         }
         j = strlen (newtext);
         break;
      case '\002':             /* bold */
         if (bold)
         {
            bold = FALSE;
            strcpy (&newtext[j], "\033[22m");
         } else
         {
            bold = TRUE;
            strcpy (&newtext[j], "\033[1m");
         }
         j = strlen (newtext);
         break;
      case '\007':
         if (!prefs.filterbeep)
         {
            newtext[j] = text[i];
            j++;
         }
         break;
      case '\017':             /* reset all */
         strcpy (&newtext[j], "\033[0m");
         j = strlen (newtext);
         reverse = FALSE;
         bold = FALSE;
         under = FALSE;
         break;
      case '\n':
         newtext[j] = '\r';
         j++;
         if (prefs.timestamp)
            dotime = TRUE;
      default:
         newtext[j] = text[i];
         j++;
      }
    jump2:
      i++;
    jump:
   }
   newtext[j] = 0;
   zvt_term_feed ((ZvtTerm *) textwidget, newtext, j);
   free (newtext);
   /*vt_draw_cursor (((ZvtTerm*)textwidget)->vx, 0);*/
}

#endif

void 
cut_down_text (GtkWidget * textwidget)
{
   if (prefs.bufsize > 0)
   {
      long n = gtk_text_get_length ((GtkText *) textwidget);
      if (n > prefs.bufsize)
      {
         gtk_text_set_point ((GtkText *) textwidget, n - prefs.bufsize);
         gtk_text_backward_delete ((GtkText *) textwidget, n - prefs.bufsize);
         gtk_text_set_point ((GtkText *) textwidget, gtk_text_get_length ((GtkText *) textwidget));
      }
   }
}

void 
PrintTextRaw (GtkWidget * textwidget, unsigned char *text,
              GdkFont * fontnorm,
              GdkFont * fontbold)
{
   int dotime = FALSE;
   int esc = FALSE;
   int comma = FALSE;
   unsigned char buf[4096];
   unsigned char num[4];
   int bcol = 1, col = 0, j = 0, i = 0, k = 0;
   int scroll = FALSE;
   GtkAdjustment *adj;
   GdkFont *font = fontnorm;
   int bold = FALSE;

   adj = (GTK_TEXT (textwidget))->vadj;
   if (adj->value == adj->upper - adj->lower - adj->page_size)
      scroll = TRUE;

   if (prefs.timestamp)
   {
      buf[0] = 0;
      j = timecat (buf);
   }
   gtk_text_freeze (GTK_TEXT (textwidget));
   while (1)
   {
      if (dotime && text[i] != 0)
      {
         dotime = FALSE;
         buf[j] = 0;
         j = timecat (buf);
      }
      if (esc)
      {
         if (!isdigit (text[i]))
         {
            bcol = 1;
            col = 0;
            goto norm;
         } else
         {
            while (text[i] >= '0' && text[i] <= '9' && k < 2)
            {
               num[k] = text[i];
               k++;
               i++;
            }
            num[k] = 0;
            k = 0;
            switch (text[i])
            {
            case ',':
               comma = TRUE;
               col = atoi (num);
               if (col < 0 || col > 15)
                  col = 1;
               break;
            default:
               if (comma)
               {
                  comma = FALSE;
                  bcol = atoi (num);
                  if (bcol < 0 || bcol > 15)
                     bcol = 1;
               } else
               {
                  col = atoi (num);
                  if (col < 0 || col > 15)
                     col = 0;
               }
               if (bcol == 1 && col == 1)
                  col = 0;
               if (bcol == 0 && col == 0)
                  col = 1;
               goto norm;
            }
         }
      } else
      {
       norm:
         esc = FALSE;
         switch (text[i])
         {
         case 0:
            goto jump;
         case 3:               /* CTRL-C */
            buf[j] = 0;
            if (j > 0)
            {
               gtk_text_insert (GTK_TEXT (textwidget), font, &colors[col], &colors[bcol], buf, j);
               j = 0;
            }
            esc = TRUE;
            k = 0;
            break;
         case '\007':          /* beep */
            if (!prefs.filterbeep)
            {
               buf[j] = 7;
               j++;
               buf[j] = 0;
               gtk_text_insert (GTK_TEXT (textwidget), font, &colors[col], &colors[bcol], buf, j);
               j = 0;
               gdk_beep ();
            }
            break;
         case '\026':          /* REVERSE */
         case '\037':          /* UNDERLINE */
         case '\002':          /* BOLD */
            buf[j] = 0;
            if (j > 0)
            {
               gtk_text_insert (GTK_TEXT (textwidget), font, &colors[col], &colors[bcol], buf, j);
               j = 0;
            }
            if (bold)
            {
               bold = FALSE;
               font = fontnorm;
            } else
            {
               bold = TRUE;
               font = fontbold;
            }
            break;
         case '\017':          /* ALL OFF */
            buf[j] = 0;
            if (j > 0)
            {
               gtk_text_insert (GTK_TEXT (textwidget), font, &colors[col], &colors[bcol], buf, j);
               bcol = 1;
               col = 0;
               bold = FALSE;
               j = 0;
            }
            font = fontnorm;
            break;
         default:
            esc = FALSE;
            comma = FALSE;
            buf[j] = text[i];
            j++;
            if (j == 4095)
               j = 4094;
            if (text[i] == '\n' && prefs.timestamp)
               dotime = TRUE;
         }
      }
      i++;
   }
 jump:
   if (j)
   {
      gtk_text_insert (GTK_TEXT (textwidget), font, &colors[col], &colors[bcol], buf, j);
      cut_down_text (textwidget);
      gtk_text_thaw (GTK_TEXT (textwidget));
      if (scroll)
         gtk_adjustment_set_value (adj, adj->upper - adj->lower - adj->page_size);
   } else
   {
      cut_down_text (textwidget);
      gtk_text_thaw (GTK_TEXT (textwidget));
   }
}

void 
PrintText (struct session *sess, unsigned char *text)
{
   if (!sess)
      sess = (struct session *) sess_list->data;

   if (sess->logfd != -1 && prefs.logging)
   {
      time_t timval = time (0);
      char *tim = ctime (&timval) + 11;
      char *temp = strip_color (text);
      tim[9] = 0;
      if (prefs.timestamp)
         write (sess->logfd, tim, strlen (tim));
      write (sess->logfd, temp, strlen (temp));
      free (temp);
   }
#ifdef USE_GNOME
   if (sess->zvt)
      PrintTextRawZvt (sess->textgad, text);
   else
   {
      if (sess->is_dialog)
         PrintTextRaw (sess->textgad, text,
                       dialog_font_normal, dialog_font_bold);
      else
         PrintTextRaw (sess->textgad, text,
              font_normal, font_bold);
   }
#else
   if (sess->is_dialog)
      PrintTextRaw (sess->textgad, text,
                    dialog_font_normal, dialog_font_bold);
   else
      PrintTextRaw (sess->textgad, text,
              font_normal, font_bold);
#endif

#ifdef USE_PANEL
   if (!sess->new_data && !sess->nick_said && sess != current_tab)
   {
      if (sess->panel_button)
         gtk_widget_set_style (GTK_BIN (sess->panel_button)->child, redtab_style);
   }
#endif

   if (!sess->new_data && sess != current_tab &&
     sess->is_tab && !sess->nick_said)
   {
      sess->new_data = TRUE;
      gtk_widget_set_style (sess->changad, redtab_style);
   }
}

void 
end_logging (int fd)
{
   char obuf[512];
   time_t currenttime;

   currenttime = time (NULL);
   write (fd, obuf, snprintf (obuf, 510, "**** ENDING LOGGING AT %s\n", ctime (&currenttime)));
   close (fd);
}

int 
open_log_file (char *servname, char *channame)
{
   char buf[512];
   int fd;
   time_t currenttime;
   struct stat st;

   snprintf (buf, 510, "%s/xchatlogs", get_xdir ());
   if (stat (buf, &st) < 0)
      mkdir (buf, S_IRUSR | S_IWUSR | S_IXUSR);

   if (prefs.no_server_logs)
      snprintf (buf, 510, "%s/xchatlogs/%s.xchatlog", get_xdir (), channame);
   else
      snprintf (buf, 510, "%s/xchatlogs/%s,%s.xchatlog", get_xdir (), servname, channame);
   fd = open (buf, O_CREAT | O_APPEND | O_WRONLY, 0x180);
   if (fd < 0)
      return -1;
   currenttime = time (NULL);
   write (fd, buf, snprintf (buf, 510, "**** BEGIN LOGGING AT %s\n", ctime (&currenttime)));

   return fd;
}

void 
setup_logging (struct session *sess)
{
   if (sess->logfd != -1)
      end_logging (sess->logfd);
   sess->logfd = open_log_file (sess->server->servername, sess->channel);
}
/* Print Events stuff here --AGL */

/* Consider the following a NOTES file:

   The main upshot of this is:
   * Plugins and Perl scripts (when I get round to signaling perl.c) can intercept text events and do what they like
   * The default text engine can be config'ed

   By default it should appear *exactly* the same (I'm working hard not to change the default style) but if you go into Settings->Edit Event Texts you can change the text's. The format is thus:

   The normal %Cx (color) and %B (bold) etc work

   $x is replaced with the data in var x (e.g. $1 is often the nick)

   $axxx is replace with a single byte of value xxx (in base 10)

   AGL (990507)
 */

/* These lists are thus:
   pntevts_text[] are the strings the user sees (WITH %x etc)
   pntevts[] are the data strings with \000 etc
   evtnames[] are the names of the events
 */

/* To add a new event:

   Think up a name (like JOIN)
   Make up a pevt_name_help struct (with a NULL at the end)
   add a struct to the end of te with,
     {"Name", help_struct, "Default text", num_of_args, NULL}
   Open up plugin.h
   add one to NUM_XP
   add a #define XP_TE_NAME with a value of +1 on the last XP_
*/

/* As of 990528 this code is in BETA --AGL */

/* Internals:

   On startup ~/.xchat/printevents.conf is loaded if it doesn't exist the
   defaults are loaded. Any missing events are filled from defaults.
   Each event is parsed by pevt_build_string and a binary output is produced
   which looks like:

   (byte) value: 0 = {
   (int) numbers of bytes
   (char []) that number of byte to be memcpy'ed into the buffer
   }
   1 =
   (byte) number of varable to insert
   2 = end of buffer

   Each XP_TE_* signal is hard coded to call textsignal_handler which calls
   display_event which decodes the data

   This means that this system *should be faster* than snprintf because
   it always 'knows' that format of the string (basically is preparses much
   of the work)

   --AGL
 */

char **pntevts_text;
char **pntevts;

char *pevt_join_help[] =
{
   "The nick of the joining person",
   "The channel being joined",
   "The host of the person",
   NULL
};

char *pevt_chanaction_help[] =
{
 "Nickname (which may contain color)",
   "The action",
   NULL
};

char *pevt_chanmsg_help[] =
{
 "Nickname (which may contain color)",
   "The text",
   NULL
};

char *pevt_privmsg_help[] =
{
   "Nickname",
   "The message",
   NULL
};

char *pevt_changenick_help[] =
{
   "Old nickname",
   "New nickname",
   NULL
};

char *pevt_newtopic_help[] =
{
   "Nick of person who changed the topic",
   "Topic",
   "Channel",
   NULL
};

char *pevt_topic_help[] =
{
   "Channel",
   "Topic",
   NULL
};

char *pevt_kick_help[] =
{
   "The nickname of the kicker",
   "The person being kicked",
   "The channel",
   "The reason",
   NULL
};

char *pevt_part_help[] =
{
   "The nick of the person leaving",
   "The host of the person",
   "The channel",
   NULL
};

char *pevt_chandate_help[] =
{
   "The channel",
   "The time",
   NULL
};

char *pevt_topicdate_help[] =
{
   "The channel",
   "The creator",
   "The time",
   NULL
};

char *pevt_quit_help[] =
{
   "Nick",
   "Reason",
   NULL
};

char *pevt_pingrep_help[] =
{
   "Who it's from",
 "The time in x.x format (see below)",
   NULL
};

char *pevt_notice_help[] =
{
   "Who it's from",
   "The message",
   NULL
};

char *pevt_ujoin_help[] =
{
   "The nick of the joining person",
   "The channel being joined",
   "The host of the person",
   NULL
};

char *pevt_uchanmsg_help[] =
{
 "Nickname (which may contain color)",
   "The text",
   NULL
};

char *pevt_dprivmsg_help[] =
{
   "Nickname",
   "The message",
   NULL
};

char *pevt_uchangenick_help[] =
{
   "Old nickname",
   "New nickname",
   NULL
};

char *pevt_ukick_help[] =
{
   "The nickname of the kicker",
   "The person being kicked",
   "The channel",
   "The reason",
   NULL
};

char *pevt_upart_help[] =
{
   "The nick of the person leaving",
   "The host of the person",
   "The channel",
   NULL
};

char *pevt_ctcpsnd_help[] =
{
   "The sound",
   "The nick of the person",
   NULL
};

char *pevt_ctcpgen_help[] =
{
   "The CTCP event",
   "The nick of the person",
   NULL
};

char *pevt_ctcpgenc_help[] =
{
   "The CTCP event",
   "The nick of the person",
   "The Channel it's going to",
   NULL
};

char *pevt_chansetkey_help[] =
{
   "The nick of the person who set the key",
   "The key",
   NULL
};

char *pevt_chansetlimit_help[] =
{
   "The nick of the person who set the limit",
   "The limit",
   NULL
};

char *pevt_chanop_help[] =
{
   "The nick of the person who has been op'ed",
   "The nick of the person of did the op'ing",
   NULL
};

char *pevt_chanvoice_help[] =
{
   "The nick of the person who has been voice'ed",
   "The nick of the person of did the voice'ing",
   NULL
};

char *pevt_chanban_help[] =
{
   "The nick of the person who did the banning",
   "The ban mask",
   NULL
};

char *pevt_chanrmkey_help[] =
{
   "The nick who removed the key",
   NULL
};

char *pevt_chanrmlimit_help[] =
{
   "The nick who removed the limit",
   NULL
};

char *pevt_chandeop_help[] =
{
   "The nick of the person of did the deop'ing",
   "The nick of the person who has been deop'ed",
   NULL
};

char *pevt_chandevoice_help[] =
{
   "The nick of the person of did the devoice'ing",
   "The nick of the person who has been devoice'ed",
   NULL
};

char *pevt_chanunban_help[] =
{
   "The nick of the person of did the unban'ing",
   "The ban mask",
   NULL
};

char *pevt_chanmodegen_help[] =
{
   "The nick of the person setting the mode",
   "The mode's sign (+/-)",
   "The mode letter",
   "The channel it's being set on",
   NULL
};

char *pevt_whois1_help[] =
{
   "Nickname",
   "Username",
   "Host",
   "Full name",
   NULL
};

char *pevt_whois2_help[] =
{
   "Nickname",
   "Channel Membership/\"is an IRC operator\"",
   NULL
};

char *pevt_whois3_help[] =
{
   "Nickname",
   "Server Information",
   NULL
};

char *pevt_whois4_help[] =
{
   "Nickname",
   "Idle time",
   NULL
};

char *pevt_whois4t_help[] =
{
   "Nickname",
   "Idle time",
   "Signon time",
   NULL
};

char *pevt_whois5_help[] =
{
   "Nickname",
   "Away reason",
   NULL
};

char *pevt_whois6_help[] =
{
   "Nickname",
   NULL
};

char *pevt_generic_channel_help[] =
{
   "Channel Name",
   NULL
};

char *pevt_generic_none_help[] =
{
   NULL
};

char *pevt_servertext_help[] =
{
   "Text",
   NULL
};

char *pevt_invited_help[] =
{
   "Channel Name",
   "Nick of person who invited you",
   NULL
};

char *pevt_usersonchan_help[] =
{
   "Channel Name",
   "Users",
   NULL
};

char *pevt_nickclash_help[] =
{
   "Nickname in use",
   "Nick being tried",
   NULL
};

char *pevt_connfail_help[] =
{
   "Error String",
   NULL
};

char *pevt_connect_help[] =
{
   "Host",
   "IP",
   "Port",
   NULL
};

char *pevt_sconnect_help[] =
{
   "PID",
   NULL
};

char *pevt_generic_nick_help[] =
{
   "Nickname",
   NULL
};

char *pevt_wintype_help[] =
{
   "Type of window",
   NULL
};

char *pevt_chanmodes_help[] =
{
   "Channel name",
   "Modes string",
   NULL
};

char *pevt_rawmodes_help[] =
{
   "Nickname",
   "Modes string",
   NULL
};

char *pevt_kill_help[] =
{
   "Nickname",
   "Reason",
   NULL
};

char *pevt_dccchaterr_help[] =
{
   "Nickname",
   "IP address",
   "port",
   NULL
};

char *pevt_dccstall_help[] =
{
   "DCC Type",
   "Filename",
   "Nickname",
   NULL
};

char *pevt_generic_file_help[] =
{
   "Filename",
   NULL
};

char *pevt_dccrecverr_help[] =
{
   "Filename",
   "Destination filename",
   "Nickname",
   NULL
};

char *pevt_dccrecvcomp_help[] =
{
   "Filename",
   "Destination filename",
   "Nickname",
   "CPS",
   NULL
};

char *pevt_dccconfail_help[] =
{
   "DCC Type",
   "Nickname",
   "Error string",
   NULL
};

char *pevt_dcccon_help[] =
{
   "DCC Type",
   "Nickname",
   "IP address",
   "{to|from}",
   NULL
};

char *pevt_dccsendfail_help[] =
{
   "Filename",
   "Nickname",
   NULL
};

char *pevt_dccsendcomp_help[] =
{
   "Filename",
   "Nickname",
   "CPS",
   NULL
};

char *pevt_dccoffer_help[] =
{
   "Filename",
   "Nickname",
   NULL
};

char *pevt_dccabort_help[] =
{
   "DCC type",
   "Filename",
   "Nickname",
   NULL
};

char *pevt_dccresumeoffer_help[] =
{
   "Nickname",
   "Filename",
   "Position",
   NULL
};

char *pevt_dccsendoffer_help[] =
{
   "Nickname",
   "Filename",
   "Size",
   NULL
};

char *pevt_dccgenericoffer_help[] =
{
   "DCC String",
   "Nickname",
   NULL
};

char *pevt_notifynumber_help[] =
{
   "Number of notify items",
   NULL
};

char *pevt_serverlookup_help[] =
{
   "Servername",
   NULL
};

char *pevt_servererror_help[] =
{
   "Text",
   NULL
};

char *pevt_servergenmessage_help[] =
{
   "Text",
   NULL
};

char *pevt_foundip_help[] =
{
   "IP",
   NULL
};

char *pevt_dccrename_help[] =
{
   "Old Filename",
   "New Filename",
   NULL
};

char *pevt_ctcpsend_help[] =
{
   "Receiver",
   "Message",
   NULL
};

char *pevt_ignorelist_help[] =
{
   "Hostmask",
   "Priv Yes/No",
   "Noti Yes/No",
   "Chan Yes/No",
   "Ctcp Yes/No",
   "Invi Yes/No",
   "Unig Yes/No",
   NULL
};

char *pevt_ignoreaddremove_help[] =
{
   "Hostmask",
   NULL
};

char *pevt_resolvinguser_help[] =
{
   "Nickname",
   "Hostname",
   NULL
};

char *pevt_dccmalformed_help[] =
{
   "The Packet",
   NULL
};

struct text_event {
  char *name;
  char **help;
  char *def;
  int num_args;
  char *sound;
};

/* *BBIIGG* struct ahead!! --AGL */

struct text_event te[] = {
   /* Padding for all the non-text signals */
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL}, 
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},

   /* Now we get down to business */

   {"Join", pevt_join_help, "%C4*%C*%C4*%C11 %B$1%B %C14(%C10$3%C14)%C has joined $2", 3, NULL},
   {"Channel Action", pevt_chanaction_help, "%C13*%C $1 $2%O", 2, NULL},
   {"Channel Message", pevt_chanmsg_help, "%C2<%O$1%C2>%C $2%O", 2, NULL},
   {"Private Message", pevt_privmsg_help, "%C12*%C13$1%C12* %C$2%O", 2, NULL},
   {"Change Nick", pevt_changenick_help, "%C4*%C*%C4*%C $1 is now known as $2", 2, NULL},
   {"New Topic", pevt_newtopic_help, "%C4*%C*%C4*%C $1 has changed the topic to: $2%O", 3, NULL},
   {"Topic", pevt_topic_help, "%C4*%C*%C4*%C Topic for %C11$1%C is %C11$2%O", 2, NULL},
   {"Kick", pevt_kick_help, "%C4*%C*%C4*%C $1 has kicked $2 from $3 ($4%O)", 4, NULL},
   {"Part", pevt_part_help, "%C4*%C*%C4*%C $1 %C14(%O$2%C14)%C has left $3", 3, NULL},
   {"Channel Creation", pevt_chandate_help, "%C4*%C*%C4*%C Channel $1 created on $2", 2, NULL},
   {"Topic Creation", pevt_topicdate_help, "%C4*%C*%C4*%C Topic for %C11$1%C set by %C11$2%C at %C11$3%O", 3, NULL},
   {"Quit", pevt_quit_help, "%C4*%C*%C4*%C $1 has quit %C14(%O$2%O%C14)%O", 2, NULL},
   {"Ping Reply", pevt_pingrep_help, "%C4*%C*%C4*%C Ping reply from $1 : $2 second(s)", 2, NULL},
   {"Notice", pevt_notice_help, "%C12-%C13$1%C12-%C $2%O", 2, NULL},
   {"You Joining", pevt_ujoin_help, "%C4*%C*%C4*%C%C11 %B$1%B %C14(%C10$3%C14)%C has joined $2", 3, NULL},
   {"Your Message", pevt_uchanmsg_help, "%C6<%O$1%C6>%C $2%O", 2, NULL},
   {"Private Message to Dialog", pevt_dprivmsg_help, "%C2<%O$1%C2>%C $2%O", 2, NULL},
   {"Your Nick Changing", pevt_uchangenick_help, "%C4*%C*%C4*%C You are now known as $2", 2, NULL},
   {"You're Kicked", pevt_ukick_help, "%C4*%C*%C4*%C You have been kicked from $2 by $3 ($4%O)", 4, NULL},
   {"You Parting", pevt_upart_help, "%C4*%C*%C4*%C You have left channel $3", 3, NULL},
   {"CTCP Sound", pevt_ctcpsnd_help, "%C4*%C*%C4*%C Received a CTCP Sound $1 from $2", 2, NULL},
   {"CTCP Generic", pevt_ctcpgen_help, "%C4*%C*%C4*%C Received a CTCP $1 from $2", 2, NULL},
   {"CTCP Generic to Channel", pevt_ctcpgenc_help, "%C4*%C*%C4*%C Received a CTCP $1 from $2 (to $3)", 3, NULL},
   {"Channel Set Key", pevt_chansetkey_help, "%C4*%C*%C4*%C $1 sets channel keyword to $2", 2, NULL},
   {"Channel Set Limit", pevt_chansetlimit_help, "%C4*%C*%C4*%C $1 sets channel limit to $2", 2, NULL},
   {"Channel Operator", pevt_chanop_help, "%C4*%C*%C4*%C $1 gives channel operator status to $2", 2, NULL},
   {"Channel Voice", pevt_chanvoice_help, "%C4*%C*%C4*%C $1 gives voice to $2", 2, NULL},
   {"Channel Ban", pevt_chanban_help, "%C4*%C*%C4*%C $1 sets ban on $2", 2, NULL},
   {"Channel Remove Keyword", pevt_chanrmkey_help, "%C4*%C*%C4*%C $1 removes channel keyword", 1, NULL},
   {"Channel Remove Limit", pevt_chanrmlimit_help, "%C4*%C*%C4*%C $1 removes user limit", 1, NULL},
   {"Channel DeOp", pevt_chandeop_help, "%C4*%C*%C4*%C $1 removes channel operator status from $2", 2, NULL},
   {"Channel DeVoice", pevt_chandevoice_help, "%C4*%C*%C4*%C $1 removes voice from $2", 2, NULL},
   {"Channel UnBan", pevt_chanunban_help, "%C4*%C*%C4*%C $1 removes ban on $2", 2, NULL},
   {"Channel Mode Generic", pevt_chanmodegen_help, "%C4*%C*%C4*%C $1 sets mode $2$3 $4", 4, NULL},
   {"WhoIs Name Line", pevt_whois1_help, "%C4*%C*%C4*%C %C12[%O$1%C12] %C14(%O$2@$3%C14) %C: $4%O", 4, NULL},
   {"WhoIs Channel/Oper Line", pevt_whois2_help, "%C4*%C*%C4*%C %C12[%O$1%C12]%C $2", 2, NULL},
   {"WhoIs Server Line", pevt_whois3_help, "%C4*%C*%C4*%C %C12[%O$1%C12]%C $2", 2, NULL},
   {"WhoIs Idle Line", pevt_whois4_help, "%C4*%C*%C4*%C %C12[%O$1%C12] %Cidle %C11$2%O", 2, NULL},
   {"WhoIs Idle Line with Signon", pevt_whois4t_help,"%C4*%C*%C4*%C %C12[%O$1%C12] %Cidle %C11$2%C, signon: %C11$3%O", 3, NULL},
   {"WhoIs Away Line", pevt_whois5_help, "%C4*%C*%C4*%C %C12[%O$1%C12] %Cis away %C14(%O$2%O%C14)", 2, NULL},
   {"WhoIs End", pevt_whois6_help, "%C4*%C*%C4*%C %C12[%O$1%C12] %CEnd of WHOIS list.", 1, NULL},
   {"User Limit", pevt_generic_channel_help, "%C4*%C*%C4*%C Cannot join%C11 %B$1 %O(User limit reached).", 1, NULL},
   {"Banned", pevt_generic_channel_help, "%C4*%C*%C4*%C Cannot join%C11 %B$1 %O(You are banned).", 1, NULL},
   {"Invite", pevt_generic_channel_help, "%C4*%C*%C4*%C Cannot join%C11 %B$1 %O(Channel is invite only).", 1, NULL},
   {"Keyword", pevt_generic_channel_help, "%C4*%C*%C4*%C Cannot join%C11 %B$1 %O(Requires keyword).", 1, NULL},
   {"MOTD Skipped", pevt_generic_none_help, "%C4*%C*%C4*%C MOTD Skipped.", 0, NULL},
   {"Server Text", pevt_servertext_help, "%C4*%C*%C4*%C $1%O", 1, NULL},
   {"Invited", pevt_invited_help, "%C4*%C*%C4*%C You have been invited to %C11$1%C by %C11$2", 2, NULL},
   {"Users On Channel", pevt_usersonchan_help, "%C4*%C*%C4*%C %C11Users on $1:%C $2", 2, NULL},
   {"Nick Clash", pevt_nickclash_help, "%C4*%C*%C4*%C $1 already in use. Retrying with $2..", 2, NULL},
   {"Nick Failed", pevt_generic_none_help, "%C4*%C*%C4*%C Nickname already in use. Use /NICK to try another.", 0, NULL},
   {"Unknown Host", pevt_generic_none_help, "%C3*%C*%C3*%C Unknown host. Maybe you misspelled it?", 0, NULL},
   {"Connection Failed", pevt_connfail_help, "%C3*%C*%C3*%C Connection failed. Error: $1", 1, NULL},
   {"Connecting", pevt_connect_help, "%C3*%C*%C3*%C Connecting to %C11$1 %C14(%C11$2%C14)%C port %C11$3%C..", 3, NULL},
   {"Connected", pevt_generic_none_help, "%C3*%C*%C3*%C Connected. Now logging in..", 0, NULL},
   {"Stop Connection", pevt_sconnect_help, "%C3*%C*%C3*%C Stopped previous connection attempt (pid=$1)", 1, NULL},
   {"Disconnected", pevt_generic_none_help, "%C3*%C*%C3*%C Disconnected.", 0, NULL},
   {"No DCC", pevt_generic_none_help, "%C3*%C*%C3*%C No such DCC.", 0, NULL},
   {"Delete Notify", pevt_generic_nick_help, "%C3*%C*%C3*%C $1 deleted from notify list.", 1, NULL},
   {"Add Notify", pevt_generic_nick_help, "%C3*%C*%C3*%C $1 added to notify list.", 1, NULL},
   {"Window Type Error", pevt_wintype_help, "%C3*%C*%C3*%C You can't do that in a $1 window.", 1, NULL},
   {"Channel Modes", pevt_chanmodes_help, "%C4*%C*%C4*%C Channel $1 modes: $2", 2, NULL},
   {"Raw Modes", pevt_rawmodes_help, "%C3*%C*%C3*%C $1 sets modes%B %C14[%O$2%B%C14]%O", 2, NULL},
   {"Killed", pevt_kill_help, "%C3*%C*%C3*%C You have been killed by $1 ($2%O)", 2, NULL},
   {"DCC Stall", pevt_dccstall_help, "%C4*%C*%C4*%C DCC $1 %C11$2 %Cto %C11$3 %Cstalled - aborting.", 3, NULL},
   {"DCC Timeout", pevt_dccstall_help, "%C4*%C*%C4*%C DCC $1 %C11$2 %Cto %C11$3 %Ctimed out - aborting.", 3, NULL},
   {"DCC CHAT Failed", pevt_dccchaterr_help, "%C4*%C*%C4*%C DCC CHAT failed. Connection to $1 %C14[%O$2:$3%C14]%O lost.", 3, NULL},
   {"DCC RECV File Open Error", pevt_generic_file_help, "%C4*%C*%C4*%C DCC RECV: Cannot open $1 for writing - aborting.", 1, NULL},
   {"DCC RECV Failed", pevt_dccrecverr_help, "%C4*%C*%C4*%C DCC RECV $1 ($2) failed. Connection to $3 lost.", 3, NULL},
   {"DCC RECV Complete", pevt_dccrecvcomp_help, "%C4*%C*%C4*%C DCC RECV %C11$1%O from %C11$3%O complete %C14[%C11$4%O cps%C14]%O.", 4, NULL},
   {"DCC Conection Failed", pevt_dccconfail_help, "%C4*%C*%C4*%C DCC $1 connect attempt to %C11$2%O failed (err=$3).", 3, NULL},
   {"DCC Connected", pevt_dcccon_help, "%C4*%C*%C4*%C DCC $1 connection established $4 %C11$2 %C14[%O$3%C14]%O", 4, NULL},
   {"DCC SEND Failed", pevt_dccsendfail_help, "%C4*%C*%C4*%C DCC SEND %C11$1%O failed. Connection to %C11$2%O lost.", 2, NULL},
   {"DCC SEND Complete", pevt_dccsendcomp_help, "%C4*%C*%C4*%C DCC SEND %C11$1%O to %C11$2%O complete %C14[%C11$3%O cps%C14]%O.", 3, NULL},
   {"DCC Offer", pevt_dccoffer_help, "%C4*%C*%C4*%C Offering %C11$1 %Cto %C11$2%O", 2, NULL},
   {"DCC Abort", pevt_dccabort_help, "%C4*%C*%C4*%C DCC $1 %C11$2 %Cto %C11$3 %Caborted.", 3, NULL},
   {"DCC Offer Not Valid", pevt_generic_none_help, "%C4*%C*%C4*%C No such DCC offer.", 0, NULL},
   {"DCC CHAT Reoffer", pevt_generic_nick_help, "%C4*%C*%C4*%C Already offering CHAT to $1", 1, NULL},
   {"DCC CHAT Offering", pevt_generic_nick_help, "%C4*%C*%C4*%C Offering DCC CHAT to $1", 1, NULL},
   {"DCC DRAW Offer", pevt_generic_nick_help, "%C4*%C*%C4*%C Received a DCC DRAW offer from $1. Type %C11/DCC DRAW $1 %Cto accept", 1, NULL},
   {"DCC CHAT Offer", pevt_generic_nick_help, "%C4*%C*%C4*%C Received a DCC CHAT offer from $1", 1, NULL},
   {"DCC RESUME Request", pevt_dccresumeoffer_help, "%C4*%C*%C4*%C %C11$1 %Chas requested to resume %C11$2 %Cfrom %C11$3%C.", 3, NULL},
   {"DCC SEND Offer", pevt_dccsendoffer_help, "%C4*%C*%C4*%C %C11$1 %Chas offered %C11$2 %C(%C11$3 %Cbytes)", 3, NULL},
   {"DCC Generic Offer", pevt_dccgenericoffer_help, "%C4*%C*%C4*%C Received '$1%O' from $2", 2, NULL},
   {"Notify Online", pevt_generic_nick_help, "%C4*%C*%C4*%C Notify: $1 is online ($2).", 2, NULL},
   {"Notify Number", pevt_notifynumber_help, "%C4*%C*%C4*%C $1 users in notify list.", 1, NULL},
   {"Notify Empty", pevt_generic_none_help, "%C4*%C*%C4*%C Notify list is empty.", 0, NULL},
   {"No Running Process", pevt_generic_none_help, "%C4*%C*%C4*%C No process is currently running", 0, NULL},
   {"Process Already Running", pevt_generic_none_help, "%C4*%C*%C4*%C A process is already running", 0, NULL},
   {"Server Lookup", pevt_serverlookup_help, "%C3*%C*%C3*%C Looking up %C11$1%C..", 1, NULL},
   {"Server Connected", pevt_generic_none_help, "%C3*%C*%C3*%C Connected.", 0, NULL},
   {"Server Error", pevt_servererror_help, "%C4*%C*%C4*%C $1%O", 1, NULL},
   {"Server Generic Message", pevt_servergenmessage_help, "%C4*%C*%C4*%C $1%O", 1, NULL},
   {"Found IP", pevt_foundip_help, "%C4*%C*%C4*%C Found your IP: [$1]", 1, NULL},
   {"DCC Rename", pevt_dccrename_help, "%C4*%C*%C4*%C The file %C11$1%C already exists, it has been renamed to %C11$2%C.", 2, NULL},
   {"CTCP Send", pevt_ctcpsend_help, "%C3>%O$1%C3<%O CTCP $2%O", 2, NULL},
   {"Message Send", pevt_ctcpsend_help, "%C3>%O$1%C3<%O $2%O", 2, NULL},
   {"Notice Send", pevt_ctcpsend_help, "%C3>%O$1%C3<%O $2%O", 2, NULL},
   {"Receive Wallops", pevt_dprivmsg_help, "%C12-%C13$1/Wallops%C12-%C $2%O", 2, NULL},
   /* XP_HIGHLIGHT */
   {NULL, NULL, NULL, 0, NULL},
   {"Ignore Header", pevt_generic_none_help, "%C08,02 Hostmask             PRIV NOTI CHAN CTCP INVI UNIG %O", 0, NULL},
   {"Ignore List", pevt_ignorelist_help, " $1 $2  $3  $4  $5  $6  $7", 7, NULL},
   {"Ignore Footer", pevt_generic_none_help, "%C08,02                                                    %O", 0 , NULL},
   {"Ignore Add", pevt_ignoreaddremove_help, "%O%C11$1%O added to ignore list.", 1, NULL},
   {"Ignore Remove",  pevt_ignoreaddremove_help, "%O%C11$1%O removed from ignore list.", 1, NULL},
   {"Resolving User", pevt_resolvinguser_help, "%C4*%C*%C4*%C Looking up IP number for%C11 $1%O..", 2, NULL},
   {"Ignorelist Empty", pevt_generic_none_help, "  Ignore list is empty.", 0, NULL},
   {"Ignore Changed", pevt_ignoreaddremove_help, "Ignore on %C11$1%O changed.", 0, NULL},
   {"Notify Offline", pevt_generic_nick_help, "%C4*%C*%C4*%C Notify: $1 is offline ($2).", 2, NULL},
   {"DCC Malformed From", pevt_generic_nick_help, "%C4*%C*%C4*%C Received a malformed DCC request from %C11$1%O.", 1, NULL},
   {"DCC Malformed Packet", pevt_dccmalformed_help, "%C4*%C*%C4*%C Contents of packet: \"$1%O\".", 1, NULL},
};

int
text_event (int i)
{
   if (te[i].name == NULL)
      return 0;
   else
      return 1;
}

void
pevent_load_defaults ()
{
   int i, len;

   for (i = 0; i < NUM_XP; i++)
   {
      if (!text_event(i))
	 continue;
      len = strlen (te[i].def);
      len++;
      if (pntevts_text[i])
         free (pntevts_text[i]);
      pntevts_text[i] = malloc (len);
      memcpy (pntevts_text[i], te[i].def, len);
   }
}

void
pevent_make_pntevts ()
{
   int i, m, len;
   char out[1024];

   for (i = 0; i < NUM_XP; i++)
   {
      if (!text_event(i))
	 continue;
      if (pntevts[i] != NULL)
         free (pntevts[i]);
      if (pevt_build_string (pntevts_text[i], &(pntevts[i]), &m) != 0)
      {
         snprintf (out, sizeof (out), "Error parsing event %s.\nLoading default", te[i].name);
         gtkutil_simpledialog (out);
         free (pntevts_text[i]);
         len = strlen (te[i].def) + 1;
         pntevts_text[i] = malloc (len);
         memcpy (pntevts_text[i], te[i].def, len);
         if (pevt_build_string (pntevts_text[i], &(pntevts[i]), &m) != 0)
         {
            fprintf (stderr, "XChat CRITICAL *** default event text failed to build!\n");
            abort ();
         }
      }
      check_special_chars (pntevts[i]);
   }
}

int
pevent_load (char *filename)
{
   char *buf, *ibuf;
   int fd, i = 0, n, len, err = 0, pnt = 0;
   struct stat st;

   buf = malloc (1000);
   if (filename == NULL)
      snprintf (buf, 1000, "%s/printevents.conf", get_xdir ());
   else
      strcpy (buf, filename);

   fd = open (buf, O_RDONLY);
   free (buf);
   if (fd < 0)
      return 1;
   if (fstat (fd, &st) != 0)
      return 1;
   ibuf = malloc (st.st_size);
   read (fd, ibuf, st.st_size);
   close (fd);

   while (buf_get_line (ibuf, &buf, &pnt, st.st_size))
   {
   startevent:
      if (buf[0] == '#')
         continue;
      if (strlen (buf) == 0)
         continue;
      
      while (!text_event(i)) {
	 i++;
	 if (i == NUM_XP) {
	    /* This action might be wrong --AGL */
	    free(ibuf);
	    return 1;
	 }
	    
      }
      
      if (strcmp (buf, te[i].name) != 0)
      {
         /* Sigh - it's out of order */
	 for (n = 0; n < NUM_XP; n++)
         {
	    if (!text_event(n))
	       continue;
            if (strcmp (buf, te[n].name) == 0)
            {
               i = n;
               break;
            }
         }
         if (n == NUM_XP)
         {
            err++;
            gtkutil_simpledialog ("Config file contained data for unknown event");
            if (strstr (buf, "$1"))
            {
               gtkutil_simpledialog ("The config file seems to be off by one,\nattempting to correct\n");
               continue;
            }
            if (err == 3)
            {
               gtkutil_simpledialog ("Too many errors, aborting load");
               free (ibuf);
               return 1;
            }
            buf_get_line (ibuf, &buf, &pnt, st.st_size);
            continue;
         }
      }
      if (!buf_get_line (ibuf, &buf, &pnt, st.st_size))
      {
         gtkutil_simpledialog (
		"The Print Events config file is corrupted, aborting.\n"
		"The Print Events may be corrupted");
         free (ibuf);
         return 1;
      }
      err = 0;
      len = strlen (buf);
      len++;
      if (pntevts_text[i])
         free (pntevts_text[i]);
      pntevts_text[i] = malloc (len);
      memcpy (pntevts_text[i], buf, len);

      if (!buf_get_line (ibuf, &buf, &pnt, st.st_size)) {
	   /*gtkutil_simpledialog (
		"The Print Events config file is corrupted, aborting.\n"
		"The Print Events may be corrupted");*/
         free (ibuf);
         return 0;
      }
      if (buf[0] != '!' && buf[0] != '?') {
	 i++;
	 if (i == NUM_XP)
	   break;
	 goto startevent;
      }
      
      len = strlen (buf);
      len++;
      if (te[i].sound)
	 free (te[i].sound);
      if (buf[0] == '!') {
	 te[i].sound = malloc (len);
	 memcpy (te[i].sound, &buf[1], len - 1);
      } else {
	 te[i].sound = NULL;
      }
      
      i++;
      if (i == NUM_XP)
      {
         break;
      }
   }

   free (ibuf);
   return 0;
}

void
pevent_check_all_loaded ()
{
   int i, len;
/*   char out[1024]; */

   for (i = 0; i < NUM_XP; i++)
   {
      if (!text_event(i))
	 continue;
      if (pntevts_text[i] == NULL)
      {
         /*snprintf(out, sizeof(out), "The data for event %s failed to load. Reverting to defaults.\nThis may be because a new version of XChat is loading an old config file.\n\nCheck all print event texts are correct", evtnames[i]);
            gtkutil_simpledialog(out); */
         len = strlen (te[i].def) + 1;
         pntevts_text[i] = malloc (len);
         memcpy (pntevts_text[i], te[i].def, len);
      }
   }
}

void
load_text_events ()
{
   /* I don't free these as the only time when we don't need them
      is once XChat has exit(2)'ed, so why bother ?? --AGL */

   pntevts_text = malloc (sizeof (char *) * (NUM_XP));
   memset (pntevts_text, 0, sizeof (char *) * (NUM_XP));
   pntevts = malloc (sizeof (char *) * (NUM_XP));
   memset (pntevts, 0, sizeof (char *) * (NUM_XP));

   if (pevent_load (NULL))
      pevent_load_defaults ();
   pevent_check_all_loaded ();
   pevent_make_pntevts ();
}

void
display_event (char *i, struct session *sess, int numargs, char **args)
{
   int len, oi, ii, *ar;
   char o[4096], d, a, done_all = FALSE;

   oi = ii = len = d = a = 0;

   if (i == NULL)
      abort ();

   while (done_all == FALSE)
   {
      d = i[ii++];
      switch (d)
      {
      case 0:
         /**(&len) = (int) (*(i + ii));*//* <- endian problem? */
         memcpy (&len, &(i[ii]), sizeof (int));
         ii += sizeof (int);
         if (oi + len > sizeof (o))
         {
            printf ("Overflow in display_event (%s)\n", i);
            return;
         }
         memcpy (&(o[oi]), &(i[ii]), len);
         oi += len;
         ii += len;
         break;
      case 1:
         a = i[ii++];
         if (a > numargs)
         {
            printf ("XChat DEBUG: display_event: arg > numargs (%d %d %s), ABORTING\n", a, numargs, i);
            abort ();
         }
         ar = (int *) args[(int) a];
         if (ar == NULL)
         {
            printf ("Error args[a] == NULL in display_event\n");
            abort ();
         }
         len = strlen ((char *) ar);
         memcpy (&o[oi], ar, len);
         oi += len;
         break;
      case 2:
         o[oi++] = '\n';
         o[oi++] = 0;
         done_all = TRUE;
         continue;
      }
   }
   o[oi] = 0;
   if (*o == '\n') return;
   PrintText (sess, o);
}

int
pevt_build_string (char *input, char **output, int *max_arg)
{
   struct pevt_stage1 *s = NULL, *base = NULL,
              *last = NULL, *next;
   int clen;
   char o[4096], d, *obuf, *i;
   int oi, ii, ti, max = -1, len, x;

   len = strlen (input);
   i = malloc (len + 1);
   memcpy (i, input, len + 1);
   check_special_chars (i);

   len = strlen (i);

   clen = oi = ii = ti = 0;

   for (;;)
   {
      if (ii == len)
         break;
      d = i[ii++];
      if (d != '$')
      {
         o[oi++] = d;
         continue;
      }
      if (i[ii] == '$')
      {
         o[oi++] = '$';
         continue;
      }
      if (oi > 0)
      {
         s = (struct pevt_stage1 *) malloc (sizeof (struct pevt_stage1));
         if (base == NULL)
            base = s;
         if (last != NULL)
            last->next = s;
         last = s;
         s->next = NULL;
         s->data = malloc (oi + sizeof (int) + 1);
         s->len = oi + sizeof (int) + 1;
         clen += oi + sizeof (int) + 1;
         s->data[0] = 0;
         memcpy (&(s->data[1]), &oi, sizeof (int));
         memcpy (&(s->data[1 + sizeof (int)]), o, oi);
         oi = 0;
      }
      if (ii == len)
      {
         gtkutil_simpledialog ("String ends with a $");
         return 1;
      }
      d = i[ii++];
      if (d == 'a')
      {                         /* Hex value */
         x = 0;
         if (ii == len)
            goto a_len_error;
         d = i[ii++];
         d -= '0';
         x = d * 100;
         if (ii == len)
            goto a_len_error;
         d = i[ii++];
         d -= '0';
         x += d * 10;
         if (ii == len)
            goto a_len_error;
         d = i[ii++];
         d -= '0';
         x += d;
         if (x > 255)
            goto a_range_error;
         o[oi++] = x;
         continue;

       a_len_error:
         gtkutil_simpledialog ("String ends in $a");
         return 1;
       a_range_error:
         gtkutil_simpledialog ("$a value is greater then 255");
         return 1;
      }
      if (d < '0' || d > '9')
      {
         snprintf (o, sizeof (o), "Error, invalid argument $%c\n", d);
         gtkutil_simpledialog (o);
         return 1;
      }
      d -= '0';
      if (max < d)
         max = d;
      s = (struct pevt_stage1 *) malloc (sizeof (struct pevt_stage1));
      if (base == NULL)
         base = s;
      if (last != NULL)
         last->next = s;
      last = s;
      s->next = NULL;
      s->data = malloc (2);
      s->len = 2;
      clen += 2;
      s->data[0] = 1;
      s->data[1] = d - 1;
   }
   if (oi > 0)
   {
      s = (struct pevt_stage1 *) malloc (sizeof (struct pevt_stage1));
      if (base == NULL)
         base = s;
      if (last != NULL)
         last->next = s;
      last = s;
      s->next = NULL;
      s->data = malloc (oi + sizeof (int) + 1);
      s->len = oi + sizeof (int) + 1;
      clen += oi + sizeof (int) + 1;
      s->data[0] = 0;
      memcpy (&(s->data[1]), &oi, sizeof (int));
      memcpy (&(s->data[1 + sizeof (int)]), o, oi);
      oi = 0;
   }
   s = (struct pevt_stage1 *) malloc (sizeof (struct pevt_stage1));
   if (base == NULL)
      base = s;
   if (last != NULL)
      last->next = s;
   last = s;
   s->next = NULL;
   s->data = malloc (1);
   s->len = 1;
   clen += 1;
   s->data[0] = 2;

   oi = 0;
   s = base;
   obuf = malloc (clen);
   while (s)
   {
      next = s->next;
      memcpy (&obuf[oi], s->data, s->len);
      oi += s->len;
      free (s->data);
      free (s);
      s = next;
   }

   free (i);

   if (max_arg)
      *max_arg = max;
   if (output)
      *output = obuf;
   return 0;
}

void
play_wave (char *file)
{
   char *buf = malloc(512);
   int pid;

   snprintf(buf, 512, "%s/%s", prefs.sounddir, file);
   if (access (buf, R_OK) == 0)
   {
	snprintf(buf, 512, "%s %s/%s", prefs.soundcmd, prefs.sounddir, file);
	pid = fork();
	if (pid == -1) {
	   free (buf);
	   return;
	}
	if (pid == 0) {
	   char *argv[4];
	   argv[0] = "sh";
	   argv[1] = "-c";
	   argv[2] = buf;
	   argv[3] = 0;
	   execve ("/bin/sh", argv, environ);
	}
   }

   free (buf);
}

int
textsignal_handler (struct session *sess, void *b, void *c,
             void *d, void *e, char f)
{
   /* This handler *MUST* always be the last in the chain
      because it doesn't call the next handler
    */
   
   char *args[8];
   int numargs, i;

   if (!text_event(current_signal))
   {
      printf ("error, textsignal_handler passed non TE signal (%d)\n", current_signal);
      abort ();
   }

   numargs = te[current_signal].num_args;
   i = 0;

#ifdef USE_PERL
   if(perl_print(te[current_signal].name, sess, b,c,d,e)) return TRUE;
#endif

   if (te[current_signal].sound)
	play_wave (te[current_signal].sound);

   args[0] = b;
   args[1] = c;
   args[2] = d;
   args[3] = e;

   display_event (pntevts[(int) current_signal], sess, numargs, args);
   return 0;
}

void
printevent_setup ()
{
   int evt = 0;
   struct xp_signal *sig;

   sig = (struct xp_signal *) malloc (sizeof (struct xp_signal));

   sig->signal = -1;
   sig->naddr = NULL;
   sig->callback = XP_CALLBACK (textsignal_handler);
   sig->last = sig->next = NULL;
#ifdef USE_PLUGIN
   sig->mod = NULL;
#endif

   for (evt = 0; evt < NUM_XP; evt++)
   {
      if (!text_event(evt))
	 continue;
      sigroots[evt] = sig;
      sighandler[evt] = XP_CALLBACK (textsignal_handler);
   }
}

void
pevent_dialog_save (char *fn)
{
   int fd, i;
   char buf[512];

   if (!fn)
      snprintf (buf, 510, "%s/printevents.conf", get_xdir ());
   else
      snprintf (buf, 510, "%s", fn);
   fd = open (buf, O_CREAT | O_TRUNC | O_WRONLY, 0x180);
   if (fd < 0)
   {
      gtkutil_simpledialog ("Error opening config file\n");
      return;
   }
   write (fd, buf, snprintf (buf, 510, "# XChat text events config file\n\n"));

   for (i = 0; i < NUM_XP; i++)
   {
      if (!text_event(i))
	 continue;
      write (fd, buf, snprintf (buf, 510, "%s\n", te[i].name));
      write (fd, buf, snprintf (buf, 510, "%s\n", pntevts_text[i]));
      if (te[i].sound && te[i].sound[0])
	 write (fd, buf, snprintf (buf, 510, "!%s\n", te[i].sound));
      else
	 write (fd, buf, snprintf (buf, 510, "?\n"));
   }
   close (fd);
}

static void
pevent_dialog_close (GtkWidget * wid, gpointer * arg)
{
   pevent_dialog = NULL;
   pevent_dialog_save (NULL);
}

static void
pevent_dialog_update_sound (GtkWidget * wid, GtkWidget * clist)
{
   int row, sig;

   row = gtkutil_clist_selection (pevent_dialog_list);
   if (row == -1)
      return;

   gtk_clist_set_text (GTK_CLIST(clist), row, 2,
				gtk_entry_get_text(GTK_ENTRY(wid)));

   sig = (int) gtk_clist_get_row_data(GTK_CLIST(clist), row);

   if (te[sig].sound)
	free (te[sig].sound);

   te[sig].sound = strdup (gtk_entry_get_text(GTK_ENTRY(wid)) );
}

static void
pevent_dialog_update (GtkWidget * wid, GtkWidget * twid)
{
   int row, len, m;
   char *text, *out;
   int sig;

   row = gtkutil_clist_selection (pevent_dialog_list);
   if (row == -1)
      return;

   sig = (int) gtk_clist_get_row_data(GTK_CLIST(pevent_dialog_list), row);

   text = gtk_entry_get_text (GTK_ENTRY (wid));
   len = strlen (text);

   if (pevt_build_string (text, &out, &m) != 0)
   {
      gtkutil_simpledialog ("There was an error parsing the string");
      return;
   }
   if (m > te[sig].num_args)
   {
      free (out);
      out = malloc (4096);
      snprintf (out, 4096, "This signal is only passed %d args, $%d is invalid", te[sig].num_args, m);
      gtkutil_simpledialog (out);
      free (out);
      return;
   }
   gtk_clist_set_text (GTK_CLIST (pevent_dialog_list), row, 1, text);

   if (pntevts_text[sig])
      free (pntevts_text[sig]);
   if (pntevts[sig])
      free (pntevts[sig]);

   pntevts_text[sig] = malloc (len + 1);
   memcpy (pntevts_text[sig], text, len + 1);
   pntevts[sig] = out;

   out = malloc (len + 2);
   memcpy (out, text, len + 1);
   out[len] = '\n';
   out[len + 1] = 0;
   check_special_chars (out);

   PrintTextRaw (twid, out,
              font_normal, font_bold);
   free (out);
}

static void
pevent_dialog_unselect (GtkWidget * clist, gint row, gint column,
 GdkEventButton * even, gpointer none)
{
   gtk_entry_set_text (GTK_ENTRY (pevent_dialog_sound_entry), "");
   gtk_entry_set_text (GTK_ENTRY (pevent_dialog_entry), "");
   gtk_clist_clear (GTK_CLIST (pevent_dialog_hlist));
}

static void
pevent_dialog_hfill (GtkWidget * list, int e)
{
   gchar *nnew[2];
   int i = 0;
   char *text, buf[64];

   if (!text_event(e))
      return;
   
   text = te[e].help[i];

   gtk_clist_clear (GTK_CLIST (list));
   while (text)
   {
      snprintf (buf, sizeof (buf), "%d", i + 1);
      if (text[0] == '\001')
         text++;
      nnew[0] = buf;
      nnew[1] = text;
      gtk_clist_append (GTK_CLIST (list), nnew);

      i++;
      text = te[e].help[i];
   }
}

static void
pevent_dialog_select (GtkWidget * clist, gint row, gint column,
 GdkEventButton * even, gpointer none)
{
   char *cmd, *snd;
   int sig;

   row = gtkutil_clist_selection (pevent_dialog_list);
   if (row != -1)
   {
      gtk_clist_get_text (GTK_CLIST (clist), row, 1, &cmd);
      gtk_entry_set_text (GTK_ENTRY (pevent_dialog_entry), cmd);
	gtk_clist_get_text (GTK_CLIST (clist), row, 2, &snd);
	gtk_entry_set_text (GTK_ENTRY (pevent_dialog_sound_entry), snd);
      sig = (int) gtk_clist_get_row_data(GTK_CLIST(clist), row);
      pevent_dialog_hfill (pevent_dialog_hlist, sig);
   } else
   {
      pevent_dialog_unselect (0, 0, 0, 0, 0);
   }
}

static void
pevent_dialog_fill (GtkWidget * list)
{
   int i, row;
   gchar *nnew[3];

   gtk_clist_clear (GTK_CLIST (list));

   for (i = 0; i < NUM_XP; i++)
   {
      if (!text_event(i))
	 continue;
      nnew[0] = te[i].name;
      nnew[1] = pntevts_text[i];
	if (te[i].sound)
	   nnew[2] = te[i].sound;
	else
	   nnew[2] = "";
      row = gtk_clist_append (GTK_CLIST (list), nnew);
      gtk_clist_set_row_data(GTK_CLIST(list), row, (void *) i);
   }
}

static void
pevent_save_req_cb (void *arg1, void *arg2, char *file)
{
   pevent_dialog_save (file);
   free (file);
}

static void
pevent_save_cb (GtkWidget * wid, void *data)
{
   if (data)
   {
      gtkutil_file_req ("Print Texts File", pevent_save_req_cb, NULL, NULL,
                        TRUE);
      return;
   }
   pevent_dialog_save (NULL);
}

static void
pevent_load_req_cb (void *arg1, void *arg2, char *file)
{
   pevent_load (file);
   pevent_make_pntevts();
   pevent_dialog_fill (pevent_dialog_list);
   pevent_dialog_select (pevent_dialog_list, -1, -1, NULL, NULL);
   free (file);
}

static void
pevent_load_cb (GtkWidget * wid, void *data)
{
   gtkutil_file_req ("Print Texts File", pevent_load_req_cb, NULL, NULL, FALSE);
}

static void
pevent_ok_cb (GtkWidget * wid, void *data)
{
   gtk_widget_destroy (pevent_dialog);
}

static void
pevent_test_cb (GtkWidget * wid, GtkWidget * twid)
{
   int len, n;
   char *out, *text;

   for (n = 0; n < NUM_XP; n++)
   {
      if (!text_event(n))
	 continue;
      text = pntevts_text[n];
      len = strlen (text);

      out = malloc (len + 2);
      memcpy (out, text, len + 1);
      out[len] = '\n';
      out[len + 1] = 0;
      check_special_chars (out);

      PrintTextRaw (twid, out,
              font_normal, font_bold);
      free (out);
   }
}

/* from settings.c - but its not there anymore */
void 
gui_entry (char *label, int max, GtkWidget * box, char *def, char *suffix, GtkWidget ** entry, void *suffixcallback)
{
   GtkWidget *wid, *hbox;

   hbox = gtk_hbox_new (0, 5);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 3);
   gtk_widget_show (hbox);

   wid = gtk_label_new (label);
   gtk_widget_set_usize (wid, 132, 0);
   gtk_misc_set_alignment (GTK_MISC (wid), 1, 0.5);
   gtk_box_pack_start (GTK_BOX (hbox), wid, 0, 0, 0);
   gtk_widget_show (wid);

   wid = gtk_entry_new_with_max_length (max);
   if (entry)
      *entry = wid;
   gtk_entry_set_text (GTK_ENTRY (wid), def);
   gtk_container_add (GTK_CONTAINER (hbox), wid);
   gtk_widget_show (wid);

   if (suffixcallback)
   {
      wid = gtk_button_new_with_label ("...");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                          GTK_SIGNAL_FUNC (suffixcallback), *entry);
      gtk_box_pack_end (GTK_BOX (hbox), wid, 0, 0, 0);
      gtk_widget_show (wid);
   } else
   {
      if (suffix)
      {
         wid = gtk_label_new (suffix);
         gtk_widget_show (wid);
         gtk_box_pack_end (GTK_BOX (hbox), wid, 0, 0, 0);
      }
   }
   gtk_box_pack_start (GTK_BOX (box), hbox, 0, 0, 0);
}

void
pevent_dialog_show ()
{
   GtkWidget *vbox, *vbox2, *hbox, *tbox,
            *wid, *bh, *th;
   gchar *titles[] =
   {"Event", "Text", "Sound"};
   gchar *help_titles[] =
   {"$ Number", "Description"};

   if (pevent_dialog)
   {
      gdk_window_show (pevent_dialog->window);
      return;
   }

   pevent_dialog = gtkutil_window_new ("Edit Events", "Edit Events", 600, 455,
              pevent_dialog_close, 0);
   gtk_window_set_policy (GTK_WINDOW (pevent_dialog), TRUE, TRUE, FALSE);

   vbox2 = gtk_vbox_new (0, 2);
   vbox = gtk_vbox_new (0, 2);
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
   gtk_container_add (GTK_CONTAINER (pevent_dialog), vbox);
   gtk_widget_show (vbox);

   wid = gtk_vpaned_new ();
   th = gtk_vbox_new (0, 2);
   bh = gtk_vbox_new (0, 2);
   gtk_widget_show (th);
   gtk_widget_show (bh);
   gtk_paned_pack1 (GTK_PANED (wid), th, 1, 1);
   gtk_paned_pack2 (GTK_PANED (wid), bh, 0, 1);
   gtk_box_pack_start (GTK_BOX (vbox), wid, 1, 1, 0);
   gtk_widget_show (wid);
   pevent_dialog_list = gtkutil_clist_new (3, titles, th, GTK_POLICY_ALWAYS,
              pevent_dialog_select, 0,
            pevent_dialog_unselect, 0,
                GTK_SELECTION_BROWSE);
   gtk_clist_set_column_width (GTK_CLIST (pevent_dialog_list), 0, 80);
   gtk_clist_set_column_width (GTK_CLIST (pevent_dialog_list), 1, 380);

   pevent_dialog_twid = gtk_text_new (0, 0);

   pevent_dialog_entry = gtk_entry_new_with_max_length (255);
   gtk_widget_set_usize (pevent_dialog_entry, 96, 0);

   gtk_signal_connect (GTK_OBJECT (pevent_dialog_entry), "activate",
                       GTK_SIGNAL_FUNC (pevent_dialog_update), pevent_dialog_twid);

   gtk_box_pack_start (GTK_BOX (bh), pevent_dialog_entry, 0, 0, 0);
   gtk_widget_show (pevent_dialog_entry);

   gui_entry("Sound file: ", 64, bh, "", 0, &pevent_dialog_sound_entry, 0);
   gtk_signal_connect (GTK_OBJECT (pevent_dialog_sound_entry), "activate",
                       GTK_SIGNAL_FUNC (pevent_dialog_update_sound), pevent_dialog_list);

   tbox = gtk_hbox_new (0, 0);
   gtk_container_add (GTK_CONTAINER (bh), tbox);
   gtk_widget_show (tbox);

   gtk_widget_set_style (pevent_dialog_twid, channelwin_style);
   gtk_widget_set_usize (pevent_dialog_twid,
                         150,
                         20);
   gtk_text_set_word_wrap (GTK_TEXT (pevent_dialog_twid), TRUE);
   gtk_container_add (GTK_CONTAINER (tbox), pevent_dialog_twid);
   gtk_widget_show (pevent_dialog_twid);

   wid = gtk_vscrollbar_new (GTK_TEXT (pevent_dialog_twid)->vadj);
   gtk_box_pack_start (GTK_BOX (tbox), wid, FALSE, FALSE, 0);
   show_and_unfocus (wid);

   pevent_dialog_hlist = gtkutil_clist_new (2, help_titles, bh,
                    GTK_POLICY_ALWAYS,
                     NULL, 0, NULL, 0,
                GTK_SELECTION_BROWSE);
   gtk_clist_set_column_width (GTK_CLIST (pevent_dialog_list), 0, 65);
   gtk_widget_show (pevent_dialog_hlist);

   pevent_dialog_fill (pevent_dialog_list);

   hbox = gtk_hbox_new (0, 2);
   gtk_box_pack_end (GTK_BOX (vbox), hbox, 0, 0, 0);
   wid = gtk_button_new_with_label ("Save");
   gtk_box_pack_end (GTK_BOX (hbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (pevent_save_cb), NULL);
   gtk_widget_show (wid);
   wid = gtk_button_new_with_label ("Save As");
   gtk_box_pack_end (GTK_BOX (hbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (pevent_save_cb), (void *) 1);
   gtk_widget_show (wid);
   wid = gtk_button_new_with_label ("Load From");
   gtk_box_pack_end (GTK_BOX (hbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (pevent_load_cb), (void *) 0);
   gtk_widget_show (wid);
   wid = gtk_button_new_with_label ("Test All");
   gtk_box_pack_end (GTK_BOX (hbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (pevent_test_cb), pevent_dialog_twid);
   gtk_widget_show (wid);
#ifdef USE_GNOME
   wid = gnome_stock_button (GNOME_STOCK_BUTTON_OK);
#else
   wid = gtk_button_new_with_label ("Ok");
#endif
   gtk_box_pack_start (GTK_BOX (hbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (pevent_ok_cb), NULL);
   gtk_widget_show (wid);

   gtk_widget_show (hbox);

   gtk_widget_show (pevent_dialog);
}
