/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "cfgfiles.h"
#include "xchat.h"
#include "menu.h"
#include "gtkutil.h"
#include "userlist.h"
#include <gdk/gdkkeysyms.h>
#include <fcntl.h>
#include <arpa/inet.h>
#ifdef USE_IMLIB
#include <gdk_imlib.h>
#endif
#ifdef USE_GNOME
#include <zvt/zvtterm.h>
#endif
#ifdef USE_PANEL
#include <applet-widget.h>
extern int nopanel;
extern GtkWidget *panel_applet, *panel_box;
GtkWidget *panel_popup = NULL;
#endif

struct session *current_tab = 0;
GtkWidget *main_window = 0;
GtkWidget *main_book;
GtkStyle *normaltab_style = 0;
GtkStyle *redtab_style;
GtkStyle *bluetab_style;

extern struct session *menu_sess;
extern struct xchatprefs prefs;
extern GSList *sess_list;
extern GSList *button_list;
extern GtkStyle *channelwin_style;
extern GdkFont *dialog_font_normal;
extern GdkFont *dialog_font_bold;
extern GdkFont *font_normal;
extern GdkFont *font_bold;
extern gint xchat_is_quitting;

extern void palette_alloc (GtkWidget * widget);
extern void menu_urlmenu (struct session *sess, GdkEventButton * event, char *url);
extern void menu_hostmenu (struct session *sess, GdkEventButton * event, char *url);
extern void menu_chanmenu (struct session *sess, GdkEventButton * event, char *url);
extern void userlist_dnd_drop (GtkWidget *, GdkDragContext *, gint, gint, GtkSelectionData *, guint, guint32, struct session *);
extern int userlist_dnd_motion (GtkWidget *, GdkDragContext *, gint, gint, guint);
extern int userlist_dnd_leave (GtkWidget *, GdkDragContext *, guint time);
extern int tcp_send_len (struct server *serv, char *buf, int len);
extern int tcp_send (struct server *serv, char *buf);
extern void end_logging (int fd);
extern void menu_popup (struct session *sess, GdkEventButton * event, char *nick, char *);
extern void clear_user_list (struct session *sess);
extern void xchat_cleanup (void);
extern void kill_session_callback (GtkWidget * win, struct session *sess);
#ifdef USE_GNOME
extern void createmenus (void *app, struct session *sess);
#else
extern GtkWidget *createmenus (struct session *sess);
#endif
extern void handle_inputgad (GtkWidget * igad, struct session *sess);
extern struct session *find_dialog (struct server *serv, char *nick);
extern void my_gtk_entry_set_text (GtkWidget * wid, char *text, struct session *sess);
extern void my_gtk_toggle_state (GtkWidget * wid, int state, char which, struct session *sess);
extern void popup_cmd (GtkWidget * igad, char *cmd);
extern void path_part (char *file, char *path);
extern struct session *new_dialog (struct session *sess);
extern int close_dialog (GtkWidget * wid, struct session *sess);
extern struct user *find_name (struct session *sess, char *name);
extern void create_panel_widget ();
extern void popup_eval_command (struct session *sess, char *cmd, char *nick, char *allnick);
int key_handle_key_press (GtkWidget *, GdkEventKey *, gpointer);

void userlist_button (GtkWidget * box, char *label, char *cmd,
                      struct session *sess, int a, int b, int c, int d);

GdkColor colors[] =
{
   {0, 0xffff, 0xffff, 0xffff}, /* 0  white */
   {0, 0, 0, 0},                /* 1  black */
   {0, 0, 0, 0xcccc},           /* 2  blue */
   {0, 0, 0xcccc, 0},           /* 3  green */
   {0, 0xcccc, 0, 0},           /* 4  red */
   {0, 0xbbbb, 0xbbbb, 0},      /* 5  yellow/brown */
   {0, 0xbbbb, 0, 0xbbbb},      /* 6  purple */
   {0, 0xffff, 0xaaaa, 0},      /* 7  orange */
   {0, 0xffff, 0xffff, 0},      /* 8  yellow */
   {0, 0, 0xffff, 0},           /* 9  green */
   {0, 0, 0xbbbb, 0xbbbb},      /* 10 aqua */
   {0, 0, 0xffff, 0xffff},      /* 11 light aqua */
   {0, 0, 0, 0xffff},           /* 12 blue */
   {0, 0xffff, 0, 0xffff},      /* 13 pink */
   {0, 0x7777, 0x7777, 0x7777}, /* 14 grey */
   {0, 0x9999, 0x9999, 0x9999}, /* 15 light grey */
/*     { 0, 0xf8f4, 0xffff, 0xbbbb }, *//* 16 light yellow for tooltips */
};


void
make_non_channel_window (struct session *sess, int state)
{
   int i;

   if (sess->flag_wid[0])
   {
      for (i = 0; i < 8; i++)
      {
         gtk_widget_set_sensitive (sess->flag_wid[i], state);
      }
      gtk_widget_set_sensitive (sess->limit_entry, state);
      gtk_widget_set_sensitive (sess->key_entry, state);
   }
   gtk_entry_set_editable ((GtkEntry *) sess->topicgad, state);
}

void
maingui_createbuttons (struct session *sess)
{
   struct popup *pop;
   GSList *list = button_list;
   int a = 0, b = 0;

   sess->button_box = gtk_table_new (5, 2, FALSE);
   gtk_box_pack_end (GTK_BOX (sess->nl_box), sess->button_box, FALSE, FALSE, 1);
   gtk_widget_show (sess->button_box);

   while (list)
   {
      pop = (struct popup *) list->data;
      if (pop->cmd[0])
      {
         userlist_button (sess->button_box, pop->name, pop->cmd, sess, a, a + 1, b, b + 1);
         a++;
         if (a == 2)
         {
            a = 0;
            b++;
         }
      }
      list = list->next;
   }
}

void
maingui_updatebuttons (struct session *sess)
{
   if (sess->button_box)
   {
      gtk_widget_destroy (sess->button_box);
      sess->button_box = 0;
   }
   if (!prefs.nouserlistbuttons)
      maingui_createbuttons (sess);
}

void
gui_set_title (struct session *sess)
{
   char tbuf[200];
   if (!sess->server->connected && !sess->is_dialog)
      strcpy (tbuf, "X-Chat (" VERSION ")");
   else
   {
      if (sess->channel[0] == 0 || sess->is_server)
         snprintf (tbuf, sizeof tbuf, "X-Chat (" VERSION "): %s", sess->server->servername);
      else
         snprintf (tbuf, sizeof tbuf, "X-Chat (" VERSION "): %s / %s", sess->server->servername, sess->channel);
   }
   if (sess->is_tab)
   {
      if (!main_window)
         return;
      if (current_tab == sess)
         gtk_window_set_title ((GtkWindow *) main_window, tbuf);
   } else
      gtk_window_set_title ((GtkWindow *) sess->window, tbuf);
}

void
set_channel (struct session *sess)
{
   if (!sess->is_tab && sess->is_dialog)
      return;
   gtk_label_set_text (GTK_LABEL (sess->changad), sess->channel);
#ifdef USE_PANEL
   if (sess->panel_button)
      gtk_label_set_text (GTK_LABEL (GTK_BIN (sess->panel_button)->child), sess->channel);
#endif
}

extern char chan_flags[];

void
clear_channel (struct session *sess)
{
   sess->channel[0] = 0;
   gtk_entry_set_text (GTK_ENTRY (sess->topicgad), "");
   gtk_label_set_text (GTK_LABEL (sess->namelistinfo), " ");
   gtk_label_set_text (GTK_LABEL (sess->changad), "<none>");
#ifdef USE_PANEL
   if (sess->panel_button)
      gtk_label_set_text (GTK_LABEL (GTK_BIN (sess->panel_button)->child), "<none>");
#endif
   gui_set_title (sess);
   clear_user_list (sess);
   if (sess->flag_wid[0])
   {
      int i;
      for (i = 0; i < 8; i++)
         my_gtk_toggle_state (sess->flag_wid[i], FALSE, chan_flags[i], sess);
      my_gtk_entry_set_text (sess->limit_entry, "", sess);
      my_gtk_entry_set_text (sess->key_entry, "", sess);
   }
   if (sess->op_xpm)
      gtk_widget_destroy (sess->op_xpm);
   sess->op_xpm = 0;
   if (sess->logfd != -1)
   {
      end_logging (sess->logfd);
	sess->logfd = -1;
   }
}

void
gui_change_nick (struct server *serv, char *newnick)
{
   GSList *list = sess_list;
   struct session *sess;
   strcpy (serv->nick, newnick);
   while (list)
   {
      sess = (struct session *) list->data;
      if (sess->server == serv && !sess->is_dialog)
         gtk_label_set_text (GTK_LABEL (sess->nickgad), newnick);
      list = list->next;
   }
}

void
handle_topicgad (GtkWidget * igad, struct session *sess)
{
   char *topic = gtk_entry_get_text (GTK_ENTRY (igad));
   char tbuf[300];
   if (sess->channel[0] && sess->server->connected)
   {
      snprintf (tbuf, sizeof tbuf, "TOPIC %s :%s\r\n", sess->channel, topic);
      tcp_send (sess->server, tbuf);
   } else
      gtk_entry_set_text (GTK_ENTRY (igad), "");
}

char *
find_selected_nick (struct session *sess)
{
   int row;
   struct user *user = sess->userlist;

   row = gtkutil_clist_selection (sess->namelistgad);
   if (row == -1)
      return 0;

   user = gtk_clist_get_row_data (GTK_CLIST (sess->namelistgad), row);
   if (!user)
      return 0;
   return user->user;
}

void
ul_button_rel (GtkWidget * widget, GdkEventButton * even, struct session *sess)
{
   struct user *user;
   char *nick;
   int row, col;
   char buf[67];

   if (!even)
      return;
   if (even->button == 3)
   {
      if (gtk_clist_get_selection_info (GTK_CLIST (widget), even->x, even->y, &row, &col) < 0)
         return;
	gtk_clist_unselect_all (GTK_CLIST (widget));
	gtk_clist_select_row (GTK_CLIST (widget), row, 0);
      nick = find_selected_nick (sess);
      if (nick)
      {
         user = find_name (sess, nick);
         if (user)
            menu_popup (sess, even, nick, user->hostname);
         else
            menu_popup (sess, even, nick, 0);
      }
      return;
   }
   if (even->button == 2)
   {
      if (gtk_clist_get_selection_info (GTK_CLIST (widget), even->x, even->y, &row, &col) != -1)
      {
         gtk_clist_select_row (GTK_CLIST (widget), row, col);
         nick = find_selected_nick (sess);
         if (nick)
         {
            snprintf (buf, sizeof (buf), "%s: ", nick);
            gtk_entry_set_text (GTK_ENTRY (sess->inputgad), buf);
            gtk_widget_grab_focus (sess->inputgad);
         }
      }
   }
}

void
add_tip (GtkWidget * wid, char *text)
{
   GtkTooltips *tip = gtk_tooltips_new ();
   gtk_tooltips_set_tip (tip, wid, text, 0);
}

void
focus_in (GtkWindow * win, GtkWidget * wid, struct session *sess)
{
   if (!sess)
   {
      if (current_tab)
      {
         if (!current_tab->is_shell)
            gtk_widget_grab_focus (current_tab->inputgad);
         else
            gtk_widget_grab_focus (current_tab->textgad);
         menu_sess = current_tab;
	   if (!prefs.use_server_tab)
      	current_tab->server->front_session = current_tab;
      }
   } else
   {
	if (!prefs.use_server_tab)
   	   sess->server->front_session = sess;
      if (!sess->is_shell)
         gtk_widget_grab_focus (sess->inputgad);
      else
         gtk_widget_grab_focus (sess->textgad);
      menu_sess = sess;
#ifdef USE_PANEL
      if (sess->panel_button)
         gtk_widget_set_style (GTK_BIN (sess->panel_button)->child, normaltab_style);
#endif
   }
}

int
check_is_number (char *t)
{
   int len = strlen (t), c;

   for (c = 0; c < len; c++)
   {
      if (t[c] < '0' || t[c] > '9')
         return FALSE;
   }
   return TRUE;
}

void
change_channel_flag (GtkWidget * wid, struct session *sess, char flag)
{
   if (sess->server->connected && sess->channel[0])
   {
      char outbuf[512];
      if (GTK_TOGGLE_BUTTON (wid)->active)
         sprintf (outbuf, "MODE %s +%c\r\n", sess->channel, flag);
      else
         sprintf (outbuf, "MODE %s -%c\r\n", sess->channel, flag);
      tcp_send (sess->server, outbuf);
      sprintf (outbuf, "MODE %s\r\n", sess->channel);
      tcp_send (sess->server, outbuf);
      sess->ignore_mode = TRUE;
      sess->ignore_date = TRUE;
   }
}

void
flagt_hit (GtkWidget * wid, struct session *sess)
{
   change_channel_flag (wid, sess, 't');
}

void
flagn_hit (GtkWidget * wid, struct session *sess)
{
   change_channel_flag (wid, sess, 'n');
}

void
flags_hit (GtkWidget * wid, struct session *sess)
{
   change_channel_flag (wid, sess, 's');
}

void
flagi_hit (GtkWidget * wid, struct session *sess)
{
   change_channel_flag (wid, sess, 'i');
}

void
flagp_hit (GtkWidget * wid, struct session *sess)
{
   change_channel_flag (wid, sess, 'p');
}

void
flagm_hit (GtkWidget * wid, struct session *sess)
{
   change_channel_flag (wid, sess, 'm');
}

void
flagl_hit (GtkWidget * wid, struct session *sess)
{
   if (GTK_TOGGLE_BUTTON (wid)->active)
   {
      if (sess->server->connected && sess->channel[0])
      {
         char outbuf[512];
         if (check_is_number (gtk_entry_get_text (GTK_ENTRY (sess->limit_entry))) == FALSE)
         {
            gtkutil_simpledialog ("User limit must be a number!\n");
            gtk_entry_set_text (GTK_ENTRY (sess->limit_entry), "");
            gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (wid), FALSE);
            return;
         }
         sprintf (outbuf, "MODE %s +l %d\r\n", sess->channel, atoi (gtk_entry_get_text (GTK_ENTRY (sess->limit_entry))));
         tcp_send (sess->server, outbuf);
         sprintf (outbuf, "MODE %s\r\n", sess->channel);
         tcp_send (sess->server, outbuf);
      }
   } else
      change_channel_flag (wid, sess, 'l');
}


void
flagk_hit (GtkWidget * wid, struct session *sess)
{
   char outbuf[512];

   if (GTK_TOGGLE_BUTTON (wid)->active)
   {
      if (sess->server->connected && sess->channel[0])
      {
         snprintf (outbuf, 512, "MODE %s +k %s\r\n", sess->channel, gtk_entry_get_text (GTK_ENTRY (sess->key_entry)));
         tcp_send (sess->server, outbuf);
         snprintf (outbuf, 512, "MODE %s\r\n", sess->channel);
         tcp_send (sess->server, outbuf);
      }
   } else
   {
      if (sess->server->connected && sess->channel[0])
      {
         snprintf (outbuf, 512, "MODE %s -k %s\r\n", sess->channel, gtk_entry_get_text (GTK_ENTRY (sess->key_entry)));
         tcp_send (sess->server, outbuf);
         snprintf (outbuf, 512, "MODE %s\r\n", sess->channel);
         tcp_send (sess->server, outbuf);
      }
   }
}

void
key_entry (GtkWidget * igad, struct session *sess)
{
   if (sess->server->connected && sess->channel[0])
   {
      char outbuf[512];
      sprintf (outbuf, "MODE %s +k %s\r\n", sess->channel, gtk_entry_get_text (GTK_ENTRY (igad)));
      tcp_send (sess->server, outbuf);
      sprintf (outbuf, "MODE %s\r\n", sess->channel);
      tcp_send (sess->server, outbuf);
   }
}

void
limit_entry (GtkWidget * igad, struct session *sess)
{
   char outbuf[512];
   if (sess->server->connected && sess->channel[0])
   {
      if (check_is_number (gtk_entry_get_text (GTK_ENTRY (igad))) == FALSE)
      {
         gtk_entry_set_text (GTK_ENTRY (igad), "");
         gtkutil_simpledialog ("User limit must be a number!\n");
         gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (sess->flag_l), FALSE);
         return;
      }
      sprintf (outbuf, "MODE %s +l %d\r\n", sess->channel, atoi (gtk_entry_get_text (GTK_ENTRY (igad))));
      tcp_send (sess->server, outbuf);
      sprintf (outbuf, "MODE %s\r\n", sess->channel);
      tcp_send (sess->server, outbuf);
   }
}

void
show_and_unfocus (GtkWidget * wid)
{
   GTK_WIDGET_UNSET_FLAGS (wid, GTK_CAN_FOCUS);
   gtk_widget_show (wid);
}

void
add_flag_wid (GtkWidget ** wid, char *tip, GtkWidget * box, char *face,
    void *callback, gpointer userdata)
{
   *wid = gtk_toggle_button_new_with_label (face);
   gtk_widget_set_usize (*wid, 18, -1);
   add_tip (*wid, tip);
   gtk_box_pack_end (GTK_BOX (box), *wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (*wid), "toggled",
                       GTK_SIGNAL_FUNC (callback), userdata);
   show_and_unfocus (*wid);
}

void
userlist_button (GtkWidget * box, char *label, char *cmd,
                 struct session *sess, int a, int b, int c, int d)
{
   GtkWidget *wid = gtk_button_new_with_label (label);
   gtk_object_set_user_data (GTK_OBJECT (wid), sess);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
    GTK_SIGNAL_FUNC (popup_cmd), cmd);
   gtk_table_attach_defaults (GTK_TABLE (box), wid, a, b, c, d);
   show_and_unfocus (wid);
}


#ifdef USE_GNOME

void
zvt_clear (GtkWidget * wid)
{
   zvt_term_set_scrollback ((ZvtTerm *) wid, 0);
   zvt_term_set_scrollback ((ZvtTerm *) wid, prefs.zvt_lines);
   zvt_term_feed ((ZvtTerm *) wid, "\e[2J\e[1H", 8);
}

void
zvt_set_style (GtkWidget * zvt, char *bg_pic, int trans, int tint,
               GdkFont * norm_font, GdkFont * bold_font)
{
   zvt_term_set_fonts ((ZvtTerm *) zvt, norm_font, bold_font);
   if (trans)
      zvt_term_set_background ((ZvtTerm *) zvt, 0, TRUE, tint);
   else
   {
      if (bg_pic[0])
      {
         if (access (bg_pic, R_OK) == 0)
         {
            zvt_term_set_background ((ZvtTerm *) zvt, bg_pic, FALSE, FALSE);
         } else
         {
            char buf[256];
            sprintf (buf, "Cannot access %s", bg_pic);
            gtkutil_simpledialog (buf);
         }
      } else
         zvt_term_set_background ((ZvtTerm *) zvt, 0, FALSE, FALSE);
   }
}

#endif

GtkStyle *
my_widget_get_style (char *bg_pic)
{
   char buf[256];
   GtkStyle *style;
   GdkPixmap *pixmap;
#ifdef USE_IMLIB
   GdkImlibImage *img;
#endif

   style = gtk_style_new ();

   style->base[GTK_STATE_NORMAL] = colors[1];
   style->bg[GTK_STATE_NORMAL] = colors[1];

   if (bg_pic[0])
   {
      if (access (bg_pic, R_OK) == 0)
      {
#ifdef USE_IMLIB
         img = gdk_imlib_load_image (bg_pic);
         if (img)
         {
            gdk_imlib_render (img, img->rgb_width, img->rgb_height);
            pixmap = gdk_imlib_move_image (img);
            gdk_imlib_destroy_image (img);
            if (pixmap)
               style->bg_pixmap[GTK_STATE_NORMAL] = pixmap;
         }
#else
         pixmap = gdk_pixmap_create_from_xpm (0, 0,
                                              &style->bg[GTK_STATE_NORMAL], bg_pic);
         if (pixmap)
            style->bg_pixmap[0] = pixmap;
#endif
      } else
      {
         snprintf (buf, sizeof buf, "Cannot access %s", bg_pic);
         gtkutil_simpledialog (buf);
      }
   }
   return style;
}

int
textgad_get_focus_cb (GtkWidget * wid, GdkEventKey * event, struct session *sess)
{
   char text[2] = " ";

   if (event->keyval >= GDK_space && event->keyval <= GDK_asciitilde)
   {
      text[0] = event->keyval;
      gtk_entry_append_text (GTK_ENTRY (sess->inputgad), text);
      gtk_widget_grab_focus (sess->inputgad);
      return FALSE;
   } else if (event->keyval == GDK_BackSpace)
   {
      gtk_widget_grab_focus (sess->inputgad);
      return FALSE;
   }
   return TRUE;
}

#ifdef USE_GNOME

gushort zvt_red[18];
gushort zvt_grn[18];
gushort zvt_blu[18];            /* these are filled in palette.c palette_prepare() */

void
term_change_pos (GtkWidget * widget)
{
   gtk_widget_queue_draw (widget);
}

#endif

static int
is_delimiter (char c)
{
   switch (c)
   {
   case ' ':
   case '|':
   case '[':
   case ']':
   case '(':
   case ')':
   case '<':
   case '>':
   case '\0':
   case '\n':
   case '"':
      return 1;
   }
   return 0;
}

/* mouse click inside text area */

static void
maingui_word_clicked (struct session *sess, GdkEventButton * even, char *word)
{
   struct user *user;
   int url = 0;

   if (even && word[0] && (word[0] != ' ' && word[1] != 0))
   {
      if (even->button == 3)
      {
	   if(word[0] == '#')
	   {
		menu_chanmenu (sess, even, word);
		return;
	   }
         user = find_name (sess, word);
         if (user)              /* see if this word is in the userlist */
         {
            menu_popup (sess, even, word, user->hostname);
            return;
         } else
         {
            if (!strncasecmp (word, "ftp.", 4))
               url = 1;
            if (!strncasecmp (word, "ftp:", 4))
               url = 1;
            if (!strncasecmp (word, "www.", 4))
               url = 1;
            if (!strncasecmp (word, "http:", 5))
               url = 1;
            if (!url)
            {
               char *at = strchr (word, '@');  /* check for email addy */
               char *dot = strrchr (word, '.');
               if (at && dot)
               {
                  if ((unsigned long) at < (unsigned long) dot)
                  {
                     char *newword = malloc (strlen (word) + 10);
                     if (*word == '~')
                        word++;
                     sprintf (newword, "mailto:%s", word);
                     menu_urlmenu (sess, even, newword);
                     free (newword);
                     return;
                  }
               }
            }
            if (url)
            {
               menu_urlmenu (sess, even, word);
               return;
            } else
            {
               char *dot = strrchr (word, '.');
               if (dot)
               {
                  dot++;
                  if (!strcasecmp (dot, "org"))
                     url = 1;
                  else if (!strcasecmp (dot, "com"))
                     url = 1;
                  else if (!strcasecmp (dot, "net"))
                     url = 1;
                  else if (!strcasecmp (dot, "edu"))
                     url = 1;
			else if (word[strlen(word)-3] == '.')
			   url = 1;
                  if (url)
                  {
                     menu_hostmenu (sess, even, word);
                     return;
                  }
               }
               {
                  int i, dots = 0;  /* check if it's an IP number */

                  for (i = 0; i < strlen (word); i++)
                  {
                     if (word[i] == '.')
                        dots++;
                  }
                  if (dots == 3)
                  {
                     if (inet_addr (word) != -1)
                     {
                        menu_hostmenu (sess, even, word);
                        return;
                     }
                  }
               }
            }
         }
      } else if (even->button == 4)
      {
         GtkAdjustment *adj;
         gfloat new_value;
         adj = gtk_range_get_adjustment (GTK_RANGE (sess->vscrollbar));
         new_value = adj->value - adj->page_increment;
         if (new_value < adj->lower)
            new_value = adj->lower;
         gtk_adjustment_set_value (adj, new_value);
      } else if (even->button == 5)
      {
         GtkAdjustment *adj;
         gfloat new_value;
         adj = gtk_range_get_adjustment (GTK_RANGE (sess->vscrollbar));
         new_value = adj->value + adj->page_increment;
         if (new_value > (adj->upper - adj->page_size))
            new_value = adj->upper - adj->page_size;
         gtk_adjustment_set_value (adj, new_value);
      }
   }
   if (even && even->button == 3 && sess->is_dialog)
   {
      menu_popup (sess, even, sess->channel, gtk_entry_get_text (GTK_ENTRY (sess->topicgad)));
   }
}

#ifdef USE_GNOME

gint
maingui_textgad_clicked_zvt (GtkWidget * wid, GdkEventButton * even, struct session *sess)
{
   char *text, *text2, *wordstart, *word;
   int font_x, font_y, len, len2, pos, end, cur;
   GtkAdjustment *adj;

   if (even->button == 3)
      gtk_signal_emit_stop_by_name (GTK_OBJECT (wid), "button_press_event");

   adj = ((ZvtTerm *) sess->textgad)->adjustment;
   end = adj->upper - adj->lower - adj->page_size;
   cur = adj->value + (adj->page_size / 2);
   if (cur < end)
   {
      /* we're not on the last page, so forget about it. */
      /* FIXME: maybe use the adjustment to offset font_y negatively?? */
      return TRUE;
   }
   font_x = even->x / ((ZvtTerm *) sess->textgad)->charwidth;
   font_y = even->y / ((ZvtTerm *) sess->textgad)->charheight;

   text = zvt_term_get_buffer ((ZvtTerm *) sess->textgad, &len, VT_SELTYPE_CHAR, 0,
                               font_y, ((ZvtTerm *) sess->textgad)->grid_width, font_y);

   if (!text || !len)
      return TRUE;

   /* let's get the next line too, and join them, incase the url is
      split across 2 lines. */
   text2 = zvt_term_get_buffer ((ZvtTerm *) sess->textgad, &len2, VT_SELTYPE_CHAR, 0,
                                font_y + 1, ((ZvtTerm *) sess->textgad)->grid_width - 1, font_y + 1);

   if (text2 && len2)
   {
      char *newtext = g_malloc (len + len2 + 2);
      memcpy (newtext, text, len);
      memcpy (newtext + len, text2, len2);
      newtext[len + len2] = 0;
      g_free (text);
      g_free (text2);
      text = newtext;
   }
   pos = font_x;
   while (!is_delimiter (text[pos]))
      pos--;

   pos++;
   wordstart = &text[pos];

   len = 0;
   while (!is_delimiter (text[pos]))
   {
      pos++;
      len++;
   }

   word = malloc (len + 1);
   memcpy (word, wordstart, len);
   word[len] = 0;
   g_free (text);

   maingui_word_clicked (sess, even, word);

   free (word);

   return TRUE;
}

#endif

gint
maingui_textgad_clicked_gtk (GtkWidget * wid, GdkEventButton * even, struct session *sess)
{
   char *text, *word, *wordstart;
   int len, pos, from, to;

   if (even->button == 3)
      gtk_signal_emit_stop_by_name (GTK_OBJECT (wid), "button_press_event");

   len = gtk_text_get_length (GTK_TEXT (sess->textgad));
   pos = gtk_editable_get_position (GTK_EDITABLE (sess->textgad));

   if (pos >= len)
   {
	if(sess->is_dialog)
   	   maingui_word_clicked (sess, even, "");
      return TRUE;
   }

   if (pos < 100)
      from = 0;
   else
      from = pos - 100;

   if (pos + 100 > len)
      to = len;
   else
      to = pos + 100;

   text = gtk_editable_get_chars (GTK_EDITABLE (sess->textgad), from, to);

   if (pos > 100)
      pos = 100;

   while (!is_delimiter (text[pos]))
      pos--;

   pos++;
   wordstart = &text[pos];

   len = 0;
   while (!is_delimiter (text[pos]))
   {
      pos++;
      len++;
   }

   word = malloc (len + 1);
   memcpy (word, wordstart, len);
   word[len] = 0;
   g_free (text);

   maingui_word_clicked (sess, even, word);

   free (word);

   return TRUE;
}


void
maingui_create_textlist (struct session *sess, GtkWidget * leftpane)
{
#ifdef USE_GNOME
   if (prefs.zvt)
   {
      sess->zvt = TRUE;
      sess->textgad = zvt_term_new ();
      if (sess->is_tab)
         gtk_signal_connect_object (GTK_OBJECT (main_window), "configure_event",
                                    GTK_SIGNAL_FUNC (term_change_pos),
          GTK_OBJECT (sess->textgad));
      else
         gtk_signal_connect_object (GTK_OBJECT (sess->window), "configure_event",
                                    GTK_SIGNAL_FUNC (term_change_pos),
          GTK_OBJECT (sess->textgad));
      gtk_widget_set_usize (sess->textgad,
         prefs.mainwindow_width - 115,
       prefs.mainwindow_height - 105);
      gtk_signal_connect (GTK_OBJECT (sess->textgad), "key_press_event",
                          GTK_SIGNAL_FUNC (textgad_get_focus_cb), sess);
      gtk_signal_connect (GTK_OBJECT (sess->textgad), "button_press_event",
                          GTK_SIGNAL_FUNC (maingui_textgad_clicked_zvt), sess);
      zvt_term_set_scrollback ((ZvtTerm *) sess->textgad, prefs.zvt_lines);
	zvt_term_set_blink ((ZvtTerm *) sess->textgad, FALSE);
      zvt_set_style (sess->textgad, prefs.background, prefs.transparent,
                     prefs.tint, font_normal, font_bold);
      gtk_container_add (GTK_CONTAINER (leftpane), sess->textgad);
      gtk_widget_show (sess->textgad);

      sess->vscrollbar = gtk_vscrollbar_new (((ZvtTerm *) sess->textgad)->adjustment);
      gtk_box_pack_start (GTK_BOX (leftpane), sess->vscrollbar, FALSE, FALSE, 1);
      show_and_unfocus (sess->vscrollbar);
   } else
   {
#endif
      if (!sess->is_tab)
         gtk_widget_realize (sess->window);
      sess->textgad = gtk_text_new (0, 0);
      gtk_widget_set_style (sess->textgad, channelwin_style);
      gtk_widget_set_usize (sess->textgad, prefs.mainwindow_width - 115,
							 prefs.mainwindow_height - 105);
      gtk_signal_connect (GTK_OBJECT (sess->textgad), "key_press_event",
                          GTK_SIGNAL_FUNC (textgad_get_focus_cb), sess);
      gtk_signal_connect (GTK_OBJECT (sess->textgad), "button_press_event",
                          GTK_SIGNAL_FUNC (maingui_textgad_clicked_gtk), sess);
      gtk_text_set_word_wrap (GTK_TEXT (sess->textgad), TRUE);
      gtk_container_add (GTK_CONTAINER (leftpane), sess->textgad);
      gtk_widget_show (sess->textgad);

      sess->vscrollbar = gtk_vscrollbar_new (GTK_TEXT (sess->textgad)->vadj);
      gtk_box_pack_start (GTK_BOX (leftpane), sess->vscrollbar, FALSE, FALSE, 1);
      show_and_unfocus (sess->vscrollbar);
#ifdef USE_GNOME
   }
#endif
}

void
gui_new_tab (struct session *sess)
{
   current_tab = sess;
   menu_sess = sess;
   if (!prefs.use_server_tab)
	sess->server->front_session = sess;
   gui_set_title (sess);
   if (!sess->is_shell)
	gtk_widget_grab_focus (sess->inputgad);

#ifdef USE_PANEL
   if (sess->panel_button)
      gtk_widget_set_style (GTK_BIN (sess->panel_button)->child, normaltab_style);
#endif
   if (sess->new_data || sess->nick_said)
   {
      sess->nick_said = FALSE;
      sess->new_data = FALSE;
      gtk_widget_set_style (sess->changad, normaltab_style);
   }
}

void
gui_new_tab_callback (GtkWidget * widget, GtkNotebookPage * nbpage, guint page)
{
   struct session *sess;
   GSList *list = sess_list;

   if (xchat_is_quitting)
      return;

   while (list)
   {
      sess = (struct session *) list->data;
      if (sess->window == nbpage->child)
      {
         gui_new_tab (sess);
         return;
      }
      list = list->next;
   }
}

int
maingui_pagetofront (int page)
{
     gtk_notebook_set_page (GTK_NOTEBOOK (main_book), page);
     return 0;
}

void
gui_mainbook_invalid (GtkWidget * w, GtkWidget * main_window)
{
   main_book = NULL;
   gtk_widget_destroy (main_window);
}

void
gui_main_window_kill (GtkWidget * win, struct session *sess)
{
   if (!sess_list)
      xchat_cleanup ();
   main_window = 0;
   current_tab = 0;
}

void
userlist_hide (GtkWidget * igad, struct session *sess)
{
#ifdef USE_GNOME
   if (sess->userlisthidden)
   {
      if (sess->paned)
         gtk_paned_set_position (GTK_PANED (sess->paned), sess->userlisthidden);
      else
         gtk_widget_show (sess->userlistbox);
      sess->userlisthidden = FALSE;
   } else
   {
      if (sess->paned)
      {
         sess->userlisthidden = GTK_PANED (sess->paned)->handle_xpos;
         gtk_paned_set_position (GTK_PANED (sess->paned), 1200);
      } else
      {
         sess->userlisthidden = TRUE;
         gtk_widget_hide (sess->userlistbox);
      }
   }
#else
   if (sess->userlisthidden)
   {
      gtk_label_set (GTK_LABEL (GTK_BIN (igad)->child), ">");
      if (sess->paned)
         gtk_paned_set_position (GTK_PANED (sess->paned), sess->userlisthidden);
      else
         gtk_widget_show (sess->userlistbox);
      sess->userlisthidden = FALSE;
   } else
   {
      gtk_label_set (GTK_LABEL (GTK_BIN (igad)->child), "<");
      if (sess->paned)
      {
         sess->userlisthidden = GTK_PANED (sess->paned)->handle_xpos;
         gtk_paned_set_position (GTK_PANED (sess->paned), 1200);
      } else
      {
         sess->userlisthidden = TRUE;
         gtk_widget_hide (sess->userlistbox);
      }
   }
#endif
}

void
maingui_userlist_selected (GtkWidget * clist, gint row, gint column,
                GdkEventButton * even)
{
   char *nick;

   if (even)
   {
	if (even->type == GDK_2BUTTON_PRESS)
	{
   	   if (prefs.doubleclickuser[0])
   	   {

      	if (gtk_clist_get_cell_type (GTK_CLIST (clist), row, column) == GTK_CELL_PIXTEXT)
      	{
         	   void *unused;
         	   gtk_clist_get_pixtext (GTK_CLIST (clist), row, column, &nick, (guint8 *) & unused, (GdkPixmap **) & unused, (GdkBitmap **) & unused);
      	} else
         	   gtk_clist_get_text (GTK_CLIST (clist), row, column, &nick);
      	popup_eval_command (menu_sess, prefs.doubleclickuser, nick, nick);
   	   }
	} else
	{
	   if(!(even->state & GDK_SHIFT_MASK))
	   {
		gtk_clist_unselect_all (GTK_CLIST (clist));
		gtk_clist_select_row (GTK_CLIST (clist), row, column);
	   }
	}
   }
}

void
gui_make_tab_window (struct session *sess)
{
   GtkWidget *main_box;
#ifndef USE_GNOME
   GtkWidget *wid;
#endif

   if (!main_window)
   {
      current_tab = 0;
#ifdef USE_GNOME
      main_window = gnome_app_new ("X-Chat", "X-Chat");
#else
      main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
#endif
      gtk_widget_realize (main_window);
      gtk_signal_connect ((GtkObject *) main_window, "destroy",
                          GTK_SIGNAL_FUNC (gui_main_window_kill), sess);
      gtk_signal_connect ((GtkObject *) main_window, "focus_in_event",
       GTK_SIGNAL_FUNC (focus_in), 0);
      gtk_window_set_policy ((GtkWindow *) main_window, TRUE, TRUE, FALSE);

      main_box = gtk_vbox_new (0, 0);
#ifdef USE_GNOME
      gnome_app_set_contents (GNOME_APP (main_window), main_box);
#else
      gtk_container_add (GTK_CONTAINER (main_window), main_box);
#endif
      gtk_widget_show (main_box);

#ifdef USE_GNOME
      createmenus (main_window, sess);
#else
      wid = createmenus (sess);
      gtk_box_pack_start (GTK_BOX (main_box), wid, FALSE, TRUE, 0);
      gtk_widget_show (wid);
#endif
      main_book = gtk_notebook_new ();
      if (prefs.tabsatbottom)
         gtk_notebook_set_tab_pos ((GtkNotebook *) main_book, GTK_POS_BOTTOM);
      gtk_notebook_set_scrollable ((GtkNotebook *) main_book, TRUE);
      gtk_signal_connect ((GtkObject *) main_book, "switch_page",
                          GTK_SIGNAL_FUNC (gui_new_tab_callback), 0);
      gtk_signal_connect ((GtkObject *) main_book, "destroy",
                          GTK_SIGNAL_FUNC (gui_mainbook_invalid), main_window);
      gtk_container_add (GTK_CONTAINER (main_box), main_book);
      gtk_widget_show (main_book);
   }
}

/* this shouldn't be necessary, but if we don't, the scrollbar disappears.
 * So let's kill the old scrolled window and create a new one - HACK! */

void
scrollbar_hack (struct session *sess)
{
   GtkWidget *win, *box, *oldwin;

   if (sess->is_dialog)
      return;

   oldwin = sess->namelistgad->parent;  /* the old scrolled window */
   box = oldwin->parent;

   win = gtk_scrolled_window_new (0, 0);
   gtk_container_add (GTK_CONTAINER (box), win);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (win),
                 GTK_POLICY_AUTOMATIC,
                GTK_POLICY_AUTOMATIC);
   gtk_widget_show (win);

   gtk_widget_reparent (sess->namelistgad, win);
   gtk_widget_destroy (oldwin);
   gtk_widget_set_usize (sess->namelistgad->parent, 115, 0);
}

/* This function takes a session and toggles it between a tab and a free window */
int
relink_window (GtkWidget * w, struct session *sess)
{
   GtkWidget *old, *wid;
   int num, need = 0;

   if (sess->is_tab)
   {
      /* It is currently a tab so we need to make a window */

      /* Warning! Hack alert!, since we need to destroy the main_book on the last page,
         I get the current page. If it is 0 (the first page) then I
         try to get page 1. If that doesn't exist I assume it is the only page.
         It works with GTK 1.2.0 but GTK could change in the future. WATCH OUT! 
         -- AGL (9/4/99) */

      num = gtk_notebook_get_current_page (GTK_NOTEBOOK (main_book));
      if (num == 0)
      {
         if (gtk_notebook_get_nth_page (GTK_NOTEBOOK (main_book), 1) == NULL)
            need = TRUE;
      }
      sess->is_tab = 0;
      old = sess->window;
      gtk_signal_disconnect_by_data (GTK_OBJECT (old), sess);
#ifdef USE_GNOME
      sess->window = gnome_app_new ("X-Chat", "X-Chat");
#else
      sess->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
#endif
      if (!sess->is_dialog)
      {
#ifdef USE_GNOME
         createmenus (GNOME_APP (sess->window), sess);
         sess->menu = GNOME_APP (sess->window)->menubar;
#else
         wid = createmenus (sess);
         gtk_box_pack_start (GTK_BOX (sess->vbox), wid, FALSE, FALSE, 0);
         gtk_box_reorder_child (GTK_BOX (sess->vbox), wid, 0);
         gtk_widget_show (wid);
         sess->menu = wid;
#endif
      }
      gui_set_title (sess);
      gtk_signal_connect ((GtkObject *) sess->window, "destroy",
                          GTK_SIGNAL_FUNC (kill_session_callback), sess);
      gtk_signal_connect ((GtkObject *) sess->window, "focus_in_event",
                          GTK_SIGNAL_FUNC (focus_in), sess);
      gtk_window_set_policy ((GtkWindow *) sess->window, TRUE, TRUE, FALSE);
#ifdef USE_GNOME
      wid = gtk_hbox_new (0, 0);
      gtk_widget_reparent (sess->vbox, wid);
      gnome_app_set_contents (GNOME_APP (sess->window), wid);
#else
      gtk_widget_reparent (sess->vbox, sess->window);
#endif
      if (!sess->is_dialog)
      {
         if (sess->channel[0] == 0)
            sess->changad = gtk_label_new ("<none>");
         else
            sess->changad = gtk_label_new (sess->channel);
         gtk_box_pack_start (GTK_BOX (sess->tbox), sess->changad, FALSE, FALSE, 5);
         gtk_box_reorder_child (GTK_BOX (sess->tbox), sess->changad, 2);
         gtk_widget_show (sess->changad);
      } else
         sess->changad = NULL;

      scrollbar_hack (sess);

      gtk_widget_show_all (sess->window);
      num = gtk_notebook_page_num (GTK_NOTEBOOK (main_book), old);
      gtk_notebook_remove_page (GTK_NOTEBOOK (main_book), num);
      if (prefs.mainwindow_left || prefs.mainwindow_top)
         gdk_window_move (sess->window->window,
                prefs.mainwindow_left,
                prefs.mainwindow_top);
#ifdef USE_GNOME
      if (sess->zvt)
      {
         zvt_term_set_color_scheme ((ZvtTerm *) sess->textgad, zvt_red, zvt_grn, zvt_blu);
         gtk_widget_hide (sess->textgad);
         gtk_widget_show (sess->textgad);
      }
#endif
      if (need)
         gtk_widget_destroy (main_book);
      return 0;
   } else
   {
      /* We have a free window so we need to make it a tab */
      if (main_book == NULL)
      {
         /* Oh dear, we don't *have* a main_book, so we need to make one */
         gui_make_tab_window (sess);
         need = TRUE;
      }
      if (sess->menu)
      {
         gtk_widget_destroy (sess->menu);
         sess->menu = NULL;
      }
      sess->is_tab = 1;
      old = sess->window;
      gtk_signal_disconnect_by_data (GTK_OBJECT (old), sess);
      sess->window = gtk_hbox_new (0, 0);
      gtk_signal_connect ((GtkObject *) sess->window, "destroy",
                          GTK_SIGNAL_FUNC (kill_session_callback), sess);
      gtk_widget_reparent (sess->vbox, sess->window);
      gtk_widget_destroy (old);

      /* FIXME: Erm, I think gtk_notebook_remove_page will destroy the changad,
         if not then we have a memory leak. Could someoue check this?
         -- AGL (9/4/99) */

      if (sess->changad != NULL)
         gtk_widget_destroy (sess->changad);
      if (sess->channel[0] == 0)
         sess->changad = gtk_label_new ("<none>");
      else
         sess->changad = gtk_label_new (sess->channel);

      scrollbar_hack (sess);

      gtk_widget_show (sess->changad);
      gtk_notebook_append_page (GTK_NOTEBOOK (main_book), sess->window, sess->changad);
      gtk_widget_show_all (sess->window);
      if (need)
         gtk_widget_show_all (main_window);
#ifdef USE_GNOME
      if (sess->zvt)
         zvt_term_set_color_scheme ((ZvtTerm *) sess->textgad, zvt_red, zvt_grn, zvt_blu);
#endif
      return 0;
   }
}

#ifdef USE_GNOME
static GtkTargetEntry dnd_targets[] =
{
   {"text/uri-list", 0, 1}
};
#endif

#ifdef USE_PANEL

static void
gui_panel_destroy_popup (GtkWidget * wid, GtkWidget * popup)
{
   gtk_widget_destroy (panel_popup);
   panel_popup = NULL;
}

static void
gui_panel_remove_clicked (GtkWidget * button, struct session *sess)
{
   gtk_widget_show (sess->window);
   gtk_widget_destroy (sess->panel_button);
   sess->panel_button = 0;
   if (main_window)             /* this fixes a little refresh glitch */
   {
      int page;

      page = gtk_notebook_page_num (GTK_NOTEBOOK (main_book), sess->window);
	gtk_idle_add ((GtkFunction)maingui_pagetofront, (gpointer)page);
   }
   if (panel_popup)
      gui_panel_destroy_popup (NULL, NULL);
}

static void
gui_panel_hide_clicked (GtkWidget * button, struct session *sess)
{
   gtk_widget_hide (sess->window);
}

static void
gui_panel_show_clicked (GtkWidget * button, struct session *sess)
{
   gtk_widget_show (sess->window);
}

static void
gui_panel_here_clicked (GtkWidget * button, struct session *sess)
{
   if (sess->is_tab)
   {
      if (main_window)
      {
         gtk_widget_hide (main_window);
         gtk_window_set_position (GTK_WINDOW (main_window), GTK_WIN_POS_MOUSE);
         gtk_widget_show (main_window);
      }
   } else
   {
      gtk_widget_hide (sess->window);
      gtk_window_set_position (GTK_WINDOW (sess->window), GTK_WIN_POS_MOUSE);
      gtk_widget_show (sess->window);
   }
}

static void
gui_panel_destroy_view_popup (GtkWidget * popup, /* Event */ gpointer * arg2, struct session *sess)
{
   /* BODGE ALERT !! BODGE ALERT !! --AGL */
   gtk_widget_reparent (sess->textgad, sess->leftpane);
   gtk_box_reorder_child (GTK_BOX (sess->leftpane), sess->textgad, 0);

   gtk_widget_destroy (popup);
}

static void
gui_panel_view_clicked (GtkWidget * button, struct session *sess)
{
   GtkWidget *view_popup;

   view_popup = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_widget_show_all (view_popup);
   gtk_widget_reparent (sess->textgad, view_popup);
#ifdef USE_GNOME
   if (sess->zvt)
      zvt_term_set_color_scheme ((ZvtTerm *) sess->textgad, zvt_red, zvt_grn, zvt_blu);
#endif
   gtk_signal_connect (GTK_OBJECT (view_popup), "leave_notify_event", gui_panel_destroy_view_popup, sess);
}

static gint
gui_panel_button_event (GtkWidget * button, GdkEvent * event, struct session *sess)
{
   if (event->type == GDK_BUTTON_PRESS &&
       event->button.button == 3)
   {
      GtkWidget *vbox, *wid;

      panel_popup = gtk_window_new (GTK_WINDOW_POPUP);
      gtk_window_set_position (GTK_WINDOW (panel_popup), GTK_WIN_POS_MOUSE);
      vbox = gtk_vbox_new (0, 0);
      gtk_container_add (GTK_CONTAINER (panel_popup), vbox);

      wid = gtk_label_new ("");
      if (sess->channel[0])
         gtk_label_set_text (GTK_LABEL (wid), sess->channel);
      else
         gtk_label_set_text (GTK_LABEL (wid), "No Channel");
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);

      wid = gtk_label_new ("");
      if (sess->server->hostname[0])
         gtk_label_set_text (GTK_LABEL (wid), sess->server->servername);
      else
         gtk_label_set_text (GTK_LABEL (wid), "No Server");
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);

      wid = gtk_label_new ("");
      if (sess->is_tab)
         gtk_label_set_text (GTK_LABEL (wid), "Is Tab");
      else
         gtk_label_set_text (GTK_LABEL (wid), "Is Not Tab");
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);

      wid = gtk_button_new_with_label ("Close");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked", GTK_SIGNAL_FUNC (gtkutil_destroy), sess->window);
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);
      wid = gtk_button_new_with_label ("Remove");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked", GTK_SIGNAL_FUNC (gui_panel_remove_clicked), sess);
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);
      wid = gtk_button_new_with_label ("Hide");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked", GTK_SIGNAL_FUNC (gui_panel_hide_clicked), sess);
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);
      wid = gtk_button_new_with_label ("Show");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked", GTK_SIGNAL_FUNC (gui_panel_show_clicked), sess);
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);
      wid = gtk_button_new_with_label ("De/Link");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked", GTK_SIGNAL_FUNC (relink_window), sess);
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);
      wid = gtk_button_new_with_label ("Move Here");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked", GTK_SIGNAL_FUNC (gui_panel_here_clicked), sess);
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);
      wid = gtk_button_new_with_label ("View");
      gtk_signal_connect (GTK_OBJECT (wid), "clicked", GTK_SIGNAL_FUNC (gui_panel_view_clicked), sess);
      gtk_box_pack_start (GTK_BOX (vbox), wid, 0, 0, 0);

      gtk_signal_connect (GTK_OBJECT (panel_popup), "leave_notify_event", gui_panel_destroy_popup, panel_popup);
      gtk_widget_show_all (panel_popup);

   }
   return 0;
}

void
maingui_unpanelize (GtkWidget * button, struct session *sess)
{
   if (!sess->is_tab)
      gtk_widget_set_style (GTK_BIN (sess->panel_button)->child, normaltab_style);
   if (prefs.panelize_hide)
   {
      gtk_container_remove (GTK_CONTAINER (button->parent), button);
      gtk_widget_destroy (button);
   }
   gtk_widget_show (sess->window);
   if (prefs.panelize_hide)
      sess->panel_button = 0;
   if (main_window)             /* this fixes a little refresh glitch */
   {
      int page;

      page = gtk_notebook_page_num (GTK_NOTEBOOK (main_book), sess->window);
	gtk_idle_add ((GtkFunction)maingui_pagetofront, (gpointer)page);
   }
}

void
maingui_panelize (GtkWidget * button, struct session *sess)
{
   char tbuf[128];

   if (!panel_applet)
      create_panel_widget ();

   if (sess->panel_button != NULL)
      return;

   if (prefs.panelize_hide)
      gtk_widget_hide (sess->window);

   if (sess->channel[0] == 0)
      button = gtk_button_new_with_label ("<none>");
   else
      button = gtk_button_new_with_label (sess->channel);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
                       GTK_SIGNAL_FUNC (maingui_unpanelize), sess);
   gtk_container_add (GTK_CONTAINER (panel_box), button);
   gtk_signal_connect (GTK_OBJECT (button), "button_press_event", GTK_SIGNAL_FUNC (gui_panel_button_event), sess);
   gtk_widget_show (button);
   sess->panel_button = button;

   if (sess->channel[0])
   {
      snprintf (tbuf, sizeof tbuf, "%s: %s", sess->server->servername, sess->channel);
      add_tip (button, tbuf);
   }
}

#endif

/* 'X' button pressed */

void
X_session (GtkWidget * button, struct session *sess)
{
   gtk_widget_destroy (sess->window);
}

void
maingui_init_styles (GtkStyle *style)
{
   normaltab_style = gtk_style_new ();
   normaltab_style->font = style->font;

   redtab_style = gtk_style_new ();
   redtab_style->font = style->font;
   memcpy (redtab_style->fg, &colors[4], sizeof (GdkColor));

   bluetab_style = gtk_style_new ();
   bluetab_style->font = style->font;
   memcpy (bluetab_style->fg, &colors[12], sizeof (GdkColor));
}

void
create_window (struct session *sess)
{
   GtkWidget *leftpane, *rightpane;
   GtkWidget *vbox, *tbox, *bbox, *nlbox, *wid;
   int justopened = FALSE;

   if (sess->is_dialog)
   {
      new_dialog (sess);
      return;
   }
   if (!sess->server->front_session)
      sess->server->front_session = sess;

   if (prefs.tabchannels)
   {
      sess->is_tab = TRUE;
      if (!main_window)
      {
         justopened = TRUE;
         gui_make_tab_window (sess);
      }
      sess->window = gtk_hbox_new (0, 0);
      gtk_signal_connect ((GtkObject *) sess->window, "destroy",
                          GTK_SIGNAL_FUNC (kill_session_callback), sess);
      if (!current_tab)
      {
         current_tab = sess;
         gui_set_title (sess);
      }
   } else
   {

#ifdef USE_GNOME
      sess->window = gnome_app_new ("X-Chat", "X-Chat");
#else
      sess->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
#endif
      gui_set_title (sess);
      gtk_signal_connect ((GtkObject *) sess->window, "destroy",
                          GTK_SIGNAL_FUNC (kill_session_callback), sess);
      gtk_signal_connect ((GtkObject *) sess->window, "focus_in_event",
                          GTK_SIGNAL_FUNC (focus_in), sess);
      gtk_window_set_policy ((GtkWindow *) sess->window, TRUE, TRUE, FALSE);
   }

   palette_alloc (sess->window);

   vbox = gtk_vbox_new (FALSE, 0);
   sess->vbox = vbox;
   gtk_container_set_border_width ((GtkContainer *) vbox, 2);
   if (!prefs.tabchannels)
   {
#ifdef USE_GNOME
      gnome_app_set_contents (GNOME_APP (sess->window), vbox);
#else
      gtk_container_add (GTK_CONTAINER (sess->window), vbox);
#endif
   } else
      gtk_container_add ((GtkContainer *) sess->window, vbox);
   gtk_widget_show (vbox);

   if (!prefs.tabchannels)
   {
#ifdef USE_GNOME
      createmenus (sess->window, sess);
      sess->menu = GNOME_APP (sess->window)->menubar;
#else
      wid = createmenus (sess);
      gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, TRUE, 0);
      gtk_widget_show (wid);
      sess->menu = wid;
#endif
   }
   tbox = gtk_hbox_new (FALSE, 0);
   sess->tbox = tbox;
   gtk_container_set_border_width (GTK_CONTAINER (tbox), 0);
   gtk_box_pack_start (GTK_BOX (vbox), tbox, FALSE, TRUE, 2);
   gtk_widget_show (tbox);

#ifdef USE_GNOME
   wid = gtkutil_button (sess->window, GNOME_STOCK_BUTTON_CANCEL,
               0, X_session, sess, 0);
   gtk_box_pack_start (GTK_BOX (tbox), wid, 0, 0, 0);
   gtk_widget_show (wid);
   add_tip (wid, "Close Channel");

   wid = gtkutil_button (sess->window, GNOME_STOCK_BUTTON_UP,
           0, relink_window, sess, 0);
   gtk_box_pack_start (GTK_BOX (tbox), wid, 0, 0, 0);
   gtk_widget_show (wid);
   add_tip (wid, "Link/DeLink this tab");
#else
   wid = gtk_button_new_with_label ("X");
   gtk_box_pack_start (GTK_BOX (tbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (X_session), sess);
   gtk_widget_show (wid);
   add_tip (wid, "Close Channel");

   wid = gtk_button_new_with_label ("^");
   gtk_box_pack_start (GTK_BOX (tbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (relink_window), sess);
   gtk_widget_show (wid);
   add_tip (wid, "Link/DeLink this tab");
#endif

#ifdef USE_PANEL
   if (!nopanel)
   {
      wid = gtkutil_button (sess->window, GNOME_STOCK_BUTTON_DOWN,
        0, maingui_panelize, sess, 0);
      gtk_box_pack_start (GTK_BOX (tbox), wid, 0, 0, 0);
      add_tip (wid, "Panelize");
   }
#endif

   if (!prefs.tabchannels)
   {
      sess->changad = gtk_label_new ("<none>");
      gtk_box_pack_start (GTK_BOX (tbox), sess->changad, FALSE, FALSE, 5);
      gtk_widget_show (sess->changad);
   }
   sess->topicgad = gtk_entry_new_with_max_length (255);
   gtk_signal_connect (GTK_OBJECT (sess->topicgad), "activate",
                       GTK_SIGNAL_FUNC (handle_topicgad), sess);
   gtk_container_add (GTK_CONTAINER (tbox), sess->topicgad);
   gtk_widget_show (sess->topicgad);

   add_tip (sess->topicgad, "The channel topic");

   if (!prefs.zvt)
   {
#ifdef USE_GNOME
      if (sess->is_tab)
         wid = gtkutil_button (main_window, GNOME_STOCK_PIXMAP_FORWARD, 0,
              userlist_hide, sess, 0);
      else
         wid = gtkutil_button (sess->window, GNOME_STOCK_PIXMAP_FORWARD, 0,
              userlist_hide, sess, 0);
      gtk_box_pack_end (GTK_BOX (tbox), wid, 0, 0, 0);
#else
      wid = gtk_button_new_with_label (">");
      gtk_box_pack_end (GTK_BOX (tbox), wid, 0, 0, 0);
      gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                          GTK_SIGNAL_FUNC (userlist_hide), (gpointer) sess);
#endif
      show_and_unfocus (wid);
      add_tip (wid, "Hide/Show Userlist");
   }
   if (!prefs.nochanmodebuttons)
   {
      sess->key_entry = gtk_entry_new_with_max_length (16);
      gtk_widget_set_usize (sess->key_entry, 30, -1);
      gtk_box_pack_end (GTK_BOX (tbox), sess->key_entry, 0, 0, 0);
      gtk_signal_connect (GTK_OBJECT (sess->key_entry), "activate",
                          GTK_SIGNAL_FUNC (key_entry), (gpointer) sess);
      gtk_widget_show (sess->key_entry);

      add_flag_wid (&(sess->flag_k), "Keyword", tbox, "K", flagk_hit, sess);

      sess->limit_entry = gtk_entry_new_with_max_length (10);
      gtk_widget_set_usize (sess->limit_entry, 30, -1);
      gtk_box_pack_end (GTK_BOX (tbox), sess->limit_entry, 0, 0, 0);
      gtk_signal_connect (GTK_OBJECT (sess->limit_entry), "activate",
                          GTK_SIGNAL_FUNC (limit_entry), (gpointer) sess);
      gtk_widget_show (sess->limit_entry);

      add_flag_wid (&(sess->flag_l), "User Limit", tbox, "L", flagl_hit, sess);

      add_flag_wid (&(sess->flag_m), "Moderated", tbox, "M", flagm_hit, sess);

      add_flag_wid (&(sess->flag_p), "Private", tbox, "P", flagp_hit, sess);

      add_flag_wid (&(sess->flag_i), "Invite Only", tbox, "I", flagi_hit, sess);

      add_flag_wid (&(sess->flag_s), "Secret", tbox, "S", flags_hit, sess);

      add_flag_wid (&(sess->flag_n), "No outside messages", tbox, "N", flagn_hit, sess);

      add_flag_wid (&(sess->flag_t), "Topic Protection", tbox, "T", flagt_hit, sess);

   } else
      sess->flag_wid[0] = 0;

   leftpane = gtk_hbox_new (FALSE, 0);
   gtk_widget_show (leftpane);

   if (!prefs.nopaned)
   {
      sess->paned = gtk_hpaned_new ();
      gtk_container_add (GTK_CONTAINER (vbox), sess->paned);
      gtk_widget_show (sess->paned);
   }
   rightpane = gtk_hbox_new (FALSE, 8);
   gtk_widget_show (rightpane);
   sess->userlistbox = rightpane;

   if (!prefs.nopaned)
   {
      gtk_paned_pack1 (GTK_PANED (sess->paned), leftpane, TRUE, TRUE);
      gtk_paned_pack2 (GTK_PANED (sess->paned), rightpane, FALSE, TRUE);
      gtk_paned_set_gutter_size (GTK_PANED (sess->paned), 10);
   } else
   {
      wid = gtk_hbox_new (0, 2);
      gtk_container_add (GTK_CONTAINER (vbox), wid);
      gtk_widget_show (wid);
      gtk_container_add (GTK_CONTAINER (wid), leftpane);
      gtk_box_pack_end (GTK_BOX (wid), rightpane, 0, 0, 0);
   }

   sess->nl_box = nlbox = gtk_vbox_new (FALSE, 2);
   gtk_container_add (GTK_CONTAINER (rightpane), nlbox);
   gtk_widget_show (nlbox);

   wid = gtk_frame_new (0);
   gtk_box_pack_start (GTK_BOX (nlbox), wid, 0, 0, 0);
   gtk_widget_show (wid);

   sess->namelistinfo = gtk_label_new (" ");
   gtk_container_add (GTK_CONTAINER (wid), sess->namelistinfo);
   gtk_widget_show (sess->namelistinfo);

   maingui_create_textlist (sess, leftpane);
   sess->leftpane = leftpane;

   sess->namelistgad = gtkutil_clist_new (1, 0, nlbox, GTK_POLICY_AUTOMATIC,
   maingui_userlist_selected, sess, 0,
           0, GTK_SELECTION_MULTIPLE);
#ifdef USE_GNOME
   /* set up drops */
   gtk_drag_dest_set (sess->namelistgad, GTK_DEST_DEFAULT_ALL,
                      dnd_targets, 1, GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);
   gtk_signal_connect (GTK_OBJECT (sess->namelistgad), "drag_motion",
                       GTK_SIGNAL_FUNC (userlist_dnd_motion), 0);
   gtk_signal_connect (GTK_OBJECT (sess->namelistgad), "drag_leave",
                       GTK_SIGNAL_FUNC (userlist_dnd_leave), 0);
   gtk_signal_connect (GTK_OBJECT (sess->namelistgad), "drag_data_received",
                       GTK_SIGNAL_FUNC (userlist_dnd_drop), sess);
#endif

   gtk_clist_set_selection_mode (GTK_CLIST (sess->namelistgad),
              GTK_SELECTION_MULTIPLE);
   gtk_clist_set_column_width (GTK_CLIST (sess->namelistgad), 0, 10);
   gtk_widget_set_usize (sess->namelistgad->parent, 115, 0);
   gtk_signal_connect (GTK_OBJECT (sess->namelistgad), "button_press_event",
                       GTK_SIGNAL_FUNC (ul_button_rel), sess);

   if (!prefs.nouserlistbuttons)
      maingui_createbuttons (sess);
   else
      sess->button_box = 0;

   bbox = gtk_hbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (bbox), 0);
   gtk_box_pack_end (GTK_BOX (vbox), bbox, FALSE, TRUE, 2);
   gtk_widget_show (bbox);

   sess->op_box = gtk_hbox_new (0, 0);
   gtk_box_pack_start (GTK_BOX (bbox), sess->op_box, FALSE, FALSE, 2);
   gtk_widget_show (sess->op_box);

   sess->nickgad = gtk_label_new (sess->server->nick);
   gtk_box_pack_start (GTK_BOX (bbox), sess->nickgad, FALSE, FALSE, 4);
   gtk_widget_show (sess->nickgad);

   sess->inputgad = gtk_entry_new_with_max_length (2048);
   gtk_container_add (GTK_CONTAINER (bbox), sess->inputgad);
   gtk_signal_connect (GTK_OBJECT (sess->inputgad), "activate",
                       GTK_SIGNAL_FUNC (handle_inputgad), sess);
   gtk_signal_connect (GTK_OBJECT (sess->inputgad), "key_press_event",
                       GTK_SIGNAL_FUNC (key_handle_key_press), sess);
   gtk_widget_show (sess->inputgad);
   if (prefs.newtabstofront || justopened)
      gtk_widget_grab_focus (sess->inputgad);

   sess->confbutton = wid = gtk_toggle_button_new_with_label ("C");
   gtk_box_pack_end (GTK_BOX (bbox), wid, 0, 0, 1);
   gtk_widget_show (wid);
   add_tip (wid, "Conference mode (no join/part msgs)");

   gtk_widget_show (sess->window);

   if (!normaltab_style && prefs.tabchannels)
   {
	if (prefs.mainwindow_left || prefs.mainwindow_top)
	   gdk_window_move (main_window->window,
                          prefs.mainwindow_left,
                          prefs.mainwindow_top);
   }

   if (prefs.tabchannels)
   {
	sess->changad = gtk_label_new ("<none>");
      gtk_widget_show (sess->changad);

      gtk_notebook_append_page (GTK_NOTEBOOK (main_book), sess->window, sess->changad);
      gtk_widget_realize (sess->textgad);
      if (justopened)
         gtk_widget_show (main_window);

	if (!normaltab_style)
	   maingui_init_styles (gtk_widget_get_style(sess->changad));

#ifdef USE_GNOME
      if (prefs.zvt)
      {
         zvt_term_set_color_scheme ((ZvtTerm *) sess->textgad, zvt_red, zvt_grn, zvt_blu);
         gtk_widget_hide (sess->textgad);
         gtk_widget_show (sess->textgad);
      }
#endif

      if (prefs.newtabstofront)
      {
         int page;

         page = gtk_notebook_page_num (GTK_NOTEBOOK (main_book), sess->window);
	   gtk_idle_add ((GtkFunction)maingui_pagetofront, (gpointer)page);

         sess->new_data = TRUE;
         gui_new_tab (sess);
      }
   } else
   {
	if (!normaltab_style)
   	   maingui_init_styles (gtk_widget_get_style(sess->changad));

      if (prefs.mainwindow_left || prefs.mainwindow_top)
         gdk_window_move (sess->window->window,
                prefs.mainwindow_left,
                prefs.mainwindow_top);
   }
#ifdef USE_GNOME
   if (prefs.zvt && !prefs.tabchannels)
   {
      zvt_term_set_color_scheme ((ZvtTerm *) sess->textgad, zvt_red, zvt_grn, zvt_blu);
      gtk_widget_hide (sess->textgad);
      gtk_widget_show (sess->textgad);
   }
#endif
}
