/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include <pwd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "cfgfiles.h"
#include "serverlist.h"
#include "gtkutil.h"


extern struct xchatprefs prefs;

extern void connect_server(struct session *sess, char *server, int port, int quiet);
extern struct session *new_session(struct server *serv, char *from);
extern struct server *new_server(void);
extern void gui_change_nick(struct server *serv, char *newnick);


GtkWidget *slwin = 0;
GtkWidget *sllist;
GtkWidget *entry_server;
GtkWidget *entry_channel;
GtkWidget *entry_port;
GtkWidget *entry_comment;
GtkWidget *entry_password;

struct serverentry *new_serverentry(void);
gint close_server_list(GtkWidget *, int);
void add_defaults(void);
void add_server_entry(struct serverentry *serv);


char *read_next_server_entry(char *my_cfg, struct serverentry *serv)
{
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "servername ", serv->server);
   if(my_cfg)
   {
      my_cfg = cfg_get_str(my_cfg, "port ", serv->channel);
      serv->port = atoi(serv->channel);
   }
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "channel ", serv->channel);
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "password ", serv->password);
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "comment ", serv->comment);

   return my_cfg;
}

void load_serverentrys(void)
{
   struct serverentry serv;
   struct stat st;
   int fh;
   char *cfg, *my_cfg;
   char file[256];

   snprintf(file, sizeof file, "%s/servers.conf", get_xdir());
   fh = open(file, O_RDONLY);
   if(fh != -1)
   {
      fstat(fh, &st);
      cfg = malloc(st.st_size);
      if(cfg)
      {
	 read(fh, cfg, st.st_size);
	 my_cfg = cfg;
	 while((my_cfg = read_next_server_entry(my_cfg, &serv)))
	 {
	    add_server_entry(&serv);
	 }
	 free(cfg);
      }
      close(fh);
   } else
     add_defaults();
}

void save_serverentrys(void)
{
   int fh, i = 0;
   char file[256], buf[1024];
   char *chan, *servname, *port, *pass, *comment;

   check_prefs_dir();

   snprintf(file, sizeof file, "%s/servers.conf", get_xdir());
   fh = open(file, O_TRUNC | O_WRONLY | O_CREAT, 0600);
   if(fh != -1)
   {
      while(1)
      {
	 if(!gtk_clist_get_text(GTK_CLIST(sllist), i, 0, &chan)) break;
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 1, &servname);
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 2, &port);
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 3, &pass);
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 4, &comment);
	 snprintf(buf, sizeof buf,
		  "servername = %s\n"
		  "port = %d\n"
		  "channel = %s\n"
		  "password = %s\n"
		  "comment = %s\n\n",
		  servname, atoi(port), chan,
		  pass, comment);
	 write(fh, buf, strlen(buf));
	 i++;
      }
      close(fh);
   }
}

void add_server_entry(struct serverentry *serv)
{
   char tbuf[16];
   gchar *new[1][6];

   sprintf(tbuf, "%d", serv->port);

   new[0][0] = serv->channel;
   new[0][1] = serv->server;
   new[0][2] = tbuf;
   new[0][3] = serv->password;
   new[0][4] = serv->comment;

   gtk_clist_append(GTK_CLIST(sllist), new[0]);
}

void connect_clicked(GtkWidget *wid, struct session *sess)
{
   char s[128], p[128];
   char *server, *portstr, *chan, *pass;
   int port, row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      if(!gtk_clist_get_text(GTK_CLIST(sllist), row, 0, &chan)) return;
      gtk_clist_get_text(GTK_CLIST(sllist), row, 1, &server);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 2, &portstr);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 3, &pass); 
      port = atoi(portstr);
      strcpy(s, server);
      strcpy(p, pass);
      strncpy(sess->willjoinchannel, chan, 200);
      close_server_list(0, TRUE);
      if(!sess)
      {
	 struct server *serv = new_server();
	 if(serv) sess = new_session(serv, 0);
      }
      strncpy(sess->server->password, p, 85);
      if(strcmp(sess->server->nick, prefs.nick1))
      {
	 gui_change_nick(sess->server, prefs.nick1);
      }

      connect_server(sess, s, port, FALSE);
   }
}

void connect_new_clicked(GtkWidget *wid, struct session *sess)
{
   char s[128], c[128], p[64];
   char *server, *portstr, *chan, *pass;
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      struct server *serv;
      int port;
      gtk_clist_get_text(GTK_CLIST(sllist), row, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 1, &server);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 2, &portstr);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 3, &pass); 
      port = atoi(portstr);
      strcpy(c, chan);
      strcpy(s, server);
      strcpy(p, pass);
      serv = new_server();
      if(serv)
      {
	 close_server_list(0, TRUE);
	 sess = new_session(serv, 0);
	 strncpy(sess->server->password, p, 85);
	 strncpy(sess->willjoinchannel, c, 200);
	 strcpy(sess->server->nick, prefs.nick1);
	 while(gtk_events_pending()) gtk_main_iteration();
	 connect_server(sess, s, port, FALSE);
      }
   }
}

void delete_server_clicked(GtkWidget *wid, struct session *sess)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      gtk_clist_unselect_all(GTK_CLIST(sllist));
      gtk_clist_remove((GtkCList*)sllist, row);
   }
}

void new_server_clicked(GtkWidget *wid, struct session *sess)
{
   int i, row;
   gchar *new[1][5];
   char *server, *portstr, *pass;

/*
  If there is an existing selection, copy it.
  By Alistair Cunningham, ac@gnu.org, 5th May 1999
*/

   new[0][0] = "#new";
   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      gtk_clist_get_text(GTK_CLIST(sllist), row, 1, &server);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 2, &portstr);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 3, &pass); 
      new[0][1] = server;
      new[0][2] = portstr;
      new[0][3] = pass;
      new[0][4] = "";
   } else {
      new[0][1] = "EDIT ME";
      new[0][2] = "6667";
      new[0][3] = "";
      new[0][4] = "";
   }

   i = gtk_clist_insert(GTK_CLIST(sllist), row+1, new[0]);
   gtk_clist_select_row(GTK_CLIST(sllist), i, 0);
   gtk_clist_moveto(GTK_CLIST(sllist), i, 0, 0.5, 0);
}

void sl_row_unselected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even)
{
   gtk_entry_set_text(GTK_ENTRY(entry_server), "");
   gtk_entry_set_text(GTK_ENTRY(entry_channel), "");
   gtk_entry_set_text(GTK_ENTRY(entry_port), "");
   gtk_entry_set_text(GTK_ENTRY(entry_password), "");
   gtk_entry_set_text(GTK_ENTRY(entry_comment), "");
}

void sl_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer sess)
{
   gchar *server, *channel, *port, *comment, *password;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      gtk_clist_get_text(GTK_CLIST(sllist), row, 0, &channel);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 1, &server);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 2, &port);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 3, &password);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 4, &comment);

      gtk_entry_set_text(GTK_ENTRY(entry_channel), channel);  
      gtk_entry_set_text(GTK_ENTRY(entry_server), server);
      gtk_entry_set_text(GTK_ENTRY(entry_port), port);
      gtk_entry_set_text(GTK_ENTRY(entry_password), password);
      gtk_entry_set_text(GTK_ENTRY(entry_comment), comment);

      if(even && even->type == GDK_2BUTTON_PRESS)
      {
	 connect_clicked(0, (struct session *)sess);
      }
   } else
     sl_row_unselected(0, 0, 0, 0);
}

void move_up_clicked(GtkWidget *igad, struct session *sess)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1 && row > 0)
   {
      gtk_clist_freeze(GTK_CLIST(sllist));
      gtk_clist_swap_rows(GTK_CLIST(sllist), row-1, row);
      gtk_clist_thaw(GTK_CLIST(sllist));
      row--;
      if(gtk_clist_row_is_visible(GTK_CLIST(sllist), row) != GTK_VISIBILITY_FULL)
	gtk_clist_moveto(GTK_CLIST(sllist), row, 0, 0.1, 0);
   }
}

void move_dn_clicked(GtkWidget *igad, struct session *sess)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *temp;
      if(!gtk_clist_get_text(GTK_CLIST(sllist), row+1, 0, &temp)) return;
      gtk_clist_freeze(GTK_CLIST(sllist));
      gtk_clist_swap_rows(GTK_CLIST(sllist), row, row+1);
      gtk_clist_thaw(GTK_CLIST(sllist));
      row++;
      if(!gtk_clist_row_is_visible(GTK_CLIST(sllist), row) != GTK_VISIBILITY_FULL)
	gtk_clist_moveto(GTK_CLIST(sllist), row, 0, 0.9, 0);
   } 
}

void skip_motd(GtkWidget *igad, gpointer serv)
{
   if(GTK_TOGGLE_BUTTON(igad) -> active)
      prefs.skipmotd = TRUE;
   else
      prefs.skipmotd = FALSE;
}

void handle_server(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *server = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 1, server);
   }
}

void handle_channel(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *chan = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 0, chan);
   }
}

void handle_comment(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *comment = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 4, comment);
   }
}

void handle_password(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *pw = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 3, pw);
   }
}

void handle_port(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *port = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 2, port);
   }
}

GtkWidget *nick1gad;
GtkWidget *nick2gad;
GtkWidget *nick3gad;
GtkWidget *realnamegad;
GtkWidget *usernamegad;

int close_server_list(GtkWidget *win, int destroy)
{
   strcpy(prefs.nick1, gtk_entry_get_text((GtkEntry*)nick1gad));
   strcpy(prefs.nick2, gtk_entry_get_text((GtkEntry*)nick2gad));
   strcpy(prefs.nick3, gtk_entry_get_text((GtkEntry*)nick3gad));
   strcpy(prefs.username, gtk_entry_get_text((GtkEntry*)usernamegad));
   strcpy(prefs.realname, gtk_entry_get_text((GtkEntry*)realnamegad));
   save_serverentrys();
   if(destroy && slwin) gtk_widget_destroy(slwin);
   slwin = 0;
   return 0;
}

void serverlist_autoconnect(struct session *sess, int aut)
{
   while(gtk_events_pending()) gtk_main_iteration();
   gtk_clist_select_row(GTK_CLIST(sllist), aut - 1, 0);
   connect_clicked(0, sess);
}

void open_server_list(GtkWidget *widd, struct session *sess)
{
   static gchar *titles[] = {"Channel", "Server", "Port", "Password", "Comment"};
   GtkWidget *vbox, *box, *wid;

   if(slwin)
   {
      close_server_list(0, TRUE);
      return;
   }

   slwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(slwin), "X-Chat: Server List");
   gtk_signal_connect(GTK_OBJECT(slwin), "delete_event",
                       GTK_SIGNAL_FUNC(close_server_list), 0);
   gtk_window_set_policy(GTK_WINDOW(slwin), TRUE, TRUE, FALSE);
   gtk_window_set_wmclass(GTK_WINDOW(slwin), "servlist", "X-Chat");

   vbox = gtk_vbox_new(0,2);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(slwin), vbox);
   gtk_widget_show(vbox);

   box = gtk_hbox_new(0,5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   gtkutil_label_new("Nicknames:", box);

   nick1gad = gtkutil_entry_new(63, box, 0, 0);
   gtk_entry_set_text(GTK_ENTRY(nick1gad), prefs.nick1);

   nick2gad = gtkutil_entry_new(63, box, 0, 0);
   gtk_entry_set_text(GTK_ENTRY(nick2gad), prefs.nick2);

   nick3gad = gtkutil_entry_new(63, box, 0, 0);
   gtk_entry_set_text(GTK_ENTRY(nick3gad), prefs.nick3);

   box = gtk_hbox_new(0,5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   gtkutil_label_new("Realname:", box);

   realnamegad = gtkutil_entry_new(63, box, 0, 0);
   gtk_entry_set_text(GTK_ENTRY(realnamegad), prefs.realname);

   gtkutil_label_new("Username:", box);

   usernamegad = gtkutil_entry_new(63, box, 0, 0);
   gtk_entry_set_text(GTK_ENTRY(usernamegad), prefs.username);

   wid = gtk_toggle_button_new_with_label("Skip MOTD?");
   gtk_signal_connect(GTK_OBJECT(wid), "toggled",
		      GTK_SIGNAL_FUNC(skip_motd), 0);
   gtk_container_add(GTK_CONTAINER(box), wid);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wid),
			       (prefs.skipmotd));
   gtk_widget_show(wid);

   sllist = gtkutil_clist_new(5, titles, vbox, GTK_POLICY_ALWAYS,
			      sl_row_selected, sess,
			      sl_row_unselected, 0, GTK_SELECTION_BROWSE);
   /*gtk_clist_set_selection_mode(GTK_CLIST(sllist), GTK_SELECTION_MULTIPLE);*/
   gtk_widget_set_usize(sllist, -1, 200);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 0, 90);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 1, 165);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 2, 40);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 3, 60);

   box = gtk_hbox_new(0,0);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   gtkutil_label_new("Server Name:", box);

   entry_server = gtkutil_entry_new(99, box, handle_server, 0);

   gtkutil_label_new("Port:", box);

   entry_port = gtkutil_entry_new(14, box, handle_port, 0);
   gtk_widget_set_usize(entry_port, 80, -1);

   gtkutil_label_new("Password:", box);

   entry_password = gtkutil_entry_new(85, box, handle_password, 0);
   gtk_widget_set_usize(entry_password, 80, -1);

   box = gtk_hbox_new(0,5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   gtkutil_label_new("Comment:", box);

   entry_comment = gtkutil_entry_new(99, box, handle_comment, 0);
   gtk_widget_set_usize(entry_comment, 160, -1);

   gtkutil_label_new("Channel:", box);

   entry_channel = gtkutil_entry_new(85, box, handle_channel, 0);

   wid = gtk_hseparator_new();
   gtk_box_pack_start(GTK_BOX(vbox), wid, FALSE, FALSE, 8);
   gtk_widget_show(wid);

   box = gtk_hbox_new(0, 2);
   gtk_container_set_border_width(GTK_CONTAINER(box), 2);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 2);
   gtk_widget_show(box);

   gtkutil_button_new("Connect", 0, 0, connect_clicked, sess, box);
   if(prefs.tabchannels)
     gtkutil_button_new("Connect in New Tab", 0, 0, connect_new_clicked, sess, box);
   else
     gtkutil_button_new("Connect in New Win", 0, 0, connect_new_clicked, sess, box);

#ifdef USE_GNOME
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_NEW, "New Server",
			  new_server_clicked, sess, box);
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_CLOSE, "Delete Server",
			  delete_server_clicked, sess, box);
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_UP, "Move Up",
			  move_up_clicked, sess, box);
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_DOWN, "Move Dn",
			  move_dn_clicked, sess, box);
#else
   gtkutil_button_new("New Server", 0, 0, new_server_clicked, sess, box);
   gtkutil_button_new("Delete Server", 0, 0, delete_server_clicked, sess, box);
   gtkutil_button_new("Move Up", 0, 0, move_up_clicked, sess, box);
   gtkutil_button_new("Move Dn", 0, 0, move_dn_clicked, sess, box);
#endif

   load_serverentrys();

   gtk_widget_show(slwin);
}

struct defaultserv dserv[] = 
{
     {"#linux", "jp.elitenet.org", "X-Chat Channel",6667},
     {"#linux", "us.elitenet.org", "X-Chat Channel",6667},

     {"", "irc.stealth.net", "IRCNet",6667},
     {"", "irc.funet.fi", "IRCNet",6667},

     {"", "irc.dal.net", "DALNet",6667},
     {"", "stlouis.mo.us.dal.net", "DALNet", 6667},

     {"", "irc.darkchar.net", "DarkcharNet", 6667},
     {"", "highvoltage.darkchar.net", "DarkcharNet", 6667},

     {"", "irc.efnet.net", "EFNet",6667},
     {"", "irc.ais.net", "EFNet",6667},

     {"", "sprynet.us.galaxynet.org", "GalaxyNet",6667},
     {"", "atlanta.ga.us.galaxynet.org", "GalaxyNet",6667},

     {"", "us.undernet.org", "UnderNet",6667},
     {"", "atlanta.ga.us.undernet.org", "UnderNet",6667},

     {0,0,0,0}
};

void add_defaults(void)
{
   struct serverentry serv;
   int i = 0;

   serv.password[0] = 0;
   while(1)
   {
      if(dserv[i].channel == 0) return;
      strcpy(serv.channel, dserv[i].channel);
      strcpy(serv.server, dserv[i].server);
      strcpy(serv.comment, dserv[i].comment);
      serv.port = dserv[i].port;
      add_server_entry(&serv);
      i++;
   }
}
