/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include "menu.h"
#include <fcntl.h>
#include "cfgfiles.h"
#include "gtkutil.h"
#include <sys/stat.h>

extern void maingui_updatebuttons(struct session *sess);
extern int buf_get_line(char *, char **, int *, int);

extern GSList *sess_list;
extern GSList *ctcp_list;
extern GSList *popup_list;
extern GSList *button_list;
extern GSList *command_list;
extern GSList *replace_list;

GtkWidget *editlist_gui_entry_name;
GtkWidget *editlist_gui_entry_cmd;
GtkWidget *editlist_gui_window;
GtkWidget *editlist_gui_list;
GSList *editlist_list;
char *editlist_file;
char *editlist_help;



void list_free(GSList **list)
{
   GSList *temp;
   void *data;
   while(*list)
   {
      temp = *list;
      data = (void *)temp->data;
      free(data);
      *list = g_slist_remove(*list, data);
   }
}

int list_delentry(GSList **list, char *name)
{
   struct popup *pop;
   GSList *alist = *list;

   while( alist )
   {
      pop = (struct popup *)alist -> data;
      if(!strcasecmp(name, pop->name))
      {
	 *list = g_slist_remove(*list, pop);
	 return 1;
      }
      alist = alist -> next;
   }
   return 0;
}

void list_addentry(GSList **list, char *cmd, char *name)
{
   struct popup *pop = malloc(sizeof(struct popup));
   if(!pop) return;
   if(!cmd) {
     strcpy(pop->name, name);
     pop->cmd[0] = 0;
   }
   else {
      strcpy(pop->name, name);
      strcpy(pop->cmd, cmd);
   }
   *list = g_slist_append(*list, pop);
}

void list_loadconf(char *file, GSList **list, char *defaultconf)
{
   char cmd[132];
   char name[82];
   char *buf, *ibuf;
   int fh, pnt = 0;
   struct stat st;

   buf = malloc(1000);
   snprintf(buf, 1000, "%s/%s", get_xdir(), file);
   fh = open(buf, O_RDONLY);
   if(fh == -1)
   {
      fh = open(buf, O_TRUNC | O_WRONLY | O_CREAT, 0600);
      if(fh != -1)
      {
	 write(fh, defaultconf, strlen(defaultconf));
	 close(fh);
	 free(buf);
	 list_loadconf(file, list, defaultconf);
      }
      return;
   }
   free(buf);
   if (fstat(fh, &st) != 0) {
      perror("fstat");
      abort();
   }
   ibuf = malloc(st.st_size);
   read(fh, ibuf, st.st_size);
   close(fh);

   cmd[0] = 0;
   name[0] = 0;
   
   while(buf_get_line(ibuf, &buf, &pnt, st.st_size))
   {
      if(*buf != '#')
      {
	 if(!strncasecmp(buf, "NAME ", 5)) strcpy(name, buf+5);
	 if(!strncasecmp(buf, "CMD ", 4)) {
	   strcpy(cmd, buf+4);
	   if(*name) {
	     list_addentry(list, cmd, name);
	     cmd[0] = 0;
	     name[0] = 0;
	   }
	 }
      }
   }
   free(ibuf);
}

void editlist_gui_load(GtkWidget *listgad)
{
   gchar *new[1][2];
   GSList *list = editlist_list;
   struct popup *pop;

   while(list)
   {
      pop = (struct popup*)list->data;
      new[0][0] = pop->name;
      new[0][1] = pop->cmd;
      gtk_clist_append(GTK_CLIST(listgad), new[0]);
      list = list->next;
   }
}

void editlist_gui_row_unselected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   gtk_entry_set_text(GTK_ENTRY(editlist_gui_entry_name), "");  
   gtk_entry_set_text(GTK_ENTRY(editlist_gui_entry_cmd), "");
}

void editlist_gui_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   char *name, *cmd;
   
   row = gtkutil_clist_selection(editlist_gui_list);
   if(row != -1)
   {
      gtk_clist_get_text(GTK_CLIST(clist), row, 0, &name);
      gtk_clist_get_text(GTK_CLIST(clist), row, 1, &cmd);

      gtk_entry_set_text(GTK_ENTRY(editlist_gui_entry_name), name);  
      gtk_entry_set_text(GTK_ENTRY(editlist_gui_entry_cmd), cmd);
   } else {
      editlist_gui_row_unselected(0, 0, 0, 0, 0);
   }
}

void editlist_gui_handle_cmd(GtkWidget *igad)
{
   int row;
   char *reply;

   row = gtkutil_clist_selection(editlist_gui_list);
   if(row != -1)
   {
      reply = gtk_entry_get_text(GTK_ENTRY(igad));
      gtk_clist_set_text(GTK_CLIST(editlist_gui_list), row, 1, reply);
   }
}

void editlist_gui_handle_name(GtkWidget *igad)
{
   int row;
   char *ctcp;

   row = gtkutil_clist_selection(editlist_gui_list);
   if(row != -1)
   {
      ctcp = gtk_entry_get_text(GTK_ENTRY(igad));
      gtk_clist_set_text(GTK_CLIST(editlist_gui_list), row, 0, ctcp);
   }
}

void editlist_gui_addnew(GtkWidget *igad)
{
   int i;
   gchar *new[1][2];
   new[0][0] = "*NEW*";
   new[0][1] = "EDIT ME";
   i = gtk_clist_append(GTK_CLIST(editlist_gui_list), new[0]);
   gtk_clist_select_row(GTK_CLIST(editlist_gui_list), i, 0);
   gtk_clist_moveto(GTK_CLIST(editlist_gui_list), i, 0, 0.5, 0);
}

void editlist_gui_delete(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(editlist_gui_list);
   if(row != -1)
   {
      gtk_clist_unselect_all(GTK_CLIST(editlist_gui_list));
      gtk_clist_remove(GTK_CLIST(editlist_gui_list), row);
   }
}

void editlist_gui_save(GtkWidget *igad)
{
   int fh, i = 0;
   char buf[256];
   char *a, *b;

   sprintf(buf, "%s/%s", get_xdir(), editlist_file);
   fh = open(buf, O_TRUNC | O_WRONLY | O_CREAT, 0600);
   if(fh != -1)
   {
      while(1)
      {
	 if(!gtk_clist_get_text(GTK_CLIST(editlist_gui_list), i, 0, &a)) break;
	 gtk_clist_get_text(GTK_CLIST(editlist_gui_list), i, 1, &b);
	 if(!strcasecmp(a, "SEP"))
	   strcpy(buf, "SEP\n");
	 else
	   sprintf(buf, "NAME %s\nCMD %s\n\n", a, b);
	 write(fh, buf, strlen(buf));
	 i++;
      }
      close(fh);
      gtk_widget_destroy(editlist_gui_window);
      if (editlist_list == replace_list) {
         list_free(&replace_list);
	 list_loadconf(editlist_file, &replace_list, 0);
      }
      else if(editlist_list == popup_list)
      {
	 list_free(&popup_list);
	 list_loadconf(editlist_file, &popup_list, 0);
      } else {
	 if(editlist_list == button_list)
	 {
	    GSList *list = sess_list;
	    struct session *sess;
	    list_free(&button_list);
	    list_loadconf(editlist_file, &button_list, 0);
	    while(list)
	    {
	       sess = (struct session *)list -> data;
	       maingui_updatebuttons(sess);
	       list = list -> next;
	    }
	 } else {
	    if(editlist_list == ctcp_list)
	    {
	       list_free(&ctcp_list);
	       list_loadconf(editlist_file, &ctcp_list, 0);
	    } else {
	       list_free(&command_list);
	       list_loadconf(editlist_file, &command_list, 0);
	    }
	 }
      }
   }
}

void editlist_gui_help(GtkWidget *igad)
{
   if(editlist_help) gtkutil_simpledialog(editlist_help);
}

void editlist_gui_sort(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(editlist_gui_list);
   if(row != -1)
     gtk_clist_unselect_row(GTK_CLIST(editlist_gui_list), row, 0);
   gtk_clist_sort(GTK_CLIST(editlist_gui_list));
}

void editlist_gui_movedown(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(editlist_gui_list);
   if(row != -1)
   {
      char *temp;
      if(!gtk_clist_get_text(GTK_CLIST(editlist_gui_list), row+1, 0, &temp)) return;
      gtk_clist_freeze(GTK_CLIST(editlist_gui_list));
      gtk_clist_swap_rows(GTK_CLIST(editlist_gui_list), row, row+1);
      gtk_clist_thaw(GTK_CLIST(editlist_gui_list));
      row++;
      if(!gtk_clist_row_is_visible(GTK_CLIST(editlist_gui_list), row) != GTK_VISIBILITY_FULL)
	gtk_clist_moveto(GTK_CLIST(editlist_gui_list), row, 0, 0.9, 0);
   }
}

void editlist_gui_moveup(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(editlist_gui_list);
   if(row != -1 && row > 0)
   {
      gtk_clist_freeze(GTK_CLIST(editlist_gui_list));
      gtk_clist_swap_rows(GTK_CLIST(editlist_gui_list), row-1, row);
      gtk_clist_thaw(GTK_CLIST(editlist_gui_list));
      row--;
      if(gtk_clist_row_is_visible(GTK_CLIST(editlist_gui_list), row) != GTK_VISIBILITY_FULL)
	gtk_clist_moveto(GTK_CLIST(editlist_gui_list), row, 0, 0.1, 0);
   }
}

void editlist_gui_close(void)
{
   editlist_gui_window = 0;
}

void editlist_gui_open(GSList *list, char *title, char *wmclass, char *file, char *help)
{
   gchar *titles[] = { "Name", "Command" };
   GtkWidget *vbox, *hbox, *button;

   if(editlist_gui_window) return;

   editlist_list = list;
   editlist_file = file;
   editlist_help = help;

   editlist_gui_window = gtkutil_window_new(title, wmclass, 450, 250,
					    editlist_gui_close, 0);
   gtk_window_set_policy(GTK_WINDOW(editlist_gui_window), TRUE, TRUE, FALSE);

   vbox = gtk_vbox_new(0, 2);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(editlist_gui_window), vbox);
   gtk_widget_show(vbox);

   editlist_gui_list = gtkutil_clist_new(2, titles, vbox, GTK_POLICY_ALWAYS,
					 editlist_gui_row_selected, 0,
					 editlist_gui_row_unselected, 0,
					 GTK_SELECTION_BROWSE);
   gtk_clist_set_column_width(GTK_CLIST(editlist_gui_list), 0, 90);

   hbox = gtk_hbox_new(0, 2);
   gtk_box_pack_end(GTK_BOX(vbox), hbox, 0, 0, 0);
   gtk_widget_show(hbox);

#ifndef USE_GNOME
   gtkutil_button_new("Move Up", 100, 0, editlist_gui_moveup, 0, hbox);
   gtkutil_button_new("Move Dn", 100, 0, editlist_gui_movedown, 0, hbox);
#else
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_PIXMAP_UP, "Move Up",
			editlist_gui_moveup, 0, hbox);
   gtk_widget_set_usize(button, 100, 0);
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_PIXMAP_DOWN, "Move Dn",
			editlist_gui_movedown, 0, hbox);
   gtk_widget_set_usize(button, 100, 0);
#endif
     
   button = gtk_vseparator_new();
   gtk_container_add(GTK_CONTAINER(hbox), button);
   gtk_widget_show(button);

#ifdef USE_GNOME
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_BUTTON_CANCEL, "Cancel",
				 gtkutil_destroy, editlist_gui_window, hbox);
   gtk_widget_set_usize(button, 100, 0);
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_PIXMAP_SAVE, "Save",
				 editlist_gui_save, 0, hbox);
   gtk_widget_set_usize(button, 100, 0);
#else
   gtkutil_button_new("Cancel", 100, 0, gtkutil_destroy, editlist_gui_window, hbox);
   gtkutil_button_new("Save", 100, 0, editlist_gui_save, 0, hbox);
#endif

   hbox = gtk_hbox_new(0, 2);
   gtk_box_pack_end(GTK_BOX(vbox), hbox, 0, 0, 0);
   gtk_widget_show(hbox);

#ifdef USE_GNOME
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_PIXMAP_NEW, "Add New",
				 editlist_gui_addnew, 0, hbox);
   gtk_widget_set_usize(button, 100, 0);
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_PIXMAP_CUT, "Delete",
				 editlist_gui_delete, 0, hbox);
   gtk_widget_set_usize(button, 100, 0);
#else
   gtkutil_button_new("Add New", 100, 0, editlist_gui_addnew, 0, hbox);
   gtkutil_button_new("Delete", 100, 0, editlist_gui_delete, 0, hbox);
#endif

   button = gtk_vseparator_new();
   gtk_container_add(GTK_CONTAINER(hbox), button);
   gtk_widget_show(button);

#ifdef USE_GNOME
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_PIXMAP_SPELLCHECK, "Sort",
			editlist_gui_sort, 0, hbox);
   gtk_widget_set_usize(button, 100, 0);
   button = gtkutil_stock_button(editlist_gui_window, GNOME_STOCK_PIXMAP_HELP, "Help",
			editlist_gui_help, 0, hbox);
   gtk_widget_set_usize(button, 100, 0);
#else
   gtkutil_button_new("Sort", 100, 0, editlist_gui_sort, 0, hbox);
   button = gtkutil_button_new("Help", 100, 0, editlist_gui_help, 0, hbox);
#endif
   if(!help) gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

   hbox = gtk_hbox_new(0, 2);
   gtk_box_pack_end(GTK_BOX(vbox), hbox, 0, 0, 0);
   gtk_widget_show(hbox);

   editlist_gui_entry_name = gtk_entry_new_with_max_length(40);
   gtk_widget_set_usize(editlist_gui_entry_name, 96, 0);
   gtk_signal_connect(GTK_OBJECT(editlist_gui_entry_name), "changed",
		      GTK_SIGNAL_FUNC(editlist_gui_handle_name), 0);
   gtk_box_pack_start(GTK_BOX(hbox), editlist_gui_entry_name, 0, 0, 0);
   gtk_widget_show(editlist_gui_entry_name);

   editlist_gui_entry_cmd = gtk_entry_new_with_max_length(128);
   gtk_signal_connect(GTK_OBJECT(editlist_gui_entry_cmd), "changed",
		      GTK_SIGNAL_FUNC(editlist_gui_handle_cmd), 0);
   gtk_container_add(GTK_CONTAINER(hbox), editlist_gui_entry_cmd);
   gtk_widget_show(editlist_gui_entry_cmd);

   hbox = gtk_hbox_new(0, 2);
   gtk_box_pack_end(GTK_BOX(vbox), hbox, 0, 0, 0);
   gtk_widget_show(hbox);

   editlist_gui_load(editlist_gui_list);

   gtk_widget_show(editlist_gui_window);
}
