/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#undef USE_GNOME
#include "style.h"
#include "xchat.h"
#include "plugin.h"
#include <sys/utsname.h>
#include <sys/time.h>
#include <unistd.h>
#include <netdb.h>
#include <time.h>

/* xchat.c */

extern struct session *find_session_from_nick(char *nick, struct server *serv);
extern struct session *find_session_from_channel(char *chan, struct server *serv);
extern struct session *find_session_from_waitchannel(char *chan, struct server *serv);
extern void show_generic_channel(struct server *serv, char *chan, char *msg);
extern struct session *new_session(struct server *serv);

/* text.c */

extern void PrintText(struct session *sess, char *text);
extern void setup_logging (struct session *sess);

/* maingui.c */

extern void make_non_channel_window(struct session *sess);
extern void key_entry(GtkWidget *igad, struct session *sess);
extern void limit_entry(GtkWidget *igad, struct session *sess);
extern void set_channel(struct session *);
extern void clear_channel(struct session *sess);
extern void flagk_hit(GtkWidget *wid, struct session *sess);
extern void flagl_hit(GtkWidget *wid, struct session *sess);
extern void flagm_hit(GtkWidget *wid, struct session *sess);
extern void flagp_hit(GtkWidget *wid, struct session *sess);
extern void flagi_hit(GtkWidget *wid, struct session *sess);
extern void flags_hit(GtkWidget *wid, struct session *sess);
extern void flagn_hit(GtkWidget *wid, struct session *sess);
extern void flagt_hit(GtkWidget *wid, struct session *sess);
extern void gui_set_title(struct session *sess);

/* rawlog.c */

extern void add_rawlog(struct server *serv, char *text);

/* dcc.c */

extern void handle_dcc(struct session *sess, char *outbuf, char *nick, char *word[], char *word_eol[]);

/* dialog.c */

extern void dialog_change_nick(struct server *serv, char *nick, char *newnick);
extern int add_to_dialog(struct server *, char *tbuf, char *nick, char *text, char *ip, int type);

/* userlist.c */

extern struct user *find_name(struct session *sess, char *name);
extern void update_user_list(struct session *sess);
extern void clear_user_list(struct session *sess);
extern void sort_namelist(struct session *sess);
extern void voice_name(struct session *sess, char *name);
extern void devoice_name(struct session *sess, char *name);
extern void op_name(struct session *sess, char *name);
extern void deop_name(struct session *sess, char *name);
extern void add_name(struct session *sess, char *name);
extern int sub_name(struct session *sess, char *name);
extern void change_nick(struct session *sess, char *oldnick, char *newnick);
extern void gui_change_nick(struct server *serv, char *newnick);
extern int module_inbound (struct session *sess, struct server *serv, char *buf);

/* urlgrab.c */

extern void url_checkurl(char *buf);

/* outbound.c */

extern void process_data_init(unsigned char *buf, char *cmd, char *word[], char *word_eol[]);

/* chanlist.c */

extern void chanlist_addentry(struct server *serv, char *chan, char *users, char *topic);

/* util.c */

extern int get_mhz(void);
extern char *nocasestrstr(char *text, char *tofind);

/* notify.c */

extern void notify_markonline(struct session *sess, char *outbuf, char *word[]);

/* color.c */

extern int color_of(char *name);

#ifdef USE_PERL
/* perl.c */

int perl_inbound(struct session *sess, struct server *serv, char *buf);
#endif

/* ctcp.c */

extern int ctcp_check(void *sess, char *tbuf, char *nick, char *word[], char *word_eol[], char *ctcp);


extern GSList *sess_list;
extern struct xchatprefs prefs;

#define find_word_to_end(a, b) word_eol[b]
#define find_word(a, b) word[b]

char chan_flags[] = {'t','n','s','i','p','m','l','k'};

struct session *find_unused_session(struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->channel[0] == 0 && sess->server == serv) return sess;
      list = list -> next;
   }
   return 0;
}

struct session* tab_msg_session(char* target, struct server* serv)
{
   struct session *sess = find_session_from_waitchannel(target, serv);
   if(!sess)
   {
     sess = find_unused_session(serv);
     if(!sess)
       sess = new_session(serv);
   }
   if(sess)
   {
     strncpy(sess->channel, target, 200);
     sess->waitchannel[0] = 0;
     sess->ignore_date = TRUE;
     sess->ignore_mode = TRUE;
     sess->ignore_names = TRUE;
     sess->end_of_names = FALSE;
     setup_logging(sess);
     set_channel(sess);
     gui_set_title(sess);
     clear_user_list(sess);
   }
   return sess;
}

void private_msg(struct server *serv, char *tbuf, char *from, char *ip, char *text)
{
   struct session *sess;

   if (EMIT_SIGNAL(XP_PRIVMSG, serv, from, ip, text, NULL, 0) == 1)
     return;

   if(prefs.beepmsg) 
     gdk_beep();
   if (prefs.privmsgtab) {
      sess = find_session_from_channel(from, serv);
      if(sess || prefs.autodialog)
      {
	 if (!sess) {
	    sess = tab_msg_session(from, serv);
	    make_non_channel_window(sess);
	 }
	 sprintf(tbuf, CHANNEL_MSG, from, text);
	 PrintText(sess, tbuf);
	 if(ip && ip[0]) gtk_entry_set_text(GTK_ENTRY(sess->topicgad), ip);
	 return;
      }
   }

   if(add_to_dialog(serv, tbuf, from, text, ip, 0)) return;
   sprintf(tbuf, PRIVMSG_TEXT, from, text);
   sess = find_session_from_nick(from, serv);
   if(!sess) sess = serv->front_session;
   PrintText(sess, tbuf);
}

void channel_action(struct session *sess, char *tbuf, char *chan, char *from, char *text, char fromme)
{
   struct session *def = sess;
   
   if (EMIT_SIGNAL(XP_CHANACTION, sess, chan, from, text, NULL, fromme) == 1)
     return;

   if(*chan != '#' && *chan != '&')
   {
     if (prefs.privmsgtab) {
       struct session* msess = find_session_from_channel(from, sess->server);
       if (!sess) {
	 msess = tab_msg_session(from, msess->server);
         PrintText(msess, tbuf);
	 return;
       }
       sprintf(tbuf, PRIVMSG_TEXT, from, text);
     } else {
       if(add_to_dialog(sess->server, tbuf, from, text, "", 2)) return;
     }
   }

   if (fromme)
   {
      if (prefs.colorednicks)
	 sprintf(tbuf, ACTION_TEXT_COLOR, 2, from, text);
      else
	 sprintf(tbuf, ACTION_TEXT, from, text);
   } else {
      if (prefs.colorednicks)
	 sprintf(tbuf, ACTION_TEXT_COLOR, color_of(from), from, text);
      else
	 sprintf(tbuf, ACTION_TEXT, from, text);
   }

   sess = find_session_from_channel(chan, sess->server);
   if(sess)
     PrintText(sess, tbuf);
   else
     PrintText(def, tbuf);
}

void channel_msg(struct server *serv, char *outbuf, char *chan, char *from, char *text, char fromme)
{
   char *real_outbuf = outbuf;

   if (EMIT_SIGNAL(XP_CHANMSG, serv, chan, from, text, NULL, fromme) == 1)
     return;

   if(prefs.tabnicks)
   {
      int i, len;
      len = 9 - strlen(from);
      if(len > 0)
      {
	 for(i=0; i<len; i++)
	 {
	    *outbuf = ' ';
	    outbuf++;
	 }
      }
   }
   if(fromme)
   {
      if(prefs.colorednicks)
	 sprintf(outbuf, CHANNEL_MSG_COLOR, 2, from, text);
      else
	 sprintf(outbuf, CHANNEL_MSG_FROM_USER, from, text);
   } else {
      if (prefs.colorednicks)
	 sprintf(outbuf, CHANNEL_MSG_COLOR, color_of(from), from, text);
      else {
	 if(nocasestrstr(text, serv->nick))
	    sprintf(outbuf, CHANNEL_MSG_WITH_NICK, from, text);
	 else
	    sprintf(outbuf, CHANNEL_MSG, from, text);
      }
   }
   if(prefs.tabnicks)
   {
      char *po;
      if((po = strchr(outbuf, '<'))) po[0] = ' ';
      if((po = strchr(po, '>'))) po[0] = '|';
   }
   show_generic_channel(serv, chan, real_outbuf);
}

void user_new_nick(struct server *serv, char *outbuf, char *nick, char *newnick, int quiet)
{
   int me;
   struct session *sess;
   GSList *list = sess_list;

   if(*newnick == ':') newnick++;
   if(!strcasecmp(nick, serv->nick)) me = TRUE; else me = FALSE;

   if (EMIT_SIGNAL(XP_CHANGENICK, serv, nick, newnick, NULL, NULL, me) == 1)
      return;

   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 if(me || find_name(sess, nick))
	 {
	    if(!quiet)
	    {
	       if(me)
	       	 sprintf(outbuf, STARTON" You are now known as %s\n", newnick);
	       else
		 sprintf(outbuf, STARTON" %s is now known as %s\n",
		    nick, newnick);
	       PrintText(sess, outbuf);
	    }
	    change_nick(sess, nick, newnick);
	 }
	 if(!strcasecmp(sess->channel, nick))
	 {
	    strncpy(sess->channel, newnick, 200);
	    set_channel(sess);
	    gui_set_title(sess);
	 }
      }
      list = list->next;
   }

   dialog_change_nick(serv, nick, newnick);

   if(me) gui_change_nick(serv, newnick);
}

void you_joined(struct server *serv, char *outbuf, char *chan, char *nick, char *ip)
{
  struct session* sess = tab_msg_session(chan, serv);
  if (sess) {
     sprintf(outbuf, "MODE %s\r\n", chan);
     send(sess->server->sok, outbuf, strlen(outbuf), 0);
     clear_user_list(sess);
     sprintf(outbuf, STARTON"\00311 \002%s\002 \00314(\00310%s\00314)\003  has joined %s\n", nick, ip, chan);
     PrintText(sess, outbuf);
   }
}

void you_kicked(struct server *serv, char *tbuf, char *chan, char *kicker, char *reason)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess)
   {
      sprintf(tbuf, STARTON" You have been kicked from %s by %s (%s)\n",
	      chan, kicker, reason);
      PrintText(sess, tbuf);  
      clear_channel(sess);
      if(prefs.autorejoin)
      {
	 sprintf(tbuf, "JOIN %s\r\n", chan);
	 send(sess->server->sok, tbuf, strlen(tbuf), 0);
	 strcpy(sess->waitchannel, chan);
      }
   }
}

void you_parted(struct server *serv, char *tbuf, char *chan)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess)
   {
      sprintf(tbuf, STARTON" You have left channel %s\n", chan);
      PrintText(sess, tbuf);
      clear_channel(sess);
   }
}

void names_list(struct server *serv, char *tbuf, char *buf)
{
   char *po;
   buf+=2;
   po = strchr(buf, ' ');
   if(po)
   {
      struct session *sess;
      char name[52];
      int pos = 0;

      po[0] = 0;
 
      sess = find_session_from_channel(buf, serv);
      if(!sess) return;
 
      if(!sess->ignore_names)
      {
	 sprintf(tbuf, STARTON" \00311Users: \003 %s\n", po+2); /* buf = chan */
	 PrintText(sess, tbuf);
      }

      if(sess->end_of_names)
      {
	 sess->end_of_names = FALSE;
	 clear_user_list(sess);
      }

      buf = po+1;

      gtk_clist_freeze(GTK_CLIST(sess->namelistgad));
      while(1)
      {
	 buf++;
	 switch(*buf)
	 {
	  case 0:
	    name[pos] = 0;
	    if(pos != 0) add_name(sess, name);
	    gtk_clist_thaw(GTK_CLIST(sess->namelistgad));
	    return;
	  case ' ':
	    name[pos] = 0;
	    pos = 0;
	    add_name(sess, name);
	    break;
	  default:
	    name[pos] = *buf;
	    pos++;
	    if(pos == 51) pos = 50;
	 }
      }
   }
}

void topic(struct server *serv, char *tbuf, char *buf)
{
   char *po = strchr(buf, ' ');
   if(po)
   {
      struct session *sess;
      po[0] = 0;
      sess = find_session_from_channel(buf, serv);
      if(sess)
      {
	 gtk_entry_set_text(GTK_ENTRY(sess->topicgad), po+2);
	 sprintf(tbuf, STARTON" Topic for\00311 %s \003 is\00311 %s\n", buf, po+2);
	 PrintText(sess, tbuf);
      }
   }
}

void new_topic(struct server *serv, char *tbuf, char *nick, char *chan, char *topic)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess)
   {
      gtk_entry_set_text(GTK_ENTRY(sess->topicgad), topic);
      sprintf(tbuf, STARTON" %s has changed the topic to: %s\n", nick, topic);
      PrintText(sess, tbuf);
   }
}

void user_joined(struct server *serv, char *outbuf, char *chan, char *user, char *ip)
{
   struct session *sess = find_session_from_channel(chan, serv);

   if (EMIT_SIGNAL(XP_JOIN, serv, chan, user, ip, NULL, 0) == 1)
      return;

   if(sess)
   {
      sprintf(outbuf, STARTON"\00311 \002%s\002 \00314(\00310%s\00314)\003  has joined %s\n", user, ip, chan);
      PrintText(sess, outbuf);
      add_name(sess, user);
   }
}

void user_kicked(struct server *serv, char *outbuf, char *chan, char *user, char *kicker, char *reason)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess)
   {
      sprintf(outbuf, STARTON" %s has kicked %s from %s (%s)\n",
	      kicker, user, chan, reason);
      PrintText(sess, outbuf);
      sub_name(sess, user);
   }
}

void user_parted(struct server *serv, char *chan, char *user, char *ip)
{
   char outbuf[136];
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess)
   {
      sprintf(outbuf, STARTON" %s \00314(\003 %s\00314)\003  has left %s\n", user, ip, chan);
      PrintText(sess, outbuf);
      sub_name(sess, user);
   } /*else
     printf("- stray PART %s %s\n", chan, user);*/
}

void channel_date(struct session *sess, char *tbuf, char *chan, char *timestr)
{
   long n = atol(timestr);
   sprintf(tbuf, STARTON" Channel %s created on %s",
	   chan, ctime(&n));
   PrintText(sess, tbuf);
   /*show_generic_channel(serv, chan, tbuf);*/
}

void topic_nametime(struct server *serv, char *tbuf, char *chan, char *nick, char *date)
{
   long n = atol(date);
   sprintf(tbuf, STARTON" Topic for\00311 %s \003 set by \00311%s \003 at \00311%s",
	   chan, nick, ctime(&n));
   show_generic_channel(serv, chan, tbuf);
}

void set_server_name(char *tbuf, struct server *serv, char *name)
{
   GSList *list = sess_list;
   struct session *sess;
   strcpy(serv->servername, name);
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv) gui_set_title(sess);
      list = list->next;
   }
}

extern int hidever;

void handle_ctcp(struct session *sess, char *outbuf, char *to, char *nick, char *msg, char *word[], char *word_eol[])
{
   char *po;
   if(!strncasecmp(msg, "VERSION", 7))
   {
      if(!hidever)
      {
	 struct utsname un;
	 int mhz;

	 if(uname(&un) < 0) return;

	 mhz = get_mhz();
	 if(mhz)
	   sprintf(outbuf, "NOTICE %s :\001VERSION X-Chat "VERSION" By ZED - %s %s [%s/%dMHz]: http://xchat.linuxpower.org\001\r\n",
		   nick, un.sysname, un.release, un.machine, mhz);
	 else
	   sprintf(outbuf, "NOTICE %s :\001VERSION X-Chat "VERSION" By ZED - %s %s [%s]: http://xchat.linuxpower.org\001\r\n",
		   nick, un.sysname, un.release, un.machine);

	 send(sess->server->sok, outbuf, strlen(outbuf), 0);
      }
   }

   if(ctcp_check((void *)sess, outbuf, nick, word, word_eol, word[4]+2)) goto jump;

   if(!strncasecmp(msg, "ACTION", 6))
   {
      char *po = strchr(msg+7, '\001');
      if(po) po[0] = 0;
      channel_action(sess, outbuf, to, nick, msg+7, FALSE);
      return;
   }
   if(!strncasecmp(msg, "DCC", 3))
   {
      handle_dcc(sess, outbuf, nick, word, word_eol);
      return;
   }
   if(!strncasecmp(msg, "SOUND", 5))
   {
      char *po = strchr(word[5], '\001');
      if(po) po[0] = 0;
      sprintf(outbuf, STARTON" Received a CTCP Sound %s from %s\n",
	      word[5], nick);
      PrintText(sess->server->front_session, outbuf);
      sprintf(outbuf, "%s/%s", prefs.sounddir, word[5]);
      if(access(outbuf, R_OK) == 0)
      {
	 sprintf(outbuf, "%s %s/%s", prefs.soundcmd, prefs.sounddir, word[5]);
	 popen(outbuf, "r");
      }
      return;
   }
   jump:   
   po = strchr(msg, '\001');
   if(po) po[0] = 0;
   if(*to != '#' && *to != '&')
     sprintf(outbuf, STARTON" Received a CTCP %s from %s\n", msg, nick);
   else {
      struct session *chansess = find_session_from_channel(to, sess->server);
      sprintf(outbuf, STARTON" Received a CTCP %s from %s (to %s)\n", msg, nick, to);
      if(chansess)
      {
	 PrintText(chansess, outbuf);
	 return;
      }
   }
   PrintText(sess->server->front_session, outbuf);
}

void user_quit(struct server *serv, char *outbuf, char *nick, char *reason)
{
   GSList *list = sess_list;
   struct session *sess;
   sprintf(outbuf, STARTON" %s has quit \00314(\003 %s\00314)\n", nick, reason);
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 if(sub_name(sess, nick)) PrintText(sess, outbuf);
      }
      list = list->next;
   }
}

void got_ping_reply(struct session *sess, char *outbuf,
		    char *timestring, char *from)
{
   struct timeval timev;
   unsigned long tim, nowtim, dif;

   gettimeofday(&timev, 0);
   sscanf(timestring, "%lu", &tim);
   nowtim = (timev.tv_sec - 50000) * 1000000 + timev.tv_usec;
   dif = nowtim - tim;

   if(atol(timestring) == 0)
     sprintf(outbuf,
	     STARTON" Ping reply from %s.\n",
	     from);
   else
     sprintf(outbuf,
	   STARTON" Ping reply from %s: %ld.%ld second(s)\n",
	   from,
	   dif / 1000000,
	   (dif / 100000) % 10);
   PrintText(sess, outbuf);
}

void notice(struct server *serv, char *outbuf, char *to, char *nick, char *msg, char *ip)
{
   char *po;
   struct session *sess = 0;

   if(to[0] == '#' || to[0] == '&') sess = find_session_from_channel(to, serv);
   if(!sess)
   {
      sess = find_session_from_nick(nick, serv);
      if(!sess) sess = serv->front_session;
   }

   if(msg[0] == 1)
   {
      msg++;
      if(!strncmp(msg, "PING", 4))
      {
	 got_ping_reply(sess, outbuf, msg+5, nick);
	 return;
      }
   }
   po = strchr(msg, '\001');
   if(po) po[0] = 0;
   if(add_to_dialog(sess->server, outbuf, nick, msg, ip, 1)) return;
   sprintf(outbuf, "\00312-\00313%s\00312- \003 %s\n", nick, msg);
   PrintText(sess, outbuf);
}

void my_gtk_entry_set_text(GtkWidget *wid, char *text, struct session *sess)
{
   void *callback;
   if(sess->flag_wid[0] == 0) return;
   if(wid == sess->limit_entry)
     callback = (void *)limit_entry;
   else
     callback = (void *)key_entry;
   gtk_signal_disconnect_by_data(GTK_OBJECT(wid), sess);
   gtk_entry_set_text(GTK_ENTRY(wid), text);
   gtk_signal_connect(GTK_OBJECT(wid), "activate",
		      GTK_SIGNAL_FUNC(callback), sess);
}

void my_gtk_toggle_state(GtkWidget *wid, int state, char which, struct session *sess)
{
   void *callback;
   switch(which)
   {
    case 'k': callback = flagk_hit; break;
    case 'l': callback = flagl_hit; break;
    case 'm': callback = flagm_hit; break;
    case 'p': callback = flagp_hit; break;
    case 'i': callback = flagi_hit; break;
    case 's': callback = flags_hit; break;
    case 'n': callback = flagn_hit; break;
    case 't': callback = flagt_hit; break;
    default:
      gtk_toggle_button_set_state((GtkToggleButton*)wid, state);
      return;
   }
   gtk_signal_disconnect_by_data(GTK_OBJECT(wid), sess);
   gtk_toggle_button_set_state((GtkToggleButton*)wid, state);
   if(callback)
     gtk_signal_connect(GTK_OBJECT(wid), "toggled",
		      GTK_SIGNAL_FUNC(callback), sess);
}

void update_channelmode_buttons(struct session *sess, char mode, char sign)
{
   int state, i;

   if(sign == '+') state = TRUE; else state = FALSE;

   if(sess->flag_wid[0])
   {
      for(i=0; i<8; i++)
      {
	 if(chan_flags[i] == mode)
	 {
	    if(GTK_TOGGLE_BUTTON(sess->flag_wid[i])->active != state)
	      my_gtk_toggle_state(sess->flag_wid[i], state, chan_flags[i], sess);
	 }
      }
   }
}

void channel_mode(struct server *serv, char *outbuf, char *chan, char *nick, char sign, char mode, char *extra, int quiet)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess)
   {
      switch(sign)
      {
       case '+':
	 switch(mode)
	 {
	  case 'k':
	    if (EMIT_SIGNAL(XP_CHANSETKEY, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    my_gtk_entry_set_text(sess->key_entry, extra, sess);
	    update_channelmode_buttons(sess, mode, sign);
	    if(!quiet)
	    {
	       sprintf(outbuf, STARTON" %s sets channel keyword to %s\n", nick, extra);
	       PrintText(sess, outbuf);
	    }
	    return;
	  case 'l':
	    if (EMIT_SIGNAL(XP_CHANSETLIMIT, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    my_gtk_entry_set_text(sess->limit_entry, extra, sess);
	    update_channelmode_buttons(sess, mode, sign);
	    if(!quiet)
	    {
	       sprintf(outbuf, STARTON" %s sets user limit to %d\n", nick, atoi(extra));
	       PrintText(sess, outbuf);
	    }
	    return;
	  case 'o':
	    if (EMIT_SIGNAL(XP_CHANOP, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    op_name(sess, extra);
	    sprintf(outbuf, STARTON" %s gives channel operator status to %s\n", nick, extra);
	    PrintText(sess, outbuf);
	    return;
	  case 'v':
	    if (EMIT_SIGNAL(XP_CHANVOICE, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    voice_name(sess, extra);
	    sprintf(outbuf, STARTON" %s gives voice to %s\n", nick, extra);
	    PrintText(sess, outbuf);
	    return;
	  case 'b':
	    if (EMIT_SIGNAL(XP_CHANBAN, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    sprintf(outbuf, STARTON" %s sets ban on %s\n", nick, extra);
	    PrintText(sess, outbuf);
	    return;
	 }
	 break;
       case '-':
	 switch(mode)
	 {
	   case 'k':
	    if (EMIT_SIGNAL(XP_CHANRMKEY, sess, chan, nick, NULL, NULL, 0) == 1)
	       return;
	    my_gtk_entry_set_text(sess->key_entry, "", sess);
	    update_channelmode_buttons(sess, mode, sign);
	    if(!quiet)
	    {
	       sprintf(outbuf, STARTON" %s removes channel keyword\n", nick);
	       PrintText(sess, outbuf);
	    }
	    return;
	  case 'l':
	    if (EMIT_SIGNAL(XP_CHANRMLIMIT, sess, chan, nick, NULL, NULL, 0) == 1)
	       return;
	    my_gtk_entry_set_text(sess->limit_entry, "", sess);
	    update_channelmode_buttons(sess, mode, sign);
	    if(!quiet)
	    {
	       sprintf(outbuf, STARTON" %s removes user limit\n", nick);
	       PrintText(sess, outbuf);
	    }
	    return;  
	  case 'o':
	    if (EMIT_SIGNAL(XP_CHANDEOP, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    deop_name(sess, extra);
	    sprintf(outbuf, STARTON" %s removes channel operator status from %s\n", nick, extra);
	    PrintText(sess, outbuf);
	    return;
	  case 'v':
	    if (EMIT_SIGNAL(XP_CHANDEVOICE, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    devoice_name(sess, extra);
	    sprintf(outbuf, STARTON" %s removes voice from %s\n", nick, extra);
	    PrintText(sess, outbuf);
	    return;
	  case 'b':
	    if (EMIT_SIGNAL(XP_CHANUNBAN, sess, chan, nick, extra, NULL, 0) == 1)
	       return;
	    sprintf(outbuf, STARTON" %s removes ban on %s\n", nick, extra);
	    PrintText(sess, outbuf);
	    return;  
	 }
	 break;
      }
      
      if(!quiet)
      {
	 if(extra && *extra)
	   sprintf(outbuf, STARTON" %s sets mode %c%c %s\n", nick, sign, mode, extra);
	 else
	   sprintf(outbuf, STARTON" %s sets mode %c%c %s\n", nick, sign, mode, chan);
	 PrintText(sess, outbuf);
      }

      update_channelmode_buttons(sess, mode, sign);

   } else {
      if(!quiet)
      {
	 sprintf(outbuf, STARTON" %s sets mode %c%c %s\n", nick, sign, mode, chan);
	 PrintText(serv->front_session, outbuf);
      }
   }
}

void channel_modes(struct server *serv, char *outbuf, char *word[], char *nick, int displacement)
{
   char *chan = find_word(pdibuf, 3+displacement);
   if(*chan)
   {
      char *modes = find_word(pdibuf, 4+displacement);
      if(*modes)
      {
	 int i = 5+displacement;
	 char sign;
	 if(*modes == ':') modes++; /* not a channel mode, eg +i */
	 sign = modes[0];
	 modes++;
	 while(1)
	 {
	    if(sign == '+')
	    {
	       switch(modes[0])
	       {
		case 'o':
		case 'v':
		case 'k':
		case 'l':
		case 'i':
		case 'b':
		  channel_mode(serv, outbuf, chan, nick, sign, modes[0], word[i], displacement);
		  i++;
		  break;
		default:
		  channel_mode(serv, outbuf, chan, nick, sign, modes[0], "", displacement);
	       }
	    } else {
	       switch(modes[0])
	       {
		case 'o':
		case 'v':
		case 'i':
		case 'k':
		case 'b':
		  channel_mode(serv, outbuf, chan, nick, sign, modes[0], word[i], displacement);
		  i++;
		  break;
		default:
		  channel_mode(serv, outbuf, chan, nick, sign, modes[0], "", displacement);
	       }
	    }
	    modes++;
	    if(modes[0] == 0 || modes[0] == ' ') return;
	    if(modes[0] == '-' || modes[0] == '+')
	    {
	       sign = modes[0];
	       modes++;
	       if(modes[0] == 0 || modes[0] == ' ') return;
	    }
	 }
      }
   }
}

void end_of_names(struct server *serv, char *outbuf, char *chan, char *text)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess)
   {
      sess->end_of_names = TRUE;
      sess->ignore_names = FALSE;
   }
}

void check_willjoin_channels(struct server *serv, char *tbuf)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 if(sess->willjoinchannel[0] != 0)
	 {
	    strcpy(sess->waitchannel, sess->willjoinchannel);
	    sess->willjoinchannel[0] = 0;
	    sprintf(tbuf, "JOIN %s\r\n", sess->waitchannel);
	    send(serv->sok, tbuf, strlen(tbuf), 0);
	 }
      }
      list = list->next;
   }
}

void next_nick(struct session *sess, char *outbuf, char *nick)
{
   sess->server->nickcount++;

   switch(sess->server->nickcount)
   {
    case 2:
      sprintf(outbuf, "NICK %s\r\n", prefs.nick2);
      send(sess->server->sok, outbuf, strlen(outbuf), 0);
      sprintf(outbuf, STARTON" %s already in use. Retrying with %s..\n", nick, prefs.nick2);
      PrintText(sess, outbuf);
      break;

    case 3:
      sprintf(outbuf, "NICK %s\r\n", prefs.nick3);
      send(sess->server->sok, outbuf, strlen(outbuf), 0);
      sprintf(outbuf, STARTON" %s already in use. Retrying with %s..\n", nick, prefs.nick3);
      PrintText(sess, outbuf);
      break;

    default:
      PrintText(sess, STARTON" Nickname already in use. Use /NICK to try another.\n");
   }
}

void process_line(struct session *sess, struct server *serv, char *buf)
{
   char pdibuf[4096];
   char outbuf[4096];
   char *word[32];
   char *word_eol[32];
   int n;

   add_rawlog(serv, buf);
   url_checkurl(buf);

   process_data_init(pdibuf, buf+1, word, word_eol);

#ifdef USE_PERL
   if(*buf == ':')
   {
      sess = find_session_from_channel(word[3], serv);
      if(!sess) sess = serv->front_session;
   }  
   if (perl_inbound(sess, serv, buf)) return;
#endif
   if (EMIT_SIGNAL(XP_INBOUND, sess, serv, buf, NULL, NULL, 0) == 1)
      return;

   if(*buf != ':')
   {
      if(!strncmp(buf, "NOTICE ", 7))
      {
	 buf += 7;
      }
      if(!strncmp(buf, "PING :", 6))
      {
	 sprintf(outbuf, "PONG :%s\r\n", buf+6);
	 send(serv->sok, outbuf, strlen(outbuf), 0);
	 return;
      }
      if(!strncmp(buf, "ERROR", 5))
      {
	 sprintf(outbuf, STARTON" %s\n", buf+7);
	 PrintText(serv->front_session, outbuf);
	 return;
      }
      PrintText(sess, STARTON" ");
      PrintText(sess, buf);
      PrintText(sess, "\n");
   } else {
      buf++;

      n = atoi(find_word(pdibuf, 2));
      if(n)
      {
	 struct session *realsess = 0;
	 char *text = find_word_to_end(buf, 3);
	 if(*text)
	 {
	    if(!strncasecmp(serv->nick, text, strlen(serv->nick)))
	       text += strlen(serv->nick) + 1;
	    if(*text == ':') text++;
	    switch(n)
	    {
	     case 1:
	       user_new_nick(serv, outbuf, serv->nick, word[3], TRUE);
	       set_server_name(outbuf, serv, pdibuf); goto def;
	     case 301:
	       sprintf(outbuf, STARTON" \00312[\003 %s\00312] \003 is away \00314(\003 %s\00314)\n",
		       find_word(pdibuf, 4), find_word_to_end(buf, 5)+1);
	       if(!add_to_dialog(serv, 0, find_word(pdibuf, 4), outbuf, "", 3))
		 PrintText(serv->front_session, outbuf);
	       break;
	     case 303:
	       word[4]++;
	       notify_markonline(serv->front_session, outbuf, word);
	       break;
	     case 312:
	       sprintf(outbuf, STARTON" \00312[\003 %s\00312] \003 %s\n",
		      word[4], word_eol[5]);
	       if(!add_to_dialog(serv, 0, word[4], outbuf, "", 3))
		 PrintText(serv->front_session, outbuf);
	       break;
	     case 311:
	     case 314:
	       sprintf(outbuf, STARTON" \00312[\003 %s\00312] \00314(\003 %s@%s\00314) \003 : %s\n",
		       find_word(pdibuf, 4), find_word(pdibuf, 5),
		       find_word(pdibuf, 6), find_word_to_end(buf, 8)+1);
	       if(!add_to_dialog(serv, 0, word[4], outbuf, "", 3))
		 PrintText(serv->front_session, outbuf);
	       break;
	     case 317:
	       {
		  long n = atol(find_word(pdibuf, 6));
		  long idle = atol(find_word(pdibuf, 5));
		  sprintf(outbuf, STARTON" \00312[\003 %s\00312] \003 idle \00311%02ld:%02ld:%02ld\003 , signon: \00311%s",
		       find_word(pdibuf, 4),
		       idle/3600, (idle/60)%60, idle%60,
		       ctime(&n));
		  if(!add_to_dialog(serv, 0, word[4], outbuf, "", 3))
		    PrintText(serv->front_session, outbuf);
	       }
	       break;
	     case 318:
	       sprintf(outbuf, STARTON" \00312[\003 %s\00312] \003 End of WHOIS list.\n",
		       word[4]);
	       if(!add_to_dialog(serv, 0, word[4], outbuf, "", 3))
		 PrintText(serv->front_session, outbuf);
	       break;
	     case 313:
	     case 319:
	       sprintf(outbuf, STARTON" \00312[\003 %s\00312] \003 %s\n",
		       word[4], word_eol[5]+1);
	       if(!add_to_dialog(serv, 0, word[4], outbuf, "", 3))
		 PrintText(serv->front_session, outbuf);
	       break;
	     case 321:
	       if(!sess->server->chanlist_window)
		 PrintText(sess, STARTON" Channel          Users   Topic\n"); break;
	     case 322:
	       if(sess->server->chanlist_window)
	       {
		  chanlist_addentry(sess->server, find_word(pdibuf, 4),
				    find_word(pdibuf, 5),
				    find_word_to_end(buf, 6)+1);
	       } else {
		  sprintf(outbuf, STARTON" %-16.16s %-7d %s\n",
		       find_word(pdibuf, 4),
		       atoi(find_word(pdibuf, 5)),
		       find_word_to_end(buf, 6)+1);
		  PrintText(sess, outbuf);
	       }
	       break;
	     case 323:
	       	if(!sess->server->chanlist_window) goto def;
		break;
	     case 324:
	       sess = find_session_from_channel(word[4], serv);
	       if(sess)
	       {
		  if(sess->ignore_mode)
		  {
		     sess->ignore_mode = FALSE;
		  } else {
		     sprintf(outbuf, STARTON" Channel %s modes: %s\n",
			     word[4], word_eol[5]);
		     PrintText(sess, outbuf);
		  }
		  update_channelmode_buttons(sess, 't', '-');
		  update_channelmode_buttons(sess, 'n', '-');
		  update_channelmode_buttons(sess, 's', '-');
		  update_channelmode_buttons(sess, 'i', '-');
		  update_channelmode_buttons(sess, 'p', '-');
		  update_channelmode_buttons(sess, 'm', '-');
		  update_channelmode_buttons(sess, 'l', '-');
		  update_channelmode_buttons(sess, 'k', '-');
		  channel_modes(serv, outbuf, word, word[3], 1);
	       }
	       break;
	     case 329:
	       sess = find_session_from_channel(word[4], serv);
	       if(sess)
	       {
		  if(sess->ignore_date)
		    sess->ignore_date = FALSE;
		  else
		    channel_date(sess, outbuf, word[4], word[5]);
	       }
	       break;
	     case 332:
	       topic(serv, outbuf, text); break;
	     case 333:
	       topic_nametime(serv, outbuf, find_word(pdibuf, 4), 
			      find_word(pdibuf, 5),
			      find_word(pdibuf, 6)); break;
	     case 353:
	       names_list(serv, outbuf, text); break;
	     case 366:
	       end_of_names(serv, outbuf, find_word(pdibuf, 4), text); break;
	     case 376:
	     case 422:
	       if (prefs.wallops)
	       {
		  sprintf(outbuf, "MODE %s +w\r\n", serv->nick);
		  send(serv->sok, outbuf, strlen(outbuf), 0);
	       }
	       if (prefs.servernotice)
               {
		  sprintf(outbuf, "MODE %s +s\r\n", serv->nick);
		  send(serv->sok, outbuf, strlen(outbuf), 0);
	       }
	       if (prefs.invisible)
	       {
		  sprintf(outbuf, "MODE %s +i\r\n", serv->nick);
		  send(serv->sok, outbuf, strlen(outbuf), 0);
	       }
	       check_willjoin_channels(serv, outbuf); goto def;
	     case 433:
	       next_nick(sess, outbuf, word[4]); break;
	     default:
	       def:
	       if(prefs.skipmotd && !serv->motd_skipped)
	       {
		  if(n == 375 || n == 372) return;
		  if(n == 376)
		  {
		     serv->motd_skipped = TRUE;
		     PrintText(sess, STARTON" MOTD Skipped.\n");
		     return;
		  }
	       }
	       sprintf(outbuf, STARTON" %s\n", text);
	       if(*text == '#' || *text == '&')
	       {
		  char *chan = find_word(pdibuf, 3);
		  if(!strncasecmp(serv->nick, chan, strlen(serv->nick)))
		    chan += strlen(serv->nick) + 1;
		  if(*chan == ':') chan++;
		  realsess = find_session_from_channel(chan, serv);
		  if(!realsess) realsess = sess;
		  PrintText(realsess, outbuf);
	       } else {
		  PrintText(serv->front_session, outbuf);
	       }
	    }
	 }
      } else {
	 char t;
	 char nick[64], ip[80];
	 char *po2, *po = strchr(buf, '!');

	 if(po)
	 {
	    po2 = strchr(buf, ' ');
	    if((unsigned long)po2 < (unsigned long)po) po = 0;
	 }

	 if(!po) /* SERVER Message */
	 {
	       strcpy(ip, pdibuf);
	       strcpy(nick, pdibuf);
	       goto j2;
	 }

	 if(po)
	 {
	    t = *po;
	    *po = 0;
	    strcpy(nick, buf);
	    *po = t;
	    po2 = strchr(po, ' ');
	    if(po2)
	    {
	       char *cmd;
	       t = *po2;
	       *po2 = 0;
	       strcpy(ip, po+1);
	       *po2 = t;
	       j2:	       
	       cmd = find_word(pdibuf, 2);

	       if(!strcmp("INVITE", cmd))
	       {
		  sprintf(outbuf, STARTON" You have been invited to \00311%s \003 by \00311%s\n",
			  word[4]+1, nick);
		  PrintText(sess, outbuf);
		  return;
	       }

	       if(!strcmp("JOIN", cmd))
	       {
		  if(!strcmp(nick, serv->nick))
		    you_joined(serv, outbuf, find_word_to_end(buf, 3)+1, nick, ip);
		  else
		    user_joined(serv, outbuf, find_word_to_end(buf, 3)+1, nick, ip);
		  return;
	       }

	       if(!strcmp("MODE", cmd))
	       {
		  channel_modes(serv, outbuf, word, nick, 0);
		  return;
	       }

	       if(!strcmp("NICK", cmd))
	       {
		  user_new_nick(serv, outbuf, nick, find_word_to_end(buf, 3), FALSE);
		  return;
	       }

	       if(!strcmp("NOTICE", cmd))
	       {
		  char *to = find_word(pdibuf, 3);
		  if(*to)
		  {
		     char *msg = find_word_to_end(buf, 4)+1;
		     if(*msg)
		     {
			notice(serv, outbuf, to, nick, msg, ip);
			return;
		     }
		  }
	       }

	       if(!strcmp("PART", cmd))
	       {
		  if(!strcmp(nick, serv->nick))
		    you_parted(serv, outbuf, cmd+5);
		  else
		    user_parted(serv, cmd+5, nick, ip);
		  return;
	       }

	       if(!strcmp("PRIVMSG", cmd))
	       {
		  char *to = find_word(pdibuf, 3);
		  if(*to)
		  {
		     char *msg = find_word_to_end(buf, 4)+1;
		     if(msg[0] == 1) /* CTCP */
			handle_ctcp(sess, outbuf, to, nick, msg+1, word, word_eol);
		     else {
			if(to[0] == '#' || to[0] == '&')
			   channel_msg(serv, outbuf, to, nick, msg, FALSE);
			else
			   private_msg(serv, outbuf, nick, ip, msg);
		     }
		     return;
		  }
	       }

	       if(!strcmp("PONG", cmd))
	       {
		  got_ping_reply(serv->front_session, outbuf,
				 find_word(pdibuf, 4)+1,
				 find_word(pdibuf, 3));
		  return;
	       }

	       if(!strcmp("QUIT", cmd))
	       {
		  user_quit(serv, outbuf, nick, find_word_to_end(buf, 3)+1);
		  return;
	       }

	       if(!strcmp("TOPIC", cmd))
	       {
		  new_topic(serv, outbuf, nick, find_word(pdibuf, 3),
			    find_word_to_end(buf, 4)+1);
		  return;
	       }

	       if(!strcmp("KICK", cmd))
	       {
		  char *kicked = find_word(pdibuf, 4);
		  if(*kicked)
		  {
		     if(!strcmp(kicked, serv->nick))
			 you_kicked(serv, outbuf, find_word(pdibuf, 3), nick,
				    find_word_to_end(buf, 5)+1);
		     else
		       user_kicked(serv, outbuf, find_word(pdibuf, 3), kicked,
				   nick, find_word_to_end(buf, 5)+1);
		     return;
		  }
	       }

	       if(!strcmp("KILL", cmd))
	       {
		  sprintf(outbuf,
			  STARTON" You have been killed by %s %s\n",
			  word[3], word_eol[5]);
		  PrintText(sess, outbuf);
		  return;
	       }

	       sprintf(outbuf, "(%s/%s) %s\n", nick, ip, find_word_to_end(buf, 2));
	       PrintText(sess, outbuf);
	       return;
	    }
	 }
	 sprintf(outbuf, STARTON" %s\n", buf);
	 PrintText(sess, outbuf);
      }
   }
}
