/* readonlycollection.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_COLLECTION_OBJECT (gee_collection_object_get_type ())
#define GEE_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObject))
#define GEE_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))
#define GEE_IS_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_IS_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_COLLECTION_OBJECT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))

typedef struct _GeeCollectionObject GeeCollectionObject;
typedef struct _GeeCollectionObjectClass GeeCollectionObjectClass;
typedef struct _GeeCollectionObjectPrivate GeeCollectionObjectPrivate;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_READ_ONLY_COLLECTION (gee_read_only_collection_get_type ())
#define GEE_READ_ONLY_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_READ_ONLY_COLLECTION, GeeReadOnlyCollection))
#define GEE_READ_ONLY_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_READ_ONLY_COLLECTION, GeeReadOnlyCollectionClass))
#define GEE_IS_READ_ONLY_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_READ_ONLY_COLLECTION))
#define GEE_IS_READ_ONLY_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_READ_ONLY_COLLECTION))
#define GEE_READ_ONLY_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_READ_ONLY_COLLECTION, GeeReadOnlyCollectionClass))

typedef struct _GeeReadOnlyCollection GeeReadOnlyCollection;
typedef struct _GeeReadOnlyCollectionClass GeeReadOnlyCollectionClass;
typedef struct _GeeReadOnlyCollectionPrivate GeeReadOnlyCollectionPrivate;

#define GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR (gee_read_only_collection_iterator_get_type ())
#define GEE_READ_ONLY_COLLECTION_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR, GeeReadOnlyCollectionIterator))
#define GEE_READ_ONLY_COLLECTION_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR, GeeReadOnlyCollectionIteratorClass))
#define GEE_READ_ONLY_COLLECTION_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR))
#define GEE_READ_ONLY_COLLECTION_IS_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR))
#define GEE_READ_ONLY_COLLECTION_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR, GeeReadOnlyCollectionIteratorClass))

typedef struct _GeeReadOnlyCollectionIterator GeeReadOnlyCollectionIterator;
typedef struct _GeeReadOnlyCollectionIteratorClass GeeReadOnlyCollectionIteratorClass;
typedef struct _GeeReadOnlyCollectionIteratorPrivate GeeReadOnlyCollectionIteratorPrivate;

/**
 * Base class for all collections.
 */
struct _GeeCollectionObject {
	GTypeInstance parent_instance;
	volatile int ref_count;
	GeeCollectionObjectPrivate * priv;
};

struct _GeeCollectionObjectClass {
	GTypeClass parent_class;
	void (*finalize) (GeeCollectionObject *self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GType (*get_element_type) (GeeIterable* self);
	GeeIterator* (*iterator) (GeeIterable* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gint (*get_size) (GeeCollection* self);
};

/**
 * Represents a read-only collection of items.
 */
struct _GeeReadOnlyCollection {
	GeeCollectionObject parent_instance;
	GeeReadOnlyCollectionPrivate * priv;
};

struct _GeeReadOnlyCollectionClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeReadOnlyCollectionPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	GeeCollection* _collection;
};

struct _GeeReadOnlyCollectionIterator {
	GeeCollectionObject parent_instance;
	GeeReadOnlyCollectionIteratorPrivate * priv;
};

struct _GeeReadOnlyCollectionIteratorClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeReadOnlyCollectionIteratorPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
};


static gpointer gee_read_only_collection_iterator_parent_class = NULL;
static GeeIteratorIface* gee_read_only_collection_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_read_only_collection_parent_class = NULL;
static GeeIterableIface* gee_read_only_collection_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_read_only_collection_gee_collection_parent_iface = NULL;

gpointer gee_collection_object_ref (gpointer instance);
void gee_collection_object_unref (gpointer instance);
GParamSpec* gee_param_spec_collection_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void gee_value_set_collection_object (GValue* value, gpointer v_object);
gpointer gee_value_get_collection_object (const GValue* value);
GType gee_collection_object_get_type (void);
GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_read_only_collection_get_type (void);
#define GEE_READ_ONLY_COLLECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_READ_ONLY_COLLECTION, GeeReadOnlyCollectionPrivate))
enum  {
	GEE_READ_ONLY_COLLECTION_DUMMY_PROPERTY
};
void gee_read_only_collection_set_collection (GeeReadOnlyCollection* self, GeeCollection* value);
GeeReadOnlyCollection* gee_read_only_collection_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeCollection* collection);
GeeReadOnlyCollection* gee_read_only_collection_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeCollection* collection);
static GType gee_read_only_collection_real_get_element_type (GeeIterable* base);
static GeeReadOnlyCollectionIterator* gee_read_only_collection_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func);
static GeeReadOnlyCollectionIterator* gee_read_only_collection_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func);
static GType gee_read_only_collection_iterator_get_type (void);
GeeIterator* gee_iterable_iterator (GeeIterable* self);
static GeeIterator* gee_read_only_collection_real_iterator (GeeIterable* base);
gboolean gee_collection_contains (GeeCollection* self, gconstpointer item);
static gboolean gee_read_only_collection_real_contains (GeeCollection* base, gconstpointer item);
static gboolean gee_read_only_collection_real_add (GeeCollection* base, gconstpointer item);
static gboolean gee_read_only_collection_real_remove (GeeCollection* base, gconstpointer item);
static void gee_read_only_collection_real_clear (GeeCollection* base);
gint gee_collection_get_size (GeeCollection* self);
#define GEE_READ_ONLY_COLLECTION_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR, GeeReadOnlyCollectionIteratorPrivate))
enum  {
	GEE_READ_ONLY_COLLECTION_ITERATOR_DUMMY_PROPERTY
};
static gboolean gee_read_only_collection_iterator_real_next (GeeIterator* base);
static gpointer gee_read_only_collection_iterator_real_get (GeeIterator* base);
static void gee_read_only_collection_finalize (GeeCollectionObject* obj);



GeeReadOnlyCollection* gee_read_only_collection_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeCollection* collection) {
	GeeReadOnlyCollection* self;
	self = (GeeReadOnlyCollection*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	gee_read_only_collection_set_collection (self, collection);
	return self;
}


GeeReadOnlyCollection* gee_read_only_collection_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeCollection* collection) {
	return gee_read_only_collection_construct (GEE_TYPE_READ_ONLY_COLLECTION, g_type, g_dup_func, g_destroy_func, collection);
}


static GType gee_read_only_collection_real_get_element_type (GeeIterable* base) {
	GeeReadOnlyCollection * self;
	GType result;
	self = (GeeReadOnlyCollection*) base;
	result = self->priv->g_type;
	return result;
}


static GeeIterator* gee_read_only_collection_real_iterator (GeeIterable* base) {
	GeeReadOnlyCollection * self;
	GeeIterator* result;
	self = (GeeReadOnlyCollection*) base;
	if (self->priv->_collection == NULL) {
		result = (GeeIterator*) gee_read_only_collection_iterator_new (self->priv->g_type, (GBoxedCopyFunc) self->priv->g_dup_func, self->priv->g_destroy_func);
		return result;
	}
	result = gee_iterable_iterator ((GeeIterable*) self->priv->_collection);
	return result;
}


static gboolean gee_read_only_collection_real_contains (GeeCollection* base, gconstpointer item) {
	GeeReadOnlyCollection * self;
	gboolean result;
	self = (GeeReadOnlyCollection*) base;
	if (self->priv->_collection == NULL) {
		result = FALSE;
		return result;
	}
	result = gee_collection_contains (self->priv->_collection, item);
	return result;
}


static gboolean gee_read_only_collection_real_add (GeeCollection* base, gconstpointer item) {
	GeeReadOnlyCollection * self;
	gboolean result;
	self = (GeeReadOnlyCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_read_only_collection_real_remove (GeeCollection* base, gconstpointer item) {
	GeeReadOnlyCollection * self;
	gboolean result;
	self = (GeeReadOnlyCollection*) base;
	g_assert_not_reached ();
}


static void gee_read_only_collection_real_clear (GeeCollection* base) {
	GeeReadOnlyCollection * self;
	self = (GeeReadOnlyCollection*) base;
	g_assert_not_reached ();
}


static gint gee_read_only_collection_real_get_size (GeeCollection* base) {
	gint result;
	GeeReadOnlyCollection* self;
	self = (GeeReadOnlyCollection*) base;
	result = gee_collection_get_size (self->priv->_collection);
	return result;
}


void gee_read_only_collection_set_collection (GeeReadOnlyCollection* self, GeeCollection* value) {
	GeeCollection* _tmp1_;
	GeeCollection* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_collection = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : gee_collection_object_ref (_tmp0_)), (self->priv->_collection == NULL) ? NULL : (self->priv->_collection = (gee_collection_object_unref (self->priv->_collection), NULL)), _tmp1_);
}


static gboolean gee_read_only_collection_iterator_real_next (GeeIterator* base) {
	GeeReadOnlyCollectionIterator * self;
	gboolean result;
	self = (GeeReadOnlyCollectionIterator*) base;
	result = FALSE;
	return result;
}


static gpointer gee_read_only_collection_iterator_real_get (GeeIterator* base) {
	GeeReadOnlyCollectionIterator * self;
	gpointer result;
	self = (GeeReadOnlyCollectionIterator*) base;
	result = NULL;
	return result;
}


static GeeReadOnlyCollectionIterator* gee_read_only_collection_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func) {
	GeeReadOnlyCollectionIterator* self;
	self = (GeeReadOnlyCollectionIterator*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	return self;
}


static GeeReadOnlyCollectionIterator* gee_read_only_collection_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func) {
	return gee_read_only_collection_iterator_construct (GEE_READ_ONLY_COLLECTION_TYPE_ITERATOR, g_type, g_dup_func, g_destroy_func);
}


static void gee_read_only_collection_iterator_class_init (GeeReadOnlyCollectionIteratorClass * klass) {
	gee_read_only_collection_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeReadOnlyCollectionIteratorPrivate));
}


static void gee_read_only_collection_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_read_only_collection_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_read_only_collection_iterator_real_next;
	iface->get = gee_read_only_collection_iterator_real_get;
}


static void gee_read_only_collection_iterator_instance_init (GeeReadOnlyCollectionIterator * self) {
	self->priv = GEE_READ_ONLY_COLLECTION_ITERATOR_GET_PRIVATE (self);
}


static GType gee_read_only_collection_iterator_get_type (void) {
	static GType gee_read_only_collection_iterator_type_id = 0;
	if (gee_read_only_collection_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeReadOnlyCollectionIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_read_only_collection_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeReadOnlyCollectionIterator), 0, (GInstanceInitFunc) gee_read_only_collection_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_read_only_collection_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_read_only_collection_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeReadOnlyCollectionIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_read_only_collection_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_read_only_collection_iterator_type_id;
}


static void gee_read_only_collection_class_init (GeeReadOnlyCollectionClass * klass) {
	gee_read_only_collection_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_read_only_collection_finalize;
	g_type_class_add_private (klass, sizeof (GeeReadOnlyCollectionPrivate));
}


static void gee_read_only_collection_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_read_only_collection_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_read_only_collection_real_get_element_type;
	iface->iterator = gee_read_only_collection_real_iterator;
}


static void gee_read_only_collection_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_read_only_collection_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->contains = gee_read_only_collection_real_contains;
	iface->add = gee_read_only_collection_real_add;
	iface->remove = gee_read_only_collection_real_remove;
	iface->clear = gee_read_only_collection_real_clear;
	iface->get_size = gee_read_only_collection_real_get_size;
}


static void gee_read_only_collection_instance_init (GeeReadOnlyCollection * self) {
	self->priv = GEE_READ_ONLY_COLLECTION_GET_PRIVATE (self);
}


static void gee_read_only_collection_finalize (GeeCollectionObject* obj) {
	GeeReadOnlyCollection * self;
	self = GEE_READ_ONLY_COLLECTION (obj);
	(self->priv->_collection == NULL) ? NULL : (self->priv->_collection = (gee_collection_object_unref (self->priv->_collection), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_read_only_collection_parent_class)->finalize (obj);
}


GType gee_read_only_collection_get_type (void) {
	static GType gee_read_only_collection_type_id = 0;
	if (gee_read_only_collection_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeReadOnlyCollectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_read_only_collection_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeReadOnlyCollection), 0, (GInstanceInitFunc) gee_read_only_collection_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_read_only_collection_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_read_only_collection_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_read_only_collection_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeReadOnlyCollection", &g_define_type_info, 0);
		g_type_add_interface_static (gee_read_only_collection_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_read_only_collection_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
	}
	return gee_read_only_collection_type_id;
}




