/* valaccodecompiler.vala
 *
 * Copyright (C) 2007-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <vala.h>
#include <gee.h>
#include <stdio.h>
#include <glib/gstdio.h>
#include <gobject/gvaluecollector.h>


#define VALA_TYPE_CCODE_COMPILER (vala_ccode_compiler_get_type ())
#define VALA_CCODE_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_COMPILER, ValaCCodeCompiler))
#define VALA_CCODE_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_COMPILER, ValaCCodeCompilerClass))
#define VALA_IS_CCODE_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_COMPILER))
#define VALA_IS_CCODE_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_COMPILER))
#define VALA_CCODE_COMPILER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_COMPILER, ValaCCodeCompilerClass))

typedef struct _ValaCCodeCompiler ValaCCodeCompiler;
typedef struct _ValaCCodeCompilerClass ValaCCodeCompilerClass;
typedef struct _ValaCCodeCompilerPrivate ValaCCodeCompilerPrivate;
typedef struct _ValaParamSpecCCodeCompiler ValaParamSpecCCodeCompiler;

/**
 * Interface to the C compiler.
 */
struct _ValaCCodeCompiler {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeCompilerPrivate * priv;
};

struct _ValaCCodeCompilerClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeCompiler *self);
};

struct _ValaParamSpecCCodeCompiler {
	GParamSpec parent_instance;
};


static gpointer vala_ccode_compiler_parent_class = NULL;

gpointer vala_ccode_compiler_ref (gpointer instance);
void vala_ccode_compiler_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_compiler (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_compiler (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_compiler (const GValue* value);
GType vala_ccode_compiler_get_type (void);
enum  {
	VALA_CCODE_COMPILER_DUMMY_PROPERTY
};
ValaCCodeCompiler* vala_ccode_compiler_new (void);
ValaCCodeCompiler* vala_ccode_compiler_construct (GType object_type);
static gboolean vala_ccode_compiler_package_exists (const char* package_name);
void vala_ccode_compiler_compile (ValaCCodeCompiler* self, ValaCodeContext* context, const char* cc_command, char** cc_options, int cc_options_length1);
static void vala_ccode_compiler_finalize (ValaCCodeCompiler* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaCCodeCompiler* vala_ccode_compiler_construct (GType object_type) {
	ValaCCodeCompiler* self;
	self = (ValaCCodeCompiler*) g_type_create_instance (object_type);
	return self;
}


ValaCCodeCompiler* vala_ccode_compiler_new (void) {
	return vala_ccode_compiler_construct (VALA_TYPE_CCODE_COMPILER);
}


static gboolean vala_ccode_compiler_package_exists (const char* package_name) {
	gboolean result;
	GError * _inner_error_;
	char* pc;
	gint exit_status;
	g_return_val_if_fail (package_name != NULL, FALSE);
	_inner_error_ = NULL;
	pc = g_strconcat ("pkg-config --exists ", package_name, NULL);
	exit_status = 0;
	{
		g_spawn_command_line_sync (pc, NULL, NULL, &exit_status, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_SPAWN_ERROR) {
				goto __catch0_g_spawn_error;
			}
			goto __finally0;
		}
		result = 0 == exit_status;
		pc = (g_free (pc), NULL);
		return result;
	}
	goto __finally0;
	__catch0_g_spawn_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			vala_report_error (NULL, e->message);
			result = FALSE;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			pc = (g_free (pc), NULL);
			return result;
		}
	}
	__finally0:
	if (_inner_error_ != NULL) {
		pc = (g_free (pc), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return FALSE;
	}
	pc = (g_free (pc), NULL);
}


/**
 * Compile generated C code to object code and optionally link object
 * files.
 *
 * @param context a code context
 */
void vala_ccode_compiler_compile (ValaCCodeCompiler* self, ValaCodeContext* context, const char* cc_command, char** cc_options, int cc_options_length1) {
	GError * _inner_error_;
	gboolean use_pkgconfig;
	char* pc;
	char* pkgflags;
	const char* _tmp11_;
	char* cmdline;
	GeeList* source_files;
	GeeList* c_source_files;
	char* _tmp29_;
	char* _tmp28_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_inner_error_ = NULL;
	use_pkgconfig = FALSE;
	pc = g_strdup ("pkg-config --cflags");
	if (!vala_code_context_get_compile_only (context)) {
		char* _tmp0_;
		_tmp0_ = NULL;
		pc = (_tmp0_ = g_strconcat (pc, " --libs", NULL), pc = (g_free (pc), NULL), _tmp0_);
	}
	if (vala_code_context_get_profile (context) == VALA_PROFILE_GOBJECT) {
		char* _tmp1_;
		use_pkgconfig = TRUE;
		_tmp1_ = NULL;
		pc = (_tmp1_ = g_strconcat (pc, " gobject-2.0", NULL), pc = (g_free (pc), NULL), _tmp1_);
		if (vala_code_context_get_thread (context)) {
			char* _tmp2_;
			_tmp2_ = NULL;
			pc = (_tmp2_ = g_strconcat (pc, " gthread-2.0", NULL), pc = (g_free (pc), NULL), _tmp2_);
		}
	}
	{
		GeeList* _tmp3_;
		GeeIterator* _tmp4_;
		GeeIterator* _pkg_it;
		_tmp3_ = NULL;
		_tmp4_ = NULL;
		_pkg_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = vala_code_context_get_packages (context))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), _tmp4_);
		while (TRUE) {
			char* pkg;
			if (!gee_iterator_next (_pkg_it)) {
				break;
			}
			pkg = (char*) gee_iterator_get (_pkg_it);
			if (vala_ccode_compiler_package_exists (pkg)) {
				char* _tmp6_;
				char* _tmp5_;
				use_pkgconfig = TRUE;
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				pc = (_tmp6_ = g_strconcat (pc, _tmp5_ = g_strconcat (" ", pkg, NULL), NULL), pc = (g_free (pc), NULL), _tmp6_);
				_tmp5_ = (g_free (_tmp5_), NULL);
			}
			pkg = (g_free (pkg), NULL);
		}
		(_pkg_it == NULL) ? NULL : (_pkg_it = (gee_collection_object_unref (_pkg_it), NULL));
	}
	pkgflags = g_strdup ("");
	if (use_pkgconfig) {
		{
			gint exit_status;
			char* _tmp9_;
			gboolean _tmp8_;
			char* _tmp7_;
			exit_status = 0;
			_tmp9_ = NULL;
			_tmp7_ = NULL;
			_tmp8_ = g_spawn_command_line_sync (pc, &_tmp7_, NULL, &exit_status, &_inner_error_);
			pkgflags = (_tmp9_ = _tmp7_, pkgflags = (g_free (pkgflags), NULL), _tmp9_);
			_tmp8_;
			if (_inner_error_ != NULL) {
				if (_inner_error_->domain == G_SPAWN_ERROR) {
					goto __catch1_g_spawn_error;
				}
				goto __finally1;
			}
			if (exit_status != 0) {
				char* _tmp10_;
				_tmp10_ = NULL;
				vala_report_error (NULL, _tmp10_ = g_strdup_printf ("pkg-config exited with status %d", exit_status));
				_tmp10_ = (g_free (_tmp10_), NULL);
				pc = (g_free (pc), NULL);
				pkgflags = (g_free (pkgflags), NULL);
				return;
			}
		}
		goto __finally1;
		__catch1_g_spawn_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				vala_report_error (NULL, e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				pc = (g_free (pc), NULL);
				pkgflags = (g_free (pkgflags), NULL);
				return;
			}
		}
		__finally1:
		if (_inner_error_ != NULL) {
			pc = (g_free (pc), NULL);
			pkgflags = (g_free (pkgflags), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return;
		}
	}
	/* TODO compile the C code files in parallel*/
	if (cc_command == NULL) {
		cc_command = "cc";
	}
	_tmp11_ = NULL;
	cmdline = (_tmp11_ = cc_command, (_tmp11_ == NULL) ? NULL : g_strdup (_tmp11_));
	if (vala_code_context_get_debug (context)) {
		char* _tmp12_;
		_tmp12_ = NULL;
		cmdline = (_tmp12_ = g_strconcat (cmdline, " -g", NULL), cmdline = (g_free (cmdline), NULL), _tmp12_);
	}
	if (vala_code_context_get_compile_only (context)) {
		char* _tmp13_;
		_tmp13_ = NULL;
		cmdline = (_tmp13_ = g_strconcat (cmdline, " -c", NULL), cmdline = (g_free (cmdline), NULL), _tmp13_);
	} else {
		if (vala_code_context_get_output (context) != NULL) {
			const char* _tmp14_;
			char* output;
			gboolean _tmp15_;
			gboolean _tmp16_;
			char* _tmp20_;
			char* _tmp19_;
			char* _tmp18_;
			_tmp14_ = NULL;
			output = (_tmp14_ = vala_code_context_get_output (context), (_tmp14_ == NULL) ? NULL : g_strdup (_tmp14_));
			_tmp15_ = FALSE;
			_tmp16_ = FALSE;
			if (vala_code_context_get_directory (context) != NULL) {
				_tmp16_ = _vala_strcmp0 (vala_code_context_get_directory (context), "") != 0;
			} else {
				_tmp16_ = FALSE;
			}
			if (_tmp16_) {
				_tmp15_ = !g_path_is_absolute (vala_code_context_get_output (context));
			} else {
				_tmp15_ = FALSE;
			}
			if (_tmp15_) {
				char* _tmp17_;
				_tmp17_ = NULL;
				output = (_tmp17_ = g_strdup_printf ("%s%c%s", vala_code_context_get_directory (context), (gint) G_DIR_SEPARATOR, vala_code_context_get_output (context)), output = (g_free (output), NULL), _tmp17_);
			}
			_tmp20_ = NULL;
			_tmp19_ = NULL;
			_tmp18_ = NULL;
			cmdline = (_tmp20_ = g_strconcat (cmdline, _tmp19_ = g_strconcat (" -o ", _tmp18_ = g_shell_quote (output), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp20_);
			_tmp19_ = (g_free (_tmp19_), NULL);
			_tmp18_ = (g_free (_tmp18_), NULL);
			output = (g_free (output), NULL);
		}
	}
	/* we're only interested in non-pkg source files */
	source_files = vala_code_context_get_source_files (context);
	{
		GeeIterator* _file_it;
		_file_it = gee_iterable_iterator ((GeeIterable*) source_files);
		while (TRUE) {
			ValaSourceFile* file;
			if (!gee_iterator_next (_file_it)) {
				break;
			}
			file = (ValaSourceFile*) gee_iterator_get (_file_it);
			if (!vala_source_file_get_external_package (file)) {
				char* _tmp24_;
				char* _tmp23_;
				char* _tmp22_;
				char* _tmp21_;
				_tmp24_ = NULL;
				_tmp23_ = NULL;
				_tmp22_ = NULL;
				_tmp21_ = NULL;
				cmdline = (_tmp24_ = g_strconcat (cmdline, _tmp23_ = g_strconcat (" ", _tmp22_ = g_shell_quote (_tmp21_ = vala_source_file_get_csource_filename (file)), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp24_);
				_tmp23_ = (g_free (_tmp23_), NULL);
				_tmp22_ = (g_free (_tmp22_), NULL);
				_tmp21_ = (g_free (_tmp21_), NULL);
			}
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
	}
	c_source_files = vala_code_context_get_c_source_files (context);
	{
		GeeIterator* _file_it;
		_file_it = gee_iterable_iterator ((GeeIterable*) c_source_files);
		while (TRUE) {
			char* file;
			char* _tmp27_;
			char* _tmp26_;
			char* _tmp25_;
			if (!gee_iterator_next (_file_it)) {
				break;
			}
			file = (char*) gee_iterator_get (_file_it);
			_tmp27_ = NULL;
			_tmp26_ = NULL;
			_tmp25_ = NULL;
			cmdline = (_tmp27_ = g_strconcat (cmdline, _tmp26_ = g_strconcat (" ", _tmp25_ = g_shell_quote (file), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp27_);
			_tmp26_ = (g_free (_tmp26_), NULL);
			_tmp25_ = (g_free (_tmp25_), NULL);
			file = (g_free (file), NULL);
		}
		(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
	}
	/* add libraries after source files to fix linking
	 with --as-needed and on Windows*/
	_tmp29_ = NULL;
	_tmp28_ = NULL;
	cmdline = (_tmp29_ = g_strconcat (cmdline, _tmp28_ = g_strconcat (" ", g_strstrip (pkgflags), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp29_);
	_tmp28_ = (g_free (_tmp28_), NULL);
	{
		char** cc_option_collection;
		int cc_option_collection_length1;
		int cc_option_it;
		cc_option_collection = cc_options;
		cc_option_collection_length1 = cc_options_length1;
		for (cc_option_it = 0; cc_option_it < cc_options_length1; cc_option_it = cc_option_it + 1) {
			const char* _tmp33_;
			char* cc_option;
			_tmp33_ = NULL;
			cc_option = (_tmp33_ = cc_option_collection[cc_option_it], (_tmp33_ == NULL) ? NULL : g_strdup (_tmp33_));
			{
				char* _tmp32_;
				char* _tmp31_;
				char* _tmp30_;
				_tmp32_ = NULL;
				_tmp31_ = NULL;
				_tmp30_ = NULL;
				cmdline = (_tmp32_ = g_strconcat (cmdline, _tmp31_ = g_strconcat (" ", _tmp30_ = g_shell_quote (cc_option), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp32_);
				_tmp31_ = (g_free (_tmp31_), NULL);
				_tmp30_ = (g_free (_tmp30_), NULL);
				cc_option = (g_free (cc_option), NULL);
			}
		}
	}
	if (vala_code_context_get_verbose_mode (context)) {
		fprintf (stdout, "%s\n", cmdline);
	}
	{
		gint exit_status;
		exit_status = 0;
		g_spawn_command_line_sync (cmdline, NULL, NULL, &exit_status, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_SPAWN_ERROR) {
				goto __catch2_g_spawn_error;
			}
			goto __finally2;
		}
		if (exit_status != 0) {
			char* _tmp34_;
			_tmp34_ = NULL;
			vala_report_error (NULL, _tmp34_ = g_strdup_printf ("cc exited with status %d", exit_status));
			_tmp34_ = (g_free (_tmp34_), NULL);
		}
	}
	goto __finally2;
	__catch2_g_spawn_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			vala_report_error (NULL, e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally2:
	if (_inner_error_ != NULL) {
		pc = (g_free (pc), NULL);
		pkgflags = (g_free (pkgflags), NULL);
		cmdline = (g_free (cmdline), NULL);
		(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
		(c_source_files == NULL) ? NULL : (c_source_files = (gee_collection_object_unref (c_source_files), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	/* remove generated C source and header files */
	{
		GeeIterator* _file_it;
		/* remove generated C source and header files */
		_file_it = gee_iterable_iterator ((GeeIterable*) source_files);
		/* remove generated C source and header files */
		while (TRUE) {
			ValaSourceFile* file;
			/* remove generated C source and header files */
			if (!gee_iterator_next (_file_it)) {
				/* remove generated C source and header files */
				break;
			}
			/* remove generated C source and header files */
			file = (ValaSourceFile*) gee_iterator_get (_file_it);
			if (!vala_source_file_get_external_package (file)) {
				if (!vala_code_context_get_save_csources (context)) {
					char* _tmp35_;
					_tmp35_ = NULL;
					g_unlink (_tmp35_ = vala_source_file_get_csource_filename (file));
					_tmp35_ = (g_free (_tmp35_), NULL);
				}
			}
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
	}
	pc = (g_free (pc), NULL);
	pkgflags = (g_free (pkgflags), NULL);
	cmdline = (g_free (cmdline), NULL);
	(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
	(c_source_files == NULL) ? NULL : (c_source_files = (gee_collection_object_unref (c_source_files), NULL));
}


static void vala_value_ccode_compiler_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_ccode_compiler_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_ccode_compiler_unref (value->data[0].v_pointer);
	}
}


static void vala_value_ccode_compiler_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_ccode_compiler_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_ccode_compiler_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_ccode_compiler_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCCodeCompiler* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_ccode_compiler_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_ccode_compiler_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCCodeCompiler** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_ccode_compiler_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_ccode_compiler (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecCCodeCompiler* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_CCODE_COMPILER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_ccode_compiler (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CCODE_COMPILER), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_ccode_compiler (GValue* value, gpointer v_object) {
	ValaCCodeCompiler* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CCODE_COMPILER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_CCODE_COMPILER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_ccode_compiler_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_ccode_compiler_unref (old);
	}
}


static void vala_ccode_compiler_class_init (ValaCCodeCompilerClass * klass) {
	vala_ccode_compiler_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_COMPILER_CLASS (klass)->finalize = vala_ccode_compiler_finalize;
}


static void vala_ccode_compiler_instance_init (ValaCCodeCompiler * self) {
	self->ref_count = 1;
}


static void vala_ccode_compiler_finalize (ValaCCodeCompiler* obj) {
	ValaCCodeCompiler * self;
	self = VALA_CCODE_COMPILER (obj);
}


GType vala_ccode_compiler_get_type (void) {
	static GType vala_ccode_compiler_type_id = 0;
	if (vala_ccode_compiler_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_ccode_compiler_init, vala_value_ccode_compiler_free_value, vala_value_ccode_compiler_copy_value, vala_value_ccode_compiler_peek_pointer, "p", vala_value_ccode_compiler_collect_value, "p", vala_value_ccode_compiler_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeCompiler), 0, (GInstanceInitFunc) vala_ccode_compiler_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_ccode_compiler_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCCodeCompiler", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_ccode_compiler_type_id;
}


gpointer vala_ccode_compiler_ref (gpointer instance) {
	ValaCCodeCompiler* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_ccode_compiler_unref (gpointer instance) {
	ValaCCodeCompiler* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_CCODE_COMPILER_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




