/* valasourcereference.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gobject/gvaluecollector.h>


#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;
typedef struct _ValaSourceReferencePrivate ValaSourceReferencePrivate;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;
typedef struct _ValaParamSpecSourceReference ValaParamSpecSourceReference;

/**
 * Represents a reference to a location in a source file.
 */
struct _ValaSourceReference {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaSourceReferencePrivate * priv;
};

struct _ValaSourceReferenceClass {
	GTypeClass parent_class;
	void (*finalize) (ValaSourceReference *self);
};

struct _ValaSourceReferencePrivate {
	ValaSourceFile* _file;
	gint _first_line;
	gint _first_column;
	gint _last_line;
	gint _last_column;
	char* _comment;
};

struct _ValaParamSpecSourceReference {
	GParamSpec parent_instance;
};


static gpointer vala_source_reference_parent_class = NULL;

gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
#define VALA_SOURCE_REFERENCE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferencePrivate))
enum  {
	VALA_SOURCE_REFERENCE_DUMMY_PROPERTY
};
void vala_source_reference_set_file (ValaSourceReference* self, ValaSourceFile* value);
void vala_source_reference_set_first_line (ValaSourceReference* self, gint value);
void vala_source_reference_set_first_column (ValaSourceReference* self, gint value);
void vala_source_reference_set_last_line (ValaSourceReference* self, gint value);
void vala_source_reference_set_last_column (ValaSourceReference* self, gint value);
ValaSourceReference* vala_source_reference_new (ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column);
ValaSourceReference* vala_source_reference_construct (GType object_type, ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column);
void vala_source_reference_set_comment (ValaSourceReference* self, const char* value);
ValaSourceReference* vala_source_reference_new_with_comment (ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column, const char* _comment);
ValaSourceReference* vala_source_reference_construct_with_comment (GType object_type, ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column, const char* _comment);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
char* vala_source_file_get_relative_filename (ValaSourceFile* self);
gint vala_source_reference_get_first_line (ValaSourceReference* self);
gint vala_source_reference_get_first_column (ValaSourceReference* self);
gint vala_source_reference_get_last_line (ValaSourceReference* self);
gint vala_source_reference_get_last_column (ValaSourceReference* self);
char* vala_source_reference_to_string (ValaSourceReference* self);
const char* vala_source_reference_get_comment (ValaSourceReference* self);
static void vala_source_reference_finalize (ValaSourceReference* obj);



/**
 * Creates a new source reference.
 *
 * @param file         a source file
 * @param first_line   first line number
 * @param first_column first column number
 * @param last_line    last line number
 * @param last_column  last column number
 * @return             newly created source reference
 */
ValaSourceReference* vala_source_reference_construct (GType object_type, ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column) {
	ValaSourceReference* self;
	g_return_val_if_fail (_file != NULL, NULL);
	self = (ValaSourceReference*) g_type_create_instance (object_type);
	vala_source_reference_set_file (self, _file);
	vala_source_reference_set_first_line (self, _first_line);
	vala_source_reference_set_first_column (self, _first_column);
	vala_source_reference_set_last_line (self, _last_line);
	vala_source_reference_set_last_column (self, _last_column);
	return self;
}


ValaSourceReference* vala_source_reference_new (ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column) {
	return vala_source_reference_construct (VALA_TYPE_SOURCE_REFERENCE, _file, _first_line, _first_column, _last_line, _last_column);
}


/**
 * Creates a new commented source reference.
 *
 * @param file         a source file
 * @param first_line   first line number
 * @param first_column first column number
 * @param last_line    last line number
 * @param last_column  last column number
 * @param comment      code comment
 * @return             newly created source reference
 */
ValaSourceReference* vala_source_reference_construct_with_comment (GType object_type, ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column, const char* _comment) {
	ValaSourceReference* self;
	g_return_val_if_fail (_file != NULL, NULL);
	self = (ValaSourceReference*) g_type_create_instance (object_type);
	vala_source_reference_set_file (self, _file);
	vala_source_reference_set_first_line (self, _first_line);
	vala_source_reference_set_first_column (self, _first_column);
	vala_source_reference_set_last_line (self, _last_line);
	vala_source_reference_set_last_column (self, _last_column);
	vala_source_reference_set_comment (self, _comment);
	return self;
}


ValaSourceReference* vala_source_reference_new_with_comment (ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column, const char* _comment) {
	return vala_source_reference_construct_with_comment (VALA_TYPE_SOURCE_REFERENCE, _file, _first_line, _first_column, _last_line, _last_column, _comment);
}


/**
 * Returns a string representation of this source reference.
 *
 * @return human-readable string
 */
char* vala_source_reference_to_string (ValaSourceReference* self) {
	char* result;
	char* _tmp0_;
	char* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	result = (_tmp1_ = g_strdup_printf ("%s:%d.%d-%d.%d", _tmp0_ = vala_source_file_get_relative_filename (self->priv->_file), self->priv->_first_line, self->priv->_first_column, self->priv->_last_line, self->priv->_last_column), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	return result;
}


ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self) {
	ValaSourceFile* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_file;
	return result;
}


void vala_source_reference_set_file (ValaSourceReference* self, ValaSourceFile* value) {
	g_return_if_fail (self != NULL);
	self->priv->_file = value;
}


gint vala_source_reference_get_first_line (ValaSourceReference* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_first_line;
	return result;
}


void vala_source_reference_set_first_line (ValaSourceReference* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_first_line = value;
}


gint vala_source_reference_get_first_column (ValaSourceReference* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_first_column;
	return result;
}


void vala_source_reference_set_first_column (ValaSourceReference* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_first_column = value;
}


gint vala_source_reference_get_last_line (ValaSourceReference* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_last_line;
	return result;
}


void vala_source_reference_set_last_line (ValaSourceReference* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_last_line = value;
}


gint vala_source_reference_get_last_column (ValaSourceReference* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_last_column;
	return result;
}


void vala_source_reference_set_last_column (ValaSourceReference* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_last_column = value;
}


const char* vala_source_reference_get_comment (ValaSourceReference* self) {
	const char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_comment;
	return result;
}


void vala_source_reference_set_comment (ValaSourceReference* self, const char* value) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_comment = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->_comment = (g_free (self->priv->_comment), NULL), _tmp1_);
}


static void vala_value_source_reference_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_source_reference_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_source_reference_unref (value->data[0].v_pointer);
	}
}


static void vala_value_source_reference_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_source_reference_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_source_reference_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_source_reference_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaSourceReference* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_source_reference_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_source_reference_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaSourceReference** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_source_reference_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecSourceReference* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_SOURCE_REFERENCE), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_source_reference (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SOURCE_REFERENCE), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_source_reference (GValue* value, gpointer v_object) {
	ValaSourceReference* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SOURCE_REFERENCE));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_SOURCE_REFERENCE));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_source_reference_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_source_reference_unref (old);
	}
}


static void vala_source_reference_class_init (ValaSourceReferenceClass * klass) {
	vala_source_reference_parent_class = g_type_class_peek_parent (klass);
	VALA_SOURCE_REFERENCE_CLASS (klass)->finalize = vala_source_reference_finalize;
	g_type_class_add_private (klass, sizeof (ValaSourceReferencePrivate));
}


static void vala_source_reference_instance_init (ValaSourceReference * self) {
	self->priv = VALA_SOURCE_REFERENCE_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_source_reference_finalize (ValaSourceReference* obj) {
	ValaSourceReference * self;
	self = VALA_SOURCE_REFERENCE (obj);
	self->priv->_comment = (g_free (self->priv->_comment), NULL);
}


GType vala_source_reference_get_type (void) {
	static GType vala_source_reference_type_id = 0;
	if (vala_source_reference_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_source_reference_init, vala_value_source_reference_free_value, vala_value_source_reference_copy_value, vala_value_source_reference_peek_pointer, "p", vala_value_source_reference_collect_value, "p", vala_value_source_reference_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaSourceReferenceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_source_reference_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSourceReference), 0, (GInstanceInitFunc) vala_source_reference_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_source_reference_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaSourceReference", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_source_reference_type_id;
}


gpointer vala_source_reference_ref (gpointer instance) {
	ValaSourceReference* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_source_reference_unref (gpointer instance) {
	ValaSourceReference* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_SOURCE_REFERENCE_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




