/* valainterface.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;
typedef struct _ValaTypeSymbolPrivate ValaTypeSymbolPrivate;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;
typedef struct _ValaObjectTypeSymbolPrivate ValaObjectTypeSymbolPrivate;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_SIGNAL (vala_signal_get_type ())
#define VALA_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SIGNAL, ValaSignal))
#define VALA_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SIGNAL, ValaSignalClass))
#define VALA_IS_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SIGNAL))
#define VALA_IS_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SIGNAL))
#define VALA_SIGNAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SIGNAL, ValaSignalClass))

typedef struct _ValaSignal ValaSignal;
typedef struct _ValaSignalClass ValaSignalClass;

#define VALA_TYPE_PROPERTY (vala_property_get_type ())
#define VALA_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY, ValaProperty))
#define VALA_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY, ValaPropertyClass))
#define VALA_IS_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY))
#define VALA_IS_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY))
#define VALA_PROPERTY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY, ValaPropertyClass))

typedef struct _ValaProperty ValaProperty;
typedef struct _ValaPropertyClass ValaPropertyClass;

#define VALA_TYPE_INTERFACE (vala_interface_get_type ())
#define VALA_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTERFACE, ValaInterface))
#define VALA_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTERFACE, ValaInterfaceClass))
#define VALA_IS_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTERFACE))
#define VALA_IS_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTERFACE))
#define VALA_INTERFACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTERFACE, ValaInterfaceClass))

typedef struct _ValaInterface ValaInterface;
typedef struct _ValaInterfaceClass ValaInterfaceClass;
typedef struct _ValaInterfacePrivate ValaInterfacePrivate;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;

#define VALA_TYPE_CLASS (vala_class_get_type ())
#define VALA_CLASS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CLASS, ValaClass))
#define VALA_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CLASS, ValaClassClass))
#define VALA_IS_CLASS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CLASS))
#define VALA_IS_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CLASS))
#define VALA_CLASS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CLASS, ValaClassClass))

typedef struct _ValaClass ValaClass;
typedef struct _ValaClassClass ValaClassClass;

#define VALA_TYPE_STRUCT (vala_struct_get_type ())
#define VALA_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STRUCT, ValaStruct))
#define VALA_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_STRUCT, ValaStructClass))
#define VALA_IS_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STRUCT))
#define VALA_IS_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_STRUCT))
#define VALA_STRUCT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_STRUCT, ValaStructClass))

typedef struct _ValaStruct ValaStruct;
typedef struct _ValaStructClass ValaStructClass;

#define VALA_TYPE_ENUM (vala_enum_get_type ())
#define VALA_ENUM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM, ValaEnum))
#define VALA_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM, ValaEnumClass))
#define VALA_IS_ENUM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM))
#define VALA_IS_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM))
#define VALA_ENUM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM, ValaEnumClass))

typedef struct _ValaEnum ValaEnum;
typedef struct _ValaEnumClass ValaEnumClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_CREATION_METHOD (vala_creation_method_get_type ())
#define VALA_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethod))
#define VALA_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))
#define VALA_IS_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CREATION_METHOD))
#define VALA_IS_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CREATION_METHOD))
#define VALA_CREATION_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))

typedef struct _ValaCreationMethod ValaCreationMethod;
typedef struct _ValaCreationMethodClass ValaCreationMethodClass;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_OBJECT_TYPE (vala_object_type_get_type ())
#define VALA_OBJECT_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE, ValaObjectType))
#define VALA_OBJECT_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE, ValaObjectTypeClass))
#define VALA_IS_OBJECT_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE))
#define VALA_IS_OBJECT_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE))
#define VALA_OBJECT_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE, ValaObjectTypeClass))

typedef struct _ValaObjectType ValaObjectType;
typedef struct _ValaObjectTypeClass ValaObjectTypeClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_VOID_TYPE (vala_void_type_get_type ())
#define VALA_VOID_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_VOID_TYPE, ValaVoidType))
#define VALA_VOID_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_VOID_TYPE, ValaVoidTypeClass))
#define VALA_IS_VOID_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_VOID_TYPE))
#define VALA_IS_VOID_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_VOID_TYPE))
#define VALA_VOID_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_VOID_TYPE, ValaVoidTypeClass))

typedef struct _ValaVoidType ValaVoidType;
typedef struct _ValaVoidTypeClass ValaVoidTypeClass;

#define VALA_TYPE_TYPEPARAMETER (vala_typeparameter_get_type ())
#define VALA_TYPEPARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPEPARAMETER, ValaTypeParameter))
#define VALA_TYPEPARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPEPARAMETER, ValaTypeParameterClass))
#define VALA_IS_TYPEPARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPEPARAMETER))
#define VALA_IS_TYPEPARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPEPARAMETER))
#define VALA_TYPEPARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPEPARAMETER, ValaTypeParameterClass))

typedef struct _ValaTypeParameter ValaTypeParameter;
typedef struct _ValaTypeParameterClass ValaTypeParameterClass;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
	gboolean (*is_instance_member) (ValaSymbol* self);
	gboolean (*is_class_member) (ValaSymbol* self);
};

/**
 * Represents a runtime data type. This data type may be defined in Vala source
 * code or imported from an external library with a Vala API file.
 */
struct _ValaTypeSymbol {
	ValaSymbol parent_instance;
	ValaTypeSymbolPrivate * priv;
};

struct _ValaTypeSymbolClass {
	ValaSymbolClass parent_class;
	char* (*get_cname) (ValaTypeSymbol* self, gboolean const_type);
	gboolean (*is_reference_type) (ValaTypeSymbol* self);
	char* (*get_dup_function) (ValaTypeSymbol* self);
	char* (*get_free_function) (ValaTypeSymbol* self);
	char* (*get_copy_function) (ValaTypeSymbol* self);
	char* (*get_destroy_function) (ValaTypeSymbol* self);
	gboolean (*is_reference_counting) (ValaTypeSymbol* self);
	char* (*get_ref_function) (ValaTypeSymbol* self);
	char* (*get_unref_function) (ValaTypeSymbol* self);
	char* (*get_ref_sink_function) (ValaTypeSymbol* self);
	char* (*get_type_id) (ValaTypeSymbol* self);
	char* (*get_marshaller_type_name) (ValaTypeSymbol* self);
	char* (*get_param_spec_function) (ValaTypeSymbol* self);
	char* (*get_get_value_function) (ValaTypeSymbol* self);
	char* (*get_set_value_function) (ValaTypeSymbol* self);
	char* (*get_upper_case_cname) (ValaTypeSymbol* self, const char* infix);
	char* (*get_default_value) (ValaTypeSymbol* self);
	gboolean (*is_subtype_of) (ValaTypeSymbol* self, ValaTypeSymbol* t);
	gint (*get_type_parameter_index) (ValaTypeSymbol* self, const char* name);
	char* (*get_type_signature) (ValaTypeSymbol* self);
};

/**
 * Represents a runtime data type for objects and interfaces. This data type may
 * be defined in Vala source code or imported from an external library with a 
 * Vala API file.
 */
struct _ValaObjectTypeSymbol {
	ValaTypeSymbol parent_instance;
	ValaObjectTypeSymbolPrivate * priv;
};

struct _ValaObjectTypeSymbolClass {
	ValaTypeSymbolClass parent_class;
	GeeList* (*get_methods) (ValaObjectTypeSymbol* self);
	GeeList* (*get_signals) (ValaObjectTypeSymbol* self);
	GeeList* (*get_properties) (ValaObjectTypeSymbol* self);
};

/**
 * Represents a class declaration in the source code.
 */
struct _ValaInterface {
	ValaObjectTypeSymbol parent_instance;
	ValaInterfacePrivate * priv;
};

struct _ValaInterfaceClass {
	ValaObjectTypeSymbolClass parent_class;
};

struct _ValaInterfacePrivate {
	GeeList* prerequisites;
	GeeList* methods;
	GeeList* fields;
	GeeList* properties;
	GeeList* signals;
	GeeList* classes;
	GeeList* structs;
	GeeList* enums;
	GeeList* delegates;
	char* cname;
	char* lower_case_csuffix;
	char* type_cname;
	char* type_id;
};

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;


static gpointer vala_interface_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_object_type_symbol_get_type (void);
GType vala_member_get_type (void);
GType vala_method_get_type (void);
GType vala_signal_get_type (void);
GType vala_property_get_type (void);
GType vala_interface_get_type (void);
GType vala_field_get_type (void);
GType vala_class_get_type (void);
GType vala_struct_get_type (void);
GType vala_enum_get_type (void);
GType vala_delegate_get_type (void);
#define VALA_INTERFACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE, ValaInterfacePrivate))
enum  {
	VALA_INTERFACE_DUMMY_PROPERTY
};
GeeList* vala_interface_get_classes (ValaInterface* self);
GeeList* vala_interface_get_structs (ValaInterface* self);
GeeList* vala_interface_get_enums (ValaInterface* self);
GeeList* vala_interface_get_delegates (ValaInterface* self);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaObjectTypeSymbol* vala_object_type_symbol_new (const char* name, ValaSourceReference* source_reference);
ValaObjectTypeSymbol* vala_object_type_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaInterface* vala_interface_new (const char* name, ValaSourceReference* source_reference);
ValaInterface* vala_interface_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
void vala_interface_add_prerequisite (ValaInterface* self, ValaDataType* type);
void vala_interface_prepend_prerequisite (ValaInterface* self, ValaDataType* type);
GeeList* vala_interface_get_prerequisites (ValaInterface* self);
GType vala_creation_method_get_type (void);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
GType member_binding_get_type (void);
MemberBinding vala_method_get_binding (ValaMethod* self);
ValaObjectType* vala_object_type_new (ValaObjectTypeSymbol* type_symbol);
ValaObjectType* vala_object_type_construct (GType object_type, ValaObjectTypeSymbol* type_symbol);
GType vala_reference_type_get_type (void);
GType vala_object_type_get_type (void);
ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
GType vala_formal_parameter_get_type (void);
void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym);
ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self);
const char* vala_symbol_get_name (ValaSymbol* self);
ValaDataType* vala_method_get_return_type (ValaMethod* self);
GType vala_void_type_get_type (void);
GeeList* vala_method_get_postconditions (ValaMethod* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
ValaLocalVariable* vala_local_variable_new (ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference);
ValaLocalVariable* vala_local_variable_construct (GType object_type, ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference);
void vala_method_set_result_var (ValaMethod* self, ValaLocalVariable* value);
void vala_local_variable_set_is_result (ValaLocalVariable* self, gboolean value);
ValaLocalVariable* vala_method_get_result_var (ValaMethod* self);
void vala_interface_add_method (ValaInterface* self, ValaMethod* m);
static GeeList* vala_interface_real_get_methods (ValaObjectTypeSymbol* base);
void vala_interface_add_field (ValaInterface* self, ValaField* f);
GeeList* vala_interface_get_fields (ValaInterface* self);
void vala_property_set_this_parameter (ValaProperty* self, ValaFormalParameter* value);
ValaFormalParameter* vala_property_get_this_parameter (ValaProperty* self);
void vala_interface_add_property (ValaInterface* self, ValaProperty* prop);
static GeeList* vala_interface_real_get_properties (ValaObjectTypeSymbol* base);
void vala_interface_add_signal (ValaInterface* self, ValaSignal* sig);
static GeeList* vala_interface_real_get_signals (ValaObjectTypeSymbol* base);
void vala_interface_add_class (ValaInterface* self, ValaClass* cl);
void vala_interface_add_struct (ValaInterface* self, ValaStruct* st);
void vala_interface_add_enum (ValaInterface* self, ValaEnum* en);
void vala_interface_add_delegate (ValaInterface* self, ValaDelegate* d);
ValaAttribute* vala_code_node_get_attribute (ValaCodeNode* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
char* vala_symbol_get_cprefix (ValaSymbol* self);
static char* vala_interface_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
void vala_interface_set_cname (ValaInterface* self, const char* cname);
char* vala_interface_get_default_lower_case_csuffix (ValaInterface* self);
char* vala_interface_get_lower_case_csuffix (ValaInterface* self);
char* vala_symbol_camel_case_to_lower_case (const char* camel_case);
void vala_interface_set_lower_case_csuffix (ValaInterface* self, const char* csuffix);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
static char* vala_interface_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix);
static char* vala_interface_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_interface_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
void vala_code_visitor_visit_interface (ValaCodeVisitor* self, ValaInterface* iface);
static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
GType vala_typeparameter_get_type (void);
GeeList* vala_object_type_symbol_get_type_parameters (ValaObjectTypeSymbol* self);
static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_interface_real_is_reference_type (ValaTypeSymbol* base);
static gboolean vala_interface_real_is_reference_counting (ValaTypeSymbol* base);
ValaTypeSymbol* vala_data_type_get_data_type (ValaDataType* self);
char* vala_typesymbol_get_ref_function (ValaTypeSymbol* self);
static char* vala_interface_real_get_ref_function (ValaTypeSymbol* base);
char* vala_typesymbol_get_unref_function (ValaTypeSymbol* self);
static char* vala_interface_real_get_unref_function (ValaTypeSymbol* base);
char* vala_typesymbol_get_ref_sink_function (ValaTypeSymbol* self);
static char* vala_interface_real_get_ref_sink_function (ValaTypeSymbol* base);
gboolean vala_typesymbol_is_subtype_of (ValaTypeSymbol* self, ValaTypeSymbol* t);
static gboolean vala_interface_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
void vala_interface_set_type_cname (ValaInterface* self, const char* type_cname);
void vala_typesymbol_add_cheader_filename (ValaTypeSymbol* self, const char* filename);
static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_interface_process_attributes (ValaInterface* self);
char* vala_typesymbol_get_cname (ValaTypeSymbol* self, gboolean const_type);
char* vala_interface_get_type_cname (ValaInterface* self);
static char* vala_interface_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_interface_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_interface_real_get_set_value_function (ValaTypeSymbol* base);
char* vala_typesymbol_get_upper_case_cname (ValaTypeSymbol* self, const char* infix);
static char* vala_interface_real_get_type_id (ValaTypeSymbol* base);
static void vala_interface_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
ValaObjectTypeSymbol* vala_object_type_get_type_symbol (ValaObjectType* self);
char* vala_typesymbol_get_param_spec_function (ValaTypeSymbol* self);
static char* vala_interface_real_get_param_spec_function (ValaTypeSymbol* base);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_semantic_analyzer_get_current_source_file (ValaSemanticAnalyzer* self);
ValaSymbol* vala_semantic_analyzer_get_current_symbol (ValaSemanticAnalyzer* self);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
void vala_semantic_analyzer_set_current_source_file (ValaSemanticAnalyzer* self, ValaSourceFile* value);
void vala_semantic_analyzer_set_current_symbol (ValaSemanticAnalyzer* self, ValaSymbol* value);
gboolean vala_semantic_analyzer_is_type_accessible (ValaSemanticAnalyzer* self, ValaSymbol* sym, ValaDataType* type);
char* vala_code_node_to_string (ValaCodeNode* self);
char* vala_symbol_get_full_name (ValaSymbol* self);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_interface_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_interface_finalize (ValaCodeNode* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Returns a copy of the list of classes.
 *
 * @return list of classes
 */
GeeList* vala_interface_get_classes (ValaInterface* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->classes);
	return result;
}


/**
 * Returns a copy of the list of structs.
 *
 * @return list of structs
 */
GeeList* vala_interface_get_structs (ValaInterface* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->structs);
	return result;
}


/**
 * Returns a copy of the list of enums.
 *
 * @return list of enums
 */
GeeList* vala_interface_get_enums (ValaInterface* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->enums);
	return result;
}


/**
 * Returns a copy of the list of delegates.
 *
 * @return list of delegates
 */
GeeList* vala_interface_get_delegates (ValaInterface* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->delegates);
	return result;
}


/**
 * Creates a new interface.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created interface
 */
ValaInterface* vala_interface_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaInterface* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaInterface*) vala_object_type_symbol_construct (object_type, name, source_reference);
	return self;
}


ValaInterface* vala_interface_new (const char* name, ValaSourceReference* source_reference) {
	return vala_interface_construct (VALA_TYPE_INTERFACE, name, source_reference);
}


/**
 * Adds the specified interface or class to the list of prerequisites of
 * this interface.
 *
 * @param type an interface or class reference
 */
void vala_interface_add_prerequisite (ValaInterface* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	gee_collection_add ((GeeCollection*) self->priv->prerequisites, type);
	vala_code_node_set_parent_node ((ValaCodeNode*) type, (ValaCodeNode*) self);
}


/**
 * Prepends the specified interface or class to the list of
 * prerequisites of this interface.
 *
 * @param type an interface or class reference
 */
void vala_interface_prepend_prerequisite (ValaInterface* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	gee_list_insert (self->priv->prerequisites, 0, type);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeList* vala_interface_get_prerequisites (ValaInterface* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->prerequisites);
	return result;
}


/**
 * Adds the specified method as a member to this interface.
 *
 * @param m a method
 */
void vala_interface_add_method (ValaInterface* self, ValaMethod* m) {
	gboolean _tmp2_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaFormalParameter* _tmp1_;
		ValaObjectType* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		vala_method_set_this_parameter (m, _tmp1_ = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0_ = vala_object_type_new ((ValaObjectTypeSymbol*) self)), NULL));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL));
		(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)), (ValaSymbol*) vala_method_get_this_parameter (m));
	}
	_tmp2_ = FALSE;
	if (!VALA_IS_VOID_TYPE (vala_method_get_return_type (m))) {
		GeeList* _tmp3_;
		_tmp3_ = NULL;
		_tmp2_ = gee_collection_get_size ((GeeCollection*) (_tmp3_ = vala_method_get_postconditions (m))) > 0;
		(_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL));
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		ValaLocalVariable* _tmp5_;
		ValaDataType* _tmp4_;
		_tmp5_ = NULL;
		_tmp4_ = NULL;
		vala_method_set_result_var (m, _tmp5_ = vala_local_variable_new (_tmp4_ = vala_data_type_copy (vala_method_get_return_type (m)), "result", NULL, NULL));
		(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
		(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
		vala_local_variable_set_is_result (vala_method_get_result_var (m), TRUE);
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
static GeeList* vala_interface_real_get_methods (ValaObjectTypeSymbol* base) {
	ValaInterface * self;
	GeeList* result;
	self = (ValaInterface*) base;
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
	return result;
}


/**
 * Adds the specified field as a member to this interface. The field
 * must be private and static.
 *
 * @param f a field
 */
void vala_interface_add_field (ValaInterface* self, ValaField* f) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	gee_collection_add ((GeeCollection*) self->priv->fields, f);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) f), (ValaSymbol*) f);
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeList* vala_interface_get_fields (ValaInterface* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->fields);
	return result;
}


/**
 * Adds the specified property as a member to this interface.
 *
 * @param prop a property
 */
void vala_interface_add_property (ValaInterface* self, ValaProperty* prop) {
	ValaFormalParameter* _tmp1_;
	ValaObjectType* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (prop != NULL);
	gee_collection_add ((GeeCollection*) self->priv->properties, prop);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) prop), (ValaSymbol*) prop);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	vala_property_set_this_parameter (prop, _tmp1_ = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0_ = vala_object_type_new ((ValaObjectTypeSymbol*) self)), NULL));
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL));
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) prop), vala_symbol_get_name ((ValaSymbol*) vala_property_get_this_parameter (prop)), (ValaSymbol*) vala_property_get_this_parameter (prop));
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
static GeeList* vala_interface_real_get_properties (ValaObjectTypeSymbol* base) {
	ValaInterface * self;
	GeeList* result;
	self = (ValaInterface*) base;
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_PROPERTY, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->properties);
	return result;
}


/**
 * Adds the specified signal as a member to this interface.
 *
 * @param sig a signal
 */
void vala_interface_add_signal (ValaInterface* self, ValaSignal* sig) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (sig != NULL);
	gee_collection_add ((GeeCollection*) self->priv->signals, sig);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) sig), (ValaSymbol*) sig);
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
static GeeList* vala_interface_real_get_signals (ValaObjectTypeSymbol* base) {
	ValaInterface * self;
	GeeList* result;
	self = (ValaInterface*) base;
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->signals);
	return result;
}


/**
 * Adds the specified class as an inner class.
 *
 * @param cl a class
 */
void vala_interface_add_class (ValaInterface* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	gee_collection_add ((GeeCollection*) self->priv->classes, cl);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) cl), (ValaSymbol*) cl);
}


/**
 * Adds the specified struct as an inner struct.
 *
 * @param st a struct
 */
void vala_interface_add_struct (ValaInterface* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	gee_collection_add ((GeeCollection*) self->priv->structs, st);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) st), (ValaSymbol*) st);
}


/**
 * Adds the specified enum as an inner enum.
 *
 * @param en an enum
 */
void vala_interface_add_enum (ValaInterface* self, ValaEnum* en) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
	gee_collection_add ((GeeCollection*) self->priv->enums, en);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) en), (ValaSymbol*) en);
}


/**
 * Adds the specified delegate as an inner delegate.
 *
 * @param d a delegate
 */
void vala_interface_add_delegate (ValaInterface* self, ValaDelegate* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
	gee_collection_add ((GeeCollection*) self->priv->delegates, d);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) d), (ValaSymbol*) d);
}


static char* vala_interface_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaInterface * self;
	char* result;
	const char* _tmp3_;
	self = (ValaInterface*) base;
	if (self->priv->cname == NULL) {
		ValaAttribute* attr;
		attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
		if (attr != NULL) {
			char* _tmp0_;
			_tmp0_ = NULL;
			self->priv->cname = (_tmp0_ = vala_attribute_get_string (attr, "cname"), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0_);
		}
		if (self->priv->cname == NULL) {
			char* _tmp2_;
			char* _tmp1_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			self->priv->cname = (_tmp2_ = g_strdup_printf ("%s%s", _tmp1_ = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp2_);
			_tmp1_ = (g_free (_tmp1_), NULL);
		}
		(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
	}
	_tmp3_ = NULL;
	result = (_tmp3_ = self->priv->cname, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_));
	return result;
}


void vala_interface_set_cname (ValaInterface* self, const char* cname) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->cname = (_tmp1_ = (_tmp0_ = cname, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1_);
}


/**
 * Returns the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @return the suffix to be used in C code
 */
char* vala_interface_get_lower_case_csuffix (ValaInterface* self) {
	char* result;
	const char* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0_;
		_tmp0_ = NULL;
		self->priv->lower_case_csuffix = (_tmp0_ = vala_interface_get_default_lower_case_csuffix (self), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp0_);
	}
	_tmp1_ = NULL;
	result = (_tmp1_ = self->priv->lower_case_csuffix, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
	return result;
}


static char* string_substring (const char* self, glong offset, glong len) {
	char* result;
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	result = g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
	return result;
}


/**
 * Returns default string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @return the suffix to be used in C code
 */
char* vala_interface_get_default_lower_case_csuffix (ValaInterface* self) {
	char* result;
	char* _result_;
	g_return_val_if_fail (self != NULL, NULL);
	_result_ = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self));
	/* remove underscores in some cases to avoid conflicts of type macros*/
	if (g_str_has_prefix (_result_, "type_")) {
		char* _tmp0_;
		_tmp0_ = NULL;
		_result_ = (_tmp0_ = g_strconcat ("type", g_utf8_offset_to_pointer (_result_, g_utf8_strlen ("type_", -1)), NULL), _result_ = (g_free (_result_), NULL), _tmp0_);
	} else {
		if (g_str_has_prefix (_result_, "is_")) {
			char* _tmp1_;
			_tmp1_ = NULL;
			_result_ = (_tmp1_ = g_strconcat ("is", g_utf8_offset_to_pointer (_result_, g_utf8_strlen ("is_", -1)), NULL), _result_ = (g_free (_result_), NULL), _tmp1_);
		}
	}
	if (g_str_has_suffix (_result_, "_class")) {
		char* _tmp3_;
		char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_result_ = (_tmp3_ = g_strconcat (_tmp2_ = string_substring (_result_, (glong) 0, g_utf8_strlen (_result_, -1) - g_utf8_strlen ("_class", -1)), "class", NULL), _result_ = (g_free (_result_), NULL), _tmp3_);
		_tmp2_ = (g_free (_tmp2_), NULL);
	}
	result = _result_;
	return result;
}


/**
 * Sets the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @param csuffix the suffix to be used in C code
 */
void vala_interface_set_lower_case_csuffix (ValaInterface* self, const char* csuffix) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (csuffix != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->lower_case_csuffix = (_tmp1_ = (_tmp0_ = csuffix, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp1_);
}


static char* vala_interface_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaInterface * self;
	char* result;
	char* _tmp1_;
	char* _tmp0_;
	char* _tmp2_;
	self = (ValaInterface*) base;
	if (infix == NULL) {
		infix = "";
	}
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	result = (_tmp2_ = g_strdup_printf ("%s%s%s", _tmp0_ = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), infix, _tmp1_ = vala_interface_get_lower_case_csuffix (self)), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	return result;
}


static char* vala_interface_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaInterface * self;
	char* result;
	char* _tmp0_;
	char* _tmp1_;
	self = (ValaInterface*) base;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	result = (_tmp1_ = g_strdup_printf ("%s_", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	return result;
}


static char* vala_interface_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaInterface * self;
	char* result;
	char* _tmp0_;
	char* _tmp1_;
	self = (ValaInterface*) base;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	result = (_tmp1_ = g_utf8_strup (_tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, infix), -1), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	return result;
}


static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_interface (visitor, self);
}


static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _type_it;
		_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (TRUE) {
			ValaDataType* type;
			if (!gee_iterator_next (_type_it)) {
				break;
			}
			type = (ValaDataType*) gee_iterator_get (_type_it);
			vala_code_node_accept ((ValaCodeNode*) type, visitor);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	{
		GeeList* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _p_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_p_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) self))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (TRUE) {
			ValaTypeParameter* p;
			if (!gee_iterator_next (_p_it)) {
				break;
			}
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_accept ((ValaCodeNode*) p, visitor);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeIterator* _en_it;
		/* process enums first to avoid order problems in C code */
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		/* process enums first to avoid order problems in C code */
		while (TRUE) {
			ValaEnum* en;
			/* process enums first to avoid order problems in C code */
			if (!gee_iterator_next (_en_it)) {
				/* process enums first to avoid order problems in C code */
				break;
			}
			/* process enums first to avoid order problems in C code */
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_accept ((ValaCodeNode*) en, visitor);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (TRUE) {
			ValaMethod* m;
			if (!gee_iterator_next (_m_it)) {
				break;
			}
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (TRUE) {
			ValaField* f;
			if (!gee_iterator_next (_f_it)) {
				break;
			}
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_accept ((ValaCodeNode*) f, visitor);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _prop_it;
		_prop_it = gee_iterable_iterator ((GeeIterable*) self->priv->properties);
		while (TRUE) {
			ValaProperty* prop;
			if (!gee_iterator_next (_prop_it)) {
				break;
			}
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			vala_code_node_accept ((ValaCodeNode*) prop, visitor);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	{
		GeeIterator* _sig_it;
		_sig_it = gee_iterable_iterator ((GeeIterable*) self->priv->signals);
		while (TRUE) {
			ValaSignal* sig;
			if (!gee_iterator_next (_sig_it)) {
				break;
			}
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			vala_code_node_accept ((ValaCodeNode*) sig, visitor);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (TRUE) {
			ValaClass* cl;
			if (!gee_iterator_next (_cl_it)) {
				break;
			}
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_accept ((ValaCodeNode*) cl, visitor);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (TRUE) {
			ValaStruct* st;
			if (!gee_iterator_next (_st_it)) {
				break;
			}
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_accept ((ValaCodeNode*) st, visitor);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (TRUE) {
			ValaDelegate* d;
			if (!gee_iterator_next (_d_it)) {
				break;
			}
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_accept ((ValaCodeNode*) d, visitor);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
}


static gboolean vala_interface_real_is_reference_type (ValaTypeSymbol* base) {
	ValaInterface * self;
	gboolean result;
	self = (ValaInterface*) base;
	result = TRUE;
	return result;
}


static gboolean vala_interface_real_is_reference_counting (ValaTypeSymbol* base) {
	ValaInterface * self;
	gboolean result;
	self = (ValaInterface*) base;
	result = TRUE;
	return result;
}


static char* vala_interface_real_get_ref_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	self = (ValaInterface*) base;
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (TRUE) {
			ValaDataType* prerequisite;
			char* ref_func;
			if (!gee_iterator_next (_prerequisite_it)) {
				break;
			}
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			ref_func = vala_typesymbol_get_ref_function (vala_data_type_get_data_type (prerequisite));
			if (ref_func != NULL) {
				result = ref_func;
				(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
				(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
				return result;
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			ref_func = (g_free (ref_func), NULL);
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	result = NULL;
	return result;
}


static char* vala_interface_real_get_unref_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	self = (ValaInterface*) base;
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (TRUE) {
			ValaDataType* prerequisite;
			char* unref_func;
			if (!gee_iterator_next (_prerequisite_it)) {
				break;
			}
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			unref_func = vala_typesymbol_get_unref_function (vala_data_type_get_data_type (prerequisite));
			if (unref_func != NULL) {
				result = unref_func;
				(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
				(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
				return result;
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			unref_func = (g_free (unref_func), NULL);
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	result = NULL;
	return result;
}


static char* vala_interface_real_get_ref_sink_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	self = (ValaInterface*) base;
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (TRUE) {
			ValaDataType* prerequisite;
			char* ref_sink_func;
			if (!gee_iterator_next (_prerequisite_it)) {
				break;
			}
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			ref_sink_func = vala_typesymbol_get_ref_sink_function (vala_data_type_get_data_type (prerequisite));
			if (ref_sink_func != NULL) {
				result = ref_sink_func;
				(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
				(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
				return result;
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			ref_sink_func = (g_free (ref_sink_func), NULL);
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	result = NULL;
	return result;
}


static gboolean vala_interface_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t) {
	ValaInterface * self;
	gboolean result;
	self = (ValaInterface*) base;
	g_return_val_if_fail (t != NULL, FALSE);
	if (VALA_TYPESYMBOL (self) == t) {
		result = TRUE;
		return result;
	}
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (TRUE) {
			ValaDataType* prerequisite;
			gboolean _tmp0_;
			if (!gee_iterator_next (_prerequisite_it)) {
				break;
			}
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			_tmp0_ = FALSE;
			if (vala_data_type_get_data_type (prerequisite) != NULL) {
				_tmp0_ = vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (prerequisite), t);
			} else {
				_tmp0_ = FALSE;
			}
			if (_tmp0_) {
				result = TRUE;
				(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
				(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
				return result;
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	result = FALSE;
	return result;
}


static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "type_cname")) {
		char* _tmp0_;
		_tmp0_ = NULL;
		vala_interface_set_type_cname (self, _tmp0_ = vala_attribute_get_string (a, "type_cname"));
		_tmp0_ = (g_free (_tmp0_), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp1_;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp1_ = NULL;
			filename_collection = _tmp1_ = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp1_);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp1_); filename_it = filename_it + 1) {
				const char* _tmp2_;
				char* filename;
				_tmp2_ = NULL;
				filename = (_tmp2_ = filename_collection[filename_it], (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp3_;
		_tmp3_ = NULL;
		self->priv->lower_case_csuffix = (_tmp3_ = vala_attribute_get_string (a, "lower_case_csuffix"), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp3_);
	}
}


/**
 * Process all associated attributes.
 */
void vala_interface_process_attributes (ValaInterface* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0_;
			ValaAttribute* a;
			_tmp0_ = NULL;
			a = (_tmp0_ = (ValaAttribute*) a_it->data, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_interface_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


/**
 * Returns the name of the type struct as it is used in C code.
 *
 * @return the type struct name to be used in C code
 */
char* vala_interface_get_type_cname (ValaInterface* self) {
	char* result;
	const char* _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->type_cname == NULL) {
		char* _tmp1_;
		char* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		self->priv->type_cname = (_tmp1_ = g_strdup_printf ("%sIface", _tmp0_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) self, FALSE)), self->priv->type_cname = (g_free (self->priv->type_cname), NULL), _tmp1_);
		_tmp0_ = (g_free (_tmp0_), NULL);
	}
	_tmp2_ = NULL;
	result = (_tmp2_ = self->priv->type_cname, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
	return result;
}


/**
 * Sets the name of the type struct as it is used in C code.
 *
 * @param type_cname the type struct name to be used in C code
 */
void vala_interface_set_type_cname (ValaInterface* self, const char* type_cname) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type_cname != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->type_cname = (_tmp1_ = (_tmp0_ = type_cname, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->type_cname = (g_free (self->priv->type_cname), NULL), _tmp1_);
}


static char* vala_interface_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	self = (ValaInterface*) base;
	result = g_strdup ("OBJECT");
	return result;
}


static char* vala_interface_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	self = (ValaInterface*) base;
	result = g_strdup ("g_value_get_object");
	return result;
}


static char* vala_interface_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	self = (ValaInterface*) base;
	result = g_strdup ("g_value_set_object");
	return result;
}


static char* vala_interface_real_get_type_id (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	const char* _tmp1_;
	self = (ValaInterface*) base;
	if (self->priv->type_id == NULL) {
		char* _tmp0_;
		_tmp0_ = NULL;
		self->priv->type_id = (_tmp0_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, "TYPE_"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp0_);
	}
	_tmp1_ = NULL;
	result = (_tmp1_ = self->priv->type_id, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
	return result;
}


static void vala_interface_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	{
		gint i;
		i = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				ValaDataType* _tmp1_;
				gboolean _tmp2_;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < gee_collection_get_size ((GeeCollection*) self->priv->prerequisites))) {
					break;
				}
				_tmp1_ = NULL;
				if ((_tmp2_ = (_tmp1_ = (ValaDataType*) gee_list_get (self->priv->prerequisites, i)) == old_type, (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL)), _tmp2_)) {
					gee_list_set (self->priv->prerequisites, i, new_type);
					return;
				}
			}
		}
	}
}


static char* vala_interface_real_get_param_spec_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	char* result;
	self = (ValaInterface*) base;
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (TRUE) {
			ValaDataType* prerequisite;
			ValaObjectType* _tmp1_;
			ValaDataType* _tmp0_;
			ValaObjectType* prereq;
			ValaClass* _tmp3_;
			ValaObjectTypeSymbol* _tmp2_;
			ValaClass* cl;
			ValaInterface* _tmp5_;
			ValaObjectTypeSymbol* _tmp4_;
			ValaInterface* interf;
			if (!gee_iterator_next (_prerequisite_it)) {
				break;
			}
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			prereq = (_tmp1_ = (_tmp0_ = prerequisite, VALA_IS_OBJECT_TYPE (_tmp0_) ? ((ValaObjectType*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			cl = (_tmp3_ = (_tmp2_ = vala_object_type_get_type_symbol (prereq), VALA_IS_CLASS (_tmp2_) ? ((ValaClass*) _tmp2_) : NULL), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
			if (cl != NULL) {
				result = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) cl);
				(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
				(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
				(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
				return result;
			}
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			interf = (_tmp5_ = (_tmp4_ = vala_object_type_get_type_symbol (prereq), VALA_IS_INTERFACE (_tmp4_) ? ((ValaInterface*) _tmp4_) : NULL), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
			if (interf != NULL) {
				char* param_spec_function;
				param_spec_function = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) interf);
				if (param_spec_function != NULL) {
					result = param_spec_function;
					(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
					(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
					(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
					(interf == NULL) ? NULL : (interf = (vala_code_node_unref (interf), NULL));
					(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
					return result;
				}
				param_spec_function = (g_free (param_spec_function), NULL);
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			(interf == NULL) ? NULL : (interf = (vala_code_node_unref (interf), NULL));
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	result = NULL;
	return result;
}


static gboolean vala_interface_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaInterface * self;
	gboolean result;
	ValaSourceFile* _tmp0_;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp1_;
	ValaSymbol* old_symbol;
	ValaClass* prereq_class;
	self = (ValaInterface*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_interface_process_attributes (self);
	_tmp0_ = NULL;
	old_source_file = (_tmp0_ = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp0_ == NULL) ? NULL : vala_source_file_ref (_tmp0_));
	_tmp1_ = NULL;
	old_symbol = (_tmp1_ = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	{
		GeeList* _tmp2_;
		GeeIterator* _tmp3_;
		GeeIterator* _prerequisite_reference_it;
		_tmp2_ = NULL;
		_tmp3_ = NULL;
		_prerequisite_reference_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = vala_interface_get_prerequisites (self))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp3_);
		while (TRUE) {
			ValaDataType* prerequisite_reference;
			if (!gee_iterator_next (_prerequisite_reference_it)) {
				break;
			}
			prerequisite_reference = (ValaDataType*) gee_iterator_get (_prerequisite_reference_it);
			/* check whether prerequisite is at least as accessible as the interface*/
			if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, prerequisite_reference)) {
				char* _tmp6_;
				char* _tmp5_;
				char* _tmp4_;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp6_ = g_strdup_printf ("prerequisite `%s` is less accessible than interface `%s`", _tmp4_ = vala_code_node_to_string ((ValaCodeNode*) prerequisite_reference), _tmp5_ = vala_symbol_get_full_name ((ValaSymbol*) self)));
				_tmp6_ = (g_free (_tmp6_), NULL);
				_tmp5_ = (g_free (_tmp5_), NULL);
				_tmp4_ = (g_free (_tmp4_), NULL);
				result = FALSE;
				(prerequisite_reference == NULL) ? NULL : (prerequisite_reference = (vala_code_node_unref (prerequisite_reference), NULL));
				(_prerequisite_reference_it == NULL) ? NULL : (_prerequisite_reference_it = (gee_collection_object_unref (_prerequisite_reference_it), NULL));
				(old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL));
				(old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL));
				return result;
			}
			(prerequisite_reference == NULL) ? NULL : (prerequisite_reference = (vala_code_node_unref (prerequisite_reference), NULL));
		}
		(_prerequisite_reference_it == NULL) ? NULL : (_prerequisite_reference_it = (gee_collection_object_unref (_prerequisite_reference_it), NULL));
	}
	/* check prerequisites */
	prereq_class = NULL;
	{
		GeeList* _tmp7_;
		GeeIterator* _tmp8_;
		GeeIterator* _prereq_it;
		_tmp7_ = NULL;
		_tmp8_ = NULL;
		_prereq_it = (_tmp8_ = gee_iterable_iterator ((GeeIterable*) (_tmp7_ = vala_interface_get_prerequisites (self))), (_tmp7_ == NULL) ? NULL : (_tmp7_ = (gee_collection_object_unref (_tmp7_), NULL)), _tmp8_);
		while (TRUE) {
			ValaDataType* prereq;
			ValaTypeSymbol* _tmp9_;
			ValaTypeSymbol* class_or_interface;
			if (!gee_iterator_next (_prereq_it)) {
				break;
			}
			prereq = (ValaDataType*) gee_iterator_get (_prereq_it);
			_tmp9_ = NULL;
			class_or_interface = (_tmp9_ = vala_data_type_get_data_type (prereq), (_tmp9_ == NULL) ? NULL : vala_code_node_ref (_tmp9_));
			/* skip on previous errors */
			if (class_or_interface == NULL) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
				(class_or_interface == NULL) ? NULL : (class_or_interface = (vala_code_node_unref (class_or_interface), NULL));
				continue;
			}
			if (!VALA_IS_OBJECT_TYPE_SYMBOL (class_or_interface)) {
				char* _tmp12_;
				char* _tmp11_;
				char* _tmp10_;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp12_ = NULL;
				_tmp11_ = NULL;
				_tmp10_ = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp12_ = g_strdup_printf ("Prerequisite `%s` of interface `%s` is not a class or interface", _tmp10_ = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp11_ = vala_code_node_to_string ((ValaCodeNode*) class_or_interface)));
				_tmp12_ = (g_free (_tmp12_), NULL);
				_tmp11_ = (g_free (_tmp11_), NULL);
				_tmp10_ = (g_free (_tmp10_), NULL);
				result = FALSE;
				(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
				(class_or_interface == NULL) ? NULL : (class_or_interface = (vala_code_node_unref (class_or_interface), NULL));
				(_prereq_it == NULL) ? NULL : (_prereq_it = (gee_collection_object_unref (_prereq_it), NULL));
				(old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL));
				(old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL));
				(prereq_class == NULL) ? NULL : (prereq_class = (vala_code_node_unref (prereq_class), NULL));
				return result;
			}
			/* interfaces are not allowed to have multiple instantiable prerequisites */
			if (VALA_IS_CLASS (class_or_interface)) {
				ValaClass* _tmp18_;
				ValaClass* _tmp17_;
				if (prereq_class != NULL) {
					char* _tmp16_;
					char* _tmp15_;
					char* _tmp14_;
					char* _tmp13_;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp16_ = NULL;
					_tmp15_ = NULL;
					_tmp14_ = NULL;
					_tmp13_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp16_ = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", _tmp13_ = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp14_ = vala_symbol_get_full_name ((ValaSymbol*) class_or_interface), _tmp15_ = vala_symbol_get_full_name ((ValaSymbol*) prereq_class)));
					_tmp16_ = (g_free (_tmp16_), NULL);
					_tmp15_ = (g_free (_tmp15_), NULL);
					_tmp14_ = (g_free (_tmp14_), NULL);
					_tmp13_ = (g_free (_tmp13_), NULL);
					result = FALSE;
					(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
					(class_or_interface == NULL) ? NULL : (class_or_interface = (vala_code_node_unref (class_or_interface), NULL));
					(_prereq_it == NULL) ? NULL : (_prereq_it = (gee_collection_object_unref (_prereq_it), NULL));
					(old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL));
					(old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL));
					(prereq_class == NULL) ? NULL : (prereq_class = (vala_code_node_unref (prereq_class), NULL));
					return result;
				}
				_tmp18_ = NULL;
				_tmp17_ = NULL;
				prereq_class = (_tmp18_ = (_tmp17_ = VALA_CLASS (class_or_interface), (_tmp17_ == NULL) ? NULL : vala_code_node_ref (_tmp17_)), (prereq_class == NULL) ? NULL : (prereq_class = (vala_code_node_unref (prereq_class), NULL)), _tmp18_);
			}
			(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
			(class_or_interface == NULL) ? NULL : (class_or_interface = (vala_code_node_unref (class_or_interface), NULL));
		}
		(_prereq_it == NULL) ? NULL : (_prereq_it = (gee_collection_object_unref (_prereq_it), NULL));
	}
	{
		GeeIterator* _type_it;
		_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (TRUE) {
			ValaDataType* type;
			if (!gee_iterator_next (_type_it)) {
				break;
			}
			type = (ValaDataType*) gee_iterator_get (_type_it);
			vala_code_node_check ((ValaCodeNode*) type, analyzer);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	{
		GeeList* _tmp19_;
		GeeIterator* _tmp20_;
		GeeIterator* _p_it;
		_tmp19_ = NULL;
		_tmp20_ = NULL;
		_p_it = (_tmp20_ = gee_iterable_iterator ((GeeIterable*) (_tmp19_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) self))), (_tmp19_ == NULL) ? NULL : (_tmp19_ = (gee_collection_object_unref (_tmp19_), NULL)), _tmp20_);
		while (TRUE) {
			ValaTypeParameter* p;
			if (!gee_iterator_next (_p_it)) {
				break;
			}
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_check ((ValaCodeNode*) p, analyzer);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	{
		GeeIterator* _en_it;
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		while (TRUE) {
			ValaEnum* en;
			if (!gee_iterator_next (_en_it)) {
				break;
			}
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_check ((ValaCodeNode*) en, analyzer);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (TRUE) {
			ValaMethod* m;
			if (!gee_iterator_next (_m_it)) {
				break;
			}
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_check ((ValaCodeNode*) m, analyzer);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (TRUE) {
			ValaField* f;
			if (!gee_iterator_next (_f_it)) {
				break;
			}
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_check ((ValaCodeNode*) f, analyzer);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _prop_it;
		_prop_it = gee_iterable_iterator ((GeeIterable*) self->priv->properties);
		while (TRUE) {
			ValaProperty* prop;
			if (!gee_iterator_next (_prop_it)) {
				break;
			}
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			vala_code_node_check ((ValaCodeNode*) prop, analyzer);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	{
		GeeIterator* _sig_it;
		_sig_it = gee_iterable_iterator ((GeeIterable*) self->priv->signals);
		while (TRUE) {
			ValaSignal* sig;
			if (!gee_iterator_next (_sig_it)) {
				break;
			}
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			vala_code_node_check ((ValaCodeNode*) sig, analyzer);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (TRUE) {
			ValaClass* cl;
			if (!gee_iterator_next (_cl_it)) {
				break;
			}
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_check ((ValaCodeNode*) cl, analyzer);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (TRUE) {
			ValaStruct* st;
			if (!gee_iterator_next (_st_it)) {
				break;
			}
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_check ((ValaCodeNode*) st, analyzer);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (TRUE) {
			ValaDelegate* d;
			if (!gee_iterator_next (_d_it)) {
				break;
			}
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_check ((ValaCodeNode*) d, analyzer);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	(old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL));
	(old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL));
	(prereq_class == NULL) ? NULL : (prereq_class = (vala_code_node_unref (prereq_class), NULL));
	return result;
}


static void vala_interface_class_init (ValaInterfaceClass * klass) {
	vala_interface_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_interface_finalize;
	g_type_class_add_private (klass, sizeof (ValaInterfacePrivate));
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_methods = vala_interface_real_get_methods;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_properties = vala_interface_real_get_properties;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_signals = vala_interface_real_get_signals;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_interface_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_interface_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_interface_real_get_lower_case_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_interface_real_get_upper_case_cname;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_interface_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_interface_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_interface_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_counting = vala_interface_real_is_reference_counting;
	VALA_TYPESYMBOL_CLASS (klass)->get_ref_function = vala_interface_real_get_ref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_unref_function = vala_interface_real_get_unref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_ref_sink_function = vala_interface_real_get_ref_sink_function;
	VALA_TYPESYMBOL_CLASS (klass)->is_subtype_of = vala_interface_real_is_subtype_of;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_interface_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_interface_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_interface_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_interface_real_get_type_id;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_interface_real_replace_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_param_spec_function = vala_interface_real_get_param_spec_function;
	VALA_CODE_NODE_CLASS (klass)->check = vala_interface_real_check;
}


static void vala_interface_instance_init (ValaInterface * self) {
	self->priv = VALA_INTERFACE_GET_PRIVATE (self);
	self->priv->prerequisites = (GeeList*) gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->fields = (GeeList*) gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->properties = (GeeList*) gee_array_list_new (VALA_TYPE_PROPERTY, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->signals = (GeeList*) gee_array_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->classes = (GeeList*) gee_array_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->structs = (GeeList*) gee_array_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->enums = (GeeList*) gee_array_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->delegates = (GeeList*) gee_array_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_interface_finalize (ValaCodeNode* obj) {
	ValaInterface * self;
	self = VALA_INTERFACE (obj);
	(self->priv->prerequisites == NULL) ? NULL : (self->priv->prerequisites = (gee_collection_object_unref (self->priv->prerequisites), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	(self->priv->fields == NULL) ? NULL : (self->priv->fields = (gee_collection_object_unref (self->priv->fields), NULL));
	(self->priv->properties == NULL) ? NULL : (self->priv->properties = (gee_collection_object_unref (self->priv->properties), NULL));
	(self->priv->signals == NULL) ? NULL : (self->priv->signals = (gee_collection_object_unref (self->priv->signals), NULL));
	(self->priv->classes == NULL) ? NULL : (self->priv->classes = (gee_collection_object_unref (self->priv->classes), NULL));
	(self->priv->structs == NULL) ? NULL : (self->priv->structs = (gee_collection_object_unref (self->priv->structs), NULL));
	(self->priv->enums == NULL) ? NULL : (self->priv->enums = (gee_collection_object_unref (self->priv->enums), NULL));
	(self->priv->delegates == NULL) ? NULL : (self->priv->delegates = (gee_collection_object_unref (self->priv->delegates), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	self->priv->type_cname = (g_free (self->priv->type_cname), NULL);
	self->priv->type_id = (g_free (self->priv->type_id), NULL);
	VALA_CODE_NODE_CLASS (vala_interface_parent_class)->finalize (obj);
}


GType vala_interface_get_type (void) {
	static GType vala_interface_type_id = 0;
	if (vala_interface_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterface), 0, (GInstanceInitFunc) vala_interface_instance_init, NULL };
		vala_interface_type_id = g_type_register_static (VALA_TYPE_OBJECT_TYPE_SYMBOL, "ValaInterface", &g_define_type_info, 0);
	}
	return vala_interface_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




