/* valagirwriter.vala
 *
 * Copyright (C) 2008-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <gee.h>


#define VALA_TYPE_GIR_WRITER (vala_gir_writer_get_type ())
#define VALA_GIR_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GIR_WRITER, ValaGIRWriter))
#define VALA_GIR_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GIR_WRITER, ValaGIRWriterClass))
#define VALA_IS_GIR_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GIR_WRITER))
#define VALA_IS_GIR_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GIR_WRITER))
#define VALA_GIR_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GIR_WRITER, ValaGIRWriterClass))

typedef struct _ValaGIRWriter ValaGIRWriter;
typedef struct _ValaGIRWriterClass ValaGIRWriterClass;
typedef struct _ValaGIRWriterPrivate ValaGIRWriterPrivate;

/**
 * Code visitor generating .gir file for the public interface.
 */
struct _ValaGIRWriter {
	ValaCodeVisitor parent_instance;
	ValaGIRWriterPrivate * priv;
};

struct _ValaGIRWriterClass {
	ValaCodeVisitorClass parent_class;
};

struct _ValaGIRWriterPrivate {
	ValaCodeContext* context;
	char* directory;
	char* gir_namespace;
	char* gir_version;
	FILE* stream;
	gint indent;
	ValaTypeSymbol* gobject_type;
	gint enum_value;
};


static gpointer vala_gir_writer_parent_class = NULL;

GType vala_gir_writer_get_type (void);
#define VALA_GIR_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIR_WRITER, ValaGIRWriterPrivate))
enum  {
	VALA_GIR_WRITER_DUMMY_PROPERTY
};
static void vala_gir_writer_write_package (ValaGIRWriter* self, const char* package);
void vala_gir_writer_write_file (ValaGIRWriter* self, ValaCodeContext* context, const char* directory, const char* gir_namespace, const char* gir_version, const char* package);
static void vala_gir_writer_write_indent (ValaGIRWriter* self);
static void vala_gir_writer_write_c_include (ValaGIRWriter* self, const char* name);
static void vala_gir_writer_write_c_includes (ValaGIRWriter* self, ValaNamespace* ns);
static void vala_gir_writer_write_annotations (ValaGIRWriter* self, ValaCodeNode* node);
static void vala_gir_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static gboolean vala_gir_writer_check_accessibility (ValaGIRWriter* self, ValaSymbol* sym);
static void vala_gir_writer_write_gtype_attributes (ValaGIRWriter* self, ValaTypeSymbol* symbol);
static void vala_gir_writer_write_ctype_attributes (ValaGIRWriter* self, ValaTypeSymbol* symbol, const char* suffix);
static void vala_gir_writer_write_signature (ValaGIRWriter* self, ValaMethod* m, const char* tag_name, gboolean instance);
static void vala_gir_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_gir_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_gir_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_gir_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static char* vala_gir_writer_literal_expression_to_value_string (ValaGIRWriter* self, ValaExpression* literal);
static void vala_gir_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_gir_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain);
static void vala_gir_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode);
static void vala_gir_writer_write_type (ValaGIRWriter* self, ValaDataType* type);
static void vala_gir_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_gir_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_gir_writer_write_params (ValaGIRWriter* self, GeeList* params, ValaDataType* instance_type);
static void vala_gir_writer_write_return_type (ValaGIRWriter* self, ValaDataType* type, gboolean constructor);
static void vala_gir_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb);
static void vala_gir_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
ValaDataType* vala_ccode_base_module_get_data_type_for_symbol (ValaTypeSymbol* sym);
static void vala_gir_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_gir_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_gir_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static char* vala_gir_writer_gi_type_name (ValaGIRWriter* self, ValaTypeSymbol* type_symbol);
static char* vala_gir_writer_camel_case_to_canonical (ValaGIRWriter* self, const char* name);
static char* vala_gir_writer_vala_to_gi_type_name (ValaGIRWriter* self, const char* name);
ValaGIRWriter* vala_gir_writer_new (void);
ValaGIRWriter* vala_gir_writer_construct (GType object_type);
ValaGIRWriter* vala_gir_writer_new (void);
static void vala_gir_writer_finalize (ValaCodeVisitor* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_gir_writer_write_file (ValaGIRWriter* self, ValaCodeContext* context, const char* directory, const char* gir_namespace, const char* gir_version, const char* package) {
	ValaCodeContext* _tmp1_;
	ValaCodeContext* _tmp0_;
	char* _tmp3_;
	const char* _tmp2_;
	char* _tmp5_;
	const char* _tmp4_;
	char* _tmp7_;
	const char* _tmp6_;
	ValaNamespace* _tmp8_;
	ValaNamespace* root_symbol;
	ValaSymbol* glib_ns;
	ValaTypeSymbol* _tmp9_;
	char* filename;
	FILE* _tmp10_;
	FILE* _tmp11_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	g_return_if_fail (directory != NULL);
	g_return_if_fail (gir_namespace != NULL);
	g_return_if_fail (gir_version != NULL);
	g_return_if_fail (package != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->context = (_tmp1_ = (_tmp0_ = context, (_tmp0_ == NULL) ? NULL : vala_code_context_ref (_tmp0_)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1_);
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	self->priv->directory = (_tmp3_ = (_tmp2_ = directory, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), self->priv->directory = (g_free (self->priv->directory), NULL), _tmp3_);
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	self->priv->gir_namespace = (_tmp5_ = (_tmp4_ = gir_namespace, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), self->priv->gir_namespace = (g_free (self->priv->gir_namespace), NULL), _tmp5_);
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	self->priv->gir_version = (_tmp7_ = (_tmp6_ = gir_version, (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_)), self->priv->gir_version = (g_free (self->priv->gir_version), NULL), _tmp7_);
	_tmp8_ = NULL;
	root_symbol = (_tmp8_ = vala_code_context_get_root (context), (_tmp8_ == NULL) ? NULL : vala_code_node_ref (_tmp8_));
	glib_ns = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) root_symbol), "GLib");
	_tmp9_ = NULL;
	self->priv->gobject_type = (_tmp9_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Object")), (self->priv->gobject_type == NULL) ? NULL : (self->priv->gobject_type = (vala_code_node_unref (self->priv->gobject_type), NULL)), _tmp9_);
	filename = g_strdup_printf ("%s%c%s-%s.gir", directory, (gint) G_DIR_SEPARATOR, gir_namespace, gir_version);
	_tmp10_ = NULL;
	self->priv->stream = (_tmp10_ = fopen (filename, "w"), (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp10_);
	fprintf (self->priv->stream, "<?xml version=\"1.0\"?>\n");
	fprintf (self->priv->stream, "<repository version=\"1.0\"");
	fprintf (self->priv->stream, " xmlns=\"http://www.gtk.org/introspection/core/1.0\"");
	fprintf (self->priv->stream, " xmlns:c=\"http://www.gtk.org/introspection/c/1.0\"");
	fprintf (self->priv->stream, " xmlns:glib=\"http://www.gtk.org/introspection/glib/1.0\"");
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_package (self, package);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
	self->priv->indent--;
	fprintf (self->priv->stream, "</repository>\n");
	_tmp11_ = NULL;
	self->priv->stream = (_tmp11_ = NULL, (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp11_);
	(root_symbol == NULL) ? NULL : (root_symbol = (vala_code_node_unref (root_symbol), NULL));
	(glib_ns == NULL) ? NULL : (glib_ns = (vala_code_node_unref (glib_ns), NULL));
	filename = (g_free (filename), NULL);
}


static void vala_gir_writer_write_package (ValaGIRWriter* self, const char* package) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (package != NULL);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<package name=\"%s\"/>\n", package);
}


static void vala_gir_writer_write_c_includes (ValaGIRWriter* self, ValaNamespace* ns) {
	GeeSet* header_filenames;
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
	/* Collect C header filenames*/
	header_filenames = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	{
		GeeList* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _c_header_filename_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_c_header_filename_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = vala_symbol_get_cheader_filenames ((ValaSymbol*) ns))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (TRUE) {
			char* c_header_filename;
			if (!gee_iterator_next (_c_header_filename_it)) {
				break;
			}
			c_header_filename = (char*) gee_iterator_get (_c_header_filename_it);
			gee_collection_add ((GeeCollection*) header_filenames, c_header_filename);
			c_header_filename = (g_free (c_header_filename), NULL);
		}
		(_c_header_filename_it == NULL) ? NULL : (_c_header_filename_it = (gee_collection_object_unref (_c_header_filename_it), NULL));
	}
	{
		GeeCollection* _tmp3_;
		GeeMap* _tmp2_;
		GeeIterator* _tmp4_;
		GeeIterator* _symbol_it;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_tmp4_ = NULL;
		_symbol_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = gee_map_get_values (_tmp2_ = vala_scope_get_symbol_table (vala_symbol_get_scope ((ValaSymbol*) ns))))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp4_);
		while (TRUE) {
			ValaSymbol* symbol;
			if (!gee_iterator_next (_symbol_it)) {
				break;
			}
			symbol = (ValaSymbol*) gee_iterator_get (_symbol_it);
			{
				GeeList* _tmp5_;
				GeeIterator* _tmp6_;
				GeeIterator* _c_header_filename_it;
				_tmp5_ = NULL;
				_tmp6_ = NULL;
				_c_header_filename_it = (_tmp6_ = gee_iterable_iterator ((GeeIterable*) (_tmp5_ = vala_symbol_get_cheader_filenames (symbol))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (gee_collection_object_unref (_tmp5_), NULL)), _tmp6_);
				while (TRUE) {
					char* c_header_filename;
					if (!gee_iterator_next (_c_header_filename_it)) {
						break;
					}
					c_header_filename = (char*) gee_iterator_get (_c_header_filename_it);
					gee_collection_add ((GeeCollection*) header_filenames, c_header_filename);
					c_header_filename = (g_free (c_header_filename), NULL);
				}
				(_c_header_filename_it == NULL) ? NULL : (_c_header_filename_it = (gee_collection_object_unref (_c_header_filename_it), NULL));
			}
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_symbol_it == NULL) ? NULL : (_symbol_it = (gee_collection_object_unref (_symbol_it), NULL));
	}
	/* Generate c:include tags*/
	{
		GeeIterator* _c_header_filename_it;
		/* Generate c:include tags*/
		_c_header_filename_it = gee_iterable_iterator ((GeeIterable*) header_filenames);
		/* Generate c:include tags*/
		while (TRUE) {
			char* c_header_filename;
			/* Generate c:include tags*/
			if (!gee_iterator_next (_c_header_filename_it)) {
				/* Generate c:include tags*/
				break;
			}
			/* Generate c:include tags*/
			c_header_filename = (char*) gee_iterator_get (_c_header_filename_it);
			vala_gir_writer_write_c_include (self, c_header_filename);
			c_header_filename = (g_free (c_header_filename), NULL);
		}
		(_c_header_filename_it == NULL) ? NULL : (_c_header_filename_it = (gee_collection_object_unref (_c_header_filename_it), NULL));
	}
	(header_filenames == NULL) ? NULL : (header_filenames = (gee_collection_object_unref (header_filenames), NULL));
}


static void vala_gir_writer_write_c_include (ValaGIRWriter* self, const char* name) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<c:include name=\"%s\"/>\n", name);
}


static void vala_gir_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaGIRWriter * self;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (ns != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) ns)) {
		return;
	}
	if (vala_symbol_get_name ((ValaSymbol*) ns) == NULL) {
		/* global namespace*/
		vala_code_node_accept_children ((ValaCodeNode*) ns, (ValaCodeVisitor*) self);
		return;
	}
	if (vala_symbol_get_name (vala_symbol_get_parent_symbol ((ValaSymbol*) ns)) != NULL) {
		/* nested namespace
		 not supported in GIR at the moment*/
		return;
	}
	vala_gir_writer_write_c_includes (self, ns);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<namespace name=\"%s\" version=\"%s\">\n", self->priv->gir_namespace, self->priv->gir_version);
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) ns);
	vala_code_node_accept_children ((ValaCodeNode*) ns, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</namespace>\n");
}


static void vala_gir_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaGIRWriter * self;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (cl != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) cl)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) cl)) {
		return;
	}
	if (vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) cl, self->priv->gobject_type)) {
		char* gtype_struct_name;
		char* _tmp0_;
		gboolean first;
		gtype_struct_name = g_strconcat (vala_symbol_get_name ((ValaSymbol*) cl), "Class", NULL);
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<class name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) cl));
		vala_gir_writer_write_gtype_attributes (self, (ValaTypeSymbol*) cl);
		fprintf (self->priv->stream, " glib:type-struct=\"%s\"", gtype_struct_name);
		_tmp0_ = NULL;
		fprintf (self->priv->stream, " parent=\"%s\"", _tmp0_ = vala_symbol_get_full_name ((ValaSymbol*) vala_class_get_base_class (cl)));
		_tmp0_ = (g_free (_tmp0_), NULL);
		if (vala_class_get_is_abstract (cl)) {
			fprintf (self->priv->stream, " abstract=\"1\"");
		}
		fprintf (self->priv->stream, ">\n");
		self->priv->indent++;
		/* write implemented interfaces*/
		first = TRUE;
		{
			GeeList* _tmp1_;
			GeeIterator* _tmp2_;
			GeeIterator* _base_type_it;
			_tmp1_ = NULL;
			_tmp2_ = NULL;
			_base_type_it = (_tmp2_ = gee_iterable_iterator ((GeeIterable*) (_tmp1_ = vala_class_get_base_types (cl))), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (gee_collection_object_unref (_tmp1_), NULL)), _tmp2_);
			while (TRUE) {
				ValaDataType* base_type;
				ValaObjectType* _tmp3_;
				ValaObjectType* object_type;
				if (!gee_iterator_next (_base_type_it)) {
					break;
				}
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				_tmp3_ = NULL;
				object_type = (_tmp3_ = VALA_OBJECT_TYPE (base_type), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
				if (VALA_IS_INTERFACE (vala_object_type_get_type_symbol (object_type))) {
					char* _tmp4_;
					if (first) {
						vala_gir_writer_write_indent (self);
						fprintf (self->priv->stream, "<implements>\n");
						self->priv->indent++;
						first = FALSE;
					}
					vala_gir_writer_write_indent (self);
					_tmp4_ = NULL;
					fprintf (self->priv->stream, "<interface name=\"%s\"/>\n", _tmp4_ = vala_symbol_get_full_name ((ValaSymbol*) vala_object_type_get_type_symbol (object_type)));
					_tmp4_ = (g_free (_tmp4_), NULL);
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
		if (!first) {
			self->priv->indent--;
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "</implements>\n");
		}
		vala_gir_writer_write_annotations (self, (ValaCodeNode*) cl);
		vala_code_node_accept_children ((ValaCodeNode*) cl, (ValaCodeVisitor*) self);
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</class>\n");
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<record name=\"%s\"", gtype_struct_name);
		vala_gir_writer_write_ctype_attributes (self, (ValaTypeSymbol*) cl, "Class");
		fprintf (self->priv->stream, " glib:is-gtype-struct-for=\"%s\"", vala_symbol_get_name ((ValaSymbol*) cl));
		fprintf (self->priv->stream, ">\n");
		self->priv->indent++;
		{
			GeeList* _tmp5_;
			GeeIterator* _tmp6_;
			GeeIterator* _m_it;
			_tmp5_ = NULL;
			_tmp6_ = NULL;
			_m_it = (_tmp6_ = gee_iterable_iterator ((GeeIterable*) (_tmp5_ = vala_object_type_symbol_get_methods ((ValaObjectTypeSymbol*) cl))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (gee_collection_object_unref (_tmp5_), NULL)), _tmp6_);
			while (TRUE) {
				ValaMethod* m;
				gboolean _tmp7_;
				if (!gee_iterator_next (_m_it)) {
					break;
				}
				m = (ValaMethod*) gee_iterator_get (_m_it);
				_tmp7_ = FALSE;
				if (vala_method_get_is_abstract (m)) {
					_tmp7_ = TRUE;
				} else {
					_tmp7_ = vala_method_get_is_virtual (m);
				}
				if (_tmp7_) {
					vala_gir_writer_write_signature (self, m, "callback", TRUE);
				}
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			}
			(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
		}
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</record>\n");
		gtype_struct_name = (g_free (gtype_struct_name), NULL);
	} else {
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<record name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) cl));
		fprintf (self->priv->stream, ">\n");
		self->priv->indent++;
		vala_gir_writer_write_annotations (self, (ValaCodeNode*) cl);
		vala_code_node_accept_children ((ValaCodeNode*) cl, (ValaCodeVisitor*) self);
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</record>\n");
	}
}


static void vala_gir_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaGIRWriter * self;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (st != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) st)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) st)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<record name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) st));
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) st);
	vala_code_node_accept_children ((ValaCodeNode*) st, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</record>\n");
}


static void vala_gir_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaGIRWriter * self;
	char* gtype_struct_name;
	GeeList* _tmp0_;
	gboolean _tmp1_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (iface != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) iface)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) iface)) {
		return;
	}
	gtype_struct_name = g_strconcat (vala_symbol_get_name ((ValaSymbol*) iface), "Iface", NULL);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<interface name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) iface));
	vala_gir_writer_write_gtype_attributes (self, (ValaTypeSymbol*) iface);
	fprintf (self->priv->stream, " glib:type-struct=\"%s\"", gtype_struct_name);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	/* write prerequisites*/
	_tmp0_ = NULL;
	if ((_tmp1_ = gee_collection_get_size ((GeeCollection*) (_tmp0_ = vala_interface_get_prerequisites (iface))) > 0, (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_)) {
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<requires>\n");
		self->priv->indent++;
		{
			GeeList* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _base_type_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_base_type_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = vala_interface_get_prerequisites (iface))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp3_);
			while (TRUE) {
				ValaDataType* base_type;
				ValaObjectType* _tmp4_;
				ValaObjectType* object_type;
				if (!gee_iterator_next (_base_type_it)) {
					break;
				}
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				_tmp4_ = NULL;
				object_type = (_tmp4_ = VALA_OBJECT_TYPE (base_type), (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
				if (VALA_IS_CLASS (vala_object_type_get_type_symbol (object_type))) {
					char* _tmp5_;
					vala_gir_writer_write_indent (self);
					_tmp5_ = NULL;
					fprintf (self->priv->stream, "<object name=\"%s\"/>\n", _tmp5_ = vala_symbol_get_full_name ((ValaSymbol*) vala_object_type_get_type_symbol (object_type)));
					_tmp5_ = (g_free (_tmp5_), NULL);
				} else {
					if (VALA_IS_INTERFACE (vala_object_type_get_type_symbol (object_type))) {
						char* _tmp6_;
						vala_gir_writer_write_indent (self);
						_tmp6_ = NULL;
						fprintf (self->priv->stream, "<interface name=\"%s\"/>\n", _tmp6_ = vala_symbol_get_full_name ((ValaSymbol*) vala_object_type_get_type_symbol (object_type)));
						_tmp6_ = (g_free (_tmp6_), NULL);
					} else {
						g_assert_not_reached ();
					}
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</requires>\n");
	}
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) iface);
	vala_code_node_accept_children ((ValaCodeNode*) iface, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</interface>\n");
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<record name=\"%s\"", gtype_struct_name);
	vala_gir_writer_write_ctype_attributes (self, (ValaTypeSymbol*) iface, "Iface");
	fprintf (self->priv->stream, " glib:is-gtype-struct-for=\"%s\"", vala_symbol_get_name ((ValaSymbol*) iface));
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	{
		GeeList* _tmp7_;
		GeeIterator* _tmp8_;
		GeeIterator* _m_it;
		_tmp7_ = NULL;
		_tmp8_ = NULL;
		_m_it = (_tmp8_ = gee_iterable_iterator ((GeeIterable*) (_tmp7_ = vala_object_type_symbol_get_methods ((ValaObjectTypeSymbol*) iface))), (_tmp7_ == NULL) ? NULL : (_tmp7_ = (gee_collection_object_unref (_tmp7_), NULL)), _tmp8_);
		while (TRUE) {
			ValaMethod* m;
			gboolean _tmp9_;
			if (!gee_iterator_next (_m_it)) {
				break;
			}
			m = (ValaMethod*) gee_iterator_get (_m_it);
			_tmp9_ = FALSE;
			if (vala_method_get_is_abstract (m)) {
				_tmp9_ = TRUE;
			} else {
				_tmp9_ = vala_method_get_is_virtual (m);
			}
			if (_tmp9_) {
				vala_gir_writer_write_signature (self, m, "callback", TRUE);
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</record>\n");
	gtype_struct_name = (g_free (gtype_struct_name), NULL);
}


static void vala_gir_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaGIRWriter * self;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (en != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) en)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) en)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<enumeration name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) en));
	vala_gir_writer_write_gtype_attributes (self, (ValaTypeSymbol*) en);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) en);
	self->priv->enum_value = 0;
	vala_code_node_accept_children ((ValaCodeNode*) en, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</enumeration>\n");
}


static void vala_gir_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev) {
	ValaGIRWriter * self;
	char* _tmp1_;
	char* _tmp0_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (ev != NULL);
	vala_gir_writer_write_indent (self);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	fprintf (self->priv->stream, "<member name=\"%s\" c:identifier=\"%s\"", _tmp0_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) ev), -1), _tmp1_ = vala_enum_value_get_cname (ev));
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp0_ = (g_free (_tmp0_), NULL);
	if (vala_enum_value_get_value (ev) != NULL) {
		char* value;
		value = vala_gir_writer_literal_expression_to_value_string (self, vala_enum_value_get_value (ev));
		fprintf (self->priv->stream, " value=\"%s\"", value);
		value = (g_free (value), NULL);
	} else {
		fprintf (self->priv->stream, " value=\"%d\"", self->priv->enum_value++);
	}
	fprintf (self->priv->stream, "/>\n");
}


static void vala_gir_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain) {
	ValaGIRWriter * self;
	char* _tmp0_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (edomain != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) edomain)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) edomain)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<errordomain name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) edomain));
	_tmp0_ = NULL;
	fprintf (self->priv->stream, " get-quark=\"%squark\"", _tmp0_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) edomain));
	_tmp0_ = (g_free (_tmp0_), NULL);
	fprintf (self->priv->stream, " codes=\"%s\"", vala_symbol_get_name ((ValaSymbol*) edomain));
	fprintf (self->priv->stream, ">\n");
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) edomain);
	fprintf (self->priv->stream, "</errordomain>\n");
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<enumeration name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) edomain));
	vala_gir_writer_write_ctype_attributes (self, (ValaTypeSymbol*) edomain, "");
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	self->priv->enum_value = 0;
	vala_code_node_accept_children ((ValaCodeNode*) edomain, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</enumeration>\n");
}


static void vala_gir_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode) {
	ValaGIRWriter * self;
	char* _tmp1_;
	char* _tmp0_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (ecode != NULL);
	vala_gir_writer_write_indent (self);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	fprintf (self->priv->stream, "<member name=\"%s\" c:identifier=\"%s\"", _tmp0_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) ecode), -1), _tmp1_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) ecode, FALSE));
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp0_ = (g_free (_tmp0_), NULL);
	if (vala_error_code_get_value (ecode) != NULL) {
		char* value;
		value = vala_gir_writer_literal_expression_to_value_string (self, vala_error_code_get_value (ecode));
		fprintf (self->priv->stream, " value=\"%s\"", value);
		value = (g_free (value), NULL);
	} else {
		fprintf (self->priv->stream, " value=\"%d\"", self->priv->enum_value++);
	}
	fprintf (self->priv->stream, "/>\n");
}


static void vala_gir_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaGIRWriter * self;
	ValaExpression* _tmp0_;
	ValaExpression* initializer;
	char* value;
	char* _tmp1_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (c != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) c)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) c)) {
		return;
	}
	/*TODO Add better constant evaluation*/
	_tmp0_ = NULL;
	initializer = (_tmp0_ = vala_constant_get_initializer (c), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	value = vala_gir_writer_literal_expression_to_value_string (self, initializer);
	vala_gir_writer_write_indent (self);
	_tmp1_ = NULL;
	fprintf (self->priv->stream, "<constant name=\"%s\" c:identifier=\"%s\"", vala_symbol_get_name ((ValaSymbol*) c), _tmp1_ = vala_constant_get_cname (c));
	_tmp1_ = (g_free (_tmp1_), NULL);
	fprintf (self->priv->stream, " value=\"%s\"", value);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_type (self, vala_expression_get_value_type (initializer));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</constant>\n");
	(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
	value = (g_free (value), NULL);
}


static void vala_gir_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaGIRWriter * self;
	char* _tmp0_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (f != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) f)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) f)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0_ = NULL;
	fprintf (self->priv->stream, "<field name=\"%s\"", _tmp0_ = vala_field_get_cname (f));
	_tmp0_ = (g_free (_tmp0_), NULL);
	if (vala_data_type_get_nullable (vala_field_get_field_type (f))) {
		fprintf (self->priv->stream, " allow-none=\"1\"");
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) f);
	vala_gir_writer_write_type (self, vala_field_get_field_type (f));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</field>\n");
}


static void vala_gir_writer_write_params (ValaGIRWriter* self, GeeList* params, ValaDataType* instance_type) {
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (params != NULL);
	_tmp0_ = FALSE;
	if (gee_collection_get_size ((GeeCollection*) params) == 0) {
		_tmp0_ = instance_type == NULL;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<parameters>\n");
	self->priv->indent++;
	if (instance_type != NULL) {
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<parameter name=\"self\" transfer-ownership=\"none\">\n");
		self->priv->indent++;
		vala_gir_writer_write_type (self, instance_type);
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</parameter>\n");
	}
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) params);
		while (TRUE) {
			ValaFormalParameter* param;
			if (!gee_iterator_next (_param_it)) {
				break;
			}
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "<parameter name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) param));
			if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_REF) {
				fprintf (self->priv->stream, " direction=\"inout\"");
				/* in/out paramter*/
				if (vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param))) {
					fprintf (self->priv->stream, " transfer-ownership=\"full\"");
				} else {
					fprintf (self->priv->stream, " transfer-ownership=\"none\"");
				}
			} else {
				if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_OUT) {
					/* out paramter*/
					fprintf (self->priv->stream, " direction=\"out\"");
					if (vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param))) {
						fprintf (self->priv->stream, " transfer-ownership=\"full\"");
					} else {
						fprintf (self->priv->stream, " transfer-ownership=\"none\"");
					}
				} else {
					/* normal in paramter*/
					if (vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param))) {
						fprintf (self->priv->stream, " transfer-ownership=\"full\"");
					} else {
						fprintf (self->priv->stream, " transfer-ownership=\"none\"");
					}
				}
			}
			if (vala_data_type_get_nullable (vala_formal_parameter_get_parameter_type (param))) {
				fprintf (self->priv->stream, " allow-none=\"1\"");
			}
			fprintf (self->priv->stream, ">\n");
			self->priv->indent++;
			vala_gir_writer_write_annotations (self, (ValaCodeNode*) param);
			vala_gir_writer_write_type (self, vala_formal_parameter_get_parameter_type (param));
			self->priv->indent--;
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "</parameter>\n");
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</parameters>\n");
}


static void vala_gir_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb) {
	ValaGIRWriter * self;
	char* _tmp0_;
	GeeList* _tmp1_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (cb != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) cb)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) cb)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<callback name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) cb));
	_tmp0_ = NULL;
	fprintf (self->priv->stream, " c:type=\"%s\"", _tmp0_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cb, FALSE));
	_tmp0_ = (g_free (_tmp0_), NULL);
	if (vala_code_node_get_tree_can_fail ((ValaCodeNode*) cb)) {
		fprintf (self->priv->stream, " throws=\"1\"");
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) cb);
	_tmp1_ = NULL;
	vala_gir_writer_write_params (self, _tmp1_ = vala_delegate_get_parameters (cb), NULL);
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (gee_collection_object_unref (_tmp1_), NULL));
	vala_gir_writer_write_return_type (self, vala_delegate_get_return_type (cb), FALSE);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</callback>\n");
}


static void vala_gir_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaGIRWriter * self;
	gboolean _tmp0_;
	gboolean _tmp1_;
	char* tag_name;
	ValaSymbol* _tmp4_;
	ValaSymbol* parent;
	gboolean _tmp5_;
	gboolean _tmp7_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (m != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) m)) {
		return;
	}
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) m)) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = vala_method_get_overrides (m);
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		gboolean _tmp2_;
		gboolean _tmp3_;
		_tmp2_ = FALSE;
		_tmp3_ = FALSE;
		if (vala_method_get_base_interface_method (m) != NULL) {
			_tmp3_ = !vala_method_get_is_abstract (m);
		} else {
			_tmp3_ = FALSE;
		}
		if (_tmp3_) {
			_tmp2_ = !vala_method_get_is_virtual (m);
		} else {
			_tmp2_ = FALSE;
		}
		_tmp0_ = _tmp2_;
	}
	/* don't write interface implementation unless it's an abstract or virtual method*/
	if (_tmp0_) {
		return;
	}
	tag_name = g_strdup ("method");
	_tmp4_ = NULL;
	parent = (_tmp4_ = vala_symbol_get_parent_symbol ((ValaSymbol*) m), (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
	_tmp5_ = FALSE;
	if (VALA_IS_NAMESPACE (parent)) {
		_tmp5_ = TRUE;
	} else {
		_tmp5_ = vala_method_get_binding (m) == MEMBER_BINDING_STATIC;
	}
	if (_tmp5_) {
		char* _tmp6_;
		_tmp6_ = NULL;
		tag_name = (_tmp6_ = g_strdup ("function"), tag_name = (g_free (tag_name), NULL), _tmp6_);
	}
	vala_gir_writer_write_signature (self, m, tag_name, FALSE);
	_tmp7_ = FALSE;
	if (vala_method_get_is_abstract (m)) {
		_tmp7_ = TRUE;
	} else {
		_tmp7_ = vala_method_get_is_virtual (m);
	}
	if (_tmp7_) {
		vala_gir_writer_write_signature (self, m, "virtual-method", FALSE);
	}
	tag_name = (g_free (tag_name), NULL);
	(parent == NULL) ? NULL : (parent = (vala_code_node_unref (parent), NULL));
}


static void vala_gir_writer_write_signature (ValaGIRWriter* self, ValaMethod* m, const char* tag_name, gboolean instance) {
	ValaDataType* instance_type;
	GeeList* _tmp3_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (tag_name != NULL);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<%s name=\"%s\"", tag_name, vala_symbol_get_name ((ValaSymbol*) m));
	if (_vala_strcmp0 (tag_name, "virtual-method") == 0) {
		fprintf (self->priv->stream, " invoker=\"%s\"", vala_symbol_get_name ((ValaSymbol*) m));
	} else {
		if (_vala_strcmp0 (tag_name, "callback") == 0) {
			char* _tmp0_;
			_tmp0_ = NULL;
			fprintf (self->priv->stream, " c:type=\"%s\"", _tmp0_ = vala_method_get_cname (m));
			_tmp0_ = (g_free (_tmp0_), NULL);
		} else {
			char* _tmp1_;
			_tmp1_ = NULL;
			fprintf (self->priv->stream, " c:identifier=\"%s\"", _tmp1_ = vala_method_get_cname (m));
			_tmp1_ = (g_free (_tmp1_), NULL);
		}
	}
	if (vala_code_node_get_tree_can_fail ((ValaCodeNode*) m)) {
		fprintf (self->priv->stream, " throws=\"1\"");
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) m);
	instance_type = NULL;
	if (instance) {
		ValaDataType* _tmp2_;
		_tmp2_ = NULL;
		instance_type = (_tmp2_ = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) m))), (instance_type == NULL) ? NULL : (instance_type = (vala_code_node_unref (instance_type), NULL)), _tmp2_);
	}
	_tmp3_ = NULL;
	vala_gir_writer_write_params (self, _tmp3_ = vala_method_get_parameters (m), instance_type);
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL));
	vala_gir_writer_write_return_type (self, vala_method_get_return_type (m), FALSE);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</%s>\n", tag_name);
	(instance_type == NULL) ? NULL : (instance_type = (vala_code_node_unref (instance_type), NULL));
}


static void vala_gir_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaGIRWriter * self;
	char* _tmp0_;
	GeeList* _tmp1_;
	ValaDataType* datatype;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (m != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) m)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) m)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0_ = NULL;
	fprintf (self->priv->stream, "<constructor name=\"%s\" c:identifier=\"%s\"", vala_symbol_get_name ((ValaSymbol*) m), _tmp0_ = vala_method_get_cname ((ValaMethod*) m));
	_tmp0_ = (g_free (_tmp0_), NULL);
	if (vala_code_node_get_tree_can_fail ((ValaCodeNode*) m)) {
		fprintf (self->priv->stream, " throws=\"1\"");
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) m);
	_tmp1_ = NULL;
	vala_gir_writer_write_params (self, _tmp1_ = vala_method_get_parameters ((ValaMethod*) m), NULL);
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (gee_collection_object_unref (_tmp1_), NULL));
	datatype = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) m)));
	vala_gir_writer_write_return_type (self, datatype, TRUE);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</constructor>\n");
	(datatype == NULL) ? NULL : (datatype = (vala_code_node_unref (datatype), NULL));
}


static void vala_gir_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaGIRWriter * self;
	gboolean _tmp0_;
	gboolean _tmp1_;
	char* _tmp4_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (prop != NULL);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) prop)) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = vala_property_get_overrides (prop);
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		gboolean _tmp2_;
		gboolean _tmp3_;
		_tmp2_ = FALSE;
		_tmp3_ = FALSE;
		if (vala_property_get_base_interface_property (prop) != NULL) {
			_tmp3_ = !vala_property_get_is_abstract (prop);
		} else {
			_tmp3_ = FALSE;
		}
		if (_tmp3_) {
			_tmp2_ = !vala_property_get_is_virtual (prop);
		} else {
			_tmp2_ = FALSE;
		}
		_tmp0_ = _tmp2_;
	}
	if (_tmp0_) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp4_ = NULL;
	fprintf (self->priv->stream, "<property name=\"%s\"", _tmp4_ = vala_property_get_canonical_name (prop));
	_tmp4_ = (g_free (_tmp4_), NULL);
	if (vala_property_get_get_accessor (prop) == NULL) {
		fprintf (self->priv->stream, " readable=\"0\"");
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		fprintf (self->priv->stream, " writable=\"1\"");
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (prop))) {
			if (!vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
				fprintf (self->priv->stream, " construct-only=\"1\"");
			} else {
				fprintf (self->priv->stream, " construct=\"1\"");
			}
		}
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) prop);
	vala_gir_writer_write_type (self, vala_property_get_property_type (prop));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</property>\n");
}


static void vala_gir_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaGIRWriter * self;
	char* _tmp0_;
	GeeList* _tmp1_;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (sig != NULL);
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) sig)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0_ = NULL;
	fprintf (self->priv->stream, "<glib:signal name=\"%s\"", _tmp0_ = vala_signal_get_cname (sig));
	_tmp0_ = (g_free (_tmp0_), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_annotations (self, (ValaCodeNode*) sig);
	_tmp1_ = NULL;
	vala_gir_writer_write_params (self, _tmp1_ = vala_signal_get_parameters (sig), NULL);
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (gee_collection_object_unref (_tmp1_), NULL));
	vala_gir_writer_write_return_type (self, vala_signal_get_return_type (sig), FALSE);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</glib:signal>\n");
}


static void vala_gir_writer_write_indent (ValaGIRWriter* self) {
	gint i;
	g_return_if_fail (self != NULL);
	i = 0;
	{
		gboolean _tmp0_;
		i = 0;
		_tmp0_ = TRUE;
		while (TRUE) {
			if (!_tmp0_) {
				i++;
			}
			_tmp0_ = FALSE;
			if (!(i < self->priv->indent)) {
				break;
			}
			fputc ('\t', self->priv->stream);
		}
	}
}


static void vala_gir_writer_write_return_type (ValaGIRWriter* self, ValaDataType* type, gboolean constructor) {
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<return-value");
	_tmp0_ = FALSE;
	if (vala_data_type_get_value_owned (type)) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = constructor;
	}
	if (_tmp0_) {
		fprintf (self->priv->stream, " transfer-ownership=\"full\"");
	} else {
		fprintf (self->priv->stream, " transfer-ownership=\"none\"");
	}
	if (vala_data_type_get_nullable (type)) {
		fprintf (self->priv->stream, " allow-none=\"1\"");
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_type (self, type);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</return-value>\n");
}


static void vala_gir_writer_write_ctype_attributes (ValaGIRWriter* self, ValaTypeSymbol* symbol, const char* suffix) {
	char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (symbol != NULL);
	g_return_if_fail (suffix != NULL);
	_tmp0_ = NULL;
	fprintf (self->priv->stream, " c:type=\"%s%s\"", _tmp0_ = vala_typesymbol_get_cname (symbol, FALSE), suffix);
	_tmp0_ = (g_free (_tmp0_), NULL);
}


static void vala_gir_writer_write_gtype_attributes (ValaGIRWriter* self, ValaTypeSymbol* symbol) {
	char* _tmp0_;
	char* _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (symbol != NULL);
	vala_gir_writer_write_ctype_attributes (self, symbol, "");
	_tmp0_ = NULL;
	fprintf (self->priv->stream, " glib:type-name=\"%s\"", _tmp0_ = vala_typesymbol_get_cname (symbol, FALSE));
	_tmp0_ = (g_free (_tmp0_), NULL);
	_tmp1_ = NULL;
	fprintf (self->priv->stream, " glib:get-type=\"%sget_type\"", _tmp1_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) symbol));
	_tmp1_ = (g_free (_tmp1_), NULL);
}


static void vala_gir_writer_write_type (ValaGIRWriter* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	if (VALA_IS_ARRAY_TYPE (type)) {
		ValaArrayType* _tmp0_;
		ValaArrayType* array_type;
		_tmp0_ = NULL;
		array_type = (_tmp0_ = VALA_ARRAY_TYPE (type), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<array>\n");
		self->priv->indent++;
		vala_gir_writer_write_type (self, vala_array_type_get_element_type (array_type));
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</array>\n");
		(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
	} else {
		if (VALA_IS_VOID_TYPE (type)) {
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "<type name=\"none\"/>\n");
		} else {
			if (VALA_IS_POINTER_TYPE (type)) {
				char* _tmp1_;
				vala_gir_writer_write_indent (self);
				_tmp1_ = NULL;
				fprintf (self->priv->stream, "<type name=\"any\" c:type=\"%s\"/>\n", _tmp1_ = vala_data_type_get_cname (type));
				_tmp1_ = (g_free (_tmp1_), NULL);
			} else {
				if (vala_data_type_get_data_type (type) != NULL) {
					char* _tmp3_;
					char* _tmp2_;
					GeeList* type_arguments;
					vala_gir_writer_write_indent (self);
					_tmp3_ = NULL;
					_tmp2_ = NULL;
					fprintf (self->priv->stream, "<type name=\"%s\" c:type=\"%s\"", _tmp2_ = vala_gir_writer_gi_type_name (self, vala_data_type_get_data_type (type)), _tmp3_ = vala_data_type_get_cname (type));
					_tmp3_ = (g_free (_tmp3_), NULL);
					_tmp2_ = (g_free (_tmp2_), NULL);
					type_arguments = vala_data_type_get_type_arguments (type);
					if (gee_collection_get_size ((GeeCollection*) type_arguments) == 0) {
						fprintf (self->priv->stream, "/>\n");
					} else {
						fprintf (self->priv->stream, ">\n");
						self->priv->indent++;
						{
							GeeIterator* _type_argument_it;
							_type_argument_it = gee_iterable_iterator ((GeeIterable*) type_arguments);
							while (TRUE) {
								ValaDataType* type_argument;
								if (!gee_iterator_next (_type_argument_it)) {
									break;
								}
								type_argument = (ValaDataType*) gee_iterator_get (_type_argument_it);
								vala_gir_writer_write_type (self, type_argument);
								(type_argument == NULL) ? NULL : (type_argument = (vala_code_node_unref (type_argument), NULL));
							}
							(_type_argument_it == NULL) ? NULL : (_type_argument_it = (gee_collection_object_unref (_type_argument_it), NULL));
						}
						self->priv->indent--;
						vala_gir_writer_write_indent (self);
						fprintf (self->priv->stream, "</type>\n");
					}
					(type_arguments == NULL) ? NULL : (type_arguments = (gee_collection_object_unref (type_arguments), NULL));
				} else {
					char* _tmp4_;
					vala_gir_writer_write_indent (self);
					_tmp4_ = NULL;
					fprintf (self->priv->stream, "<type name=\"%s\"/>\n", _tmp4_ = vala_code_node_to_string ((ValaCodeNode*) type));
					_tmp4_ = (g_free (_tmp4_), NULL);
				}
			}
		}
	}
}


static void vala_gir_writer_write_annotations (ValaGIRWriter* self, ValaCodeNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	{
		GList* attr_collection;
		GList* attr_it;
		attr_collection = node->attributes;
		for (attr_it = attr_collection; attr_it != NULL; attr_it = attr_it->next) {
			ValaAttribute* _tmp3_;
			ValaAttribute* attr;
			_tmp3_ = NULL;
			attr = (_tmp3_ = (ValaAttribute*) attr_it->data, (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
			{
				char* name;
				name = vala_gir_writer_camel_case_to_canonical (self, vala_attribute_get_name (attr));
				{
					GeeSet* _tmp0_;
					GeeIterator* _tmp1_;
					GeeIterator* _arg_name_it;
					_tmp0_ = NULL;
					_tmp1_ = NULL;
					_arg_name_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_keys (attr->args))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
					while (TRUE) {
						char* arg_name;
						ValaExpression* arg;
						char* value;
						if (!gee_iterator_next (_arg_name_it)) {
							break;
						}
						arg_name = (char*) gee_iterator_get (_arg_name_it);
						arg = (ValaExpression*) gee_map_get (attr->args, arg_name);
						value = vala_gir_writer_literal_expression_to_value_string (self, (ValaExpression*) VALA_LITERAL (arg));
						if (value != NULL) {
							char* _tmp2_;
							vala_gir_writer_write_indent (self);
							_tmp2_ = NULL;
							fprintf (self->priv->stream, "<annotation key=\"%s.%s\" value=\"%s\"/>\n", name, _tmp2_ = vala_gir_writer_camel_case_to_canonical (self, arg_name), value);
							_tmp2_ = (g_free (_tmp2_), NULL);
						}
						arg_name = (g_free (arg_name), NULL);
						(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
						value = (g_free (value), NULL);
					}
					(_arg_name_it == NULL) ? NULL : (_arg_name_it = (gee_collection_object_unref (_arg_name_it), NULL));
				}
				(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
				name = (g_free (name), NULL);
			}
		}
	}
}


static char* vala_gir_writer_gi_type_name (ValaGIRWriter* self, ValaTypeSymbol* type_symbol) {
	char* result;
	char* _tmp0_;
	char* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_symbol != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	result = (_tmp1_ = vala_gir_writer_vala_to_gi_type_name (self, _tmp0_ = vala_symbol_get_full_name ((ValaSymbol*) type_symbol)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	return result;
}


static gboolean string_contains (const char* self, const char* needle) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (needle != NULL, FALSE);
	result = strstr (self, needle) != NULL;
	return result;
}


static char* vala_gir_writer_vala_to_gi_type_name (ValaGIRWriter* self, const char* name) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	if (_vala_strcmp0 (name, "bool") == 0) {
		result = g_strdup ("boolean");
		return result;
	} else {
		if (_vala_strcmp0 (name, "string") == 0) {
			result = g_strdup ("utf8");
			return result;
		} else {
			if (!string_contains (name, ".")) {
				const char* _tmp0_;
				_tmp0_ = NULL;
				result = (_tmp0_ = name, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
				return result;
			} else {
				char** _tmp2_;
				gint split_name_size;
				gint split_name_length1;
				char** _tmp1_;
				char** split_name;
				GString* type_name;
				const char* _tmp4_;
				_tmp2_ = NULL;
				_tmp1_ = NULL;
				split_name = (_tmp2_ = _tmp1_ = g_strsplit (name, ".", 0), split_name_length1 = _vala_array_length (_tmp1_), split_name_size = split_name_length1, _tmp2_);
				type_name = g_string_new ("");
				g_string_append (type_name, split_name[0]);
				g_string_append_unichar (type_name, (gunichar) '.');
				{
					gint i;
					i = 1;
					{
						gboolean _tmp3_;
						_tmp3_ = TRUE;
						while (TRUE) {
							if (!_tmp3_) {
								i++;
							}
							_tmp3_ = FALSE;
							if (!(i < split_name_length1)) {
								break;
							}
							g_string_append (type_name, split_name[i]);
						}
					}
				}
				if (_vala_strcmp0 (type_name->str, "GLib.Object") == 0) {
					result = g_strdup ("GObject.Object");
					split_name = (_vala_array_free (split_name, split_name_length1, (GDestroyNotify) g_free), NULL);
					(type_name == NULL) ? NULL : (type_name = (g_string_free (type_name, TRUE), NULL));
					return result;
				}
				_tmp4_ = NULL;
				result = (_tmp4_ = type_name->str, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_));
				split_name = (_vala_array_free (split_name, split_name_length1, (GDestroyNotify) g_free), NULL);
				(type_name == NULL) ? NULL : (type_name = (g_string_free (type_name, TRUE), NULL));
				return result;
				split_name = (_vala_array_free (split_name, split_name_length1, (GDestroyNotify) g_free), NULL);
				(type_name == NULL) ? NULL : (type_name = (g_string_free (type_name, TRUE), NULL));
			}
		}
	}
}


static char* vala_gir_writer_literal_expression_to_value_string (ValaGIRWriter* self, ValaExpression* literal) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (literal != NULL, NULL);
	if (VALA_IS_STRING_LITERAL (literal)) {
		ValaStringLiteral* _tmp1_;
		ValaExpression* _tmp0_;
		ValaStringLiteral* lit;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		lit = (_tmp1_ = (_tmp0_ = literal, VALA_IS_STRING_LITERAL (_tmp0_) ? ((ValaStringLiteral*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
		if (lit != NULL) {
			char* _tmp2_;
			char* _tmp3_;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			result = (_tmp3_ = g_markup_escape_text (_tmp2_ = vala_string_literal_eval (lit), -1), _tmp2_ = (g_free (_tmp2_), NULL), _tmp3_);
			(lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL));
			return result;
		}
		(lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL));
	} else {
		if (VALA_IS_CHARACTER_LITERAL (literal)) {
			result = g_strdup_printf ("%lc", (glong) vala_character_literal_get_char (VALA_CHARACTER_LITERAL (literal)));
			return result;
		} else {
			if (VALA_IS_BOOLEAN_LITERAL (literal)) {
				const char* _tmp4_;
				const char* _tmp5_;
				_tmp4_ = NULL;
				if (vala_boolean_literal_get_value (VALA_BOOLEAN_LITERAL (literal))) {
					_tmp4_ = "true";
				} else {
					_tmp4_ = "false";
				}
				_tmp5_ = NULL;
				result = (_tmp5_ = _tmp4_, (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_));
				return result;
			} else {
				if (VALA_IS_REAL_LITERAL (literal)) {
					const char* _tmp6_;
					_tmp6_ = NULL;
					result = (_tmp6_ = vala_real_literal_get_value (VALA_REAL_LITERAL (literal)), (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_));
					return result;
				} else {
					if (VALA_IS_INTEGER_LITERAL (literal)) {
						const char* _tmp7_;
						_tmp7_ = NULL;
						result = (_tmp7_ = vala_integer_literal_get_value (VALA_INTEGER_LITERAL (literal)), (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_));
						return result;
					} else {
						if (VALA_IS_UNARY_EXPRESSION (literal)) {
							ValaUnaryExpression* _tmp8_;
							ValaUnaryExpression* unary;
							_tmp8_ = NULL;
							unary = (_tmp8_ = VALA_UNARY_EXPRESSION (literal), (_tmp8_ == NULL) ? NULL : vala_code_node_ref (_tmp8_));
							if (vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_MINUS) {
								if (VALA_IS_REAL_LITERAL (vala_unary_expression_get_inner (unary))) {
									result = g_strconcat ("-", vala_real_literal_get_value (VALA_REAL_LITERAL (vala_unary_expression_get_inner (unary))), NULL);
									(unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL));
									return result;
								} else {
									if (VALA_IS_INTEGER_LITERAL (vala_unary_expression_get_inner (unary))) {
										result = g_strconcat ("-", vala_integer_literal_get_value (VALA_INTEGER_LITERAL (vala_unary_expression_get_inner (unary))), NULL);
										(unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL));
										return result;
									}
								}
							}
							(unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL));
						}
					}
				}
			}
		}
	}
	result = NULL;
	return result;
}


static char* vala_gir_writer_camel_case_to_canonical (ValaGIRWriter* self, const char* name) {
	char* result;
	char** _tmp3_;
	gint parts_size;
	gint parts_length1;
	char** _tmp1_;
	char* _tmp0_;
	char** _tmp2_;
	char** parts;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	_tmp3_ = NULL;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	parts = (_tmp3_ = (_tmp2_ = _tmp1_ = g_strsplit (_tmp0_ = vala_symbol_camel_case_to_lower_case (name), "_", 0), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_), parts_length1 = _vala_array_length (_tmp1_), parts_size = parts_length1, _tmp3_);
	result = g_strjoinv ("-", parts);
	parts = (_vala_array_free (parts, parts_length1, (GDestroyNotify) g_free), NULL);
	return result;
}


static gboolean vala_gir_writer_check_accessibility (ValaGIRWriter* self, ValaSymbol* sym) {
	gboolean result;
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (sym != NULL, FALSE);
	_tmp0_ = FALSE;
	if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
	}
	if (_tmp0_) {
		result = TRUE;
		return result;
	}
	result = FALSE;
	return result;
}


/**
 * Code visitor generating .gir file for the public interface.
 */
ValaGIRWriter* vala_gir_writer_construct (GType object_type) {
	ValaGIRWriter* self;
	self = (ValaGIRWriter*) g_type_create_instance (object_type);
	return self;
}


ValaGIRWriter* vala_gir_writer_new (void) {
	return vala_gir_writer_construct (VALA_TYPE_GIR_WRITER);
}


static void vala_gir_writer_class_init (ValaGIRWriterClass * klass) {
	vala_gir_writer_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_gir_writer_finalize;
	g_type_class_add_private (klass, sizeof (ValaGIRWriterPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_gir_writer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_gir_writer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_gir_writer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_gir_writer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_gir_writer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_gir_writer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_gir_writer_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_code = vala_gir_writer_real_visit_error_code;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_gir_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_gir_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_gir_writer_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_gir_writer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_gir_writer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_gir_writer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_gir_writer_real_visit_signal;
}


static void vala_gir_writer_instance_init (ValaGIRWriter * self) {
	self->priv = VALA_GIR_WRITER_GET_PRIVATE (self);
}


static void vala_gir_writer_finalize (ValaCodeVisitor* obj) {
	ValaGIRWriter * self;
	self = VALA_GIR_WRITER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	self->priv->directory = (g_free (self->priv->directory), NULL);
	self->priv->gir_namespace = (g_free (self->priv->gir_namespace), NULL);
	self->priv->gir_version = (g_free (self->priv->gir_version), NULL);
	(self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL));
	(self->priv->gobject_type == NULL) ? NULL : (self->priv->gobject_type = (vala_code_node_unref (self->priv->gobject_type), NULL));
	VALA_CODE_VISITOR_CLASS (vala_gir_writer_parent_class)->finalize (obj);
}


GType vala_gir_writer_get_type (void) {
	static GType vala_gir_writer_type_id = 0;
	if (vala_gir_writer_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIRWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gir_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIRWriter), 0, (GInstanceInitFunc) vala_gir_writer_instance_init, NULL };
		vala_gir_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIRWriter", &g_define_type_info, 0);
	}
	return vala_gir_writer_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




