/* valaassignment.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaExpressionPrivate ValaExpressionPrivate;

#define VALA_TYPE_ASSIGNMENT (vala_assignment_get_type ())
#define VALA_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ASSIGNMENT, ValaAssignment))
#define VALA_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ASSIGNMENT, ValaAssignmentClass))
#define VALA_IS_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ASSIGNMENT))
#define VALA_IS_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ASSIGNMENT))
#define VALA_ASSIGNMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ASSIGNMENT, ValaAssignmentClass))

typedef struct _ValaAssignment ValaAssignment;
typedef struct _ValaAssignmentClass ValaAssignmentClass;
typedef struct _ValaAssignmentPrivate ValaAssignmentPrivate;

#define VALA_TYPE_ASSIGNMENT_OPERATOR (vala_assignment_operator_get_type ())

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_MEMBER_ACCESS (vala_member_access_get_type ())
#define VALA_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccess))
#define VALA_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))
#define VALA_IS_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER_ACCESS))
#define VALA_IS_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER_ACCESS))
#define VALA_MEMBER_ACCESS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))

typedef struct _ValaMemberAccess ValaMemberAccess;
typedef struct _ValaMemberAccessClass ValaMemberAccessClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_SIGNAL (vala_signal_get_type ())
#define VALA_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SIGNAL, ValaSignal))
#define VALA_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SIGNAL, ValaSignalClass))
#define VALA_IS_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SIGNAL))
#define VALA_IS_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SIGNAL))
#define VALA_SIGNAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SIGNAL, ValaSignalClass))

typedef struct _ValaSignal ValaSignal;
typedef struct _ValaSignalClass ValaSignalClass;

#define VALA_TYPE_PROPERTY (vala_property_get_type ())
#define VALA_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY, ValaProperty))
#define VALA_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY, ValaPropertyClass))
#define VALA_IS_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY))
#define VALA_IS_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY))
#define VALA_PROPERTY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY, ValaPropertyClass))

typedef struct _ValaProperty ValaProperty;
typedef struct _ValaPropertyClass ValaPropertyClass;

#define VALA_TYPE_DYNAMIC_PROPERTY (vala_dynamic_property_get_type ())
#define VALA_DYNAMIC_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DYNAMIC_PROPERTY, ValaDynamicProperty))
#define VALA_DYNAMIC_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DYNAMIC_PROPERTY, ValaDynamicPropertyClass))
#define VALA_IS_DYNAMIC_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DYNAMIC_PROPERTY))
#define VALA_IS_DYNAMIC_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DYNAMIC_PROPERTY))
#define VALA_DYNAMIC_PROPERTY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DYNAMIC_PROPERTY, ValaDynamicPropertyClass))

typedef struct _ValaDynamicProperty ValaDynamicProperty;
typedef struct _ValaDynamicPropertyClass ValaDynamicPropertyClass;

#define VALA_TYPE_DYNAMIC_SIGNAL (vala_dynamic_signal_get_type ())
#define VALA_DYNAMIC_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignal))
#define VALA_DYNAMIC_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignalClass))
#define VALA_IS_DYNAMIC_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DYNAMIC_SIGNAL))
#define VALA_IS_DYNAMIC_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DYNAMIC_SIGNAL))
#define VALA_DYNAMIC_SIGNAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignalClass))

typedef struct _ValaDynamicSignal ValaDynamicSignal;
typedef struct _ValaDynamicSignalClass ValaDynamicSignalClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;

#define VALA_TYPE_DELEGATE_TYPE (vala_delegate_type_get_type ())
#define VALA_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateType))
#define VALA_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))
#define VALA_IS_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE_TYPE))
#define VALA_IS_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE_TYPE))
#define VALA_DELEGATE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))

typedef struct _ValaDelegateType ValaDelegateType;
typedef struct _ValaDelegateTypeClass ValaDelegateTypeClass;

#define VALA_TYPE_ELEMENT_ACCESS (vala_element_access_get_type ())
#define VALA_ELEMENT_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ELEMENT_ACCESS, ValaElementAccess))
#define VALA_ELEMENT_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ELEMENT_ACCESS, ValaElementAccessClass))
#define VALA_IS_ELEMENT_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ELEMENT_ACCESS))
#define VALA_IS_ELEMENT_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ELEMENT_ACCESS))
#define VALA_ELEMENT_ACCESS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ELEMENT_ACCESS, ValaElementAccessClass))

typedef struct _ValaElementAccess ValaElementAccess;
typedef struct _ValaElementAccessClass ValaElementAccessClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_METHOD_CALL (vala_method_call_get_type ())
#define VALA_METHOD_CALL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD_CALL, ValaMethodCall))
#define VALA_METHOD_CALL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD_CALL, ValaMethodCallClass))
#define VALA_IS_METHOD_CALL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD_CALL))
#define VALA_IS_METHOD_CALL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD_CALL))
#define VALA_METHOD_CALL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD_CALL, ValaMethodCallClass))

typedef struct _ValaMethodCall ValaMethodCall;
typedef struct _ValaMethodCallClass ValaMethodCallClass;

#define VALA_TYPE_POINTER_INDIRECTION (vala_pointer_indirection_get_type ())
#define VALA_POINTER_INDIRECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_POINTER_INDIRECTION, ValaPointerIndirection))
#define VALA_POINTER_INDIRECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_POINTER_INDIRECTION, ValaPointerIndirectionClass))
#define VALA_IS_POINTER_INDIRECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_POINTER_INDIRECTION))
#define VALA_IS_POINTER_INDIRECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_POINTER_INDIRECTION))
#define VALA_POINTER_INDIRECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_POINTER_INDIRECTION, ValaPointerIndirectionClass))

typedef struct _ValaPointerIndirection ValaPointerIndirection;
typedef struct _ValaPointerIndirectionClass ValaPointerIndirectionClass;

#define VALA_TYPE_BINARY_OPERATOR (vala_binary_operator_get_type ())

#define VALA_TYPE_BINARY_EXPRESSION (vala_binary_expression_get_type ())
#define VALA_BINARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpression))
#define VALA_BINARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpressionClass))
#define VALA_IS_BINARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_BINARY_EXPRESSION))
#define VALA_IS_BINARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_BINARY_EXPRESSION))
#define VALA_BINARY_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpressionClass))

typedef struct _ValaBinaryExpression ValaBinaryExpression;
typedef struct _ValaBinaryExpressionClass ValaBinaryExpressionClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_OBJECT_TYPE (vala_object_type_get_type ())
#define VALA_OBJECT_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE, ValaObjectType))
#define VALA_OBJECT_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE, ValaObjectTypeClass))
#define VALA_IS_OBJECT_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE))
#define VALA_IS_OBJECT_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE))
#define VALA_OBJECT_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE, ValaObjectTypeClass))

typedef struct _ValaObjectType ValaObjectType;
typedef struct _ValaObjectTypeClass ValaObjectTypeClass;

#define VALA_TYPE_PROPERTY_ACCESSOR (vala_property_accessor_get_type ())
#define VALA_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessor))
#define VALA_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))
#define VALA_IS_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_IS_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_PROPERTY_ACCESSOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))

typedef struct _ValaPropertyAccessor ValaPropertyAccessor;
typedef struct _ValaPropertyAccessorClass ValaPropertyAccessorClass;

#define VALA_TYPE_CREATION_METHOD (vala_creation_method_get_type ())
#define VALA_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethod))
#define VALA_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))
#define VALA_IS_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CREATION_METHOD))
#define VALA_IS_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CREATION_METHOD))
#define VALA_CREATION_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))

typedef struct _ValaCreationMethod ValaCreationMethod;
typedef struct _ValaCreationMethodClass ValaCreationMethodClass;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;

#define VALA_TYPE_POINTER_TYPE (vala_pointer_type_get_type ())
#define VALA_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerType))
#define VALA_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))
#define VALA_IS_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_POINTER_TYPE))
#define VALA_IS_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_POINTER_TYPE))
#define VALA_POINTER_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))

typedef struct _ValaPointerType ValaPointerType;
typedef struct _ValaPointerTypeClass ValaPointerTypeClass;

#define VALA_TYPE_ARRAY_TYPE (vala_array_type_get_type ())
#define VALA_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayType))
#define VALA_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))
#define VALA_IS_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_TYPE))
#define VALA_IS_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_TYPE))
#define VALA_ARRAY_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))

typedef struct _ValaArrayType ValaArrayType;
typedef struct _ValaArrayTypeClass ValaArrayTypeClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Base class for all code nodes that might be used as an expression.
 */
struct _ValaExpression {
	ValaCodeNode parent_instance;
	ValaExpressionPrivate * priv;
	GeeArrayList* temp_vars;
};

struct _ValaExpressionClass {
	ValaCodeNodeClass parent_class;
	gboolean (*is_constant) (ValaExpression* self);
	gboolean (*is_pure) (ValaExpression* self);
	gboolean (*is_non_null) (ValaExpression* self);
};

/**
 * Represents an assignment expression in the source code.
 *
 * Supports =, |=, &=, ^=, +=, -=, *=, /=, %=, <<=, >>=.
 */
struct _ValaAssignment {
	ValaExpression parent_instance;
	ValaAssignmentPrivate * priv;
};

struct _ValaAssignmentClass {
	ValaExpressionClass parent_class;
};

typedef enum  {
	VALA_ASSIGNMENT_OPERATOR_NONE,
	VALA_ASSIGNMENT_OPERATOR_SIMPLE,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_OR,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_AND,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR,
	VALA_ASSIGNMENT_OPERATOR_ADD,
	VALA_ASSIGNMENT_OPERATOR_SUB,
	VALA_ASSIGNMENT_OPERATOR_MUL,
	VALA_ASSIGNMENT_OPERATOR_DIV,
	VALA_ASSIGNMENT_OPERATOR_PERCENT,
	VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT,
	VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT
} ValaAssignmentOperator;

struct _ValaAssignmentPrivate {
	ValaAssignmentOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};

typedef enum  {
	VALA_BINARY_OPERATOR_NONE,
	VALA_BINARY_OPERATOR_PLUS,
	VALA_BINARY_OPERATOR_MINUS,
	VALA_BINARY_OPERATOR_MUL,
	VALA_BINARY_OPERATOR_DIV,
	VALA_BINARY_OPERATOR_MOD,
	VALA_BINARY_OPERATOR_SHIFT_LEFT,
	VALA_BINARY_OPERATOR_SHIFT_RIGHT,
	VALA_BINARY_OPERATOR_LESS_THAN,
	VALA_BINARY_OPERATOR_GREATER_THAN,
	VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL,
	VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL,
	VALA_BINARY_OPERATOR_EQUALITY,
	VALA_BINARY_OPERATOR_INEQUALITY,
	VALA_BINARY_OPERATOR_BITWISE_AND,
	VALA_BINARY_OPERATOR_BITWISE_OR,
	VALA_BINARY_OPERATOR_BITWISE_XOR,
	VALA_BINARY_OPERATOR_AND,
	VALA_BINARY_OPERATOR_OR,
	VALA_BINARY_OPERATOR_IN
} ValaBinaryOperator;


static gpointer vala_assignment_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_assignment_get_type (void);
GType vala_assignment_operator_get_type (void);
#define VALA_ASSIGNMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ASSIGNMENT, ValaAssignmentPrivate))
enum  {
	VALA_ASSIGNMENT_DUMMY_PROPERTY
};
void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value);
void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value);
ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
void vala_code_visitor_visit_assignment (ValaCodeVisitor* self, ValaAssignment* a);
void vala_code_visitor_visit_expression (ValaCodeVisitor* self, ValaExpression* expr);
static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaExpression* vala_assignment_get_left (ValaAssignment* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaExpression* vala_assignment_get_right (ValaAssignment* self);
static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_assignment_real_is_pure (ValaExpression* base);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
void vala_expression_set_lvalue (ValaExpression* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
GType vala_member_access_get_type (void);
ValaSymbol* vala_expression_get_symbol_reference (ValaExpression* self);
GType vala_member_get_type (void);
GType vala_signal_get_type (void);
GType vala_property_get_type (void);
GType vala_dynamic_property_get_type (void);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
gboolean vala_member_access_get_prototype_access (ValaMemberAccess* self);
char* vala_symbol_get_full_name (ValaSymbol* self);
GType vala_dynamic_signal_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_delegate_get_type (void);
ValaDelegate* vala_signal_get_delegate (ValaSignal* self, ValaDataType* sender_type, ValaCodeNode* node_reference);
ValaExpression* vala_member_access_get_inner (ValaMemberAccess* self);
ValaDelegateType* vala_delegate_type_new (ValaDelegate* delegate_symbol);
ValaDelegateType* vala_delegate_type_construct (GType object_type, ValaDelegate* delegate_symbol);
GType vala_delegate_type_get_type (void);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
ValaDataType* vala_expression_get_formal_value_type (ValaExpression* self);
void vala_expression_set_formal_target_type (ValaExpression* self, ValaDataType* value);
GType vala_element_access_get_type (void);
ValaExpression* vala_element_access_get_container (ValaElementAccess* self);
ValaSymbol* vala_data_type_get_member (ValaDataType* self, const char* member_name);
GType vala_method_get_type (void);
ValaMemberAccess* vala_member_access_new (ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference);
ValaMemberAccess* vala_member_access_construct (GType object_type, ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference);
ValaMethodCall* vala_method_call_new (ValaExpression* call, ValaSourceReference* source_reference);
ValaMethodCall* vala_method_call_construct (GType object_type, ValaExpression* call, ValaSourceReference* source_reference);
GType vala_method_call_get_type (void);
GeeList* vala_element_access_get_indices (ValaElementAccess* self);
void vala_method_call_add_argument (ValaMethodCall* self, ValaExpression* arg);
ValaCodeNode* vala_code_node_get_parent_node (ValaCodeNode* self);
void vala_code_node_replace_expression (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
GType vala_pointer_indirection_get_type (void);
ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self);
const char* vala_member_access_get_member_name (ValaMemberAccess* self);
GType vala_binary_operator_get_type (void);
ValaBinaryExpression* vala_binary_expression_new (ValaBinaryOperator op, ValaExpression* _left, ValaExpression* _right, ValaSourceReference* source);
ValaBinaryExpression* vala_binary_expression_construct (GType object_type, ValaBinaryOperator op, ValaExpression* _left, ValaExpression* _right, ValaSourceReference* source);
GType vala_binary_expression_get_type (void);
ValaDataType* vala_expression_get_target_type (ValaExpression* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
void vala_binary_expression_set_operator (ValaBinaryExpression* self, ValaBinaryOperator value);
ValaExpression* vala_dynamic_signal_get_handler (ValaDynamicSignal* self);
GType vala_formal_parameter_get_type (void);
GeeList* vala_data_type_get_parameters (ValaDataType* self);
void vala_signal_add_parameter (ValaSignal* self, ValaFormalParameter* param);
ValaFormalParameter* vala_formal_parameter_copy (ValaFormalParameter* self);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
GType vala_object_type_symbol_get_type (void);
ValaObjectType* vala_object_type_new (ValaObjectTypeSymbol* type_symbol);
ValaObjectType* vala_object_type_construct (GType object_type, ValaObjectTypeSymbol* type_symbol);
GType vala_reference_type_get_type (void);
GType vala_object_type_get_type (void);
gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type);
char* vala_code_node_to_string (ValaCodeNode* self);
ValaDelegate* vala_delegate_type_get_delegate_symbol (ValaDelegateType* self);
char* vala_delegate_get_prototype_string (ValaDelegate* self, const char* name);
const char* vala_symbol_get_name (ValaSymbol* self);
void vala_property_set_property_type (ValaProperty* self, ValaDataType* value);
ValaDataType* vala_property_get_property_type (ValaProperty* self);
void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value);
GType vala_property_accessor_get_type (void);
ValaPropertyAccessor* vala_property_get_set_accessor (ValaProperty* self);
gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self);
ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
GType vala_creation_method_get_type (void);
gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
ValaDataType* vala_local_variable_get_variable_type (ValaLocalVariable* self);
gboolean vala_delegate_matches_method (ValaDelegate* self, ValaMethod* m);
GType vala_field_get_type (void);
ValaDataType* vala_field_get_field_type (ValaField* self);
gboolean vala_data_type_is_disposable (ValaDataType* self);
GType vala_pointer_type_get_type (void);
gboolean vala_data_type_get_value_owned (ValaDataType* self);
GType vala_array_type_get_type (void);
ValaDataType* vala_array_type_get_element_type (ValaArrayType* self);
GeeList* vala_data_type_get_type_arguments (ValaDataType* self);
void vala_code_node_add_error_types (ValaCodeNode* self, GeeList* error_types);
GeeList* vala_code_node_get_error_types (ValaCodeNode* self);
static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_get_defined_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_get_used_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
static void vala_assignment_finalize (ValaCodeNode* obj);



/**
 * Creates a new assignment.
 *
 * @param left             left hand side
 * @param operator         assignment operator
 * @param right            right hand side
 * @param source_reference reference to source code
 * @return                 newly created assignment
 */
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	ValaAssignment* self;
	g_return_val_if_fail (left != NULL, NULL);
	g_return_val_if_fail (right != NULL, NULL);
	self = (ValaAssignment*) g_type_create_instance (object_type);
	vala_assignment_set_right (self, right);
	vala_assignment_set_operator (self, operator);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_assignment_set_left (self, left);
	return self;
}


ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	return vala_assignment_construct (VALA_TYPE_ASSIGNMENT, left, right, operator, source_reference);
}


static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_assignment (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_left (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_right (self), visitor);
}


static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_assignment_get_left (self) == old_node) {
		vala_assignment_set_left (self, new_node);
	}
	if (vala_assignment_get_right (self) == old_node) {
		vala_assignment_set_right (self, new_node);
	}
}


static gboolean vala_assignment_real_is_pure (ValaExpression* base) {
	ValaAssignment * self;
	gboolean result;
	self = (ValaAssignment*) base;
	result = FALSE;
	return result;
}


static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaAssignment * self;
	gboolean result;
	gboolean _tmp21_;
	GeeList* _tmp91_;
	GeeList* _tmp92_;
	self = (ValaAssignment*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_expression_set_lvalue (vala_assignment_get_left (self), TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_left (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		result = FALSE;
		return result;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
		ValaMemberAccess* _tmp0_;
		ValaMemberAccess* ma;
		gboolean _tmp1_;
		gboolean _tmp2_;
		gboolean _tmp5_;
		_tmp0_ = NULL;
		ma = (_tmp0_ = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		_tmp1_ = FALSE;
		_tmp2_ = FALSE;
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			_tmp2_ = TRUE;
		} else {
			_tmp2_ = VALA_IS_DYNAMIC_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma));
		}
		if (!_tmp2_) {
			_tmp1_ = vala_expression_get_value_type ((ValaExpression*) ma) == NULL;
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
			result = FALSE;
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			return result;
		}
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp4_;
			char* _tmp3_;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4_ = g_strdup_printf ("Access to instance member `%s' denied", _tmp3_ = vala_symbol_get_full_name (vala_expression_get_symbol_reference ((ValaExpression*) ma))));
			_tmp4_ = (g_free (_tmp4_), NULL);
			_tmp3_ = (g_free (_tmp3_), NULL);
			result = FALSE;
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			return result;
		}
		_tmp5_ = FALSE;
		if (vala_code_node_get_error ((ValaCodeNode*) ma)) {
			_tmp5_ = TRUE;
		} else {
			_tmp5_ = vala_expression_get_symbol_reference ((ValaExpression*) ma) == NULL;
		}
		if (_tmp5_) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			/* if no symbol found, skip this check */
			result = FALSE;
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			return result;
		}
		if (VALA_IS_DYNAMIC_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		} else {
			if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaSignal* _tmp6_;
				ValaSignal* sig;
				ValaDelegateType* _tmp8_;
				ValaDelegate* _tmp7_;
				_tmp6_ = NULL;
				sig = (_tmp6_ = VALA_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
				_tmp8_ = NULL;
				_tmp7_ = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp8_ = vala_delegate_type_new (_tmp7_ = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
				(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				vala_expression_set_formal_target_type (vala_assignment_get_right (self), vala_expression_get_formal_value_type ((ValaExpression*) ma));
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type ((ValaExpression*) ma));
			}
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
			ValaElementAccess* _tmp9_;
			ValaElementAccess* ea;
			gboolean _tmp10_;
			_tmp9_ = NULL;
			ea = (_tmp9_ = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp9_ == NULL) ? NULL : vala_code_node_ref (_tmp9_));
			_tmp10_ = FALSE;
			if (VALA_IS_MEMBER_ACCESS (vala_element_access_get_container (ea))) {
				_tmp10_ = VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea)));
			} else {
				_tmp10_ = FALSE;
			}
			if (_tmp10_) {
				ValaMemberAccess* _tmp11_;
				ValaMemberAccess* ma;
				ValaSignal* _tmp12_;
				ValaSignal* sig;
				ValaDelegateType* _tmp14_;
				ValaDelegate* _tmp13_;
				_tmp11_ = NULL;
				ma = (_tmp11_ = VALA_MEMBER_ACCESS (vala_element_access_get_container (ea)), (_tmp11_ == NULL) ? NULL : vala_code_node_ref (_tmp11_));
				_tmp12_ = NULL;
				sig = (_tmp12_ = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea))), (_tmp12_ == NULL) ? NULL : vala_code_node_ref (_tmp12_));
				_tmp14_ = NULL;
				_tmp13_ = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp14_ = vala_delegate_type_new (_tmp13_ = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_code_node_unref (_tmp14_), NULL));
				(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_code_node_unref (_tmp13_), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				ValaSymbol* _tmp15_;
				gboolean _tmp16_;
				_tmp15_ = NULL;
				if ((_tmp16_ = VALA_IS_METHOD (_tmp15_ = vala_data_type_get_member (vala_expression_get_value_type (vala_element_access_get_container (ea)), "set")), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_code_node_unref (_tmp15_), NULL)), _tmp16_)) {
					ValaMemberAccess* _tmp17_;
					ValaMethodCall* _tmp18_;
					ValaMethodCall* set_call;
					_tmp17_ = NULL;
					_tmp18_ = NULL;
					set_call = (_tmp18_ = vala_method_call_new ((ValaExpression*) (_tmp17_ = vala_member_access_new (vala_element_access_get_container (ea), "set", NULL)), NULL), (_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_code_node_unref (_tmp17_), NULL)), _tmp18_);
					{
						GeeList* _tmp19_;
						GeeIterator* _tmp20_;
						GeeIterator* _e_it;
						_tmp19_ = NULL;
						_tmp20_ = NULL;
						_e_it = (_tmp20_ = gee_iterable_iterator ((GeeIterable*) (_tmp19_ = vala_element_access_get_indices (ea))), (_tmp19_ == NULL) ? NULL : (_tmp19_ = (gee_collection_object_unref (_tmp19_), NULL)), _tmp20_);
						while (TRUE) {
							ValaExpression* e;
							if (!gee_iterator_next (_e_it)) {
								break;
							}
							e = (ValaExpression*) gee_iterator_get (_e_it);
							vala_method_call_add_argument (set_call, e);
							(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
						}
						(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					}
					vala_method_call_add_argument (set_call, vala_assignment_get_right (self));
					vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) set_call);
					result = vala_code_node_check ((ValaCodeNode*) set_call, analyzer);
					(set_call == NULL) ? NULL : (set_call = (vala_code_node_unref (set_call), NULL));
					(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
					return result;
				} else {
					vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
				}
			}
			(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
		} else {
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (self))) {
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
			} else {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
				result = FALSE;
				return result;
			}
		}
	}
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		result = FALSE;
		return result;
	}
	_tmp21_ = FALSE;
	if (self->priv->_operator != VALA_ASSIGNMENT_OPERATOR_SIMPLE) {
		_tmp21_ = VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self));
	} else {
		_tmp21_ = FALSE;
	}
	if (_tmp21_) {
		ValaMemberAccess* _tmp22_;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		_tmp22_ = NULL;
		ma = (_tmp22_ = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp22_ == NULL) ? NULL : vala_code_node_ref (_tmp22_));
		if (!VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			ValaMemberAccess* old_value;
			ValaBinaryExpression* bin;
			ValaDataType* _tmp23_;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			bin = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, (ValaExpression*) old_value, vala_assignment_get_right (self), NULL);
			vala_expression_set_target_type ((ValaExpression*) bin, vala_expression_get_target_type (vala_assignment_get_right (self)));
			_tmp23_ = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), _tmp23_ = vala_data_type_copy (vala_expression_get_target_type (vala_assignment_get_right (self))));
			(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_code_node_unref (_tmp23_), NULL));
			vala_data_type_set_value_owned (vala_expression_get_target_type (vala_assignment_get_right (self)), FALSE);
			if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (self, (ValaExpression*) bin);
			vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer);
			vala_assignment_set_operator (self, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL) ? NULL : (old_value = (vala_code_node_unref (old_value), NULL));
			(bin == NULL) ? NULL : (bin = (vala_code_node_unref (bin), NULL));
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self)))) {
		ValaSignal* _tmp24_;
		ValaSignal* sig;
		ValaMethod* _tmp26_;
		ValaSymbol* _tmp25_;
		ValaMethod* m;
		ValaDynamicSignal* _tmp28_;
		ValaSignal* _tmp27_;
		ValaDynamicSignal* dynamic_sig;
		ValaMemberAccess* _tmp30_;
		ValaExpression* _tmp29_;
		ValaMemberAccess* right_ma;
		_tmp24_ = NULL;
		sig = (_tmp24_ = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self))), (_tmp24_ == NULL) ? NULL : vala_code_node_ref (_tmp24_));
		_tmp26_ = NULL;
		_tmp25_ = NULL;
		m = (_tmp26_ = (_tmp25_ = vala_expression_get_symbol_reference (vala_assignment_get_right (self)), VALA_IS_METHOD (_tmp25_) ? ((ValaMethod*) _tmp25_) : NULL), (_tmp26_ == NULL) ? NULL : vala_code_node_ref (_tmp26_));
		if (m == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), "unsupported expression for signal handler");
			result = FALSE;
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			return result;
		}
		_tmp28_ = NULL;
		_tmp27_ = NULL;
		dynamic_sig = (_tmp28_ = (_tmp27_ = sig, VALA_IS_DYNAMIC_SIGNAL (_tmp27_) ? ((ValaDynamicSignal*) _tmp27_) : NULL), (_tmp28_ == NULL) ? NULL : vala_code_node_ref (_tmp28_));
		_tmp30_ = NULL;
		_tmp29_ = NULL;
		right_ma = (_tmp30_ = (_tmp29_ = vala_assignment_get_right (self), VALA_IS_MEMBER_ACCESS (_tmp29_) ? ((ValaMemberAccess*) _tmp29_) : NULL), (_tmp30_ == NULL) ? NULL : vala_code_node_ref (_tmp30_));
		if (dynamic_sig != NULL) {
			gboolean first;
			ValaDelegateType* _tmp36_;
			ValaDelegate* _tmp35_;
			ValaObjectType* _tmp34_;
			first = TRUE;
			{
				GeeList* _tmp31_;
				GeeIterator* _tmp32_;
				GeeIterator* _param_it;
				_tmp31_ = NULL;
				_tmp32_ = NULL;
				_param_it = (_tmp32_ = gee_iterable_iterator ((GeeIterable*) (_tmp31_ = vala_data_type_get_parameters (vala_expression_get_value_type (vala_dynamic_signal_get_handler (dynamic_sig))))), (_tmp31_ == NULL) ? NULL : (_tmp31_ = (gee_collection_object_unref (_tmp31_), NULL)), _tmp32_);
				while (TRUE) {
					ValaFormalParameter* param;
					if (!gee_iterator_next (_param_it)) {
						break;
					}
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					if (first) {
						/* skip sender parameter*/
						first = FALSE;
					} else {
						ValaFormalParameter* _tmp33_;
						_tmp33_ = NULL;
						vala_signal_add_parameter ((ValaSignal*) dynamic_sig, _tmp33_ = vala_formal_parameter_copy (param));
						(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_code_node_unref (_tmp33_), NULL));
					}
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			_tmp36_ = NULL;
			_tmp35_ = NULL;
			_tmp34_ = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp36_ = vala_delegate_type_new (_tmp35_ = vala_signal_get_delegate (sig, (ValaDataType*) (_tmp34_ = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) sig)))), (ValaCodeNode*) self))));
			(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_code_node_unref (_tmp36_), NULL));
			(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_code_node_unref (_tmp35_), NULL));
			(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_code_node_unref (_tmp34_), NULL));
		} else {
			if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_target_type (vala_assignment_get_right (self)))) {
				ValaDelegateType* _tmp37_;
				ValaDelegateType* delegate_type;
				char* _tmp41_;
				char* _tmp40_;
				char* _tmp39_;
				char* _tmp38_;
				_tmp37_ = NULL;
				delegate_type = (_tmp37_ = VALA_DELEGATE_TYPE (vala_expression_get_target_type (vala_assignment_get_right (self))), (_tmp37_ == NULL) ? NULL : vala_code_node_ref (_tmp37_));
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp41_ = NULL;
				_tmp40_ = NULL;
				_tmp39_ = NULL;
				_tmp38_ = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp41_ = g_strdup_printf ("method `%s' is incompatible with signal `%s', expected `%s'", _tmp38_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp39_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type (vala_assignment_get_right (self))), _tmp40_ = vala_delegate_get_prototype_string (vala_delegate_type_get_delegate_symbol (delegate_type), vala_symbol_get_name ((ValaSymbol*) m))));
				_tmp41_ = (g_free (_tmp41_), NULL);
				_tmp40_ = (g_free (_tmp40_), NULL);
				_tmp39_ = (g_free (_tmp39_), NULL);
				_tmp38_ = (g_free (_tmp38_), NULL);
				result = FALSE;
				(delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
				(dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL));
				(right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL));
				return result;
			} else {
				gboolean _tmp42_;
				_tmp42_ = FALSE;
				if (right_ma != NULL) {
					_tmp42_ = vala_member_access_get_prototype_access (right_ma);
				} else {
					_tmp42_ = FALSE;
				}
				if (_tmp42_) {
					char* _tmp44_;
					char* _tmp43_;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp44_ = NULL;
					_tmp43_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp44_ = g_strdup_printf ("Access to instance member `%s' denied", _tmp43_ = vala_symbol_get_full_name ((ValaSymbol*) m)));
					_tmp44_ = (g_free (_tmp44_), NULL);
					_tmp43_ = (g_free (_tmp43_), NULL);
					result = FALSE;
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
					(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					(dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL));
					(right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL));
					return result;
				}
			}
		}
		(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		(dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL));
		(right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL));
	} else {
		if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
			ValaMemberAccess* _tmp45_;
			ValaMemberAccess* ma;
			gboolean _tmp76_;
			_tmp45_ = NULL;
			ma = (_tmp45_ = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp45_ == NULL) ? NULL : vala_code_node_ref (_tmp45_));
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaProperty* _tmp46_;
				ValaProperty* prop;
				ValaDynamicProperty* _tmp48_;
				ValaProperty* _tmp47_;
				ValaDynamicProperty* dynamic_prop;
				gboolean _tmp51_;
				_tmp46_ = NULL;
				prop = (_tmp46_ = VALA_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp46_ == NULL) ? NULL : vala_code_node_ref (_tmp46_));
				_tmp48_ = NULL;
				_tmp47_ = NULL;
				dynamic_prop = (_tmp48_ = (_tmp47_ = prop, VALA_IS_DYNAMIC_PROPERTY (_tmp47_) ? ((ValaDynamicProperty*) _tmp47_) : NULL), (_tmp48_ == NULL) ? NULL : vala_code_node_ref (_tmp48_));
				if (dynamic_prop != NULL) {
					ValaDataType* _tmp49_;
					ValaDataType* _tmp50_;
					_tmp49_ = NULL;
					vala_property_set_property_type ((ValaProperty*) dynamic_prop, _tmp49_ = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_right (self))));
					(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_code_node_unref (_tmp49_), NULL));
					_tmp50_ = NULL;
					vala_expression_set_value_type (vala_assignment_get_left (self), _tmp50_ = vala_data_type_copy (vala_property_get_property_type ((ValaProperty*) dynamic_prop)));
					(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_code_node_unref (_tmp50_), NULL));
				}
				_tmp51_ = FALSE;
				if (vala_property_get_set_accessor (prop) == NULL) {
					_tmp51_ = TRUE;
				} else {
					gboolean _tmp52_;
					_tmp52_ = FALSE;
					if (!vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
						gboolean _tmp53_;
						ValaMethod* _tmp54_;
						gboolean _tmp55_;
						_tmp53_ = FALSE;
						_tmp54_ = NULL;
						if ((_tmp55_ = VALA_IS_CREATION_METHOD (_tmp54_ = vala_semantic_analyzer_find_current_method (analyzer)), (_tmp54_ == NULL) ? NULL : (_tmp54_ = (vala_code_node_unref (_tmp54_), NULL)), _tmp55_)) {
							_tmp53_ = TRUE;
						} else {
							_tmp53_ = vala_semantic_analyzer_is_in_constructor (analyzer);
						}
						_tmp52_ = !_tmp53_;
					} else {
						_tmp52_ = FALSE;
					}
					_tmp51_ = _tmp52_;
				}
				if (_tmp51_) {
					char* _tmp57_;
					char* _tmp56_;
					vala_code_node_set_error ((ValaCodeNode*) ma, TRUE);
					_tmp57_ = NULL;
					_tmp56_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) ma), _tmp57_ = g_strdup_printf ("Property `%s' is read-only", _tmp56_ = vala_symbol_get_full_name ((ValaSymbol*) prop)));
					_tmp57_ = (g_free (_tmp57_), NULL);
					_tmp56_ = (g_free (_tmp56_), NULL);
					result = FALSE;
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					(dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL));
					(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
					return result;
				}
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				(dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL));
			} else {
				gboolean _tmp58_;
				_tmp58_ = FALSE;
				if (VALA_IS_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					_tmp58_ = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
				} else {
					_tmp58_ = FALSE;
				}
				if (_tmp58_) {
					ValaLocalVariable* _tmp59_;
					ValaLocalVariable* local;
					gboolean _tmp60_;
					_tmp59_ = NULL;
					local = (_tmp59_ = VALA_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp59_ == NULL) ? NULL : vala_code_node_ref (_tmp59_));
					_tmp60_ = FALSE;
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
						_tmp60_ = VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (local));
					} else {
						_tmp60_ = FALSE;
					}
					if (_tmp60_) {
						ValaMethod* _tmp61_;
						ValaMethod* m;
						ValaDelegateType* _tmp62_;
						ValaDelegateType* dt;
						ValaDelegate* _tmp63_;
						ValaDelegate* cb;
						_tmp61_ = NULL;
						m = (_tmp61_ = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp61_ == NULL) ? NULL : vala_code_node_ref (_tmp61_));
						_tmp62_ = NULL;
						dt = (_tmp62_ = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (local)), (_tmp62_ == NULL) ? NULL : vala_code_node_ref (_tmp62_));
						_tmp63_ = NULL;
						cb = (_tmp63_ = vala_delegate_type_get_delegate_symbol (dt), (_tmp63_ == NULL) ? NULL : vala_code_node_ref (_tmp63_));
						/* check whether method matches callback type */
						if (!vala_delegate_matches_method (cb, m)) {
							char* _tmp66_;
							char* _tmp65_;
							char* _tmp64_;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							_tmp66_ = NULL;
							_tmp65_ = NULL;
							_tmp64_ = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp66_ = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp64_ = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp65_ = vala_symbol_get_full_name ((ValaSymbol*) cb)));
							_tmp66_ = (g_free (_tmp66_), NULL);
							_tmp65_ = (g_free (_tmp65_), NULL);
							_tmp64_ = (g_free (_tmp64_), NULL);
							result = FALSE;
							(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
							(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
							(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
							(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
							(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
							return result;
						}
						vala_expression_set_value_type (vala_assignment_get_right (self), vala_local_variable_get_variable_type (local));
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
						(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
						(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
					} else {
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
						result = FALSE;
						(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
						(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
						return result;
					}
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				} else {
					gboolean _tmp67_;
					_tmp67_ = FALSE;
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
						_tmp67_ = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
					} else {
						_tmp67_ = FALSE;
					}
					if (_tmp67_) {
						ValaField* _tmp68_;
						ValaField* f;
						gboolean _tmp69_;
						_tmp68_ = NULL;
						f = (_tmp68_ = VALA_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp68_ == NULL) ? NULL : vala_code_node_ref (_tmp68_));
						_tmp69_ = FALSE;
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
							_tmp69_ = VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f));
						} else {
							_tmp69_ = FALSE;
						}
						if (_tmp69_) {
							ValaMethod* _tmp70_;
							ValaMethod* m;
							ValaDelegateType* _tmp71_;
							ValaDelegateType* dt;
							ValaDelegate* _tmp72_;
							ValaDelegate* cb;
							_tmp70_ = NULL;
							m = (_tmp70_ = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp70_ == NULL) ? NULL : vala_code_node_ref (_tmp70_));
							_tmp71_ = NULL;
							dt = (_tmp71_ = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp71_ == NULL) ? NULL : vala_code_node_ref (_tmp71_));
							_tmp72_ = NULL;
							cb = (_tmp72_ = vala_delegate_type_get_delegate_symbol (dt), (_tmp72_ == NULL) ? NULL : vala_code_node_ref (_tmp72_));
							/* check whether method matches callback type */
							if (!vala_delegate_matches_method (cb, m)) {
								char* _tmp75_;
								char* _tmp74_;
								char* _tmp73_;
								vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
								_tmp75_ = NULL;
								_tmp74_ = NULL;
								_tmp73_ = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp75_ = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp73_ = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp74_ = vala_symbol_get_full_name ((ValaSymbol*) cb)));
								_tmp75_ = (g_free (_tmp75_), NULL);
								_tmp74_ = (g_free (_tmp74_), NULL);
								_tmp73_ = (g_free (_tmp73_), NULL);
								result = FALSE;
								(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
								(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
								(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
								(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
								return result;
							}
							vala_expression_set_value_type (vala_assignment_get_right (self), vala_field_get_field_type (f));
							(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
							(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
							(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
						} else {
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
							result = FALSE;
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
							(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
							return result;
						}
						(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					}
				}
			}
			_tmp76_ = FALSE;
			if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
				_tmp76_ = vala_expression_get_value_type (vala_assignment_get_right (self)) != NULL;
			} else {
				_tmp76_ = FALSE;
			}
			if (_tmp76_) {
				/* if there was an error on either side,
				 * i.e. {left|right}.value_type == null, skip type check */
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp79_;
					char* _tmp78_;
					char* _tmp77_;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp79_ = NULL;
					_tmp78_ = NULL;
					_tmp77_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp79_ = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp77_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp78_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp79_ = (g_free (_tmp79_), NULL);
					_tmp78_ = (g_free (_tmp78_), NULL);
					_tmp77_ = (g_free (_tmp77_), NULL);
					result = FALSE;
					(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
					return result;
				}
				if (!VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
						gboolean _tmp80_;
						_tmp80_ = FALSE;
						if (!VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
							_tmp80_ = !vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)));
						} else {
							_tmp80_ = FALSE;
						}
						/* rhs transfers ownership of the expression */
						if (_tmp80_) {
							/* lhs doesn't own the value */
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						}
					} else {
						if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
						}
					}
				}
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		} else {
			if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
				ValaElementAccess* _tmp81_;
				ValaElementAccess* ea;
				_tmp81_ = NULL;
				ea = (_tmp81_ = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp81_ == NULL) ? NULL : vala_code_node_ref (_tmp81_));
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp84_;
					char* _tmp83_;
					char* _tmp82_;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp84_ = NULL;
					_tmp83_ = NULL;
					_tmp82_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp84_ = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp82_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp83_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp84_ = (g_free (_tmp84_), NULL);
					_tmp83_ = (g_free (_tmp83_), NULL);
					_tmp82_ = (g_free (_tmp82_), NULL);
					result = FALSE;
					(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
					return result;
				}
				if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
					ValaDataType* element_type;
					gboolean _tmp89_;
					/* rhs transfers ownership of the expression */
					element_type = NULL;
					if (VALA_IS_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea)))) {
						ValaArrayType* _tmp85_;
						ValaArrayType* array_type;
						ValaDataType* _tmp87_;
						ValaDataType* _tmp86_;
						_tmp85_ = NULL;
						array_type = (_tmp85_ = VALA_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea))), (_tmp85_ == NULL) ? NULL : vala_code_node_ref (_tmp85_));
						_tmp87_ = NULL;
						_tmp86_ = NULL;
						element_type = (_tmp87_ = (_tmp86_ = vala_array_type_get_element_type (array_type), (_tmp86_ == NULL) ? NULL : vala_code_node_ref (_tmp86_)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp87_);
						(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					} else {
						GeeList* args;
						ValaDataType* _tmp88_;
						args = vala_data_type_get_type_arguments (vala_expression_get_value_type (vala_element_access_get_container (ea)));
						g_assert (gee_collection_get_size ((GeeCollection*) args) == 1);
						_tmp88_ = NULL;
						element_type = (_tmp88_ = (ValaDataType*) gee_list_get (args, 0), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp88_);
						(args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL));
					}
					_tmp89_ = FALSE;
					if (!VALA_IS_POINTER_TYPE (element_type)) {
						_tmp89_ = !vala_data_type_get_value_owned (element_type);
					} else {
						_tmp89_ = FALSE;
					}
					if (_tmp89_) {
						/* lhs doesn't own the value */
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						result = FALSE;
						(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
						(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
						return result;
					}
					(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				} else {
					if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					}
				}
				(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
			} else {
				/* lhs wants to own the value
				 * rhs doesn't transfer the ownership
				 * code generator needs to add reference
				 * increment calls */
				result = TRUE;
				return result;
			}
		}
	}
	if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
		ValaDataType* _tmp90_;
		_tmp90_ = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp90_ = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_left (self))));
		(_tmp90_ == NULL) ? NULL : (_tmp90_ = (vala_code_node_unref (_tmp90_), NULL));
		vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
	} else {
		vala_expression_set_value_type ((ValaExpression*) self, NULL);
	}
	_tmp91_ = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp91_ = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_left (self)));
	(_tmp91_ == NULL) ? NULL : (_tmp91_ = (gee_collection_object_unref (_tmp91_), NULL));
	_tmp92_ = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp92_ = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_right (self)));
	(_tmp92_ == NULL) ? NULL : (_tmp92_ = (gee_collection_object_unref (_tmp92_), NULL));
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	return result;
}


static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaLocalVariable* _tmp1_;
	ValaSymbol* _tmp0_;
	ValaLocalVariable* local;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_left (self), collection);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	local = (_tmp1_ = (_tmp0_ = vala_expression_get_symbol_reference (vala_assignment_get_left (self)), VALA_IS_LOCAL_VARIABLE (_tmp0_) ? ((ValaLocalVariable*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (local != NULL) {
		gee_collection_add (collection, local);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaMemberAccess* _tmp1_;
	ValaExpression* _tmp0_;
	ValaMemberAccess* ma;
	ValaElementAccess* _tmp3_;
	ValaExpression* _tmp2_;
	ValaElementAccess* ea;
	gboolean _tmp4_;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	ma = (_tmp1_ = (_tmp0_ = vala_assignment_get_left (self), VALA_IS_MEMBER_ACCESS (_tmp0_) ? ((ValaMemberAccess*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	ea = (_tmp3_ = (_tmp2_ = vala_assignment_get_left (self), VALA_IS_ELEMENT_ACCESS (_tmp2_) ? ((ValaElementAccess*) _tmp2_) : NULL), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
	_tmp4_ = FALSE;
	if (ma != NULL) {
		_tmp4_ = vala_member_access_get_inner (ma) != NULL;
	} else {
		_tmp4_ = FALSE;
	}
	if (_tmp4_) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_member_access_get_inner (ma), collection);
	} else {
		if (ea != NULL) {
			vala_code_node_get_used_variables ((ValaCodeNode*) ea, collection);
		}
	}
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
}


ValaExpression* vala_assignment_get_left (ValaAssignment* self) {
	ValaExpression* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_left;
	return result;
}


void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp1_;
	ValaExpression* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_left = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL)), _tmp1_);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_left, (ValaCodeNode*) self);
}


ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self) {
	ValaAssignmentOperator result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_operator;
	return result;
}


void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaExpression* vala_assignment_get_right (ValaAssignment* self) {
	ValaExpression* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_right;
	return result;
}


void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp1_;
	ValaExpression* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_right = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL)), _tmp1_);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_right, (ValaCodeNode*) self);
}


static void vala_assignment_class_init (ValaAssignmentClass * klass) {
	vala_assignment_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_assignment_finalize;
	g_type_class_add_private (klass, sizeof (ValaAssignmentPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_assignment_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_assignment_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_assignment_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_assignment_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_assignment_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_assignment_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_assignment_real_get_used_variables;
}


static void vala_assignment_instance_init (ValaAssignment * self) {
	self->priv = VALA_ASSIGNMENT_GET_PRIVATE (self);
}


static void vala_assignment_finalize (ValaCodeNode* obj) {
	ValaAssignment * self;
	self = VALA_ASSIGNMENT (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL));
	VALA_CODE_NODE_CLASS (vala_assignment_parent_class)->finalize (obj);
}


GType vala_assignment_get_type (void) {
	static GType vala_assignment_type_id = 0;
	if (vala_assignment_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaAssignmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_assignment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaAssignment), 0, (GInstanceInitFunc) vala_assignment_instance_init, NULL };
		vala_assignment_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaAssignment", &g_define_type_info, 0);
	}
	return vala_assignment_type_id;
}



GType vala_assignment_operator_get_type (void) {
	static GType vala_assignment_operator_type_id = 0;
	if (G_UNLIKELY (vala_assignment_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_ASSIGNMENT_OPERATOR_NONE, "VALA_ASSIGNMENT_OPERATOR_NONE", "none"}, {VALA_ASSIGNMENT_OPERATOR_SIMPLE, "VALA_ASSIGNMENT_OPERATOR_SIMPLE", "simple"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_OR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_AND, "VALA_ASSIGNMENT_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_ASSIGNMENT_OPERATOR_ADD, "VALA_ASSIGNMENT_OPERATOR_ADD", "add"}, {VALA_ASSIGNMENT_OPERATOR_SUB, "VALA_ASSIGNMENT_OPERATOR_SUB", "sub"}, {VALA_ASSIGNMENT_OPERATOR_MUL, "VALA_ASSIGNMENT_OPERATOR_MUL", "mul"}, {VALA_ASSIGNMENT_OPERATOR_DIV, "VALA_ASSIGNMENT_OPERATOR_DIV", "div"}, {VALA_ASSIGNMENT_OPERATOR_PERCENT, "VALA_ASSIGNMENT_OPERATOR_PERCENT", "percent"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT", "shift-right"}, {0, NULL, NULL}};
		vala_assignment_operator_type_id = g_enum_register_static ("ValaAssignmentOperator", values);
	}
	return vala_assignment_operator_type_id;
}




