/* valaarraycreationexpression.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaExpressionPrivate ValaExpressionPrivate;

#define VALA_TYPE_ARRAY_CREATION_EXPRESSION (vala_array_creation_expression_get_type ())
#define VALA_ARRAY_CREATION_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_CREATION_EXPRESSION, ValaArrayCreationExpression))
#define VALA_ARRAY_CREATION_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_CREATION_EXPRESSION, ValaArrayCreationExpressionClass))
#define VALA_IS_ARRAY_CREATION_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_CREATION_EXPRESSION))
#define VALA_IS_ARRAY_CREATION_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_CREATION_EXPRESSION))
#define VALA_ARRAY_CREATION_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_CREATION_EXPRESSION, ValaArrayCreationExpressionClass))

typedef struct _ValaArrayCreationExpression ValaArrayCreationExpression;
typedef struct _ValaArrayCreationExpressionClass ValaArrayCreationExpressionClass;
typedef struct _ValaArrayCreationExpressionPrivate ValaArrayCreationExpressionPrivate;

#define VALA_TYPE_INITIALIZER_LIST (vala_initializer_list_get_type ())
#define VALA_INITIALIZER_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INITIALIZER_LIST, ValaInitializerList))
#define VALA_INITIALIZER_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListClass))
#define VALA_IS_INITIALIZER_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INITIALIZER_LIST))
#define VALA_IS_INITIALIZER_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INITIALIZER_LIST))
#define VALA_INITIALIZER_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListClass))

typedef struct _ValaInitializerList ValaInitializerList;
typedef struct _ValaInitializerListClass ValaInitializerListClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_LITERAL (vala_literal_get_type ())
#define VALA_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LITERAL, ValaLiteral))
#define VALA_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LITERAL, ValaLiteralClass))
#define VALA_IS_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LITERAL))
#define VALA_IS_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LITERAL))
#define VALA_LITERAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LITERAL, ValaLiteralClass))

typedef struct _ValaLiteral ValaLiteral;
typedef struct _ValaLiteralClass ValaLiteralClass;

#define VALA_TYPE_INTEGER_LITERAL (vala_integer_literal_get_type ())
#define VALA_INTEGER_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTEGER_LITERAL, ValaIntegerLiteral))
#define VALA_INTEGER_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTEGER_LITERAL, ValaIntegerLiteralClass))
#define VALA_IS_INTEGER_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTEGER_LITERAL))
#define VALA_IS_INTEGER_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTEGER_LITERAL))
#define VALA_INTEGER_LITERAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTEGER_LITERAL, ValaIntegerLiteralClass))

typedef struct _ValaIntegerLiteral ValaIntegerLiteral;
typedef struct _ValaIntegerLiteralClass ValaIntegerLiteralClass;

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_ARRAY_TYPE (vala_array_type_get_type ())
#define VALA_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayType))
#define VALA_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))
#define VALA_IS_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_TYPE))
#define VALA_IS_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_TYPE))
#define VALA_ARRAY_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))

typedef struct _ValaArrayType ValaArrayType;
typedef struct _ValaArrayTypeClass ValaArrayTypeClass;

#define VALA_TYPE_VALUE_TYPE (vala_value_type_get_type ())
#define VALA_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_VALUE_TYPE, ValaValueType))
#define VALA_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_VALUE_TYPE, ValaValueTypeClass))
#define VALA_IS_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_VALUE_TYPE))
#define VALA_IS_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_VALUE_TYPE))
#define VALA_VALUE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_VALUE_TYPE, ValaValueTypeClass))

typedef struct _ValaValueType ValaValueType;
typedef struct _ValaValueTypeClass ValaValueTypeClass;

#define VALA_TYPE_INTEGER_TYPE (vala_integer_type_get_type ())
#define VALA_INTEGER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTEGER_TYPE, ValaIntegerType))
#define VALA_INTEGER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTEGER_TYPE, ValaIntegerTypeClass))
#define VALA_IS_INTEGER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTEGER_TYPE))
#define VALA_IS_INTEGER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTEGER_TYPE))
#define VALA_INTEGER_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTEGER_TYPE, ValaIntegerTypeClass))

typedef struct _ValaIntegerType ValaIntegerType;
typedef struct _ValaIntegerTypeClass ValaIntegerTypeClass;

#define VALA_TYPE_ENUM_VALUE_TYPE (vala_enum_value_type_get_type ())
#define VALA_ENUM_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM_VALUE_TYPE, ValaEnumValueType))
#define VALA_ENUM_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM_VALUE_TYPE, ValaEnumValueTypeClass))
#define VALA_IS_ENUM_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM_VALUE_TYPE))
#define VALA_IS_ENUM_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM_VALUE_TYPE))
#define VALA_ENUM_VALUE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM_VALUE_TYPE, ValaEnumValueTypeClass))

typedef struct _ValaEnumValueType ValaEnumValueType;
typedef struct _ValaEnumValueTypeClass ValaEnumValueTypeClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Base class for all code nodes that might be used as an expression.
 */
struct _ValaExpression {
	ValaCodeNode parent_instance;
	ValaExpressionPrivate * priv;
	GeeArrayList* temp_vars;
};

struct _ValaExpressionClass {
	ValaCodeNodeClass parent_class;
	gboolean (*is_constant) (ValaExpression* self);
	gboolean (*is_pure) (ValaExpression* self);
	gboolean (*is_non_null) (ValaExpression* self);
};

/**
 * Represents an array creation expression e.g. "new int[] {1,2,3}".
 */
struct _ValaArrayCreationExpression {
	ValaExpression parent_instance;
	ValaArrayCreationExpressionPrivate * priv;
};

struct _ValaArrayCreationExpressionClass {
	ValaExpressionClass parent_class;
};

struct _ValaArrayCreationExpressionPrivate {
	gint _rank;
	GeeList* sizes;
	ValaDataType* _element_type;
	ValaInitializerList* _initializer_list;
};


static gpointer vala_array_creation_expression_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_array_creation_expression_get_type (void);
GType vala_initializer_list_get_type (void);
#define VALA_ARRAY_CREATION_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ARRAY_CREATION_EXPRESSION, ValaArrayCreationExpressionPrivate))
enum  {
	VALA_ARRAY_CREATION_EXPRESSION_DUMMY_PROPERTY
};
void vala_array_creation_expression_append_size (ValaArrayCreationExpression* self, ValaExpression* size);
GeeList* vala_array_creation_expression_get_sizes (ValaArrayCreationExpression* self);
void vala_array_creation_expression_set_element_type (ValaArrayCreationExpression* self, ValaDataType* value);
void vala_array_creation_expression_set_rank (ValaArrayCreationExpression* self, gint value);
void vala_array_creation_expression_set_initializer_list (ValaArrayCreationExpression* self, ValaInitializerList* value);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
ValaArrayCreationExpression* vala_array_creation_expression_new (ValaDataType* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference);
ValaArrayCreationExpression* vala_array_creation_expression_construct (GType object_type, ValaDataType* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference);
ValaArrayCreationExpression* vala_array_creation_expression_new (ValaDataType* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference);
ValaDataType* vala_array_creation_expression_get_element_type (ValaArrayCreationExpression* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaInitializerList* vala_array_creation_expression_get_initializer_list (ValaArrayCreationExpression* self);
static void vala_array_creation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
void vala_code_visitor_visit_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr);
void vala_code_visitor_visit_expression (ValaCodeVisitor* self, ValaExpression* expr);
static void vala_array_creation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_array_creation_expression_real_is_pure (ValaExpression* base);
static void vala_array_creation_expression_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
gint vala_array_creation_expression_get_rank (ValaArrayCreationExpression* self);
gint vala_initializer_list_get_size (ValaInitializerList* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
ValaIntegerLiteral* vala_integer_literal_new (const char* i, ValaSourceReference* source);
ValaIntegerLiteral* vala_integer_literal_construct (GType object_type, const char* i, ValaSourceReference* source);
GType vala_literal_get_type (void);
GType vala_integer_literal_get_type (void);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
GeeList* vala_initializer_list_get_initializers (ValaInitializerList* self);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
void vala_report_error (ValaSourceReference* source, const char* message);
static gint vala_array_creation_expression_create_sizes_from_initializer_list (ValaArrayCreationExpression* self, ValaSemanticAnalyzer* analyzer, ValaInitializerList* il, gint rank, GeeList* sl);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
ValaArrayType* vala_array_type_new (ValaDataType* element_type, gint rank, ValaSourceReference* source_reference);
ValaArrayType* vala_array_type_construct (GType object_type, ValaDataType* element_type, gint rank, ValaSourceReference* source_reference);
GType vala_reference_type_get_type (void);
GType vala_array_type_get_type (void);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
GType vala_value_type_get_type (void);
GType vala_integer_type_get_type (void);
GType vala_enum_value_type_get_type (void);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value);
static gboolean vala_array_creation_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
static void vala_array_creation_expression_finalize (ValaCodeNode* obj);



/**
 * Add a size expression.
 */
void vala_array_creation_expression_append_size (ValaArrayCreationExpression* self, ValaExpression* size) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (size != NULL);
	gee_collection_add ((GeeCollection*) self->priv->sizes, size);
}


/**
 * Get the sizes for all dimensions ascending from left to right.
 */
GeeList* vala_array_creation_expression_get_sizes (ValaArrayCreationExpression* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->sizes);
	return result;
}


ValaArrayCreationExpression* vala_array_creation_expression_construct (GType object_type, ValaDataType* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference) {
	ValaArrayCreationExpression* self;
	g_return_val_if_fail (element_type != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaArrayCreationExpression*) g_type_create_instance (object_type);
	vala_array_creation_expression_set_element_type (self, element_type);
	vala_array_creation_expression_set_rank (self, rank);
	vala_array_creation_expression_set_initializer_list (self, initializer_list);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaArrayCreationExpression* vala_array_creation_expression_new (ValaDataType* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference) {
	return vala_array_creation_expression_construct (VALA_TYPE_ARRAY_CREATION_EXPRESSION, element_type, rank, initializer_list, source_reference);
}


static void vala_array_creation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaArrayCreationExpression * self;
	self = (ValaArrayCreationExpression*) base;
	g_return_if_fail (visitor != NULL);
	if (vala_array_creation_expression_get_element_type (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_array_creation_expression_get_element_type (self), visitor);
	}
	{
		GeeIterator* _e_it;
		_e_it = gee_iterable_iterator ((GeeIterable*) self->priv->sizes);
		while (TRUE) {
			ValaExpression* e;
			if (!gee_iterator_next (_e_it)) {
				break;
			}
			e = (ValaExpression*) gee_iterator_get (_e_it);
			vala_code_node_accept ((ValaCodeNode*) e, visitor);
			(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
		}
		(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
	}
	if (vala_array_creation_expression_get_initializer_list (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_array_creation_expression_get_initializer_list (self), visitor);
	}
}


static void vala_array_creation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaArrayCreationExpression * self;
	self = (ValaArrayCreationExpression*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_array_creation_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static gboolean vala_array_creation_expression_real_is_pure (ValaExpression* base) {
	ValaArrayCreationExpression * self;
	gboolean result;
	self = (ValaArrayCreationExpression*) base;
	result = FALSE;
	return result;
}


static void vala_array_creation_expression_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaArrayCreationExpression * self;
	self = (ValaArrayCreationExpression*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_array_creation_expression_get_element_type (self) == old_type) {
		vala_array_creation_expression_set_element_type (self, new_type);
	}
}


static gint vala_array_creation_expression_create_sizes_from_initializer_list (ValaArrayCreationExpression* self, ValaSemanticAnalyzer* analyzer, ValaInitializerList* il, gint rank, GeeList* sl) {
	gint result;
	gint subsize;
	g_return_val_if_fail (self != NULL, 0);
	g_return_val_if_fail (analyzer != NULL, 0);
	g_return_val_if_fail (il != NULL, 0);
	g_return_val_if_fail (sl != NULL, 0);
	if (gee_collection_get_size ((GeeCollection*) sl) == (self->priv->_rank - rank)) {
		char* _tmp0_;
		ValaIntegerLiteral* _tmp1_;
		ValaIntegerLiteral* init;
		/* only add size if this is the first initializer list of the current dimension*/
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		init = (_tmp1_ = vala_integer_literal_new (_tmp0_ = g_strdup_printf ("%i", vala_initializer_list_get_size (il)), vala_code_node_get_source_reference ((ValaCodeNode*) il)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
		vala_code_node_check ((ValaCodeNode*) init, analyzer);
		gee_collection_add ((GeeCollection*) sl, (ValaLiteral*) init);
		(init == NULL) ? NULL : (init = (vala_code_node_unref (init), NULL));
	}
	subsize = -1;
	{
		GeeList* _tmp2_;
		GeeIterator* _tmp3_;
		GeeIterator* _e_it;
		_tmp2_ = NULL;
		_tmp3_ = NULL;
		_e_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = vala_initializer_list_get_initializers (il))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp3_);
		while (TRUE) {
			ValaExpression* e;
			if (!gee_iterator_next (_e_it)) {
				break;
			}
			e = (ValaExpression*) gee_iterator_get (_e_it);
			if (VALA_IS_INITIALIZER_LIST (e)) {
				gint size;
				gboolean _tmp4_;
				if (rank == 1) {
					vala_code_node_set_error ((ValaCodeNode*) il, TRUE);
					vala_code_node_set_error ((ValaCodeNode*) e, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), "Expected array element, got array initializer list");
					result = -1;
					(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
					(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					return result;
				}
				size = vala_array_creation_expression_create_sizes_from_initializer_list (self, analyzer, VALA_INITIALIZER_LIST (e), rank - 1, sl);
				if (size == (-1)) {
					result = -1;
					(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
					(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					return result;
				}
				_tmp4_ = FALSE;
				if (subsize >= 0) {
					_tmp4_ = subsize != size;
				} else {
					_tmp4_ = FALSE;
				}
				if (_tmp4_) {
					char* _tmp5_;
					vala_code_node_set_error ((ValaCodeNode*) il, TRUE);
					_tmp5_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) il), _tmp5_ = g_strdup_printf ("Expected initializer list of size %d, got size %d", subsize, size));
					_tmp5_ = (g_free (_tmp5_), NULL);
					result = -1;
					(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
					(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					return result;
				} else {
					subsize = size;
				}
			} else {
				if (rank != 1) {
					vala_code_node_set_error ((ValaCodeNode*) il, TRUE);
					vala_code_node_set_error ((ValaCodeNode*) e, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), "Expected array initializer list, got array element");
					result = -1;
					(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
					(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					return result;
				}
			}
			(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
		}
		(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
	}
	result = vala_initializer_list_get_size (il);
	return result;
}


static gboolean vala_array_creation_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaArrayCreationExpression * self;
	gboolean result;
	GeeList* sizes;
	ValaInitializerList* _tmp0_;
	ValaInitializerList* initlist;
	GeeArrayList* calc_sizes;
	gboolean _tmp3_;
	ValaArrayType* _tmp4_;
	self = (ValaArrayCreationExpression*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	sizes = vala_array_creation_expression_get_sizes (self);
	_tmp0_ = NULL;
	initlist = (_tmp0_ = vala_array_creation_expression_get_initializer_list (self), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	if (vala_array_creation_expression_get_element_type (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_array_creation_expression_get_element_type (self), analyzer);
	}
	{
		GeeIterator* _e_it;
		_e_it = gee_iterable_iterator ((GeeIterable*) sizes);
		while (TRUE) {
			ValaExpression* e;
			if (!gee_iterator_next (_e_it)) {
				break;
			}
			e = (ValaExpression*) gee_iterator_get (_e_it);
			vala_code_node_check ((ValaCodeNode*) e, analyzer);
			(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
		}
		(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
	}
	calc_sizes = gee_array_list_new (VALA_TYPE_LITERAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (initlist != NULL) {
		ValaArrayType* _tmp1_;
		gint ret;
		_tmp1_ = NULL;
		vala_expression_set_target_type ((ValaExpression*) initlist, (ValaDataType*) (_tmp1_ = vala_array_type_new (vala_array_creation_expression_get_element_type (self), self->priv->_rank, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL));
		vala_code_node_check ((ValaCodeNode*) initlist, analyzer);
		ret = vala_array_creation_expression_create_sizes_from_initializer_list (self, analyzer, initlist, self->priv->_rank, (GeeList*) calc_sizes);
		if (ret == (-1)) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		}
	}
	if (gee_collection_get_size ((GeeCollection*) sizes) > 0) {
		/* check for errors in the size list */
		{
			GeeIterator* _e_it;
			/* check for errors in the size list */
			_e_it = gee_iterable_iterator ((GeeIterable*) sizes);
			/* check for errors in the size list */
			while (TRUE) {
				ValaExpression* e;
				/* check for errors in the size list */
				if (!gee_iterator_next (_e_it)) {
					/* check for errors in the size list */
					break;
				}
				/* check for errors in the size list */
				e = (ValaExpression*) gee_iterator_get (_e_it);
				if (vala_expression_get_value_type (e) == NULL) {
					/* return on previous error */
					result = FALSE;
					(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
					(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					(sizes == NULL) ? NULL : (sizes = (gee_collection_object_unref (sizes), NULL));
					(initlist == NULL) ? NULL : (initlist = (vala_code_node_unref (initlist), NULL));
					(calc_sizes == NULL) ? NULL : (calc_sizes = (gee_collection_object_unref (calc_sizes), NULL));
					return result;
				} else {
					gboolean _tmp2_;
					_tmp2_ = FALSE;
					if (VALA_IS_INTEGER_TYPE (vala_expression_get_value_type (e))) {
						_tmp2_ = TRUE;
					} else {
						_tmp2_ = VALA_IS_ENUM_VALUE_TYPE (vala_expression_get_value_type (e));
					}
					if (!_tmp2_) {
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), "Expression of integer type expected");
					}
				}
				(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
			}
			(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
		}
	} else {
		if (initlist == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			/* this is an internal error because it is already handeld by the parser */
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "internal error: initializer list expected");
		} else {
			{
				GeeIterator* _size_it;
				_size_it = gee_iterable_iterator ((GeeIterable*) calc_sizes);
				while (TRUE) {
					ValaExpression* size;
					if (!gee_iterator_next (_size_it)) {
						break;
					}
					size = (ValaExpression*) ((ValaLiteral*) gee_iterator_get (_size_it));
					vala_array_creation_expression_append_size (self, size);
					(size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL));
				}
				(_size_it == NULL) ? NULL : (_size_it = (gee_collection_object_unref (_size_it), NULL));
			}
		}
	}
	if (vala_code_node_get_error ((ValaCodeNode*) self)) {
		result = FALSE;
		(sizes == NULL) ? NULL : (sizes = (gee_collection_object_unref (sizes), NULL));
		(initlist == NULL) ? NULL : (initlist = (vala_code_node_unref (initlist), NULL));
		(calc_sizes == NULL) ? NULL : (calc_sizes = (gee_collection_object_unref (calc_sizes), NULL));
		return result;
	}
	_tmp3_ = FALSE;
	if (vala_array_creation_expression_get_initializer_list (self) != NULL) {
		_tmp3_ = vala_expression_get_value_type ((ValaExpression*) vala_array_creation_expression_get_initializer_list (self)) == NULL;
	} else {
		_tmp3_ = FALSE;
	}
	/* check for wrong elements inside the initializer */
	if (_tmp3_) {
		result = FALSE;
		(sizes == NULL) ? NULL : (sizes = (gee_collection_object_unref (sizes), NULL));
		(initlist == NULL) ? NULL : (initlist = (vala_code_node_unref (initlist), NULL));
		(calc_sizes == NULL) ? NULL : (calc_sizes = (gee_collection_object_unref (calc_sizes), NULL));
		return result;
	}
	/* try to construct the type of the array */
	if (vala_array_creation_expression_get_element_type (self) == NULL) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Cannot determine the element type of the created array");
		result = FALSE;
		(sizes == NULL) ? NULL : (sizes = (gee_collection_object_unref (sizes), NULL));
		(initlist == NULL) ? NULL : (initlist = (vala_code_node_unref (initlist), NULL));
		(calc_sizes == NULL) ? NULL : (calc_sizes = (gee_collection_object_unref (calc_sizes), NULL));
		return result;
	}
	vala_data_type_set_value_owned (vala_array_creation_expression_get_element_type (self), TRUE);
	_tmp4_ = NULL;
	vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp4_ = vala_array_type_new (vala_array_creation_expression_get_element_type (self), self->priv->_rank, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
	vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), TRUE);
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	(sizes == NULL) ? NULL : (sizes = (gee_collection_object_unref (sizes), NULL));
	(initlist == NULL) ? NULL : (initlist = (vala_code_node_unref (initlist), NULL));
	(calc_sizes == NULL) ? NULL : (calc_sizes = (gee_collection_object_unref (calc_sizes), NULL));
	return result;
}


ValaDataType* vala_array_creation_expression_get_element_type (ValaArrayCreationExpression* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_element_type;
	return result;
}


void vala_array_creation_expression_set_element_type (ValaArrayCreationExpression* self, ValaDataType* value) {
	ValaDataType* _tmp1_;
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_element_type = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_)), (self->priv->_element_type == NULL) ? NULL : (self->priv->_element_type = (vala_code_node_unref (self->priv->_element_type), NULL)), _tmp1_);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_element_type, (ValaCodeNode*) self);
}


gint vala_array_creation_expression_get_rank (ValaArrayCreationExpression* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_rank;
	return result;
}


void vala_array_creation_expression_set_rank (ValaArrayCreationExpression* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_rank = value;
}


ValaInitializerList* vala_array_creation_expression_get_initializer_list (ValaArrayCreationExpression* self) {
	ValaInitializerList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_initializer_list;
	return result;
}


void vala_array_creation_expression_set_initializer_list (ValaArrayCreationExpression* self, ValaInitializerList* value) {
	ValaInitializerList* _tmp1_;
	ValaInitializerList* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_initializer_list = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_)), (self->priv->_initializer_list == NULL) ? NULL : (self->priv->_initializer_list = (vala_code_node_unref (self->priv->_initializer_list), NULL)), _tmp1_);
	if (self->priv->_initializer_list != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_initializer_list, (ValaCodeNode*) self);
	}
}


static void vala_array_creation_expression_class_init (ValaArrayCreationExpressionClass * klass) {
	vala_array_creation_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_array_creation_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaArrayCreationExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_array_creation_expression_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_array_creation_expression_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_array_creation_expression_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_array_creation_expression_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_array_creation_expression_real_check;
}


static void vala_array_creation_expression_instance_init (ValaArrayCreationExpression * self) {
	self->priv = VALA_ARRAY_CREATION_EXPRESSION_GET_PRIVATE (self);
	self->priv->sizes = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_array_creation_expression_finalize (ValaCodeNode* obj) {
	ValaArrayCreationExpression * self;
	self = VALA_ARRAY_CREATION_EXPRESSION (obj);
	(self->priv->sizes == NULL) ? NULL : (self->priv->sizes = (gee_collection_object_unref (self->priv->sizes), NULL));
	(self->priv->_element_type == NULL) ? NULL : (self->priv->_element_type = (vala_code_node_unref (self->priv->_element_type), NULL));
	(self->priv->_initializer_list == NULL) ? NULL : (self->priv->_initializer_list = (vala_code_node_unref (self->priv->_initializer_list), NULL));
	VALA_CODE_NODE_CLASS (vala_array_creation_expression_parent_class)->finalize (obj);
}


GType vala_array_creation_expression_get_type (void) {
	static GType vala_array_creation_expression_type_id = 0;
	if (vala_array_creation_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaArrayCreationExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_array_creation_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaArrayCreationExpression), 0, (GInstanceInitFunc) vala_array_creation_expression_instance_init, NULL };
		vala_array_creation_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaArrayCreationExpression", &g_define_type_info, 0);
	}
	return vala_array_creation_expression_type_id;
}




