/* valadbusservermodule.vala
 *
 * Copyright (C) 2007-2009  Jürg Billeter
*  Copyright (C) 2008  Philip Van Hoof
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Philip Van Hoof <pvanhoof@gnome.org>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdlib.h>
#include <string.h>
#include <valaccode.h>
#include <gee.h>


#define VALA_TYPE_CCODE_MODULE (vala_ccode_module_get_type ())
#define VALA_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModule))
#define VALA_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))
#define VALA_IS_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_MODULE))
#define VALA_IS_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_MODULE))
#define VALA_CCODE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))

typedef struct _ValaCCodeModule ValaCCodeModule;
typedef struct _ValaCCodeModuleClass ValaCCodeModuleClass;
typedef struct _ValaCCodeModulePrivate ValaCCodeModulePrivate;

#define VALA_TYPE_CCODE_BASE_MODULE (vala_ccode_base_module_get_type ())
#define VALA_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModule))
#define VALA_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))
#define VALA_IS_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_IS_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_CCODE_BASE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))

typedef struct _ValaCCodeBaseModule ValaCCodeBaseModule;
typedef struct _ValaCCodeBaseModuleClass ValaCCodeBaseModuleClass;
typedef struct _ValaCCodeBaseModulePrivate ValaCCodeBaseModulePrivate;

#define VALA_TYPE_CCODE_DECLARATION_SPACE (vala_ccode_declaration_space_get_type ())
#define VALA_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpace))
#define VALA_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))
#define VALA_IS_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_IS_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_CCODE_DECLARATION_SPACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))

typedef struct _ValaCCodeDeclarationSpace ValaCCodeDeclarationSpace;
typedef struct _ValaCCodeDeclarationSpaceClass ValaCCodeDeclarationSpaceClass;

#define VALA_TYPE_CCODE_STRUCT_MODULE (vala_ccode_struct_module_get_type ())
#define VALA_CCODE_STRUCT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModule))
#define VALA_CCODE_STRUCT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModuleClass))
#define VALA_IS_CCODE_STRUCT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_STRUCT_MODULE))
#define VALA_IS_CCODE_STRUCT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_STRUCT_MODULE))
#define VALA_CCODE_STRUCT_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModuleClass))

typedef struct _ValaCCodeStructModule ValaCCodeStructModule;
typedef struct _ValaCCodeStructModuleClass ValaCCodeStructModuleClass;
typedef struct _ValaCCodeStructModulePrivate ValaCCodeStructModulePrivate;

#define VALA_TYPE_CCODE_METHOD_MODULE (vala_ccode_method_module_get_type ())
#define VALA_CCODE_METHOD_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_METHOD_MODULE, ValaCCodeMethodModule))
#define VALA_CCODE_METHOD_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_METHOD_MODULE, ValaCCodeMethodModuleClass))
#define VALA_IS_CCODE_METHOD_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_METHOD_MODULE))
#define VALA_IS_CCODE_METHOD_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_METHOD_MODULE))
#define VALA_CCODE_METHOD_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_METHOD_MODULE, ValaCCodeMethodModuleClass))

typedef struct _ValaCCodeMethodModule ValaCCodeMethodModule;
typedef struct _ValaCCodeMethodModuleClass ValaCCodeMethodModuleClass;
typedef struct _ValaCCodeMethodModulePrivate ValaCCodeMethodModulePrivate;

#define VALA_TYPE_CCODE_CONTROL_FLOW_MODULE (vala_ccode_control_flow_module_get_type ())
#define VALA_CCODE_CONTROL_FLOW_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_CONTROL_FLOW_MODULE, ValaCCodeControlFlowModule))
#define VALA_CCODE_CONTROL_FLOW_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_CONTROL_FLOW_MODULE, ValaCCodeControlFlowModuleClass))
#define VALA_IS_CCODE_CONTROL_FLOW_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_CONTROL_FLOW_MODULE))
#define VALA_IS_CCODE_CONTROL_FLOW_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_CONTROL_FLOW_MODULE))
#define VALA_CCODE_CONTROL_FLOW_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_CONTROL_FLOW_MODULE, ValaCCodeControlFlowModuleClass))

typedef struct _ValaCCodeControlFlowModule ValaCCodeControlFlowModule;
typedef struct _ValaCCodeControlFlowModuleClass ValaCCodeControlFlowModuleClass;
typedef struct _ValaCCodeControlFlowModulePrivate ValaCCodeControlFlowModulePrivate;

#define VALA_TYPE_CCODE_MEMBER_ACCESS_MODULE (vala_ccode_member_access_module_get_type ())
#define VALA_CCODE_MEMBER_ACCESS_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_MEMBER_ACCESS_MODULE, ValaCCodeMemberAccessModule))
#define VALA_CCODE_MEMBER_ACCESS_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_MEMBER_ACCESS_MODULE, ValaCCodeMemberAccessModuleClass))
#define VALA_IS_CCODE_MEMBER_ACCESS_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_MEMBER_ACCESS_MODULE))
#define VALA_IS_CCODE_MEMBER_ACCESS_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_MEMBER_ACCESS_MODULE))
#define VALA_CCODE_MEMBER_ACCESS_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_MEMBER_ACCESS_MODULE, ValaCCodeMemberAccessModuleClass))

typedef struct _ValaCCodeMemberAccessModule ValaCCodeMemberAccessModule;
typedef struct _ValaCCodeMemberAccessModuleClass ValaCCodeMemberAccessModuleClass;
typedef struct _ValaCCodeMemberAccessModulePrivate ValaCCodeMemberAccessModulePrivate;

#define VALA_TYPE_CCODE_ASSIGNMENT_MODULE (vala_ccode_assignment_module_get_type ())
#define VALA_CCODE_ASSIGNMENT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_ASSIGNMENT_MODULE, ValaCCodeAssignmentModule))
#define VALA_CCODE_ASSIGNMENT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_ASSIGNMENT_MODULE, ValaCCodeAssignmentModuleClass))
#define VALA_IS_CCODE_ASSIGNMENT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_ASSIGNMENT_MODULE))
#define VALA_IS_CCODE_ASSIGNMENT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_ASSIGNMENT_MODULE))
#define VALA_CCODE_ASSIGNMENT_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_ASSIGNMENT_MODULE, ValaCCodeAssignmentModuleClass))

typedef struct _ValaCCodeAssignmentModule ValaCCodeAssignmentModule;
typedef struct _ValaCCodeAssignmentModuleClass ValaCCodeAssignmentModuleClass;
typedef struct _ValaCCodeAssignmentModulePrivate ValaCCodeAssignmentModulePrivate;

#define VALA_TYPE_CCODE_METHOD_CALL_MODULE (vala_ccode_method_call_module_get_type ())
#define VALA_CCODE_METHOD_CALL_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_METHOD_CALL_MODULE, ValaCCodeMethodCallModule))
#define VALA_CCODE_METHOD_CALL_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_METHOD_CALL_MODULE, ValaCCodeMethodCallModuleClass))
#define VALA_IS_CCODE_METHOD_CALL_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_METHOD_CALL_MODULE))
#define VALA_IS_CCODE_METHOD_CALL_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_METHOD_CALL_MODULE))
#define VALA_CCODE_METHOD_CALL_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_METHOD_CALL_MODULE, ValaCCodeMethodCallModuleClass))

typedef struct _ValaCCodeMethodCallModule ValaCCodeMethodCallModule;
typedef struct _ValaCCodeMethodCallModuleClass ValaCCodeMethodCallModuleClass;
typedef struct _ValaCCodeMethodCallModulePrivate ValaCCodeMethodCallModulePrivate;

#define VALA_TYPE_CCODE_ARRAY_MODULE (vala_ccode_array_module_get_type ())
#define VALA_CCODE_ARRAY_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_ARRAY_MODULE, ValaCCodeArrayModule))
#define VALA_CCODE_ARRAY_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_ARRAY_MODULE, ValaCCodeArrayModuleClass))
#define VALA_IS_CCODE_ARRAY_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_ARRAY_MODULE))
#define VALA_IS_CCODE_ARRAY_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_ARRAY_MODULE))
#define VALA_CCODE_ARRAY_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_ARRAY_MODULE, ValaCCodeArrayModuleClass))

typedef struct _ValaCCodeArrayModule ValaCCodeArrayModule;
typedef struct _ValaCCodeArrayModuleClass ValaCCodeArrayModuleClass;
typedef struct _ValaCCodeArrayModulePrivate ValaCCodeArrayModulePrivate;

#define VALA_TYPE_CCODE_DELEGATE_MODULE (vala_ccode_delegate_module_get_type ())
#define VALA_CCODE_DELEGATE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_DELEGATE_MODULE, ValaCCodeDelegateModule))
#define VALA_CCODE_DELEGATE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_DELEGATE_MODULE, ValaCCodeDelegateModuleClass))
#define VALA_IS_CCODE_DELEGATE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_DELEGATE_MODULE))
#define VALA_IS_CCODE_DELEGATE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_DELEGATE_MODULE))
#define VALA_CCODE_DELEGATE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_DELEGATE_MODULE, ValaCCodeDelegateModuleClass))

typedef struct _ValaCCodeDelegateModule ValaCCodeDelegateModule;
typedef struct _ValaCCodeDelegateModuleClass ValaCCodeDelegateModuleClass;
typedef struct _ValaCCodeDelegateModulePrivate ValaCCodeDelegateModulePrivate;

#define VALA_TYPE_GERROR_MODULE (vala_gerror_module_get_type ())
#define VALA_GERROR_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GERROR_MODULE, ValaGErrorModule))
#define VALA_GERROR_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GERROR_MODULE, ValaGErrorModuleClass))
#define VALA_IS_GERROR_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GERROR_MODULE))
#define VALA_IS_GERROR_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GERROR_MODULE))
#define VALA_GERROR_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GERROR_MODULE, ValaGErrorModuleClass))

typedef struct _ValaGErrorModule ValaGErrorModule;
typedef struct _ValaGErrorModuleClass ValaGErrorModuleClass;
typedef struct _ValaGErrorModulePrivate ValaGErrorModulePrivate;

#define VALA_TYPE_GTYPE_MODULE (vala_gtype_module_get_type ())
#define VALA_GTYPE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GTYPE_MODULE, ValaGTypeModule))
#define VALA_GTYPE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GTYPE_MODULE, ValaGTypeModuleClass))
#define VALA_IS_GTYPE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GTYPE_MODULE))
#define VALA_IS_GTYPE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GTYPE_MODULE))
#define VALA_GTYPE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GTYPE_MODULE, ValaGTypeModuleClass))

typedef struct _ValaGTypeModule ValaGTypeModule;
typedef struct _ValaGTypeModuleClass ValaGTypeModuleClass;
typedef struct _ValaGTypeModulePrivate ValaGTypeModulePrivate;

#define VALA_TYPE_GOBJECT_MODULE (vala_gobject_module_get_type ())
#define VALA_GOBJECT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GOBJECT_MODULE, ValaGObjectModule))
#define VALA_GOBJECT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GOBJECT_MODULE, ValaGObjectModuleClass))
#define VALA_IS_GOBJECT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GOBJECT_MODULE))
#define VALA_IS_GOBJECT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GOBJECT_MODULE))
#define VALA_GOBJECT_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GOBJECT_MODULE, ValaGObjectModuleClass))

typedef struct _ValaGObjectModule ValaGObjectModule;
typedef struct _ValaGObjectModuleClass ValaGObjectModuleClass;
typedef struct _ValaGObjectModulePrivate ValaGObjectModulePrivate;

#define VALA_TYPE_GSIGNAL_MODULE (vala_gsignal_module_get_type ())
#define VALA_GSIGNAL_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GSIGNAL_MODULE, ValaGSignalModule))
#define VALA_GSIGNAL_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GSIGNAL_MODULE, ValaGSignalModuleClass))
#define VALA_IS_GSIGNAL_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GSIGNAL_MODULE))
#define VALA_IS_GSIGNAL_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GSIGNAL_MODULE))
#define VALA_GSIGNAL_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GSIGNAL_MODULE, ValaGSignalModuleClass))

typedef struct _ValaGSignalModule ValaGSignalModule;
typedef struct _ValaGSignalModuleClass ValaGSignalModuleClass;
typedef struct _ValaGSignalModulePrivate ValaGSignalModulePrivate;

#define VALA_TYPE_GASYNC_MODULE (vala_gasync_module_get_type ())
#define VALA_GASYNC_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GASYNC_MODULE, ValaGAsyncModule))
#define VALA_GASYNC_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GASYNC_MODULE, ValaGAsyncModuleClass))
#define VALA_IS_GASYNC_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GASYNC_MODULE))
#define VALA_IS_GASYNC_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GASYNC_MODULE))
#define VALA_GASYNC_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GASYNC_MODULE, ValaGAsyncModuleClass))

typedef struct _ValaGAsyncModule ValaGAsyncModule;
typedef struct _ValaGAsyncModuleClass ValaGAsyncModuleClass;
typedef struct _ValaGAsyncModulePrivate ValaGAsyncModulePrivate;

#define VALA_TYPE_DBUS_MODULE (vala_dbus_module_get_type ())
#define VALA_DBUS_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DBUS_MODULE, ValaDBusModule))
#define VALA_DBUS_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DBUS_MODULE, ValaDBusModuleClass))
#define VALA_IS_DBUS_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DBUS_MODULE))
#define VALA_IS_DBUS_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DBUS_MODULE))
#define VALA_DBUS_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DBUS_MODULE, ValaDBusModuleClass))

typedef struct _ValaDBusModule ValaDBusModule;
typedef struct _ValaDBusModuleClass ValaDBusModuleClass;
typedef struct _ValaDBusModulePrivate ValaDBusModulePrivate;

#define VALA_TYPE_DBUS_CLIENT_MODULE (vala_dbus_client_module_get_type ())
#define VALA_DBUS_CLIENT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DBUS_CLIENT_MODULE, ValaDBusClientModule))
#define VALA_DBUS_CLIENT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DBUS_CLIENT_MODULE, ValaDBusClientModuleClass))
#define VALA_IS_DBUS_CLIENT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DBUS_CLIENT_MODULE))
#define VALA_IS_DBUS_CLIENT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DBUS_CLIENT_MODULE))
#define VALA_DBUS_CLIENT_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DBUS_CLIENT_MODULE, ValaDBusClientModuleClass))

typedef struct _ValaDBusClientModule ValaDBusClientModule;
typedef struct _ValaDBusClientModuleClass ValaDBusClientModuleClass;
typedef struct _ValaDBusClientModulePrivate ValaDBusClientModulePrivate;

#define VALA_TYPE_DBUS_SERVER_MODULE (vala_dbus_server_module_get_type ())
#define VALA_DBUS_SERVER_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DBUS_SERVER_MODULE, ValaDBusServerModule))
#define VALA_DBUS_SERVER_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DBUS_SERVER_MODULE, ValaDBusServerModuleClass))
#define VALA_IS_DBUS_SERVER_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DBUS_SERVER_MODULE))
#define VALA_IS_DBUS_SERVER_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DBUS_SERVER_MODULE))
#define VALA_DBUS_SERVER_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DBUS_SERVER_MODULE, ValaDBusServerModuleClass))

typedef struct _ValaDBusServerModule ValaDBusServerModule;
typedef struct _ValaDBusServerModuleClass ValaDBusServerModuleClass;
typedef struct _ValaDBusServerModulePrivate ValaDBusServerModulePrivate;

#define VALA_TYPE_CCODE_GENERATOR (vala_ccode_generator_get_type ())
#define VALA_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGenerator))
#define VALA_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))
#define VALA_IS_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_GENERATOR))
#define VALA_IS_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_GENERATOR))
#define VALA_CCODE_GENERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))

typedef struct _ValaCCodeGenerator ValaCCodeGenerator;
typedef struct _ValaCCodeGeneratorClass ValaCCodeGeneratorClass;

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeModule {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeModulePrivate * priv;
};

struct _ValaCCodeModuleClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeModule *self);
	void (*emit) (ValaCCodeModule* self, ValaCodeContext* context);
	void (*visit_source_file) (ValaCCodeModule* self, ValaSourceFile* source_file);
	void (*visit_class) (ValaCCodeModule* self, ValaClass* cl);
	void (*visit_interface) (ValaCCodeModule* self, ValaInterface* iface);
	void (*visit_struct) (ValaCCodeModule* self, ValaStruct* st);
	void (*visit_enum) (ValaCCodeModule* self, ValaEnum* en);
	void (*visit_error_domain) (ValaCCodeModule* self, ValaErrorDomain* edomain);
	void (*visit_delegate) (ValaCCodeModule* self, ValaDelegate* d);
	void (*visit_member) (ValaCCodeModule* self, ValaMember* m);
	void (*visit_constant) (ValaCCodeModule* self, ValaConstant* c);
	void (*visit_field) (ValaCCodeModule* self, ValaField* f);
	void (*visit_method) (ValaCCodeModule* self, ValaMethod* m);
	void (*visit_creation_method) (ValaCCodeModule* self, ValaCreationMethod* m);
	void (*visit_formal_parameter) (ValaCCodeModule* self, ValaFormalParameter* p);
	void (*visit_property) (ValaCCodeModule* self, ValaProperty* prop);
	void (*visit_property_accessor) (ValaCCodeModule* self, ValaPropertyAccessor* acc);
	void (*visit_signal) (ValaCCodeModule* self, ValaSignal* sig);
	void (*visit_constructor) (ValaCCodeModule* self, ValaConstructor* c);
	void (*visit_destructor) (ValaCCodeModule* self, ValaDestructor* d);
	void (*visit_block) (ValaCCodeModule* self, ValaBlock* b);
	void (*visit_empty_statement) (ValaCCodeModule* self, ValaEmptyStatement* stmt);
	void (*visit_declaration_statement) (ValaCCodeModule* self, ValaDeclarationStatement* stmt);
	void (*visit_local_variable) (ValaCCodeModule* self, ValaLocalVariable* local);
	void (*visit_initializer_list) (ValaCCodeModule* self, ValaInitializerList* list);
	void (*visit_end_full_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_expression_statement) (ValaCCodeModule* self, ValaExpressionStatement* stmt);
	void (*visit_if_statement) (ValaCCodeModule* self, ValaIfStatement* stmt);
	void (*visit_switch_statement) (ValaCCodeModule* self, ValaSwitchStatement* stmt);
	void (*visit_switch_section) (ValaCCodeModule* self, ValaSwitchSection* section);
	void (*visit_switch_label) (ValaCCodeModule* self, ValaSwitchLabel* label);
	void (*visit_while_statement) (ValaCCodeModule* self, ValaWhileStatement* stmt);
	void (*visit_do_statement) (ValaCCodeModule* self, ValaDoStatement* stmt);
	void (*visit_for_statement) (ValaCCodeModule* self, ValaForStatement* stmt);
	void (*visit_foreach_statement) (ValaCCodeModule* self, ValaForeachStatement* stmt);
	void (*visit_break_statement) (ValaCCodeModule* self, ValaBreakStatement* stmt);
	void (*visit_continue_statement) (ValaCCodeModule* self, ValaContinueStatement* stmt);
	void (*visit_return_statement) (ValaCCodeModule* self, ValaReturnStatement* stmt);
	void (*visit_yield_statement) (ValaCCodeModule* self, ValaYieldStatement* stmt);
	void (*visit_throw_statement) (ValaCCodeModule* self, ValaThrowStatement* stmt);
	void (*visit_try_statement) (ValaCCodeModule* self, ValaTryStatement* stmt);
	void (*visit_catch_clause) (ValaCCodeModule* self, ValaCatchClause* clause);
	void (*visit_lock_statement) (ValaCCodeModule* self, ValaLockStatement* stmt);
	void (*visit_delete_statement) (ValaCCodeModule* self, ValaDeleteStatement* stmt);
	void (*visit_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_array_creation_expression) (ValaCCodeModule* self, ValaArrayCreationExpression* expr);
	void (*visit_boolean_literal) (ValaCCodeModule* self, ValaBooleanLiteral* expr);
	void (*visit_character_literal) (ValaCCodeModule* self, ValaCharacterLiteral* expr);
	void (*visit_integer_literal) (ValaCCodeModule* self, ValaIntegerLiteral* expr);
	void (*visit_real_literal) (ValaCCodeModule* self, ValaRealLiteral* expr);
	void (*visit_string_literal) (ValaCCodeModule* self, ValaStringLiteral* expr);
	void (*visit_null_literal) (ValaCCodeModule* self, ValaNullLiteral* expr);
	void (*visit_member_access) (ValaCCodeModule* self, ValaMemberAccess* expr);
	void (*visit_method_call) (ValaCCodeModule* self, ValaMethodCall* expr);
	void (*visit_element_access) (ValaCCodeModule* self, ValaElementAccess* expr);
	void (*visit_base_access) (ValaCCodeModule* self, ValaBaseAccess* expr);
	void (*visit_postfix_expression) (ValaCCodeModule* self, ValaPostfixExpression* expr);
	void (*visit_object_creation_expression) (ValaCCodeModule* self, ValaObjectCreationExpression* expr);
	void (*visit_sizeof_expression) (ValaCCodeModule* self, ValaSizeofExpression* expr);
	void (*visit_typeof_expression) (ValaCCodeModule* self, ValaTypeofExpression* expr);
	void (*visit_unary_expression) (ValaCCodeModule* self, ValaUnaryExpression* expr);
	void (*visit_cast_expression) (ValaCCodeModule* self, ValaCastExpression* expr);
	void (*visit_pointer_indirection) (ValaCCodeModule* self, ValaPointerIndirection* expr);
	void (*visit_addressof_expression) (ValaCCodeModule* self, ValaAddressofExpression* expr);
	void (*visit_reference_transfer_expression) (ValaCCodeModule* self, ValaReferenceTransferExpression* expr);
	void (*visit_binary_expression) (ValaCCodeModule* self, ValaBinaryExpression* expr);
	void (*visit_type_check) (ValaCCodeModule* self, ValaTypeCheck* expr);
	void (*visit_lambda_expression) (ValaCCodeModule* self, ValaLambdaExpression* l);
	void (*visit_assignment) (ValaCCodeModule* self, ValaAssignment* a);
	char* (*get_custom_creturn_type) (ValaCCodeModule* self, ValaMethod* m);
	void (*generate_dynamic_method_wrapper) (ValaCCodeModule* self, ValaDynamicMethod* method);
	gboolean (*method_has_wrapper) (ValaCCodeModule* self, ValaMethod* method);
	ValaCCodeIdentifier* (*get_value_setter_function) (ValaCCodeModule* self, ValaDataType* type_reference);
	ValaCCodeExpression* (*get_construct_property_assignment) (ValaCCodeModule* self, ValaCCodeConstant* canonical_cconstant, ValaDataType* property_type, ValaCCodeExpression* value);
	ValaCCodeFunctionCall* (*get_param_spec) (ValaCCodeModule* self, ValaProperty* prop);
	ValaCCodeFunctionCall* (*get_signal_creation) (ValaCCodeModule* self, ValaSignal* sig, ValaTypeSymbol* type);
	ValaCCodeFragment* (*register_dbus_info) (ValaCCodeModule* self, ValaObjectTypeSymbol* bindable);
	char* (*get_dynamic_property_getter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_property_setter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_signal_cname) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_connect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_disconnect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	void (*generate_marshaller) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, gboolean dbus);
	char* (*get_marshaller_function) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, const char* prefix, gboolean dbus);
	char* (*get_array_length_cname) (ValaCCodeModule* self, const char* array_cname, gint dim);
	ValaCCodeExpression* (*get_array_length_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr, gint dim);
	char* (*get_array_size_cname) (ValaCCodeModule* self, const char* array_cname);
	ValaCCodeExpression* (*get_array_size_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr);
	void (*add_simple_check) (ValaCCodeModule* self, ValaCodeNode* node, ValaCCodeFragment* cfrag);
};

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeBaseModule {
	ValaCCodeModule parent_instance;
	ValaCCodeBaseModulePrivate * priv;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaTypeSymbol* current_type_symbol;
	ValaClass* current_class;
	ValaMethod* current_method;
	ValaDataType* current_return_type;
	ValaTryStatement* current_try;
	ValaPropertyAccessor* current_property_accessor;
	ValaCCodeDeclarationSpace* header_declarations;
	ValaCCodeDeclarationSpace* internal_header_declarations;
	ValaCCodeDeclarationSpace* source_declarations;
	ValaCCodeFragment* source_signal_marshaller_declaration;
	ValaCCodeFragment* source_type_member_definition;
	ValaCCodeFragment* class_init_fragment;
	ValaCCodeFragment* base_init_fragment;
	ValaCCodeFragment* class_finalize_fragment;
	ValaCCodeFragment* base_finalize_fragment;
	ValaCCodeFragment* instance_init_fragment;
	ValaCCodeFragment* instance_finalize_fragment;
	ValaCCodeFragment* source_signal_marshaller_definition;
	ValaCCodeFragment* module_init_fragment;
	ValaCCodeStruct* param_spec_struct;
	ValaCCodeStruct* closure_struct;
	ValaCCodeEnum* prop_enum;
	ValaCCodeFunction* function;
	ValaCCodeFragment* pre_statement_fragment;
	GeeArrayList* temp_vars;
	GeeArrayList* temp_ref_vars;
	GeeSet* user_marshal_set;
	GeeSet* predefined_marshal_set;
	gint next_temp_var_id;
	gboolean in_creation_method;
	gboolean in_constructor;
	gboolean in_static_or_class_ctor;
	gboolean current_method_inner_error;
	gint next_coroutine_state;
	ValaDataType* bool_type;
	ValaDataType* char_type;
	ValaDataType* uchar_type;
	ValaDataType* unichar_type;
	ValaDataType* short_type;
	ValaDataType* ushort_type;
	ValaDataType* int_type;
	ValaDataType* uint_type;
	ValaDataType* long_type;
	ValaDataType* ulong_type;
	ValaDataType* int8_type;
	ValaDataType* uint8_type;
	ValaDataType* int16_type;
	ValaDataType* uint16_type;
	ValaDataType* int32_type;
	ValaDataType* uint32_type;
	ValaDataType* int64_type;
	ValaDataType* uint64_type;
	ValaDataType* string_type;
	ValaDataType* float_type;
	ValaDataType* double_type;
	ValaTypeSymbol* gtype_type;
	ValaTypeSymbol* gobject_type;
	ValaErrorType* gerror_type;
	ValaClass* glist_type;
	ValaClass* gslist_type;
	ValaTypeSymbol* gstringbuilder_type;
	ValaTypeSymbol* garray_type;
	ValaTypeSymbol* gbytearray_type;
	ValaTypeSymbol* gptrarray_type;
	ValaDataType* gquark_type;
	ValaStruct* gvalue_type;
	ValaStruct* mutex_type;
	ValaTypeSymbol* type_module_type;
	ValaTypeSymbol* dbus_object_type;
	gboolean in_plugin;
	char* module_init_param_name;
	gboolean gvaluecollector_h_needed;
	gboolean gio_h_needed;
	gboolean requires_array_free;
	gboolean requires_array_move;
	gboolean requires_array_length;
	gboolean requires_strcmp0;
	gboolean dbus_glib_h_needed;
	gboolean dbus_glib_h_needed_in_header;
	GeeSet* wrappers;
	GeeMap* variable_name_map;
};

struct _ValaCCodeBaseModuleClass {
	ValaCCodeModuleClass parent_class;
	void (*append_vala_array_free) (ValaCCodeBaseModule* self);
	void (*append_vala_array_move) (ValaCCodeBaseModule* self);
	void (*append_vala_array_length) (ValaCCodeBaseModule* self);
	void (*generate_class_struct_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_struct_declaration) (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_delegate_declaration) (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_cparameters) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
	ValaCCodeExpression* (*get_dup_func_expression) (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference);
	ValaCCodeExpression* (*get_unref_expression) (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr);
	char* (*get_delegate_target_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	ValaCCodeExpression* (*get_delegate_target_cexpression) (ValaCCodeBaseModule* self, ValaExpression* delegate_expr);
	char* (*get_delegate_target_destroy_notify_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	ValaCCodeExpression* (*get_ref_cexpression) (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node);
	void (*generate_class_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_interface_declaration) (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_method_declaration) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_error_domain_declaration) (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space);
	ValaCCodeExpression* (*get_implicit_cast_expression) (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
};

struct _ValaCCodeStructModule {
	ValaCCodeBaseModule parent_instance;
	ValaCCodeStructModulePrivate * priv;
};

struct _ValaCCodeStructModuleClass {
	ValaCCodeBaseModuleClass parent_class;
};

/**
 * The link between a method and generated code.
 */
struct _ValaCCodeMethodModule {
	ValaCCodeStructModule parent_instance;
	ValaCCodeMethodModulePrivate * priv;
};

struct _ValaCCodeMethodModuleClass {
	ValaCCodeStructModuleClass parent_class;
	void (*generate_method_result_declaration) (ValaCCodeMethodModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, ValaCCodeFunction* cfunc, GeeMap* cparam_map, GeeMap* carg_map);
	void (*generate_parameter) (ValaCCodeMethodModule* self, ValaFormalParameter* param, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, GeeMap* carg_map);
};

struct _ValaCCodeControlFlowModule {
	ValaCCodeMethodModule parent_instance;
	ValaCCodeControlFlowModulePrivate * priv;
};

struct _ValaCCodeControlFlowModuleClass {
	ValaCCodeMethodModuleClass parent_class;
};

struct _ValaCCodeMemberAccessModule {
	ValaCCodeControlFlowModule parent_instance;
	ValaCCodeMemberAccessModulePrivate * priv;
};

struct _ValaCCodeMemberAccessModuleClass {
	ValaCCodeControlFlowModuleClass parent_class;
};

/**
 * The link between an assignment and generated code.
 */
struct _ValaCCodeAssignmentModule {
	ValaCCodeMemberAccessModule parent_instance;
	ValaCCodeAssignmentModulePrivate * priv;
};

struct _ValaCCodeAssignmentModuleClass {
	ValaCCodeMemberAccessModuleClass parent_class;
};

struct _ValaCCodeMethodCallModule {
	ValaCCodeAssignmentModule parent_instance;
	ValaCCodeMethodCallModulePrivate * priv;
};

struct _ValaCCodeMethodCallModuleClass {
	ValaCCodeAssignmentModuleClass parent_class;
};

struct _ValaCCodeArrayModule {
	ValaCCodeMethodCallModule parent_instance;
	ValaCCodeArrayModulePrivate * priv;
};

struct _ValaCCodeArrayModuleClass {
	ValaCCodeMethodCallModuleClass parent_class;
};

/**
 * The link between an assignment and generated code.
 */
struct _ValaCCodeDelegateModule {
	ValaCCodeArrayModule parent_instance;
	ValaCCodeDelegateModulePrivate * priv;
};

struct _ValaCCodeDelegateModuleClass {
	ValaCCodeArrayModuleClass parent_class;
};

struct _ValaGErrorModule {
	ValaCCodeDelegateModule parent_instance;
	ValaGErrorModulePrivate * priv;
};

struct _ValaGErrorModuleClass {
	ValaCCodeDelegateModuleClass parent_class;
	ValaCCodeStatement* (*return_with_exception) (ValaGErrorModule* self, ValaCCodeExpression* error_expr);
};

struct _ValaGTypeModule {
	ValaGErrorModule parent_instance;
	ValaGTypeModulePrivate * priv;
};

struct _ValaGTypeModuleClass {
	ValaGErrorModuleClass parent_class;
	void (*generate_virtual_method_declaration) (ValaGTypeModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, ValaCCodeStruct* type_struct);
	void (*generate_class_init) (ValaGTypeModule* self, ValaClass* cl, ValaCCodeBlock* init_block);
};

struct _ValaGObjectModule {
	ValaGTypeModule parent_instance;
	ValaGObjectModulePrivate * priv;
};

struct _ValaGObjectModuleClass {
	ValaGTypeModuleClass parent_class;
};

struct _ValaGSignalModule {
	ValaGObjectModule parent_instance;
	ValaGSignalModulePrivate * priv;
};

struct _ValaGSignalModuleClass {
	ValaGObjectModuleClass parent_class;
	ValaCCodeExpression* (*get_dbus_g_type) (ValaGSignalModule* self, ValaDataType* data_type);
};

struct _ValaGAsyncModule {
	ValaGSignalModule parent_instance;
	ValaGAsyncModulePrivate * priv;
};

struct _ValaGAsyncModuleClass {
	ValaGSignalModuleClass parent_class;
};

struct _ValaDBusModule {
	ValaGAsyncModule parent_instance;
	ValaDBusModulePrivate * priv;
};

struct _ValaDBusModuleClass {
	ValaGAsyncModuleClass parent_class;
};

/**
 * The link between a dynamic method and generated code.
 */
struct _ValaDBusClientModule {
	ValaDBusModule parent_instance;
	ValaDBusClientModulePrivate * priv;
};

struct _ValaDBusClientModuleClass {
	ValaDBusModuleClass parent_class;
};

/**
 * The link between a dynamic method and generated code.
 */
struct _ValaDBusServerModule {
	ValaDBusClientModule parent_instance;
	ValaDBusServerModulePrivate * priv;
};

struct _ValaDBusServerModuleClass {
	ValaDBusClientModuleClass parent_class;
};



gpointer vala_ccode_module_ref (gpointer instance);
void vala_ccode_module_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_module (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_module (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_module (const GValue* value);
GType vala_ccode_module_get_type (void);
GType vala_ccode_base_module_get_type (void);
gpointer vala_ccode_declaration_space_ref (gpointer instance);
void vala_ccode_declaration_space_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_declaration_space (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_declaration_space (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_declaration_space (const GValue* value);
GType vala_ccode_declaration_space_get_type (void);
GType vala_ccode_struct_module_get_type (void);
GType vala_ccode_method_module_get_type (void);
GType vala_ccode_control_flow_module_get_type (void);
GType vala_ccode_member_access_module_get_type (void);
GType vala_ccode_assignment_module_get_type (void);
GType vala_ccode_method_call_module_get_type (void);
GType vala_ccode_array_module_get_type (void);
GType vala_ccode_delegate_module_get_type (void);
GType vala_gerror_module_get_type (void);
GType vala_gtype_module_get_type (void);
GType vala_gobject_module_get_type (void);
GType vala_gsignal_module_get_type (void);
GType vala_gasync_module_get_type (void);
GType vala_dbus_module_get_type (void);
GType vala_dbus_client_module_get_type (void);
GType vala_dbus_server_module_get_type (void);
enum  {
	VALA_DBUS_SERVER_MODULE_DUMMY_PROPERTY
};
GType vala_ccode_generator_get_type (void);
ValaDBusClientModule* vala_dbus_client_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaDBusClientModule* vala_dbus_client_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaDBusServerModule* vala_dbus_server_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaDBusServerModule* vala_dbus_server_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaDBusServerModule* vala_dbus_server_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
static gboolean vala_dbus_server_module_is_dbus_visible (ValaDBusServerModule* self, ValaCodeNode* node);
static char* vala_dbus_server_module_dbus_result_name (ValaDBusServerModule* self, ValaMethod* m);
ValaCCodeExpression* vala_ccode_base_module_default_value_for_type (ValaCCodeBaseModule* self, ValaDataType* type, gboolean initializer_expression);
char* vala_ccode_module_get_array_length_cname (ValaCCodeModule* self, const char* array_cname, gint dim);
ValaCCodeExpression* vala_dbus_module_read_expression (ValaDBusModule* self, ValaCCodeFragment* fragment, ValaDataType* type, ValaCCodeExpression* iter_expr, ValaCCodeExpression* expr);
void vala_dbus_module_write_expression (ValaDBusModule* self, ValaCCodeFragment* fragment, ValaDataType* type, ValaCCodeExpression* iter_expr, ValaCCodeExpression* expr);
void vala_ccode_declaration_space_add_type_member_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
static char* vala_dbus_server_module_generate_dbus_wrapper (ValaDBusServerModule* self, ValaMethod* m, ValaObjectTypeSymbol* sym);
void vala_ccode_method_module_generate_parameter (ValaCCodeMethodModule* self, ValaFormalParameter* param, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, GeeMap* carg_map);
ValaCCodeNode* vala_ccode_base_module_get_ccodenode (ValaCCodeBaseModule* self, ValaCodeNode* node);
ValaCCodeModule* vala_ccode_module_get_head (ValaCCodeModule* self);
static char* vala_dbus_server_module_generate_dbus_signal_wrapper (ValaDBusServerModule* self, ValaSignal* sig, ValaObjectTypeSymbol* sym, const char* dbus_iface_name);
static ValaCCodeExpression* vala_dbus_server_module_get_path_vtable (ValaDBusServerModule* self, ValaObjectType* object_type);
static void vala_dbus_server_module_handle_signals (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, ValaCCodeBlock* block);
static gboolean vala_dbus_server_module_type_implements_dbus_interface (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym);
static void vala_dbus_server_module_generate_register_function (ValaDBusServerModule* self, ValaObjectType* object_type);
static void vala_dbus_server_module_generate_unregister_function (ValaDBusServerModule* self, ValaObjectType* object_type);
static void vala_dbus_server_module_handle_method (ValaDBusServerModule* self, const char* dbus_iface_name, const char* dbus_method_name, const char* handler_name, ValaCCodeBlock* block, ValaCCodeIfStatement** clastif);
static void vala_dbus_server_module_handle_methods (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name, ValaCCodeBlock* block, ValaCCodeIfStatement** clastif);
static char* vala_dbus_server_module_generate_dbus_property_get_wrapper (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name);
static char* vala_dbus_server_module_generate_dbus_property_get_all_wrapper (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name);
static char* vala_dbus_server_module_generate_dbus_property_set_wrapper (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name);
static char* vala_dbus_server_module_get_dbus_type_introspection (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym);
static char* vala_dbus_server_module_generate_dbus_introspect (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym);
static void vala_dbus_server_module_generate_message_function (ValaDBusServerModule* self, ValaObjectType* object_type);
void vala_ccode_declaration_space_add_constant_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
static ValaCCodeExpression* vala_dbus_server_module_get_vtable (ValaDBusServerModule* self, ValaObjectType* object_type);
void vala_ccode_module_visit_method_call (ValaCCodeModule* self, ValaMethodCall* expr);
ValaCCodeGenerator* vala_ccode_module_get_codegen (ValaCCodeModule* self);
static void vala_dbus_server_module_real_visit_method_call (ValaCCodeModule* base, ValaMethodCall* expr);
static ValaCCodeFragment* vala_dbus_server_module_real_register_dbus_info (ValaCCodeModule* base, ValaObjectTypeSymbol* sym);
static gpointer vala_dbus_server_module_parent_class = NULL;
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaDBusServerModule* vala_dbus_server_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	ValaDBusServerModule* self;
	g_return_val_if_fail (codegen != NULL, NULL);
	self = (ValaDBusServerModule*) vala_dbus_client_module_construct (object_type, codegen, next);
	return self;
}


ValaDBusServerModule* vala_dbus_server_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	return vala_dbus_server_module_construct (VALA_TYPE_DBUS_SERVER_MODULE, codegen, next);
}


static gboolean vala_dbus_server_module_is_dbus_visible (ValaDBusServerModule* self, ValaCodeNode* node) {
	ValaAttribute* dbus_attribute;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (node != NULL, FALSE);
	dbus_attribute = vala_code_node_get_attribute (node, "DBus");
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (dbus_attribute != NULL) {
		_tmp1_ = vala_attribute_has_argument (dbus_attribute, "visible");
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = !vala_attribute_get_bool (dbus_attribute, "visible");
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		gboolean _tmp2_;
		return (_tmp2_ = FALSE, (dbus_attribute == NULL) ? NULL : (dbus_attribute = (vala_code_node_unref (dbus_attribute), NULL)), _tmp2_);
	}
	return (_tmp3_ = TRUE, (dbus_attribute == NULL) ? NULL : (dbus_attribute = (vala_code_node_unref (dbus_attribute), NULL)), _tmp3_);
}


static char* vala_dbus_server_module_dbus_result_name (ValaDBusServerModule* self, ValaMethod* m) {
	ValaAttribute* dbus_attribute;
	gboolean _tmp0_;
	char* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (m != NULL, NULL);
	dbus_attribute = vala_code_node_get_attribute ((ValaCodeNode*) m, "DBus");
	_tmp0_ = FALSE;
	if (dbus_attribute != NULL) {
		_tmp0_ = vala_attribute_has_argument (dbus_attribute, "result");
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		char* result_name;
		gboolean _tmp1_;
		result_name = vala_attribute_get_string (dbus_attribute, "result");
		_tmp1_ = FALSE;
		if (result_name != NULL) {
			_tmp1_ = _vala_strcmp0 (result_name, "") != 0;
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			char* _tmp2_;
			_tmp2_ = NULL;
			return (_tmp2_ = result_name, (dbus_attribute == NULL) ? NULL : (dbus_attribute = (vala_code_node_unref (dbus_attribute), NULL)), _tmp2_);
		}
		result_name = (g_free (result_name), NULL);
	}
	_tmp3_ = NULL;
	return (_tmp3_ = g_strdup ("result"), (dbus_attribute == NULL) ? NULL : (dbus_attribute = (vala_code_node_unref (dbus_attribute), NULL)), _tmp3_);
}


static char* vala_dbus_server_module_generate_dbus_wrapper (ValaDBusServerModule* self, ValaMethod* m, ValaObjectTypeSymbol* sym) {
	char* _tmp0_;
	char* _tmp1_;
	char* wrapper_name;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp4_;
	char* _tmp3_;
	char* _tmp2_;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* _tmp6_;
	ValaCCodeBlock* block;
	ValaCCodeFragment* prefragment;
	ValaCCodeFragment* postfragment;
	ValaCCodeDeclaration* _tmp7_;
	ValaCCodeVariableDeclarator* _tmp8_;
	ValaCCodeDeclaration* _tmp9_;
	ValaCCodeVariableDeclarator* _tmp11_;
	ValaCCodeConstant* _tmp10_;
	ValaCCodeIdentifier* _tmp12_;
	ValaCCodeFunctionCall* _tmp13_;
	ValaCCodeFunctionCall* message_signature;
	ValaCCodeIdentifier* _tmp14_;
	ValaCCodeIdentifier* _tmp15_;
	ValaCCodeFunctionCall* _tmp16_;
	ValaCCodeFunctionCall* signature_check;
	ValaCCodeBlock* signature_error_block;
	ValaCCodeReturnStatement* _tmp18_;
	ValaCCodeConstant* _tmp17_;
	ValaCCodeIfStatement* _tmp19_;
	ValaCCodeIdentifier* _tmp20_;
	ValaCCodeFunctionCall* _tmp21_;
	ValaCCodeFunctionCall* iter_call;
	ValaCCodeIdentifier* _tmp22_;
	ValaCCodeUnaryExpression* _tmp24_;
	ValaCCodeIdentifier* _tmp23_;
	ValaCCodeExpressionStatement* _tmp25_;
	ValaCCodeFunctionCall* _tmp27_;
	ValaCCodeIdentifier* _tmp26_;
	ValaCCodeIdentifier* _tmp28_;
	ValaCCodeUnaryExpression* _tmp30_;
	ValaCCodeIdentifier* _tmp29_;
	ValaCCodeExpressionStatement* _tmp31_;
	ValaCCodeIdentifier* _tmp33_;
	char* _tmp32_;
	ValaCCodeFunctionCall* _tmp34_;
	ValaCCodeFunctionCall* ccall;
	ValaCCodeIdentifier* _tmp35_;
	char* type_signature;
	ValaCCodeConstant* _tmp76_;
	char* _tmp75_;
	ValaCCodeDeclaration* _tmp96_;
	ValaCCodeVariableDeclarator* _tmp97_;
	GeeList* _tmp98_;
	gboolean _tmp99_;
	ValaCCodeIdentifier* _tmp115_;
	ValaCCodeFunctionCall* _tmp116_;
	ValaCCodeFunctionCall* msgcall;
	ValaCCodeIdentifier* _tmp117_;
	ValaCCodeExpressionStatement* _tmp120_;
	ValaCCodeAssignment* _tmp119_;
	ValaCCodeIdentifier* _tmp118_;
	ValaCCodeReturnStatement* _tmp122_;
	ValaCCodeIdentifier* _tmp121_;
	ValaCCodeFunction* _tmp123_;
	char* _tmp124_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (m != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	wrapper_name = (_tmp1_ = g_strdup_printf ("_dbus_%s", _tmp0_ = vala_method_get_cname (m)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	/* declaration*/
	_cdecl_ = NULL;
	function = vala_ccode_function_new (wrapper_name, "DBusMessage*");
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp4_ = vala_ccode_formal_parameter_new ("self", _tmp3_ = g_strconcat (_tmp2_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) sym, FALSE), "*", NULL)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp5_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp6_ = vala_ccode_formal_parameter_new ("message", "DBusMessage*"));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	block = vala_ccode_block_new ();
	prefragment = vala_ccode_fragment_new ();
	postfragment = vala_ccode_fragment_new ();
	_tmp7_ = NULL;
	_cdecl_ = (_tmp7_ = vala_ccode_declaration_new ("DBusMessageIter"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp7_);
	_tmp8_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp8_ = vala_ccode_variable_declarator_new ("iter", NULL, NULL)));
	(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp9_ = NULL;
	_cdecl_ = (_tmp9_ = vala_ccode_declaration_new ("GError*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp9_);
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp11_ = vala_ccode_variable_declarator_new ("error", (ValaCCodeExpression*) (_tmp10_ = vala_ccode_constant_new ("NULL")), NULL)));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) prefragment);
	_tmp12_ = NULL;
	_tmp13_ = NULL;
	message_signature = (_tmp13_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp12_ = vala_ccode_identifier_new ("dbus_message_get_signature"))), (_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL)), _tmp13_);
	_tmp14_ = NULL;
	vala_ccode_function_call_add_argument (message_signature, (ValaCCodeExpression*) (_tmp14_ = vala_ccode_identifier_new ("message")));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	_tmp15_ = NULL;
	_tmp16_ = NULL;
	signature_check = (_tmp16_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new ("strcmp"))), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL)), _tmp16_);
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) message_signature);
	signature_error_block = vala_ccode_block_new ();
	_tmp18_ = NULL;
	_tmp17_ = NULL;
	vala_ccode_block_add_statement (signature_error_block, (ValaCCodeNode*) (_tmp18_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp17_ = vala_ccode_constant_new ("NULL")))));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
	_tmp19_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp19_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) signature_check, (ValaCCodeStatement*) signature_error_block, NULL)));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	_tmp20_ = NULL;
	_tmp21_ = NULL;
	iter_call = (_tmp21_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new ("dbus_message_iter_init"))), (_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL)), _tmp21_);
	_tmp22_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("message")));
	(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
	_tmp24_ = NULL;
	_tmp23_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp24_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp23_ = vala_ccode_identifier_new ("iter")))));
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	_tmp25_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp25_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
	_tmp27_ = NULL;
	_tmp26_ = NULL;
	iter_call = (_tmp27_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp26_ = vala_ccode_identifier_new ("dbus_message_iter_init_append"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp27_);
	(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
	_tmp28_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("reply")));
	(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL));
	_tmp30_ = NULL;
	_tmp29_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp30_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp29_ = vala_ccode_identifier_new ("iter")))));
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
	(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
	_tmp31_ = NULL;
	vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp31_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
	_tmp33_ = NULL;
	_tmp32_ = NULL;
	_tmp34_ = NULL;
	ccall = (_tmp34_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp33_ = vala_ccode_identifier_new (_tmp32_ = vala_method_get_cname (m)))), (_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL)), _tmp32_ = (g_free (_tmp32_), NULL), _tmp34_);
	_tmp35_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp35_ = vala_ccode_identifier_new ("self")));
	(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
	/* expected type signature for input parameters*/
	type_signature = g_strdup ("");
	{
		GeeList* _tmp36_;
		GeeIterator* _tmp37_;
		GeeIterator* _param_it;
		_tmp36_ = NULL;
		_tmp37_ = NULL;
		_param_it = (_tmp37_ = gee_iterable_iterator ((GeeIterable*) (_tmp36_ = vala_method_get_parameters (m))), (_tmp36_ == NULL) ? NULL : (_tmp36_ = (gee_collection_object_unref (_tmp36_), NULL)), _tmp37_);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			ValaCCodeDeclaration* _tmp39_;
			char* _tmp38_;
			ValaCCodeVariableDeclarator* _tmp41_;
			ValaCCodeExpression* _tmp40_;
			gboolean _tmp42_;
			gboolean _tmp43_;
			gboolean _tmp44_;
			char* _tmp49_;
			gboolean _tmp50_;
			ValaStruct* _tmp54_;
			ValaTypeSymbol* _tmp53_;
			ValaStruct* st;
			gboolean _tmp55_;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			_tmp39_ = NULL;
			_tmp38_ = NULL;
			_cdecl_ = (_tmp39_ = vala_ccode_declaration_new (_tmp38_ = vala_data_type_get_cname (vala_formal_parameter_get_parameter_type (param))), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp39_);
			_tmp38_ = (g_free (_tmp38_), NULL);
			_tmp41_ = NULL;
			_tmp40_ = NULL;
			vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp41_ = vala_ccode_variable_declarator_new (vala_symbol_get_name ((ValaSymbol*) param), _tmp40_ = vala_ccode_base_module_default_value_for_type ((ValaCCodeBaseModule*) self, vala_formal_parameter_get_parameter_type (param), TRUE), NULL)));
			(_tmp41_ == NULL) ? NULL : (_tmp41_ = (vala_ccode_node_unref (_tmp41_), NULL));
			(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_ccode_node_unref (_tmp40_), NULL));
			vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
			_tmp42_ = FALSE;
			_tmp43_ = FALSE;
			_tmp44_ = FALSE;
			if (_vala_strcmp0 (type_signature, "") == 0) {
				_tmp44_ = vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN;
			} else {
				_tmp44_ = FALSE;
			}
			if (_tmp44_) {
				_tmp43_ = vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param)) != NULL;
			} else {
				_tmp43_ = FALSE;
			}
			if (_tmp43_) {
				char* _tmp45_;
				_tmp45_ = NULL;
				_tmp42_ = _vala_strcmp0 (_tmp45_ = vala_symbol_get_full_name ((ValaSymbol*) vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param))), "DBus.BusName") == 0;
				_tmp45_ = (g_free (_tmp45_), NULL);
			} else {
				_tmp42_ = FALSE;
			}
			if (_tmp42_) {
				ValaCCodeIdentifier* _tmp46_;
				ValaCCodeFunctionCall* _tmp47_;
				ValaCCodeFunctionCall* get_sender;
				ValaCCodeIdentifier* _tmp48_;
				/* first parameter is a string parameter called 'sender'
				 pass bus name of sender*/
				_tmp46_ = NULL;
				_tmp47_ = NULL;
				get_sender = (_tmp47_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp46_ = vala_ccode_identifier_new ("dbus_message_get_sender"))), (_tmp46_ == NULL) ? NULL : (_tmp46_ = (vala_ccode_node_unref (_tmp46_), NULL)), _tmp47_);
				_tmp48_ = NULL;
				vala_ccode_function_call_add_argument (get_sender, (ValaCCodeExpression*) (_tmp48_ = vala_ccode_identifier_new ("message")));
				(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL));
				vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) get_sender);
				(get_sender == NULL) ? NULL : (get_sender = (vala_ccode_node_unref (get_sender), NULL));
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				continue;
			}
			_tmp49_ = NULL;
			if ((_tmp50_ = (_tmp49_ = vala_data_type_get_type_signature (vala_formal_parameter_get_parameter_type (param))) == NULL, _tmp49_ = (g_free (_tmp49_), NULL), _tmp50_)) {
				char* _tmp52_;
				char* _tmp51_;
				_tmp52_ = NULL;
				_tmp51_ = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (param)), _tmp52_ = g_strdup_printf ("D-Bus serialization of type `%s' is not supported", _tmp51_ = vala_code_node_to_string ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (param))));
				_tmp52_ = (g_free (_tmp52_), NULL);
				_tmp51_ = (g_free (_tmp51_), NULL);
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				continue;
			}
			_tmp54_ = NULL;
			_tmp53_ = NULL;
			st = (_tmp54_ = (_tmp53_ = vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param)), VALA_IS_STRUCT (_tmp53_) ? ((ValaStruct*) _tmp53_) : NULL), (_tmp54_ == NULL) ? NULL : vala_code_node_ref (_tmp54_));
			_tmp55_ = FALSE;
			if (vala_formal_parameter_get_direction (param) != VALA_PARAMETER_DIRECTION_IN) {
				_tmp55_ = TRUE;
			} else {
				gboolean _tmp56_;
				_tmp56_ = FALSE;
				if (st != NULL) {
					_tmp56_ = !vala_struct_is_simple_type (st);
				} else {
					_tmp56_ = FALSE;
				}
				_tmp55_ = _tmp56_;
			}
			if (_tmp55_) {
				ValaCCodeUnaryExpression* _tmp58_;
				ValaCCodeIdentifier* _tmp57_;
				_tmp58_ = NULL;
				_tmp57_ = NULL;
				vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp58_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp57_ = vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) param))))));
				(_tmp58_ == NULL) ? NULL : (_tmp58_ = (vala_ccode_node_unref (_tmp58_), NULL));
				(_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_ccode_node_unref (_tmp57_), NULL));
			} else {
				ValaCCodeIdentifier* _tmp59_;
				_tmp59_ = NULL;
				vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp59_ = vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) param))));
				(_tmp59_ == NULL) ? NULL : (_tmp59_ = (vala_ccode_node_unref (_tmp59_), NULL));
			}
			if (VALA_IS_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param))) {
				ValaArrayType* _tmp60_;
				ValaArrayType* array_type;
				_tmp60_ = NULL;
				array_type = (_tmp60_ = VALA_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param)), (_tmp60_ == NULL) ? NULL : vala_code_node_ref (_tmp60_));
				{
					gint dim;
					dim = 1;
					for (; dim <= vala_array_type_get_rank (array_type); dim++) {
						char* length_cname;
						ValaCCodeDeclaration* _tmp61_;
						ValaCCodeVariableDeclarator* _tmp63_;
						ValaCCodeConstant* _tmp62_;
						length_cname = vala_ccode_module_get_array_length_cname ((ValaCCodeModule*) self, vala_symbol_get_name ((ValaSymbol*) param), dim);
						_tmp61_ = NULL;
						_cdecl_ = (_tmp61_ = vala_ccode_declaration_new ("int"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp61_);
						_tmp63_ = NULL;
						_tmp62_ = NULL;
						vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp63_ = vala_ccode_variable_declarator_new (length_cname, (ValaCCodeExpression*) (_tmp62_ = vala_ccode_constant_new ("0")), NULL)));
						(_tmp63_ == NULL) ? NULL : (_tmp63_ = (vala_ccode_node_unref (_tmp63_), NULL));
						(_tmp62_ == NULL) ? NULL : (_tmp62_ = (vala_ccode_node_unref (_tmp62_), NULL));
						vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
						if (vala_formal_parameter_get_direction (param) != VALA_PARAMETER_DIRECTION_IN) {
							ValaCCodeUnaryExpression* _tmp65_;
							ValaCCodeIdentifier* _tmp64_;
							_tmp65_ = NULL;
							_tmp64_ = NULL;
							vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp65_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp64_ = vala_ccode_identifier_new (length_cname)))));
							(_tmp65_ == NULL) ? NULL : (_tmp65_ = (vala_ccode_node_unref (_tmp65_), NULL));
							(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_ccode_node_unref (_tmp64_), NULL));
						} else {
							ValaCCodeIdentifier* _tmp66_;
							_tmp66_ = NULL;
							vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp66_ = vala_ccode_identifier_new (length_cname)));
							(_tmp66_ == NULL) ? NULL : (_tmp66_ = (vala_ccode_node_unref (_tmp66_), NULL));
						}
						length_cname = (g_free (length_cname), NULL);
					}
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			}
			if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN) {
				char* _tmp68_;
				char* _tmp67_;
				ValaCCodeIdentifier* target;
				ValaCCodeIdentifier* _tmp69_;
				ValaCCodeExpression* _tmp70_;
				ValaCCodeExpression* expr;
				ValaCCodeExpressionStatement* _tmp72_;
				ValaCCodeAssignment* _tmp71_;
				_tmp68_ = NULL;
				_tmp67_ = NULL;
				type_signature = (_tmp68_ = g_strconcat (type_signature, _tmp67_ = vala_data_type_get_type_signature (vala_formal_parameter_get_parameter_type (param)), NULL), type_signature = (g_free (type_signature), NULL), _tmp68_);
				_tmp67_ = (g_free (_tmp67_), NULL);
				target = vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) param));
				_tmp69_ = NULL;
				_tmp70_ = NULL;
				expr = (_tmp70_ = vala_dbus_module_read_expression ((ValaDBusModule*) self, prefragment, vala_formal_parameter_get_parameter_type (param), (ValaCCodeExpression*) (_tmp69_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) target), (_tmp69_ == NULL) ? NULL : (_tmp69_ = (vala_ccode_node_unref (_tmp69_), NULL)), _tmp70_);
				_tmp72_ = NULL;
				_tmp71_ = NULL;
				vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp72_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp71_ = vala_ccode_assignment_new ((ValaCCodeExpression*) target, expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
				(_tmp72_ == NULL) ? NULL : (_tmp72_ = (vala_ccode_node_unref (_tmp72_), NULL));
				(_tmp71_ == NULL) ? NULL : (_tmp71_ = (vala_ccode_node_unref (_tmp71_), NULL));
				(target == NULL) ? NULL : (target = (vala_ccode_node_unref (target), NULL));
				(expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL));
			} else {
				ValaCCodeIdentifier* _tmp74_;
				ValaCCodeIdentifier* _tmp73_;
				_tmp74_ = NULL;
				_tmp73_ = NULL;
				vala_dbus_module_write_expression ((ValaDBusModule*) self, postfragment, vala_formal_parameter_get_parameter_type (param), (ValaCCodeExpression*) (_tmp73_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) (_tmp74_ = vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) param))));
				(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_ccode_node_unref (_tmp74_), NULL));
				(_tmp73_ == NULL) ? NULL : (_tmp73_ = (vala_ccode_node_unref (_tmp73_), NULL));
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	_tmp76_ = NULL;
	_tmp75_ = NULL;
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) (_tmp76_ = vala_ccode_constant_new (_tmp75_ = g_strdup_printf ("\"%s\"", type_signature))));
	(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_ccode_node_unref (_tmp76_), NULL));
	_tmp75_ = (g_free (_tmp75_), NULL);
	if (!VALA_IS_VOID_TYPE (vala_method_get_return_type (m))) {
		char* _tmp77_;
		gboolean _tmp78_;
		_tmp77_ = NULL;
		if ((_tmp78_ = (_tmp77_ = vala_data_type_get_type_signature (vala_method_get_return_type (m))) == NULL, _tmp77_ = (g_free (_tmp77_), NULL), _tmp78_)) {
			char* _tmp80_;
			char* _tmp79_;
			_tmp80_ = NULL;
			_tmp79_ = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_method_get_return_type (m)), _tmp80_ = g_strdup_printf ("D-Bus serialization of type `%s' is not supported", _tmp79_ = vala_code_node_to_string ((ValaCodeNode*) vala_method_get_return_type (m))));
			_tmp80_ = (g_free (_tmp80_), NULL);
			_tmp79_ = (g_free (_tmp79_), NULL);
		} else {
			ValaCCodeDeclaration* _tmp82_;
			char* _tmp81_;
			ValaCCodeVariableDeclarator* _tmp83_;
			ValaCCodeExpressionStatement* _tmp86_;
			ValaCCodeAssignment* _tmp85_;
			ValaCCodeIdentifier* _tmp84_;
			ValaCCodeIdentifier* _tmp94_;
			ValaCCodeIdentifier* _tmp93_;
			_tmp82_ = NULL;
			_tmp81_ = NULL;
			_cdecl_ = (_tmp82_ = vala_ccode_declaration_new (_tmp81_ = vala_data_type_get_cname (vala_method_get_return_type (m))), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp82_);
			_tmp81_ = (g_free (_tmp81_), NULL);
			_tmp83_ = NULL;
			vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp83_ = vala_ccode_variable_declarator_new ("result", NULL, NULL)));
			(_tmp83_ == NULL) ? NULL : (_tmp83_ = (vala_ccode_node_unref (_tmp83_), NULL));
			vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
			_tmp86_ = NULL;
			_tmp85_ = NULL;
			_tmp84_ = NULL;
			vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp86_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp85_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new ("result")), (ValaCCodeExpression*) ccall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
			(_tmp86_ == NULL) ? NULL : (_tmp86_ = (vala_ccode_node_unref (_tmp86_), NULL));
			(_tmp85_ == NULL) ? NULL : (_tmp85_ = (vala_ccode_node_unref (_tmp85_), NULL));
			(_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL));
			if (VALA_IS_ARRAY_TYPE (vala_method_get_return_type (m))) {
				ValaArrayType* _tmp87_;
				ValaArrayType* array_type;
				_tmp87_ = NULL;
				array_type = (_tmp87_ = VALA_ARRAY_TYPE (vala_method_get_return_type (m)), (_tmp87_ == NULL) ? NULL : vala_code_node_ref (_tmp87_));
				{
					gint dim;
					dim = 1;
					for (; dim <= vala_array_type_get_rank (array_type); dim++) {
						char* length_cname;
						ValaCCodeDeclaration* _tmp88_;
						ValaCCodeVariableDeclarator* _tmp90_;
						ValaCCodeConstant* _tmp89_;
						ValaCCodeUnaryExpression* _tmp92_;
						ValaCCodeIdentifier* _tmp91_;
						length_cname = vala_ccode_module_get_array_length_cname ((ValaCCodeModule*) self, "result", dim);
						_tmp88_ = NULL;
						_cdecl_ = (_tmp88_ = vala_ccode_declaration_new ("int"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp88_);
						_tmp90_ = NULL;
						_tmp89_ = NULL;
						vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp90_ = vala_ccode_variable_declarator_new (length_cname, (ValaCCodeExpression*) (_tmp89_ = vala_ccode_constant_new ("0")), NULL)));
						(_tmp90_ == NULL) ? NULL : (_tmp90_ = (vala_ccode_node_unref (_tmp90_), NULL));
						(_tmp89_ == NULL) ? NULL : (_tmp89_ = (vala_ccode_node_unref (_tmp89_), NULL));
						vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
						_tmp92_ = NULL;
						_tmp91_ = NULL;
						vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp92_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp91_ = vala_ccode_identifier_new (length_cname)))));
						(_tmp92_ == NULL) ? NULL : (_tmp92_ = (vala_ccode_node_unref (_tmp92_), NULL));
						(_tmp91_ == NULL) ? NULL : (_tmp91_ = (vala_ccode_node_unref (_tmp91_), NULL));
						length_cname = (g_free (length_cname), NULL);
					}
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			}
			_tmp94_ = NULL;
			_tmp93_ = NULL;
			vala_dbus_module_write_expression ((ValaDBusModule*) self, postfragment, vala_method_get_return_type (m), (ValaCCodeExpression*) (_tmp93_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) (_tmp94_ = vala_ccode_identifier_new ("result")));
			(_tmp94_ == NULL) ? NULL : (_tmp94_ = (vala_ccode_node_unref (_tmp94_), NULL));
			(_tmp93_ == NULL) ? NULL : (_tmp93_ = (vala_ccode_node_unref (_tmp93_), NULL));
		}
	} else {
		ValaCCodeExpressionStatement* _tmp95_;
		_tmp95_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp95_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccall)));
		(_tmp95_ == NULL) ? NULL : (_tmp95_ = (vala_ccode_node_unref (_tmp95_), NULL));
	}
	_tmp96_ = NULL;
	_cdecl_ = (_tmp96_ = vala_ccode_declaration_new ("DBusMessage*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp96_);
	_tmp97_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp97_ = vala_ccode_variable_declarator_new ("reply", NULL, NULL)));
	(_tmp97_ == NULL) ? NULL : (_tmp97_ = (vala_ccode_node_unref (_tmp97_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp98_ = NULL;
	if ((_tmp99_ = gee_collection_get_size ((GeeCollection*) (_tmp98_ = vala_code_node_get_error_types ((ValaCodeNode*) m))) > 0, (_tmp98_ == NULL) ? NULL : (_tmp98_ = (gee_collection_object_unref (_tmp98_), NULL)), _tmp99_)) {
		ValaCCodeUnaryExpression* _tmp101_;
		ValaCCodeIdentifier* _tmp100_;
		ValaCCodeBlock* error_block;
		ValaCCodeIdentifier* _tmp102_;
		ValaCCodeFunctionCall* _tmp103_;
		ValaCCodeFunctionCall* msgcall;
		ValaCCodeIdentifier* _tmp104_;
		ValaCCodeIdentifier* _tmp105_;
		ValaCCodeMemberAccess* _tmp107_;
		ValaCCodeIdentifier* _tmp106_;
		ValaCCodeExpressionStatement* _tmp110_;
		ValaCCodeAssignment* _tmp109_;
		ValaCCodeIdentifier* _tmp108_;
		ValaCCodeReturnStatement* _tmp112_;
		ValaCCodeIdentifier* _tmp111_;
		ValaCCodeIfStatement* _tmp114_;
		ValaCCodeIdentifier* _tmp113_;
		_tmp101_ = NULL;
		_tmp100_ = NULL;
		vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp101_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp100_ = vala_ccode_identifier_new ("error")))));
		(_tmp101_ == NULL) ? NULL : (_tmp101_ = (vala_ccode_node_unref (_tmp101_), NULL));
		(_tmp100_ == NULL) ? NULL : (_tmp100_ = (vala_ccode_node_unref (_tmp100_), NULL));
		error_block = vala_ccode_block_new ();
		_tmp102_ = NULL;
		_tmp103_ = NULL;
		msgcall = (_tmp103_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp102_ = vala_ccode_identifier_new ("dbus_message_new_error"))), (_tmp102_ == NULL) ? NULL : (_tmp102_ = (vala_ccode_node_unref (_tmp102_), NULL)), _tmp103_);
		_tmp104_ = NULL;
		vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp104_ = vala_ccode_identifier_new ("message")));
		(_tmp104_ == NULL) ? NULL : (_tmp104_ = (vala_ccode_node_unref (_tmp104_), NULL));
		_tmp105_ = NULL;
		vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp105_ = vala_ccode_identifier_new ("DBUS_ERROR_FAILED")));
		(_tmp105_ == NULL) ? NULL : (_tmp105_ = (vala_ccode_node_unref (_tmp105_), NULL));
		_tmp107_ = NULL;
		_tmp106_ = NULL;
		vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp107_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp106_ = vala_ccode_identifier_new ("error")), "message")));
		(_tmp107_ == NULL) ? NULL : (_tmp107_ = (vala_ccode_node_unref (_tmp107_), NULL));
		(_tmp106_ == NULL) ? NULL : (_tmp106_ = (vala_ccode_node_unref (_tmp106_), NULL));
		_tmp110_ = NULL;
		_tmp109_ = NULL;
		_tmp108_ = NULL;
		vala_ccode_block_add_statement (error_block, (ValaCCodeNode*) (_tmp110_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp109_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp108_ = vala_ccode_identifier_new ("reply")), (ValaCCodeExpression*) msgcall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
		(_tmp110_ == NULL) ? NULL : (_tmp110_ = (vala_ccode_node_unref (_tmp110_), NULL));
		(_tmp109_ == NULL) ? NULL : (_tmp109_ = (vala_ccode_node_unref (_tmp109_), NULL));
		(_tmp108_ == NULL) ? NULL : (_tmp108_ = (vala_ccode_node_unref (_tmp108_), NULL));
		_tmp112_ = NULL;
		_tmp111_ = NULL;
		vala_ccode_block_add_statement (error_block, (ValaCCodeNode*) (_tmp112_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp111_ = vala_ccode_identifier_new ("reply")))));
		(_tmp112_ == NULL) ? NULL : (_tmp112_ = (vala_ccode_node_unref (_tmp112_), NULL));
		(_tmp111_ == NULL) ? NULL : (_tmp111_ = (vala_ccode_node_unref (_tmp111_), NULL));
		_tmp114_ = NULL;
		_tmp113_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp114_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) (_tmp113_ = vala_ccode_identifier_new ("error")), (ValaCCodeStatement*) error_block, NULL)));
		(_tmp114_ == NULL) ? NULL : (_tmp114_ = (vala_ccode_node_unref (_tmp114_), NULL));
		(_tmp113_ == NULL) ? NULL : (_tmp113_ = (vala_ccode_node_unref (_tmp113_), NULL));
		(error_block == NULL) ? NULL : (error_block = (vala_ccode_node_unref (error_block), NULL));
		(msgcall == NULL) ? NULL : (msgcall = (vala_ccode_node_unref (msgcall), NULL));
	}
	_tmp115_ = NULL;
	_tmp116_ = NULL;
	msgcall = (_tmp116_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp115_ = vala_ccode_identifier_new ("dbus_message_new_method_return"))), (_tmp115_ == NULL) ? NULL : (_tmp115_ = (vala_ccode_node_unref (_tmp115_), NULL)), _tmp116_);
	_tmp117_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp117_ = vala_ccode_identifier_new ("message")));
	(_tmp117_ == NULL) ? NULL : (_tmp117_ = (vala_ccode_node_unref (_tmp117_), NULL));
	_tmp120_ = NULL;
	_tmp119_ = NULL;
	_tmp118_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp120_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp119_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp118_ = vala_ccode_identifier_new ("reply")), (ValaCCodeExpression*) msgcall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp120_ == NULL) ? NULL : (_tmp120_ = (vala_ccode_node_unref (_tmp120_), NULL));
	(_tmp119_ == NULL) ? NULL : (_tmp119_ = (vala_ccode_node_unref (_tmp119_), NULL));
	(_tmp118_ == NULL) ? NULL : (_tmp118_ = (vala_ccode_node_unref (_tmp118_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) postfragment);
	_tmp122_ = NULL;
	_tmp121_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp122_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp121_ = vala_ccode_identifier_new ("reply")))));
	(_tmp122_ == NULL) ? NULL : (_tmp122_ = (vala_ccode_node_unref (_tmp122_), NULL));
	(_tmp121_ == NULL) ? NULL : (_tmp121_ = (vala_ccode_node_unref (_tmp121_), NULL));
	_tmp123_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp123_ = vala_ccode_function_copy (function)));
	(_tmp123_ == NULL) ? NULL : (_tmp123_ = (vala_ccode_node_unref (_tmp123_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp124_ = NULL;
	return (_tmp124_ = wrapper_name, (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), (prefragment == NULL) ? NULL : (prefragment = (vala_ccode_node_unref (prefragment), NULL)), (postfragment == NULL) ? NULL : (postfragment = (vala_ccode_node_unref (postfragment), NULL)), (message_signature == NULL) ? NULL : (message_signature = (vala_ccode_node_unref (message_signature), NULL)), (signature_check == NULL) ? NULL : (signature_check = (vala_ccode_node_unref (signature_check), NULL)), (signature_error_block == NULL) ? NULL : (signature_error_block = (vala_ccode_node_unref (signature_error_block), NULL)), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), type_signature = (g_free (type_signature), NULL), (msgcall == NULL) ? NULL : (msgcall = (vala_ccode_node_unref (msgcall), NULL)), _tmp124_);
}


static char* vala_dbus_server_module_generate_dbus_signal_wrapper (ValaDBusServerModule* self, ValaSignal* sig, ValaObjectTypeSymbol* sym, const char* dbus_iface_name) {
	char* _tmp1_;
	char* _tmp0_;
	char* _tmp2_;
	char* wrapper_name;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp3_;
	ValaCCodeFormalParameter* _tmp11_;
	ValaCCodeBlock* block;
	ValaCCodeFragment* prefragment;
	ValaCCodeIdentifier* _tmp12_;
	ValaCCodeFunctionCall* _tmp13_;
	ValaCCodeFunctionCall* path;
	ValaCCodeIdentifier* _tmp14_;
	ValaCCodeConstant* _tmp15_;
	ValaCCodeDeclaration* _tmp16_;
	ValaCCodeVariableDeclarator* _tmp17_;
	ValaCCodeDeclaration* _tmp18_;
	ValaCCodeVariableDeclarator* _tmp19_;
	ValaCCodeDeclaration* _tmp20_;
	ValaCCodeVariableDeclarator* _tmp21_;
	ValaCCodeIdentifier* _tmp22_;
	ValaCCodeFunctionCall* _tmp23_;
	ValaCCodeFunctionCall* msgcall;
	ValaCCodeIdentifier* _tmp24_;
	ValaCCodeConstant* _tmp26_;
	char* _tmp25_;
	ValaCCodeConstant* _tmp29_;
	char* _tmp28_;
	char* _tmp27_;
	ValaCCodeExpressionStatement* _tmp32_;
	ValaCCodeAssignment* _tmp31_;
	ValaCCodeIdentifier* _tmp30_;
	ValaCCodeIdentifier* _tmp33_;
	ValaCCodeFunctionCall* _tmp34_;
	ValaCCodeFunctionCall* iter_call;
	ValaCCodeIdentifier* _tmp35_;
	ValaCCodeUnaryExpression* _tmp37_;
	ValaCCodeIdentifier* _tmp36_;
	ValaCCodeExpressionStatement* _tmp38_;
	ValaCCodeIdentifier* _tmp43_;
	ValaCCodeFunctionCall* _tmp44_;
	ValaCCodeFunctionCall* ccall;
	ValaCCodeIdentifier* _tmp45_;
	ValaCCodeIdentifier* _tmp46_;
	ValaCCodeConstant* _tmp47_;
	ValaCCodeExpressionStatement* _tmp48_;
	ValaCCodeIdentifier* _tmp49_;
	ValaCCodeFunctionCall* _tmp50_;
	ValaCCodeFunctionCall* message_unref;
	ValaCCodeIdentifier* _tmp51_;
	ValaCCodeExpressionStatement* _tmp52_;
	ValaCCodeFunction* _tmp53_;
	char* _tmp54_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sig != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	g_return_val_if_fail (dbus_iface_name != NULL, NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	wrapper_name = (_tmp2_ = g_strdup_printf ("_dbus_%s_%s", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) sym, NULL), _tmp1_ = vala_signal_get_cname (sig)), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	/* declaration*/
	_cdecl_ = NULL;
	function = vala_ccode_function_new (wrapper_name, "void");
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp3_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp3_ = vala_ccode_formal_parameter_new ("_sender", "GObject*"));
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
	{
		GeeList* _tmp4_;
		GeeIterator* _tmp5_;
		GeeIterator* _param_it;
		_tmp4_ = NULL;
		_tmp5_ = NULL;
		_param_it = (_tmp5_ = gee_iterable_iterator ((GeeIterable*) (_tmp4_ = vala_signal_get_parameters (sig))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (gee_collection_object_unref (_tmp4_), NULL)), _tmp5_);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			GeeHashMap* _tmp6_;
			ValaCCodeFormalParameter* _tmp7_;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			/* ensure ccodenode of parameter is set*/
			_tmp6_ = NULL;
			vala_ccode_method_module_generate_parameter ((ValaCCodeMethodModule*) self, param, ((ValaCCodeBaseModule*) self)->source_declarations, (GeeMap*) (_tmp6_ = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal)), NULL);
			(_tmp6_ == NULL) ? NULL : (_tmp6_ = (gee_collection_object_unref (_tmp6_), NULL));
			_tmp7_ = NULL;
			vala_ccode_function_add_parameter (function, _tmp7_ = VALA_CCODE_FORMAL_PARAMETER (vala_ccode_base_module_get_ccodenode ((ValaCCodeBaseModule*) self, (ValaCodeNode*) param)));
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			if (VALA_IS_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param))) {
				ValaArrayType* _tmp8_;
				ValaArrayType* array_type;
				_tmp8_ = NULL;
				array_type = (_tmp8_ = VALA_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param)), (_tmp8_ == NULL) ? NULL : vala_code_node_ref (_tmp8_));
				{
					gint dim;
					dim = 1;
					for (; dim <= vala_array_type_get_rank (array_type); dim++) {
						ValaCCodeFormalParameter* _tmp10_;
						char* _tmp9_;
						_tmp10_ = NULL;
						_tmp9_ = NULL;
						vala_ccode_function_add_parameter (function, _tmp10_ = vala_ccode_formal_parameter_new (_tmp9_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), vala_symbol_get_name ((ValaSymbol*) param), dim), "int"));
						(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
						_tmp9_ = (g_free (_tmp9_), NULL);
					}
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	_tmp11_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp11_ = vala_ccode_formal_parameter_new ("_connection", "DBusConnection*"));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	block = vala_ccode_block_new ();
	prefragment = vala_ccode_fragment_new ();
	_tmp12_ = NULL;
	_tmp13_ = NULL;
	path = (_tmp13_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp12_ = vala_ccode_identifier_new ("g_object_get_data"))), (_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL)), _tmp13_);
	_tmp14_ = NULL;
	vala_ccode_function_call_add_argument (path, (ValaCCodeExpression*) (_tmp14_ = vala_ccode_identifier_new ("_sender")));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	_tmp15_ = NULL;
	vala_ccode_function_call_add_argument (path, (ValaCCodeExpression*) (_tmp15_ = vala_ccode_constant_new ("\"dbus_object_path\"")));
	(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
	_tmp16_ = NULL;
	_cdecl_ = (_tmp16_ = vala_ccode_declaration_new ("const char *"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp16_);
	_tmp17_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp17_ = vala_ccode_variable_declarator_new ("_path", (ValaCCodeExpression*) path, NULL)));
	(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp18_ = NULL;
	_cdecl_ = (_tmp18_ = vala_ccode_declaration_new ("DBusMessage"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp18_);
	_tmp19_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp19_ = vala_ccode_variable_declarator_new ("*_message", NULL, NULL)));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp20_ = NULL;
	_cdecl_ = (_tmp20_ = vala_ccode_declaration_new ("DBusMessageIter"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp20_);
	_tmp21_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp21_ = vala_ccode_variable_declarator_new ("_iter", NULL, NULL)));
	(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) prefragment);
	_tmp22_ = NULL;
	_tmp23_ = NULL;
	msgcall = (_tmp23_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("dbus_message_new_signal"))), (_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL)), _tmp23_);
	_tmp24_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp24_ = vala_ccode_identifier_new ("_path")));
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	_tmp26_ = NULL;
	_tmp25_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp26_ = vala_ccode_constant_new (_tmp25_ = g_strdup_printf ("\"%s\"", dbus_iface_name))));
	(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
	_tmp25_ = (g_free (_tmp25_), NULL);
	_tmp29_ = NULL;
	_tmp28_ = NULL;
	_tmp27_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp29_ = vala_ccode_constant_new (_tmp28_ = g_strdup_printf ("\"%s\"", _tmp27_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) sig))))));
	(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
	_tmp28_ = (g_free (_tmp28_), NULL);
	_tmp27_ = (g_free (_tmp27_), NULL);
	_tmp32_ = NULL;
	_tmp31_ = NULL;
	_tmp30_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp32_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp31_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new ("_message")), (ValaCCodeExpression*) msgcall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
	_tmp33_ = NULL;
	_tmp34_ = NULL;
	iter_call = (_tmp34_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp33_ = vala_ccode_identifier_new ("dbus_message_iter_init_append"))), (_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL)), _tmp34_);
	_tmp35_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp35_ = vala_ccode_identifier_new ("_message")));
	(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
	_tmp37_ = NULL;
	_tmp36_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp37_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp36_ = vala_ccode_identifier_new ("_iter")))));
	(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
	(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
	_tmp38_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp38_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
	{
		GeeList* _tmp39_;
		GeeIterator* _tmp40_;
		GeeIterator* _param_it;
		_tmp39_ = NULL;
		_tmp40_ = NULL;
		_param_it = (_tmp40_ = gee_iterable_iterator ((GeeIterable*) (_tmp39_ = vala_signal_get_parameters (sig))), (_tmp39_ == NULL) ? NULL : (_tmp39_ = (gee_collection_object_unref (_tmp39_), NULL)), _tmp40_);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			ValaCCodeExpression* expr;
			ValaCCodeIdentifier* _tmp42_;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			expr = (ValaCCodeExpression*) vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) param));
			if (vala_data_type_is_real_struct_type (vala_formal_parameter_get_parameter_type (param))) {
				ValaCCodeExpression* _tmp41_;
				_tmp41_ = NULL;
				expr = (_tmp41_ = (ValaCCodeExpression*) vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION, expr), (expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL)), _tmp41_);
			}
			_tmp42_ = NULL;
			vala_dbus_module_write_expression ((ValaDBusModule*) self, prefragment, vala_formal_parameter_get_parameter_type (param), (ValaCCodeExpression*) (_tmp42_ = vala_ccode_identifier_new ("_iter")), expr);
			(_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL));
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			(expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	_tmp43_ = NULL;
	_tmp44_ = NULL;
	ccall = (_tmp44_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp43_ = vala_ccode_identifier_new ("dbus_connection_send"))), (_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL)), _tmp44_);
	_tmp45_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp45_ = vala_ccode_identifier_new ("_connection")));
	(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
	_tmp46_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp46_ = vala_ccode_identifier_new ("_message")));
	(_tmp46_ == NULL) ? NULL : (_tmp46_ = (vala_ccode_node_unref (_tmp46_), NULL));
	_tmp47_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp47_ = vala_ccode_constant_new ("NULL")));
	(_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL));
	_tmp48_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp48_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccall)));
	(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL));
	_tmp49_ = NULL;
	_tmp50_ = NULL;
	message_unref = (_tmp50_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp49_ = vala_ccode_identifier_new ("dbus_message_unref"))), (_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL)), _tmp50_);
	_tmp51_ = NULL;
	vala_ccode_function_call_add_argument (message_unref, (ValaCCodeExpression*) (_tmp51_ = vala_ccode_identifier_new ("_message")));
	(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
	_tmp52_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp52_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) message_unref)));
	(_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_ccode_node_unref (_tmp52_), NULL));
	_tmp53_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp53_ = vala_ccode_function_copy (function)));
	(_tmp53_ == NULL) ? NULL : (_tmp53_ = (vala_ccode_node_unref (_tmp53_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp54_ = NULL;
	return (_tmp54_ = wrapper_name, (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), (prefragment == NULL) ? NULL : (prefragment = (vala_ccode_node_unref (prefragment), NULL)), (path == NULL) ? NULL : (path = (vala_ccode_node_unref (path), NULL)), (msgcall == NULL) ? NULL : (msgcall = (vala_ccode_node_unref (msgcall), NULL)), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), (message_unref == NULL) ? NULL : (message_unref = (vala_ccode_node_unref (message_unref), NULL)), _tmp54_);
}


static void vala_dbus_server_module_generate_register_function (ValaDBusServerModule* self, ValaObjectType* object_type) {
	ValaObjectTypeSymbol* _tmp0_;
	ValaObjectTypeSymbol* sym;
	char* _tmp2_;
	char* _tmp1_;
	ValaCCodeFunction* _tmp3_;
	ValaCCodeFunction* cfunc;
	ValaCCodeFormalParameter* _tmp4_;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* _tmp6_;
	ValaCCodeFunction* _tmp7_;
	ValaCCodeBlock* block;
	ValaCCodeIdentifier* _tmp8_;
	ValaCCodeFunctionCall* _tmp9_;
	ValaCCodeFunctionCall* get_path;
	ValaCCodeIdentifier* _tmp10_;
	ValaCCodeConstant* _tmp11_;
	ValaCCodeUnaryExpression* register_check;
	ValaCCodeBlock* register_block;
	ValaCCodeIdentifier* _tmp12_;
	ValaCCodeFunctionCall* _tmp13_;
	ValaCCodeFunctionCall* path_dup;
	ValaCCodeIdentifier* _tmp14_;
	ValaCCodeIdentifier* _tmp15_;
	ValaCCodeFunctionCall* _tmp16_;
	ValaCCodeFunctionCall* set_path;
	ValaCCodeIdentifier* _tmp17_;
	ValaCCodeConstant* _tmp18_;
	ValaCCodeExpressionStatement* _tmp19_;
	ValaCCodeIdentifier* _tmp20_;
	ValaCCodeFunctionCall* _tmp21_;
	ValaCCodeFunctionCall* cregister;
	ValaCCodeIdentifier* _tmp22_;
	ValaCCodeIdentifier* _tmp23_;
	ValaCCodeUnaryExpression* _tmp25_;
	ValaCCodeExpression* _tmp24_;
	ValaCCodeIdentifier* _tmp26_;
	ValaCCodeExpressionStatement* _tmp27_;
	ValaCCodeIdentifier* _tmp28_;
	ValaCCodeFunctionCall* _tmp29_;
	ValaCCodeFunctionCall* weak_unregister;
	ValaCCodeIdentifier* _tmp30_;
	ValaCCodeIdentifier* _tmp31_;
	ValaCCodeIdentifier* _tmp32_;
	ValaCCodeExpressionStatement* _tmp33_;
	ValaCCodeIfStatement* _tmp34_;
	ValaClass* _tmp36_;
	ValaObjectTypeSymbol* _tmp35_;
	ValaClass* cl;
	g_return_if_fail (self != NULL);
	g_return_if_fail (object_type != NULL);
	_tmp0_ = NULL;
	sym = (_tmp0_ = vala_object_type_get_type_symbol (object_type), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp3_ = NULL;
	cfunc = (_tmp3_ = vala_ccode_function_new (_tmp2_ = g_strconcat (_tmp1_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), "dbus_register_object", NULL), "void"), _tmp2_ = (g_free (_tmp2_), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp3_);
	_tmp4_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp4_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp5_ = vala_ccode_formal_parameter_new ("path", "const char*"));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp6_ = vala_ccode_formal_parameter_new ("object", "void*"));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	if (!vala_symbol_is_private_symbol ((ValaSymbol*) sym)) {
		((ValaCCodeBaseModule*) self)->dbus_glib_h_needed_in_header = TRUE;
	} else {
		((ValaCCodeBaseModule*) self)->dbus_glib_h_needed = TRUE;
		vala_ccode_function_set_modifiers (cfunc, vala_ccode_function_get_modifiers (cfunc) | VALA_CCODE_MODIFIERS_STATIC);
	}
	_tmp7_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp7_ = vala_ccode_function_copy (cfunc)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	block = vala_ccode_block_new ();
	vala_ccode_function_set_block (cfunc, block);
	_tmp8_ = NULL;
	_tmp9_ = NULL;
	get_path = (_tmp9_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp8_ = vala_ccode_identifier_new ("g_object_get_data"))), (_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL)), _tmp9_);
	_tmp10_ = NULL;
	vala_ccode_function_call_add_argument (get_path, (ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new ("object")));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	_tmp11_ = NULL;
	vala_ccode_function_call_add_argument (get_path, (ValaCCodeExpression*) (_tmp11_ = vala_ccode_constant_new ("\"dbus_object_path\"")));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	register_check = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_LOGICAL_NEGATION, (ValaCCodeExpression*) get_path);
	register_block = vala_ccode_block_new ();
	_tmp12_ = NULL;
	_tmp13_ = NULL;
	path_dup = (_tmp13_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp12_ = vala_ccode_identifier_new ("g_strdup"))), (_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL)), _tmp13_);
	_tmp14_ = NULL;
	vala_ccode_function_call_add_argument (path_dup, (ValaCCodeExpression*) (_tmp14_ = vala_ccode_identifier_new ("path")));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	_tmp15_ = NULL;
	_tmp16_ = NULL;
	set_path = (_tmp16_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new ("g_object_set_data"))), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL)), _tmp16_);
	_tmp17_ = NULL;
	vala_ccode_function_call_add_argument (set_path, (ValaCCodeExpression*) (_tmp17_ = vala_ccode_identifier_new ("object")));
	(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
	_tmp18_ = NULL;
	vala_ccode_function_call_add_argument (set_path, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_constant_new ("\"dbus_object_path\"")));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	vala_ccode_function_call_add_argument (set_path, (ValaCCodeExpression*) path_dup);
	_tmp19_ = NULL;
	vala_ccode_block_add_statement (register_block, (ValaCCodeNode*) (_tmp19_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) set_path)));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	_tmp20_ = NULL;
	_tmp21_ = NULL;
	cregister = (_tmp21_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new ("dbus_connection_register_object_path"))), (_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL)), _tmp21_);
	_tmp22_ = NULL;
	vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("connection")));
	(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
	_tmp23_ = NULL;
	vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) (_tmp23_ = vala_ccode_identifier_new ("path")));
	(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	_tmp25_ = NULL;
	_tmp24_ = NULL;
	vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) (_tmp25_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, _tmp24_ = vala_dbus_server_module_get_path_vtable (self, object_type))));
	(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	_tmp26_ = NULL;
	vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) (_tmp26_ = vala_ccode_identifier_new ("object")));
	(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
	_tmp27_ = NULL;
	vala_ccode_block_add_statement (register_block, (ValaCCodeNode*) (_tmp27_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) cregister)));
	(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
	_tmp28_ = NULL;
	_tmp29_ = NULL;
	weak_unregister = (_tmp29_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("g_object_weak_ref"))), (_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL)), _tmp29_);
	_tmp30_ = NULL;
	vala_ccode_function_call_add_argument (weak_unregister, (ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new ("object")));
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
	_tmp31_ = NULL;
	vala_ccode_function_call_add_argument (weak_unregister, (ValaCCodeExpression*) (_tmp31_ = vala_ccode_identifier_new ("_vala_dbus_unregister_object")));
	(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
	_tmp32_ = NULL;
	vala_ccode_function_call_add_argument (weak_unregister, (ValaCCodeExpression*) (_tmp32_ = vala_ccode_identifier_new ("connection")));
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	_tmp33_ = NULL;
	vala_ccode_block_add_statement (register_block, (ValaCCodeNode*) (_tmp33_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) weak_unregister)));
	(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
	_tmp34_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp34_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) register_check, (ValaCCodeStatement*) register_block, NULL)));
	(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
	vala_dbus_server_module_handle_signals (self, vala_object_type_get_type_symbol (object_type), block);
	_tmp36_ = NULL;
	_tmp35_ = NULL;
	cl = (_tmp36_ = (_tmp35_ = sym, VALA_IS_CLASS (_tmp35_) ? ((ValaClass*) _tmp35_) : NULL), (_tmp36_ == NULL) ? NULL : vala_code_node_ref (_tmp36_));
	if (cl != NULL) {
		{
			GeeList* _tmp37_;
			GeeIterator* _tmp38_;
			GeeIterator* _base_type_it;
			_tmp37_ = NULL;
			_tmp38_ = NULL;
			_base_type_it = (_tmp38_ = gee_iterable_iterator ((GeeIterable*) (_tmp37_ = vala_class_get_base_types (cl))), (_tmp37_ == NULL) ? NULL : (_tmp37_ = (gee_collection_object_unref (_tmp37_), NULL)), _tmp38_);
			while (gee_iterator_next (_base_type_it)) {
				ValaDataType* base_type;
				ValaObjectType* _tmp40_;
				ValaDataType* _tmp39_;
				ValaObjectType* base_obj_type;
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				_tmp40_ = NULL;
				_tmp39_ = NULL;
				base_obj_type = (_tmp40_ = (_tmp39_ = base_type, VALA_IS_OBJECT_TYPE (_tmp39_) ? ((ValaObjectType*) _tmp39_) : NULL), (_tmp40_ == NULL) ? NULL : vala_code_node_ref (_tmp40_));
				if (vala_dbus_server_module_type_implements_dbus_interface (self, vala_object_type_get_type_symbol (base_obj_type))) {
					ValaCCodeIdentifier* _tmp43_;
					char* _tmp42_;
					char* _tmp41_;
					ValaCCodeFunctionCall* _tmp44_;
					ValaCCodeFunctionCall* base_register;
					ValaCCodeIdentifier* _tmp45_;
					ValaCCodeIdentifier* _tmp46_;
					ValaCCodeIdentifier* _tmp47_;
					ValaCCodeExpressionStatement* _tmp48_;
					_tmp43_ = NULL;
					_tmp42_ = NULL;
					_tmp41_ = NULL;
					_tmp44_ = NULL;
					base_register = (_tmp44_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp43_ = vala_ccode_identifier_new (_tmp42_ = g_strconcat (_tmp41_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) vala_object_type_get_type_symbol (base_obj_type)), "dbus_register_object", NULL)))), (_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL)), _tmp42_ = (g_free (_tmp42_), NULL), _tmp41_ = (g_free (_tmp41_), NULL), _tmp44_);
					_tmp45_ = NULL;
					vala_ccode_function_call_add_argument (base_register, (ValaCCodeExpression*) (_tmp45_ = vala_ccode_identifier_new ("connection")));
					(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
					_tmp46_ = NULL;
					vala_ccode_function_call_add_argument (base_register, (ValaCCodeExpression*) (_tmp46_ = vala_ccode_identifier_new ("path")));
					(_tmp46_ == NULL) ? NULL : (_tmp46_ = (vala_ccode_node_unref (_tmp46_), NULL));
					_tmp47_ = NULL;
					vala_ccode_function_call_add_argument (base_register, (ValaCCodeExpression*) (_tmp47_ = vala_ccode_identifier_new ("object")));
					(_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL));
					_tmp48_ = NULL;
					vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp48_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) base_register)));
					(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL));
					(base_register == NULL) ? NULL : (base_register = (vala_ccode_node_unref (base_register), NULL));
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(base_obj_type == NULL) ? NULL : (base_obj_type = (vala_code_node_unref (base_obj_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
	}
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) cfunc);
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
	(cfunc == NULL) ? NULL : (cfunc = (vala_ccode_node_unref (cfunc), NULL));
	(block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL));
	(get_path == NULL) ? NULL : (get_path = (vala_ccode_node_unref (get_path), NULL));
	(register_check == NULL) ? NULL : (register_check = (vala_ccode_node_unref (register_check), NULL));
	(register_block == NULL) ? NULL : (register_block = (vala_ccode_node_unref (register_block), NULL));
	(path_dup == NULL) ? NULL : (path_dup = (vala_ccode_node_unref (path_dup), NULL));
	(set_path == NULL) ? NULL : (set_path = (vala_ccode_node_unref (set_path), NULL));
	(cregister == NULL) ? NULL : (cregister = (vala_ccode_node_unref (cregister), NULL));
	(weak_unregister == NULL) ? NULL : (weak_unregister = (vala_ccode_node_unref (weak_unregister), NULL));
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
}


static void vala_dbus_server_module_generate_unregister_function (ValaDBusServerModule* self, ValaObjectType* object_type) {
	ValaObjectTypeSymbol* _tmp0_;
	ValaObjectTypeSymbol* sym;
	char* _tmp3_;
	char* _tmp2_;
	char* _tmp1_;
	ValaCCodeFunction* _tmp4_;
	ValaCCodeFunction* cfunc;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* _tmp6_;
	ValaCCodeFunction* _tmp7_;
	ValaCCodeBlock* block;
	g_return_if_fail (self != NULL);
	g_return_if_fail (object_type != NULL);
	_tmp0_ = NULL;
	sym = (_tmp0_ = vala_object_type_get_type_symbol (object_type), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp4_ = NULL;
	cfunc = (_tmp4_ = vala_ccode_function_new (_tmp3_ = g_strconcat (_tmp2_ = g_strconcat ("_", _tmp1_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), NULL), "dbus_unregister", NULL), "void"), _tmp3_ = (g_free (_tmp3_), NULL), _tmp2_ = (g_free (_tmp2_), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp4_);
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp5_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp6_ = vala_ccode_formal_parameter_new ("user_data", "void*"));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	_tmp7_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp7_ = vala_ccode_function_copy (cfunc)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	block = vala_ccode_block_new ();
	vala_ccode_function_set_block (cfunc, block);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) cfunc);
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
	(cfunc == NULL) ? NULL : (cfunc = (vala_ccode_node_unref (cfunc), NULL));
	(block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL));
}


static void vala_dbus_server_module_handle_method (ValaDBusServerModule* self, const char* dbus_iface_name, const char* dbus_method_name, const char* handler_name, ValaCCodeBlock* block, ValaCCodeIfStatement** clastif) {
	ValaCCodeIdentifier* _tmp0_;
	ValaCCodeFunctionCall* _tmp1_;
	ValaCCodeFunctionCall* ccheck;
	ValaCCodeIdentifier* _tmp2_;
	ValaCCodeConstant* _tmp4_;
	char* _tmp3_;
	ValaCCodeConstant* _tmp6_;
	char* _tmp5_;
	ValaCCodeBlock* callblock;
	ValaCCodeIdentifier* _tmp7_;
	ValaCCodeFunctionCall* _tmp8_;
	ValaCCodeFunctionCall* ccall;
	ValaCCodeIdentifier* _tmp9_;
	ValaCCodeIdentifier* _tmp10_;
	ValaCCodeIdentifier* _tmp11_;
	ValaCCodeExpressionStatement* _tmp14_;
	ValaCCodeAssignment* _tmp13_;
	ValaCCodeIdentifier* _tmp12_;
	ValaCCodeIfStatement* cif;
	ValaCCodeIfStatement* _tmp16_;
	ValaCCodeIfStatement* _tmp15_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (dbus_iface_name != NULL);
	g_return_if_fail (dbus_method_name != NULL);
	g_return_if_fail (handler_name != NULL);
	g_return_if_fail (block != NULL);
	g_return_if_fail (clastif != NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	ccheck = (_tmp1_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp0_ = vala_ccode_identifier_new ("dbus_message_is_method_call"))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL)), _tmp1_);
	_tmp2_ = NULL;
	vala_ccode_function_call_add_argument (ccheck, (ValaCCodeExpression*) (_tmp2_ = vala_ccode_identifier_new ("message")));
	(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	vala_ccode_function_call_add_argument (ccheck, (ValaCCodeExpression*) (_tmp4_ = vala_ccode_constant_new (_tmp3_ = g_strdup_printf ("\"%s\"", dbus_iface_name))));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	vala_ccode_function_call_add_argument (ccheck, (ValaCCodeExpression*) (_tmp6_ = vala_ccode_constant_new (_tmp5_ = g_strdup_printf ("\"%s\"", dbus_method_name))));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	_tmp5_ = (g_free (_tmp5_), NULL);
	callblock = vala_ccode_block_new ();
	_tmp7_ = NULL;
	_tmp8_ = NULL;
	ccall = (_tmp8_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (handler_name))), (_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL)), _tmp8_);
	_tmp9_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new ("object")));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	_tmp10_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new ("connection")));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	_tmp11_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp11_ = vala_ccode_identifier_new ("message")));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	_tmp14_ = NULL;
	_tmp13_ = NULL;
	_tmp12_ = NULL;
	vala_ccode_block_add_statement (callblock, (ValaCCodeNode*) (_tmp14_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp12_ = vala_ccode_identifier_new ("reply")), (ValaCCodeExpression*) ccall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	cif = vala_ccode_if_statement_new ((ValaCCodeExpression*) ccheck, (ValaCCodeStatement*) callblock, NULL);
	if ((*clastif) == NULL) {
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) cif);
	} else {
		vala_ccode_if_statement_set_false_statement ((*clastif), (ValaCCodeStatement*) cif);
	}
	_tmp16_ = NULL;
	_tmp15_ = NULL;
	(*clastif) = (_tmp16_ = (_tmp15_ = cif, (_tmp15_ == NULL) ? NULL : vala_ccode_node_ref (_tmp15_)), ((*clastif) == NULL) ? NULL : ((*clastif) = (vala_ccode_node_unref ((*clastif)), NULL)), _tmp16_);
	(ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL));
	(callblock == NULL) ? NULL : (callblock = (vala_ccode_node_unref (callblock), NULL));
	(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
	(cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL));
}


static void vala_dbus_server_module_handle_methods (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name, ValaCCodeBlock* block, ValaCCodeIfStatement** clastif) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (sym != NULL);
	g_return_if_fail (dbus_iface_name != NULL);
	g_return_if_fail (block != NULL);
	g_return_if_fail (clastif != NULL);
	{
		GeeList* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _m_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_m_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = vala_object_type_symbol_get_methods (sym))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			gboolean _tmp2_;
			gboolean _tmp3_;
			gboolean _tmp4_;
			char* _tmp6_;
			char* _tmp5_;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			_tmp2_ = FALSE;
			_tmp3_ = FALSE;
			_tmp4_ = FALSE;
			if (VALA_IS_CREATION_METHOD (m)) {
				_tmp4_ = TRUE;
			} else {
				_tmp4_ = vala_method_get_binding (m) != MEMBER_BINDING_INSTANCE;
			}
			if (_tmp4_) {
				_tmp3_ = TRUE;
			} else {
				_tmp3_ = vala_method_get_overrides (m);
			}
			if (_tmp3_) {
				_tmp2_ = TRUE;
			} else {
				_tmp2_ = vala_symbol_get_access ((ValaSymbol*) m) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
			}
			if (_tmp2_) {
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) m)) {
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
				continue;
			}
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			vala_dbus_server_module_handle_method (self, dbus_iface_name, _tmp5_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) m)), _tmp6_ = vala_dbus_server_module_generate_dbus_wrapper (self, m, sym), block, &(*clastif));
			_tmp6_ = (g_free (_tmp6_), NULL);
			_tmp5_ = (g_free (_tmp5_), NULL);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
}


static char* vala_dbus_server_module_generate_dbus_property_get_wrapper (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name) {
	char* _tmp0_;
	char* _tmp1_;
	char* wrapper_name;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp4_;
	char* _tmp3_;
	char* _tmp2_;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* _tmp6_;
	ValaCCodeBlock* block;
	ValaCCodeFragment* prefragment;
	ValaCCodeDeclaration* _tmp7_;
	ValaCCodeVariableDeclarator* _tmp8_;
	ValaCCodeDeclaration* _tmp9_;
	ValaCCodeVariableDeclarator* _tmp10_;
	ValaCCodeVariableDeclarator* _tmp11_;
	ValaCCodeVariableDeclarator* _tmp12_;
	ValaCCodeIdentifier* _tmp13_;
	ValaCCodeFunctionCall* _tmp14_;
	ValaCCodeFunctionCall* message_signature;
	ValaCCodeIdentifier* _tmp15_;
	ValaCCodeIdentifier* _tmp16_;
	ValaCCodeFunctionCall* _tmp17_;
	ValaCCodeFunctionCall* signature_check;
	ValaCCodeConstant* _tmp18_;
	ValaCCodeBlock* signature_error_block;
	ValaCCodeReturnStatement* _tmp20_;
	ValaCCodeConstant* _tmp19_;
	ValaCCodeIfStatement* _tmp21_;
	ValaCCodeIdentifier* _tmp22_;
	ValaCCodeFunctionCall* _tmp23_;
	ValaCCodeFunctionCall* iter_call;
	ValaCCodeIdentifier* _tmp24_;
	ValaCCodeUnaryExpression* _tmp26_;
	ValaCCodeIdentifier* _tmp25_;
	ValaCCodeExpressionStatement* _tmp27_;
	ValaCCodeIdentifier* _tmp28_;
	ValaCCodeFunctionCall* _tmp29_;
	ValaCCodeFunctionCall* msgcall;
	ValaCCodeIdentifier* _tmp30_;
	ValaCCodeExpressionStatement* _tmp33_;
	ValaCCodeAssignment* _tmp32_;
	ValaCCodeIdentifier* _tmp31_;
	ValaCCodeFunctionCall* _tmp35_;
	ValaCCodeIdentifier* _tmp34_;
	ValaCCodeIdentifier* _tmp36_;
	ValaCCodeUnaryExpression* _tmp38_;
	ValaCCodeIdentifier* _tmp37_;
	ValaCCodeExpressionStatement* _tmp39_;
	ValaCCodeDeclaration* _tmp40_;
	ValaCCodeVariableDeclarator* _tmp41_;
	ValaCCodeIdentifier* target;
	ValaCCodeIdentifier* _tmp42_;
	ValaCCodeExpression* _tmp43_;
	ValaCCodeExpression* expr;
	ValaCCodeExpressionStatement* _tmp45_;
	ValaCCodeAssignment* _tmp44_;
	ValaCCodeDeclaration* _tmp46_;
	ValaCCodeVariableDeclarator* _tmp47_;
	ValaCCodeIdentifier* _tmp48_;
	ValaCCodeExpression* _tmp50_;
	ValaCCodeIdentifier* _tmp49_;
	ValaCCodeExpressionStatement* _tmp52_;
	ValaCCodeAssignment* _tmp51_;
	ValaCCodeIfStatement* clastif;
	ValaCCodeFunction* _tmp104_;
	char* _tmp105_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	g_return_val_if_fail (dbus_iface_name != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	wrapper_name = (_tmp1_ = g_strdup_printf ("_dbus_%s_property_get", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) sym, NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	_cdecl_ = NULL;
	function = vala_ccode_function_new (wrapper_name, "DBusMessage*");
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp4_ = vala_ccode_formal_parameter_new ("self", _tmp3_ = g_strconcat (_tmp2_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) sym, FALSE), "*", NULL)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp5_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp6_ = vala_ccode_formal_parameter_new ("message", "DBusMessage*"));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	block = vala_ccode_block_new ();
	prefragment = vala_ccode_fragment_new ();
	_tmp7_ = NULL;
	_cdecl_ = (_tmp7_ = vala_ccode_declaration_new ("DBusMessage*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp7_);
	_tmp8_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp8_ = vala_ccode_variable_declarator_new ("reply", NULL, NULL)));
	(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp9_ = NULL;
	_cdecl_ = (_tmp9_ = vala_ccode_declaration_new ("DBusMessageIter"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp9_);
	_tmp10_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp10_ = vala_ccode_variable_declarator_new ("iter", NULL, NULL)));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	_tmp11_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp11_ = vala_ccode_variable_declarator_new ("reply_iter", NULL, NULL)));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	_tmp12_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp12_ = vala_ccode_variable_declarator_new ("subiter", NULL, NULL)));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp13_ = NULL;
	_tmp14_ = NULL;
	message_signature = (_tmp14_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_identifier_new ("dbus_message_get_signature"))), (_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL)), _tmp14_);
	_tmp15_ = NULL;
	vala_ccode_function_call_add_argument (message_signature, (ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new ("message")));
	(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
	_tmp16_ = NULL;
	_tmp17_ = NULL;
	signature_check = (_tmp17_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new ("strcmp"))), (_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL)), _tmp17_);
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) message_signature);
	_tmp18_ = NULL;
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_constant_new ("\"ss\"")));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	signature_error_block = vala_ccode_block_new ();
	_tmp20_ = NULL;
	_tmp19_ = NULL;
	vala_ccode_block_add_statement (signature_error_block, (ValaCCodeNode*) (_tmp20_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp19_ = vala_ccode_constant_new ("NULL")))));
	(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	_tmp21_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp21_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) signature_check, (ValaCCodeStatement*) signature_error_block, NULL)));
	(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
	_tmp22_ = NULL;
	_tmp23_ = NULL;
	iter_call = (_tmp23_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("dbus_message_iter_init"))), (_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL)), _tmp23_);
	_tmp24_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp24_ = vala_ccode_identifier_new ("message")));
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	_tmp26_ = NULL;
	_tmp25_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp26_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp25_ = vala_ccode_identifier_new ("iter")))));
	(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
	(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
	_tmp27_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp27_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
	_tmp28_ = NULL;
	_tmp29_ = NULL;
	msgcall = (_tmp29_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("dbus_message_new_method_return"))), (_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL)), _tmp29_);
	_tmp30_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new ("message")));
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
	_tmp33_ = NULL;
	_tmp32_ = NULL;
	_tmp31_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp33_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp32_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp31_ = vala_ccode_identifier_new ("reply")), (ValaCCodeExpression*) msgcall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
	_tmp35_ = NULL;
	_tmp34_ = NULL;
	iter_call = (_tmp35_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp34_ = vala_ccode_identifier_new ("dbus_message_iter_init_append"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp35_);
	(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
	_tmp36_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp36_ = vala_ccode_identifier_new ("reply")));
	(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
	_tmp38_ = NULL;
	_tmp37_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp38_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp37_ = vala_ccode_identifier_new ("reply_iter")))));
	(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
	(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
	_tmp39_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp39_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) prefragment);
	_tmp40_ = NULL;
	_cdecl_ = (_tmp40_ = vala_ccode_declaration_new ("char*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp40_);
	_tmp41_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp41_ = vala_ccode_variable_declarator_new ("interface_name", NULL, NULL)));
	(_tmp41_ == NULL) ? NULL : (_tmp41_ = (vala_ccode_node_unref (_tmp41_), NULL));
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
	target = vala_ccode_identifier_new ("interface_name");
	_tmp42_ = NULL;
	_tmp43_ = NULL;
	expr = (_tmp43_ = vala_dbus_module_read_expression ((ValaDBusModule*) self, prefragment, ((ValaCCodeBaseModule*) self)->string_type, (ValaCCodeExpression*) (_tmp42_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) target), (_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL)), _tmp43_);
	_tmp45_ = NULL;
	_tmp44_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp45_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp44_ = vala_ccode_assignment_new ((ValaCCodeExpression*) target, expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
	(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
	_tmp46_ = NULL;
	_cdecl_ = (_tmp46_ = vala_ccode_declaration_new ("char*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp46_);
	_tmp47_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp47_ = vala_ccode_variable_declarator_new ("property_name", NULL, NULL)));
	(_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL));
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
	_tmp48_ = NULL;
	target = (_tmp48_ = vala_ccode_identifier_new ("property_name"), (target == NULL) ? NULL : (target = (vala_ccode_node_unref (target), NULL)), _tmp48_);
	_tmp50_ = NULL;
	_tmp49_ = NULL;
	expr = (_tmp50_ = vala_dbus_module_read_expression ((ValaDBusModule*) self, prefragment, ((ValaCCodeBaseModule*) self)->string_type, (ValaCCodeExpression*) (_tmp49_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) target), (expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL)), _tmp50_);
	(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
	_tmp52_ = NULL;
	_tmp51_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp52_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp51_ = vala_ccode_assignment_new ((ValaCCodeExpression*) target, expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_ccode_node_unref (_tmp52_), NULL));
	(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
	clastif = NULL;
	{
		GeeList* _tmp53_;
		GeeIterator* _tmp54_;
		GeeIterator* _prop_it;
		_tmp53_ = NULL;
		_tmp54_ = NULL;
		_prop_it = (_tmp54_ = gee_iterable_iterator ((GeeIterable*) (_tmp53_ = vala_object_type_symbol_get_properties (sym))), (_tmp53_ == NULL) ? NULL : (_tmp53_ = (gee_collection_object_unref (_tmp53_), NULL)), _tmp54_);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			gboolean _tmp55_;
			gboolean _tmp56_;
			ValaCCodeBlock* prop_block;
			ValaCCodeFragment* postfragment;
			ValaCCodeIdentifier* _tmp57_;
			ValaCCodeFunctionCall* _tmp58_;
			ValaCCodeFunctionCall* ccmp;
			ValaCCodeIdentifier* _tmp59_;
			ValaCCodeConstant* _tmp61_;
			char* _tmp60_;
			ValaCCodeConstant* _tmp62_;
			ValaCCodeBinaryExpression* _tmp63_;
			ValaCCodeBinaryExpression* ccheck1;
			ValaCCodeFunctionCall* _tmp65_;
			ValaCCodeIdentifier* _tmp64_;
			ValaCCodeIdentifier* _tmp66_;
			ValaCCodeConstant* _tmp69_;
			char* _tmp68_;
			char* _tmp67_;
			ValaCCodeConstant* _tmp70_;
			ValaCCodeBinaryExpression* _tmp71_;
			ValaCCodeBinaryExpression* ccheck2;
			ValaCCodeBinaryExpression* ccheck;
			ValaCCodeFunctionCall* _tmp73_;
			ValaCCodeIdentifier* _tmp72_;
			ValaCCodeUnaryExpression* _tmp75_;
			ValaCCodeIdentifier* _tmp74_;
			ValaCCodeIdentifier* _tmp76_;
			ValaCCodeConstant* _tmp79_;
			char* _tmp78_;
			char* _tmp77_;
			ValaCCodeUnaryExpression* _tmp81_;
			ValaCCodeIdentifier* _tmp80_;
			ValaCCodeExpressionStatement* _tmp82_;
			ValaCCodeIdentifier* _tmp84_;
			char* _tmp83_;
			ValaCCodeFunctionCall* _tmp85_;
			ValaCCodeFunctionCall* ccall;
			ValaCCodeIdentifier* _tmp86_;
			ValaCCodeIdentifier* _tmp87_;
			ValaCCodeFunctionCall* _tmp89_;
			ValaCCodeIdentifier* _tmp88_;
			ValaCCodeUnaryExpression* _tmp91_;
			ValaCCodeIdentifier* _tmp90_;
			ValaCCodeUnaryExpression* _tmp93_;
			ValaCCodeIdentifier* _tmp92_;
			ValaCCodeExpressionStatement* _tmp94_;
			ValaCCodeIfStatement* cif;
			ValaCCodeIfStatement* _tmp96_;
			ValaCCodeIfStatement* _tmp95_;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			_tmp55_ = FALSE;
			_tmp56_ = FALSE;
			if (vala_property_get_binding (prop) != MEMBER_BINDING_INSTANCE) {
				_tmp56_ = TRUE;
			} else {
				_tmp56_ = vala_property_get_overrides (prop);
			}
			if (_tmp56_) {
				_tmp55_ = TRUE;
			} else {
				_tmp55_ = vala_symbol_get_access ((ValaSymbol*) prop) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
			}
			if (_tmp55_) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) prop)) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (vala_property_get_get_accessor (prop) == NULL) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			prop_block = vala_ccode_block_new ();
			postfragment = vala_ccode_fragment_new ();
			vala_ccode_block_add_statement (prop_block, (ValaCCodeNode*) postfragment);
			_tmp57_ = NULL;
			_tmp58_ = NULL;
			ccmp = (_tmp58_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp57_ = vala_ccode_identifier_new ("strcmp"))), (_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_ccode_node_unref (_tmp57_), NULL)), _tmp58_);
			_tmp59_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp59_ = vala_ccode_identifier_new ("interface_name")));
			(_tmp59_ == NULL) ? NULL : (_tmp59_ = (vala_ccode_node_unref (_tmp59_), NULL));
			_tmp61_ = NULL;
			_tmp60_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp61_ = vala_ccode_constant_new (_tmp60_ = g_strdup_printf ("\"%s\"", dbus_iface_name))));
			(_tmp61_ == NULL) ? NULL : (_tmp61_ = (vala_ccode_node_unref (_tmp61_), NULL));
			_tmp60_ = (g_free (_tmp60_), NULL);
			_tmp62_ = NULL;
			_tmp63_ = NULL;
			ccheck1 = (_tmp63_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) ccmp, (ValaCCodeExpression*) (_tmp62_ = vala_ccode_constant_new ("0"))), (_tmp62_ == NULL) ? NULL : (_tmp62_ = (vala_ccode_node_unref (_tmp62_), NULL)), _tmp63_);
			_tmp65_ = NULL;
			_tmp64_ = NULL;
			ccmp = (_tmp65_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp64_ = vala_ccode_identifier_new ("strcmp"))), (ccmp == NULL) ? NULL : (ccmp = (vala_ccode_node_unref (ccmp), NULL)), _tmp65_);
			(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_ccode_node_unref (_tmp64_), NULL));
			_tmp66_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp66_ = vala_ccode_identifier_new ("property_name")));
			(_tmp66_ == NULL) ? NULL : (_tmp66_ = (vala_ccode_node_unref (_tmp66_), NULL));
			_tmp69_ = NULL;
			_tmp68_ = NULL;
			_tmp67_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp69_ = vala_ccode_constant_new (_tmp68_ = g_strdup_printf ("\"%s\"", _tmp67_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) prop))))));
			(_tmp69_ == NULL) ? NULL : (_tmp69_ = (vala_ccode_node_unref (_tmp69_), NULL));
			_tmp68_ = (g_free (_tmp68_), NULL);
			_tmp67_ = (g_free (_tmp67_), NULL);
			_tmp70_ = NULL;
			_tmp71_ = NULL;
			ccheck2 = (_tmp71_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) ccmp, (ValaCCodeExpression*) (_tmp70_ = vala_ccode_constant_new ("0"))), (_tmp70_ == NULL) ? NULL : (_tmp70_ = (vala_ccode_node_unref (_tmp70_), NULL)), _tmp71_);
			ccheck = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_AND, (ValaCCodeExpression*) ccheck1, (ValaCCodeExpression*) ccheck2);
			_tmp73_ = NULL;
			_tmp72_ = NULL;
			iter_call = (_tmp73_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp72_ = vala_ccode_identifier_new ("dbus_message_iter_open_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp73_);
			(_tmp72_ == NULL) ? NULL : (_tmp72_ = (vala_ccode_node_unref (_tmp72_), NULL));
			_tmp75_ = NULL;
			_tmp74_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp75_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp74_ = vala_ccode_identifier_new ("reply_iter")))));
			(_tmp75_ == NULL) ? NULL : (_tmp75_ = (vala_ccode_node_unref (_tmp75_), NULL));
			(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_ccode_node_unref (_tmp74_), NULL));
			_tmp76_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp76_ = vala_ccode_identifier_new ("DBUS_TYPE_VARIANT")));
			(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_ccode_node_unref (_tmp76_), NULL));
			_tmp79_ = NULL;
			_tmp78_ = NULL;
			_tmp77_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp79_ = vala_ccode_constant_new (_tmp78_ = g_strdup_printf ("\"%s\"", _tmp77_ = vala_data_type_get_type_signature (vala_property_get_property_type (prop))))));
			(_tmp79_ == NULL) ? NULL : (_tmp79_ = (vala_ccode_node_unref (_tmp79_), NULL));
			_tmp78_ = (g_free (_tmp78_), NULL);
			_tmp77_ = (g_free (_tmp77_), NULL);
			_tmp81_ = NULL;
			_tmp80_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp81_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp80_ = vala_ccode_identifier_new ("subiter")))));
			(_tmp81_ == NULL) ? NULL : (_tmp81_ = (vala_ccode_node_unref (_tmp81_), NULL));
			(_tmp80_ == NULL) ? NULL : (_tmp80_ = (vala_ccode_node_unref (_tmp80_), NULL));
			_tmp82_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp82_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
			(_tmp82_ == NULL) ? NULL : (_tmp82_ = (vala_ccode_node_unref (_tmp82_), NULL));
			_tmp84_ = NULL;
			_tmp83_ = NULL;
			_tmp85_ = NULL;
			ccall = (_tmp85_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new (_tmp83_ = vala_property_accessor_get_cname (vala_property_get_get_accessor (prop))))), (_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL)), _tmp83_ = (g_free (_tmp83_), NULL), _tmp85_);
			_tmp86_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp86_ = vala_ccode_identifier_new ("self")));
			(_tmp86_ == NULL) ? NULL : (_tmp86_ = (vala_ccode_node_unref (_tmp86_), NULL));
			_tmp87_ = NULL;
			vala_dbus_module_write_expression ((ValaDBusModule*) self, postfragment, vala_property_get_property_type (prop), (ValaCCodeExpression*) (_tmp87_ = vala_ccode_identifier_new ("subiter")), (ValaCCodeExpression*) ccall);
			(_tmp87_ == NULL) ? NULL : (_tmp87_ = (vala_ccode_node_unref (_tmp87_), NULL));
			_tmp89_ = NULL;
			_tmp88_ = NULL;
			iter_call = (_tmp89_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp88_ = vala_ccode_identifier_new ("dbus_message_iter_close_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp89_);
			(_tmp88_ == NULL) ? NULL : (_tmp88_ = (vala_ccode_node_unref (_tmp88_), NULL));
			_tmp91_ = NULL;
			_tmp90_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp91_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp90_ = vala_ccode_identifier_new ("reply_iter")))));
			(_tmp91_ == NULL) ? NULL : (_tmp91_ = (vala_ccode_node_unref (_tmp91_), NULL));
			(_tmp90_ == NULL) ? NULL : (_tmp90_ = (vala_ccode_node_unref (_tmp90_), NULL));
			_tmp93_ = NULL;
			_tmp92_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp93_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp92_ = vala_ccode_identifier_new ("subiter")))));
			(_tmp93_ == NULL) ? NULL : (_tmp93_ = (vala_ccode_node_unref (_tmp93_), NULL));
			(_tmp92_ == NULL) ? NULL : (_tmp92_ = (vala_ccode_node_unref (_tmp92_), NULL));
			_tmp94_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp94_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
			(_tmp94_ == NULL) ? NULL : (_tmp94_ = (vala_ccode_node_unref (_tmp94_), NULL));
			cif = vala_ccode_if_statement_new ((ValaCCodeExpression*) ccheck, (ValaCCodeStatement*) prop_block, NULL);
			if (clastif == NULL) {
				vala_ccode_block_add_statement (block, (ValaCCodeNode*) cif);
			} else {
				vala_ccode_if_statement_set_false_statement (clastif, (ValaCCodeStatement*) cif);
			}
			_tmp96_ = NULL;
			_tmp95_ = NULL;
			clastif = (_tmp96_ = (_tmp95_ = cif, (_tmp95_ == NULL) ? NULL : vala_ccode_node_ref (_tmp95_)), (clastif == NULL) ? NULL : (clastif = (vala_ccode_node_unref (clastif), NULL)), _tmp96_);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			(prop_block == NULL) ? NULL : (prop_block = (vala_ccode_node_unref (prop_block), NULL));
			(postfragment == NULL) ? NULL : (postfragment = (vala_ccode_node_unref (postfragment), NULL));
			(ccmp == NULL) ? NULL : (ccmp = (vala_ccode_node_unref (ccmp), NULL));
			(ccheck1 == NULL) ? NULL : (ccheck1 = (vala_ccode_node_unref (ccheck1), NULL));
			(ccheck2 == NULL) ? NULL : (ccheck2 = (vala_ccode_node_unref (ccheck2), NULL));
			(ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL));
			(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
			(cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	if (clastif == NULL) {
		ValaCCodeBlock* _tmp97_;
		ValaCCodeReturnStatement* _tmp99_;
		ValaCCodeConstant* _tmp98_;
		_tmp97_ = NULL;
		block = (_tmp97_ = vala_ccode_block_new (), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), _tmp97_);
		_tmp99_ = NULL;
		_tmp98_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp99_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp98_ = vala_ccode_constant_new ("NULL")))));
		(_tmp99_ == NULL) ? NULL : (_tmp99_ = (vala_ccode_node_unref (_tmp99_), NULL));
		(_tmp98_ == NULL) ? NULL : (_tmp98_ = (vala_ccode_node_unref (_tmp98_), NULL));
	} else {
		ValaCCodeBlock* else_block;
		ValaCCodeReturnStatement* _tmp101_;
		ValaCCodeConstant* _tmp100_;
		ValaCCodeReturnStatement* _tmp103_;
		ValaCCodeIdentifier* _tmp102_;
		else_block = vala_ccode_block_new ();
		_tmp101_ = NULL;
		_tmp100_ = NULL;
		vala_ccode_block_add_statement (else_block, (ValaCCodeNode*) (_tmp101_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp100_ = vala_ccode_constant_new ("NULL")))));
		(_tmp101_ == NULL) ? NULL : (_tmp101_ = (vala_ccode_node_unref (_tmp101_), NULL));
		(_tmp100_ == NULL) ? NULL : (_tmp100_ = (vala_ccode_node_unref (_tmp100_), NULL));
		vala_ccode_if_statement_set_false_statement (clastif, (ValaCCodeStatement*) else_block);
		_tmp103_ = NULL;
		_tmp102_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp103_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp102_ = vala_ccode_identifier_new ("reply")))));
		(_tmp103_ == NULL) ? NULL : (_tmp103_ = (vala_ccode_node_unref (_tmp103_), NULL));
		(_tmp102_ == NULL) ? NULL : (_tmp102_ = (vala_ccode_node_unref (_tmp102_), NULL));
		(else_block == NULL) ? NULL : (else_block = (vala_ccode_node_unref (else_block), NULL));
	}
	_tmp104_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp104_ = vala_ccode_function_copy (function)));
	(_tmp104_ == NULL) ? NULL : (_tmp104_ = (vala_ccode_node_unref (_tmp104_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp105_ = NULL;
	return (_tmp105_ = wrapper_name, (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), (prefragment == NULL) ? NULL : (prefragment = (vala_ccode_node_unref (prefragment), NULL)), (message_signature == NULL) ? NULL : (message_signature = (vala_ccode_node_unref (message_signature), NULL)), (signature_check == NULL) ? NULL : (signature_check = (vala_ccode_node_unref (signature_check), NULL)), (signature_error_block == NULL) ? NULL : (signature_error_block = (vala_ccode_node_unref (signature_error_block), NULL)), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), (msgcall == NULL) ? NULL : (msgcall = (vala_ccode_node_unref (msgcall), NULL)), (target == NULL) ? NULL : (target = (vala_ccode_node_unref (target), NULL)), (expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL)), (clastif == NULL) ? NULL : (clastif = (vala_ccode_node_unref (clastif), NULL)), _tmp105_);
}


static char* vala_dbus_server_module_generate_dbus_property_get_all_wrapper (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name) {
	char* _tmp0_;
	char* _tmp1_;
	char* wrapper_name;
	gboolean has_readable_properties;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp8_;
	char* _tmp7_;
	char* _tmp6_;
	ValaCCodeFormalParameter* _tmp9_;
	ValaCCodeFormalParameter* _tmp10_;
	ValaCCodeBlock* block;
	ValaCCodeFragment* prefragment;
	ValaCCodeDeclaration* _tmp11_;
	ValaCCodeVariableDeclarator* _tmp12_;
	ValaCCodeDeclaration* _tmp13_;
	ValaCCodeVariableDeclarator* _tmp14_;
	ValaCCodeVariableDeclarator* _tmp15_;
	ValaCCodeVariableDeclarator* _tmp16_;
	ValaCCodeIdentifier* _tmp19_;
	ValaCCodeFunctionCall* _tmp20_;
	ValaCCodeFunctionCall* message_signature;
	ValaCCodeIdentifier* _tmp21_;
	ValaCCodeIdentifier* _tmp22_;
	ValaCCodeFunctionCall* _tmp23_;
	ValaCCodeFunctionCall* signature_check;
	ValaCCodeConstant* _tmp24_;
	ValaCCodeBlock* signature_error_block;
	ValaCCodeReturnStatement* _tmp26_;
	ValaCCodeConstant* _tmp25_;
	ValaCCodeIfStatement* _tmp27_;
	ValaCCodeIdentifier* _tmp28_;
	ValaCCodeFunctionCall* _tmp29_;
	ValaCCodeFunctionCall* iter_call;
	ValaCCodeIdentifier* _tmp30_;
	ValaCCodeUnaryExpression* _tmp32_;
	ValaCCodeIdentifier* _tmp31_;
	ValaCCodeExpressionStatement* _tmp33_;
	ValaCCodeIdentifier* _tmp34_;
	ValaCCodeFunctionCall* _tmp35_;
	ValaCCodeFunctionCall* msgcall;
	ValaCCodeIdentifier* _tmp36_;
	ValaCCodeExpressionStatement* _tmp39_;
	ValaCCodeAssignment* _tmp38_;
	ValaCCodeIdentifier* _tmp37_;
	ValaCCodeFunctionCall* _tmp41_;
	ValaCCodeIdentifier* _tmp40_;
	ValaCCodeIdentifier* _tmp42_;
	ValaCCodeUnaryExpression* _tmp44_;
	ValaCCodeIdentifier* _tmp43_;
	ValaCCodeExpressionStatement* _tmp45_;
	ValaCCodeDeclaration* _tmp46_;
	ValaCCodeVariableDeclarator* _tmp47_;
	ValaCCodeIdentifier* target;
	ValaCCodeIdentifier* _tmp48_;
	ValaCCodeExpression* _tmp49_;
	ValaCCodeExpression* expr;
	ValaCCodeExpressionStatement* _tmp51_;
	ValaCCodeAssignment* _tmp50_;
	ValaCCodeBlock* prop_block;
	ValaCCodeFunctionCall* _tmp55_;
	ValaCCodeIdentifier* _tmp54_;
	ValaCCodeUnaryExpression* _tmp57_;
	ValaCCodeIdentifier* _tmp56_;
	ValaCCodeIdentifier* _tmp58_;
	ValaCCodeConstant* _tmp59_;
	ValaCCodeUnaryExpression* _tmp61_;
	ValaCCodeIdentifier* _tmp60_;
	ValaCCodeExpressionStatement* _tmp62_;
	ValaCCodeFunctionCall* _tmp121_;
	ValaCCodeIdentifier* _tmp120_;
	ValaCCodeUnaryExpression* _tmp123_;
	ValaCCodeIdentifier* _tmp122_;
	ValaCCodeUnaryExpression* _tmp125_;
	ValaCCodeIdentifier* _tmp124_;
	ValaCCodeExpressionStatement* _tmp126_;
	ValaCCodeIdentifier* _tmp127_;
	ValaCCodeFunctionCall* _tmp128_;
	ValaCCodeFunctionCall* ccmp;
	ValaCCodeIdentifier* _tmp129_;
	ValaCCodeConstant* _tmp131_;
	char* _tmp130_;
	ValaCCodeConstant* _tmp132_;
	ValaCCodeBinaryExpression* _tmp133_;
	ValaCCodeBinaryExpression* ccheck;
	ValaCCodeBlock* else_block;
	ValaCCodeReturnStatement* _tmp135_;
	ValaCCodeConstant* _tmp134_;
	ValaCCodeIfStatement* _tmp136_;
	ValaCCodeReturnStatement* _tmp138_;
	ValaCCodeIdentifier* _tmp137_;
	ValaCCodeFunction* _tmp139_;
	char* _tmp140_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	g_return_val_if_fail (dbus_iface_name != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	wrapper_name = (_tmp1_ = g_strdup_printf ("_dbus_%s_property_get_all", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) sym, NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	has_readable_properties = FALSE;
	{
		GeeList* _tmp2_;
		GeeIterator* _tmp3_;
		GeeIterator* _prop_it;
		_tmp2_ = NULL;
		_tmp3_ = NULL;
		_prop_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = vala_object_type_symbol_get_properties (sym))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp3_);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			gboolean _tmp4_;
			gboolean _tmp5_;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			_tmp4_ = FALSE;
			_tmp5_ = FALSE;
			if (vala_property_get_binding (prop) != MEMBER_BINDING_INSTANCE) {
				_tmp5_ = TRUE;
			} else {
				_tmp5_ = vala_property_get_overrides (prop);
			}
			if (_tmp5_) {
				_tmp4_ = TRUE;
			} else {
				_tmp4_ = vala_symbol_get_access ((ValaSymbol*) prop) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
			}
			if (_tmp4_) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) prop)) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (vala_property_get_get_accessor (prop) != NULL) {
				has_readable_properties = TRUE;
			}
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	_cdecl_ = NULL;
	function = vala_ccode_function_new (wrapper_name, "DBusMessage*");
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp8_ = NULL;
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp8_ = vala_ccode_formal_parameter_new ("self", _tmp7_ = g_strconcat (_tmp6_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) sym, FALSE), "*", NULL)));
	(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
	_tmp7_ = (g_free (_tmp7_), NULL);
	_tmp6_ = (g_free (_tmp6_), NULL);
	_tmp9_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp9_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	_tmp10_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp10_ = vala_ccode_formal_parameter_new ("message", "DBusMessage*"));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	block = vala_ccode_block_new ();
	prefragment = vala_ccode_fragment_new ();
	_tmp11_ = NULL;
	_cdecl_ = (_tmp11_ = vala_ccode_declaration_new ("DBusMessage*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp11_);
	_tmp12_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp12_ = vala_ccode_variable_declarator_new ("reply", NULL, NULL)));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp13_ = NULL;
	_cdecl_ = (_tmp13_ = vala_ccode_declaration_new ("DBusMessageIter"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp13_);
	_tmp14_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp14_ = vala_ccode_variable_declarator_new ("iter", NULL, NULL)));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	_tmp15_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp15_ = vala_ccode_variable_declarator_new ("reply_iter", NULL, NULL)));
	(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
	_tmp16_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp16_ = vala_ccode_variable_declarator_new ("subiter", NULL, NULL)));
	(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
	if (has_readable_properties) {
		ValaCCodeVariableDeclarator* _tmp17_;
		ValaCCodeVariableDeclarator* _tmp18_;
		_tmp17_ = NULL;
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp17_ = vala_ccode_variable_declarator_new ("entry_iter", NULL, NULL)));
		(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
		_tmp18_ = NULL;
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp18_ = vala_ccode_variable_declarator_new ("value_iter", NULL, NULL)));
		(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	}
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp19_ = NULL;
	_tmp20_ = NULL;
	message_signature = (_tmp20_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp19_ = vala_ccode_identifier_new ("dbus_message_get_signature"))), (_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL)), _tmp20_);
	_tmp21_ = NULL;
	vala_ccode_function_call_add_argument (message_signature, (ValaCCodeExpression*) (_tmp21_ = vala_ccode_identifier_new ("message")));
	(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
	_tmp22_ = NULL;
	_tmp23_ = NULL;
	signature_check = (_tmp23_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("strcmp"))), (_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL)), _tmp23_);
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) message_signature);
	_tmp24_ = NULL;
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) (_tmp24_ = vala_ccode_constant_new ("\"s\"")));
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	signature_error_block = vala_ccode_block_new ();
	_tmp26_ = NULL;
	_tmp25_ = NULL;
	vala_ccode_block_add_statement (signature_error_block, (ValaCCodeNode*) (_tmp26_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp25_ = vala_ccode_constant_new ("NULL")))));
	(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
	(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
	_tmp27_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp27_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) signature_check, (ValaCCodeStatement*) signature_error_block, NULL)));
	(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
	_tmp28_ = NULL;
	_tmp29_ = NULL;
	iter_call = (_tmp29_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("dbus_message_iter_init"))), (_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL)), _tmp29_);
	_tmp30_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new ("message")));
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
	_tmp32_ = NULL;
	_tmp31_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp32_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp31_ = vala_ccode_identifier_new ("iter")))));
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
	_tmp33_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp33_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
	_tmp34_ = NULL;
	_tmp35_ = NULL;
	msgcall = (_tmp35_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp34_ = vala_ccode_identifier_new ("dbus_message_new_method_return"))), (_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL)), _tmp35_);
	_tmp36_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp36_ = vala_ccode_identifier_new ("message")));
	(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
	_tmp39_ = NULL;
	_tmp38_ = NULL;
	_tmp37_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp39_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp38_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp37_ = vala_ccode_identifier_new ("reply")), (ValaCCodeExpression*) msgcall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
	(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
	(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
	_tmp41_ = NULL;
	_tmp40_ = NULL;
	iter_call = (_tmp41_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp40_ = vala_ccode_identifier_new ("dbus_message_iter_init_append"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp41_);
	(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_ccode_node_unref (_tmp40_), NULL));
	_tmp42_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp42_ = vala_ccode_identifier_new ("reply")));
	(_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL));
	_tmp44_ = NULL;
	_tmp43_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp44_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp43_ = vala_ccode_identifier_new ("reply_iter")))));
	(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
	(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL));
	_tmp45_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp45_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) prefragment);
	_tmp46_ = NULL;
	_cdecl_ = (_tmp46_ = vala_ccode_declaration_new ("char*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp46_);
	_tmp47_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp47_ = vala_ccode_variable_declarator_new ("interface_name", NULL, NULL)));
	(_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL));
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
	target = vala_ccode_identifier_new ("interface_name");
	_tmp48_ = NULL;
	_tmp49_ = NULL;
	expr = (_tmp49_ = vala_dbus_module_read_expression ((ValaDBusModule*) self, prefragment, ((ValaCCodeBaseModule*) self)->string_type, (ValaCCodeExpression*) (_tmp48_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) target), (_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL)), _tmp49_);
	_tmp51_ = NULL;
	_tmp50_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp51_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp50_ = vala_ccode_assignment_new ((ValaCCodeExpression*) target, expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
	(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
	if (has_readable_properties) {
		ValaCCodeDeclaration* _tmp52_;
		ValaCCodeVariableDeclarator* _tmp53_;
		_tmp52_ = NULL;
		_cdecl_ = (_tmp52_ = vala_ccode_declaration_new ("const char*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp52_);
		_tmp53_ = NULL;
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp53_ = vala_ccode_variable_declarator_new ("property_name", NULL, NULL)));
		(_tmp53_ == NULL) ? NULL : (_tmp53_ = (vala_ccode_node_unref (_tmp53_), NULL));
		vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
	}
	prop_block = vala_ccode_block_new ();
	_tmp55_ = NULL;
	_tmp54_ = NULL;
	iter_call = (_tmp55_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp54_ = vala_ccode_identifier_new ("dbus_message_iter_open_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp55_);
	(_tmp54_ == NULL) ? NULL : (_tmp54_ = (vala_ccode_node_unref (_tmp54_), NULL));
	_tmp57_ = NULL;
	_tmp56_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp57_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp56_ = vala_ccode_identifier_new ("reply_iter")))));
	(_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_ccode_node_unref (_tmp57_), NULL));
	(_tmp56_ == NULL) ? NULL : (_tmp56_ = (vala_ccode_node_unref (_tmp56_), NULL));
	_tmp58_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp58_ = vala_ccode_identifier_new ("DBUS_TYPE_ARRAY")));
	(_tmp58_ == NULL) ? NULL : (_tmp58_ = (vala_ccode_node_unref (_tmp58_), NULL));
	_tmp59_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp59_ = vala_ccode_constant_new ("\"{sv}\"")));
	(_tmp59_ == NULL) ? NULL : (_tmp59_ = (vala_ccode_node_unref (_tmp59_), NULL));
	_tmp61_ = NULL;
	_tmp60_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp61_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp60_ = vala_ccode_identifier_new ("subiter")))));
	(_tmp61_ == NULL) ? NULL : (_tmp61_ = (vala_ccode_node_unref (_tmp61_), NULL));
	(_tmp60_ == NULL) ? NULL : (_tmp60_ = (vala_ccode_node_unref (_tmp60_), NULL));
	_tmp62_ = NULL;
	vala_ccode_block_add_statement (prop_block, (ValaCCodeNode*) (_tmp62_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp62_ == NULL) ? NULL : (_tmp62_ = (vala_ccode_node_unref (_tmp62_), NULL));
	{
		GeeList* _tmp63_;
		GeeIterator* _tmp64_;
		GeeIterator* _prop_it;
		_tmp63_ = NULL;
		_tmp64_ = NULL;
		_prop_it = (_tmp64_ = gee_iterable_iterator ((GeeIterable*) (_tmp63_ = vala_object_type_symbol_get_properties (sym))), (_tmp63_ == NULL) ? NULL : (_tmp63_ = (gee_collection_object_unref (_tmp63_), NULL)), _tmp64_);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			gboolean _tmp65_;
			gboolean _tmp66_;
			ValaCCodeFragment* postfragment;
			ValaCCodeFunctionCall* _tmp68_;
			ValaCCodeIdentifier* _tmp67_;
			ValaCCodeUnaryExpression* _tmp70_;
			ValaCCodeIdentifier* _tmp69_;
			ValaCCodeIdentifier* _tmp71_;
			ValaCCodeConstant* _tmp72_;
			ValaCCodeUnaryExpression* _tmp74_;
			ValaCCodeIdentifier* _tmp73_;
			ValaCCodeExpressionStatement* _tmp75_;
			ValaCCodeExpressionStatement* _tmp81_;
			ValaCCodeAssignment* _tmp80_;
			ValaCCodeConstant* _tmp79_;
			char* _tmp78_;
			char* _tmp77_;
			ValaCCodeIdentifier* _tmp76_;
			ValaCCodeFunctionCall* _tmp83_;
			ValaCCodeIdentifier* _tmp82_;
			ValaCCodeUnaryExpression* _tmp85_;
			ValaCCodeIdentifier* _tmp84_;
			ValaCCodeIdentifier* _tmp86_;
			ValaCCodeUnaryExpression* _tmp88_;
			ValaCCodeIdentifier* _tmp87_;
			ValaCCodeExpressionStatement* _tmp89_;
			ValaCCodeFunctionCall* _tmp91_;
			ValaCCodeIdentifier* _tmp90_;
			ValaCCodeUnaryExpression* _tmp93_;
			ValaCCodeIdentifier* _tmp92_;
			ValaCCodeIdentifier* _tmp94_;
			ValaCCodeConstant* _tmp97_;
			char* _tmp96_;
			char* _tmp95_;
			ValaCCodeUnaryExpression* _tmp99_;
			ValaCCodeIdentifier* _tmp98_;
			ValaCCodeExpressionStatement* _tmp100_;
			ValaCCodeIdentifier* _tmp102_;
			char* _tmp101_;
			ValaCCodeFunctionCall* _tmp103_;
			ValaCCodeFunctionCall* ccall;
			ValaCCodeIdentifier* _tmp104_;
			ValaCCodeIdentifier* _tmp105_;
			ValaCCodeFunctionCall* _tmp107_;
			ValaCCodeIdentifier* _tmp106_;
			ValaCCodeUnaryExpression* _tmp109_;
			ValaCCodeIdentifier* _tmp108_;
			ValaCCodeUnaryExpression* _tmp111_;
			ValaCCodeIdentifier* _tmp110_;
			ValaCCodeExpressionStatement* _tmp112_;
			ValaCCodeFunctionCall* _tmp114_;
			ValaCCodeIdentifier* _tmp113_;
			ValaCCodeUnaryExpression* _tmp116_;
			ValaCCodeIdentifier* _tmp115_;
			ValaCCodeUnaryExpression* _tmp118_;
			ValaCCodeIdentifier* _tmp117_;
			ValaCCodeExpressionStatement* _tmp119_;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			_tmp65_ = FALSE;
			_tmp66_ = FALSE;
			if (vala_property_get_binding (prop) != MEMBER_BINDING_INSTANCE) {
				_tmp66_ = TRUE;
			} else {
				_tmp66_ = vala_property_get_overrides (prop);
			}
			if (_tmp66_) {
				_tmp65_ = TRUE;
			} else {
				_tmp65_ = vala_symbol_get_access ((ValaSymbol*) prop) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
			}
			if (_tmp65_) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) prop)) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (vala_property_get_get_accessor (prop) == NULL) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			postfragment = vala_ccode_fragment_new ();
			vala_ccode_block_add_statement (prop_block, (ValaCCodeNode*) postfragment);
			_tmp68_ = NULL;
			_tmp67_ = NULL;
			iter_call = (_tmp68_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp67_ = vala_ccode_identifier_new ("dbus_message_iter_open_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp68_);
			(_tmp67_ == NULL) ? NULL : (_tmp67_ = (vala_ccode_node_unref (_tmp67_), NULL));
			_tmp70_ = NULL;
			_tmp69_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp70_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp69_ = vala_ccode_identifier_new ("subiter")))));
			(_tmp70_ == NULL) ? NULL : (_tmp70_ = (vala_ccode_node_unref (_tmp70_), NULL));
			(_tmp69_ == NULL) ? NULL : (_tmp69_ = (vala_ccode_node_unref (_tmp69_), NULL));
			_tmp71_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp71_ = vala_ccode_identifier_new ("DBUS_TYPE_DICT_ENTRY")));
			(_tmp71_ == NULL) ? NULL : (_tmp71_ = (vala_ccode_node_unref (_tmp71_), NULL));
			_tmp72_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp72_ = vala_ccode_constant_new ("NULL")));
			(_tmp72_ == NULL) ? NULL : (_tmp72_ = (vala_ccode_node_unref (_tmp72_), NULL));
			_tmp74_ = NULL;
			_tmp73_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp74_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp73_ = vala_ccode_identifier_new ("entry_iter")))));
			(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_ccode_node_unref (_tmp74_), NULL));
			(_tmp73_ == NULL) ? NULL : (_tmp73_ = (vala_ccode_node_unref (_tmp73_), NULL));
			_tmp75_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp75_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
			(_tmp75_ == NULL) ? NULL : (_tmp75_ = (vala_ccode_node_unref (_tmp75_), NULL));
			_tmp81_ = NULL;
			_tmp80_ = NULL;
			_tmp79_ = NULL;
			_tmp78_ = NULL;
			_tmp77_ = NULL;
			_tmp76_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp81_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp80_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp76_ = vala_ccode_identifier_new ("property_name")), (ValaCCodeExpression*) (_tmp79_ = vala_ccode_constant_new (_tmp78_ = g_strdup_printf ("\"%s\"", _tmp77_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) prop))))), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
			(_tmp81_ == NULL) ? NULL : (_tmp81_ = (vala_ccode_node_unref (_tmp81_), NULL));
			(_tmp80_ == NULL) ? NULL : (_tmp80_ = (vala_ccode_node_unref (_tmp80_), NULL));
			(_tmp79_ == NULL) ? NULL : (_tmp79_ = (vala_ccode_node_unref (_tmp79_), NULL));
			_tmp78_ = (g_free (_tmp78_), NULL);
			_tmp77_ = (g_free (_tmp77_), NULL);
			(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_ccode_node_unref (_tmp76_), NULL));
			_tmp83_ = NULL;
			_tmp82_ = NULL;
			iter_call = (_tmp83_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp82_ = vala_ccode_identifier_new ("dbus_message_iter_append_basic"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp83_);
			(_tmp82_ == NULL) ? NULL : (_tmp82_ = (vala_ccode_node_unref (_tmp82_), NULL));
			_tmp85_ = NULL;
			_tmp84_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp85_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new ("entry_iter")))));
			(_tmp85_ == NULL) ? NULL : (_tmp85_ = (vala_ccode_node_unref (_tmp85_), NULL));
			(_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL));
			_tmp86_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp86_ = vala_ccode_identifier_new ("DBUS_TYPE_STRING")));
			(_tmp86_ == NULL) ? NULL : (_tmp86_ = (vala_ccode_node_unref (_tmp86_), NULL));
			_tmp88_ = NULL;
			_tmp87_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp88_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp87_ = vala_ccode_identifier_new ("property_name")))));
			(_tmp88_ == NULL) ? NULL : (_tmp88_ = (vala_ccode_node_unref (_tmp88_), NULL));
			(_tmp87_ == NULL) ? NULL : (_tmp87_ = (vala_ccode_node_unref (_tmp87_), NULL));
			_tmp89_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp89_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
			(_tmp89_ == NULL) ? NULL : (_tmp89_ = (vala_ccode_node_unref (_tmp89_), NULL));
			_tmp91_ = NULL;
			_tmp90_ = NULL;
			iter_call = (_tmp91_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp90_ = vala_ccode_identifier_new ("dbus_message_iter_open_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp91_);
			(_tmp90_ == NULL) ? NULL : (_tmp90_ = (vala_ccode_node_unref (_tmp90_), NULL));
			_tmp93_ = NULL;
			_tmp92_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp93_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp92_ = vala_ccode_identifier_new ("entry_iter")))));
			(_tmp93_ == NULL) ? NULL : (_tmp93_ = (vala_ccode_node_unref (_tmp93_), NULL));
			(_tmp92_ == NULL) ? NULL : (_tmp92_ = (vala_ccode_node_unref (_tmp92_), NULL));
			_tmp94_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp94_ = vala_ccode_identifier_new ("DBUS_TYPE_VARIANT")));
			(_tmp94_ == NULL) ? NULL : (_tmp94_ = (vala_ccode_node_unref (_tmp94_), NULL));
			_tmp97_ = NULL;
			_tmp96_ = NULL;
			_tmp95_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp97_ = vala_ccode_constant_new (_tmp96_ = g_strdup_printf ("\"%s\"", _tmp95_ = vala_data_type_get_type_signature (vala_property_get_property_type (prop))))));
			(_tmp97_ == NULL) ? NULL : (_tmp97_ = (vala_ccode_node_unref (_tmp97_), NULL));
			_tmp96_ = (g_free (_tmp96_), NULL);
			_tmp95_ = (g_free (_tmp95_), NULL);
			_tmp99_ = NULL;
			_tmp98_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp99_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp98_ = vala_ccode_identifier_new ("value_iter")))));
			(_tmp99_ == NULL) ? NULL : (_tmp99_ = (vala_ccode_node_unref (_tmp99_), NULL));
			(_tmp98_ == NULL) ? NULL : (_tmp98_ = (vala_ccode_node_unref (_tmp98_), NULL));
			_tmp100_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp100_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
			(_tmp100_ == NULL) ? NULL : (_tmp100_ = (vala_ccode_node_unref (_tmp100_), NULL));
			_tmp102_ = NULL;
			_tmp101_ = NULL;
			_tmp103_ = NULL;
			ccall = (_tmp103_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp102_ = vala_ccode_identifier_new (_tmp101_ = vala_property_accessor_get_cname (vala_property_get_get_accessor (prop))))), (_tmp102_ == NULL) ? NULL : (_tmp102_ = (vala_ccode_node_unref (_tmp102_), NULL)), _tmp101_ = (g_free (_tmp101_), NULL), _tmp103_);
			_tmp104_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp104_ = vala_ccode_identifier_new ("self")));
			(_tmp104_ == NULL) ? NULL : (_tmp104_ = (vala_ccode_node_unref (_tmp104_), NULL));
			_tmp105_ = NULL;
			vala_dbus_module_write_expression ((ValaDBusModule*) self, postfragment, vala_property_get_property_type (prop), (ValaCCodeExpression*) (_tmp105_ = vala_ccode_identifier_new ("value_iter")), (ValaCCodeExpression*) ccall);
			(_tmp105_ == NULL) ? NULL : (_tmp105_ = (vala_ccode_node_unref (_tmp105_), NULL));
			_tmp107_ = NULL;
			_tmp106_ = NULL;
			iter_call = (_tmp107_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp106_ = vala_ccode_identifier_new ("dbus_message_iter_close_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp107_);
			(_tmp106_ == NULL) ? NULL : (_tmp106_ = (vala_ccode_node_unref (_tmp106_), NULL));
			_tmp109_ = NULL;
			_tmp108_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp109_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp108_ = vala_ccode_identifier_new ("entry_iter")))));
			(_tmp109_ == NULL) ? NULL : (_tmp109_ = (vala_ccode_node_unref (_tmp109_), NULL));
			(_tmp108_ == NULL) ? NULL : (_tmp108_ = (vala_ccode_node_unref (_tmp108_), NULL));
			_tmp111_ = NULL;
			_tmp110_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp111_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp110_ = vala_ccode_identifier_new ("value_iter")))));
			(_tmp111_ == NULL) ? NULL : (_tmp111_ = (vala_ccode_node_unref (_tmp111_), NULL));
			(_tmp110_ == NULL) ? NULL : (_tmp110_ = (vala_ccode_node_unref (_tmp110_), NULL));
			_tmp112_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp112_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
			(_tmp112_ == NULL) ? NULL : (_tmp112_ = (vala_ccode_node_unref (_tmp112_), NULL));
			_tmp114_ = NULL;
			_tmp113_ = NULL;
			iter_call = (_tmp114_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp113_ = vala_ccode_identifier_new ("dbus_message_iter_close_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp114_);
			(_tmp113_ == NULL) ? NULL : (_tmp113_ = (vala_ccode_node_unref (_tmp113_), NULL));
			_tmp116_ = NULL;
			_tmp115_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp116_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp115_ = vala_ccode_identifier_new ("subiter")))));
			(_tmp116_ == NULL) ? NULL : (_tmp116_ = (vala_ccode_node_unref (_tmp116_), NULL));
			(_tmp115_ == NULL) ? NULL : (_tmp115_ = (vala_ccode_node_unref (_tmp115_), NULL));
			_tmp118_ = NULL;
			_tmp117_ = NULL;
			vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp118_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp117_ = vala_ccode_identifier_new ("entry_iter")))));
			(_tmp118_ == NULL) ? NULL : (_tmp118_ = (vala_ccode_node_unref (_tmp118_), NULL));
			(_tmp117_ == NULL) ? NULL : (_tmp117_ = (vala_ccode_node_unref (_tmp117_), NULL));
			_tmp119_ = NULL;
			vala_ccode_fragment_append (postfragment, (ValaCCodeNode*) (_tmp119_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
			(_tmp119_ == NULL) ? NULL : (_tmp119_ = (vala_ccode_node_unref (_tmp119_), NULL));
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			(postfragment == NULL) ? NULL : (postfragment = (vala_ccode_node_unref (postfragment), NULL));
			(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	_tmp121_ = NULL;
	_tmp120_ = NULL;
	iter_call = (_tmp121_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp120_ = vala_ccode_identifier_new ("dbus_message_iter_close_container"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp121_);
	(_tmp120_ == NULL) ? NULL : (_tmp120_ = (vala_ccode_node_unref (_tmp120_), NULL));
	_tmp123_ = NULL;
	_tmp122_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp123_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp122_ = vala_ccode_identifier_new ("reply_iter")))));
	(_tmp123_ == NULL) ? NULL : (_tmp123_ = (vala_ccode_node_unref (_tmp123_), NULL));
	(_tmp122_ == NULL) ? NULL : (_tmp122_ = (vala_ccode_node_unref (_tmp122_), NULL));
	_tmp125_ = NULL;
	_tmp124_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp125_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp124_ = vala_ccode_identifier_new ("subiter")))));
	(_tmp125_ == NULL) ? NULL : (_tmp125_ = (vala_ccode_node_unref (_tmp125_), NULL));
	(_tmp124_ == NULL) ? NULL : (_tmp124_ = (vala_ccode_node_unref (_tmp124_), NULL));
	_tmp126_ = NULL;
	vala_ccode_block_add_statement (prop_block, (ValaCCodeNode*) (_tmp126_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp126_ == NULL) ? NULL : (_tmp126_ = (vala_ccode_node_unref (_tmp126_), NULL));
	_tmp127_ = NULL;
	_tmp128_ = NULL;
	ccmp = (_tmp128_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp127_ = vala_ccode_identifier_new ("strcmp"))), (_tmp127_ == NULL) ? NULL : (_tmp127_ = (vala_ccode_node_unref (_tmp127_), NULL)), _tmp128_);
	_tmp129_ = NULL;
	vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp129_ = vala_ccode_identifier_new ("interface_name")));
	(_tmp129_ == NULL) ? NULL : (_tmp129_ = (vala_ccode_node_unref (_tmp129_), NULL));
	_tmp131_ = NULL;
	_tmp130_ = NULL;
	vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp131_ = vala_ccode_constant_new (_tmp130_ = g_strdup_printf ("\"%s\"", dbus_iface_name))));
	(_tmp131_ == NULL) ? NULL : (_tmp131_ = (vala_ccode_node_unref (_tmp131_), NULL));
	_tmp130_ = (g_free (_tmp130_), NULL);
	_tmp132_ = NULL;
	_tmp133_ = NULL;
	ccheck = (_tmp133_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) ccmp, (ValaCCodeExpression*) (_tmp132_ = vala_ccode_constant_new ("0"))), (_tmp132_ == NULL) ? NULL : (_tmp132_ = (vala_ccode_node_unref (_tmp132_), NULL)), _tmp133_);
	else_block = vala_ccode_block_new ();
	_tmp135_ = NULL;
	_tmp134_ = NULL;
	vala_ccode_block_add_statement (else_block, (ValaCCodeNode*) (_tmp135_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp134_ = vala_ccode_constant_new ("NULL")))));
	(_tmp135_ == NULL) ? NULL : (_tmp135_ = (vala_ccode_node_unref (_tmp135_), NULL));
	(_tmp134_ == NULL) ? NULL : (_tmp134_ = (vala_ccode_node_unref (_tmp134_), NULL));
	_tmp136_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp136_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) ccheck, (ValaCCodeStatement*) prop_block, (ValaCCodeStatement*) else_block)));
	(_tmp136_ == NULL) ? NULL : (_tmp136_ = (vala_ccode_node_unref (_tmp136_), NULL));
	_tmp138_ = NULL;
	_tmp137_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp138_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp137_ = vala_ccode_identifier_new ("reply")))));
	(_tmp138_ == NULL) ? NULL : (_tmp138_ = (vala_ccode_node_unref (_tmp138_), NULL));
	(_tmp137_ == NULL) ? NULL : (_tmp137_ = (vala_ccode_node_unref (_tmp137_), NULL));
	_tmp139_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp139_ = vala_ccode_function_copy (function)));
	(_tmp139_ == NULL) ? NULL : (_tmp139_ = (vala_ccode_node_unref (_tmp139_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp140_ = NULL;
	return (_tmp140_ = wrapper_name, (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), (prefragment == NULL) ? NULL : (prefragment = (vala_ccode_node_unref (prefragment), NULL)), (message_signature == NULL) ? NULL : (message_signature = (vala_ccode_node_unref (message_signature), NULL)), (signature_check == NULL) ? NULL : (signature_check = (vala_ccode_node_unref (signature_check), NULL)), (signature_error_block == NULL) ? NULL : (signature_error_block = (vala_ccode_node_unref (signature_error_block), NULL)), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), (msgcall == NULL) ? NULL : (msgcall = (vala_ccode_node_unref (msgcall), NULL)), (target == NULL) ? NULL : (target = (vala_ccode_node_unref (target), NULL)), (expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL)), (prop_block == NULL) ? NULL : (prop_block = (vala_ccode_node_unref (prop_block), NULL)), (ccmp == NULL) ? NULL : (ccmp = (vala_ccode_node_unref (ccmp), NULL)), (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), (else_block == NULL) ? NULL : (else_block = (vala_ccode_node_unref (else_block), NULL)), _tmp140_);
}


static char* vala_dbus_server_module_generate_dbus_property_set_wrapper (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, const char* dbus_iface_name) {
	char* _tmp0_;
	char* _tmp1_;
	char* wrapper_name;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp4_;
	char* _tmp3_;
	char* _tmp2_;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* _tmp6_;
	ValaCCodeBlock* block;
	ValaCCodeFragment* prefragment;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeVariableDeclarator* _tmp7_;
	ValaCCodeDeclaration* _tmp8_;
	ValaCCodeVariableDeclarator* _tmp9_;
	ValaCCodeVariableDeclarator* _tmp10_;
	ValaCCodeIdentifier* _tmp11_;
	ValaCCodeFunctionCall* _tmp12_;
	ValaCCodeFunctionCall* message_signature;
	ValaCCodeIdentifier* _tmp13_;
	ValaCCodeIdentifier* _tmp14_;
	ValaCCodeFunctionCall* _tmp15_;
	ValaCCodeFunctionCall* signature_check;
	ValaCCodeConstant* _tmp16_;
	ValaCCodeBlock* signature_error_block;
	ValaCCodeReturnStatement* _tmp18_;
	ValaCCodeConstant* _tmp17_;
	ValaCCodeIfStatement* _tmp19_;
	ValaCCodeIdentifier* _tmp20_;
	ValaCCodeFunctionCall* _tmp21_;
	ValaCCodeFunctionCall* iter_call;
	ValaCCodeIdentifier* _tmp22_;
	ValaCCodeUnaryExpression* _tmp24_;
	ValaCCodeIdentifier* _tmp23_;
	ValaCCodeExpressionStatement* _tmp25_;
	ValaCCodeIdentifier* _tmp26_;
	ValaCCodeFunctionCall* _tmp27_;
	ValaCCodeFunctionCall* msgcall;
	ValaCCodeIdentifier* _tmp28_;
	ValaCCodeExpressionStatement* _tmp31_;
	ValaCCodeAssignment* _tmp30_;
	ValaCCodeIdentifier* _tmp29_;
	ValaCCodeDeclaration* _tmp32_;
	ValaCCodeVariableDeclarator* _tmp33_;
	ValaCCodeIdentifier* target;
	ValaCCodeIdentifier* _tmp34_;
	ValaCCodeExpression* _tmp35_;
	ValaCCodeExpression* expr;
	ValaCCodeExpressionStatement* _tmp37_;
	ValaCCodeAssignment* _tmp36_;
	ValaCCodeDeclaration* _tmp38_;
	ValaCCodeVariableDeclarator* _tmp39_;
	ValaCCodeIdentifier* _tmp40_;
	ValaCCodeExpression* _tmp42_;
	ValaCCodeIdentifier* _tmp41_;
	ValaCCodeExpressionStatement* _tmp44_;
	ValaCCodeAssignment* _tmp43_;
	ValaCCodeFunctionCall* _tmp46_;
	ValaCCodeIdentifier* _tmp45_;
	ValaCCodeUnaryExpression* _tmp48_;
	ValaCCodeIdentifier* _tmp47_;
	ValaCCodeUnaryExpression* _tmp50_;
	ValaCCodeIdentifier* _tmp49_;
	ValaCCodeExpressionStatement* _tmp51_;
	ValaCCodeIfStatement* clastif;
	ValaCCodeFunction* _tmp95_;
	char* _tmp96_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	g_return_val_if_fail (dbus_iface_name != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	wrapper_name = (_tmp1_ = g_strdup_printf ("_dbus_%s_property_set", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) sym, NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	function = vala_ccode_function_new (wrapper_name, "DBusMessage*");
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp4_ = vala_ccode_formal_parameter_new ("self", _tmp3_ = g_strconcat (_tmp2_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) sym, FALSE), "*", NULL)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp5_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp6_ = vala_ccode_formal_parameter_new ("message", "DBusMessage*"));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	block = vala_ccode_block_new ();
	prefragment = vala_ccode_fragment_new ();
	_cdecl_ = vala_ccode_declaration_new ("DBusMessage*");
	_tmp7_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp7_ = vala_ccode_variable_declarator_new ("reply", NULL, NULL)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp8_ = NULL;
	_cdecl_ = (_tmp8_ = vala_ccode_declaration_new ("DBusMessageIter"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp8_);
	_tmp9_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp9_ = vala_ccode_variable_declarator_new ("iter", NULL, NULL)));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	_tmp10_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp10_ = vala_ccode_variable_declarator_new ("subiter", NULL, NULL)));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp11_ = NULL;
	_tmp12_ = NULL;
	message_signature = (_tmp12_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp11_ = vala_ccode_identifier_new ("dbus_message_get_signature"))), (_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL)), _tmp12_);
	_tmp13_ = NULL;
	vala_ccode_function_call_add_argument (message_signature, (ValaCCodeExpression*) (_tmp13_ = vala_ccode_identifier_new ("message")));
	(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
	_tmp14_ = NULL;
	_tmp15_ = NULL;
	signature_check = (_tmp15_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp14_ = vala_ccode_identifier_new ("strcmp"))), (_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL)), _tmp15_);
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) message_signature);
	_tmp16_ = NULL;
	vala_ccode_function_call_add_argument (signature_check, (ValaCCodeExpression*) (_tmp16_ = vala_ccode_constant_new ("\"ssv\"")));
	(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
	signature_error_block = vala_ccode_block_new ();
	_tmp18_ = NULL;
	_tmp17_ = NULL;
	vala_ccode_block_add_statement (signature_error_block, (ValaCCodeNode*) (_tmp18_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp17_ = vala_ccode_constant_new ("NULL")))));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
	_tmp19_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp19_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) signature_check, (ValaCCodeStatement*) signature_error_block, NULL)));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	_tmp20_ = NULL;
	_tmp21_ = NULL;
	iter_call = (_tmp21_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new ("dbus_message_iter_init"))), (_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL)), _tmp21_);
	_tmp22_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("message")));
	(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
	_tmp24_ = NULL;
	_tmp23_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp24_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp23_ = vala_ccode_identifier_new ("iter")))));
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	_tmp25_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp25_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
	_tmp26_ = NULL;
	_tmp27_ = NULL;
	msgcall = (_tmp27_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp26_ = vala_ccode_identifier_new ("dbus_message_new_method_return"))), (_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL)), _tmp27_);
	_tmp28_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("message")));
	(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL));
	_tmp31_ = NULL;
	_tmp30_ = NULL;
	_tmp29_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp31_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp30_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp29_ = vala_ccode_identifier_new ("reply")), (ValaCCodeExpression*) msgcall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
	(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) prefragment);
	_tmp32_ = NULL;
	_cdecl_ = (_tmp32_ = vala_ccode_declaration_new ("char*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp32_);
	_tmp33_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp33_ = vala_ccode_variable_declarator_new ("interface_name", NULL, NULL)));
	(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
	target = vala_ccode_identifier_new ("interface_name");
	_tmp34_ = NULL;
	_tmp35_ = NULL;
	expr = (_tmp35_ = vala_dbus_module_read_expression ((ValaDBusModule*) self, prefragment, ((ValaCCodeBaseModule*) self)->string_type, (ValaCCodeExpression*) (_tmp34_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) target), (_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL)), _tmp35_);
	_tmp37_ = NULL;
	_tmp36_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp37_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp36_ = vala_ccode_assignment_new ((ValaCCodeExpression*) target, expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
	(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
	_tmp38_ = NULL;
	_cdecl_ = (_tmp38_ = vala_ccode_declaration_new ("char*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp38_);
	_tmp39_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp39_ = vala_ccode_variable_declarator_new ("property_name", NULL, NULL)));
	(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
	_tmp40_ = NULL;
	target = (_tmp40_ = vala_ccode_identifier_new ("property_name"), (target == NULL) ? NULL : (target = (vala_ccode_node_unref (target), NULL)), _tmp40_);
	_tmp42_ = NULL;
	_tmp41_ = NULL;
	expr = (_tmp42_ = vala_dbus_module_read_expression ((ValaDBusModule*) self, prefragment, ((ValaCCodeBaseModule*) self)->string_type, (ValaCCodeExpression*) (_tmp41_ = vala_ccode_identifier_new ("iter")), (ValaCCodeExpression*) target), (expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL)), _tmp42_);
	(_tmp41_ == NULL) ? NULL : (_tmp41_ = (vala_ccode_node_unref (_tmp41_), NULL));
	_tmp44_ = NULL;
	_tmp43_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp44_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp43_ = vala_ccode_assignment_new ((ValaCCodeExpression*) target, expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
	(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL));
	_tmp46_ = NULL;
	_tmp45_ = NULL;
	iter_call = (_tmp46_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp45_ = vala_ccode_identifier_new ("dbus_message_iter_recurse"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp46_);
	(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
	_tmp48_ = NULL;
	_tmp47_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp48_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp47_ = vala_ccode_identifier_new ("iter")))));
	(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL));
	(_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL));
	_tmp50_ = NULL;
	_tmp49_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp50_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp49_ = vala_ccode_identifier_new ("subiter")))));
	(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
	(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
	_tmp51_ = NULL;
	vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp51_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
	clastif = NULL;
	{
		GeeList* _tmp52_;
		GeeIterator* _tmp53_;
		GeeIterator* _prop_it;
		_tmp52_ = NULL;
		_tmp53_ = NULL;
		_prop_it = (_tmp53_ = gee_iterable_iterator ((GeeIterable*) (_tmp52_ = vala_object_type_symbol_get_properties (sym))), (_tmp52_ == NULL) ? NULL : (_tmp52_ = (gee_collection_object_unref (_tmp52_), NULL)), _tmp53_);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			gboolean _tmp54_;
			gboolean _tmp55_;
			ValaCCodeBlock* prop_block;
			ValaCCodeFragment* _tmp56_;
			ValaCCodeIdentifier* _tmp57_;
			ValaCCodeFunctionCall* _tmp58_;
			ValaCCodeFunctionCall* ccmp;
			ValaCCodeIdentifier* _tmp59_;
			ValaCCodeConstant* _tmp61_;
			char* _tmp60_;
			ValaCCodeConstant* _tmp62_;
			ValaCCodeBinaryExpression* _tmp63_;
			ValaCCodeBinaryExpression* ccheck1;
			ValaCCodeFunctionCall* _tmp65_;
			ValaCCodeIdentifier* _tmp64_;
			ValaCCodeIdentifier* _tmp66_;
			ValaCCodeConstant* _tmp69_;
			char* _tmp68_;
			char* _tmp67_;
			ValaCCodeConstant* _tmp70_;
			ValaCCodeBinaryExpression* _tmp71_;
			ValaCCodeBinaryExpression* ccheck2;
			ValaCCodeBinaryExpression* ccheck;
			ValaCCodeDeclaration* _tmp73_;
			char* _tmp72_;
			ValaCCodeVariableDeclarator* _tmp74_;
			ValaCCodeIdentifier* _tmp75_;
			ValaCCodeExpression* _tmp77_;
			ValaCCodeIdentifier* _tmp76_;
			ValaCCodeExpressionStatement* _tmp79_;
			ValaCCodeAssignment* _tmp78_;
			ValaCCodeIdentifier* _tmp81_;
			char* _tmp80_;
			ValaCCodeFunctionCall* _tmp82_;
			ValaCCodeFunctionCall* ccall;
			ValaCCodeIdentifier* _tmp83_;
			ValaCCodeIdentifier* _tmp84_;
			ValaCCodeExpressionStatement* _tmp85_;
			ValaCCodeIfStatement* cif;
			ValaCCodeIfStatement* _tmp87_;
			ValaCCodeIfStatement* _tmp86_;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			_tmp54_ = FALSE;
			_tmp55_ = FALSE;
			if (vala_property_get_binding (prop) != MEMBER_BINDING_INSTANCE) {
				_tmp55_ = TRUE;
			} else {
				_tmp55_ = vala_property_get_overrides (prop);
			}
			if (_tmp55_) {
				_tmp54_ = TRUE;
			} else {
				_tmp54_ = vala_symbol_get_access ((ValaSymbol*) prop) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
			}
			if (_tmp54_) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) prop)) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (vala_property_get_set_accessor (prop) == NULL) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			prop_block = vala_ccode_block_new ();
			_tmp56_ = NULL;
			prefragment = (_tmp56_ = vala_ccode_fragment_new (), (prefragment == NULL) ? NULL : (prefragment = (vala_ccode_node_unref (prefragment), NULL)), _tmp56_);
			vala_ccode_block_add_statement (prop_block, (ValaCCodeNode*) prefragment);
			_tmp57_ = NULL;
			_tmp58_ = NULL;
			ccmp = (_tmp58_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp57_ = vala_ccode_identifier_new ("strcmp"))), (_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_ccode_node_unref (_tmp57_), NULL)), _tmp58_);
			_tmp59_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp59_ = vala_ccode_identifier_new ("interface_name")));
			(_tmp59_ == NULL) ? NULL : (_tmp59_ = (vala_ccode_node_unref (_tmp59_), NULL));
			_tmp61_ = NULL;
			_tmp60_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp61_ = vala_ccode_constant_new (_tmp60_ = g_strdup_printf ("\"%s\"", dbus_iface_name))));
			(_tmp61_ == NULL) ? NULL : (_tmp61_ = (vala_ccode_node_unref (_tmp61_), NULL));
			_tmp60_ = (g_free (_tmp60_), NULL);
			_tmp62_ = NULL;
			_tmp63_ = NULL;
			ccheck1 = (_tmp63_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) ccmp, (ValaCCodeExpression*) (_tmp62_ = vala_ccode_constant_new ("0"))), (_tmp62_ == NULL) ? NULL : (_tmp62_ = (vala_ccode_node_unref (_tmp62_), NULL)), _tmp63_);
			_tmp65_ = NULL;
			_tmp64_ = NULL;
			ccmp = (_tmp65_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp64_ = vala_ccode_identifier_new ("strcmp"))), (ccmp == NULL) ? NULL : (ccmp = (vala_ccode_node_unref (ccmp), NULL)), _tmp65_);
			(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_ccode_node_unref (_tmp64_), NULL));
			_tmp66_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp66_ = vala_ccode_identifier_new ("property_name")));
			(_tmp66_ == NULL) ? NULL : (_tmp66_ = (vala_ccode_node_unref (_tmp66_), NULL));
			_tmp69_ = NULL;
			_tmp68_ = NULL;
			_tmp67_ = NULL;
			vala_ccode_function_call_add_argument (ccmp, (ValaCCodeExpression*) (_tmp69_ = vala_ccode_constant_new (_tmp68_ = g_strdup_printf ("\"%s\"", _tmp67_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) prop))))));
			(_tmp69_ == NULL) ? NULL : (_tmp69_ = (vala_ccode_node_unref (_tmp69_), NULL));
			_tmp68_ = (g_free (_tmp68_), NULL);
			_tmp67_ = (g_free (_tmp67_), NULL);
			_tmp70_ = NULL;
			_tmp71_ = NULL;
			ccheck2 = (_tmp71_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) ccmp, (ValaCCodeExpression*) (_tmp70_ = vala_ccode_constant_new ("0"))), (_tmp70_ == NULL) ? NULL : (_tmp70_ = (vala_ccode_node_unref (_tmp70_), NULL)), _tmp71_);
			ccheck = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_AND, (ValaCCodeExpression*) ccheck1, (ValaCCodeExpression*) ccheck2);
			_tmp73_ = NULL;
			_tmp72_ = NULL;
			_cdecl_ = (_tmp73_ = vala_ccode_declaration_new (_tmp72_ = vala_data_type_get_cname (vala_property_get_property_type (prop))), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp73_);
			_tmp72_ = (g_free (_tmp72_), NULL);
			_tmp74_ = NULL;
			vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp74_ = vala_ccode_variable_declarator_new ("value", NULL, NULL)));
			(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_ccode_node_unref (_tmp74_), NULL));
			vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) _cdecl_);
			_tmp75_ = NULL;
			target = (_tmp75_ = vala_ccode_identifier_new ("value"), (target == NULL) ? NULL : (target = (vala_ccode_node_unref (target), NULL)), _tmp75_);
			_tmp77_ = NULL;
			_tmp76_ = NULL;
			expr = (_tmp77_ = vala_dbus_module_read_expression ((ValaDBusModule*) self, prefragment, vala_property_get_property_type (prop), (ValaCCodeExpression*) (_tmp76_ = vala_ccode_identifier_new ("subiter")), (ValaCCodeExpression*) target), (expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL)), _tmp77_);
			(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_ccode_node_unref (_tmp76_), NULL));
			_tmp79_ = NULL;
			_tmp78_ = NULL;
			vala_ccode_fragment_append (prefragment, (ValaCCodeNode*) (_tmp79_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp78_ = vala_ccode_assignment_new ((ValaCCodeExpression*) target, expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
			(_tmp79_ == NULL) ? NULL : (_tmp79_ = (vala_ccode_node_unref (_tmp79_), NULL));
			(_tmp78_ == NULL) ? NULL : (_tmp78_ = (vala_ccode_node_unref (_tmp78_), NULL));
			_tmp81_ = NULL;
			_tmp80_ = NULL;
			_tmp82_ = NULL;
			ccall = (_tmp82_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp81_ = vala_ccode_identifier_new (_tmp80_ = vala_property_accessor_get_cname (vala_property_get_set_accessor (prop))))), (_tmp81_ == NULL) ? NULL : (_tmp81_ = (vala_ccode_node_unref (_tmp81_), NULL)), _tmp80_ = (g_free (_tmp80_), NULL), _tmp82_);
			_tmp83_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp83_ = vala_ccode_identifier_new ("self")));
			(_tmp83_ == NULL) ? NULL : (_tmp83_ = (vala_ccode_node_unref (_tmp83_), NULL));
			_tmp84_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new ("value")));
			(_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL));
			_tmp85_ = NULL;
			vala_ccode_block_add_statement (prop_block, (ValaCCodeNode*) (_tmp85_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccall)));
			(_tmp85_ == NULL) ? NULL : (_tmp85_ = (vala_ccode_node_unref (_tmp85_), NULL));
			cif = vala_ccode_if_statement_new ((ValaCCodeExpression*) ccheck, (ValaCCodeStatement*) prop_block, NULL);
			if (clastif == NULL) {
				vala_ccode_block_add_statement (block, (ValaCCodeNode*) cif);
			} else {
				vala_ccode_if_statement_set_false_statement (clastif, (ValaCCodeStatement*) cif);
			}
			_tmp87_ = NULL;
			_tmp86_ = NULL;
			clastif = (_tmp87_ = (_tmp86_ = cif, (_tmp86_ == NULL) ? NULL : vala_ccode_node_ref (_tmp86_)), (clastif == NULL) ? NULL : (clastif = (vala_ccode_node_unref (clastif), NULL)), _tmp87_);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			(prop_block == NULL) ? NULL : (prop_block = (vala_ccode_node_unref (prop_block), NULL));
			(ccmp == NULL) ? NULL : (ccmp = (vala_ccode_node_unref (ccmp), NULL));
			(ccheck1 == NULL) ? NULL : (ccheck1 = (vala_ccode_node_unref (ccheck1), NULL));
			(ccheck2 == NULL) ? NULL : (ccheck2 = (vala_ccode_node_unref (ccheck2), NULL));
			(ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL));
			(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
			(cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	if (clastif == NULL) {
		ValaCCodeBlock* _tmp88_;
		ValaCCodeReturnStatement* _tmp90_;
		ValaCCodeConstant* _tmp89_;
		_tmp88_ = NULL;
		block = (_tmp88_ = vala_ccode_block_new (), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), _tmp88_);
		_tmp90_ = NULL;
		_tmp89_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp90_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp89_ = vala_ccode_constant_new ("NULL")))));
		(_tmp90_ == NULL) ? NULL : (_tmp90_ = (vala_ccode_node_unref (_tmp90_), NULL));
		(_tmp89_ == NULL) ? NULL : (_tmp89_ = (vala_ccode_node_unref (_tmp89_), NULL));
	} else {
		ValaCCodeBlock* else_block;
		ValaCCodeReturnStatement* _tmp92_;
		ValaCCodeConstant* _tmp91_;
		ValaCCodeReturnStatement* _tmp94_;
		ValaCCodeIdentifier* _tmp93_;
		else_block = vala_ccode_block_new ();
		_tmp92_ = NULL;
		_tmp91_ = NULL;
		vala_ccode_block_add_statement (else_block, (ValaCCodeNode*) (_tmp92_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp91_ = vala_ccode_constant_new ("NULL")))));
		(_tmp92_ == NULL) ? NULL : (_tmp92_ = (vala_ccode_node_unref (_tmp92_), NULL));
		(_tmp91_ == NULL) ? NULL : (_tmp91_ = (vala_ccode_node_unref (_tmp91_), NULL));
		vala_ccode_if_statement_set_false_statement (clastif, (ValaCCodeStatement*) else_block);
		_tmp94_ = NULL;
		_tmp93_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp94_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp93_ = vala_ccode_identifier_new ("reply")))));
		(_tmp94_ == NULL) ? NULL : (_tmp94_ = (vala_ccode_node_unref (_tmp94_), NULL));
		(_tmp93_ == NULL) ? NULL : (_tmp93_ = (vala_ccode_node_unref (_tmp93_), NULL));
		(else_block == NULL) ? NULL : (else_block = (vala_ccode_node_unref (else_block), NULL));
	}
	_tmp95_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp95_ = vala_ccode_function_copy (function)));
	(_tmp95_ == NULL) ? NULL : (_tmp95_ = (vala_ccode_node_unref (_tmp95_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp96_ = NULL;
	return (_tmp96_ = wrapper_name, (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), (prefragment == NULL) ? NULL : (prefragment = (vala_ccode_node_unref (prefragment), NULL)), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), (message_signature == NULL) ? NULL : (message_signature = (vala_ccode_node_unref (message_signature), NULL)), (signature_check == NULL) ? NULL : (signature_check = (vala_ccode_node_unref (signature_check), NULL)), (signature_error_block == NULL) ? NULL : (signature_error_block = (vala_ccode_node_unref (signature_error_block), NULL)), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), (msgcall == NULL) ? NULL : (msgcall = (vala_ccode_node_unref (msgcall), NULL)), (target == NULL) ? NULL : (target = (vala_ccode_node_unref (target), NULL)), (expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL)), (clastif == NULL) ? NULL : (clastif = (vala_ccode_node_unref (clastif), NULL)), _tmp96_);
}


static char* vala_dbus_server_module_get_dbus_type_introspection (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym) {
	char* _result_;
	ValaClass* _tmp1_;
	ValaObjectTypeSymbol* _tmp0_;
	ValaClass* cl;
	ValaAttribute* dbus;
	char* dbus_iface_name;
	char* _tmp11_;
	char* _tmp10_;
	char* _tmp55_;
	char* _tmp56_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	_result_ = g_strdup ("");
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	cl = (_tmp1_ = (_tmp0_ = sym, VALA_IS_CLASS (_tmp0_) ? ((ValaClass*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (cl != NULL) {
		{
			GeeList* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _base_type_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_base_type_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = vala_class_get_base_types (cl))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp3_);
			while (gee_iterator_next (_base_type_it)) {
				ValaDataType* base_type;
				ValaObjectType* _tmp5_;
				ValaDataType* _tmp4_;
				ValaObjectType* base_obj_type;
				char* _tmp7_;
				char* _tmp6_;
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				base_obj_type = (_tmp5_ = (_tmp4_ = base_type, VALA_IS_OBJECT_TYPE (_tmp4_) ? ((ValaObjectType*) _tmp4_) : NULL), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				_result_ = (_tmp7_ = g_strconcat (_result_, _tmp6_ = vala_dbus_server_module_get_dbus_type_introspection (self, vala_object_type_get_type_symbol (base_obj_type)), NULL), _result_ = (g_free (_result_), NULL), _tmp7_);
				_tmp6_ = (g_free (_tmp6_), NULL);
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(base_obj_type == NULL) ? NULL : (base_obj_type = (vala_code_node_unref (base_obj_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
	}
	dbus = vala_code_node_get_attribute ((ValaCodeNode*) sym, "DBus");
	if (dbus == NULL) {
		char* _tmp8_;
		_tmp8_ = NULL;
		return (_tmp8_ = _result_, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL)), _tmp8_);
	}
	dbus_iface_name = vala_attribute_get_string (dbus, "name");
	if (dbus_iface_name == NULL) {
		char* _tmp9_;
		_tmp9_ = NULL;
		return (_tmp9_ = _result_, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL)), dbus_iface_name = (g_free (dbus_iface_name), NULL), _tmp9_);
	}
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	_result_ = (_tmp11_ = g_strconcat (_result_, _tmp10_ = g_strdup_printf ("<interface name=\"%s\">\n", dbus_iface_name), NULL), _result_ = (g_free (_result_), NULL), _tmp11_);
	_tmp10_ = (g_free (_tmp10_), NULL);
	{
		GeeList* _tmp12_;
		GeeIterator* _tmp13_;
		GeeIterator* _m_it;
		_tmp12_ = NULL;
		_tmp13_ = NULL;
		_m_it = (_tmp13_ = gee_iterable_iterator ((GeeIterable*) (_tmp12_ = vala_object_type_symbol_get_methods (sym))), (_tmp12_ == NULL) ? NULL : (_tmp12_ = (gee_collection_object_unref (_tmp12_), NULL)), _tmp13_);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			gboolean _tmp14_;
			gboolean _tmp15_;
			gboolean _tmp16_;
			char* _tmp19_;
			char* _tmp18_;
			char* _tmp17_;
			char* _tmp33_;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			_tmp14_ = FALSE;
			_tmp15_ = FALSE;
			_tmp16_ = FALSE;
			if (VALA_IS_CREATION_METHOD (m)) {
				_tmp16_ = TRUE;
			} else {
				_tmp16_ = vala_method_get_binding (m) != MEMBER_BINDING_INSTANCE;
			}
			if (_tmp16_) {
				_tmp15_ = TRUE;
			} else {
				_tmp15_ = vala_method_get_overrides (m);
			}
			if (_tmp15_) {
				_tmp14_ = TRUE;
			} else {
				_tmp14_ = vala_symbol_get_access ((ValaSymbol*) m) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
			}
			if (_tmp14_) {
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) m)) {
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
				continue;
			}
			_tmp19_ = NULL;
			_tmp18_ = NULL;
			_tmp17_ = NULL;
			_result_ = (_tmp19_ = g_strconcat (_result_, _tmp18_ = g_strdup_printf ("  <method name=\"%s\">\n", _tmp17_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) m))), NULL), _result_ = (g_free (_result_), NULL), _tmp19_);
			_tmp18_ = (g_free (_tmp18_), NULL);
			_tmp17_ = (g_free (_tmp17_), NULL);
			{
				GeeList* _tmp20_;
				GeeIterator* _tmp21_;
				GeeIterator* _param_it;
				_tmp20_ = NULL;
				_tmp21_ = NULL;
				_param_it = (_tmp21_ = gee_iterable_iterator ((GeeIterable*) (_tmp20_ = vala_method_get_parameters (m))), (_tmp20_ == NULL) ? NULL : (_tmp20_ = (gee_collection_object_unref (_tmp20_), NULL)), _tmp21_);
				while (gee_iterator_next (_param_it)) {
					ValaFormalParameter* param;
					gboolean _tmp22_;
					const char* _tmp24_;
					const char* _tmp25_;
					char* direction;
					char* _tmp28_;
					char* _tmp27_;
					char* _tmp26_;
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					_tmp22_ = FALSE;
					if (vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param)) != NULL) {
						char* _tmp23_;
						_tmp23_ = NULL;
						_tmp22_ = _vala_strcmp0 (_tmp23_ = vala_symbol_get_full_name ((ValaSymbol*) vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param))), "DBus.BusName") == 0;
						_tmp23_ = (g_free (_tmp23_), NULL);
					} else {
						_tmp22_ = FALSE;
					}
					if (_tmp22_) {
						/* skip sender parameter
						 (implicit in D-Bus)*/
						(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
						continue;
					}
					_tmp24_ = NULL;
					if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN) {
						_tmp24_ = "in";
					} else {
						_tmp24_ = "out";
					}
					_tmp25_ = NULL;
					direction = (_tmp25_ = _tmp24_, (_tmp25_ == NULL) ? NULL : g_strdup (_tmp25_));
					_tmp28_ = NULL;
					_tmp27_ = NULL;
					_tmp26_ = NULL;
					_result_ = (_tmp28_ = g_strconcat (_result_, _tmp27_ = g_strdup_printf ("    <arg name=\"%s\" type=\"%s\" direction=\"%s\"/>\n", vala_symbol_get_name ((ValaSymbol*) param), _tmp26_ = vala_data_type_get_type_signature (vala_formal_parameter_get_parameter_type (param)), direction), NULL), _result_ = (g_free (_result_), NULL), _tmp28_);
					_tmp27_ = (g_free (_tmp27_), NULL);
					_tmp26_ = (g_free (_tmp26_), NULL);
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
					direction = (g_free (direction), NULL);
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			if (!VALA_IS_VOID_TYPE (vala_method_get_return_type (m))) {
				char* _tmp32_;
				char* _tmp31_;
				char* _tmp30_;
				char* _tmp29_;
				_tmp32_ = NULL;
				_tmp31_ = NULL;
				_tmp30_ = NULL;
				_tmp29_ = NULL;
				_result_ = (_tmp32_ = g_strconcat (_result_, _tmp31_ = g_strdup_printf ("    <arg name=\"%s\" type=\"%s\" direction=\"out\"/>\n", _tmp29_ = vala_dbus_server_module_dbus_result_name (self, m), _tmp30_ = vala_data_type_get_type_signature (vala_method_get_return_type (m))), NULL), _result_ = (g_free (_result_), NULL), _tmp32_);
				_tmp31_ = (g_free (_tmp31_), NULL);
				_tmp30_ = (g_free (_tmp30_), NULL);
				_tmp29_ = (g_free (_tmp29_), NULL);
			}
			_tmp33_ = NULL;
			_result_ = (_tmp33_ = g_strconcat (_result_, "  </method>\n", NULL), _result_ = (g_free (_result_), NULL), _tmp33_);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	{
		GeeList* _tmp34_;
		GeeIterator* _tmp35_;
		GeeIterator* _prop_it;
		_tmp34_ = NULL;
		_tmp35_ = NULL;
		_prop_it = (_tmp35_ = gee_iterable_iterator ((GeeIterable*) (_tmp34_ = vala_object_type_symbol_get_properties (sym))), (_tmp34_ == NULL) ? NULL : (_tmp34_ = (gee_collection_object_unref (_tmp34_), NULL)), _tmp35_);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			gboolean _tmp36_;
			gboolean _tmp37_;
			const char* _tmp38_;
			const char* _tmp39_;
			char* access;
			char* _tmp43_;
			char* _tmp42_;
			char* _tmp41_;
			char* _tmp40_;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			_tmp36_ = FALSE;
			_tmp37_ = FALSE;
			if (vala_property_get_binding (prop) != MEMBER_BINDING_INSTANCE) {
				_tmp37_ = TRUE;
			} else {
				_tmp37_ = vala_property_get_overrides (prop);
			}
			if (_tmp37_) {
				_tmp36_ = TRUE;
			} else {
				_tmp36_ = vala_symbol_get_access ((ValaSymbol*) prop) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
			}
			if (_tmp36_) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) prop)) {
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				continue;
			}
			_tmp38_ = NULL;
			if (vala_property_get_get_accessor (prop) != NULL) {
				_tmp38_ = "read";
			} else {
				_tmp38_ = "";
			}
			_tmp39_ = NULL;
			if (vala_property_get_set_accessor (prop) != NULL) {
				_tmp39_ = "write";
			} else {
				_tmp39_ = "";
			}
			access = g_strconcat (_tmp38_, _tmp39_, NULL);
			_tmp43_ = NULL;
			_tmp42_ = NULL;
			_tmp41_ = NULL;
			_tmp40_ = NULL;
			_result_ = (_tmp43_ = g_strconcat (_result_, _tmp42_ = g_strdup_printf ("  <property name=\"%s\" type=\"%s\" access=\"%s\"/>\n", _tmp40_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) prop)), _tmp41_ = vala_data_type_get_type_signature (vala_property_get_property_type (prop)), access), NULL), _result_ = (g_free (_result_), NULL), _tmp43_);
			_tmp42_ = (g_free (_tmp42_), NULL);
			_tmp41_ = (g_free (_tmp41_), NULL);
			_tmp40_ = (g_free (_tmp40_), NULL);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			access = (g_free (access), NULL);
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	{
		GeeList* _tmp44_;
		GeeIterator* _tmp45_;
		GeeIterator* _sig_it;
		_tmp44_ = NULL;
		_tmp45_ = NULL;
		_sig_it = (_tmp45_ = gee_iterable_iterator ((GeeIterable*) (_tmp44_ = vala_object_type_symbol_get_signals (sym))), (_tmp44_ == NULL) ? NULL : (_tmp44_ = (gee_collection_object_unref (_tmp44_), NULL)), _tmp45_);
		while (gee_iterator_next (_sig_it)) {
			ValaSignal* sig;
			char* _tmp48_;
			char* _tmp47_;
			char* _tmp46_;
			char* _tmp54_;
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			if (vala_symbol_get_access ((ValaSymbol*) sig) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) sig)) {
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				continue;
			}
			_tmp48_ = NULL;
			_tmp47_ = NULL;
			_tmp46_ = NULL;
			_result_ = (_tmp48_ = g_strconcat (_result_, _tmp47_ = g_strdup_printf ("  <signal name=\"%s\">\n", _tmp46_ = vala_symbol_lower_case_to_camel_case (vala_symbol_get_name ((ValaSymbol*) sig))), NULL), _result_ = (g_free (_result_), NULL), _tmp48_);
			_tmp47_ = (g_free (_tmp47_), NULL);
			_tmp46_ = (g_free (_tmp46_), NULL);
			{
				GeeList* _tmp49_;
				GeeIterator* _tmp50_;
				GeeIterator* _param_it;
				_tmp49_ = NULL;
				_tmp50_ = NULL;
				_param_it = (_tmp50_ = gee_iterable_iterator ((GeeIterable*) (_tmp49_ = vala_signal_get_parameters (sig))), (_tmp49_ == NULL) ? NULL : (_tmp49_ = (gee_collection_object_unref (_tmp49_), NULL)), _tmp50_);
				while (gee_iterator_next (_param_it)) {
					ValaFormalParameter* param;
					char* _tmp53_;
					char* _tmp52_;
					char* _tmp51_;
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					_tmp53_ = NULL;
					_tmp52_ = NULL;
					_tmp51_ = NULL;
					_result_ = (_tmp53_ = g_strconcat (_result_, _tmp52_ = g_strdup_printf ("    <arg name=\"%s\" type=\"%s\"/>\n", vala_symbol_get_name ((ValaSymbol*) param), _tmp51_ = vala_data_type_get_type_signature (vala_formal_parameter_get_parameter_type (param))), NULL), _result_ = (g_free (_result_), NULL), _tmp53_);
					_tmp52_ = (g_free (_tmp52_), NULL);
					_tmp51_ = (g_free (_tmp51_), NULL);
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			_tmp54_ = NULL;
			_result_ = (_tmp54_ = g_strconcat (_result_, "  </signal>\n", NULL), _result_ = (g_free (_result_), NULL), _tmp54_);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	_tmp55_ = NULL;
	_result_ = (_tmp55_ = g_strconcat (_result_, "</interface>\n", NULL), _result_ = (g_free (_result_), NULL), _tmp55_);
	_tmp56_ = NULL;
	return (_tmp56_ = _result_, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL)), dbus_iface_name = (g_free (dbus_iface_name), NULL), _tmp56_);
}


static char* vala_dbus_server_module_generate_dbus_introspect (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym) {
	char* _tmp0_;
	char* _tmp1_;
	char* wrapper_name;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp4_;
	char* _tmp3_;
	char* _tmp2_;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* _tmp6_;
	ValaCCodeBlock* block;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeVariableDeclarator* _tmp7_;
	ValaCCodeDeclaration* _tmp8_;
	ValaCCodeVariableDeclarator* _tmp9_;
	ValaCCodeIdentifier* _tmp10_;
	ValaCCodeFunctionCall* _tmp11_;
	ValaCCodeFunctionCall* msgcall;
	ValaCCodeIdentifier* _tmp12_;
	ValaCCodeExpressionStatement* _tmp15_;
	ValaCCodeAssignment* _tmp14_;
	ValaCCodeIdentifier* _tmp13_;
	ValaCCodeIdentifier* _tmp16_;
	ValaCCodeFunctionCall* _tmp17_;
	ValaCCodeFunctionCall* iter_call;
	ValaCCodeIdentifier* _tmp18_;
	ValaCCodeUnaryExpression* _tmp20_;
	ValaCCodeIdentifier* _tmp19_;
	ValaCCodeExpressionStatement* _tmp21_;
	ValaCCodeDeclaration* _tmp22_;
	ValaCCodeVariableDeclarator* _tmp23_;
	ValaCCodeDeclaration* _tmp24_;
	ValaCCodeVariableDeclarator* _tmp25_;
	ValaCCodeDeclaration* _tmp26_;
	ValaCCodeVariableDeclarator* _tmp27_;
	char* xml_data;
	ValaCCodeIdentifier* _tmp28_;
	ValaCCodeFunctionCall* _tmp29_;
	ValaCCodeFunctionCall* str_call;
	ValaCCodeConstant* _tmp32_;
	char* _tmp31_;
	char* _tmp30_;
	ValaCCodeExpressionStatement* _tmp35_;
	ValaCCodeAssignment* _tmp34_;
	ValaCCodeIdentifier* _tmp33_;
	char* _tmp36_;
	char* _tmp37_;
	char* _tmp39_;
	char* _tmp38_;
	ValaCCodeFunctionCall* _tmp41_;
	ValaCCodeIdentifier* _tmp40_;
	ValaCCodeIdentifier* _tmp42_;
	ValaCCodeConstant* _tmp45_;
	char* _tmp44_;
	char* _tmp43_;
	ValaCCodeExpressionStatement* _tmp46_;
	ValaCCodeIdentifier* _tmp47_;
	ValaCCodeFunctionCall* _tmp48_;
	ValaCCodeFunctionCall* get_path;
	ValaCCodeCastExpression* _tmp50_;
	ValaCCodeIdentifier* _tmp49_;
	ValaCCodeConstant* _tmp51_;
	ValaCCodeIdentifier* _tmp52_;
	ValaCCodeFunctionCall* _tmp53_;
	ValaCCodeFunctionCall* list_call;
	ValaCCodeIdentifier* _tmp54_;
	ValaCCodeUnaryExpression* _tmp56_;
	ValaCCodeIdentifier* _tmp55_;
	ValaCCodeExpressionStatement* _tmp57_;
	ValaCCodeBlock* child_block;
	ValaCCodeFunctionCall* _tmp59_;
	ValaCCodeIdentifier* _tmp58_;
	ValaCCodeIdentifier* _tmp60_;
	ValaCCodeConstant* _tmp63_;
	char* _tmp62_;
	char* _tmp61_;
	ValaCCodeElementAccess* _tmp66_;
	ValaCCodeIdentifier* _tmp65_;
	ValaCCodeIdentifier* _tmp64_;
	ValaCCodeExpressionStatement* _tmp67_;
	ValaCCodeElementAccess* _tmp70_;
	ValaCCodeIdentifier* _tmp69_;
	ValaCCodeIdentifier* _tmp68_;
	ValaCCodeForStatement* _tmp71_;
	ValaCCodeForStatement* cfor;
	ValaCCodeAssignment* _tmp74_;
	ValaCCodeConstant* _tmp73_;
	ValaCCodeIdentifier* _tmp72_;
	ValaCCodeUnaryExpression* _tmp76_;
	ValaCCodeIdentifier* _tmp75_;
	ValaCCodeIdentifier* _tmp77_;
	ValaCCodeFunctionCall* _tmp78_;
	ValaCCodeFunctionCall* list_free_call;
	ValaCCodeIdentifier* _tmp79_;
	ValaCCodeExpressionStatement* _tmp80_;
	char* _tmp81_;
	ValaCCodeFunctionCall* _tmp83_;
	ValaCCodeIdentifier* _tmp82_;
	ValaCCodeIdentifier* _tmp84_;
	ValaCCodeConstant* _tmp87_;
	char* _tmp86_;
	char* _tmp85_;
	ValaCCodeExpressionStatement* _tmp88_;
	ValaCCodeFunctionCall* _tmp90_;
	ValaCCodeIdentifier* _tmp89_;
	ValaCCodeUnaryExpression* _tmp92_;
	ValaCCodeIdentifier* _tmp91_;
	ValaCCodeIdentifier* _tmp93_;
	ValaCCodeUnaryExpression* _tmp96_;
	ValaCCodeMemberAccess* _tmp95_;
	ValaCCodeIdentifier* _tmp94_;
	ValaCCodeExpressionStatement* _tmp97_;
	ValaCCodeFunctionCall* _tmp99_;
	ValaCCodeIdentifier* _tmp98_;
	ValaCCodeIdentifier* _tmp100_;
	ValaCCodeConstant* _tmp101_;
	ValaCCodeExpressionStatement* _tmp102_;
	ValaCCodeReturnStatement* _tmp104_;
	ValaCCodeIdentifier* _tmp103_;
	ValaCCodeFunction* _tmp105_;
	char* _tmp106_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	wrapper_name = (_tmp1_ = g_strdup_printf ("_dbus_%s_introspect", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) sym, NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	function = vala_ccode_function_new (wrapper_name, "DBusMessage*");
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp4_ = vala_ccode_formal_parameter_new ("self", _tmp3_ = g_strconcat (_tmp2_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) sym, FALSE), "*", NULL)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp5_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp6_ = vala_ccode_formal_parameter_new ("message", "DBusMessage*"));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	block = vala_ccode_block_new ();
	_cdecl_ = vala_ccode_declaration_new ("DBusMessage*");
	_tmp7_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp7_ = vala_ccode_variable_declarator_new ("reply", NULL, NULL)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp8_ = NULL;
	_cdecl_ = (_tmp8_ = vala_ccode_declaration_new ("DBusMessageIter"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp8_);
	_tmp9_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp9_ = vala_ccode_variable_declarator_new ("iter", NULL, NULL)));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp10_ = NULL;
	_tmp11_ = NULL;
	msgcall = (_tmp11_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new ("dbus_message_new_method_return"))), (_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL)), _tmp11_);
	_tmp12_ = NULL;
	vala_ccode_function_call_add_argument (msgcall, (ValaCCodeExpression*) (_tmp12_ = vala_ccode_identifier_new ("message")));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	_tmp15_ = NULL;
	_tmp14_ = NULL;
	_tmp13_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp15_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp14_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_identifier_new ("reply")), (ValaCCodeExpression*) msgcall, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
	_tmp16_ = NULL;
	_tmp17_ = NULL;
	iter_call = (_tmp17_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new ("dbus_message_iter_init_append"))), (_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL)), _tmp17_);
	_tmp18_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_identifier_new ("reply")));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	_tmp20_ = NULL;
	_tmp19_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp20_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp19_ = vala_ccode_identifier_new ("iter")))));
	(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	_tmp21_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp21_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
	_tmp22_ = NULL;
	_cdecl_ = (_tmp22_ = vala_ccode_declaration_new ("GString*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp22_);
	_tmp23_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp23_ = vala_ccode_variable_declarator_new ("xml_data", NULL, NULL)));
	(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp24_ = NULL;
	_cdecl_ = (_tmp24_ = vala_ccode_declaration_new ("char**"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp24_);
	_tmp25_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp25_ = vala_ccode_variable_declarator_new ("children", NULL, NULL)));
	(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	_tmp26_ = NULL;
	_cdecl_ = (_tmp26_ = vala_ccode_declaration_new ("int"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp26_);
	_tmp27_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp27_ = vala_ccode_variable_declarator_new ("i", NULL, NULL)));
	(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	xml_data = g_strdup ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	_tmp28_ = NULL;
	_tmp29_ = NULL;
	str_call = (_tmp29_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("g_string_new"))), (_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL)), _tmp29_);
	_tmp32_ = NULL;
	_tmp31_ = NULL;
	_tmp30_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp32_ = vala_ccode_constant_new (_tmp31_ = g_strdup_printf ("\"%s\"", _tmp30_ = g_strescape (xml_data, "")))));
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	_tmp31_ = (g_free (_tmp31_), NULL);
	_tmp30_ = (g_free (_tmp30_), NULL);
	_tmp35_ = NULL;
	_tmp34_ = NULL;
	_tmp33_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp35_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp34_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp33_ = vala_ccode_identifier_new ("xml_data")), (ValaCCodeExpression*) str_call, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
	(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
	(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
	_tmp36_ = NULL;
	xml_data = (_tmp36_ = g_strdup ("<node>\n"), xml_data = (g_free (xml_data), NULL), _tmp36_);
	_tmp37_ = NULL;
	xml_data = (_tmp37_ = g_strconcat (xml_data, "<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n", NULL), xml_data = (g_free (xml_data), NULL), _tmp37_);
	_tmp39_ = NULL;
	_tmp38_ = NULL;
	xml_data = (_tmp39_ = g_strconcat (xml_data, _tmp38_ = vala_dbus_server_module_get_dbus_type_introspection (self, sym), NULL), xml_data = (g_free (xml_data), NULL), _tmp39_);
	_tmp38_ = (g_free (_tmp38_), NULL);
	_tmp41_ = NULL;
	_tmp40_ = NULL;
	str_call = (_tmp41_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp40_ = vala_ccode_identifier_new ("g_string_append"))), (str_call == NULL) ? NULL : (str_call = (vala_ccode_node_unref (str_call), NULL)), _tmp41_);
	(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_ccode_node_unref (_tmp40_), NULL));
	_tmp42_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp42_ = vala_ccode_identifier_new ("xml_data")));
	(_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL));
	_tmp45_ = NULL;
	_tmp44_ = NULL;
	_tmp43_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp45_ = vala_ccode_constant_new (_tmp44_ = g_strdup_printf ("\"%s\"", _tmp43_ = g_strescape (xml_data, "")))));
	(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
	_tmp44_ = (g_free (_tmp44_), NULL);
	_tmp43_ = (g_free (_tmp43_), NULL);
	_tmp46_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp46_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) str_call)));
	(_tmp46_ == NULL) ? NULL : (_tmp46_ = (vala_ccode_node_unref (_tmp46_), NULL));
	_tmp47_ = NULL;
	_tmp48_ = NULL;
	get_path = (_tmp48_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp47_ = vala_ccode_identifier_new ("g_object_get_data"))), (_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL)), _tmp48_);
	_tmp50_ = NULL;
	_tmp49_ = NULL;
	vala_ccode_function_call_add_argument (get_path, (ValaCCodeExpression*) (_tmp50_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) (_tmp49_ = vala_ccode_identifier_new ("self")), "GObject *")));
	(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
	(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
	_tmp51_ = NULL;
	vala_ccode_function_call_add_argument (get_path, (ValaCCodeExpression*) (_tmp51_ = vala_ccode_constant_new ("\"dbus_object_path\"")));
	(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
	_tmp52_ = NULL;
	_tmp53_ = NULL;
	list_call = (_tmp53_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp52_ = vala_ccode_identifier_new ("dbus_connection_list_registered"))), (_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_ccode_node_unref (_tmp52_), NULL)), _tmp53_);
	_tmp54_ = NULL;
	vala_ccode_function_call_add_argument (list_call, (ValaCCodeExpression*) (_tmp54_ = vala_ccode_identifier_new ("connection")));
	(_tmp54_ == NULL) ? NULL : (_tmp54_ = (vala_ccode_node_unref (_tmp54_), NULL));
	vala_ccode_function_call_add_argument (list_call, (ValaCCodeExpression*) get_path);
	_tmp56_ = NULL;
	_tmp55_ = NULL;
	vala_ccode_function_call_add_argument (list_call, (ValaCCodeExpression*) (_tmp56_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp55_ = vala_ccode_identifier_new ("children")))));
	(_tmp56_ == NULL) ? NULL : (_tmp56_ = (vala_ccode_node_unref (_tmp56_), NULL));
	(_tmp55_ == NULL) ? NULL : (_tmp55_ = (vala_ccode_node_unref (_tmp55_), NULL));
	_tmp57_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp57_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) list_call)));
	(_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_ccode_node_unref (_tmp57_), NULL));
	/* add child nodes*/
	child_block = vala_ccode_block_new ();
	_tmp59_ = NULL;
	_tmp58_ = NULL;
	str_call = (_tmp59_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp58_ = vala_ccode_identifier_new ("g_string_append_printf"))), (str_call == NULL) ? NULL : (str_call = (vala_ccode_node_unref (str_call), NULL)), _tmp59_);
	(_tmp58_ == NULL) ? NULL : (_tmp58_ = (vala_ccode_node_unref (_tmp58_), NULL));
	_tmp60_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp60_ = vala_ccode_identifier_new ("xml_data")));
	(_tmp60_ == NULL) ? NULL : (_tmp60_ = (vala_ccode_node_unref (_tmp60_), NULL));
	_tmp63_ = NULL;
	_tmp62_ = NULL;
	_tmp61_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp63_ = vala_ccode_constant_new (_tmp62_ = g_strdup_printf ("\"%s\"", _tmp61_ = g_strescape ("<node name=\"%s\"/>\n", "")))));
	(_tmp63_ == NULL) ? NULL : (_tmp63_ = (vala_ccode_node_unref (_tmp63_), NULL));
	_tmp62_ = (g_free (_tmp62_), NULL);
	_tmp61_ = (g_free (_tmp61_), NULL);
	_tmp66_ = NULL;
	_tmp65_ = NULL;
	_tmp64_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp66_ = vala_ccode_element_access_new ((ValaCCodeExpression*) (_tmp64_ = vala_ccode_identifier_new ("children")), (ValaCCodeExpression*) (_tmp65_ = vala_ccode_identifier_new ("i")))));
	(_tmp66_ == NULL) ? NULL : (_tmp66_ = (vala_ccode_node_unref (_tmp66_), NULL));
	(_tmp65_ == NULL) ? NULL : (_tmp65_ = (vala_ccode_node_unref (_tmp65_), NULL));
	(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_ccode_node_unref (_tmp64_), NULL));
	_tmp67_ = NULL;
	vala_ccode_block_add_statement (child_block, (ValaCCodeNode*) (_tmp67_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) str_call)));
	(_tmp67_ == NULL) ? NULL : (_tmp67_ = (vala_ccode_node_unref (_tmp67_), NULL));
	_tmp70_ = NULL;
	_tmp69_ = NULL;
	_tmp68_ = NULL;
	_tmp71_ = NULL;
	cfor = (_tmp71_ = vala_ccode_for_statement_new ((ValaCCodeExpression*) (_tmp70_ = vala_ccode_element_access_new ((ValaCCodeExpression*) (_tmp68_ = vala_ccode_identifier_new ("children")), (ValaCCodeExpression*) (_tmp69_ = vala_ccode_identifier_new ("i")))), (ValaCCodeStatement*) child_block), (_tmp70_ == NULL) ? NULL : (_tmp70_ = (vala_ccode_node_unref (_tmp70_), NULL)), (_tmp69_ == NULL) ? NULL : (_tmp69_ = (vala_ccode_node_unref (_tmp69_), NULL)), (_tmp68_ == NULL) ? NULL : (_tmp68_ = (vala_ccode_node_unref (_tmp68_), NULL)), _tmp71_);
	_tmp74_ = NULL;
	_tmp73_ = NULL;
	_tmp72_ = NULL;
	vala_ccode_for_statement_add_initializer (cfor, (ValaCCodeExpression*) (_tmp74_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp72_ = vala_ccode_identifier_new ("i")), (ValaCCodeExpression*) (_tmp73_ = vala_ccode_constant_new ("0")), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
	(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_ccode_node_unref (_tmp74_), NULL));
	(_tmp73_ == NULL) ? NULL : (_tmp73_ = (vala_ccode_node_unref (_tmp73_), NULL));
	(_tmp72_ == NULL) ? NULL : (_tmp72_ = (vala_ccode_node_unref (_tmp72_), NULL));
	_tmp76_ = NULL;
	_tmp75_ = NULL;
	vala_ccode_for_statement_add_iterator (cfor, (ValaCCodeExpression*) (_tmp76_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POSTFIX_INCREMENT, (ValaCCodeExpression*) (_tmp75_ = vala_ccode_identifier_new ("i")))));
	(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_ccode_node_unref (_tmp76_), NULL));
	(_tmp75_ == NULL) ? NULL : (_tmp75_ = (vala_ccode_node_unref (_tmp75_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) cfor);
	_tmp77_ = NULL;
	_tmp78_ = NULL;
	list_free_call = (_tmp78_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp77_ = vala_ccode_identifier_new ("dbus_free_string_array"))), (_tmp77_ == NULL) ? NULL : (_tmp77_ = (vala_ccode_node_unref (_tmp77_), NULL)), _tmp78_);
	_tmp79_ = NULL;
	vala_ccode_function_call_add_argument (list_free_call, (ValaCCodeExpression*) (_tmp79_ = vala_ccode_identifier_new ("children")));
	(_tmp79_ == NULL) ? NULL : (_tmp79_ = (vala_ccode_node_unref (_tmp79_), NULL));
	_tmp80_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp80_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) list_free_call)));
	(_tmp80_ == NULL) ? NULL : (_tmp80_ = (vala_ccode_node_unref (_tmp80_), NULL));
	_tmp81_ = NULL;
	xml_data = (_tmp81_ = g_strdup ("</node>\n"), xml_data = (g_free (xml_data), NULL), _tmp81_);
	_tmp83_ = NULL;
	_tmp82_ = NULL;
	str_call = (_tmp83_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp82_ = vala_ccode_identifier_new ("g_string_append"))), (str_call == NULL) ? NULL : (str_call = (vala_ccode_node_unref (str_call), NULL)), _tmp83_);
	(_tmp82_ == NULL) ? NULL : (_tmp82_ = (vala_ccode_node_unref (_tmp82_), NULL));
	_tmp84_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new ("xml_data")));
	(_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL));
	_tmp87_ = NULL;
	_tmp86_ = NULL;
	_tmp85_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp87_ = vala_ccode_constant_new (_tmp86_ = g_strdup_printf ("\"%s\"", _tmp85_ = g_strescape (xml_data, "")))));
	(_tmp87_ == NULL) ? NULL : (_tmp87_ = (vala_ccode_node_unref (_tmp87_), NULL));
	_tmp86_ = (g_free (_tmp86_), NULL);
	_tmp85_ = (g_free (_tmp85_), NULL);
	_tmp88_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp88_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) str_call)));
	(_tmp88_ == NULL) ? NULL : (_tmp88_ = (vala_ccode_node_unref (_tmp88_), NULL));
	_tmp90_ = NULL;
	_tmp89_ = NULL;
	iter_call = (_tmp90_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp89_ = vala_ccode_identifier_new ("dbus_message_iter_append_basic"))), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), _tmp90_);
	(_tmp89_ == NULL) ? NULL : (_tmp89_ = (vala_ccode_node_unref (_tmp89_), NULL));
	_tmp92_ = NULL;
	_tmp91_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp92_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp91_ = vala_ccode_identifier_new ("iter")))));
	(_tmp92_ == NULL) ? NULL : (_tmp92_ = (vala_ccode_node_unref (_tmp92_), NULL));
	(_tmp91_ == NULL) ? NULL : (_tmp91_ = (vala_ccode_node_unref (_tmp91_), NULL));
	_tmp93_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp93_ = vala_ccode_identifier_new ("DBUS_TYPE_STRING")));
	(_tmp93_ == NULL) ? NULL : (_tmp93_ = (vala_ccode_node_unref (_tmp93_), NULL));
	_tmp96_ = NULL;
	_tmp95_ = NULL;
	_tmp94_ = NULL;
	vala_ccode_function_call_add_argument (iter_call, (ValaCCodeExpression*) (_tmp96_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp95_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp94_ = vala_ccode_identifier_new ("xml_data")), "str")))));
	(_tmp96_ == NULL) ? NULL : (_tmp96_ = (vala_ccode_node_unref (_tmp96_), NULL));
	(_tmp95_ == NULL) ? NULL : (_tmp95_ = (vala_ccode_node_unref (_tmp95_), NULL));
	(_tmp94_ == NULL) ? NULL : (_tmp94_ = (vala_ccode_node_unref (_tmp94_), NULL));
	_tmp97_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp97_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) iter_call)));
	(_tmp97_ == NULL) ? NULL : (_tmp97_ = (vala_ccode_node_unref (_tmp97_), NULL));
	_tmp99_ = NULL;
	_tmp98_ = NULL;
	str_call = (_tmp99_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp98_ = vala_ccode_identifier_new ("g_string_free"))), (str_call == NULL) ? NULL : (str_call = (vala_ccode_node_unref (str_call), NULL)), _tmp99_);
	(_tmp98_ == NULL) ? NULL : (_tmp98_ = (vala_ccode_node_unref (_tmp98_), NULL));
	_tmp100_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp100_ = vala_ccode_identifier_new ("xml_data")));
	(_tmp100_ == NULL) ? NULL : (_tmp100_ = (vala_ccode_node_unref (_tmp100_), NULL));
	_tmp101_ = NULL;
	vala_ccode_function_call_add_argument (str_call, (ValaCCodeExpression*) (_tmp101_ = vala_ccode_constant_new ("TRUE")));
	(_tmp101_ == NULL) ? NULL : (_tmp101_ = (vala_ccode_node_unref (_tmp101_), NULL));
	_tmp102_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp102_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) str_call)));
	(_tmp102_ == NULL) ? NULL : (_tmp102_ = (vala_ccode_node_unref (_tmp102_), NULL));
	_tmp104_ = NULL;
	_tmp103_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp104_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp103_ = vala_ccode_identifier_new ("reply")))));
	(_tmp104_ == NULL) ? NULL : (_tmp104_ = (vala_ccode_node_unref (_tmp104_), NULL));
	(_tmp103_ == NULL) ? NULL : (_tmp103_ = (vala_ccode_node_unref (_tmp103_), NULL));
	_tmp105_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp105_ = vala_ccode_function_copy (function)));
	(_tmp105_ == NULL) ? NULL : (_tmp105_ = (vala_ccode_node_unref (_tmp105_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp106_ = NULL;
	return (_tmp106_ = wrapper_name, (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), (msgcall == NULL) ? NULL : (msgcall = (vala_ccode_node_unref (msgcall), NULL)), (iter_call == NULL) ? NULL : (iter_call = (vala_ccode_node_unref (iter_call), NULL)), xml_data = (g_free (xml_data), NULL), (str_call == NULL) ? NULL : (str_call = (vala_ccode_node_unref (str_call), NULL)), (get_path == NULL) ? NULL : (get_path = (vala_ccode_node_unref (get_path), NULL)), (list_call == NULL) ? NULL : (list_call = (vala_ccode_node_unref (list_call), NULL)), (child_block == NULL) ? NULL : (child_block = (vala_ccode_node_unref (child_block), NULL)), (cfor == NULL) ? NULL : (cfor = (vala_ccode_node_unref (cfor), NULL)), (list_free_call == NULL) ? NULL : (list_free_call = (vala_ccode_node_unref (list_free_call), NULL)), _tmp106_);
}


static void vala_dbus_server_module_handle_signals (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym, ValaCCodeBlock* block) {
	ValaAttribute* dbus;
	char* dbus_iface_name;
	g_return_if_fail (self != NULL);
	g_return_if_fail (sym != NULL);
	g_return_if_fail (block != NULL);
	dbus = vala_code_node_get_attribute ((ValaCodeNode*) sym, "DBus");
	if (dbus == NULL) {
		(dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL));
		return;
	}
	dbus_iface_name = vala_attribute_get_string (dbus, "name");
	if (dbus_iface_name == NULL) {
		(dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL));
		dbus_iface_name = (g_free (dbus_iface_name), NULL);
		return;
	}
	{
		GeeList* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _sig_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_sig_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = vala_object_type_symbol_get_signals (sym))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (gee_iterator_next (_sig_it)) {
			ValaSignal* sig;
			ValaCCodeIdentifier* _tmp2_;
			ValaCCodeFunctionCall* _tmp3_;
			ValaCCodeFunctionCall* connect;
			ValaCCodeIdentifier* _tmp4_;
			ValaCCodeConstant* _tmp5_;
			ValaCCodeCastExpression* _tmp8_;
			ValaCCodeIdentifier* _tmp7_;
			char* _tmp6_;
			ValaCCodeIdentifier* _tmp9_;
			ValaCCodeExpressionStatement* _tmp10_;
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			if (vala_symbol_get_access ((ValaSymbol*) sig) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				continue;
			}
			if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) sig)) {
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				continue;
			}
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			connect = (_tmp3_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp2_ = vala_ccode_identifier_new ("g_signal_connect"))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL)), _tmp3_);
			_tmp4_ = NULL;
			vala_ccode_function_call_add_argument (connect, (ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new ("object")));
			(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
			_tmp5_ = NULL;
			vala_ccode_function_call_add_argument (connect, (ValaCCodeExpression*) (_tmp5_ = vala_signal_get_canonical_cconstant (sig, NULL)));
			(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
			_tmp8_ = NULL;
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			vala_ccode_function_call_add_argument (connect, (ValaCCodeExpression*) (_tmp8_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (_tmp6_ = vala_dbus_server_module_generate_dbus_signal_wrapper (self, sig, sym, dbus_iface_name))), "GCallback")));
			(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			_tmp6_ = (g_free (_tmp6_), NULL);
			_tmp9_ = NULL;
			vala_ccode_function_call_add_argument (connect, (ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new ("connection")));
			(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
			_tmp10_ = NULL;
			vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp10_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) connect)));
			(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			(connect == NULL) ? NULL : (connect = (vala_ccode_node_unref (connect), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	(dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL));
	dbus_iface_name = (g_free (dbus_iface_name), NULL);
}


static void vala_dbus_server_module_generate_message_function (ValaDBusServerModule* self, ValaObjectType* object_type) {
	ValaObjectTypeSymbol* _tmp0_;
	ValaObjectTypeSymbol* sym;
	char* _tmp2_;
	char* _tmp1_;
	ValaCCodeFunction* _tmp3_;
	ValaCCodeFunction* cfunc;
	ValaCCodeFormalParameter* _tmp4_;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* _tmp6_;
	ValaCCodeFunction* _tmp7_;
	ValaCCodeBlock* block;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeVariableDeclarator* _tmp9_;
	ValaCCodeConstant* _tmp8_;
	ValaCCodeIfStatement* clastif;
	char* _tmp10_;
	ValaAttribute* dbus;
	ValaCCodeBlock* replyblock;
	ValaCCodeIdentifier* _tmp18_;
	ValaCCodeFunctionCall* _tmp19_;
	ValaCCodeFunctionCall* ccall;
	ValaCCodeIdentifier* _tmp20_;
	ValaCCodeIdentifier* _tmp21_;
	ValaCCodeConstant* _tmp22_;
	ValaCCodeExpressionStatement* _tmp23_;
	ValaCCodeFunctionCall* _tmp25_;
	ValaCCodeIdentifier* _tmp24_;
	ValaCCodeIdentifier* _tmp26_;
	ValaCCodeExpressionStatement* _tmp27_;
	ValaCCodeReturnStatement* _tmp29_;
	ValaCCodeIdentifier* _tmp28_;
	ValaCCodeIdentifier* _tmp30_;
	ValaCCodeIfStatement* _tmp31_;
	ValaCCodeIfStatement* cif;
	ValaCCodeIfStatement* _tmp33_;
	ValaCCodeIfStatement* _tmp32_;
	ValaClass* _tmp35_;
	ValaObjectTypeSymbol* _tmp34_;
	ValaClass* cl;
	ValaCCodeBlock* retblock;
	ValaCCodeReturnStatement* _tmp55_;
	ValaCCodeIdentifier* _tmp54_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (object_type != NULL);
	_tmp0_ = NULL;
	sym = (_tmp0_ = vala_object_type_get_type_symbol (object_type), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	((ValaCCodeBaseModule*) self)->dbus_glib_h_needed = TRUE;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp3_ = NULL;
	cfunc = (_tmp3_ = vala_ccode_function_new (_tmp2_ = g_strconcat (_tmp1_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), "dbus_message", NULL), "DBusHandlerResult"), _tmp2_ = (g_free (_tmp2_), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp3_);
	_tmp4_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp4_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp5_ = vala_ccode_formal_parameter_new ("message", "DBusMessage*"));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp6_ = NULL;
	vala_ccode_function_add_parameter (cfunc, _tmp6_ = vala_ccode_formal_parameter_new ("object", "void*"));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	if (vala_symbol_is_private_symbol ((ValaSymbol*) sym)) {
		vala_ccode_function_set_modifiers (cfunc, vala_ccode_function_get_modifiers (cfunc) | VALA_CCODE_MODIFIERS_STATIC);
	}
	_tmp7_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp7_ = vala_ccode_function_copy (cfunc)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	block = vala_ccode_block_new ();
	vala_ccode_function_set_block (cfunc, block);
	_cdecl_ = vala_ccode_declaration_new ("DBusMessage*");
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp9_ = vala_ccode_variable_declarator_new ("reply", (ValaCCodeExpression*) (_tmp8_ = vala_ccode_constant_new ("NULL")), NULL)));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
	clastif = NULL;
	_tmp10_ = NULL;
	vala_dbus_server_module_handle_method (self, "org.freedesktop.DBus.Introspectable", "Introspect", _tmp10_ = vala_dbus_server_module_generate_dbus_introspect (self, sym), block, &clastif);
	_tmp10_ = (g_free (_tmp10_), NULL);
	dbus = vala_code_node_get_attribute ((ValaCodeNode*) sym, "DBus");
	if (dbus != NULL) {
		char* dbus_iface_name;
		dbus_iface_name = vala_attribute_get_string (dbus, "name");
		if (dbus_iface_name != NULL) {
			gboolean need_property_get;
			gboolean need_property_set;
			char* _tmp17_;
			need_property_get = FALSE;
			need_property_set = FALSE;
			{
				GeeList* _tmp11_;
				GeeIterator* _tmp12_;
				GeeIterator* _prop_it;
				_tmp11_ = NULL;
				_tmp12_ = NULL;
				_prop_it = (_tmp12_ = gee_iterable_iterator ((GeeIterable*) (_tmp11_ = vala_object_type_symbol_get_properties (sym))), (_tmp11_ == NULL) ? NULL : (_tmp11_ = (gee_collection_object_unref (_tmp11_), NULL)), _tmp12_);
				while (gee_iterator_next (_prop_it)) {
					ValaProperty* prop;
					gboolean _tmp13_;
					gboolean _tmp14_;
					prop = (ValaProperty*) gee_iterator_get (_prop_it);
					_tmp13_ = FALSE;
					_tmp14_ = FALSE;
					if (vala_property_get_binding (prop) != MEMBER_BINDING_INSTANCE) {
						_tmp14_ = TRUE;
					} else {
						_tmp14_ = vala_property_get_overrides (prop);
					}
					if (_tmp14_) {
						_tmp13_ = TRUE;
					} else {
						_tmp13_ = vala_symbol_get_access ((ValaSymbol*) prop) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
					}
					if (_tmp13_) {
						(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						continue;
					}
					if (!vala_dbus_server_module_is_dbus_visible (self, (ValaCodeNode*) prop)) {
						(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						continue;
					}
					if (vala_property_get_get_accessor (prop) != NULL) {
						need_property_get = TRUE;
					}
					if (vala_property_get_set_accessor (prop) != NULL) {
						need_property_set = TRUE;
					}
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				}
				(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
			}
			if (need_property_get) {
				char* _tmp15_;
				_tmp15_ = NULL;
				vala_dbus_server_module_handle_method (self, "org.freedesktop.DBus.Properties", "Get", _tmp15_ = vala_dbus_server_module_generate_dbus_property_get_wrapper (self, sym, dbus_iface_name), block, &clastif);
				_tmp15_ = (g_free (_tmp15_), NULL);
			}
			if (need_property_set) {
				char* _tmp16_;
				_tmp16_ = NULL;
				vala_dbus_server_module_handle_method (self, "org.freedesktop.DBus.Properties", "Set", _tmp16_ = vala_dbus_server_module_generate_dbus_property_set_wrapper (self, sym, dbus_iface_name), block, &clastif);
				_tmp16_ = (g_free (_tmp16_), NULL);
			}
			_tmp17_ = NULL;
			vala_dbus_server_module_handle_method (self, "org.freedesktop.DBus.Properties", "GetAll", _tmp17_ = vala_dbus_server_module_generate_dbus_property_get_all_wrapper (self, sym, dbus_iface_name), block, &clastif);
			_tmp17_ = (g_free (_tmp17_), NULL);
			vala_dbus_server_module_handle_methods (self, sym, dbus_iface_name, block, &clastif);
		}
		dbus_iface_name = (g_free (dbus_iface_name), NULL);
	}
	replyblock = vala_ccode_block_new ();
	_tmp18_ = NULL;
	_tmp19_ = NULL;
	ccall = (_tmp19_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp18_ = vala_ccode_identifier_new ("dbus_connection_send"))), (_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL)), _tmp19_);
	_tmp20_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new ("connection")));
	(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
	_tmp21_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp21_ = vala_ccode_identifier_new ("reply")));
	(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
	_tmp22_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_constant_new ("NULL")));
	(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
	_tmp23_ = NULL;
	vala_ccode_block_add_statement (replyblock, (ValaCCodeNode*) (_tmp23_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccall)));
	(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	_tmp25_ = NULL;
	_tmp24_ = NULL;
	ccall = (_tmp25_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp24_ = vala_ccode_identifier_new ("dbus_message_unref"))), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), _tmp25_);
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	_tmp26_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp26_ = vala_ccode_identifier_new ("reply")));
	(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
	_tmp27_ = NULL;
	vala_ccode_block_add_statement (replyblock, (ValaCCodeNode*) (_tmp27_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccall)));
	(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
	_tmp29_ = NULL;
	_tmp28_ = NULL;
	vala_ccode_block_add_statement (replyblock, (ValaCCodeNode*) (_tmp29_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("DBUS_HANDLER_RESULT_HANDLED")))));
	(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
	(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL));
	_tmp30_ = NULL;
	_tmp31_ = NULL;
	cif = (_tmp31_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new ("reply")), (ValaCCodeStatement*) replyblock, NULL), (_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL)), _tmp31_);
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) cif);
	_tmp33_ = NULL;
	_tmp32_ = NULL;
	clastif = (_tmp33_ = (_tmp32_ = cif, (_tmp32_ == NULL) ? NULL : vala_ccode_node_ref (_tmp32_)), (clastif == NULL) ? NULL : (clastif = (vala_ccode_node_unref (clastif), NULL)), _tmp33_);
	_tmp35_ = NULL;
	_tmp34_ = NULL;
	cl = (_tmp35_ = (_tmp34_ = sym, VALA_IS_CLASS (_tmp34_) ? ((ValaClass*) _tmp34_) : NULL), (_tmp35_ == NULL) ? NULL : vala_code_node_ref (_tmp35_));
	if (cl != NULL) {
		{
			GeeList* _tmp36_;
			GeeIterator* _tmp37_;
			GeeIterator* _base_type_it;
			_tmp36_ = NULL;
			_tmp37_ = NULL;
			_base_type_it = (_tmp37_ = gee_iterable_iterator ((GeeIterable*) (_tmp36_ = vala_class_get_base_types (cl))), (_tmp36_ == NULL) ? NULL : (_tmp36_ = (gee_collection_object_unref (_tmp36_), NULL)), _tmp37_);
			while (gee_iterator_next (_base_type_it)) {
				ValaDataType* base_type;
				ValaObjectType* _tmp39_;
				ValaDataType* _tmp38_;
				ValaObjectType* base_obj_type;
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				_tmp39_ = NULL;
				_tmp38_ = NULL;
				base_obj_type = (_tmp39_ = (_tmp38_ = base_type, VALA_IS_OBJECT_TYPE (_tmp38_) ? ((ValaObjectType*) _tmp38_) : NULL), (_tmp39_ == NULL) ? NULL : vala_code_node_ref (_tmp39_));
				if (vala_dbus_server_module_type_implements_dbus_interface (self, vala_object_type_get_type_symbol (base_obj_type))) {
					ValaCCodeIdentifier* _tmp42_;
					char* _tmp41_;
					char* _tmp40_;
					ValaCCodeFunctionCall* _tmp43_;
					ValaCCodeFunctionCall* base_call;
					ValaCCodeIdentifier* _tmp44_;
					ValaCCodeIdentifier* _tmp45_;
					ValaCCodeIdentifier* _tmp46_;
					ValaCCodeIdentifier* _tmp47_;
					ValaCCodeBinaryExpression* _tmp48_;
					ValaCCodeBinaryExpression* ccheck;
					ValaCCodeBlock* base_block;
					ValaCCodeReturnStatement* _tmp50_;
					ValaCCodeIdentifier* _tmp49_;
					ValaCCodeIfStatement* _tmp51_;
					ValaCCodeIfStatement* _tmp53_;
					ValaCCodeIfStatement* _tmp52_;
					_tmp42_ = NULL;
					_tmp41_ = NULL;
					_tmp40_ = NULL;
					_tmp43_ = NULL;
					base_call = (_tmp43_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp42_ = vala_ccode_identifier_new (_tmp41_ = g_strconcat (_tmp40_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) vala_object_type_get_type_symbol (base_obj_type)), "dbus_message", NULL)))), (_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL)), _tmp41_ = (g_free (_tmp41_), NULL), _tmp40_ = (g_free (_tmp40_), NULL), _tmp43_);
					_tmp44_ = NULL;
					vala_ccode_function_call_add_argument (base_call, (ValaCCodeExpression*) (_tmp44_ = vala_ccode_identifier_new ("connection")));
					(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
					_tmp45_ = NULL;
					vala_ccode_function_call_add_argument (base_call, (ValaCCodeExpression*) (_tmp45_ = vala_ccode_identifier_new ("message")));
					(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
					_tmp46_ = NULL;
					vala_ccode_function_call_add_argument (base_call, (ValaCCodeExpression*) (_tmp46_ = vala_ccode_identifier_new ("object")));
					(_tmp46_ == NULL) ? NULL : (_tmp46_ = (vala_ccode_node_unref (_tmp46_), NULL));
					_tmp47_ = NULL;
					_tmp48_ = NULL;
					ccheck = (_tmp48_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) base_call, (ValaCCodeExpression*) (_tmp47_ = vala_ccode_identifier_new ("DBUS_HANDLER_RESULT_HANDLED"))), (_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL)), _tmp48_);
					base_block = vala_ccode_block_new ();
					_tmp50_ = NULL;
					_tmp49_ = NULL;
					vala_ccode_block_add_statement (base_block, (ValaCCodeNode*) (_tmp50_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp49_ = vala_ccode_identifier_new ("DBUS_HANDLER_RESULT_HANDLED")))));
					(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
					(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
					_tmp51_ = NULL;
					cif = (_tmp51_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) ccheck, (ValaCCodeStatement*) base_block, NULL), (cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL)), _tmp51_);
					vala_ccode_if_statement_set_false_statement (clastif, (ValaCCodeStatement*) cif);
					_tmp53_ = NULL;
					_tmp52_ = NULL;
					clastif = (_tmp53_ = (_tmp52_ = cif, (_tmp52_ == NULL) ? NULL : vala_ccode_node_ref (_tmp52_)), (clastif == NULL) ? NULL : (clastif = (vala_ccode_node_unref (clastif), NULL)), _tmp53_);
					(base_call == NULL) ? NULL : (base_call = (vala_ccode_node_unref (base_call), NULL));
					(ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL));
					(base_block == NULL) ? NULL : (base_block = (vala_ccode_node_unref (base_block), NULL));
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(base_obj_type == NULL) ? NULL : (base_obj_type = (vala_code_node_unref (base_obj_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
	}
	retblock = vala_ccode_block_new ();
	_tmp55_ = NULL;
	_tmp54_ = NULL;
	vala_ccode_block_add_statement (retblock, (ValaCCodeNode*) (_tmp55_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp54_ = vala_ccode_identifier_new ("DBUS_HANDLER_RESULT_NOT_YET_HANDLED")))));
	(_tmp55_ == NULL) ? NULL : (_tmp55_ = (vala_ccode_node_unref (_tmp55_), NULL));
	(_tmp54_ == NULL) ? NULL : (_tmp54_ = (vala_ccode_node_unref (_tmp54_), NULL));
	vala_ccode_if_statement_set_false_statement (clastif, (ValaCCodeStatement*) retblock);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) cfunc);
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
	(cfunc == NULL) ? NULL : (cfunc = (vala_ccode_node_unref (cfunc), NULL));
	(block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL));
	(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
	(clastif == NULL) ? NULL : (clastif = (vala_ccode_node_unref (clastif), NULL));
	(dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL));
	(replyblock == NULL) ? NULL : (replyblock = (vala_ccode_node_unref (replyblock), NULL));
	(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
	(cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL));
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	(retblock == NULL) ? NULL : (retblock = (vala_ccode_node_unref (retblock), NULL));
}


static ValaCCodeExpression* vala_dbus_server_module_get_vtable (ValaDBusServerModule* self, ValaObjectType* object_type) {
	ValaObjectTypeSymbol* _tmp0_;
	ValaObjectTypeSymbol* sym;
	ValaCCodeInitializerList* vtable;
	ValaCCodeIdentifier* _tmp3_;
	char* _tmp2_;
	char* _tmp1_;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeVariableDeclarator* _tmp7_;
	char* _tmp6_;
	char* _tmp5_;
	char* _tmp4_;
	char* _tmp10_;
	char* _tmp9_;
	char* _tmp8_;
	ValaCCodeExpression* _tmp11_;
	ValaCCodeExpression* _tmp12_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (object_type != NULL, NULL);
	_tmp0_ = NULL;
	sym = (_tmp0_ = vala_object_type_get_type_symbol (object_type), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	vtable = vala_ccode_initializer_list_new ();
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	vala_ccode_initializer_list_append (vtable, (ValaCCodeExpression*) (_tmp3_ = vala_ccode_identifier_new (_tmp2_ = g_strconcat (_tmp1_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), "dbus_register_object", NULL))));
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp1_ = (g_free (_tmp1_), NULL);
	vala_dbus_server_module_generate_register_function (self, object_type);
	_cdecl_ = vala_ccode_declaration_new ("const _DBusObjectVTable");
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp7_ = vala_ccode_variable_declarator_new (_tmp6_ = g_strconcat (_tmp5_ = g_strconcat ("_", _tmp4_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), NULL), "dbus_vtable", NULL), (ValaCCodeExpression*) vtable, NULL)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	_tmp6_ = (g_free (_tmp6_), NULL);
	_tmp5_ = (g_free (_tmp5_), NULL);
	_tmp4_ = (g_free (_tmp4_), NULL);
	vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
	vala_ccode_declaration_space_add_constant_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) _cdecl_);
	_tmp10_ = NULL;
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	_tmp11_ = NULL;
	_tmp12_ = NULL;
	return (_tmp12_ = (_tmp11_ = (ValaCCodeExpression*) vala_ccode_identifier_new (_tmp10_ = g_strconcat (_tmp9_ = g_strconcat ("_", _tmp8_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), NULL), "dbus_vtable", NULL)), _tmp10_ = (g_free (_tmp10_), NULL), _tmp9_ = (g_free (_tmp9_), NULL), _tmp8_ = (g_free (_tmp8_), NULL), _tmp11_), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (vtable == NULL) ? NULL : (vtable = (vala_ccode_node_unref (vtable), NULL)), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp12_);
}


static ValaCCodeExpression* vala_dbus_server_module_get_path_vtable (ValaDBusServerModule* self, ValaObjectType* object_type) {
	ValaObjectTypeSymbol* _tmp0_;
	ValaObjectTypeSymbol* sym;
	ValaCCodeInitializerList* vtable;
	ValaCCodeIdentifier* _tmp4_;
	char* _tmp3_;
	char* _tmp2_;
	char* _tmp1_;
	ValaCCodeIdentifier* _tmp7_;
	char* _tmp6_;
	char* _tmp5_;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeVariableDeclarator* _tmp11_;
	char* _tmp10_;
	char* _tmp9_;
	char* _tmp8_;
	char* _tmp14_;
	char* _tmp13_;
	char* _tmp12_;
	ValaCCodeExpression* _tmp15_;
	ValaCCodeExpression* _tmp16_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (object_type != NULL, NULL);
	_tmp0_ = NULL;
	sym = (_tmp0_ = vala_object_type_get_type_symbol (object_type), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	vtable = vala_ccode_initializer_list_new ();
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	vala_ccode_initializer_list_append (vtable, (ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new (_tmp3_ = g_strconcat (_tmp2_ = g_strconcat ("_", _tmp1_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), NULL), "dbus_unregister", NULL))));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	vala_ccode_initializer_list_append (vtable, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (_tmp6_ = g_strconcat (_tmp5_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), "dbus_message", NULL))));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	_tmp6_ = (g_free (_tmp6_), NULL);
	_tmp5_ = (g_free (_tmp5_), NULL);
	vala_dbus_server_module_generate_unregister_function (self, object_type);
	vala_dbus_server_module_generate_message_function (self, object_type);
	_cdecl_ = vala_ccode_declaration_new ("const DBusObjectPathVTable");
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp11_ = vala_ccode_variable_declarator_new (_tmp10_ = g_strconcat (_tmp9_ = g_strconcat ("_", _tmp8_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), NULL), "dbus_path_vtable", NULL), (ValaCCodeExpression*) vtable, NULL)));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	_tmp10_ = (g_free (_tmp10_), NULL);
	_tmp9_ = (g_free (_tmp9_), NULL);
	_tmp8_ = (g_free (_tmp8_), NULL);
	vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
	vala_ccode_declaration_space_add_constant_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) _cdecl_);
	_tmp14_ = NULL;
	_tmp13_ = NULL;
	_tmp12_ = NULL;
	_tmp15_ = NULL;
	_tmp16_ = NULL;
	return (_tmp16_ = (_tmp15_ = (ValaCCodeExpression*) vala_ccode_identifier_new (_tmp14_ = g_strconcat (_tmp13_ = g_strconcat ("_", _tmp12_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) sym), NULL), "dbus_path_vtable", NULL)), _tmp14_ = (g_free (_tmp14_), NULL), _tmp13_ = (g_free (_tmp13_), NULL), _tmp12_ = (g_free (_tmp12_), NULL), _tmp15_), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (vtable == NULL) ? NULL : (vtable = (vala_ccode_node_unref (vtable), NULL)), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp16_);
}


static void vala_dbus_server_module_real_visit_method_call (ValaCCodeModule* base, ValaMethodCall* expr) {
	ValaDBusServerModule * self;
	ValaMethodType* _tmp1_;
	ValaDataType* _tmp0_;
	ValaMethodType* mtype;
	gboolean _tmp2_;
	ValaMemberAccess* _tmp4_;
	ValaMemberAccess* ma;
	ValaCCodeIdentifier* _tmp5_;
	ValaCCodeFunctionCall* _tmp6_;
	ValaCCodeFunctionCall* raw_conn;
	GeeList* _tmp7_;
	GeeIterator* _tmp8_;
	GeeIterator* args_it;
	ValaExpression* path_arg;
	ValaExpression* obj_arg;
	ValaCCodeIdentifier* _tmp9_;
	ValaCCodeFunctionCall* _tmp10_;
	ValaCCodeFunctionCall* cregister;
	self = (ValaDBusServerModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	mtype = (_tmp1_ = (_tmp0_ = vala_expression_get_value_type (vala_method_call_get_call (expr)), VALA_IS_METHOD_TYPE (_tmp0_) ? ((ValaMethodType*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (mtype == NULL) {
		_tmp2_ = TRUE;
	} else {
		char* _tmp3_;
		_tmp3_ = NULL;
		_tmp2_ = _vala_strcmp0 (_tmp3_ = vala_method_get_cname (vala_method_type_get_method_symbol (mtype)), "dbus_g_connection_register_g_object") != 0;
		_tmp3_ = (g_free (_tmp3_), NULL);
	}
	if (_tmp2_) {
		VALA_CCODE_MODULE_CLASS (vala_dbus_server_module_parent_class)->visit_method_call ((ValaCCodeModule*) VALA_DBUS_CLIENT_MODULE (self), expr);
		(mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL));
		return;
	}
	((ValaCCodeBaseModule*) self)->dbus_glib_h_needed = TRUE;
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	_tmp4_ = NULL;
	ma = (_tmp4_ = VALA_MEMBER_ACCESS (vala_method_call_get_call (expr)), (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
	_tmp5_ = NULL;
	_tmp6_ = NULL;
	raw_conn = (_tmp6_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_identifier_new ("dbus_g_connection_get_connection"))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL)), _tmp6_);
	vala_ccode_function_call_add_argument (raw_conn, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_member_access_get_inner (ma))));
	_tmp7_ = NULL;
	_tmp8_ = NULL;
	args_it = (_tmp8_ = gee_iterable_iterator ((GeeIterable*) (_tmp7_ = vala_method_call_get_argument_list (expr))), (_tmp7_ == NULL) ? NULL : (_tmp7_ = (gee_collection_object_unref (_tmp7_), NULL)), _tmp8_);
	gee_iterator_next (args_it);
	path_arg = (ValaExpression*) gee_iterator_get (args_it);
	gee_iterator_next (args_it);
	obj_arg = (ValaExpression*) gee_iterator_get (args_it);
	_tmp9_ = NULL;
	_tmp10_ = NULL;
	cregister = (_tmp10_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new ("_vala_dbus_register_object"))), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL)), _tmp10_);
	vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) raw_conn);
	vala_ccode_function_call_add_argument (cregister, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) path_arg)));
	vala_ccode_function_call_add_argument (cregister, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) obj_arg)));
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) cregister);
	(mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL));
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	(raw_conn == NULL) ? NULL : (raw_conn = (vala_ccode_node_unref (raw_conn), NULL));
	(args_it == NULL) ? NULL : (args_it = (gee_collection_object_unref (args_it), NULL));
	(path_arg == NULL) ? NULL : (path_arg = (vala_code_node_unref (path_arg), NULL));
	(obj_arg == NULL) ? NULL : (obj_arg = (vala_code_node_unref (obj_arg), NULL));
	(cregister == NULL) ? NULL : (cregister = (vala_ccode_node_unref (cregister), NULL));
}


static gboolean vala_dbus_server_module_type_implements_dbus_interface (ValaDBusServerModule* self, ValaObjectTypeSymbol* sym) {
	ValaAttribute* dbus;
	ValaClass* _tmp2_;
	ValaObjectTypeSymbol* _tmp1_;
	ValaClass* cl;
	gboolean _tmp8_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (sym != NULL, FALSE);
	dbus = vala_code_node_get_attribute ((ValaCodeNode*) sym, "DBus");
	if (dbus != NULL) {
		gboolean _tmp0_;
		return (_tmp0_ = TRUE, (dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL)), _tmp0_);
	}
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	cl = (_tmp2_ = (_tmp1_ = sym, VALA_IS_CLASS (_tmp1_) ? ((ValaClass*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	if (cl != NULL) {
		{
			GeeList* _tmp3_;
			GeeIterator* _tmp4_;
			GeeIterator* _base_type_it;
			_tmp3_ = NULL;
			_tmp4_ = NULL;
			_base_type_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = vala_class_get_base_types (cl))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), _tmp4_);
			while (gee_iterator_next (_base_type_it)) {
				ValaDataType* base_type;
				ValaObjectType* _tmp6_;
				ValaDataType* _tmp5_;
				ValaObjectType* base_obj_type;
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				base_obj_type = (_tmp6_ = (_tmp5_ = base_type, VALA_IS_OBJECT_TYPE (_tmp5_) ? ((ValaObjectType*) _tmp5_) : NULL), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
				if (vala_dbus_server_module_type_implements_dbus_interface (self, vala_object_type_get_type_symbol (base_obj_type))) {
					gboolean _tmp7_;
					return (_tmp7_ = TRUE, (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), (base_obj_type == NULL) ? NULL : (base_obj_type = (vala_code_node_unref (base_obj_type), NULL)), (_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL)), (dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp7_);
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(base_obj_type == NULL) ? NULL : (base_obj_type = (vala_code_node_unref (base_obj_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
	}
	return (_tmp8_ = FALSE, (dbus == NULL) ? NULL : (dbus = (vala_code_node_unref (dbus), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp8_);
}


static ValaCCodeFragment* vala_dbus_server_module_real_register_dbus_info (ValaCCodeModule* base, ValaObjectTypeSymbol* sym) {
	ValaDBusServerModule * self;
	ValaCCodeFragment* fragment;
	ValaCCodeIdentifier* _tmp1_;
	ValaCCodeFunctionCall* _tmp2_;
	ValaCCodeFunctionCall* quark;
	ValaCCodeConstant* _tmp3_;
	ValaCCodeIdentifier* _tmp4_;
	ValaCCodeFunctionCall* _tmp5_;
	ValaCCodeFunctionCall* set_qdata;
	ValaCCodeIdentifier* _tmp7_;
	char* _tmp6_;
	ValaCCodeCastExpression* _tmp11_;
	ValaCCodeUnaryExpression* _tmp10_;
	ValaCCodeExpression* _tmp9_;
	ValaObjectType* _tmp8_;
	ValaCCodeExpressionStatement* _tmp12_;
	ValaCCodeFragment* _tmp13_;
	self = (ValaDBusServerModule*) base;
	g_return_val_if_fail (sym != NULL, NULL);
	fragment = vala_ccode_fragment_new ();
	if (!vala_dbus_server_module_type_implements_dbus_interface (self, sym)) {
		return fragment;
	}
	_tmp1_ = NULL;
	_tmp2_ = NULL;
	quark = (_tmp2_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp1_ = vala_ccode_identifier_new ("g_quark_from_static_string"))), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL)), _tmp2_);
	_tmp3_ = NULL;
	vala_ccode_function_call_add_argument (quark, (ValaCCodeExpression*) (_tmp3_ = vala_ccode_constant_new ("\"DBusObjectVTable\"")));
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
	_tmp4_ = NULL;
	_tmp5_ = NULL;
	set_qdata = (_tmp5_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new ("g_type_set_qdata"))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL)), _tmp5_);
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	vala_ccode_function_call_add_argument (set_qdata, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (_tmp6_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) sym, "TYPE_"))));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	_tmp6_ = (g_free (_tmp6_), NULL);
	vala_ccode_function_call_add_argument (set_qdata, (ValaCCodeExpression*) quark);
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	vala_ccode_function_call_add_argument (set_qdata, (ValaCCodeExpression*) (_tmp11_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) (_tmp10_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, _tmp9_ = vala_dbus_server_module_get_vtable (self, _tmp8_ = vala_object_type_new (sym)))), "void*")));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
	_tmp12_ = NULL;
	vala_ccode_fragment_append (fragment, (ValaCCodeNode*) (_tmp12_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) set_qdata)));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	_tmp13_ = NULL;
	return (_tmp13_ = fragment, (quark == NULL) ? NULL : (quark = (vala_ccode_node_unref (quark), NULL)), (set_qdata == NULL) ? NULL : (set_qdata = (vala_ccode_node_unref (set_qdata), NULL)), _tmp13_);
}


static void vala_dbus_server_module_class_init (ValaDBusServerModuleClass * klass) {
	vala_dbus_server_module_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_MODULE_CLASS (klass)->visit_method_call = vala_dbus_server_module_real_visit_method_call;
	VALA_CCODE_MODULE_CLASS (klass)->register_dbus_info = vala_dbus_server_module_real_register_dbus_info;
}


static void vala_dbus_server_module_instance_init (ValaDBusServerModule * self) {
}


GType vala_dbus_server_module_get_type (void) {
	static GType vala_dbus_server_module_type_id = 0;
	if (vala_dbus_server_module_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDBusServerModuleClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_dbus_server_module_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDBusServerModule), 0, (GInstanceInitFunc) vala_dbus_server_module_instance_init, NULL };
		vala_dbus_server_module_type_id = g_type_register_static (VALA_TYPE_DBUS_CLIENT_MODULE, "ValaDBusServerModule", &g_define_type_info, 0);
	}
	return vala_dbus_server_module_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




