/* valaccodefunction.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;
typedef struct _ValaCCodeNodePrivate ValaCCodeNodePrivate;

#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;

#define VALA_TYPE_CCODE_FUNCTION (vala_ccode_function_get_type ())
#define VALA_CCODE_FUNCTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_FUNCTION, ValaCCodeFunction))
#define VALA_CCODE_FUNCTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_FUNCTION, ValaCCodeFunctionClass))
#define VALA_IS_CCODE_FUNCTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_FUNCTION))
#define VALA_IS_CCODE_FUNCTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_FUNCTION))
#define VALA_CCODE_FUNCTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_FUNCTION, ValaCCodeFunctionClass))

typedef struct _ValaCCodeFunction ValaCCodeFunction;
typedef struct _ValaCCodeFunctionClass ValaCCodeFunctionClass;
typedef struct _ValaCCodeFunctionPrivate ValaCCodeFunctionPrivate;

#define VALA_TYPE_CCODE_MODIFIERS (vala_ccode_modifiers_get_type ())

#define VALA_TYPE_CCODE_STATEMENT (vala_ccode_statement_get_type ())
#define VALA_CCODE_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_STATEMENT, ValaCCodeStatement))
#define VALA_CCODE_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_STATEMENT, ValaCCodeStatementClass))
#define VALA_IS_CCODE_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_STATEMENT))
#define VALA_IS_CCODE_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_STATEMENT))
#define VALA_CCODE_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_STATEMENT, ValaCCodeStatementClass))

typedef struct _ValaCCodeStatement ValaCCodeStatement;
typedef struct _ValaCCodeStatementClass ValaCCodeStatementClass;

#define VALA_TYPE_CCODE_BLOCK (vala_ccode_block_get_type ())
#define VALA_CCODE_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_BLOCK, ValaCCodeBlock))
#define VALA_CCODE_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_BLOCK, ValaCCodeBlockClass))
#define VALA_IS_CCODE_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_BLOCK))
#define VALA_IS_CCODE_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_BLOCK))
#define VALA_CCODE_BLOCK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_BLOCK, ValaCCodeBlockClass))

typedef struct _ValaCCodeBlock ValaCCodeBlock;
typedef struct _ValaCCodeBlockClass ValaCCodeBlockClass;

#define VALA_TYPE_CCODE_FORMAL_PARAMETER (vala_ccode_formal_parameter_get_type ())
#define VALA_CCODE_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_FORMAL_PARAMETER, ValaCCodeFormalParameter))
#define VALA_CCODE_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_FORMAL_PARAMETER, ValaCCodeFormalParameterClass))
#define VALA_IS_CCODE_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_FORMAL_PARAMETER))
#define VALA_IS_CCODE_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_FORMAL_PARAMETER))
#define VALA_CCODE_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_FORMAL_PARAMETER, ValaCCodeFormalParameterClass))

typedef struct _ValaCCodeFormalParameter ValaCCodeFormalParameter;
typedef struct _ValaCCodeFormalParameterClass ValaCCodeFormalParameterClass;

#define VALA_TYPE_CCODE_LINE_DIRECTIVE (vala_ccode_line_directive_get_type ())
#define VALA_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirective))
#define VALA_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))
#define VALA_IS_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_IS_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_CCODE_LINE_DIRECTIVE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))

typedef struct _ValaCCodeLineDirective ValaCCodeLineDirective;
typedef struct _ValaCCodeLineDirectiveClass ValaCCodeLineDirectiveClass;

/**
 * Represents a node in the C code tree.
 */
struct _ValaCCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeNodePrivate * priv;
};

struct _ValaCCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeNode *self);
	void (*write) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_declaration) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_combined) (ValaCCodeNode* self, ValaCCodeWriter* writer);
};

/**
 * Represents a function declaration in the C code.
 */
struct _ValaCCodeFunction {
	ValaCCodeNode parent_instance;
	ValaCCodeFunctionPrivate * priv;
};

struct _ValaCCodeFunctionClass {
	ValaCCodeNodeClass parent_class;
};

typedef enum  {
	VALA_CCODE_MODIFIERS_NONE = 0,
	VALA_CCODE_MODIFIERS_STATIC = 1 << 0,
	VALA_CCODE_MODIFIERS_REGISTER = 1 << 1,
	VALA_CCODE_MODIFIERS_EXTERN = 1 << 2,
	VALA_CCODE_MODIFIERS_INLINE = 1 << 3,
	VALA_CCODE_MODIFIERS_VOLATILE = 1 << 4
} ValaCCodeModifiers;

struct _ValaCCodeFunctionPrivate {
	char* _name;
	ValaCCodeModifiers _modifiers;
	char* _return_type;
	ValaCCodeBlock* _block;
	GeeList* parameters;
};



gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
GType vala_ccode_function_get_type (void);
GType vala_ccode_modifiers_get_type (void);
GType vala_ccode_statement_get_type (void);
GType vala_ccode_block_get_type (void);
GType vala_ccode_formal_parameter_get_type (void);
#define VALA_CCODE_FUNCTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_FUNCTION, ValaCCodeFunctionPrivate))
enum  {
	VALA_CCODE_FUNCTION_DUMMY_PROPERTY
};
void vala_ccode_function_set_name (ValaCCodeFunction* self, const char* value);
void vala_ccode_function_set_return_type (ValaCCodeFunction* self, const char* value);
ValaCCodeFunction* vala_ccode_function_new (const char* name, const char* return_type);
ValaCCodeFunction* vala_ccode_function_construct (GType object_type, const char* name, const char* return_type);
ValaCCodeFunction* vala_ccode_function_new (const char* name, const char* return_type);
void vala_ccode_function_add_parameter (ValaCCodeFunction* self, ValaCCodeFormalParameter* param);
const char* vala_ccode_function_get_name (ValaCCodeFunction* self);
const char* vala_ccode_function_get_return_type (ValaCCodeFunction* self);
ValaCCodeModifiers vala_ccode_function_get_modifiers (ValaCCodeFunction* self);
void vala_ccode_function_set_modifiers (ValaCCodeFunction* self, ValaCCodeModifiers value);
ValaCCodeBlock* vala_ccode_function_get_block (ValaCCodeFunction* self);
void vala_ccode_function_set_block (ValaCCodeFunction* self, ValaCCodeBlock* value);
ValaCCodeFunction* vala_ccode_function_copy (ValaCCodeFunction* self);
GType vala_ccode_line_directive_get_type (void);
void vala_ccode_writer_write_indent (ValaCCodeWriter* self, ValaCCodeLineDirective* line);
ValaCCodeLineDirective* vala_ccode_node_get_line (ValaCCodeNode* self);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
void vala_ccode_writer_write_newline (ValaCCodeWriter* self);
static void vala_ccode_function_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_function_parent_class = NULL;
static void vala_ccode_function_finalize (ValaCCodeNode* obj);



ValaCCodeFunction* vala_ccode_function_construct (GType object_type, const char* name, const char* return_type) {
	ValaCCodeFunction* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (return_type != NULL, NULL);
	self = (ValaCCodeFunction*) g_type_create_instance (object_type);
	vala_ccode_function_set_name (self, name);
	vala_ccode_function_set_return_type (self, return_type);
	return self;
}


ValaCCodeFunction* vala_ccode_function_new (const char* name, const char* return_type) {
	return vala_ccode_function_construct (VALA_TYPE_CCODE_FUNCTION, name, return_type);
}


/**
 * Appends the specified parameter to the list of function parameters.
 *
 * @param param a formal parameter
 */
void vala_ccode_function_add_parameter (ValaCCodeFunction* self, ValaCCodeFormalParameter* param) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	gee_collection_add ((GeeCollection*) self->priv->parameters, param);
}


/**
 * Returns a copy of this function.
 *
 * @return copied function
 */
ValaCCodeFunction* vala_ccode_function_copy (ValaCCodeFunction* self) {
	ValaCCodeFunction* func;
	g_return_val_if_fail (self != NULL, NULL);
	func = vala_ccode_function_new (self->priv->_name, self->priv->_return_type);
	vala_ccode_function_set_modifiers (func, self->priv->_modifiers);
	/* no deep copy for lists available yet
	 * func.parameters = parameters.copy ();
	 */
	{
		GeeIterator* _param_it;
		/* no deep copy for lists available yet
		 * func.parameters = parameters.copy ();
		 */
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		/* no deep copy for lists available yet
		 * func.parameters = parameters.copy ();
		 */
		while (gee_iterator_next (_param_it)) {
			ValaCCodeFormalParameter* param;
			/* no deep copy for lists available yet
			 * func.parameters = parameters.copy ();
			 */
			param = (ValaCCodeFormalParameter*) gee_iterator_get (_param_it);
			gee_collection_add ((GeeCollection*) func->priv->parameters, param);
			(param == NULL) ? NULL : (param = (vala_ccode_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	vala_ccode_function_set_block (func, self->priv->_block);
	return func;
}


static void vala_ccode_function_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeFunction * self;
	gboolean first;
	self = (ValaCCodeFunction*) base;
	g_return_if_fail (writer != NULL);
	vala_ccode_writer_write_indent (writer, vala_ccode_node_get_line ((ValaCCodeNode*) self));
	if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_STATIC) == VALA_CCODE_MODIFIERS_STATIC) {
		vala_ccode_writer_write_string (writer, "static ");
	}
	if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_INLINE) == VALA_CCODE_MODIFIERS_INLINE) {
		vala_ccode_writer_write_string (writer, "inline ");
	}
	vala_ccode_writer_write_string (writer, self->priv->_return_type);
	vala_ccode_writer_write_string (writer, " ");
	vala_ccode_writer_write_string (writer, self->priv->_name);
	vala_ccode_writer_write_string (writer, " (");
	first = TRUE;
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaCCodeFormalParameter* param;
			param = (ValaCCodeFormalParameter*) gee_iterator_get (_param_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ", ");
			} else {
				first = FALSE;
			}
			vala_ccode_node_write ((ValaCCodeNode*) param, writer);
			(param == NULL) ? NULL : (param = (vala_ccode_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	if (first) {
		vala_ccode_writer_write_string (writer, "void");
	}
	vala_ccode_writer_write_string (writer, ")");
	if (self->priv->_block == NULL) {
		vala_ccode_writer_write_string (writer, ";");
	} else {
		vala_ccode_node_write ((ValaCCodeNode*) self->priv->_block, writer);
		vala_ccode_writer_write_newline (writer);
	}
	vala_ccode_writer_write_newline (writer);
}


const char* vala_ccode_function_get_name (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


void vala_ccode_function_set_name (ValaCCodeFunction* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_name = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_name = (g_free (self->priv->_name), NULL), _tmp2_);
}


ValaCCodeModifiers vala_ccode_function_get_modifiers (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_modifiers;
}


void vala_ccode_function_set_modifiers (ValaCCodeFunction* self, ValaCCodeModifiers value) {
	g_return_if_fail (self != NULL);
	self->priv->_modifiers = value;
}


const char* vala_ccode_function_get_return_type (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_return_type;
}


void vala_ccode_function_set_return_type (ValaCCodeFunction* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_return_type = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_return_type = (g_free (self->priv->_return_type), NULL), _tmp2_);
}


ValaCCodeBlock* vala_ccode_function_get_block (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_block;
}


void vala_ccode_function_set_block (ValaCCodeFunction* self, ValaCCodeBlock* value) {
	ValaCCodeBlock* _tmp2_;
	ValaCCodeBlock* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_block = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_)), (self->priv->_block == NULL) ? NULL : (self->priv->_block = (vala_ccode_node_unref (self->priv->_block), NULL)), _tmp2_);
}


static void vala_ccode_function_class_init (ValaCCodeFunctionClass * klass) {
	vala_ccode_function_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_function_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeFunctionPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_function_real_write;
}


static void vala_ccode_function_instance_init (ValaCCodeFunction * self) {
	self->priv = VALA_CCODE_FUNCTION_GET_PRIVATE (self);
	self->priv->parameters = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_ccode_function_finalize (ValaCCodeNode* obj) {
	ValaCCodeFunction * self;
	self = VALA_CCODE_FUNCTION (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	self->priv->_return_type = (g_free (self->priv->_return_type), NULL);
	(self->priv->_block == NULL) ? NULL : (self->priv->_block = (vala_ccode_node_unref (self->priv->_block), NULL));
	(self->priv->parameters == NULL) ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_function_parent_class)->finalize (obj);
}


GType vala_ccode_function_get_type (void) {
	static GType vala_ccode_function_type_id = 0;
	if (vala_ccode_function_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeFunctionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_function_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeFunction), 0, (GInstanceInitFunc) vala_ccode_function_instance_init, NULL };
		vala_ccode_function_type_id = g_type_register_static (VALA_TYPE_CCODE_NODE, "ValaCCodeFunction", &g_define_type_info, 0);
	}
	return vala_ccode_function_type_id;
}




