/* valaconstant.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;
typedef struct _ValaMemberPrivate ValaMemberPrivate;

#define VALA_TYPE_LOCKABLE (vala_lockable_get_type ())
#define VALA_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCKABLE, ValaLockable))
#define VALA_IS_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCKABLE))
#define VALA_LOCKABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_LOCKABLE, ValaLockableIface))

typedef struct _ValaLockable ValaLockable;
typedef struct _ValaLockableIface ValaLockableIface;

#define VALA_TYPE_CONSTANT (vala_constant_get_type ())
#define VALA_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CONSTANT, ValaConstant))
#define VALA_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CONSTANT, ValaConstantClass))
#define VALA_IS_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CONSTANT))
#define VALA_IS_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CONSTANT))
#define VALA_CONSTANT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CONSTANT, ValaConstantClass))

typedef struct _ValaConstant ValaConstant;
typedef struct _ValaConstantClass ValaConstantClass;
typedef struct _ValaConstantPrivate ValaConstantPrivate;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents a general class member.
 */
struct _ValaMember {
	ValaSymbol parent_instance;
	ValaMemberPrivate * priv;
};

struct _ValaMemberClass {
	ValaSymbolClass parent_class;
};

/**
 * Represents a lockable object.
 */
struct _ValaLockableIface {
	GTypeInterface parent_iface;
	gboolean (*get_lock_used) (ValaLockable* self);
	void (*set_lock_used) (ValaLockable* self, gboolean used);
};

/**
 * Represents a type member with a constant value.
 */
struct _ValaConstant {
	ValaMember parent_instance;
	ValaConstantPrivate * priv;
};

struct _ValaConstantClass {
	ValaMemberClass parent_class;
};

struct _ValaConstantPrivate {
	char* cname;
	gboolean lock_used;
	ValaDataType* _data_type;
	ValaExpression* _initializer;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_member_get_type (void);
GType vala_lockable_get_type (void);
GType vala_constant_get_type (void);
#define VALA_CONSTANT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CONSTANT, ValaConstantPrivate))
enum  {
	VALA_CONSTANT_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaMember* vala_member_new (const char* name, ValaSourceReference* source_reference);
ValaMember* vala_member_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
void vala_constant_set_type_reference (ValaConstant* self, ValaDataType* value);
void vala_constant_set_initializer (ValaConstant* self, ValaExpression* value);
ValaConstant* vala_constant_new (const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference);
ValaConstant* vala_constant_construct (GType object_type, const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference);
ValaConstant* vala_constant_new (const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference);
void vala_code_visitor_visit_member (ValaCodeVisitor* self, ValaMember* m);
void vala_code_visitor_visit_constant (ValaCodeVisitor* self, ValaConstant* c);
static void vala_constant_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaDataType* vala_constant_get_type_reference (ValaConstant* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaExpression* vala_constant_get_initializer (ValaConstant* self);
static void vala_constant_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
char* vala_constant_get_default_cname (ValaConstant* self);
char* vala_constant_get_cname (ValaConstant* self);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
const char* vala_symbol_get_name (ValaSymbol* self);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
static gboolean vala_constant_real_get_lock_used (ValaLockable* base);
static void vala_constant_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_constant_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
void vala_member_add_cheader_filename (ValaMember* self, const char* filename);
static void vala_constant_process_ccode_attribute (ValaConstant* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_constant_process_attributes (ValaConstant* self);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_semantic_analyzer_get_current_source_file (ValaSemanticAnalyzer* self);
ValaSymbol* vala_semantic_analyzer_get_current_symbol (ValaSemanticAnalyzer* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
void vala_semantic_analyzer_set_current_source_file (ValaSemanticAnalyzer* self, ValaSourceFile* value);
void vala_semantic_analyzer_set_current_symbol (ValaSemanticAnalyzer* self, ValaSymbol* value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
gboolean vala_symbol_get_external (ValaSymbol* self);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
void vala_report_error (ValaSourceReference* source, const char* message);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
gboolean vala_symbol_get_external_package (ValaSymbol* self);
gboolean vala_member_get_hides (ValaMember* self);
ValaSymbol* vala_member_get_hidden_member (ValaMember* self);
void vala_report_warning (ValaSourceReference* source, const char* message);
char* vala_symbol_get_full_name (ValaSymbol* self);
static gboolean vala_constant_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
static gpointer vala_constant_parent_class = NULL;
static ValaLockableIface* vala_constant_vala_lockable_parent_iface = NULL;
static void vala_constant_finalize (ValaCodeNode* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new constant.
 *
 * @param name             constant name
 * @param type_reference   constant type
 * @param initializer      constant value
 * @param source_reference reference to source code
 * @return                 newly created constant
 */
ValaConstant* vala_constant_construct (GType object_type, const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaConstant* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (type_reference != NULL, NULL);
	self = (ValaConstant*) vala_member_construct (object_type, name, source_reference);
	vala_constant_set_type_reference (self, type_reference);
	vala_constant_set_initializer (self, initializer);
	return self;
}


ValaConstant* vala_constant_new (const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	return vala_constant_construct (VALA_TYPE_CONSTANT, name, type_reference, initializer, source_reference);
}


static void vala_constant_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstant * self;
	self = (ValaConstant*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_member (visitor, (ValaMember*) self);
	vala_code_visitor_visit_constant (visitor, self);
}


static void vala_constant_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstant * self;
	self = (ValaConstant*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_constant_get_type_reference (self), visitor);
	if (vala_constant_get_initializer (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_constant_get_initializer (self), visitor);
	}
}


/**
 * Returns the name of this constant as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_constant_get_cname (ValaConstant* self) {
	const char* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0_;
		_tmp0_ = NULL;
		self->priv->cname = (_tmp0_ = vala_constant_get_default_cname (self), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0_);
	}
	_tmp1_ = NULL;
	return (_tmp1_ = self->priv->cname, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
}


/**
 * Returns the default name of this constant as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
char* vala_constant_get_default_cname (ValaConstant* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_symbol_get_parent_symbol ((ValaSymbol*) self) == NULL) {
		const char* _tmp0_;
		/* global constant*/
		_tmp0_ = NULL;
		return (_tmp0_ = vala_symbol_get_name ((ValaSymbol*) self), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
	} else {
		char* _tmp3_;
		char* _tmp2_;
		char* _tmp4_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_tmp4_ = NULL;
		return (_tmp4_ = g_strdup_printf ("%s%s", _tmp3_ = g_utf8_strup (_tmp2_ = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), -1), vala_symbol_get_name ((ValaSymbol*) self)), _tmp3_ = (g_free (_tmp3_), NULL), _tmp2_ = (g_free (_tmp2_), NULL), _tmp4_);
	}
}


static gboolean vala_constant_real_get_lock_used (ValaLockable* base) {
	ValaConstant * self;
	self = (ValaConstant*) base;
	return self->priv->lock_used;
}


static void vala_constant_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaConstant * self;
	self = (ValaConstant*) base;
	self->priv->lock_used = used;
}


static void vala_constant_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaConstant * self;
	self = (ValaConstant*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_constant_get_type_reference (self) == old_type) {
		vala_constant_set_type_reference (self, new_type);
	}
}


static void vala_constant_process_ccode_attribute (ValaConstant* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0_;
		_tmp0_ = NULL;
		self->priv->cname = (_tmp0_ = vala_attribute_get_string (a, "cname"), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0_);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp1_;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp1_ = NULL;
			filename_collection = _tmp1_ = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp1_);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp1_); filename_it = filename_it + 1) {
				const char* _tmp2_;
				char* filename;
				_tmp2_ = NULL;
				filename = (_tmp2_ = filename_collection[filename_it], (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
				{
					vala_member_add_cheader_filename ((ValaMember*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_constant_process_attributes (ValaConstant* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0_;
			ValaAttribute* a;
			_tmp0_ = NULL;
			a = (_tmp0_ = (ValaAttribute*) a_it->data, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_constant_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_constant_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaConstant * self;
	ValaSourceFile* _tmp1_;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2_;
	ValaSymbol* old_symbol;
	gboolean _tmp3_;
	gboolean _tmp4_;
	gboolean _tmp10_;
	self = (ValaConstant*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_constant_process_attributes (self);
	_tmp1_ = NULL;
	old_source_file = (_tmp1_ = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1_ == NULL) ? NULL : vala_source_file_ref (_tmp1_));
	_tmp2_ = NULL;
	old_symbol = (_tmp2_ = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	vala_code_node_check ((ValaCodeNode*) vala_constant_get_type_reference (self), analyzer);
	if (!vala_symbol_get_external ((ValaSymbol*) self)) {
		if (vala_constant_get_initializer (self) == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "A const field requires a initializer to be provided");
		} else {
			vala_expression_set_target_type (vala_constant_get_initializer (self), vala_constant_get_type_reference (self));
			vala_code_node_check ((ValaCodeNode*) vala_constant_get_initializer (self), analyzer);
		}
	}
	_tmp3_ = FALSE;
	_tmp4_ = FALSE;
	if (!vala_symbol_get_external_package ((ValaSymbol*) self)) {
		_tmp4_ = !vala_member_get_hides ((ValaMember*) self);
	} else {
		_tmp4_ = FALSE;
	}
	if (_tmp4_) {
		ValaSymbol* _tmp5_;
		_tmp5_ = NULL;
		_tmp3_ = (_tmp5_ = vala_member_get_hidden_member ((ValaMember*) self)) != NULL;
		(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		char* _tmp9_;
		char* _tmp8_;
		ValaSymbol* _tmp7_;
		char* _tmp6_;
		_tmp9_ = NULL;
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		_tmp6_ = NULL;
		vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp9_ = g_strdup_printf ("%s hides inherited constant `%s'. Use the `new' keyword if hiding was intentional", _tmp6_ = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp8_ = vala_symbol_get_full_name (_tmp7_ = vala_member_get_hidden_member ((ValaMember*) self))));
		_tmp9_ = (g_free (_tmp9_), NULL);
		_tmp8_ = (g_free (_tmp8_), NULL);
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
		_tmp6_ = (g_free (_tmp6_), NULL);
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	return (_tmp10_ = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), _tmp10_);
}


ValaDataType* vala_constant_get_type_reference (ValaConstant* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_constant_set_type_reference (ValaConstant* self, ValaDataType* value) {
	ValaDataType* _tmp2_;
	ValaDataType* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_data_type = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)), _tmp2_);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
}


ValaExpression* vala_constant_get_initializer (ValaConstant* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_initializer;
}


void vala_constant_set_initializer (ValaConstant* self, ValaExpression* value) {
	ValaExpression* _tmp2_;
	ValaExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_initializer = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_initializer == NULL) ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL)), _tmp2_);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_initializer, (ValaCodeNode*) self);
	}
}


static void vala_constant_class_init (ValaConstantClass * klass) {
	vala_constant_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_constant_finalize;
	g_type_class_add_private (klass, sizeof (ValaConstantPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_constant_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_constant_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_constant_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_constant_real_check;
}


static void vala_constant_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_constant_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_constant_real_get_lock_used;
	iface->set_lock_used = vala_constant_real_set_lock_used;
}


static void vala_constant_instance_init (ValaConstant * self) {
	self->priv = VALA_CONSTANT_GET_PRIVATE (self);
	self->priv->lock_used = FALSE;
}


static void vala_constant_finalize (ValaCodeNode* obj) {
	ValaConstant * self;
	self = VALA_CONSTANT (obj);
	self->priv->cname = (g_free (self->priv->cname), NULL);
	(self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL));
	(self->priv->_initializer == NULL) ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL));
	VALA_CODE_NODE_CLASS (vala_constant_parent_class)->finalize (obj);
}


GType vala_constant_get_type (void) {
	static GType vala_constant_type_id = 0;
	if (vala_constant_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaConstantClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_constant_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaConstant), 0, (GInstanceInitFunc) vala_constant_instance_init, NULL };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_constant_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_constant_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaConstant", &g_define_type_info, 0);
		g_type_add_interface_static (vala_constant_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_constant_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




