/* valaccodestructmodule.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdlib.h>
#include <string.h>
#include <valaccode.h>
#include <gee.h>


#define VALA_TYPE_CCODE_MODULE (vala_ccode_module_get_type ())
#define VALA_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModule))
#define VALA_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))
#define VALA_IS_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_MODULE))
#define VALA_IS_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_MODULE))
#define VALA_CCODE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))

typedef struct _ValaCCodeModule ValaCCodeModule;
typedef struct _ValaCCodeModuleClass ValaCCodeModuleClass;
typedef struct _ValaCCodeModulePrivate ValaCCodeModulePrivate;

#define VALA_TYPE_CCODE_BASE_MODULE (vala_ccode_base_module_get_type ())
#define VALA_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModule))
#define VALA_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))
#define VALA_IS_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_IS_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_CCODE_BASE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))

typedef struct _ValaCCodeBaseModule ValaCCodeBaseModule;
typedef struct _ValaCCodeBaseModuleClass ValaCCodeBaseModuleClass;
typedef struct _ValaCCodeBaseModulePrivate ValaCCodeBaseModulePrivate;

#define VALA_TYPE_CCODE_DECLARATION_SPACE (vala_ccode_declaration_space_get_type ())
#define VALA_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpace))
#define VALA_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))
#define VALA_IS_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_IS_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_CCODE_DECLARATION_SPACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))

typedef struct _ValaCCodeDeclarationSpace ValaCCodeDeclarationSpace;
typedef struct _ValaCCodeDeclarationSpaceClass ValaCCodeDeclarationSpaceClass;

#define VALA_TYPE_CCODE_STRUCT_MODULE (vala_ccode_struct_module_get_type ())
#define VALA_CCODE_STRUCT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModule))
#define VALA_CCODE_STRUCT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModuleClass))
#define VALA_IS_CCODE_STRUCT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_STRUCT_MODULE))
#define VALA_IS_CCODE_STRUCT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_STRUCT_MODULE))
#define VALA_CCODE_STRUCT_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModuleClass))

typedef struct _ValaCCodeStructModule ValaCCodeStructModule;
typedef struct _ValaCCodeStructModuleClass ValaCCodeStructModuleClass;
typedef struct _ValaCCodeStructModulePrivate ValaCCodeStructModulePrivate;

#define VALA_TYPE_CCODE_GENERATOR (vala_ccode_generator_get_type ())
#define VALA_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGenerator))
#define VALA_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))
#define VALA_IS_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_GENERATOR))
#define VALA_IS_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_GENERATOR))
#define VALA_CCODE_GENERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))

typedef struct _ValaCCodeGenerator ValaCCodeGenerator;
typedef struct _ValaCCodeGeneratorClass ValaCCodeGeneratorClass;

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeModule {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeModulePrivate * priv;
};

struct _ValaCCodeModuleClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeModule *self);
	void (*emit) (ValaCCodeModule* self, ValaCodeContext* context);
	void (*visit_source_file) (ValaCCodeModule* self, ValaSourceFile* source_file);
	void (*visit_class) (ValaCCodeModule* self, ValaClass* cl);
	void (*visit_interface) (ValaCCodeModule* self, ValaInterface* iface);
	void (*visit_struct) (ValaCCodeModule* self, ValaStruct* st);
	void (*visit_enum) (ValaCCodeModule* self, ValaEnum* en);
	void (*visit_error_domain) (ValaCCodeModule* self, ValaErrorDomain* edomain);
	void (*visit_delegate) (ValaCCodeModule* self, ValaDelegate* d);
	void (*visit_member) (ValaCCodeModule* self, ValaMember* m);
	void (*visit_constant) (ValaCCodeModule* self, ValaConstant* c);
	void (*visit_field) (ValaCCodeModule* self, ValaField* f);
	void (*visit_method) (ValaCCodeModule* self, ValaMethod* m);
	void (*visit_creation_method) (ValaCCodeModule* self, ValaCreationMethod* m);
	void (*visit_formal_parameter) (ValaCCodeModule* self, ValaFormalParameter* p);
	void (*visit_property) (ValaCCodeModule* self, ValaProperty* prop);
	void (*visit_property_accessor) (ValaCCodeModule* self, ValaPropertyAccessor* acc);
	void (*visit_signal) (ValaCCodeModule* self, ValaSignal* sig);
	void (*visit_constructor) (ValaCCodeModule* self, ValaConstructor* c);
	void (*visit_destructor) (ValaCCodeModule* self, ValaDestructor* d);
	void (*visit_block) (ValaCCodeModule* self, ValaBlock* b);
	void (*visit_empty_statement) (ValaCCodeModule* self, ValaEmptyStatement* stmt);
	void (*visit_declaration_statement) (ValaCCodeModule* self, ValaDeclarationStatement* stmt);
	void (*visit_local_variable) (ValaCCodeModule* self, ValaLocalVariable* local);
	void (*visit_initializer_list) (ValaCCodeModule* self, ValaInitializerList* list);
	void (*visit_end_full_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_expression_statement) (ValaCCodeModule* self, ValaExpressionStatement* stmt);
	void (*visit_if_statement) (ValaCCodeModule* self, ValaIfStatement* stmt);
	void (*visit_switch_statement) (ValaCCodeModule* self, ValaSwitchStatement* stmt);
	void (*visit_switch_section) (ValaCCodeModule* self, ValaSwitchSection* section);
	void (*visit_switch_label) (ValaCCodeModule* self, ValaSwitchLabel* label);
	void (*visit_while_statement) (ValaCCodeModule* self, ValaWhileStatement* stmt);
	void (*visit_do_statement) (ValaCCodeModule* self, ValaDoStatement* stmt);
	void (*visit_for_statement) (ValaCCodeModule* self, ValaForStatement* stmt);
	void (*visit_foreach_statement) (ValaCCodeModule* self, ValaForeachStatement* stmt);
	void (*visit_break_statement) (ValaCCodeModule* self, ValaBreakStatement* stmt);
	void (*visit_continue_statement) (ValaCCodeModule* self, ValaContinueStatement* stmt);
	void (*visit_return_statement) (ValaCCodeModule* self, ValaReturnStatement* stmt);
	void (*visit_yield_statement) (ValaCCodeModule* self, ValaYieldStatement* stmt);
	void (*visit_throw_statement) (ValaCCodeModule* self, ValaThrowStatement* stmt);
	void (*visit_try_statement) (ValaCCodeModule* self, ValaTryStatement* stmt);
	void (*visit_catch_clause) (ValaCCodeModule* self, ValaCatchClause* clause);
	void (*visit_lock_statement) (ValaCCodeModule* self, ValaLockStatement* stmt);
	void (*visit_delete_statement) (ValaCCodeModule* self, ValaDeleteStatement* stmt);
	void (*visit_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_array_creation_expression) (ValaCCodeModule* self, ValaArrayCreationExpression* expr);
	void (*visit_boolean_literal) (ValaCCodeModule* self, ValaBooleanLiteral* expr);
	void (*visit_character_literal) (ValaCCodeModule* self, ValaCharacterLiteral* expr);
	void (*visit_integer_literal) (ValaCCodeModule* self, ValaIntegerLiteral* expr);
	void (*visit_real_literal) (ValaCCodeModule* self, ValaRealLiteral* expr);
	void (*visit_string_literal) (ValaCCodeModule* self, ValaStringLiteral* expr);
	void (*visit_null_literal) (ValaCCodeModule* self, ValaNullLiteral* expr);
	void (*visit_member_access) (ValaCCodeModule* self, ValaMemberAccess* expr);
	void (*visit_method_call) (ValaCCodeModule* self, ValaMethodCall* expr);
	void (*visit_element_access) (ValaCCodeModule* self, ValaElementAccess* expr);
	void (*visit_base_access) (ValaCCodeModule* self, ValaBaseAccess* expr);
	void (*visit_postfix_expression) (ValaCCodeModule* self, ValaPostfixExpression* expr);
	void (*visit_object_creation_expression) (ValaCCodeModule* self, ValaObjectCreationExpression* expr);
	void (*visit_sizeof_expression) (ValaCCodeModule* self, ValaSizeofExpression* expr);
	void (*visit_typeof_expression) (ValaCCodeModule* self, ValaTypeofExpression* expr);
	void (*visit_unary_expression) (ValaCCodeModule* self, ValaUnaryExpression* expr);
	void (*visit_cast_expression) (ValaCCodeModule* self, ValaCastExpression* expr);
	void (*visit_pointer_indirection) (ValaCCodeModule* self, ValaPointerIndirection* expr);
	void (*visit_addressof_expression) (ValaCCodeModule* self, ValaAddressofExpression* expr);
	void (*visit_reference_transfer_expression) (ValaCCodeModule* self, ValaReferenceTransferExpression* expr);
	void (*visit_binary_expression) (ValaCCodeModule* self, ValaBinaryExpression* expr);
	void (*visit_type_check) (ValaCCodeModule* self, ValaTypeCheck* expr);
	void (*visit_lambda_expression) (ValaCCodeModule* self, ValaLambdaExpression* l);
	void (*visit_assignment) (ValaCCodeModule* self, ValaAssignment* a);
	char* (*get_custom_creturn_type) (ValaCCodeModule* self, ValaMethod* m);
	void (*generate_dynamic_method_wrapper) (ValaCCodeModule* self, ValaDynamicMethod* method);
	gboolean (*method_has_wrapper) (ValaCCodeModule* self, ValaMethod* method);
	ValaCCodeIdentifier* (*get_value_setter_function) (ValaCCodeModule* self, ValaDataType* type_reference);
	ValaCCodeExpression* (*get_construct_property_assignment) (ValaCCodeModule* self, ValaCCodeConstant* canonical_cconstant, ValaDataType* property_type, ValaCCodeExpression* value);
	ValaCCodeFunctionCall* (*get_param_spec) (ValaCCodeModule* self, ValaProperty* prop);
	ValaCCodeFunctionCall* (*get_signal_creation) (ValaCCodeModule* self, ValaSignal* sig, ValaTypeSymbol* type);
	ValaCCodeFragment* (*register_dbus_info) (ValaCCodeModule* self, ValaObjectTypeSymbol* bindable);
	char* (*get_dynamic_property_getter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_property_setter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_signal_cname) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_connect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_disconnect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	void (*generate_marshaller) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, gboolean dbus);
	char* (*get_marshaller_function) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, const char* prefix, gboolean dbus);
	char* (*get_array_length_cname) (ValaCCodeModule* self, const char* array_cname, gint dim);
	ValaCCodeExpression* (*get_array_length_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr, gint dim);
	char* (*get_array_size_cname) (ValaCCodeModule* self, const char* array_cname);
	ValaCCodeExpression* (*get_array_size_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr);
	void (*add_simple_check) (ValaCCodeModule* self, ValaCodeNode* node, ValaCCodeFragment* cfrag);
};

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeBaseModule {
	ValaCCodeModule parent_instance;
	ValaCCodeBaseModulePrivate * priv;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaTypeSymbol* current_type_symbol;
	ValaClass* current_class;
	ValaMethod* current_method;
	ValaDataType* current_return_type;
	ValaTryStatement* current_try;
	ValaPropertyAccessor* current_property_accessor;
	ValaCCodeDeclarationSpace* header_declarations;
	ValaCCodeDeclarationSpace* internal_header_declarations;
	ValaCCodeDeclarationSpace* source_declarations;
	ValaCCodeFragment* source_signal_marshaller_declaration;
	ValaCCodeFragment* source_type_member_definition;
	ValaCCodeFragment* class_init_fragment;
	ValaCCodeFragment* base_init_fragment;
	ValaCCodeFragment* class_finalize_fragment;
	ValaCCodeFragment* base_finalize_fragment;
	ValaCCodeFragment* instance_init_fragment;
	ValaCCodeFragment* instance_finalize_fragment;
	ValaCCodeFragment* source_signal_marshaller_definition;
	ValaCCodeFragment* module_init_fragment;
	ValaCCodeStruct* param_spec_struct;
	ValaCCodeStruct* closure_struct;
	ValaCCodeEnum* prop_enum;
	ValaCCodeFunction* function;
	ValaCCodeFragment* pre_statement_fragment;
	GeeArrayList* temp_vars;
	GeeArrayList* temp_ref_vars;
	GeeSet* user_marshal_set;
	GeeSet* predefined_marshal_set;
	GeeSet* c_keywords;
	gint next_temp_var_id;
	gboolean in_creation_method;
	gboolean in_constructor;
	gboolean in_static_or_class_ctor;
	gboolean current_method_inner_error;
	gint next_coroutine_state;
	ValaDataType* bool_type;
	ValaDataType* char_type;
	ValaDataType* uchar_type;
	ValaDataType* unichar_type;
	ValaDataType* short_type;
	ValaDataType* ushort_type;
	ValaDataType* int_type;
	ValaDataType* uint_type;
	ValaDataType* long_type;
	ValaDataType* ulong_type;
	ValaDataType* int8_type;
	ValaDataType* uint8_type;
	ValaDataType* int16_type;
	ValaDataType* uint16_type;
	ValaDataType* int32_type;
	ValaDataType* uint32_type;
	ValaDataType* int64_type;
	ValaDataType* uint64_type;
	ValaDataType* string_type;
	ValaDataType* float_type;
	ValaDataType* double_type;
	ValaTypeSymbol* gtype_type;
	ValaTypeSymbol* gobject_type;
	ValaErrorType* gerror_type;
	ValaClass* glist_type;
	ValaClass* gslist_type;
	ValaTypeSymbol* gstringbuilder_type;
	ValaTypeSymbol* garray_type;
	ValaTypeSymbol* gbytearray_type;
	ValaTypeSymbol* gptrarray_type;
	ValaDataType* gquark_type;
	ValaStruct* gvalue_type;
	ValaStruct* mutex_type;
	ValaTypeSymbol* type_module_type;
	ValaTypeSymbol* dbus_object_type;
	gboolean in_plugin;
	char* module_init_param_name;
	gboolean string_h_needed;
	gboolean gvaluecollector_h_needed;
	gboolean gio_h_needed;
	gboolean requires_array_free;
	gboolean requires_array_move;
	gboolean requires_array_length;
	gboolean requires_strcmp0;
	gboolean dbus_glib_h_needed;
	gboolean dbus_glib_h_needed_in_header;
	GeeSet* wrappers;
	GeeMap* variable_name_map;
};

struct _ValaCCodeBaseModuleClass {
	ValaCCodeModuleClass parent_class;
	void (*append_vala_array_free) (ValaCCodeBaseModule* self);
	void (*append_vala_array_move) (ValaCCodeBaseModule* self);
	void (*append_vala_array_length) (ValaCCodeBaseModule* self);
	void (*generate_class_struct_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_struct_declaration) (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_delegate_declaration) (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_cparameters) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
	ValaCCodeExpression* (*get_dup_func_expression) (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference);
	char* (*get_delegate_target_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	ValaCCodeExpression* (*get_delegate_target_cexpression) (ValaCCodeBaseModule* self, ValaExpression* delegate_expr);
	char* (*get_delegate_target_destroy_notify_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	void (*generate_class_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_interface_declaration) (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_method_declaration) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_error_domain_declaration) (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space);
	ValaCCodeExpression* (*get_implicit_cast_expression) (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
};

struct _ValaCCodeStructModule {
	ValaCCodeBaseModule parent_instance;
	ValaCCodeStructModulePrivate * priv;
};

struct _ValaCCodeStructModuleClass {
	ValaCCodeBaseModuleClass parent_class;
};



gpointer vala_ccode_module_ref (gpointer instance);
void vala_ccode_module_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_module (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_module (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_module (const GValue* value);
GType vala_ccode_module_get_type (void);
GType vala_ccode_base_module_get_type (void);
gpointer vala_ccode_declaration_space_ref (gpointer instance);
void vala_ccode_declaration_space_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_declaration_space (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_declaration_space (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_declaration_space (const GValue* value);
GType vala_ccode_declaration_space_get_type (void);
GType vala_ccode_struct_module_get_type (void);
enum  {
	VALA_CCODE_STRUCT_MODULE_DUMMY_PROPERTY
};
GType vala_ccode_generator_get_type (void);
ValaCCodeBaseModule* vala_ccode_base_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeBaseModule* vala_ccode_base_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeStructModule* vala_ccode_struct_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeStructModule* vala_ccode_struct_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeStructModule* vala_ccode_struct_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
gboolean vala_ccode_declaration_space_add_symbol_declaration (ValaCCodeDeclarationSpace* self, ValaSymbol* sym, const char* name);
void vala_ccode_base_module_generate_type_declaration (ValaCCodeBaseModule* self, ValaDataType* type, ValaCCodeDeclarationSpace* decl_space);
ValaCCodeModule* vala_ccode_module_get_head (ValaCCodeModule* self);
char* vala_ccode_module_get_array_length_cname (ValaCCodeModule* self, const char* array_cname, gint dim);
char* vala_ccode_module_get_array_size_cname (ValaCCodeModule* self, const char* array_cname);
char* vala_ccode_base_module_get_delegate_target_cname (ValaCCodeBaseModule* self, const char* delegate_cname);
void vala_ccode_declaration_space_add_type_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
void vala_ccode_declaration_space_add_type_definition (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
static void vala_ccode_struct_module_real_generate_struct_declaration (ValaCCodeBaseModule* base, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_struct_declaration (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
ValaCCodeGenerator* vala_ccode_module_get_codegen (ValaCCodeModule* self);
static void vala_ccode_struct_module_add_struct_copy_function (ValaCCodeStructModule* self, ValaStruct* st);
static void vala_ccode_struct_module_add_struct_destroy_function (ValaCCodeStructModule* self, ValaStruct* st);
static void vala_ccode_struct_module_add_struct_dup_function (ValaCCodeStructModule* self, ValaStruct* st);
static void vala_ccode_struct_module_add_struct_free_function (ValaCCodeStructModule* self, ValaStruct* st);
static void vala_ccode_struct_module_real_visit_struct (ValaCCodeModule* base, ValaStruct* st);
void vala_ccode_declaration_space_add_type_member_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
gboolean vala_ccode_base_module_requires_copy (ValaCCodeBaseModule* self, ValaDataType* type);
ValaDataType* vala_ccode_base_module_get_data_type_for_symbol (ValaTypeSymbol* sym);
ValaCCodeExpression* vala_ccode_base_module_get_ref_cexpression (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node);
void vala_ccode_base_module_append_temp_decl (ValaCCodeBaseModule* self, ValaCCodeFragment* cfrag, GeeList* temp_vars);
static gpointer vala_ccode_struct_module_parent_class = NULL;



ValaCCodeStructModule* vala_ccode_struct_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	ValaCCodeStructModule* self;
	g_return_val_if_fail (codegen != NULL, NULL);
	self = (ValaCCodeStructModule*) vala_ccode_base_module_construct (object_type, codegen, next);
	return self;
}


ValaCCodeStructModule* vala_ccode_struct_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	return vala_ccode_struct_module_construct (VALA_TYPE_CCODE_STRUCT_MODULE, codegen, next);
}


static void vala_ccode_struct_module_real_generate_struct_declaration (ValaCCodeBaseModule* base, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space) {
	ValaCCodeStructModule * self;
	char* _tmp0;
	gboolean _tmp1;
	char* _tmp3;
	char* _tmp2;
	ValaCCodeStruct* _tmp4;
	ValaCCodeStruct* instance_struct;
	ValaCCodeTypeDefinition* _tmp22;
	ValaCCodeVariableDeclarator* _tmp21;
	char* _tmp20;
	char* _tmp19;
	char* _tmp18;
	self = (ValaCCodeStructModule*) base;
	g_return_if_fail (st != NULL);
	g_return_if_fail (decl_space != NULL);
	_tmp0 = NULL;
	if ((_tmp1 = vala_ccode_declaration_space_add_symbol_declaration (decl_space, (ValaSymbol*) st, _tmp0 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE)), _tmp0 = (g_free (_tmp0), NULL), _tmp1)) {
		return;
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp4 = NULL;
	instance_struct = (_tmp4 = vala_ccode_struct_new (_tmp3 = g_strdup_printf ("_%s", _tmp2 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE))), _tmp3 = (g_free (_tmp3), NULL), _tmp2 = (g_free (_tmp2), NULL), _tmp4);
	{
		GeeList* _tmp5;
		GeeIterator* _tmp6;
		GeeIterator* _f_it;
		_tmp5 = NULL;
		_tmp6 = NULL;
		_f_it = (_tmp6 = gee_iterable_iterator ((GeeIterable*) (_tmp5 = vala_struct_get_fields (st))), (_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL)), _tmp6);
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			char* field_ctype;
			f = (ValaField*) gee_iterator_get (_f_it);
			field_ctype = vala_data_type_get_cname (vala_field_get_field_type (f));
			if (vala_field_get_is_volatile (f)) {
				char* _tmp7;
				_tmp7 = NULL;
				field_ctype = (_tmp7 = g_strconcat ("volatile ", field_ctype, NULL), field_ctype = (g_free (field_ctype), NULL), _tmp7);
			}
			if (vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
				char* _tmp8;
				gboolean _tmp9;
				vala_ccode_base_module_generate_type_declaration ((ValaCCodeBaseModule*) self, vala_field_get_field_type (f), decl_space);
				_tmp8 = NULL;
				vala_ccode_struct_add_field (instance_struct, field_ctype, _tmp8 = vala_field_get_cname (f));
				_tmp8 = (g_free (_tmp8), NULL);
				_tmp9 = FALSE;
				if (VALA_IS_ARRAY_TYPE (vala_field_get_field_type (f))) {
					_tmp9 = !vala_field_get_no_array_length (f);
				} else {
					_tmp9 = FALSE;
				}
				if (_tmp9) {
					ValaArrayType* _tmp10;
					ValaArrayType* array_type;
					ValaDataType* len_type;
					gboolean _tmp13;
					/* create fields to store array dimensions*/
					_tmp10 = NULL;
					array_type = (_tmp10 = VALA_ARRAY_TYPE (vala_field_get_field_type (f)), (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10));
					len_type = vala_data_type_copy (((ValaCCodeBaseModule*) self)->int_type);
					{
						gint dim;
						dim = 1;
						for (; dim <= vala_array_type_get_rank (array_type); dim++) {
							char* _tmp12;
							char* _tmp11;
							_tmp12 = NULL;
							_tmp11 = NULL;
							vala_ccode_struct_add_field (instance_struct, _tmp11 = vala_data_type_get_cname (len_type), _tmp12 = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), vala_symbol_get_name ((ValaSymbol*) f), dim));
							_tmp12 = (g_free (_tmp12), NULL);
							_tmp11 = (g_free (_tmp11), NULL);
						}
					}
					_tmp13 = FALSE;
					if (vala_array_type_get_rank (array_type) == 1) {
						_tmp13 = vala_symbol_is_internal_symbol ((ValaSymbol*) f);
					} else {
						_tmp13 = FALSE;
					}
					if (_tmp13) {
						char* _tmp15;
						char* _tmp14;
						_tmp15 = NULL;
						_tmp14 = NULL;
						vala_ccode_struct_add_field (instance_struct, _tmp14 = vala_data_type_get_cname (len_type), _tmp15 = vala_ccode_module_get_array_size_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), vala_symbol_get_name ((ValaSymbol*) f)));
						_tmp15 = (g_free (_tmp15), NULL);
						_tmp14 = (g_free (_tmp14), NULL);
					}
					(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					(len_type == NULL) ? NULL : (len_type = (vala_code_node_unref (len_type), NULL));
				} else {
					if (VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f))) {
						ValaDelegateType* _tmp16;
						ValaDelegateType* delegate_type;
						_tmp16 = NULL;
						delegate_type = (_tmp16 = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16));
						if (vala_delegate_get_has_target (vala_delegate_type_get_delegate_symbol (delegate_type))) {
							char* _tmp17;
							/* create field to store delegate target*/
							_tmp17 = NULL;
							vala_ccode_struct_add_field (instance_struct, "gpointer", _tmp17 = vala_ccode_base_module_get_delegate_target_cname ((ValaCCodeBaseModule*) self, vala_symbol_get_name ((ValaSymbol*) f)));
							_tmp17 = (g_free (_tmp17), NULL);
						}
						(delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL));
					}
				}
			}
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			field_ctype = (g_free (field_ctype), NULL);
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	_tmp22 = NULL;
	_tmp21 = NULL;
	_tmp20 = NULL;
	_tmp19 = NULL;
	_tmp18 = NULL;
	vala_ccode_declaration_space_add_type_declaration (decl_space, (ValaCCodeNode*) (_tmp22 = vala_ccode_type_definition_new (_tmp19 = g_strdup_printf ("struct _%s", _tmp18 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE)), (ValaCCodeDeclarator*) (_tmp21 = vala_ccode_variable_declarator_new (_tmp20 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), NULL)))));
	(_tmp22 == NULL) ? NULL : (_tmp22 = (vala_ccode_node_unref (_tmp22), NULL));
	(_tmp21 == NULL) ? NULL : (_tmp21 = (vala_ccode_node_unref (_tmp21), NULL));
	_tmp20 = (g_free (_tmp20), NULL);
	_tmp19 = (g_free (_tmp19), NULL);
	_tmp18 = (g_free (_tmp18), NULL);
	if (vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) st)) != NULL) {
		ValaCCodeComment* _tmp23;
		_tmp23 = NULL;
		vala_ccode_declaration_space_add_type_definition (decl_space, (ValaCCodeNode*) (_tmp23 = vala_ccode_comment_new (vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) st)))));
		(_tmp23 == NULL) ? NULL : (_tmp23 = (vala_ccode_node_unref (_tmp23), NULL));
	}
	vala_ccode_declaration_space_add_type_definition (decl_space, (ValaCCodeNode*) instance_struct);
	(instance_struct == NULL) ? NULL : (instance_struct = (vala_ccode_node_unref (instance_struct), NULL));
}


static void vala_ccode_struct_module_real_visit_struct (ValaCCodeModule* base, ValaStruct* st) {
	ValaCCodeStructModule * self;
	ValaTypeSymbol* _tmp0;
	ValaTypeSymbol* old_type_symbol;
	ValaCCodeFragment* _tmp1;
	ValaCCodeFragment* old_instance_finalize_fragment;
	ValaTypeSymbol* _tmp3;
	ValaTypeSymbol* _tmp2;
	ValaCCodeFragment* _tmp4;
	ValaTypeSymbol* _tmp6;
	ValaTypeSymbol* _tmp5;
	ValaCCodeFragment* _tmp8;
	ValaCCodeFragment* _tmp7;
	self = (ValaCCodeStructModule*) base;
	g_return_if_fail (st != NULL);
	_tmp0 = NULL;
	old_type_symbol = (_tmp0 = ((ValaCCodeBaseModule*) self)->current_type_symbol, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
	_tmp1 = NULL;
	old_instance_finalize_fragment = (_tmp1 = ((ValaCCodeBaseModule*) self)->instance_finalize_fragment, (_tmp1 == NULL) ? NULL : vala_ccode_node_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	((ValaCCodeBaseModule*) self)->current_type_symbol = (_tmp3 = (_tmp2 = (ValaTypeSymbol*) st, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2)), (((ValaCCodeBaseModule*) self)->current_type_symbol == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_type_symbol = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_type_symbol), NULL)), _tmp3);
	_tmp4 = NULL;
	((ValaCCodeBaseModule*) self)->instance_finalize_fragment = (_tmp4 = vala_ccode_fragment_new (), (((ValaCCodeBaseModule*) self)->instance_finalize_fragment == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->instance_finalize_fragment = (vala_ccode_node_unref (((ValaCCodeBaseModule*) self)->instance_finalize_fragment), NULL)), _tmp4);
	vala_ccode_base_module_generate_struct_declaration ((ValaCCodeBaseModule*) self, st, ((ValaCCodeBaseModule*) self)->source_declarations);
	if (!vala_symbol_is_internal_symbol ((ValaSymbol*) st)) {
		vala_ccode_base_module_generate_struct_declaration ((ValaCCodeBaseModule*) self, st, ((ValaCCodeBaseModule*) self)->header_declarations);
	}
	vala_ccode_base_module_generate_struct_declaration ((ValaCCodeBaseModule*) self, st, ((ValaCCodeBaseModule*) self)->internal_header_declarations);
	vala_code_node_accept_children ((ValaCodeNode*) st, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	if (vala_struct_is_disposable (st)) {
		vala_ccode_struct_module_add_struct_copy_function (self, st);
		vala_ccode_struct_module_add_struct_destroy_function (self, st);
	}
	vala_ccode_struct_module_add_struct_dup_function (self, st);
	vala_ccode_struct_module_add_struct_free_function (self, st);
	_tmp6 = NULL;
	_tmp5 = NULL;
	((ValaCCodeBaseModule*) self)->current_type_symbol = (_tmp6 = (_tmp5 = old_type_symbol, (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5)), (((ValaCCodeBaseModule*) self)->current_type_symbol == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_type_symbol = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_type_symbol), NULL)), _tmp6);
	_tmp8 = NULL;
	_tmp7 = NULL;
	((ValaCCodeBaseModule*) self)->instance_finalize_fragment = (_tmp8 = (_tmp7 = old_instance_finalize_fragment, (_tmp7 == NULL) ? NULL : vala_ccode_node_ref (_tmp7)), (((ValaCCodeBaseModule*) self)->instance_finalize_fragment == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->instance_finalize_fragment = (vala_ccode_node_unref (((ValaCCodeBaseModule*) self)->instance_finalize_fragment), NULL)), _tmp8);
	(old_type_symbol == NULL) ? NULL : (old_type_symbol = (vala_code_node_unref (old_type_symbol), NULL));
	(old_instance_finalize_fragment == NULL) ? NULL : (old_instance_finalize_fragment = (vala_ccode_node_unref (old_instance_finalize_fragment), NULL));
}


static void vala_ccode_struct_module_add_struct_dup_function (ValaCCodeStructModule* self, ValaStruct* st) {
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	ValaCCodeFunction* _tmp3;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp7;
	char* _tmp6;
	char* _tmp5;
	char* _tmp4;
	ValaCCodeFunction* _tmp8;
	ValaCCodeBlock* cblock;
	char* _tmp10;
	char* _tmp9;
	ValaCCodeDeclaration* _tmp11;
	ValaCCodeDeclaration* cdecl_;
	ValaCCodeVariableDeclarator* _tmp12;
	ValaCCodeIdentifier* _tmp13;
	ValaCCodeFunctionCall* _tmp14;
	ValaCCodeFunctionCall* creation_call;
	ValaCCodeConstant* _tmp16;
	char* _tmp15;
	ValaCCodeConstant* _tmp17;
	ValaCCodeExpressionStatement* _tmp20;
	ValaCCodeAssignment* _tmp19;
	ValaCCodeIdentifier* _tmp18;
	ValaCCodeReturnStatement* _tmp37;
	ValaCCodeIdentifier* _tmp36;
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp3 = NULL;
	function = (_tmp3 = vala_ccode_function_new (_tmp0 = vala_typesymbol_get_dup_function ((ValaTypeSymbol*) st), _tmp2 = g_strconcat (_tmp1 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), "*", NULL)), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp3);
	if (vala_symbol_get_access ((ValaSymbol*) st) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	}
	_tmp7 = NULL;
	_tmp6 = NULL;
	_tmp5 = NULL;
	_tmp4 = NULL;
	vala_ccode_function_add_parameter (function, _tmp7 = vala_ccode_formal_parameter_new ("self", _tmp6 = g_strconcat (_tmp5 = g_strconcat ("const ", _tmp4 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), NULL), "*", NULL)));
	(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_ccode_node_unref (_tmp7), NULL));
	_tmp6 = (g_free (_tmp6), NULL);
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp8 = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp8 = vala_ccode_function_copy (function)));
	(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_ccode_node_unref (_tmp8), NULL));
	cblock = vala_ccode_block_new ();
	_tmp10 = NULL;
	_tmp9 = NULL;
	_tmp11 = NULL;
	cdecl_ = (_tmp11 = vala_ccode_declaration_new (_tmp10 = g_strconcat (_tmp9 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), "*", NULL)), _tmp10 = (g_free (_tmp10), NULL), _tmp9 = (g_free (_tmp9), NULL), _tmp11);
	_tmp12 = NULL;
	vala_ccode_declaration_add_declarator (cdecl_, (ValaCCodeDeclarator*) (_tmp12 = vala_ccode_variable_declarator_new ("dup", NULL)));
	(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_ccode_node_unref (_tmp12), NULL));
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) cdecl_);
	_tmp13 = NULL;
	_tmp14 = NULL;
	creation_call = (_tmp14 = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp13 = vala_ccode_identifier_new ("g_new0"))), (_tmp13 == NULL) ? NULL : (_tmp13 = (vala_ccode_node_unref (_tmp13), NULL)), _tmp14);
	_tmp16 = NULL;
	_tmp15 = NULL;
	vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp16 = vala_ccode_constant_new (_tmp15 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE))));
	(_tmp16 == NULL) ? NULL : (_tmp16 = (vala_ccode_node_unref (_tmp16), NULL));
	_tmp15 = (g_free (_tmp15), NULL);
	_tmp17 = NULL;
	vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp17 = vala_ccode_constant_new ("1")));
	(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_ccode_node_unref (_tmp17), NULL));
	_tmp20 = NULL;
	_tmp19 = NULL;
	_tmp18 = NULL;
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp20 = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp19 = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp18 = vala_ccode_identifier_new ("dup")), (ValaCCodeExpression*) creation_call, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp20 == NULL) ? NULL : (_tmp20 = (vala_ccode_node_unref (_tmp20), NULL));
	(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_ccode_node_unref (_tmp19), NULL));
	(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_ccode_node_unref (_tmp18), NULL));
	if (vala_struct_is_disposable (st)) {
		ValaCCodeIdentifier* _tmp22;
		char* _tmp21;
		ValaCCodeFunctionCall* _tmp23;
		ValaCCodeFunctionCall* copy_call;
		ValaCCodeIdentifier* _tmp24;
		ValaCCodeIdentifier* _tmp25;
		ValaCCodeExpressionStatement* _tmp26;
		_tmp22 = NULL;
		_tmp21 = NULL;
		_tmp23 = NULL;
		copy_call = (_tmp23 = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp22 = vala_ccode_identifier_new (_tmp21 = vala_typesymbol_get_copy_function ((ValaTypeSymbol*) st)))), (_tmp22 == NULL) ? NULL : (_tmp22 = (vala_ccode_node_unref (_tmp22), NULL)), _tmp21 = (g_free (_tmp21), NULL), _tmp23);
		_tmp24 = NULL;
		vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp24 = vala_ccode_identifier_new ("self")));
		(_tmp24 == NULL) ? NULL : (_tmp24 = (vala_ccode_node_unref (_tmp24), NULL));
		_tmp25 = NULL;
		vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp25 = vala_ccode_identifier_new ("dup")));
		(_tmp25 == NULL) ? NULL : (_tmp25 = (vala_ccode_node_unref (_tmp25), NULL));
		_tmp26 = NULL;
		vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp26 = vala_ccode_expression_statement_new ((ValaCCodeExpression*) copy_call)));
		(_tmp26 == NULL) ? NULL : (_tmp26 = (vala_ccode_node_unref (_tmp26), NULL));
		(copy_call == NULL) ? NULL : (copy_call = (vala_ccode_node_unref (copy_call), NULL));
	} else {
		ValaCCodeIdentifier* _tmp27;
		ValaCCodeFunctionCall* _tmp28;
		ValaCCodeFunctionCall* sizeof_call;
		ValaCCodeConstant* _tmp30;
		char* _tmp29;
		ValaCCodeIdentifier* _tmp31;
		ValaCCodeFunctionCall* _tmp32;
		ValaCCodeFunctionCall* copy_call;
		ValaCCodeIdentifier* _tmp33;
		ValaCCodeIdentifier* _tmp34;
		ValaCCodeExpressionStatement* _tmp35;
		_tmp27 = NULL;
		_tmp28 = NULL;
		sizeof_call = (_tmp28 = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp27 = vala_ccode_identifier_new ("sizeof"))), (_tmp27 == NULL) ? NULL : (_tmp27 = (vala_ccode_node_unref (_tmp27), NULL)), _tmp28);
		_tmp30 = NULL;
		_tmp29 = NULL;
		vala_ccode_function_call_add_argument (sizeof_call, (ValaCCodeExpression*) (_tmp30 = vala_ccode_constant_new (_tmp29 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE))));
		(_tmp30 == NULL) ? NULL : (_tmp30 = (vala_ccode_node_unref (_tmp30), NULL));
		_tmp29 = (g_free (_tmp29), NULL);
		_tmp31 = NULL;
		_tmp32 = NULL;
		copy_call = (_tmp32 = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp31 = vala_ccode_identifier_new ("memcpy"))), (_tmp31 == NULL) ? NULL : (_tmp31 = (vala_ccode_node_unref (_tmp31), NULL)), _tmp32);
		_tmp33 = NULL;
		vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp33 = vala_ccode_identifier_new ("dup")));
		(_tmp33 == NULL) ? NULL : (_tmp33 = (vala_ccode_node_unref (_tmp33), NULL));
		_tmp34 = NULL;
		vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp34 = vala_ccode_identifier_new ("self")));
		(_tmp34 == NULL) ? NULL : (_tmp34 = (vala_ccode_node_unref (_tmp34), NULL));
		vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) sizeof_call);
		_tmp35 = NULL;
		vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp35 = vala_ccode_expression_statement_new ((ValaCCodeExpression*) copy_call)));
		(_tmp35 == NULL) ? NULL : (_tmp35 = (vala_ccode_node_unref (_tmp35), NULL));
		(sizeof_call == NULL) ? NULL : (sizeof_call = (vala_ccode_node_unref (sizeof_call), NULL));
		(copy_call == NULL) ? NULL : (copy_call = (vala_ccode_node_unref (copy_call), NULL));
	}
	_tmp37 = NULL;
	_tmp36 = NULL;
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp37 = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp36 = vala_ccode_identifier_new ("dup")))));
	(_tmp37 == NULL) ? NULL : (_tmp37 = (vala_ccode_node_unref (_tmp37), NULL));
	(_tmp36 == NULL) ? NULL : (_tmp36 = (vala_ccode_node_unref (_tmp36), NULL));
	vala_ccode_function_set_block (function, cblock);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	(function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL));
	(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
	(cdecl_ == NULL) ? NULL : (cdecl_ = (vala_ccode_node_unref (cdecl_), NULL));
	(creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL));
}


static void vala_ccode_struct_module_add_struct_free_function (ValaCCodeStructModule* self, ValaStruct* st) {
	char* _tmp0;
	ValaCCodeFunction* _tmp1;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp4;
	char* _tmp3;
	char* _tmp2;
	ValaCCodeFunction* _tmp5;
	ValaCCodeBlock* cblock;
	ValaCCodeIdentifier* _tmp11;
	ValaCCodeFunctionCall* _tmp12;
	ValaCCodeFunctionCall* free_call;
	ValaCCodeIdentifier* _tmp13;
	ValaCCodeExpressionStatement* _tmp14;
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	function = (_tmp1 = vala_ccode_function_new (_tmp0 = vala_typesymbol_get_free_function ((ValaTypeSymbol*) st), "void"), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	if (vala_symbol_get_access ((ValaSymbol*) st) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	vala_ccode_function_add_parameter (function, _tmp4 = vala_ccode_formal_parameter_new ("self", _tmp3 = g_strconcat (_tmp2 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), "*", NULL)));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_ccode_node_unref (_tmp4), NULL));
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp5 = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp5 = vala_ccode_function_copy (function)));
	(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_ccode_node_unref (_tmp5), NULL));
	cblock = vala_ccode_block_new ();
	if (vala_struct_is_disposable (st)) {
		ValaCCodeIdentifier* _tmp7;
		char* _tmp6;
		ValaCCodeFunctionCall* _tmp8;
		ValaCCodeFunctionCall* destroy_call;
		ValaCCodeIdentifier* _tmp9;
		ValaCCodeExpressionStatement* _tmp10;
		_tmp7 = NULL;
		_tmp6 = NULL;
		_tmp8 = NULL;
		destroy_call = (_tmp8 = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp7 = vala_ccode_identifier_new (_tmp6 = vala_typesymbol_get_destroy_function ((ValaTypeSymbol*) st)))), (_tmp7 == NULL) ? NULL : (_tmp7 = (vala_ccode_node_unref (_tmp7), NULL)), _tmp6 = (g_free (_tmp6), NULL), _tmp8);
		_tmp9 = NULL;
		vala_ccode_function_call_add_argument (destroy_call, (ValaCCodeExpression*) (_tmp9 = vala_ccode_identifier_new ("self")));
		(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_ccode_node_unref (_tmp9), NULL));
		_tmp10 = NULL;
		vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp10 = vala_ccode_expression_statement_new ((ValaCCodeExpression*) destroy_call)));
		(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_ccode_node_unref (_tmp10), NULL));
		(destroy_call == NULL) ? NULL : (destroy_call = (vala_ccode_node_unref (destroy_call), NULL));
	}
	_tmp11 = NULL;
	_tmp12 = NULL;
	free_call = (_tmp12 = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp11 = vala_ccode_identifier_new ("g_free"))), (_tmp11 == NULL) ? NULL : (_tmp11 = (vala_ccode_node_unref (_tmp11), NULL)), _tmp12);
	_tmp13 = NULL;
	vala_ccode_function_call_add_argument (free_call, (ValaCCodeExpression*) (_tmp13 = vala_ccode_identifier_new ("self")));
	(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_ccode_node_unref (_tmp13), NULL));
	_tmp14 = NULL;
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp14 = vala_ccode_expression_statement_new ((ValaCCodeExpression*) free_call)));
	(_tmp14 == NULL) ? NULL : (_tmp14 = (vala_ccode_node_unref (_tmp14), NULL));
	vala_ccode_function_set_block (function, cblock);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	(function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL));
	(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
	(free_call == NULL) ? NULL : (free_call = (vala_ccode_node_unref (free_call), NULL));
}


static void vala_ccode_struct_module_add_struct_copy_function (ValaCCodeStructModule* self, ValaStruct* st) {
	char* _tmp0;
	ValaCCodeFunction* _tmp1;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp5;
	char* _tmp4;
	char* _tmp3;
	char* _tmp2;
	ValaCCodeFormalParameter* _tmp8;
	char* _tmp7;
	char* _tmp6;
	ValaCCodeFunction* _tmp9;
	ValaCCodeBlock* cblock;
	ValaCCodeFragment* cfrag;
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	function = (_tmp1 = vala_ccode_function_new (_tmp0 = vala_typesymbol_get_copy_function ((ValaTypeSymbol*) st), "void"), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	if (vala_symbol_get_access ((ValaSymbol*) st) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	}
	_tmp5 = NULL;
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	vala_ccode_function_add_parameter (function, _tmp5 = vala_ccode_formal_parameter_new ("self", _tmp4 = g_strconcat (_tmp3 = g_strconcat ("const ", _tmp2 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), NULL), "*", NULL)));
	(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_ccode_node_unref (_tmp5), NULL));
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp8 = NULL;
	_tmp7 = NULL;
	_tmp6 = NULL;
	vala_ccode_function_add_parameter (function, _tmp8 = vala_ccode_formal_parameter_new ("dest", _tmp7 = g_strconcat (_tmp6 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), "*", NULL)));
	(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_ccode_node_unref (_tmp8), NULL));
	_tmp7 = (g_free (_tmp7), NULL);
	_tmp6 = (g_free (_tmp6), NULL);
	_tmp9 = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp9 = vala_ccode_function_copy (function)));
	(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_ccode_node_unref (_tmp9), NULL));
	cblock = vala_ccode_block_new ();
	cfrag = vala_ccode_fragment_new ();
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) cfrag);
	{
		GeeList* _tmp10;
		GeeIterator* _tmp11;
		GeeIterator* _f_it;
		_tmp10 = NULL;
		_tmp11 = NULL;
		_f_it = (_tmp11 = gee_iterable_iterator ((GeeIterable*) (_tmp10 = vala_struct_get_fields (st))), (_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL)), _tmp11);
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (_f_it);
			if (vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
				ValaCCodeIdentifier* _tmp12;
				ValaCCodeExpression* _tmp13;
				ValaCCodeExpression* copy;
				ValaCCodeIdentifier* _tmp17;
				ValaCCodeMemberAccess* _tmp18;
				ValaCCodeMemberAccess* dest;
				ValaCCodeExpressionStatement* _tmp20;
				ValaCCodeAssignment* _tmp19;
				ValaArrayType* _tmp22;
				ValaDataType* _tmp21;
				ValaArrayType* array_type;
				_tmp12 = NULL;
				_tmp13 = NULL;
				copy = (_tmp13 = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp12 = vala_ccode_identifier_new ("self")), vala_symbol_get_name ((ValaSymbol*) f)), (_tmp12 == NULL) ? NULL : (_tmp12 = (vala_ccode_node_unref (_tmp12), NULL)), _tmp13);
				if (vala_ccode_base_module_requires_copy ((ValaCCodeBaseModule*) self, vala_field_get_field_type (f))) {
					ValaMemberAccess* this_access;
					ValaDataType* _tmp14;
					ValaCCodeIdentifier* _tmp15;
					ValaMemberAccess* ma;
					ValaCCodeExpression* _tmp16;
					this_access = vala_member_access_new_simple ("this", NULL);
					_tmp14 = NULL;
					vala_expression_set_value_type ((ValaExpression*) this_access, _tmp14 = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) f))));
					(_tmp14 == NULL) ? NULL : (_tmp14 = (vala_code_node_unref (_tmp14), NULL));
					_tmp15 = NULL;
					vala_code_node_set_ccodenode ((ValaCodeNode*) this_access, (ValaCCodeNode*) (_tmp15 = vala_ccode_identifier_new ("(*self)")));
					(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_ccode_node_unref (_tmp15), NULL));
					ma = vala_member_access_new ((ValaExpression*) this_access, vala_symbol_get_name ((ValaSymbol*) f), NULL);
					vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) f);
					_tmp16 = NULL;
					copy = (_tmp16 = vala_ccode_base_module_get_ref_cexpression ((ValaCCodeBaseModule*) self, vala_field_get_field_type (f), copy, (ValaExpression*) ma, (ValaCodeNode*) f), (copy == NULL) ? NULL : (copy = (vala_ccode_node_unref (copy), NULL)), _tmp16);
					(this_access == NULL) ? NULL : (this_access = (vala_code_node_unref (this_access), NULL));
					(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				}
				_tmp17 = NULL;
				_tmp18 = NULL;
				dest = (_tmp18 = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp17 = vala_ccode_identifier_new ("dest")), vala_symbol_get_name ((ValaSymbol*) f)), (_tmp17 == NULL) ? NULL : (_tmp17 = (vala_ccode_node_unref (_tmp17), NULL)), _tmp18);
				_tmp20 = NULL;
				_tmp19 = NULL;
				vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp20 = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp19 = vala_ccode_assignment_new ((ValaCCodeExpression*) dest, copy, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
				(_tmp20 == NULL) ? NULL : (_tmp20 = (vala_ccode_node_unref (_tmp20), NULL));
				(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_ccode_node_unref (_tmp19), NULL));
				_tmp22 = NULL;
				_tmp21 = NULL;
				array_type = (_tmp22 = (_tmp21 = vala_field_get_field_type (f), VALA_IS_ARRAY_TYPE (_tmp21) ? ((ValaArrayType*) _tmp21) : NULL), (_tmp22 == NULL) ? NULL : vala_code_node_ref (_tmp22));
				if (array_type != NULL) {
					{
						gint dim;
						dim = 1;
						for (; dim <= vala_array_type_get_rank (array_type); dim++) {
							char* _tmp24;
							ValaCCodeIdentifier* _tmp23;
							ValaCCodeMemberAccess* _tmp25;
							ValaCCodeMemberAccess* len_src;
							char* _tmp27;
							ValaCCodeIdentifier* _tmp26;
							ValaCCodeMemberAccess* _tmp28;
							ValaCCodeMemberAccess* len_dest;
							ValaCCodeExpressionStatement* _tmp30;
							ValaCCodeAssignment* _tmp29;
							_tmp24 = NULL;
							_tmp23 = NULL;
							_tmp25 = NULL;
							len_src = (_tmp25 = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp23 = vala_ccode_identifier_new ("self")), _tmp24 = vala_ccode_module_get_array_length_cname ((ValaCCodeModule*) self, vala_symbol_get_name ((ValaSymbol*) f), dim)), _tmp24 = (g_free (_tmp24), NULL), (_tmp23 == NULL) ? NULL : (_tmp23 = (vala_ccode_node_unref (_tmp23), NULL)), _tmp25);
							_tmp27 = NULL;
							_tmp26 = NULL;
							_tmp28 = NULL;
							len_dest = (_tmp28 = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp26 = vala_ccode_identifier_new ("dest")), _tmp27 = vala_ccode_module_get_array_length_cname ((ValaCCodeModule*) self, vala_symbol_get_name ((ValaSymbol*) f), dim)), _tmp27 = (g_free (_tmp27), NULL), (_tmp26 == NULL) ? NULL : (_tmp26 = (vala_ccode_node_unref (_tmp26), NULL)), _tmp28);
							_tmp30 = NULL;
							_tmp29 = NULL;
							vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp30 = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp29 = vala_ccode_assignment_new ((ValaCCodeExpression*) len_dest, (ValaCCodeExpression*) len_src, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
							(_tmp30 == NULL) ? NULL : (_tmp30 = (vala_ccode_node_unref (_tmp30), NULL));
							(_tmp29 == NULL) ? NULL : (_tmp29 = (vala_ccode_node_unref (_tmp29), NULL));
							(len_src == NULL) ? NULL : (len_src = (vala_ccode_node_unref (len_src), NULL));
							(len_dest == NULL) ? NULL : (len_dest = (vala_ccode_node_unref (len_dest), NULL));
						}
					}
				}
				(copy == NULL) ? NULL : (copy = (vala_ccode_node_unref (copy), NULL));
				(dest == NULL) ? NULL : (dest = (vala_ccode_node_unref (dest), NULL));
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			}
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	vala_ccode_base_module_append_temp_decl ((ValaCCodeBaseModule*) self, cfrag, (GeeList*) ((ValaCCodeBaseModule*) self)->temp_vars);
	gee_collection_clear ((GeeCollection*) ((ValaCCodeBaseModule*) self)->temp_vars);
	vala_ccode_function_set_block (function, cblock);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	(function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL));
	(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
	(cfrag == NULL) ? NULL : (cfrag = (vala_ccode_node_unref (cfrag), NULL));
}


static void vala_ccode_struct_module_add_struct_destroy_function (ValaCCodeStructModule* self, ValaStruct* st) {
	char* _tmp0;
	ValaCCodeFunction* _tmp1;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp4;
	char* _tmp3;
	char* _tmp2;
	ValaCCodeFunction* _tmp5;
	ValaCCodeBlock* cblock;
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	function = (_tmp1 = vala_ccode_function_new (_tmp0 = vala_typesymbol_get_destroy_function ((ValaTypeSymbol*) st), "void"), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	if (vala_symbol_get_access ((ValaSymbol*) st) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	vala_ccode_function_add_parameter (function, _tmp4 = vala_ccode_formal_parameter_new ("self", _tmp3 = g_strconcat (_tmp2 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), "*", NULL)));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_ccode_node_unref (_tmp4), NULL));
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp5 = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp5 = vala_ccode_function_copy (function)));
	(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_ccode_node_unref (_tmp5), NULL));
	cblock = vala_ccode_block_new ();
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->instance_finalize_fragment);
	vala_ccode_function_set_block (function, cblock);
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) function);
	(function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL));
	(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
}


static void vala_ccode_struct_module_class_init (ValaCCodeStructModuleClass * klass) {
	vala_ccode_struct_module_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_struct_declaration = vala_ccode_struct_module_real_generate_struct_declaration;
	VALA_CCODE_MODULE_CLASS (klass)->visit_struct = vala_ccode_struct_module_real_visit_struct;
}


static void vala_ccode_struct_module_instance_init (ValaCCodeStructModule * self) {
}


GType vala_ccode_struct_module_get_type (void) {
	static GType vala_ccode_struct_module_type_id = 0;
	if (vala_ccode_struct_module_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeStructModuleClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_struct_module_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeStructModule), 0, (GInstanceInitFunc) vala_ccode_struct_module_instance_init, NULL };
		vala_ccode_struct_module_type_id = g_type_register_static (VALA_TYPE_CCODE_BASE_MODULE, "ValaCCodeStructModule", &g_define_type_info, 0);
	}
	return vala_ccode_struct_module_type_id;
}




