/* valawhilestatement.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valawhilestatement.h>
#include <stdlib.h>
#include <string.h>
#include <gee/list.h>
#include <vala/valaexpression.h>
#include <vala/valablock.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadatatype.h>
#include <vala/valareport.h>
#include <vala/valaunaryexpression.h>
#include <vala/valabreakstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valabooleanliteral.h>




struct _ValaWhileStatementPrivate {
	ValaExpression* _condition;
	ValaBlock* _body;
};

#define VALA_WHILE_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_WHILE_STATEMENT, ValaWhileStatementPrivate))
enum  {
	VALA_WHILE_STATEMENT_DUMMY_PROPERTY
};
static void vala_while_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_while_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_while_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_while_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_while_statement_parent_class = NULL;
static ValaStatementIface* vala_while_statement_vala_statement_parent_iface = NULL;
static void vala_while_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new while statement.
 *
 * @param cond   loop condition
 * @param body   loop body
 * @param source reference to source code
 * @return       newly created while statement
 */
ValaWhileStatement* vala_while_statement_construct (GType object_type, ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaWhileStatement* self;
	g_return_val_if_fail (condition != NULL, NULL);
	g_return_val_if_fail (body != NULL, NULL);
	self = (ValaWhileStatement*) g_type_create_instance (object_type);
	vala_while_statement_set_body (self, body);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_while_statement_set_condition (self, condition);
	return self;
}


ValaWhileStatement* vala_while_statement_new (ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	return vala_while_statement_construct (VALA_TYPE_WHILE_STATEMENT, condition, body, source_reference);
}


static void vala_while_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaWhileStatement * self;
	self = (ValaWhileStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_while_statement (visitor, self);
}


static void vala_while_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaWhileStatement * self;
	self = (ValaWhileStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_while_statement_get_condition (self), visitor);
	vala_code_visitor_visit_end_full_expression (visitor, vala_while_statement_get_condition (self));
	vala_code_node_accept ((ValaCodeNode*) vala_while_statement_get_body (self), visitor);
}


static void vala_while_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaWhileStatement * self;
	self = (ValaWhileStatement*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_while_statement_get_condition (self) == old_node) {
		vala_while_statement_set_condition (self, new_node);
	}
}


static gboolean vala_while_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaWhileStatement * self;
	GeeList* _tmp3;
	GeeList* _tmp4;
	self = (ValaWhileStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_code_node_check ((ValaCodeNode*) vala_while_statement_get_condition (self), analyzer);
	vala_code_node_check ((ValaCodeNode*) vala_while_statement_get_body (self), analyzer);
	if (vala_code_node_get_error ((ValaCodeNode*) vala_while_statement_get_condition (self))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (!vala_data_type_compatible (vala_expression_get_value_type (vala_while_statement_get_condition (self)), analyzer->bool_type)) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_while_statement_get_condition (self)), "Condition must be boolean");
		return FALSE;
	}
	_tmp3 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp3 = vala_code_node_get_error_types ((ValaCodeNode*) vala_while_statement_get_condition (self)));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL));
	_tmp4 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp4 = vala_code_node_get_error_types ((ValaCodeNode*) vala_while_statement_get_body (self)));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


ValaBlock* vala_while_statement_prepare_condition_split (ValaWhileStatement* self, ValaSemanticAnalyzer* analyzer) {
	ValaUnaryExpression* if_condition;
	ValaBlock* true_block;
	ValaBreakStatement* _tmp0;
	ValaIfStatement* if_stmt;
	ValaBooleanLiteral* _tmp1;
	ValaBlock* _tmp2;
	ValaBlock* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (analyzer != NULL, NULL);
	/* move condition into the loop body to allow split
	 in multiple statements*/
	if_condition = vala_unary_expression_new (VALA_UNARY_OPERATOR_LOGICAL_NEGATION, vala_while_statement_get_condition (self), vala_code_node_get_source_reference ((ValaCodeNode*) vala_while_statement_get_condition (self)));
	true_block = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) vala_while_statement_get_condition (self)));
	_tmp0 = NULL;
	vala_block_add_statement (true_block, (ValaStatement*) (_tmp0 = vala_break_statement_new (vala_code_node_get_source_reference ((ValaCodeNode*) vala_while_statement_get_condition (self)))));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
	if_stmt = vala_if_statement_new ((ValaExpression*) if_condition, true_block, NULL, vala_code_node_get_source_reference ((ValaCodeNode*) vala_while_statement_get_condition (self)));
	vala_block_insert_statement (vala_while_statement_get_body (self), 0, (ValaStatement*) if_stmt);
	_tmp1 = NULL;
	vala_while_statement_set_condition (self, (ValaExpression*) (_tmp1 = vala_boolean_literal_new (TRUE, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	vala_code_node_check ((ValaCodeNode*) vala_while_statement_get_condition (self), analyzer);
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = vala_while_statement_get_body (self), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2)), (if_condition == NULL) ? NULL : (if_condition = (vala_code_node_unref (if_condition), NULL)), (true_block == NULL) ? NULL : (true_block = (vala_code_node_unref (true_block), NULL)), (if_stmt == NULL) ? NULL : (if_stmt = (vala_code_node_unref (if_stmt), NULL)), _tmp3);
}


ValaExpression* vala_while_statement_get_condition (ValaWhileStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_condition;
}


void vala_while_statement_set_condition (ValaWhileStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_condition = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_condition, (ValaCodeNode*) self);
}


ValaBlock* vala_while_statement_get_body (ValaWhileStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_while_statement_set_body (ValaWhileStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_body, (ValaCodeNode*) self);
}


static void vala_while_statement_class_init (ValaWhileStatementClass * klass) {
	vala_while_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_while_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaWhileStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_while_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_while_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_while_statement_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->check = vala_while_statement_real_check;
}


static void vala_while_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_while_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_while_statement_instance_init (ValaWhileStatement * self) {
	self->priv = VALA_WHILE_STATEMENT_GET_PRIVATE (self);
}


static void vala_while_statement_finalize (ValaCodeNode* obj) {
	ValaWhileStatement * self;
	self = VALA_WHILE_STATEMENT (obj);
	(self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL));
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL));
	VALA_CODE_NODE_CLASS (vala_while_statement_parent_class)->finalize (obj);
}


GType vala_while_statement_get_type (void) {
	static GType vala_while_statement_type_id = 0;
	if (vala_while_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaWhileStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_while_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaWhileStatement), 0, (GInstanceInitFunc) vala_while_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_while_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_while_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaWhileStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_while_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_while_statement_type_id;
}




