/* valasymbolresolver.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valasymbolresolver.h>
#include <gee/list.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodenode.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valatypesymbol.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valanamespace.h>
#include <vala/valatypeparameter.h>
#include <vala/valaunresolvedtype.h>
#include <vala/valavoidtype.h>
#include <vala/valainvalidtype.h>
#include <vala/valagenerictype.h>
#include <vala/valadelegatetype.h>
#include <vala/valaerrortype.h>
#include <vala/valaerrorcode.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valabooleantype.h>
#include <vala/valaintegertype.h>
#include <vala/valafloatingtype.h>
#include <vala/valastructvaluetype.h>
#include <vala/valaenumvaluetype.h>
#include <vala/valareferencetype.h>




struct _ValaSymbolResolverPrivate {
	ValaSymbol* root_symbol;
	ValaScope* current_scope;
	GeeList* current_using_directives;
};

#define VALA_SYMBOL_RESOLVER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SYMBOL_RESOLVER, ValaSymbolResolverPrivate))
enum  {
	VALA_SYMBOL_RESOLVER_DUMMY_PROPERTY
};
static void vala_symbol_resolver_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_symbol_resolver_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_symbol_resolver_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_symbol_resolver_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_symbol_resolver_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_symbol_resolver_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed);
static void vala_symbol_resolver_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb);
static void vala_symbol_resolver_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_symbol_resolver_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_symbol_resolver_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_symbol_resolver_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_symbol_resolver_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_symbol_resolver_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_symbol_resolver_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_symbol_resolver_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_symbol_resolver_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_symbol_resolver_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_symbol_resolver_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_symbol_resolver_real_visit_using_directive (ValaCodeVisitor* base, ValaUsingDirective* ns);
static ValaSymbol* vala_symbol_resolver_resolve_symbol (ValaSymbolResolver* self, ValaUnresolvedSymbol* unresolved_symbol);
static ValaDataType* vala_symbol_resolver_resolve_type (ValaSymbolResolver* self, ValaUnresolvedType* unresolved_type);
static void vala_symbol_resolver_real_visit_data_type (ValaCodeVisitor* base, ValaDataType* data_type);
static void vala_symbol_resolver_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt);
static void vala_symbol_resolver_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local);
static void vala_symbol_resolver_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_symbol_resolver_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_symbol_resolver_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_symbol_resolver_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt);
static void vala_symbol_resolver_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section);
static void vala_symbol_resolver_real_visit_switch_label (ValaCodeVisitor* base, ValaSwitchLabel* label);
static void vala_symbol_resolver_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_symbol_resolver_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_symbol_resolver_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_symbol_resolver_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_symbol_resolver_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_symbol_resolver_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt);
static void vala_symbol_resolver_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_symbol_resolver_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_symbol_resolver_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_symbol_resolver_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* e);
static void vala_symbol_resolver_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static void vala_symbol_resolver_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static void vala_symbol_resolver_real_visit_method_call (ValaCodeVisitor* base, ValaMethodCall* expr);
static void vala_symbol_resolver_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_symbol_resolver_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_symbol_resolver_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static void vala_symbol_resolver_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static void vala_symbol_resolver_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_symbol_resolver_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_symbol_resolver_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_symbol_resolver_parent_class = NULL;
static void vala_symbol_resolver_finalize (ValaCodeVisitor* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Resolve symbol names in the specified code context.
 *
 * @param context a code context
 */
void vala_symbol_resolver_resolve (ValaSymbolResolver* self, ValaCodeContext* context) {
	ValaSymbol* _tmp1;
	ValaSymbol* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->root_symbol = (_tmp1 = (_tmp0 = (ValaSymbol*) vala_code_context_get_root (context), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0)), (self->priv->root_symbol == NULL) ? NULL : (self->priv->root_symbol = (vala_code_node_unref (self->priv->root_symbol), NULL)), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_symbol_get_scope (self->priv->root_symbol), (_tmp2 == NULL) ? NULL : vala_scope_ref (_tmp2)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp3);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file) {
	ValaSymbolResolver * self;
	GeeList* _tmp0;
	ValaScope* _tmp2;
	ValaScope* _tmp1;
	GeeList* _tmp3;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (file != NULL);
	_tmp0 = NULL;
	self->priv->current_using_directives = (_tmp0 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL) ? NULL : (self->priv->current_using_directives = (gee_collection_object_unref (self->priv->current_using_directives), NULL)), _tmp0);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_scope = (_tmp2 = (_tmp1 = vala_symbol_get_scope (self->priv->root_symbol), (_tmp1 == NULL) ? NULL : vala_scope_ref (_tmp1)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp2);
	vala_source_file_accept_children (file, (ValaCodeVisitor*) self);
	_tmp3 = NULL;
	self->priv->current_using_directives = (_tmp3 = NULL, (self->priv->current_using_directives == NULL) ? NULL : (self->priv->current_using_directives = (gee_collection_object_unref (self->priv->current_using_directives), NULL)), _tmp3);
}


static void vala_symbol_resolver_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp12;
	ValaScope* _tmp11;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (cl != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) cl), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) cl, (ValaCodeVisitor*) self);
	{
		GeeList* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* _type_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		_type_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = vala_class_get_base_types (cl))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			type = (ValaDataType*) gee_iterator_get (_type_it);
			if (VALA_IS_CLASS (vala_data_type_get_data_type (type))) {
				if (vala_class_get_base_class (cl) != NULL) {
					char* _tmp7;
					char* _tmp6;
					char* _tmp5;
					char* _tmp4;
					vala_code_node_set_error ((ValaCodeNode*) cl, TRUE);
					_tmp7 = NULL;
					_tmp6 = NULL;
					_tmp5 = NULL;
					_tmp4 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) type), _tmp7 = g_strdup_printf ("%s: Classes cannot have multiple base classes (`%s' and `%s')", _tmp4 = vala_symbol_get_full_name ((ValaSymbol*) cl), _tmp5 = vala_symbol_get_full_name ((ValaSymbol*) vala_class_get_base_class (cl)), _tmp6 = vala_symbol_get_full_name ((ValaSymbol*) vala_data_type_get_data_type (type))));
					_tmp7 = (g_free (_tmp7), NULL);
					_tmp6 = (g_free (_tmp6), NULL);
					_tmp5 = (g_free (_tmp5), NULL);
					_tmp4 = (g_free (_tmp4), NULL);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
					return;
				}
				vala_class_set_base_class (cl, VALA_CLASS (vala_data_type_get_data_type (type)));
				if (vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) vala_class_get_base_class (cl), (ValaTypeSymbol*) cl)) {
					char* _tmp10;
					char* _tmp9;
					char* _tmp8;
					vala_code_node_set_error ((ValaCodeNode*) cl, TRUE);
					_tmp10 = NULL;
					_tmp9 = NULL;
					_tmp8 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) type), _tmp10 = g_strdup_printf ("Base class cycle (`%s' and `%s')", _tmp8 = vala_symbol_get_full_name ((ValaSymbol*) cl), _tmp9 = vala_symbol_get_full_name ((ValaSymbol*) vala_class_get_base_class (cl))));
					_tmp10 = (g_free (_tmp10), NULL);
					_tmp9 = (g_free (_tmp9), NULL);
					_tmp8 = (g_free (_tmp8), NULL);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
					return;
				}
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	_tmp12 = NULL;
	_tmp11 = NULL;
	self->priv->current_scope = (_tmp12 = (_tmp11 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp11 == NULL) ? NULL : vala_scope_ref (_tmp11)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp12);
}


static void vala_symbol_resolver_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp8;
	ValaScope* _tmp7;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (st != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) st), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) st, (ValaCodeVisitor*) self);
	if (vala_struct_get_base_type (st) != NULL) {
		ValaStruct* _tmp3;
		ValaTypeSymbol* _tmp2;
		ValaStruct* base_type;
		_tmp3 = NULL;
		_tmp2 = NULL;
		base_type = (_tmp3 = (_tmp2 = vala_data_type_get_data_type (vala_struct_get_base_type (st)), VALA_IS_STRUCT (_tmp2) ? ((ValaStruct*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
		if (base_type != NULL) {
			if (vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) base_type, (ValaTypeSymbol*) st)) {
				char* _tmp6;
				char* _tmp5;
				char* _tmp4;
				vala_code_node_set_error ((ValaCodeNode*) st, TRUE);
				_tmp6 = NULL;
				_tmp5 = NULL;
				_tmp4 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) base_type), _tmp6 = g_strdup_printf ("Base struct cycle (`%s' and `%s')", _tmp4 = vala_symbol_get_full_name ((ValaSymbol*) st), _tmp5 = vala_symbol_get_full_name ((ValaSymbol*) base_type)));
				_tmp6 = (g_free (_tmp6), NULL);
				_tmp5 = (g_free (_tmp5), NULL);
				_tmp4 = (g_free (_tmp4), NULL);
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				return;
			}
		}
		(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
	}
	_tmp8 = NULL;
	_tmp7 = NULL;
	self->priv->current_scope = (_tmp8 = (_tmp7 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp7 == NULL) ? NULL : vala_scope_ref (_tmp7)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp8);
}


static void vala_symbol_resolver_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp9;
	ValaScope* _tmp8;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (iface != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) iface), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) iface, (ValaCodeVisitor*) self);
	{
		GeeList* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* _type_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		_type_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = vala_interface_get_prerequisites (iface))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			gboolean _tmp4;
			type = (ValaDataType*) gee_iterator_get (_type_it);
			_tmp4 = FALSE;
			if (vala_data_type_get_data_type (type) != NULL) {
				_tmp4 = vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (type), (ValaTypeSymbol*) iface);
			} else {
				_tmp4 = FALSE;
			}
			if (_tmp4) {
				char* _tmp7;
				char* _tmp6;
				char* _tmp5;
				vala_code_node_set_error ((ValaCodeNode*) iface, TRUE);
				_tmp7 = NULL;
				_tmp6 = NULL;
				_tmp5 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) type), _tmp7 = g_strdup_printf ("Prerequisite cycle (`%s' and `%s')", _tmp5 = vala_symbol_get_full_name ((ValaSymbol*) iface), _tmp6 = vala_symbol_get_full_name ((ValaSymbol*) vala_data_type_get_data_type (type))));
				_tmp7 = (g_free (_tmp7), NULL);
				_tmp6 = (g_free (_tmp6), NULL);
				_tmp5 = (g_free (_tmp5), NULL);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
				return;
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	_tmp9 = NULL;
	_tmp8 = NULL;
	self->priv->current_scope = (_tmp9 = (_tmp8 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp8 == NULL) ? NULL : vala_scope_ref (_tmp8)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp9);
}


static void vala_symbol_resolver_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (en != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) en), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) en, (ValaCodeVisitor*) self);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL) ? NULL : vala_scope_ref (_tmp2)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp3);
}


static void vala_symbol_resolver_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (ed != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) ed), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) ed, (ValaCodeVisitor*) self);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL) ? NULL : vala_scope_ref (_tmp2)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp3);
}


static void vala_symbol_resolver_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (cb != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) cb), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) cb, (ValaCodeVisitor*) self);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL) ? NULL : vala_scope_ref (_tmp2)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp3);
}


static void vala_symbol_resolver_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (c != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) c), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) c, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (f != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) f), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) f, (ValaCodeVisitor*) self);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL) ? NULL : vala_scope_ref (_tmp2)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp3);
}


static void vala_symbol_resolver_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (m != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope ((ValaSymbol*) m), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp1);
	vala_code_node_accept_children ((ValaCodeNode*) m, (ValaCodeVisitor*) self);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL) ? NULL : vala_scope_ref (_tmp2)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp3);
}


static void vala_symbol_resolver_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (m != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) m, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (p != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) p, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (prop != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) prop, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (acc != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) acc, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (sig != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) sig, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (c != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) c, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (d != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) d, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (b != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) b, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_using_directive (ValaCodeVisitor* base, ValaUsingDirective* ns) {
	ValaSymbolResolver * self;
	ValaUnresolvedSymbol* _tmp1;
	ValaSymbol* _tmp0;
	ValaUnresolvedSymbol* unresolved_symbol;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (ns != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	unresolved_symbol = (_tmp1 = (_tmp0 = vala_using_directive_get_namespace_symbol (ns), VALA_IS_UNRESOLVED_SYMBOL (_tmp0) ? ((ValaUnresolvedSymbol*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (unresolved_symbol != NULL) {
		ValaSymbol* _tmp2;
		_tmp2 = NULL;
		vala_using_directive_set_namespace_symbol (ns, _tmp2 = vala_symbol_resolver_resolve_symbol (self, unresolved_symbol));
		(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
		if (vala_using_directive_get_namespace_symbol (ns) == NULL) {
			char* _tmp4;
			char* _tmp3;
			vala_code_node_set_error ((ValaCodeNode*) ns, TRUE);
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) ns), _tmp4 = g_strdup_printf ("The namespace name `%s' could not be found", _tmp3 = vala_code_node_to_string ((ValaCodeNode*) unresolved_symbol)));
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
			(unresolved_symbol == NULL) ? NULL : (unresolved_symbol = (vala_code_node_unref (unresolved_symbol), NULL));
			return;
		}
	}
	(unresolved_symbol == NULL) ? NULL : (unresolved_symbol = (vala_code_node_unref (unresolved_symbol), NULL));
}


static ValaSymbol* vala_symbol_resolver_resolve_symbol (ValaSymbolResolver* self, ValaUnresolvedSymbol* unresolved_symbol) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (unresolved_symbol != NULL, NULL);
	if (vala_unresolved_symbol_get_qualified (unresolved_symbol)) {
		/* qualified access to global symbol*/
		return vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), vala_symbol_get_name ((ValaSymbol*) unresolved_symbol));
	} else {
		if (vala_unresolved_symbol_get_inner (unresolved_symbol) == NULL) {
			ValaSymbol* sym;
			ValaScope* _tmp1;
			ValaScope* scope;
			ValaSymbol* _tmp19;
			sym = NULL;
			_tmp1 = NULL;
			scope = (_tmp1 = self->priv->current_scope, (_tmp1 == NULL) ? NULL : vala_scope_ref (_tmp1));
			while (TRUE) {
				gboolean _tmp2;
				ValaSymbol* _tmp3;
				gboolean _tmp4;
				gboolean _tmp5;
				ValaScope* _tmp8;
				ValaScope* _tmp7;
				_tmp2 = FALSE;
				if (sym == NULL) {
					_tmp2 = scope != NULL;
				} else {
					_tmp2 = FALSE;
				}
				if (!_tmp2) {
					break;
				}
				_tmp3 = NULL;
				sym = (_tmp3 = vala_scope_lookup (scope, vala_symbol_get_name ((ValaSymbol*) unresolved_symbol)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp3);
				_tmp4 = FALSE;
				_tmp5 = FALSE;
				if (VALA_IS_NAMESPACE (sym)) {
					_tmp5 = TRUE;
				} else {
					_tmp5 = VALA_IS_TYPESYMBOL (sym);
				}
				if (_tmp5) {
					_tmp4 = TRUE;
				} else {
					_tmp4 = VALA_IS_TYPEPARAMETER (sym);
				}
				/* only look for types and type containers*/
				if (!(_tmp4)) {
					ValaSymbol* _tmp6;
					_tmp6 = NULL;
					sym = (_tmp6 = NULL, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp6);
				}
				_tmp8 = NULL;
				_tmp7 = NULL;
				scope = (_tmp8 = (_tmp7 = vala_scope_get_parent_scope (scope), (_tmp7 == NULL) ? NULL : vala_scope_ref (_tmp7)), (scope == NULL) ? NULL : (scope = (vala_scope_unref (scope), NULL)), _tmp8);
			}
			if (sym == NULL) {
				{
					GeeIterator* _ns_it;
					_ns_it = gee_iterable_iterator ((GeeIterable*) self->priv->current_using_directives);
					while (gee_iterator_next (_ns_it)) {
						ValaUsingDirective* ns;
						gboolean _tmp9;
						ValaSymbol* local_sym;
						gboolean _tmp10;
						gboolean _tmp11;
						ns = (ValaUsingDirective*) gee_iterator_get (_ns_it);
						_tmp9 = FALSE;
						if (vala_code_node_get_error ((ValaCodeNode*) ns)) {
							_tmp9 = TRUE;
						} else {
							_tmp9 = VALA_IS_UNRESOLVED_SYMBOL (vala_using_directive_get_namespace_symbol (ns));
						}
						if (_tmp9) {
							(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
							continue;
						}
						local_sym = vala_scope_lookup (vala_symbol_get_scope (vala_using_directive_get_namespace_symbol (ns)), vala_symbol_get_name ((ValaSymbol*) unresolved_symbol));
						_tmp10 = FALSE;
						_tmp11 = FALSE;
						if (VALA_IS_NAMESPACE (local_sym)) {
							_tmp11 = TRUE;
						} else {
							_tmp11 = VALA_IS_TYPESYMBOL (local_sym);
						}
						if (_tmp11) {
							_tmp10 = TRUE;
						} else {
							_tmp10 = VALA_IS_TYPEPARAMETER (sym);
						}
						/* only look for types and type containers*/
						if (!(_tmp10)) {
							ValaSymbol* _tmp12;
							_tmp12 = NULL;
							local_sym = (_tmp12 = NULL, (local_sym == NULL) ? NULL : (local_sym = (vala_code_node_unref (local_sym), NULL)), _tmp12);
						}
						if (local_sym != NULL) {
							ValaSymbol* _tmp18;
							ValaSymbol* _tmp17;
							if (sym != NULL) {
								char* _tmp15;
								char* _tmp14;
								char* _tmp13;
								ValaSymbol* _tmp16;
								vala_code_node_set_error ((ValaCodeNode*) unresolved_symbol, TRUE);
								_tmp15 = NULL;
								_tmp14 = NULL;
								_tmp13 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_symbol), _tmp15 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_symbol_get_name ((ValaSymbol*) unresolved_symbol), _tmp13 = vala_symbol_get_full_name (sym), _tmp14 = vala_symbol_get_full_name (local_sym)));
								_tmp15 = (g_free (_tmp15), NULL);
								_tmp14 = (g_free (_tmp14), NULL);
								_tmp13 = (g_free (_tmp13), NULL);
								_tmp16 = NULL;
								return (_tmp16 = NULL, (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), (local_sym == NULL) ? NULL : (local_sym = (vala_code_node_unref (local_sym), NULL)), (_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (scope == NULL) ? NULL : (scope = (vala_scope_unref (scope), NULL)), _tmp16);
							}
							_tmp18 = NULL;
							_tmp17 = NULL;
							sym = (_tmp18 = (_tmp17 = local_sym, (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp18);
						}
						(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
						(local_sym == NULL) ? NULL : (local_sym = (vala_code_node_unref (local_sym), NULL));
					}
					(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
				}
			}
			_tmp19 = NULL;
			return (_tmp19 = sym, (scope == NULL) ? NULL : (scope = (vala_scope_unref (scope), NULL)), _tmp19);
		} else {
			ValaSymbol* parent_symbol;
			ValaSymbol* _tmp22;
			parent_symbol = vala_symbol_resolver_resolve_symbol (self, vala_unresolved_symbol_get_inner (unresolved_symbol));
			if (parent_symbol == NULL) {
				char* _tmp20;
				ValaSymbol* _tmp21;
				vala_code_node_set_error ((ValaCodeNode*) unresolved_symbol, TRUE);
				_tmp20 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_unresolved_symbol_get_inner (unresolved_symbol)), _tmp20 = g_strdup_printf ("The symbol `%s' could not be found", vala_symbol_get_name ((ValaSymbol*) vala_unresolved_symbol_get_inner (unresolved_symbol))));
				_tmp20 = (g_free (_tmp20), NULL);
				_tmp21 = NULL;
				return (_tmp21 = NULL, (parent_symbol == NULL) ? NULL : (parent_symbol = (vala_code_node_unref (parent_symbol), NULL)), _tmp21);
			}
			_tmp22 = NULL;
			return (_tmp22 = vala_scope_lookup (vala_symbol_get_scope (parent_symbol), vala_symbol_get_name ((ValaSymbol*) unresolved_symbol)), (parent_symbol == NULL) ? NULL : (parent_symbol = (vala_code_node_unref (parent_symbol), NULL)), _tmp22);
		}
	}
}


static ValaDataType* vala_symbol_resolver_resolve_type (ValaSymbolResolver* self, ValaUnresolvedType* unresolved_type) {
	ValaDataType* type;
	ValaSymbol* sym;
	ValaDataType* _tmp26;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (unresolved_type != NULL, NULL);
	type = NULL;
	/* still required for vapigen*/
	if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) vala_unresolved_type_get_unresolved_symbol (unresolved_type)), "void") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = (ValaDataType*) vala_void_type_new (NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0);
	}
	sym = vala_symbol_resolver_resolve_symbol (self, vala_unresolved_type_get_unresolved_symbol (unresolved_type));
	if (sym == NULL) {
		ValaDataType* _tmp3;
		/* don't report same error twice*/
		if (!vala_code_node_get_error ((ValaCodeNode*) vala_unresolved_type_get_unresolved_symbol (unresolved_type))) {
			char* _tmp2;
			char* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_type), _tmp2 = g_strdup_printf ("The type name `%s' could not be found", _tmp1 = vala_code_node_to_string ((ValaCodeNode*) vala_unresolved_type_get_unresolved_symbol (unresolved_type))));
			_tmp2 = (g_free (_tmp2), NULL);
			_tmp1 = (g_free (_tmp1), NULL);
		}
		_tmp3 = NULL;
		return (_tmp3 = (ValaDataType*) vala_invalid_type_new (), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp3);
	}
	if (VALA_IS_TYPEPARAMETER (sym)) {
		ValaDataType* _tmp4;
		_tmp4 = NULL;
		type = (_tmp4 = (ValaDataType*) vala_generic_type_new (VALA_TYPEPARAMETER (sym)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp4);
	} else {
		if (VALA_IS_TYPESYMBOL (sym)) {
			if (VALA_IS_DELEGATE (sym)) {
				ValaDataType* _tmp5;
				_tmp5 = NULL;
				type = (_tmp5 = (ValaDataType*) vala_delegate_type_new (VALA_DELEGATE (sym)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp5);
			} else {
				if (VALA_IS_CLASS (sym)) {
					ValaClass* _tmp6;
					ValaClass* cl;
					_tmp6 = NULL;
					cl = (_tmp6 = VALA_CLASS (sym), (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6));
					if (vala_class_get_is_error_base (cl)) {
						ValaDataType* _tmp7;
						_tmp7 = NULL;
						type = (_tmp7 = (ValaDataType*) vala_error_type_new (NULL, NULL, vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_type)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp7);
					} else {
						ValaDataType* _tmp8;
						_tmp8 = NULL;
						type = (_tmp8 = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) cl), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp8);
					}
					(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
				} else {
					if (VALA_IS_INTERFACE (sym)) {
						ValaDataType* _tmp9;
						_tmp9 = NULL;
						type = (_tmp9 = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) VALA_INTERFACE (sym)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp9);
					} else {
						if (VALA_IS_STRUCT (sym)) {
							ValaStruct* _tmp10;
							ValaStruct* st;
							_tmp10 = NULL;
							st = (_tmp10 = VALA_STRUCT (sym), (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10));
							if (vala_struct_is_boolean_type (st)) {
								ValaDataType* _tmp11;
								_tmp11 = NULL;
								type = (_tmp11 = (ValaDataType*) vala_boolean_type_new (st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp11);
							} else {
								if (vala_struct_is_integer_type (st)) {
									ValaDataType* _tmp12;
									_tmp12 = NULL;
									type = (_tmp12 = (ValaDataType*) vala_integer_type_new (st, NULL, NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp12);
								} else {
									if (vala_struct_is_floating_type (st)) {
										ValaDataType* _tmp13;
										_tmp13 = NULL;
										type = (_tmp13 = (ValaDataType*) vala_floating_type_new (st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp13);
									} else {
										ValaDataType* _tmp14;
										_tmp14 = NULL;
										type = (_tmp14 = (ValaDataType*) vala_struct_value_type_new (st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp14);
									}
								}
							}
							(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
						} else {
							if (VALA_IS_ENUM (sym)) {
								ValaDataType* _tmp15;
								_tmp15 = NULL;
								type = (_tmp15 = (ValaDataType*) vala_enum_value_type_new (VALA_ENUM (sym)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp15);
							} else {
								if (VALA_IS_ERROR_DOMAIN (sym)) {
									ValaDataType* _tmp16;
									_tmp16 = NULL;
									type = (_tmp16 = (ValaDataType*) vala_error_type_new (VALA_ERROR_DOMAIN (sym), NULL, vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_type)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp16);
								} else {
									if (VALA_IS_ERROR_CODE (sym)) {
										ValaDataType* _tmp17;
										_tmp17 = NULL;
										type = (_tmp17 = (ValaDataType*) vala_error_type_new (VALA_ERROR_DOMAIN (vala_symbol_get_parent_symbol (sym)), VALA_ERROR_CODE (sym), vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_type)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp17);
									} else {
										char* _tmp19;
										char* _tmp18;
										ValaDataType* _tmp20;
										_tmp19 = NULL;
										_tmp18 = NULL;
										vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_type), _tmp19 = g_strdup_printf ("internal error: `%s' is not a supported type", _tmp18 = vala_symbol_get_full_name (sym)));
										_tmp19 = (g_free (_tmp19), NULL);
										_tmp18 = (g_free (_tmp18), NULL);
										_tmp20 = NULL;
										return (_tmp20 = (ValaDataType*) vala_invalid_type_new (), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp20);
									}
								}
							}
						}
					}
				}
			}
		} else {
			char* _tmp22;
			char* _tmp21;
			ValaDataType* _tmp23;
			_tmp22 = NULL;
			_tmp21 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_type), _tmp22 = g_strdup_printf ("`%s' is not a type", _tmp21 = vala_symbol_get_full_name (sym)));
			_tmp22 = (g_free (_tmp22), NULL);
			_tmp21 = (g_free (_tmp21), NULL);
			_tmp23 = NULL;
			return (_tmp23 = (ValaDataType*) vala_invalid_type_new (), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp23);
		}
	}
	vala_code_node_set_source_reference ((ValaCodeNode*) type, vala_code_node_get_source_reference ((ValaCodeNode*) unresolved_type));
	vala_data_type_set_value_owned (type, vala_data_type_get_value_owned ((ValaDataType*) unresolved_type));
	if (VALA_IS_GENERIC_TYPE (type)) {
		/* type parameters are always considered nullable
		 actual type argument may or may not be nullable*/
		vala_data_type_set_nullable (type, TRUE);
	} else {
		vala_data_type_set_nullable (type, vala_data_type_get_nullable ((ValaDataType*) unresolved_type));
	}
	vala_data_type_set_is_dynamic (type, vala_data_type_get_is_dynamic ((ValaDataType*) unresolved_type));
	{
		GeeList* _tmp24;
		GeeIterator* _tmp25;
		GeeIterator* _type_arg_it;
		_tmp24 = NULL;
		_tmp25 = NULL;
		_type_arg_it = (_tmp25 = gee_iterable_iterator ((GeeIterable*) (_tmp24 = vala_data_type_get_type_arguments ((ValaDataType*) unresolved_type))), (_tmp24 == NULL) ? NULL : (_tmp24 = (gee_collection_object_unref (_tmp24), NULL)), _tmp25);
		while (gee_iterator_next (_type_arg_it)) {
			ValaDataType* type_arg;
			type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
			vala_data_type_add_type_argument (type, type_arg);
			(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
		}
		(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
	}
	_tmp26 = NULL;
	return (_tmp26 = type, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp26);
}


static void vala_symbol_resolver_real_visit_data_type (ValaCodeVisitor* base, ValaDataType* data_type) {
	ValaSymbolResolver * self;
	ValaUnresolvedType* _tmp0;
	ValaUnresolvedType* unresolved_type;
	ValaDataType* _tmp1;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (data_type != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) data_type, (ValaCodeVisitor*) self);
	if (!(VALA_IS_UNRESOLVED_TYPE (data_type))) {
		return;
	}
	_tmp0 = NULL;
	unresolved_type = (_tmp0 = VALA_UNRESOLVED_TYPE (data_type), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
	_tmp1 = NULL;
	vala_code_node_replace_type (vala_code_node_get_parent_node ((ValaCodeNode*) unresolved_type), (ValaDataType*) unresolved_type, _tmp1 = vala_symbol_resolver_resolve_type (self, unresolved_type));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	(unresolved_type == NULL) ? NULL : (unresolved_type = (vala_code_node_unref (unresolved_type), NULL));
}


static void vala_symbol_resolver_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (local != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) local, (ValaCodeVisitor*) self);
	if (VALA_IS_REFERENCE_TYPE (vala_local_variable_get_variable_type (local))) {
		vala_data_type_set_nullable (vala_local_variable_get_variable_type (local), TRUE);
	}
}


static void vala_symbol_resolver_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (list != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) list, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (section != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) section, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_switch_label (ValaCodeVisitor* base, ValaSwitchLabel* label) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (label != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) label, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (clause != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) clause, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* e) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (e != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) e, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_method_call (ValaCodeVisitor* base, ValaMethodCall* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (l != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) l, (ValaCodeVisitor*) self);
}


static void vala_symbol_resolver_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a) {
	ValaSymbolResolver * self;
	self = (ValaSymbolResolver*) base;
	g_return_if_fail (a != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) a, (ValaCodeVisitor*) self);
}


/**
 * Code visitor resolving symbol names.
 */
ValaSymbolResolver* vala_symbol_resolver_construct (GType object_type) {
	ValaSymbolResolver* self;
	self = (ValaSymbolResolver*) g_type_create_instance (object_type);
	return self;
}


ValaSymbolResolver* vala_symbol_resolver_new (void) {
	return vala_symbol_resolver_construct (VALA_TYPE_SYMBOL_RESOLVER);
}


static void vala_symbol_resolver_class_init (ValaSymbolResolverClass * klass) {
	vala_symbol_resolver_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_symbol_resolver_finalize;
	g_type_class_add_private (klass, sizeof (ValaSymbolResolverPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_symbol_resolver_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_symbol_resolver_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_symbol_resolver_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_symbol_resolver_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_symbol_resolver_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_symbol_resolver_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_symbol_resolver_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_symbol_resolver_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_symbol_resolver_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_symbol_resolver_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_symbol_resolver_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_symbol_resolver_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_symbol_resolver_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_symbol_resolver_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_symbol_resolver_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_symbol_resolver_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_symbol_resolver_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_symbol_resolver_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_using_directive = vala_symbol_resolver_real_visit_using_directive;
	VALA_CODE_VISITOR_CLASS (klass)->visit_data_type = vala_symbol_resolver_real_visit_data_type;
	VALA_CODE_VISITOR_CLASS (klass)->visit_declaration_statement = vala_symbol_resolver_real_visit_declaration_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_local_variable = vala_symbol_resolver_real_visit_local_variable;
	VALA_CODE_VISITOR_CLASS (klass)->visit_initializer_list = vala_symbol_resolver_real_visit_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_symbol_resolver_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_symbol_resolver_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_statement = vala_symbol_resolver_real_visit_switch_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_symbol_resolver_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_label = vala_symbol_resolver_real_visit_switch_label;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_symbol_resolver_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_symbol_resolver_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_symbol_resolver_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_symbol_resolver_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_symbol_resolver_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_yield_statement = vala_symbol_resolver_real_visit_yield_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_symbol_resolver_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_symbol_resolver_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_symbol_resolver_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_array_creation_expression = vala_symbol_resolver_real_visit_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_symbol_resolver_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_symbol_resolver_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method_call = vala_symbol_resolver_real_visit_method_call;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_symbol_resolver_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_object_creation_expression = vala_symbol_resolver_real_visit_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_symbol_resolver_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_symbol_resolver_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_symbol_resolver_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_symbol_resolver_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_symbol_resolver_real_visit_assignment;
}


static void vala_symbol_resolver_instance_init (ValaSymbolResolver * self) {
	self->priv = VALA_SYMBOL_RESOLVER_GET_PRIVATE (self);
}


static void vala_symbol_resolver_finalize (ValaCodeVisitor* obj) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (obj);
	(self->priv->root_symbol == NULL) ? NULL : (self->priv->root_symbol = (vala_code_node_unref (self->priv->root_symbol), NULL));
	(self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL));
	(self->priv->current_using_directives == NULL) ? NULL : (self->priv->current_using_directives = (gee_collection_object_unref (self->priv->current_using_directives), NULL));
	VALA_CODE_VISITOR_CLASS (vala_symbol_resolver_parent_class)->finalize (obj);
}


GType vala_symbol_resolver_get_type (void) {
	static GType vala_symbol_resolver_type_id = 0;
	if (vala_symbol_resolver_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSymbolResolverClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_symbol_resolver_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSymbolResolver), 0, (GInstanceInitFunc) vala_symbol_resolver_instance_init, NULL };
		vala_symbol_resolver_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSymbolResolver", &g_define_type_info, 0);
	}
	return vala_symbol_resolver_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




