/* valadostatement.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valadostatement.h>
#include <stdlib.h>
#include <string.h>
#include <gee/list.h>
#include <vala/valaexpression.h>
#include <vala/valablock.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadatatype.h>
#include <vala/valareport.h>
#include <vala/valalocalvariable.h>
#include <vala/valabooleanliteral.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valasymbol.h>
#include <vala/valaunaryexpression.h>
#include <vala/valabreakstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valamemberaccess.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaassignment.h>




struct _ValaDoStatementPrivate {
	ValaExpression* _condition;
	ValaBlock* _body;
};

#define VALA_DO_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DO_STATEMENT, ValaDoStatementPrivate))
enum  {
	VALA_DO_STATEMENT_DUMMY_PROPERTY
};
static void vala_do_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_do_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_do_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_do_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_do_statement_parent_class = NULL;
static ValaStatementIface* vala_do_statement_vala_statement_parent_iface = NULL;
static void vala_do_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new do statement.
 *
 * @param cond   loop condition
 * @param body   loop body
 * @param source reference to source code
 * @return       newly created do statement
 */
ValaDoStatement* vala_do_statement_construct (GType object_type, ValaBlock* body, ValaExpression* condition, ValaSourceReference* source_reference) {
	ValaDoStatement* self;
	g_return_val_if_fail (body != NULL, NULL);
	g_return_val_if_fail (condition != NULL, NULL);
	self = (ValaDoStatement*) g_type_create_instance (object_type);
	vala_do_statement_set_condition (self, condition);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_do_statement_set_body (self, body);
	return self;
}


ValaDoStatement* vala_do_statement_new (ValaBlock* body, ValaExpression* condition, ValaSourceReference* source_reference) {
	return vala_do_statement_construct (VALA_TYPE_DO_STATEMENT, body, condition, source_reference);
}


static void vala_do_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDoStatement * self;
	self = (ValaDoStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_do_statement (visitor, self);
}


static void vala_do_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDoStatement * self;
	self = (ValaDoStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_do_statement_get_body (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_do_statement_get_condition (self), visitor);
	vala_code_visitor_visit_end_full_expression (visitor, vala_do_statement_get_condition (self));
}


static void vala_do_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaDoStatement * self;
	self = (ValaDoStatement*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_do_statement_get_condition (self) == old_node) {
		vala_do_statement_set_condition (self, new_node);
	}
}


static gboolean vala_do_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaDoStatement * self;
	GeeList* _tmp3;
	GeeList* _tmp4;
	self = (ValaDoStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_do_statement_get_condition (self), analyzer)) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (!vala_data_type_compatible (vala_expression_get_value_type (vala_do_statement_get_condition (self)), analyzer->bool_type)) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_do_statement_get_condition (self)), "Condition must be boolean");
		return FALSE;
	}
	vala_code_node_check ((ValaCodeNode*) vala_do_statement_get_body (self), analyzer);
	_tmp3 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp3 = vala_code_node_get_error_types ((ValaCodeNode*) vala_do_statement_get_condition (self)));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL));
	_tmp4 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp4 = vala_code_node_get_error_types ((ValaCodeNode*) vala_do_statement_get_body (self)));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


ValaBlock* vala_do_statement_prepare_condition_split (ValaDoStatement* self, ValaSemanticAnalyzer* analyzer) {
	ValaBooleanLiteral* _tmp2;
	char* _tmp1;
	ValaDataType* _tmp0;
	ValaLocalVariable* _tmp3;
	ValaLocalVariable* first_local;
	ValaDeclarationStatement* first_decl;
	ValaBlock* _tmp4;
	ValaBlock* block;
	ValaUnaryExpression* if_condition;
	ValaBlock* true_block;
	ValaBreakStatement* _tmp5;
	ValaIfStatement* if_stmt;
	ValaBlock* condition_block;
	ValaMemberAccess* _tmp6;
	ValaIfStatement* _tmp7;
	ValaIfStatement* first_if;
	ValaExpressionStatement* _tmp11;
	ValaAssignment* _tmp10;
	ValaBooleanLiteral* _tmp9;
	ValaMemberAccess* _tmp8;
	ValaBooleanLiteral* _tmp12;
	ValaBlock* _tmp13;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (analyzer != NULL, NULL);
	/* move condition into the loop body to allow split
	 * in multiple statements
	 *
	 * first = false;
	 * do {
	 *     if (first) {
	 *         if (!condition) {
	 *             break;
	 *         }
	 *     }
	 *     first = true;
	 *     ...
	 * } while (true);
	 */
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp3 = NULL;
	first_local = (_tmp3 = vala_local_variable_new (_tmp0 = vala_data_type_copy (analyzer->bool_type), _tmp1 = vala_code_node_get_temp_name ((ValaCodeNode*) self), (ValaExpression*) (_tmp2 = vala_boolean_literal_new (FALSE, vala_code_node_get_source_reference ((ValaCodeNode*) self))), vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)), _tmp1 = (g_free (_tmp1), NULL), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp3);
	first_decl = vala_declaration_statement_new ((ValaSymbol*) first_local, vala_code_node_get_source_reference ((ValaCodeNode*) self));
	vala_code_node_check ((ValaCodeNode*) first_decl, analyzer);
	_tmp4 = NULL;
	block = (_tmp4 = VALA_BLOCK (vala_semantic_analyzer_get_current_symbol (analyzer)), (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4));
	vala_block_insert_before (block, (ValaStatement*) self, (ValaStatement*) first_decl);
	if_condition = vala_unary_expression_new (VALA_UNARY_OPERATOR_LOGICAL_NEGATION, vala_do_statement_get_condition (self), vala_code_node_get_source_reference ((ValaCodeNode*) vala_do_statement_get_condition (self)));
	true_block = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) vala_do_statement_get_condition (self)));
	_tmp5 = NULL;
	vala_block_add_statement (true_block, (ValaStatement*) (_tmp5 = vala_break_statement_new (vala_code_node_get_source_reference ((ValaCodeNode*) vala_do_statement_get_condition (self)))));
	(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
	if_stmt = vala_if_statement_new ((ValaExpression*) if_condition, true_block, NULL, vala_code_node_get_source_reference ((ValaCodeNode*) vala_do_statement_get_condition (self)));
	condition_block = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) vala_do_statement_get_condition (self)));
	vala_block_add_statement (condition_block, (ValaStatement*) if_stmt);
	_tmp6 = NULL;
	_tmp7 = NULL;
	first_if = (_tmp7 = vala_if_statement_new ((ValaExpression*) (_tmp6 = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) first_local), vala_code_node_get_source_reference ((ValaCodeNode*) self))), condition_block, NULL, vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)), _tmp7);
	vala_block_insert_statement (vala_do_statement_get_body (self), 0, (ValaStatement*) first_if);
	_tmp11 = NULL;
	_tmp10 = NULL;
	_tmp9 = NULL;
	_tmp8 = NULL;
	vala_block_insert_statement (vala_do_statement_get_body (self), 1, (ValaStatement*) (_tmp11 = vala_expression_statement_new ((ValaExpression*) (_tmp10 = vala_assignment_new ((ValaExpression*) (_tmp8 = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) first_local), vala_code_node_get_source_reference ((ValaCodeNode*) self))), (ValaExpression*) (_tmp9 = vala_boolean_literal_new (TRUE, vala_code_node_get_source_reference ((ValaCodeNode*) self))), VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference ((ValaCodeNode*) self))), vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL));
	(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
	(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
	(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
	_tmp12 = NULL;
	vala_do_statement_set_condition (self, (ValaExpression*) (_tmp12 = vala_boolean_literal_new (TRUE, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
	vala_code_node_check ((ValaCodeNode*) vala_do_statement_get_condition (self), analyzer);
	_tmp13 = NULL;
	return (_tmp13 = condition_block, (first_local == NULL) ? NULL : (first_local = (vala_code_node_unref (first_local), NULL)), (first_decl == NULL) ? NULL : (first_decl = (vala_code_node_unref (first_decl), NULL)), (block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL)), (if_condition == NULL) ? NULL : (if_condition = (vala_code_node_unref (if_condition), NULL)), (true_block == NULL) ? NULL : (true_block = (vala_code_node_unref (true_block), NULL)), (if_stmt == NULL) ? NULL : (if_stmt = (vala_code_node_unref (if_stmt), NULL)), (first_if == NULL) ? NULL : (first_if = (vala_code_node_unref (first_if), NULL)), _tmp13);
}


ValaBlock* vala_do_statement_get_body (ValaDoStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_do_statement_set_body (ValaDoStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_body, (ValaCodeNode*) self);
}


ValaExpression* vala_do_statement_get_condition (ValaDoStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_condition;
}


void vala_do_statement_set_condition (ValaDoStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_condition = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_condition, (ValaCodeNode*) self);
}


static void vala_do_statement_class_init (ValaDoStatementClass * klass) {
	vala_do_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_do_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaDoStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_do_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_do_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_do_statement_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->check = vala_do_statement_real_check;
}


static void vala_do_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_do_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_do_statement_instance_init (ValaDoStatement * self) {
	self->priv = VALA_DO_STATEMENT_GET_PRIVATE (self);
}


static void vala_do_statement_finalize (ValaCodeNode* obj) {
	ValaDoStatement * self;
	self = VALA_DO_STATEMENT (obj);
	(self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL));
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL));
	VALA_CODE_NODE_CLASS (vala_do_statement_parent_class)->finalize (obj);
}


GType vala_do_statement_get_type (void) {
	static GType vala_do_statement_type_id = 0;
	if (vala_do_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDoStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_do_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDoStatement), 0, (GInstanceInitFunc) vala_do_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_do_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_do_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaDoStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_do_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_do_statement_type_id;
}




