/* valadatatype.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valadatatype.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valatypesymbol.h>
#include <vala/valatypeparameter.h>
#include <vala/valacodevisitor.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valascope.h>
#include <vala/valasymbol.h>
#include <vala/valadelegatetype.h>
#include <vala/valapointertype.h>
#include <vala/valaarraytype.h>
#include <vala/valaenum.h>
#include <vala/valastruct.h>
#include <vala/valagenerictype.h>
#include <vala/valasemanticanalyzer.h>




struct _ValaDataTypePrivate {
	gboolean _value_owned;
	gboolean _nullable;
	ValaTypeSymbol* _data_type;
	ValaTypeParameter* _type_parameter;
	gboolean _floating_reference;
	gboolean _is_dynamic;
	GeeList* type_argument_list;
};

#define VALA_DATA_TYPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DATA_TYPE, ValaDataTypePrivate))
enum  {
	VALA_DATA_TYPE_DUMMY_PROPERTY
};
static GeeList* vala_data_type__empty_type_list;
static void vala_data_type_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_data_type_real_get_cname (ValaDataType* self);
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix);
static char* vala_data_type_real_to_string (ValaCodeNode* base);
static char* vala_data_type_real_to_qualified_string (ValaDataType* self, ValaScope* scope);
static ValaDataType* vala_data_type_real_copy (ValaDataType* self);
static gboolean vala_data_type_real_equals (ValaDataType* self, ValaDataType* type2);
static gboolean vala_data_type_real_stricter (ValaDataType* self, ValaDataType* type2);
static void vala_data_type_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_data_type_real_compatible (ValaDataType* self, ValaDataType* target_type);
static gboolean vala_data_type_real_is_invokable (ValaDataType* self);
static ValaDataType* vala_data_type_real_get_return_type (ValaDataType* self);
static GeeList* vala_data_type_real_get_parameters (ValaDataType* self);
static gboolean vala_data_type_real_is_reference_type_or_type_parameter (ValaDataType* self);
static gboolean vala_data_type_real_is_array (ValaDataType* self);
static GeeList* vala_data_type_real_get_symbols (ValaDataType* self);
static ValaSymbol* vala_data_type_real_get_member (ValaDataType* self, const char* member_name);
static ValaSymbol* vala_data_type_real_get_pointer_member (ValaDataType* self, const char* member_name);
static gboolean vala_data_type_real_is_real_struct_type (ValaDataType* self);
static char* vala_data_type_real_get_type_id (ValaDataType* self);
static char* vala_data_type_real_get_type_signature (ValaDataType* self);
static gboolean vala_data_type_real_is_disposable (ValaDataType* self);
static gpointer vala_data_type_parent_class = NULL;
static void vala_data_type_finalize (ValaCodeNode* obj);



/**
 * Appends the specified type as generic type argument.
 *
 * @param arg a type reference
 */
void vala_data_type_add_type_argument (ValaDataType* self, ValaDataType* arg) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (arg != NULL);
	if (self->priv->type_argument_list == NULL) {
		GeeList* _tmp0;
		_tmp0 = NULL;
		self->priv->type_argument_list = (_tmp0 = (GeeList*) gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (self->priv->type_argument_list == NULL) ? NULL : (self->priv->type_argument_list = (gee_collection_object_unref (self->priv->type_argument_list), NULL)), _tmp0);
	}
	gee_collection_add ((GeeCollection*) self->priv->type_argument_list, arg);
	vala_code_node_set_parent_node ((ValaCodeNode*) arg, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the list of generic type arguments.
 *
 * @return type argument list
 */
GeeList* vala_data_type_get_type_arguments (ValaDataType* self) {
	GeeList* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->type_argument_list != NULL) {
		GeeList* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->type_argument_list, (_tmp0 == NULL) ? NULL : gee_collection_object_ref (_tmp0));
	}
	if (vala_data_type__empty_type_list == NULL) {
		GeeList* _tmp3;
		GeeArrayList* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		vala_data_type__empty_type_list = (_tmp3 = (GeeList*) gee_read_only_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, (GeeList*) (_tmp2 = gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal))), (vala_data_type__empty_type_list == NULL) ? NULL : (vala_data_type__empty_type_list = (gee_collection_object_unref (vala_data_type__empty_type_list), NULL)), _tmp3);
		(_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL));
	}
	_tmp4 = NULL;
	return (_tmp4 = vala_data_type__empty_type_list, (_tmp4 == NULL) ? NULL : gee_collection_object_ref (_tmp4));
}


/**
 * Removes all generic type arguments.
 */
void vala_data_type_remove_all_type_arguments (ValaDataType* self) {
	GeeList* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp0 = NULL;
	self->priv->type_argument_list = (_tmp0 = NULL, (self->priv->type_argument_list == NULL) ? NULL : (self->priv->type_argument_list = (gee_collection_object_unref (self->priv->type_argument_list), NULL)), _tmp0);
}


static void vala_data_type_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDataType * self;
	gboolean _tmp0;
	self = (ValaDataType*) base;
	g_return_if_fail (visitor != NULL);
	_tmp0 = FALSE;
	if (self->priv->type_argument_list != NULL) {
		_tmp0 = gee_collection_get_size ((GeeCollection*) self->priv->type_argument_list) > 0;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_argument_list);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				vala_code_node_accept ((ValaCodeNode*) type_arg, visitor);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
	}
	vala_code_visitor_visit_data_type (visitor, self);
}


/**
 * Returns the name and qualifiers of this type as it is used in C code.
 *
 * @return the type string to be used in C code
 */
static char* vala_data_type_real_get_cname (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	/* raise error*/
	vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unresolved type reference");
	return NULL;
}


char* vala_data_type_get_cname (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_cname (self);
}


/**
 * Returns the name and qualifiers of this type as it is used in C code
 * in a const declaration.
 *
 * @return the type string to be used in C code const declarations
 */
char* vala_data_type_get_const_cname (ValaDataType* self) {
	char* ptr;
	ValaTypeSymbol* t;
	char* _tmp6;
	char* _tmp7;
	char* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	ptr = NULL;
	t = NULL;
	/* FIXME: workaround to make constant arrays possible*/
	if (VALA_IS_ARRAY_TYPE (self)) {
		ValaTypeSymbol* _tmp1;
		ValaTypeSymbol* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		t = (_tmp1 = (_tmp0 = vala_array_type_get_element_type ((VALA_ARRAY_TYPE (self)))->priv->_data_type, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0)), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp1);
	} else {
		ValaTypeSymbol* _tmp3;
		ValaTypeSymbol* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		t = (_tmp3 = (_tmp2 = self->priv->_data_type, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2)), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp3);
	}
	if (!vala_typesymbol_is_reference_type (t)) {
		char* _tmp4;
		_tmp4 = NULL;
		ptr = (_tmp4 = g_strdup (""), ptr = (g_free (ptr), NULL), _tmp4);
	} else {
		char* _tmp5;
		_tmp5 = NULL;
		ptr = (_tmp5 = g_strdup ("*"), ptr = (g_free (ptr), NULL), _tmp5);
	}
	_tmp6 = NULL;
	_tmp7 = NULL;
	_tmp8 = NULL;
	return (_tmp8 = (_tmp7 = g_strdup_printf ("const %s%s", _tmp6 = vala_typesymbol_get_cname (t, FALSE), ptr), _tmp6 = (g_free (_tmp6), NULL), _tmp7), ptr = (g_free (ptr), NULL), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp8);
}


/**
 * Returns the C name of this data type in lower case. Words are
 * separated by underscores.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the lower case name to be used in C code
 */
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix) {
	g_return_val_if_fail (self != NULL, NULL);
	return vala_symbol_get_lower_case_cname ((ValaSymbol*) self->priv->_data_type, infix);
}


char* vala_data_type_get_lower_case_cname (ValaDataType* self, const char* infix) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_lower_case_cname (self, infix);
}


static char* vala_data_type_real_to_string (ValaCodeNode* base) {
	ValaDataType * self;
	self = (ValaDataType*) base;
	return vala_data_type_to_qualified_string (self, NULL);
}


static char* vala_data_type_real_to_qualified_string (ValaDataType* self, ValaScope* scope) {
	char* s;
	GeeList* type_args;
	char* _tmp20;
	g_return_val_if_fail (self != NULL, NULL);
	s = NULL;
	if (self->priv->_data_type != NULL) {
		ValaSymbol* _tmp0;
		ValaSymbol* global_symbol;
		ValaSymbol* sym;
		ValaScope* _tmp3;
		ValaScope* parent_scope;
		gboolean _tmp8;
		_tmp0 = NULL;
		global_symbol = (_tmp0 = (ValaSymbol*) self->priv->_data_type, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
		while (vala_symbol_get_name (vala_symbol_get_parent_symbol (global_symbol)) != NULL) {
			ValaSymbol* _tmp2;
			ValaSymbol* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			global_symbol = (_tmp2 = (_tmp1 = vala_symbol_get_parent_symbol (global_symbol), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (global_symbol == NULL) ? NULL : (global_symbol = (vala_code_node_unref (global_symbol), NULL)), _tmp2);
		}
		sym = NULL;
		_tmp3 = NULL;
		parent_scope = (_tmp3 = scope, (_tmp3 == NULL) ? NULL : vala_scope_ref (_tmp3));
		while (TRUE) {
			gboolean _tmp4;
			ValaSymbol* _tmp5;
			ValaScope* _tmp7;
			ValaScope* _tmp6;
			_tmp4 = FALSE;
			if (sym == NULL) {
				_tmp4 = parent_scope != NULL;
			} else {
				_tmp4 = FALSE;
			}
			if (!_tmp4) {
				break;
			}
			_tmp5 = NULL;
			sym = (_tmp5 = vala_scope_lookup (parent_scope, vala_symbol_get_name (global_symbol)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp5);
			_tmp7 = NULL;
			_tmp6 = NULL;
			parent_scope = (_tmp7 = (_tmp6 = vala_scope_get_parent_scope (parent_scope), (_tmp6 == NULL) ? NULL : vala_scope_ref (_tmp6)), (parent_scope == NULL) ? NULL : (parent_scope = (vala_scope_unref (parent_scope), NULL)), _tmp7);
		}
		_tmp8 = FALSE;
		if (sym != NULL) {
			_tmp8 = global_symbol != sym;
		} else {
			_tmp8 = FALSE;
		}
		if (_tmp8) {
			char* _tmp10;
			char* _tmp9;
			_tmp10 = NULL;
			_tmp9 = NULL;
			s = (_tmp10 = g_strconcat ("global::", _tmp9 = vala_symbol_get_full_name ((ValaSymbol*) self->priv->_data_type), NULL), s = (g_free (s), NULL), _tmp10);
			_tmp9 = (g_free (_tmp9), NULL);
			;
		} else {
			char* _tmp11;
			_tmp11 = NULL;
			s = (_tmp11 = vala_symbol_get_full_name ((ValaSymbol*) self->priv->_data_type), s = (g_free (s), NULL), _tmp11);
		}
		(global_symbol == NULL) ? NULL : (global_symbol = (vala_code_node_unref (global_symbol), NULL));
		(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
		(parent_scope == NULL) ? NULL : (parent_scope = (vala_scope_unref (parent_scope), NULL));
	} else {
		char* _tmp12;
		_tmp12 = NULL;
		s = (_tmp12 = g_strdup ("null"), s = (g_free (s), NULL), _tmp12);
	}
	type_args = vala_data_type_get_type_arguments (self);
	if (gee_collection_get_size ((GeeCollection*) type_args) > 0) {
		char* _tmp13;
		gboolean first;
		char* _tmp18;
		_tmp13 = NULL;
		s = (_tmp13 = g_strconcat (s, ("<"), NULL), s = (g_free (s), NULL), _tmp13);
		first = TRUE;
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_args);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				char* _tmp17;
				char* _tmp16;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				if (!first) {
					char* _tmp14;
					_tmp14 = NULL;
					s = (_tmp14 = g_strconcat (s, (","), NULL), s = (g_free (s), NULL), _tmp14);
				} else {
					first = FALSE;
				}
				if (!type_arg->priv->_value_owned) {
					char* _tmp15;
					_tmp15 = NULL;
					s = (_tmp15 = g_strconcat (s, ("weak "), NULL), s = (g_free (s), NULL), _tmp15);
				}
				_tmp17 = NULL;
				_tmp16 = NULL;
				s = (_tmp17 = g_strconcat (s, _tmp16 = (vala_data_type_to_qualified_string (type_arg, scope)), NULL), s = (g_free (s), NULL), _tmp17);
				_tmp16 = (g_free (_tmp16), NULL);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
		_tmp18 = NULL;
		s = (_tmp18 = g_strconcat (s, (">"), NULL), s = (g_free (s), NULL), _tmp18);
	}
	if (self->priv->_nullable) {
		char* _tmp19;
		_tmp19 = NULL;
		s = (_tmp19 = g_strconcat (s, ("?"), NULL), s = (g_free (s), NULL), _tmp19);
	}
	_tmp20 = NULL;
	return (_tmp20 = s, (type_args == NULL) ? NULL : (type_args = (gee_collection_object_unref (type_args), NULL)), _tmp20);
}


char* vala_data_type_to_qualified_string (ValaDataType* self, ValaScope* scope) {
	return VALA_DATA_TYPE_GET_CLASS (self)->to_qualified_string (self, scope);
}


static ValaDataType* vala_data_type_real_copy (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	g_critical ("Type `%s' does not implement abstract method `vala_data_type_copy'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return NULL;
}


/**
 * Creates a shallow copy of this type reference.
 *
 * @return copy of this type reference
 */
ValaDataType* vala_data_type_copy (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->copy (self);
}


/**
 * Checks two type references for equality. May only be used with
 * resolved type references.
 *
 * @param type2 a type reference
 * @return      true if this type reference is equal to type2, false
 *              otherwise
 */
static gboolean vala_data_type_real_equals (ValaDataType* self, ValaDataType* type2) {
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type2 != NULL, FALSE);
	if (type2->priv->_value_owned != self->priv->_value_owned) {
		return FALSE;
	}
	if (type2->priv->_nullable != self->priv->_nullable) {
		return FALSE;
	}
	if (type2->priv->_data_type != self->priv->_data_type) {
		return FALSE;
	}
	_tmp3 = FALSE;
	if (type2->priv->_type_parameter != NULL) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = self->priv->_type_parameter != NULL;
	}
	if (_tmp3) {
		gboolean _tmp4;
		_tmp4 = FALSE;
		if (type2->priv->_type_parameter == NULL) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = self->priv->_type_parameter == NULL;
		}
		if (_tmp4) {
			return FALSE;
		}
		if (!vala_typeparameter_equals (type2->priv->_type_parameter, self->priv->_type_parameter)) {
			return FALSE;
		}
	}
	if (type2->priv->_floating_reference != self->priv->_floating_reference) {
		return FALSE;
	}
	return TRUE;
}


gboolean vala_data_type_equals (ValaDataType* self, ValaDataType* type2) {
	return VALA_DATA_TYPE_GET_CLASS (self)->equals (self, type2);
}


/**
 * Checks whether this type reference is at least as strict as the
 * specified type reference type2.
 *
 * @param type2 a type reference
 * @return      true if this type reference is stricter or equal
 */
static gboolean vala_data_type_real_stricter (ValaDataType* self, ValaDataType* type2) {
	gboolean _tmp1;
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type2 != NULL, FALSE);
	if (type2->priv->_value_owned != self->priv->_value_owned) {
		return FALSE;
	}
	_tmp1 = FALSE;
	if (!type2->priv->_nullable) {
		_tmp1 = self->priv->_nullable;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		return FALSE;
	}
	_tmp3 = FALSE;
	if (self->priv->_type_parameter != NULL) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = type2->priv->_type_parameter != NULL;
	}
	/* temporarily ignore type parameters */
	if (_tmp3) {
		return TRUE;
	}
	if (type2->priv->_data_type != self->priv->_data_type) {
		/* FIXME: allow this type reference to refer to a
		        subtype of the type type2 is referring to*/
		return FALSE;
	}
	if (type2->priv->_floating_reference != self->priv->_floating_reference) {
		return FALSE;
	}
	return TRUE;
}


gboolean vala_data_type_stricter (ValaDataType* self, ValaDataType* type2) {
	return VALA_DATA_TYPE_GET_CLASS (self)->stricter (self, type2);
}


static void vala_data_type_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaDataType * self;
	self = (ValaDataType*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (self->priv->type_argument_list != NULL) {
		{
			gint i;
			i = 0;
			for (; i < gee_collection_get_size ((GeeCollection*) self->priv->type_argument_list); i++) {
				ValaDataType* _tmp0;
				gboolean _tmp1;
				_tmp0 = NULL;
				if ((_tmp1 = (_tmp0 = (ValaDataType*) gee_list_get ((GeeList*) self->priv->type_argument_list, i)) == old_type, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
					gee_list_set ((GeeList*) self->priv->type_argument_list, i, new_type);
					return;
				}
			}
		}
	}
}


static gboolean vala_data_type_real_compatible (ValaDataType* self, ValaDataType* target_type) {
	gboolean _tmp0;
	gboolean _tmp9;
	gboolean _tmp10;
	gboolean _tmp13;
	gboolean _tmp22;
	gboolean _tmp23;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (target_type != NULL, FALSE);
	_tmp0 = FALSE;
	if (VALA_IS_DELEGATE_TYPE (target_type)) {
		_tmp0 = VALA_IS_DELEGATE_TYPE (self);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		return vala_delegate_type_get_delegate_symbol ((VALA_DELEGATE_TYPE (target_type))) == vala_delegate_type_get_delegate_symbol ((VALA_DELEGATE_TYPE (self)));
	}
	if (VALA_IS_POINTER_TYPE (target_type)) {
		gboolean _tmp2;
		_tmp2 = FALSE;
		if (self->priv->_type_parameter != NULL) {
			_tmp2 = TRUE;
		} else {
			gboolean _tmp3;
			_tmp3 = FALSE;
			if (self->priv->_data_type != NULL) {
				gboolean _tmp4;
				_tmp4 = FALSE;
				if (vala_typesymbol_is_reference_type (self->priv->_data_type)) {
					_tmp4 = TRUE;
				} else {
					_tmp4 = VALA_IS_DELEGATE_TYPE (self);
				}
				_tmp3 = (_tmp4);
			} else {
				_tmp3 = FALSE;
			}
			_tmp2 = (_tmp3);
		}
		/* any reference or array type or pointer type can be cast to a generic pointer */
		if (_tmp2) {
			return TRUE;
		}
		return FALSE;
	}
	/* temporarily ignore type parameters */
	if (target_type->priv->_type_parameter != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY_TYPE (self) != VALA_IS_ARRAY_TYPE (target_type)) {
		return FALSE;
	}
	_tmp9 = FALSE;
	_tmp10 = FALSE;
	if (VALA_IS_ENUM (self->priv->_data_type)) {
		_tmp10 = VALA_IS_STRUCT (target_type->priv->_data_type);
	} else {
		_tmp10 = FALSE;
	}
	if (_tmp10) {
		_tmp9 = vala_struct_is_integer_type ((VALA_STRUCT (target_type->priv->_data_type)));
	} else {
		_tmp9 = FALSE;
	}
	if (_tmp9) {
		return TRUE;
	}
	if (self->priv->_data_type == target_type->priv->_data_type) {
		return TRUE;
	}
	_tmp13 = FALSE;
	if (VALA_IS_STRUCT (self->priv->_data_type)) {
		_tmp13 = VALA_IS_STRUCT (target_type->priv->_data_type);
	} else {
		_tmp13 = FALSE;
	}
	if (_tmp13) {
		ValaStruct* _tmp14;
		ValaStruct* expr_struct;
		ValaStruct* _tmp15;
		ValaStruct* expect_struct;
		gboolean _tmp16;
		gboolean _tmp18;
		gboolean _tmp19;
		_tmp14 = NULL;
		expr_struct = (_tmp14 = VALA_STRUCT (self->priv->_data_type), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
		_tmp15 = NULL;
		expect_struct = (_tmp15 = VALA_STRUCT (target_type->priv->_data_type), (_tmp15 == NULL) ? NULL : vala_code_node_ref (_tmp15));
		_tmp16 = FALSE;
		if (vala_struct_is_integer_type (expr_struct)) {
			_tmp16 = vala_struct_is_floating_type (expect_struct);
		} else {
			_tmp16 = FALSE;
		}
		/* integer types may be implicitly cast to floating point types */
		if (_tmp16) {
			gboolean _tmp17;
			return (_tmp17 = TRUE, (expr_struct == NULL) ? NULL : (expr_struct = (vala_code_node_unref (expr_struct), NULL)), (expect_struct == NULL) ? NULL : (expect_struct = (vala_code_node_unref (expect_struct), NULL)), _tmp17);
		}
		_tmp18 = FALSE;
		_tmp19 = FALSE;
		if (vala_struct_is_integer_type (expr_struct)) {
			_tmp19 = vala_struct_is_integer_type (expect_struct);
		} else {
			_tmp19 = FALSE;
		}
		if ((_tmp19)) {
			_tmp18 = TRUE;
		} else {
			gboolean _tmp20;
			_tmp20 = FALSE;
			if (vala_struct_is_floating_type (expr_struct)) {
				_tmp20 = vala_struct_is_floating_type (expect_struct);
			} else {
				_tmp20 = FALSE;
			}
			_tmp18 = (_tmp20);
		}
		if (_tmp18) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean _tmp21;
				return (_tmp21 = TRUE, (expr_struct == NULL) ? NULL : (expr_struct = (vala_code_node_unref (expr_struct), NULL)), (expect_struct == NULL) ? NULL : (expect_struct = (vala_code_node_unref (expect_struct), NULL)), _tmp21);
			}
		}
		(expr_struct == NULL) ? NULL : (expr_struct = (vala_code_node_unref (expr_struct), NULL));
		(expect_struct == NULL) ? NULL : (expect_struct = (vala_code_node_unref (expect_struct), NULL));
	}
	_tmp22 = FALSE;
	_tmp23 = FALSE;
	if (self->priv->_data_type != NULL) {
		_tmp23 = target_type->priv->_data_type != NULL;
	} else {
		_tmp23 = FALSE;
	}
	if (_tmp23) {
		_tmp22 = vala_typesymbol_is_subtype_of (self->priv->_data_type, target_type->priv->_data_type);
	} else {
		_tmp22 = FALSE;
	}
	if (_tmp22) {
		return TRUE;
	}
	return FALSE;
}


gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type) {
	return VALA_DATA_TYPE_GET_CLASS (self)->compatible (self, target_type);
}


/**
 * Returns whether instances of this type are invokable.
 *
 * @return true if invokable, false otherwise
 */
static gboolean vala_data_type_real_is_invokable (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return FALSE;
}


gboolean vala_data_type_is_invokable (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_invokable (self);
}


/**
 * Returns the return type of this invokable.
 *
 * @return return type
 */
static ValaDataType* vala_data_type_real_get_return_type (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return NULL;
}


ValaDataType* vala_data_type_get_return_type (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_return_type (self);
}


/**
 * Returns copy of the list of invocation parameters.
 *
 * @return parameter list
 */
static GeeList* vala_data_type_real_get_parameters (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return NULL;
}


GeeList* vala_data_type_get_parameters (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_parameters (self);
}


static gboolean vala_data_type_real_is_reference_type_or_type_parameter (ValaDataType* self) {
	gboolean _tmp0;
	gboolean _tmp1;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->_data_type != NULL) {
		_tmp1 = vala_typesymbol_is_reference_type (self->priv->_data_type);
	} else {
		_tmp1 = FALSE;
	}
	if ((_tmp1)) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = self->priv->_type_parameter != NULL;
	}
	return _tmp0;
}


gboolean vala_data_type_is_reference_type_or_type_parameter (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_reference_type_or_type_parameter (self);
}


static gboolean vala_data_type_real_is_array (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return FALSE;
}


gboolean vala_data_type_is_array (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_array (self);
}


/**
 * Returns a list of symbols that define this type.
 *
 * @return symbol list
 */
static GeeList* vala_data_type_real_get_symbols (ValaDataType* self) {
	GeeArrayList* symbols;
	g_return_val_if_fail (self != NULL, NULL);
	symbols = gee_array_list_new (VALA_TYPE_SYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (self->priv->_data_type != NULL) {
		gee_collection_add ((GeeCollection*) symbols, (ValaSymbol*) self->priv->_data_type);
	}
	return (GeeList*) symbols;
}


GeeList* vala_data_type_get_symbols (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_symbols (self);
}


static ValaSymbol* vala_data_type_real_get_member (ValaDataType* self, const char* member_name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	if (self->priv->_data_type != NULL) {
		return vala_semantic_analyzer_symbol_lookup_inherited ((ValaSymbol*) self->priv->_data_type, member_name);
	}
	return NULL;
}


ValaSymbol* vala_data_type_get_member (ValaDataType* self, const char* member_name) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_member (self, member_name);
}


static ValaSymbol* vala_data_type_real_get_pointer_member (ValaDataType* self, const char* member_name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	return NULL;
}


ValaSymbol* vala_data_type_get_pointer_member (ValaDataType* self, const char* member_name) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_pointer_member (self, member_name);
}


/**
 * Checks whether this data type references a real struct. A real struct
 * is a struct which is not a simple (fundamental) type.
 */
static gboolean vala_data_type_real_is_real_struct_type (ValaDataType* self) {
	ValaStruct* _tmp1;
	ValaTypeSymbol* _tmp0;
	ValaStruct* s;
	gboolean _tmp2;
	gboolean _tmp4;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp1 = NULL;
	_tmp0 = NULL;
	s = (_tmp1 = (_tmp0 = self->priv->_data_type, VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp2 = FALSE;
	if (s != NULL) {
		_tmp2 = !vala_struct_is_simple_type (s);
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		gboolean _tmp3;
		return (_tmp3 = TRUE, (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), _tmp3);
	}
	return (_tmp4 = FALSE, (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), _tmp4);
}


gboolean vala_data_type_is_real_struct_type (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_real_struct_type (self);
}


static char* vala_data_type_real_get_type_id (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_data_type != NULL) {
		return vala_typesymbol_get_type_id (self->priv->_data_type);
	} else {
		return NULL;
	}
}


char* vala_data_type_get_type_id (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_type_id (self);
}


/**
 * Returns type signature as used for GVariant and D-Bus.
 */
static char* vala_data_type_real_get_type_signature (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_data_type != NULL) {
		char* sig;
		GeeList* type_args;
		char* _tmp2;
		sig = vala_typesymbol_get_type_signature (self->priv->_data_type);
		type_args = vala_data_type_get_type_arguments (self);
		if (gee_collection_get_size ((GeeCollection*) type_args) > 0) {
			char* element_sig;
			char* _tmp1;
			g_assert (strstr (sig, "%s") != NULL);
			element_sig = g_strdup ("");
			{
				GeeIterator* _type_arg_it;
				_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_args);
				while (gee_iterator_next (_type_arg_it)) {
					ValaDataType* type_arg;
					char* s;
					type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
					s = vala_data_type_get_type_signature (type_arg);
					if (s != NULL) {
						char* _tmp0;
						_tmp0 = NULL;
						element_sig = (_tmp0 = g_strconcat (element_sig, (s), NULL), element_sig = (g_free (element_sig), NULL), _tmp0);
					}
					(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
					s = (g_free (s), NULL);
				}
				(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
			}
			_tmp1 = NULL;
			sig = (_tmp1 = g_strdup_printf (sig, element_sig), sig = (g_free (sig), NULL), _tmp1);
			element_sig = (g_free (element_sig), NULL);
		}
		_tmp2 = NULL;
		return (_tmp2 = sig, (type_args == NULL) ? NULL : (type_args = (gee_collection_object_unref (type_args), NULL)), _tmp2);
	} else {
		return NULL;
	}
}


char* vala_data_type_get_type_signature (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_type_signature (self);
}


/**
 * Returns whether the value needs to be disposed, i.e. whether
 * allocated memory or other resources need to be released when
 * the value is no longer needed.
 */
static gboolean vala_data_type_real_is_disposable (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (!self->priv->_value_owned) {
		return FALSE;
	}
	if (vala_data_type_is_reference_type_or_type_parameter (self)) {
		return TRUE;
	}
	return FALSE;
}


gboolean vala_data_type_is_disposable (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_disposable (self);
}


ValaDataType* vala_data_type_get_actual_type (ValaDataType* self, ValaDataType* derived_instance_type, ValaCodeNode* node_reference) {
	ValaDataType* _tmp2;
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (node_reference != NULL, NULL);
	if (derived_instance_type == NULL) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
	}
	_tmp2 = NULL;
	result = (_tmp2 = self, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	if (VALA_IS_GENERIC_TYPE (result)) {
		ValaDataType* _tmp3;
		_tmp3 = NULL;
		result = (_tmp3 = vala_semantic_analyzer_get_actual_type (derived_instance_type, VALA_GENERIC_TYPE (result), node_reference), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp3);
	} else {
		if (result->priv->type_argument_list != NULL) {
			ValaDataType* _tmp4;
			/* recursely get actual types for type arguments*/
			_tmp4 = NULL;
			result = (_tmp4 = vala_data_type_copy (result), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp4);
			{
				gint i;
				i = 0;
				for (; i < gee_collection_get_size ((GeeCollection*) result->priv->type_argument_list); i++) {
					ValaDataType* _tmp6;
					ValaDataType* _tmp5;
					_tmp6 = NULL;
					_tmp5 = NULL;
					gee_list_set ((GeeList*) result->priv->type_argument_list, i, _tmp6 = vala_data_type_get_actual_type (_tmp5 = (ValaDataType*) gee_list_get ((GeeList*) result->priv->type_argument_list, i), derived_instance_type, node_reference));
					(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
					(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
				}
			}
		}
	}
	return result;
}


gboolean vala_data_type_get_value_owned (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_value_owned;
}


void vala_data_type_set_value_owned (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_value_owned = value;
}


gboolean vala_data_type_get_nullable (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_nullable;
}


void vala_data_type_set_nullable (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_nullable = value;
}


ValaTypeSymbol* vala_data_type_get_data_type (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_data_type_set_data_type (ValaDataType* self, ValaTypeSymbol* value) {
	g_return_if_fail (self != NULL);
	self->priv->_data_type = value;
}


ValaTypeParameter* vala_data_type_get_type_parameter (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_type_parameter;
}


void vala_data_type_set_type_parameter (ValaDataType* self, ValaTypeParameter* value) {
	ValaTypeParameter* _tmp2;
	ValaTypeParameter* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_type_parameter == NULL) ? NULL : (self->priv->_type_parameter = (vala_code_node_unref (self->priv->_type_parameter), NULL)), _tmp2);
}


gboolean vala_data_type_get_floating_reference (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_floating_reference;
}


void vala_data_type_set_floating_reference (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_floating_reference = value;
}


gboolean vala_data_type_get_is_dynamic (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_dynamic;
}


void vala_data_type_set_is_dynamic (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_dynamic = value;
}


static void vala_data_type_class_init (ValaDataTypeClass * klass) {
	vala_data_type_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_data_type_finalize;
	g_type_class_add_private (klass, sizeof (ValaDataTypePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_data_type_real_accept;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_data_type_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_data_type_real_get_lower_case_cname;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_data_type_real_to_string;
	VALA_DATA_TYPE_CLASS (klass)->to_qualified_string = vala_data_type_real_to_qualified_string;
	VALA_DATA_TYPE_CLASS (klass)->copy = vala_data_type_real_copy;
	VALA_DATA_TYPE_CLASS (klass)->equals = vala_data_type_real_equals;
	VALA_DATA_TYPE_CLASS (klass)->stricter = vala_data_type_real_stricter;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_data_type_real_replace_type;
	VALA_DATA_TYPE_CLASS (klass)->compatible = vala_data_type_real_compatible;
	VALA_DATA_TYPE_CLASS (klass)->is_invokable = vala_data_type_real_is_invokable;
	VALA_DATA_TYPE_CLASS (klass)->get_return_type = vala_data_type_real_get_return_type;
	VALA_DATA_TYPE_CLASS (klass)->get_parameters = vala_data_type_real_get_parameters;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type_or_type_parameter = vala_data_type_real_is_reference_type_or_type_parameter;
	VALA_DATA_TYPE_CLASS (klass)->is_array = vala_data_type_real_is_array;
	VALA_DATA_TYPE_CLASS (klass)->get_symbols = vala_data_type_real_get_symbols;
	VALA_DATA_TYPE_CLASS (klass)->get_member = vala_data_type_real_get_member;
	VALA_DATA_TYPE_CLASS (klass)->get_pointer_member = vala_data_type_real_get_pointer_member;
	VALA_DATA_TYPE_CLASS (klass)->is_real_struct_type = vala_data_type_real_is_real_struct_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_data_type_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_type_signature = vala_data_type_real_get_type_signature;
	VALA_DATA_TYPE_CLASS (klass)->is_disposable = vala_data_type_real_is_disposable;
}


static void vala_data_type_instance_init (ValaDataType * self) {
	self->priv = VALA_DATA_TYPE_GET_PRIVATE (self);
}


static void vala_data_type_finalize (ValaCodeNode* obj) {
	ValaDataType * self;
	self = VALA_DATA_TYPE (obj);
	(self->priv->_type_parameter == NULL) ? NULL : (self->priv->_type_parameter = (vala_code_node_unref (self->priv->_type_parameter), NULL));
	(self->priv->type_argument_list == NULL) ? NULL : (self->priv->type_argument_list = (gee_collection_object_unref (self->priv->type_argument_list), NULL));
	VALA_CODE_NODE_CLASS (vala_data_type_parent_class)->finalize (obj);
}


GType vala_data_type_get_type (void) {
	static GType vala_data_type_type_id = 0;
	if (vala_data_type_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDataTypeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_data_type_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDataType), 0, (GInstanceInitFunc) vala_data_type_instance_init, NULL };
		vala_data_type_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaDataType", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_data_type_type_id;
}




