/* valaflowanalyzer.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaflowanalyzer.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/map.h>
#include <gee/set.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <gee/collection.h>
#include <gee/hashmap.h>
#include <gee/hashset.h>
#include <vala/valabasicblock.h>
#include <vala/valaerrorcode.h>
#include <vala/valacatchclause.h>
#include <vala/valasymbol.h>
#include <vala/valareport.h>
#include <vala/valacodenode.h>
#include <vala/valasourcereference.h>
#include <vala/valavoidtype.h>
#include <vala/valalocalvariable.h>
#include <vala/valaphifunction.h>
#include <vala/valadatatype.h>
#include <vala/valaexpression.h>
#include <vala/valamethodcall.h>
#include <vala/valamemberaccess.h>
#include <vala/valabooleanliteral.h>
#include <vala/valaswitchsection.h>
#include <vala/valastatement.h>
#include <gobject/gvaluecollector.h>


#define VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET (vala_flow_analyzer_jump_target_get_type ())
#define VALA_FLOW_ANALYZER_JUMP_TARGET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, ValaFlowAnalyzerJumpTarget))
#define VALA_FLOW_ANALYZER_JUMP_TARGET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, ValaFlowAnalyzerJumpTargetClass))
#define VALA_FLOW_ANALYZER_IS_JUMP_TARGET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET))
#define VALA_FLOW_ANALYZER_IS_JUMP_TARGET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET))
#define VALA_FLOW_ANALYZER_JUMP_TARGET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, ValaFlowAnalyzerJumpTargetClass))

typedef struct _ValaFlowAnalyzerJumpTarget ValaFlowAnalyzerJumpTarget;
typedef struct _ValaFlowAnalyzerJumpTargetClass ValaFlowAnalyzerJumpTargetClass;
typedef struct _ValaFlowAnalyzerJumpTargetPrivate ValaFlowAnalyzerJumpTargetPrivate;
typedef struct _ValaFlowAnalyzerParamSpecJumpTarget ValaFlowAnalyzerParamSpecJumpTarget;

struct _ValaFlowAnalyzerJumpTarget {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaFlowAnalyzerJumpTargetPrivate * priv;
};

struct _ValaFlowAnalyzerJumpTargetClass {
	GTypeClass parent_class;
	void (*finalize) (ValaFlowAnalyzerJumpTarget *self);
};

struct _ValaFlowAnalyzerParamSpecJumpTarget {
	GParamSpec parent_instance;
};



struct _ValaFlowAnalyzerPrivate {
	ValaCodeContext* context;
	ValaBasicBlock* current_block;
	gboolean unreachable_reported;
	GeeList* jump_stack;
	GeeMap* var_map;
	GeeSet* used_vars;
	GeeMap* phi_functions;
};

#define VALA_FLOW_ANALYZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FLOW_ANALYZER, ValaFlowAnalyzerPrivate))
enum  {
	VALA_FLOW_ANALYZER_DUMMY_PROPERTY
};
static void vala_flow_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_flow_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_flow_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_flow_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_flow_analyzer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_flow_analyzer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed);
static void vala_flow_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_flow_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static GeeList* vala_flow_analyzer_get_depth_first_list (ValaFlowAnalyzer* self, ValaMethod* m);
static void vala_flow_analyzer_depth_first_traverse (ValaFlowAnalyzer* self, ValaBasicBlock* current, GeeList* list);
static void vala_flow_analyzer_build_dominator_tree (ValaFlowAnalyzer* self, ValaMethod* m);
static void vala_flow_analyzer_build_dominator_frontier (ValaFlowAnalyzer* self, ValaMethod* m);
static GeeMap* vala_flow_analyzer_get_assignment_map (ValaFlowAnalyzer* self, ValaMethod* m);
static void vala_flow_analyzer_insert_phi_functions (ValaFlowAnalyzer* self, ValaMethod* m);
static void vala_flow_analyzer_check_variables (ValaFlowAnalyzer* self, ValaMethod* m);
static void vala_flow_analyzer_check_block_variables (ValaFlowAnalyzer* self, ValaMethod* m, ValaBasicBlock* block);
static ValaLocalVariable* vala_flow_analyzer_process_assignment (ValaFlowAnalyzer* self, ValaMethod* m, GeeMap* var_map, ValaLocalVariable* var_symbol);
static void vala_flow_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_flow_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_flow_analyzer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_flow_analyzer_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt);
static void vala_flow_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static gboolean vala_flow_analyzer_always_true (ValaFlowAnalyzer* self, ValaExpression* condition);
static gboolean vala_flow_analyzer_always_false (ValaFlowAnalyzer* self, ValaExpression* condition);
static void vala_flow_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_flow_analyzer_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt);
static void vala_flow_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_flow_analyzer_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_flow_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_flow_analyzer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_flow_analyzer_real_visit_break_statement (ValaCodeVisitor* base, ValaBreakStatement* stmt);
static void vala_flow_analyzer_real_visit_continue_statement (ValaCodeVisitor* base, ValaContinueStatement* stmt);
static void vala_flow_analyzer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_flow_analyzer_handle_errors (ValaFlowAnalyzer* self, ValaCodeNode* node);
static void vala_flow_analyzer_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt);
static void vala_flow_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_flow_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_flow_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static gboolean vala_flow_analyzer_unreachable (ValaFlowAnalyzer* self, ValaCodeNode* node);
struct _ValaFlowAnalyzerJumpTargetPrivate {
	gboolean _is_break_target;
	gboolean _is_continue_target;
	gboolean _is_return_target;
	gboolean _is_error_target;
	ValaErrorDomain* _error_domain;
	ValaErrorCode* _error_code;
	gboolean _is_finally_clause;
	ValaBasicBlock* _basic_block;
	ValaBasicBlock* _last_block;
	ValaCatchClause* _catch_clause;
};

#define VALA_FLOW_ANALYZER_JUMP_TARGET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, ValaFlowAnalyzerJumpTargetPrivate))
enum  {
	VALA_FLOW_ANALYZER_JUMP_TARGET_DUMMY_PROPERTY
};
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_break_target (GType object_type, ValaBasicBlock* basic_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_break_target (ValaBasicBlock* basic_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_continue_target (GType object_type, ValaBasicBlock* basic_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_continue_target (ValaBasicBlock* basic_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_return_target (GType object_type, ValaBasicBlock* basic_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_return_target (ValaBasicBlock* basic_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_error_target (GType object_type, ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_error_target (ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_finally_clause (GType object_type, ValaBasicBlock* basic_block, ValaBasicBlock* last_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_finally_clause (ValaBasicBlock* basic_block, ValaBasicBlock* last_block);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct (GType object_type);
static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new (void);
static gboolean vala_flow_analyzer_jump_target_get_is_break_target (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_is_break_target (ValaFlowAnalyzerJumpTarget* self, gboolean value);
static gboolean vala_flow_analyzer_jump_target_get_is_continue_target (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_is_continue_target (ValaFlowAnalyzerJumpTarget* self, gboolean value);
static gboolean vala_flow_analyzer_jump_target_get_is_return_target (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_is_return_target (ValaFlowAnalyzerJumpTarget* self, gboolean value);
static gboolean vala_flow_analyzer_jump_target_get_is_error_target (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_is_error_target (ValaFlowAnalyzerJumpTarget* self, gboolean value);
static ValaErrorDomain* vala_flow_analyzer_jump_target_get_error_domain (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_error_domain (ValaFlowAnalyzerJumpTarget* self, ValaErrorDomain* value);
static ValaErrorCode* vala_flow_analyzer_jump_target_get_error_code (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_error_code (ValaFlowAnalyzerJumpTarget* self, ValaErrorCode* value);
static gboolean vala_flow_analyzer_jump_target_get_is_finally_clause (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_is_finally_clause (ValaFlowAnalyzerJumpTarget* self, gboolean value);
static ValaBasicBlock* vala_flow_analyzer_jump_target_get_basic_block (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_basic_block (ValaFlowAnalyzerJumpTarget* self, ValaBasicBlock* value);
static ValaBasicBlock* vala_flow_analyzer_jump_target_get_last_block (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_last_block (ValaFlowAnalyzerJumpTarget* self, ValaBasicBlock* value);
static ValaCatchClause* vala_flow_analyzer_jump_target_get_catch_clause (ValaFlowAnalyzerJumpTarget* self);
static void vala_flow_analyzer_jump_target_set_catch_clause (ValaFlowAnalyzerJumpTarget* self, ValaCatchClause* value);
static GParamSpec* vala_flow_analyzer_param_spec_jump_target (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
static gpointer vala_flow_analyzer_value_get_jump_target (const GValue* value);
static void vala_flow_analyzer_value_set_jump_target (GValue* value, gpointer v_object);
static gpointer vala_flow_analyzer_jump_target_parent_class = NULL;
static void vala_flow_analyzer_jump_target_finalize (ValaFlowAnalyzerJumpTarget* obj);
static GType vala_flow_analyzer_jump_target_get_type (void);
static gpointer vala_flow_analyzer_jump_target_ref (gpointer instance);
static void vala_flow_analyzer_jump_target_unref (gpointer instance);
static gpointer vala_flow_analyzer_parent_class = NULL;
static void vala_flow_analyzer_finalize (ValaCodeVisitor* obj);



ValaFlowAnalyzer* vala_flow_analyzer_construct (GType object_type) {
	ValaFlowAnalyzer* self;
	self = (ValaFlowAnalyzer*) g_type_create_instance (object_type);
	return self;
}


ValaFlowAnalyzer* vala_flow_analyzer_new (void) {
	return vala_flow_analyzer_construct (VALA_TYPE_FLOW_ANALYZER);
}


/**
 * Build control flow graph in the specified context.
 *
 * @param context a code context
 */
void vala_flow_analyzer_analyze (ValaFlowAnalyzer* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	GeeList* source_files;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL) ? NULL : vala_code_context_ref (_tmp0)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1);
	/* we're only interested in non-pkg source files */
	source_files = vala_code_context_get_source_files (context);
	{
		GeeIterator* file_it;
		file_it = gee_iterable_iterator ((GeeIterable*) source_files);
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = (ValaSourceFile*) gee_iterator_get (file_it);
			if (!vala_source_file_get_external_package (file)) {
				vala_source_file_accept (file, (ValaCodeVisitor*) self);
			}
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(file_it == NULL) ? NULL : (file_it = (gee_collection_object_unref (file_it), NULL));
	}
	(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
}


static void vala_flow_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (source_file != NULL);
	vala_source_file_accept_children (source_file, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (cl != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) cl, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (st != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) st, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (iface != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) iface, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (en != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) en, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (ed != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) ed, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaFlowAnalyzer * self;
	gboolean _tmp0;
	gboolean _tmp1;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (f != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (vala_symbol_get_access ((ValaSymbol*) f) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
		_tmp1 = vala_symbol_get_access ((ValaSymbol*) f) != VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		_tmp0 = !vala_symbol_get_used ((ValaSymbol*) f);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		char* _tmp3;
		char* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) f), _tmp3 = g_strdup_printf ("field `%s' never used", _tmp2 = vala_symbol_get_full_name ((ValaSymbol*) f)));
		_tmp3 = (g_free (_tmp3), NULL);
		_tmp2 = (g_free (_tmp2), NULL);
	}
}


static void vala_flow_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaFlowAnalyzer * self;
	gboolean _tmp0;
	gboolean _tmp1;
	gboolean _tmp2;
	ValaBasicBlock* _tmp5;
	ValaBasicBlock* _tmp6;
	ValaBasicBlock* _tmp7;
	ValaFlowAnalyzerJumpTarget* _tmp8;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (m != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	if (vala_symbol_get_access ((ValaSymbol*) m) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
		_tmp2 = vala_symbol_get_access ((ValaSymbol*) m) != VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		_tmp1 = !vala_symbol_get_used ((ValaSymbol*) m);
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		_tmp0 = !vala_method_get_entry_point (m);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		char* _tmp4;
		char* _tmp3;
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) m), _tmp4 = g_strdup_printf ("method `%s' never used", _tmp3 = vala_symbol_get_full_name ((ValaSymbol*) m)));
		_tmp4 = (g_free (_tmp4), NULL);
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_method_get_body (m) == NULL) {
		return;
	}
	_tmp5 = NULL;
	vala_method_set_entry_block (m, _tmp5 = vala_basic_block_new_entry ());
	(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_basic_block_unref (_tmp5), NULL));
	_tmp6 = NULL;
	vala_method_set_exit_block (m, _tmp6 = vala_basic_block_new_exit ());
	(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_basic_block_unref (_tmp6), NULL));
	_tmp7 = NULL;
	self->priv->current_block = (_tmp7 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp7);
	vala_basic_block_connect (vala_method_get_entry_block (m), self->priv->current_block);
	_tmp8 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp8 = vala_flow_analyzer_jump_target_new_return_target (vala_method_get_exit_block (m)));
	(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_flow_analyzer_jump_target_unref (_tmp8), NULL));
	vala_code_node_accept_children ((ValaCodeNode*) m, (ValaCodeVisitor*) self);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	if (self->priv->current_block != NULL) {
		/* end of method body reachable*/
		if (!(VALA_IS_VOID_TYPE (vala_method_get_return_type (m)))) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "missing return statement at end of method body");
			vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		}
		vala_basic_block_connect (self->priv->current_block, vala_method_get_exit_block (m));
	}
	vala_flow_analyzer_build_dominator_tree (self, m);
	vala_flow_analyzer_build_dominator_frontier (self, m);
	vala_flow_analyzer_insert_phi_functions (self, m);
	vala_flow_analyzer_check_variables (self, m);
}


static GeeList* vala_flow_analyzer_get_depth_first_list (ValaFlowAnalyzer* self, ValaMethod* m) {
	GeeArrayList* list;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (m != NULL, NULL);
	list = gee_array_list_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, g_direct_equal);
	vala_flow_analyzer_depth_first_traverse (self, vala_method_get_entry_block (m), (GeeList*) list);
	return (GeeList*) list;
}


static void vala_flow_analyzer_depth_first_traverse (ValaFlowAnalyzer* self, ValaBasicBlock* current, GeeList* list) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (current != NULL);
	g_return_if_fail (list != NULL);
	if (gee_collection_contains ((GeeCollection*) list, current)) {
		return;
	}
	gee_collection_add ((GeeCollection*) list, current);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* succ_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		succ_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_basic_block_get_successors (current))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (succ_it)) {
			ValaBasicBlock* _tmp2;
			ValaBasicBlock* succ;
			_tmp2 = NULL;
			succ = (_tmp2 = (ValaBasicBlock*) gee_iterator_get (succ_it), (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2));
			vala_flow_analyzer_depth_first_traverse (self, succ, list);
			(succ == NULL) ? NULL : (succ = (vala_basic_block_unref (succ), NULL));
		}
		(succ_it == NULL) ? NULL : (succ_it = (gee_collection_object_unref (succ_it), NULL));
	}
}


static void vala_flow_analyzer_build_dominator_tree (ValaFlowAnalyzer* self, ValaMethod* m) {
	GeeHashMap* dom;
	GeeList* block_list;
	GeeHashSet* entry_dom_set;
	gboolean changes;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	/* set dom(n) = {E,1,2...,N,X} forall n*/
	dom = gee_hash_map_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, GEE_TYPE_SET, (GBoxedCopyFunc) gee_collection_object_ref, gee_collection_object_unref, g_direct_hash, g_direct_equal, g_direct_equal);
	block_list = vala_flow_analyzer_get_depth_first_list (self, m);
	{
		GeeIterator* block_it;
		block_it = gee_iterable_iterator ((GeeIterable*) block_list);
		while (gee_iterator_next (block_it)) {
			ValaBasicBlock* block;
			GeeHashSet* block_set;
			block = (ValaBasicBlock*) gee_iterator_get (block_it);
			block_set = gee_hash_set_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, g_direct_hash, g_direct_equal);
			{
				GeeIterator* b_it;
				b_it = gee_iterable_iterator ((GeeIterable*) block_list);
				while (gee_iterator_next (b_it)) {
					ValaBasicBlock* b;
					b = (ValaBasicBlock*) gee_iterator_get (b_it);
					gee_collection_add ((GeeCollection*) block_set, b);
					(b == NULL) ? NULL : (b = (vala_basic_block_unref (b), NULL));
				}
				(b_it == NULL) ? NULL : (b_it = (gee_collection_object_unref (b_it), NULL));
			}
			gee_map_set ((GeeMap*) dom, block, (GeeSet*) block_set);
			(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
			(block_set == NULL) ? NULL : (block_set = (gee_collection_object_unref (block_set), NULL));
		}
		(block_it == NULL) ? NULL : (block_it = (gee_collection_object_unref (block_it), NULL));
	}
	/* set dom(E) = {E}*/
	entry_dom_set = gee_hash_set_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, g_direct_hash, g_direct_equal);
	gee_collection_add ((GeeCollection*) entry_dom_set, vala_method_get_entry_block (m));
	gee_map_set ((GeeMap*) dom, vala_method_get_entry_block (m), (GeeSet*) entry_dom_set);
	changes = TRUE;
	while (changes) {
		changes = FALSE;
		{
			GeeIterator* block_it;
			block_it = gee_iterable_iterator ((GeeIterable*) block_list);
			while (gee_iterator_next (block_it)) {
				ValaBasicBlock* block;
				GeeHashSet* dom_set;
				gboolean first;
				GeeSet* _tmp3;
				gboolean _tmp4;
				block = (ValaBasicBlock*) gee_iterator_get (block_it);
				/* intersect dom(pred) forall pred: pred = predecessor(s)*/
				dom_set = gee_hash_set_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, g_direct_hash, g_direct_equal);
				first = TRUE;
				{
					GeeList* _tmp0;
					GeeIterator* _tmp1;
					GeeIterator* pred_it;
					_tmp0 = NULL;
					_tmp1 = NULL;
					pred_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_basic_block_get_predecessors (block))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
					while (gee_iterator_next (pred_it)) {
						ValaBasicBlock* _tmp2;
						ValaBasicBlock* pred;
						GeeSet* pred_dom_set;
						_tmp2 = NULL;
						pred = (_tmp2 = (ValaBasicBlock*) gee_iterator_get (pred_it), (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2));
						pred_dom_set = (GeeSet*) gee_map_get ((GeeMap*) dom, pred);
						if (first) {
							{
								GeeIterator* dom_block_it;
								dom_block_it = gee_iterable_iterator ((GeeIterable*) pred_dom_set);
								while (gee_iterator_next (dom_block_it)) {
									ValaBasicBlock* dom_block;
									dom_block = (ValaBasicBlock*) gee_iterator_get (dom_block_it);
									gee_collection_add ((GeeCollection*) dom_set, dom_block);
									(dom_block == NULL) ? NULL : (dom_block = (vala_basic_block_unref (dom_block), NULL));
								}
								(dom_block_it == NULL) ? NULL : (dom_block_it = (gee_collection_object_unref (dom_block_it), NULL));
							}
							first = FALSE;
						} else {
							GeeArrayList* remove_queue;
							remove_queue = gee_array_list_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, g_direct_equal);
							{
								GeeIterator* dom_block_it;
								dom_block_it = gee_iterable_iterator ((GeeIterable*) dom_set);
								while (gee_iterator_next (dom_block_it)) {
									ValaBasicBlock* dom_block;
									dom_block = (ValaBasicBlock*) gee_iterator_get (dom_block_it);
									if (!(gee_collection_contains ((GeeCollection*) pred_dom_set, dom_block))) {
										gee_collection_add ((GeeCollection*) remove_queue, dom_block);
									}
									(dom_block == NULL) ? NULL : (dom_block = (vala_basic_block_unref (dom_block), NULL));
								}
								(dom_block_it == NULL) ? NULL : (dom_block_it = (gee_collection_object_unref (dom_block_it), NULL));
							}
							{
								GeeIterator* dom_block_it;
								dom_block_it = gee_iterable_iterator ((GeeIterable*) remove_queue);
								while (gee_iterator_next (dom_block_it)) {
									ValaBasicBlock* dom_block;
									dom_block = (ValaBasicBlock*) gee_iterator_get (dom_block_it);
									gee_collection_remove ((GeeCollection*) dom_set, dom_block);
									(dom_block == NULL) ? NULL : (dom_block = (vala_basic_block_unref (dom_block), NULL));
								}
								(dom_block_it == NULL) ? NULL : (dom_block_it = (gee_collection_object_unref (dom_block_it), NULL));
							}
							(remove_queue == NULL) ? NULL : (remove_queue = (gee_collection_object_unref (remove_queue), NULL));
						}
						(pred == NULL) ? NULL : (pred = (vala_basic_block_unref (pred), NULL));
						(pred_dom_set == NULL) ? NULL : (pred_dom_set = (gee_collection_object_unref (pred_dom_set), NULL));
					}
					(pred_it == NULL) ? NULL : (pred_it = (gee_collection_object_unref (pred_it), NULL));
				}
				/* unite with s*/
				gee_collection_add ((GeeCollection*) dom_set, block);
				/* check for changes*/
				_tmp3 = NULL;
				if ((_tmp4 = gee_collection_get_size ((GeeCollection*) (_tmp3 = (GeeSet*) gee_map_get ((GeeMap*) dom, block))) != gee_collection_get_size ((GeeCollection*) dom_set), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4)) {
					changes = TRUE;
				} else {
					{
						GeeSet* _tmp5;
						GeeIterator* _tmp6;
						GeeIterator* dom_block_it;
						_tmp5 = NULL;
						_tmp6 = NULL;
						dom_block_it = (_tmp6 = gee_iterable_iterator ((GeeIterable*) (_tmp5 = (GeeSet*) gee_map_get ((GeeMap*) dom, block))), (_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL)), _tmp6);
						while (gee_iterator_next (dom_block_it)) {
							ValaBasicBlock* dom_block;
							dom_block = (ValaBasicBlock*) gee_iterator_get (dom_block_it);
							if (!(gee_collection_contains ((GeeCollection*) dom_set, dom_block))) {
								changes = TRUE;
							}
							(dom_block == NULL) ? NULL : (dom_block = (vala_basic_block_unref (dom_block), NULL));
						}
						(dom_block_it == NULL) ? NULL : (dom_block_it = (gee_collection_object_unref (dom_block_it), NULL));
					}
				}
				/* update set in map*/
				gee_map_set ((GeeMap*) dom, block, (GeeSet*) dom_set);
				(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
				(dom_set == NULL) ? NULL : (dom_set = (gee_collection_object_unref (dom_set), NULL));
			}
			(block_it == NULL) ? NULL : (block_it = (gee_collection_object_unref (block_it), NULL));
		}
	}
	/* build tree*/
	{
		GeeIterator* block_it;
		/* build tree*/
		block_it = gee_iterable_iterator ((GeeIterable*) block_list);
		/* build tree*/
		while (gee_iterator_next (block_it)) {
			ValaBasicBlock* block;
			ValaBasicBlock* immediate_dominator;
			/* build tree*/
			block = (ValaBasicBlock*) gee_iterator_get (block_it);
			if (block == vala_method_get_entry_block (m)) {
				(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
				continue;
			}
			immediate_dominator = NULL;
			{
				GeeSet* _tmp7;
				GeeIterator* _tmp8;
				GeeIterator* dominator_it;
				_tmp7 = NULL;
				_tmp8 = NULL;
				dominator_it = (_tmp8 = gee_iterable_iterator ((GeeIterable*) (_tmp7 = (GeeSet*) gee_map_get ((GeeMap*) dom, block))), (_tmp7 == NULL) ? NULL : (_tmp7 = (gee_collection_object_unref (_tmp7), NULL)), _tmp8);
				while (gee_iterator_next (dominator_it)) {
					ValaBasicBlock* dominator;
					dominator = (ValaBasicBlock*) gee_iterator_get (dominator_it);
					if (dominator == block) {
						(dominator == NULL) ? NULL : (dominator = (vala_basic_block_unref (dominator), NULL));
						continue;
					}
					if (immediate_dominator == NULL) {
						ValaBasicBlock* _tmp10;
						ValaBasicBlock* _tmp9;
						_tmp10 = NULL;
						_tmp9 = NULL;
						immediate_dominator = (_tmp10 = (_tmp9 = dominator, (_tmp9 == NULL) ? NULL : vala_basic_block_ref (_tmp9)), (immediate_dominator == NULL) ? NULL : (immediate_dominator = (vala_basic_block_unref (immediate_dominator), NULL)), _tmp10);
					} else {
						GeeSet* _tmp11;
						gboolean _tmp12;
						/* if immediate_dominator dominates dominator,
						 update immediate_dominator*/
						_tmp11 = NULL;
						if ((_tmp12 = gee_collection_contains ((GeeCollection*) (_tmp11 = (GeeSet*) gee_map_get ((GeeMap*) dom, dominator)), immediate_dominator), (_tmp11 == NULL) ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL)), _tmp12)) {
							ValaBasicBlock* _tmp14;
							ValaBasicBlock* _tmp13;
							_tmp14 = NULL;
							_tmp13 = NULL;
							immediate_dominator = (_tmp14 = (_tmp13 = dominator, (_tmp13 == NULL) ? NULL : vala_basic_block_ref (_tmp13)), (immediate_dominator == NULL) ? NULL : (immediate_dominator = (vala_basic_block_unref (immediate_dominator), NULL)), _tmp14);
						}
					}
					(dominator == NULL) ? NULL : (dominator = (vala_basic_block_unref (dominator), NULL));
				}
				(dominator_it == NULL) ? NULL : (dominator_it = (gee_collection_object_unref (dominator_it), NULL));
			}
			vala_basic_block_add_child (immediate_dominator, block);
			(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
			(immediate_dominator == NULL) ? NULL : (immediate_dominator = (vala_basic_block_unref (immediate_dominator), NULL));
		}
		(block_it == NULL) ? NULL : (block_it = (gee_collection_object_unref (block_it), NULL));
	}
	(dom == NULL) ? NULL : (dom = (gee_collection_object_unref (dom), NULL));
	(block_list == NULL) ? NULL : (block_list = (gee_collection_object_unref (block_list), NULL));
	(entry_dom_set == NULL) ? NULL : (entry_dom_set = (gee_collection_object_unref (entry_dom_set), NULL));
}


static void vala_flow_analyzer_build_dominator_frontier (ValaFlowAnalyzer* self, ValaMethod* m) {
	GeeList* block_list;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	block_list = vala_flow_analyzer_get_depth_first_list (self, m);
	{
		gint i;
		i = gee_collection_get_size ((GeeCollection*) block_list) - 1;
		for (; i >= 0; i--) {
			ValaBasicBlock* block;
			block = (ValaBasicBlock*) gee_list_get ((GeeList*) block_list, i);
			{
				GeeList* _tmp0;
				GeeIterator* _tmp1;
				GeeIterator* succ_it;
				_tmp0 = NULL;
				_tmp1 = NULL;
				succ_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_basic_block_get_successors (block))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
				while (gee_iterator_next (succ_it)) {
					ValaBasicBlock* _tmp2;
					ValaBasicBlock* succ;
					_tmp2 = NULL;
					succ = (_tmp2 = (ValaBasicBlock*) gee_iterator_get (succ_it), (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2));
					/* if idom(succ) != block*/
					if (vala_basic_block_get_parent (succ) != block) {
						vala_basic_block_add_dominator_frontier (block, succ);
					}
					(succ == NULL) ? NULL : (succ = (vala_basic_block_unref (succ), NULL));
				}
				(succ_it == NULL) ? NULL : (succ_it = (gee_collection_object_unref (succ_it), NULL));
			}
			{
				GeeList* _tmp3;
				GeeIterator* _tmp4;
				GeeIterator* child_it;
				_tmp3 = NULL;
				_tmp4 = NULL;
				child_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = vala_basic_block_get_children (block))), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4);
				while (gee_iterator_next (child_it)) {
					ValaBasicBlock* child;
					child = (ValaBasicBlock*) gee_iterator_get (child_it);
					{
						GeeSet* _tmp5;
						GeeIterator* _tmp6;
						GeeIterator* child_frontier_it;
						_tmp5 = NULL;
						_tmp6 = NULL;
						child_frontier_it = (_tmp6 = gee_iterable_iterator ((GeeIterable*) (_tmp5 = vala_basic_block_get_dominator_frontier (child))), (_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL)), _tmp6);
						while (gee_iterator_next (child_frontier_it)) {
							ValaBasicBlock* child_frontier;
							child_frontier = (ValaBasicBlock*) gee_iterator_get (child_frontier_it);
							/* if idom(child_frontier) != block*/
							if (vala_basic_block_get_parent (child_frontier) != block) {
								vala_basic_block_add_dominator_frontier (block, child_frontier);
							}
							(child_frontier == NULL) ? NULL : (child_frontier = (vala_basic_block_unref (child_frontier), NULL));
						}
						(child_frontier_it == NULL) ? NULL : (child_frontier_it = (gee_collection_object_unref (child_frontier_it), NULL));
					}
					(child == NULL) ? NULL : (child = (vala_basic_block_unref (child), NULL));
				}
				(child_it == NULL) ? NULL : (child_it = (gee_collection_object_unref (child_it), NULL));
			}
			(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
		}
	}
	(block_list == NULL) ? NULL : (block_list = (gee_collection_object_unref (block_list), NULL));
}


static GeeMap* vala_flow_analyzer_get_assignment_map (ValaFlowAnalyzer* self, ValaMethod* m) {
	GeeHashMap* map;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (m != NULL, NULL);
	map = gee_hash_map_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, GEE_TYPE_SET, (GBoxedCopyFunc) gee_collection_object_ref, gee_collection_object_unref, g_direct_hash, g_direct_equal, g_direct_equal);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* block_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		block_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_flow_analyzer_get_depth_first_list (self, m))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (block_it)) {
			ValaBasicBlock* block;
			GeeArrayList* defined_variables;
			block = (ValaBasicBlock*) gee_iterator_get (block_it);
			defined_variables = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
			{
				GeeList* _tmp2;
				GeeIterator* _tmp3;
				GeeIterator* node_it;
				_tmp2 = NULL;
				_tmp3 = NULL;
				node_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = vala_basic_block_get_nodes (block))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
				while (gee_iterator_next (node_it)) {
					ValaCodeNode* node;
					node = (ValaCodeNode*) gee_iterator_get (node_it);
					vala_code_node_get_defined_variables (node, (GeeCollection*) defined_variables);
					(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
				}
				(node_it == NULL) ? NULL : (node_it = (gee_collection_object_unref (node_it), NULL));
			}
			{
				GeeIterator* local_it;
				local_it = gee_iterable_iterator ((GeeIterable*) defined_variables);
				while (gee_iterator_next (local_it)) {
					ValaLocalVariable* local;
					GeeSet* block_set;
					local = (ValaLocalVariable*) gee_iterator_get (local_it);
					block_set = (GeeSet*) gee_map_get ((GeeMap*) map, local);
					if (block_set == NULL) {
						GeeSet* _tmp4;
						_tmp4 = NULL;
						block_set = (_tmp4 = (GeeSet*) gee_hash_set_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, g_direct_hash, g_direct_equal), (block_set == NULL) ? NULL : (block_set = (gee_collection_object_unref (block_set), NULL)), _tmp4);
						gee_map_set ((GeeMap*) map, local, block_set);
					}
					gee_collection_add ((GeeCollection*) block_set, block);
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
					(block_set == NULL) ? NULL : (block_set = (gee_collection_object_unref (block_set), NULL));
				}
				(local_it == NULL) ? NULL : (local_it = (gee_collection_object_unref (local_it), NULL));
			}
			(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
			(defined_variables == NULL) ? NULL : (defined_variables = (gee_collection_object_unref (defined_variables), NULL));
		}
		(block_it == NULL) ? NULL : (block_it = (gee_collection_object_unref (block_it), NULL));
	}
	return (GeeMap*) map;
}


static void vala_flow_analyzer_insert_phi_functions (ValaFlowAnalyzer* self, ValaMethod* m) {
	GeeMap* assign;
	gint counter;
	GeeArrayList* work_list;
	GeeHashMap* added;
	GeeHashMap* phi;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	assign = vala_flow_analyzer_get_assignment_map (self, m);
	counter = 0;
	work_list = gee_array_list_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, g_direct_equal);
	added = gee_hash_map_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal, g_direct_equal);
	phi = gee_hash_map_new (VALA_TYPE_BASIC_BLOCK, (GBoxedCopyFunc) vala_basic_block_ref, vala_basic_block_unref, G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal, g_direct_equal);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* block_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		block_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_flow_analyzer_get_depth_first_list (self, m))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (block_it)) {
			ValaBasicBlock* block;
			block = (ValaBasicBlock*) gee_iterator_get (block_it);
			gee_map_set ((GeeMap*) added, block, GINT_TO_POINTER (0));
			gee_map_set ((GeeMap*) phi, block, GINT_TO_POINTER (0));
			(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
		}
		(block_it == NULL) ? NULL : (block_it = (gee_collection_object_unref (block_it), NULL));
	}
	{
		GeeSet* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* local_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		local_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = gee_map_get_keys (assign))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (local_it)) {
			ValaLocalVariable* local;
			local = (ValaLocalVariable*) gee_iterator_get (local_it);
			counter++;
			{
				GeeSet* _tmp4;
				GeeIterator* _tmp5;
				GeeIterator* block_it;
				_tmp4 = NULL;
				_tmp5 = NULL;
				block_it = (_tmp5 = gee_iterable_iterator ((GeeIterable*) (_tmp4 = (GeeSet*) gee_map_get (assign, local))), (_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL)), _tmp5);
				while (gee_iterator_next (block_it)) {
					ValaBasicBlock* block;
					block = (ValaBasicBlock*) gee_iterator_get (block_it);
					gee_collection_add ((GeeCollection*) work_list, block);
					gee_map_set ((GeeMap*) added, block, GINT_TO_POINTER (counter));
					(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
				}
				(block_it == NULL) ? NULL : (block_it = (gee_collection_object_unref (block_it), NULL));
			}
			while (gee_collection_get_size ((GeeCollection*) work_list) > 0) {
				ValaBasicBlock* block;
				block = (ValaBasicBlock*) gee_list_get ((GeeList*) work_list, 0);
				gee_list_remove_at ((GeeList*) work_list, 0);
				{
					GeeSet* _tmp6;
					GeeIterator* _tmp7;
					GeeIterator* frontier_it;
					_tmp6 = NULL;
					_tmp7 = NULL;
					frontier_it = (_tmp7 = gee_iterable_iterator ((GeeIterable*) (_tmp6 = vala_basic_block_get_dominator_frontier (block))), (_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL)), _tmp7);
					while (gee_iterator_next (frontier_it)) {
						ValaBasicBlock* frontier;
						gint blockPhi;
						frontier = (ValaBasicBlock*) gee_iterator_get (frontier_it);
						blockPhi = GPOINTER_TO_INT (gee_map_get ((GeeMap*) phi, frontier));
						if (blockPhi < counter) {
							ValaPhiFunction* _tmp9;
							GeeList* _tmp8;
							gint block_added;
							_tmp9 = NULL;
							_tmp8 = NULL;
							vala_basic_block_add_phi_function (frontier, _tmp9 = vala_phi_function_new (local, gee_collection_get_size ((GeeCollection*) (_tmp8 = vala_basic_block_get_predecessors (frontier)))));
							(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_phi_function_unref (_tmp9), NULL));
							(_tmp8 == NULL) ? NULL : (_tmp8 = (gee_collection_object_unref (_tmp8), NULL));
							gee_map_set ((GeeMap*) phi, frontier, GINT_TO_POINTER (counter));
							block_added = GPOINTER_TO_INT (gee_map_get ((GeeMap*) added, frontier));
							if (block_added < counter) {
								gee_map_set ((GeeMap*) added, frontier, GINT_TO_POINTER (counter));
								gee_collection_add ((GeeCollection*) work_list, frontier);
							}
						}
						(frontier == NULL) ? NULL : (frontier = (vala_basic_block_unref (frontier), NULL));
					}
					(frontier_it == NULL) ? NULL : (frontier_it = (gee_collection_object_unref (frontier_it), NULL));
				}
				(block == NULL) ? NULL : (block = (vala_basic_block_unref (block), NULL));
			}
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(local_it == NULL) ? NULL : (local_it = (gee_collection_object_unref (local_it), NULL));
	}
	(assign == NULL) ? NULL : (assign = (gee_collection_object_unref (assign), NULL));
	(work_list == NULL) ? NULL : (work_list = (gee_collection_object_unref (work_list), NULL));
	(added == NULL) ? NULL : (added = (gee_collection_object_unref (added), NULL));
	(phi == NULL) ? NULL : (phi = (gee_collection_object_unref (phi), NULL));
}


static void vala_flow_analyzer_check_variables (ValaFlowAnalyzer* self, ValaMethod* m) {
	GeeMap* _tmp0;
	GeeSet* _tmp1;
	GeeMap* _tmp2;
	GeeArrayList* used_vars_queue;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	_tmp0 = NULL;
	self->priv->var_map = (_tmp0 = (GeeMap*) gee_hash_map_new (VALA_TYPE_SYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, GEE_TYPE_LIST, (GBoxedCopyFunc) gee_collection_object_ref, gee_collection_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->var_map == NULL) ? NULL : (self->priv->var_map = (gee_collection_object_unref (self->priv->var_map), NULL)), _tmp0);
	_tmp1 = NULL;
	self->priv->used_vars = (_tmp1 = (GeeSet*) gee_hash_set_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_hash, g_direct_equal), (self->priv->used_vars == NULL) ? NULL : (self->priv->used_vars = (gee_collection_object_unref (self->priv->used_vars), NULL)), _tmp1);
	_tmp2 = NULL;
	self->priv->phi_functions = (_tmp2 = (GeeMap*) gee_hash_map_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, VALA_TYPE_PHI_FUNCTION, (GBoxedCopyFunc) vala_phi_function_ref, vala_phi_function_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->phi_functions == NULL) ? NULL : (self->priv->phi_functions = (gee_collection_object_unref (self->priv->phi_functions), NULL)), _tmp2);
	vala_flow_analyzer_check_block_variables (self, m, vala_method_get_entry_block (m));
	/* check for variables used before initialization*/
	used_vars_queue = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	{
		GeeIterator* local_it;
		local_it = gee_iterable_iterator ((GeeIterable*) self->priv->used_vars);
		while (gee_iterator_next (local_it)) {
			ValaLocalVariable* local;
			local = (ValaLocalVariable*) gee_iterator_get (local_it);
			gee_collection_add ((GeeCollection*) used_vars_queue, local);
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(local_it == NULL) ? NULL : (local_it = (gee_collection_object_unref (local_it), NULL));
	}
	while (gee_collection_get_size ((GeeCollection*) used_vars_queue) > 0) {
		ValaLocalVariable* used_var;
		ValaPhiFunction* phi;
		used_var = (ValaLocalVariable*) gee_list_get ((GeeList*) used_vars_queue, 0);
		gee_list_remove_at ((GeeList*) used_vars_queue, 0);
		phi = (ValaPhiFunction*) gee_map_get (self->priv->phi_functions, used_var);
		if (phi != NULL) {
			{
				GeeIterator* local_it;
				local_it = gee_iterable_iterator ((GeeIterable*) vala_phi_function_get_operands (phi));
				while (gee_iterator_next (local_it)) {
					ValaLocalVariable* local;
					local = (ValaLocalVariable*) gee_iterator_get (local_it);
					if (local == NULL) {
						char* _tmp3;
						_tmp3 = NULL;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) used_var), _tmp3 = g_strdup_printf ("use of possibly unassigned local variable `%s'", vala_symbol_get_name ((ValaSymbol*) used_var)));
						_tmp3 = (g_free (_tmp3), NULL);
						(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
						continue;
					}
					if (!(gee_collection_contains ((GeeCollection*) self->priv->used_vars, local))) {
						vala_code_node_set_source_reference ((ValaCodeNode*) local, vala_code_node_get_source_reference ((ValaCodeNode*) used_var));
						gee_collection_add ((GeeCollection*) self->priv->used_vars, local);
						gee_collection_add ((GeeCollection*) used_vars_queue, local);
					}
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				}
				(local_it == NULL) ? NULL : (local_it = (gee_collection_object_unref (local_it), NULL));
			}
		}
		(used_var == NULL) ? NULL : (used_var = (vala_code_node_unref (used_var), NULL));
		(phi == NULL) ? NULL : (phi = (vala_phi_function_unref (phi), NULL));
	}
	(used_vars_queue == NULL) ? NULL : (used_vars_queue = (gee_collection_object_unref (used_vars_queue), NULL));
}


static void vala_flow_analyzer_check_block_variables (ValaFlowAnalyzer* self, ValaMethod* m, ValaBasicBlock* block) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (block != NULL);
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* phi_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		phi_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_basic_block_get_phi_functions (block))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (phi_it)) {
			ValaPhiFunction* phi;
			ValaLocalVariable* versioned_var;
			phi = (ValaPhiFunction*) gee_iterator_get (phi_it);
			versioned_var = vala_flow_analyzer_process_assignment (self, m, self->priv->var_map, vala_phi_function_get_original_variable (phi));
			gee_map_set (self->priv->phi_functions, versioned_var, phi);
			(phi == NULL) ? NULL : (phi = (vala_phi_function_unref (phi), NULL));
			(versioned_var == NULL) ? NULL : (versioned_var = (vala_code_node_unref (versioned_var), NULL));
		}
		(phi_it == NULL) ? NULL : (phi_it = (gee_collection_object_unref (phi_it), NULL));
	}
	{
		GeeList* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* node_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		node_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = vala_basic_block_get_nodes (block))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (node_it)) {
			ValaCodeNode* node;
			GeeArrayList* used_variables;
			GeeArrayList* defined_variables;
			node = (ValaCodeNode*) gee_iterator_get (node_it);
			used_variables = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
			vala_code_node_get_used_variables (node, (GeeCollection*) used_variables);
			{
				GeeIterator* var_symbol_it;
				var_symbol_it = gee_iterable_iterator ((GeeIterable*) used_variables);
				while (gee_iterator_next (var_symbol_it)) {
					ValaLocalVariable* var_symbol;
					GeeList* variable_stack;
					gboolean _tmp4;
					ValaLocalVariable* versioned_local;
					var_symbol = (ValaLocalVariable*) gee_iterator_get (var_symbol_it);
					variable_stack = (GeeList*) gee_map_get (self->priv->var_map, (ValaSymbol*) var_symbol);
					_tmp4 = FALSE;
					if (variable_stack == NULL) {
						_tmp4 = TRUE;
					} else {
						_tmp4 = gee_collection_get_size ((GeeCollection*) variable_stack) == 0;
					}
					if (_tmp4) {
						char* _tmp5;
						_tmp5 = NULL;
						vala_report_error (vala_code_node_get_source_reference (node), _tmp5 = g_strdup_printf ("use of possibly unassigned local variable `%s'", vala_symbol_get_name ((ValaSymbol*) var_symbol)));
						_tmp5 = (g_free (_tmp5), NULL);
						(var_symbol == NULL) ? NULL : (var_symbol = (vala_code_node_unref (var_symbol), NULL));
						(variable_stack == NULL) ? NULL : (variable_stack = (gee_collection_object_unref (variable_stack), NULL));
						continue;
					}
					versioned_local = (ValaLocalVariable*) gee_list_get (variable_stack, gee_collection_get_size ((GeeCollection*) variable_stack) - 1);
					if (!(gee_collection_contains ((GeeCollection*) self->priv->used_vars, versioned_local))) {
						vala_code_node_set_source_reference ((ValaCodeNode*) versioned_local, vala_code_node_get_source_reference (node));
					}
					gee_collection_add ((GeeCollection*) self->priv->used_vars, versioned_local);
					(var_symbol == NULL) ? NULL : (var_symbol = (vala_code_node_unref (var_symbol), NULL));
					(variable_stack == NULL) ? NULL : (variable_stack = (gee_collection_object_unref (variable_stack), NULL));
					(versioned_local == NULL) ? NULL : (versioned_local = (vala_code_node_unref (versioned_local), NULL));
				}
				(var_symbol_it == NULL) ? NULL : (var_symbol_it = (gee_collection_object_unref (var_symbol_it), NULL));
			}
			defined_variables = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
			vala_code_node_get_defined_variables (node, (GeeCollection*) defined_variables);
			{
				GeeIterator* local_it;
				local_it = gee_iterable_iterator ((GeeIterable*) defined_variables);
				while (gee_iterator_next (local_it)) {
					ValaLocalVariable* local;
					ValaLocalVariable* _tmp6;
					local = (ValaLocalVariable*) gee_iterator_get (local_it);
					_tmp6 = NULL;
					_tmp6 = vala_flow_analyzer_process_assignment (self, m, self->priv->var_map, local);
					(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				}
				(local_it == NULL) ? NULL : (local_it = (gee_collection_object_unref (local_it), NULL));
			}
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
			(used_variables == NULL) ? NULL : (used_variables = (gee_collection_object_unref (used_variables), NULL));
			(defined_variables == NULL) ? NULL : (defined_variables = (gee_collection_object_unref (defined_variables), NULL));
		}
		(node_it == NULL) ? NULL : (node_it = (gee_collection_object_unref (node_it), NULL));
	}
	{
		GeeList* _tmp7;
		GeeIterator* _tmp8;
		GeeIterator* succ_it;
		_tmp7 = NULL;
		_tmp8 = NULL;
		succ_it = (_tmp8 = gee_iterable_iterator ((GeeIterable*) (_tmp7 = vala_basic_block_get_successors (block))), (_tmp7 == NULL) ? NULL : (_tmp7 = (gee_collection_object_unref (_tmp7), NULL)), _tmp8);
		while (gee_iterator_next (succ_it)) {
			ValaBasicBlock* _tmp9;
			ValaBasicBlock* succ;
			gint j;
			_tmp9 = NULL;
			succ = (_tmp9 = (ValaBasicBlock*) gee_iterator_get (succ_it), (_tmp9 == NULL) ? NULL : vala_basic_block_ref (_tmp9));
			j = 0;
			{
				GeeList* _tmp10;
				GeeIterator* _tmp11;
				GeeIterator* pred_it;
				_tmp10 = NULL;
				_tmp11 = NULL;
				pred_it = (_tmp11 = gee_iterable_iterator ((GeeIterable*) (_tmp10 = vala_basic_block_get_predecessors (succ))), (_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL)), _tmp11);
				while (gee_iterator_next (pred_it)) {
					ValaBasicBlock* _tmp12;
					ValaBasicBlock* pred;
					_tmp12 = NULL;
					pred = (_tmp12 = (ValaBasicBlock*) gee_iterator_get (pred_it), (_tmp12 == NULL) ? NULL : vala_basic_block_ref (_tmp12));
					if (pred == block) {
						(pred == NULL) ? NULL : (pred = (vala_basic_block_unref (pred), NULL));
						break;
					}
					j++;
					(pred == NULL) ? NULL : (pred = (vala_basic_block_unref (pred), NULL));
				}
				(pred_it == NULL) ? NULL : (pred_it = (gee_collection_object_unref (pred_it), NULL));
			}
			{
				GeeSet* _tmp13;
				GeeIterator* _tmp14;
				GeeIterator* phi_it;
				_tmp13 = NULL;
				_tmp14 = NULL;
				phi_it = (_tmp14 = gee_iterable_iterator ((GeeIterable*) (_tmp13 = vala_basic_block_get_phi_functions (succ))), (_tmp13 == NULL) ? NULL : (_tmp13 = (gee_collection_object_unref (_tmp13), NULL)), _tmp14);
				while (gee_iterator_next (phi_it)) {
					ValaPhiFunction* phi;
					GeeList* variable_stack;
					gboolean _tmp15;
					phi = (ValaPhiFunction*) gee_iterator_get (phi_it);
					variable_stack = (GeeList*) gee_map_get (self->priv->var_map, (ValaSymbol*) vala_phi_function_get_original_variable (phi));
					_tmp15 = FALSE;
					if (variable_stack != NULL) {
						_tmp15 = gee_collection_get_size ((GeeCollection*) variable_stack) > 0;
					} else {
						_tmp15 = FALSE;
					}
					if (_tmp15) {
						ValaLocalVariable* _tmp16;
						_tmp16 = NULL;
						gee_list_set (vala_phi_function_get_operands (phi), j, _tmp16 = (ValaLocalVariable*) gee_list_get (variable_stack, gee_collection_get_size ((GeeCollection*) variable_stack) - 1));
						(_tmp16 == NULL) ? NULL : (_tmp16 = (vala_code_node_unref (_tmp16), NULL));
					}
					(phi == NULL) ? NULL : (phi = (vala_phi_function_unref (phi), NULL));
					(variable_stack == NULL) ? NULL : (variable_stack = (gee_collection_object_unref (variable_stack), NULL));
				}
				(phi_it == NULL) ? NULL : (phi_it = (gee_collection_object_unref (phi_it), NULL));
			}
			(succ == NULL) ? NULL : (succ = (vala_basic_block_unref (succ), NULL));
		}
		(succ_it == NULL) ? NULL : (succ_it = (gee_collection_object_unref (succ_it), NULL));
	}
	{
		GeeList* _tmp17;
		GeeIterator* _tmp18;
		GeeIterator* child_it;
		_tmp17 = NULL;
		_tmp18 = NULL;
		child_it = (_tmp18 = gee_iterable_iterator ((GeeIterable*) (_tmp17 = vala_basic_block_get_children (block))), (_tmp17 == NULL) ? NULL : (_tmp17 = (gee_collection_object_unref (_tmp17), NULL)), _tmp18);
		while (gee_iterator_next (child_it)) {
			ValaBasicBlock* child;
			child = (ValaBasicBlock*) gee_iterator_get (child_it);
			vala_flow_analyzer_check_block_variables (self, m, child);
			(child == NULL) ? NULL : (child = (vala_basic_block_unref (child), NULL));
		}
		(child_it == NULL) ? NULL : (child_it = (gee_collection_object_unref (child_it), NULL));
	}
	{
		GeeSet* _tmp19;
		GeeIterator* _tmp20;
		GeeIterator* phi_it;
		_tmp19 = NULL;
		_tmp20 = NULL;
		phi_it = (_tmp20 = gee_iterable_iterator ((GeeIterable*) (_tmp19 = vala_basic_block_get_phi_functions (block))), (_tmp19 == NULL) ? NULL : (_tmp19 = (gee_collection_object_unref (_tmp19), NULL)), _tmp20);
		while (gee_iterator_next (phi_it)) {
			ValaPhiFunction* phi;
			GeeList* variable_stack;
			phi = (ValaPhiFunction*) gee_iterator_get (phi_it);
			variable_stack = (GeeList*) gee_map_get (self->priv->var_map, (ValaSymbol*) vala_phi_function_get_original_variable (phi));
			gee_list_remove_at (variable_stack, gee_collection_get_size ((GeeCollection*) variable_stack) - 1);
			(phi == NULL) ? NULL : (phi = (vala_phi_function_unref (phi), NULL));
			(variable_stack == NULL) ? NULL : (variable_stack = (gee_collection_object_unref (variable_stack), NULL));
		}
		(phi_it == NULL) ? NULL : (phi_it = (gee_collection_object_unref (phi_it), NULL));
	}
	{
		GeeList* _tmp21;
		GeeIterator* _tmp22;
		GeeIterator* node_it;
		_tmp21 = NULL;
		_tmp22 = NULL;
		node_it = (_tmp22 = gee_iterable_iterator ((GeeIterable*) (_tmp21 = vala_basic_block_get_nodes (block))), (_tmp21 == NULL) ? NULL : (_tmp21 = (gee_collection_object_unref (_tmp21), NULL)), _tmp22);
		while (gee_iterator_next (node_it)) {
			ValaCodeNode* node;
			GeeArrayList* defined_variables;
			node = (ValaCodeNode*) gee_iterator_get (node_it);
			defined_variables = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
			vala_code_node_get_defined_variables (node, (GeeCollection*) defined_variables);
			{
				GeeIterator* local_it;
				local_it = gee_iterable_iterator ((GeeIterable*) defined_variables);
				while (gee_iterator_next (local_it)) {
					ValaLocalVariable* local;
					GeeList* variable_stack;
					local = (ValaLocalVariable*) gee_iterator_get (local_it);
					variable_stack = (GeeList*) gee_map_get (self->priv->var_map, (ValaSymbol*) local);
					gee_list_remove_at (variable_stack, gee_collection_get_size ((GeeCollection*) variable_stack) - 1);
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
					(variable_stack == NULL) ? NULL : (variable_stack = (gee_collection_object_unref (variable_stack), NULL));
				}
				(local_it == NULL) ? NULL : (local_it = (gee_collection_object_unref (local_it), NULL));
			}
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
			(defined_variables == NULL) ? NULL : (defined_variables = (gee_collection_object_unref (defined_variables), NULL));
		}
		(node_it == NULL) ? NULL : (node_it = (gee_collection_object_unref (node_it), NULL));
	}
}


static ValaLocalVariable* vala_flow_analyzer_process_assignment (ValaFlowAnalyzer* self, ValaMethod* m, GeeMap* var_map, ValaLocalVariable* var_symbol) {
	GeeList* variable_stack;
	ValaLocalVariable* versioned_var;
	ValaLocalVariable* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (m != NULL, NULL);
	g_return_val_if_fail (var_map != NULL, NULL);
	g_return_val_if_fail (var_symbol != NULL, NULL);
	variable_stack = (GeeList*) gee_map_get (var_map, (ValaSymbol*) var_symbol);
	if (variable_stack == NULL) {
		GeeList* _tmp0;
		_tmp0 = NULL;
		variable_stack = (_tmp0 = (GeeList*) gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (variable_stack == NULL) ? NULL : (variable_stack = (gee_collection_object_unref (variable_stack), NULL)), _tmp0);
		gee_map_set (var_map, (ValaSymbol*) var_symbol, variable_stack);
	}
	versioned_var = vala_local_variable_new (vala_local_variable_get_variable_type (var_symbol), vala_symbol_get_name ((ValaSymbol*) var_symbol), NULL, vala_code_node_get_source_reference ((ValaCodeNode*) var_symbol));
	gee_collection_add ((GeeCollection*) variable_stack, versioned_var);
	_tmp1 = NULL;
	return (_tmp1 = versioned_var, (variable_stack == NULL) ? NULL : (variable_stack = (gee_collection_object_unref (variable_stack), NULL)), _tmp1);
}


static void vala_flow_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (prop != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) prop, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* _tmp0;
	ValaBasicBlock* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaFlowAnalyzerJumpTarget* _tmp3;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (acc != NULL);
	if (vala_property_accessor_get_body (acc) == NULL) {
		return;
	}
	_tmp0 = NULL;
	vala_property_accessor_set_entry_block (acc, _tmp0 = vala_basic_block_new_entry ());
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_basic_block_unref (_tmp0), NULL));
	_tmp1 = NULL;
	vala_property_accessor_set_exit_block (acc, _tmp1 = vala_basic_block_new_exit ());
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_basic_block_unref (_tmp1), NULL));
	_tmp2 = NULL;
	self->priv->current_block = (_tmp2 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp2);
	vala_basic_block_connect (vala_property_accessor_get_entry_block (acc), self->priv->current_block);
	_tmp3 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp3 = vala_flow_analyzer_jump_target_new_return_target (vala_property_accessor_get_exit_block (acc)));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_flow_analyzer_jump_target_unref (_tmp3), NULL));
	vala_code_node_accept_children ((ValaCodeNode*) acc, (ValaCodeVisitor*) self);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	if (self->priv->current_block != NULL) {
		/* end of property accessor body reachable*/
		if (vala_property_accessor_get_readable (acc)) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) acc), "missing return statement at end of property getter body");
			vala_code_node_set_error ((ValaCodeNode*) acc, TRUE);
		}
		vala_basic_block_connect (self->priv->current_block, vala_property_accessor_get_exit_block (acc));
	}
}


static void vala_flow_analyzer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (b != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) b, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaLocalVariable* _tmp2;
	ValaSymbol* _tmp1;
	ValaLocalVariable* local;
	gboolean _tmp3;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	if (!vala_symbol_get_used (vala_declaration_statement_get_declaration (stmt))) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) vala_declaration_statement_get_declaration (stmt)), _tmp0 = g_strdup_printf ("local variable `%s' declared but never used", vala_symbol_get_name (vala_declaration_statement_get_declaration (stmt))));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) stmt);
	_tmp2 = NULL;
	_tmp1 = NULL;
	local = (_tmp2 = (_tmp1 = vala_declaration_statement_get_declaration (stmt), VALA_IS_LOCAL_VARIABLE (_tmp1) ? ((ValaLocalVariable*) _tmp1) : NULL), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	_tmp3 = FALSE;
	if (local != NULL) {
		_tmp3 = vala_local_variable_get_initializer (local) != NULL;
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_local_variable_get_initializer (local));
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


static void vala_flow_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) stmt);
	vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) stmt);
	if (VALA_IS_METHOD_CALL (vala_expression_statement_get_expression (stmt))) {
		ValaMethodCall* _tmp0;
		ValaMethodCall* expr;
		ValaMemberAccess* _tmp2;
		ValaExpression* _tmp1;
		ValaMemberAccess* ma;
		gboolean _tmp3;
		gboolean _tmp4;
		_tmp0 = NULL;
		expr = (_tmp0 = VALA_METHOD_CALL (vala_expression_statement_get_expression (stmt)), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
		_tmp2 = NULL;
		_tmp1 = NULL;
		ma = (_tmp2 = (_tmp1 = vala_method_call_get_call (expr), VALA_IS_MEMBER_ACCESS (_tmp1) ? ((ValaMemberAccess*) _tmp1) : NULL), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		if (ma != NULL) {
			_tmp4 = vala_expression_get_symbol_reference ((ValaExpression*) ma) != NULL;
		} else {
			_tmp4 = FALSE;
		}
		if (_tmp4) {
			ValaAttribute* _tmp5;
			_tmp5 = NULL;
			_tmp3 = (_tmp5 = vala_code_node_get_attribute ((ValaCodeNode*) vala_expression_get_symbol_reference ((ValaExpression*) ma), "NoReturn")) != NULL;
			(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			ValaBasicBlock* _tmp6;
			_tmp6 = NULL;
			self->priv->current_block = (_tmp6 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp6);
			self->priv->unreachable_reported = FALSE;
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			return;
		}
		(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
}


static gboolean vala_flow_analyzer_always_true (ValaFlowAnalyzer* self, ValaExpression* condition) {
	ValaBooleanLiteral* _tmp1;
	ValaExpression* _tmp0;
	ValaBooleanLiteral* literal;
	gboolean _tmp2;
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (condition != NULL, FALSE);
	_tmp1 = NULL;
	_tmp0 = NULL;
	literal = (_tmp1 = (_tmp0 = condition, VALA_IS_BOOLEAN_LITERAL (_tmp0) ? ((ValaBooleanLiteral*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp2 = FALSE;
	if (literal != NULL) {
		_tmp2 = vala_boolean_literal_get_value (literal);
	} else {
		_tmp2 = FALSE;
	}
	return (_tmp3 = (_tmp2), (literal == NULL) ? NULL : (literal = (vala_code_node_unref (literal), NULL)), _tmp3);
}


static gboolean vala_flow_analyzer_always_false (ValaFlowAnalyzer* self, ValaExpression* condition) {
	ValaBooleanLiteral* _tmp1;
	ValaExpression* _tmp0;
	ValaBooleanLiteral* literal;
	gboolean _tmp2;
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (condition != NULL, FALSE);
	_tmp1 = NULL;
	_tmp0 = NULL;
	literal = (_tmp1 = (_tmp0 = condition, VALA_IS_BOOLEAN_LITERAL (_tmp0) ? ((ValaBooleanLiteral*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp2 = FALSE;
	if (literal != NULL) {
		_tmp2 = !vala_boolean_literal_get_value (literal);
	} else {
		_tmp2 = FALSE;
	}
	return (_tmp3 = (_tmp2), (literal == NULL) ? NULL : (literal = (vala_code_node_unref (literal), NULL)), _tmp3);
}


static void vala_flow_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* _tmp0;
	ValaBasicBlock* last_block;
	ValaBasicBlock* _tmp3;
	ValaBasicBlock* last_true_block;
	ValaBasicBlock* _tmp6;
	ValaBasicBlock* last_false_block;
	gboolean _tmp7;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	/* condition*/
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) vala_if_statement_get_condition (stmt));
	vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_if_statement_get_condition (stmt));
	/* true block*/
	_tmp0 = NULL;
	last_block = (_tmp0 = self->priv->current_block, (_tmp0 == NULL) ? NULL : vala_basic_block_ref (_tmp0));
	if (vala_flow_analyzer_always_false (self, vala_if_statement_get_condition (stmt))) {
		ValaBasicBlock* _tmp1;
		_tmp1 = NULL;
		self->priv->current_block = (_tmp1 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp1);
		self->priv->unreachable_reported = FALSE;
	} else {
		ValaBasicBlock* _tmp2;
		_tmp2 = NULL;
		self->priv->current_block = (_tmp2 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp2);
		vala_basic_block_connect (last_block, self->priv->current_block);
	}
	vala_code_node_accept ((ValaCodeNode*) vala_if_statement_get_true_statement (stmt), (ValaCodeVisitor*) self);
	/* false block*/
	_tmp3 = NULL;
	last_true_block = (_tmp3 = self->priv->current_block, (_tmp3 == NULL) ? NULL : vala_basic_block_ref (_tmp3));
	if (vala_flow_analyzer_always_true (self, vala_if_statement_get_condition (stmt))) {
		ValaBasicBlock* _tmp4;
		_tmp4 = NULL;
		self->priv->current_block = (_tmp4 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp4);
		self->priv->unreachable_reported = FALSE;
	} else {
		ValaBasicBlock* _tmp5;
		_tmp5 = NULL;
		self->priv->current_block = (_tmp5 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp5);
		vala_basic_block_connect (last_block, self->priv->current_block);
	}
	if (vala_if_statement_get_false_statement (stmt) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_if_statement_get_false_statement (stmt), (ValaCodeVisitor*) self);
	}
	/* after if/else*/
	_tmp6 = NULL;
	last_false_block = (_tmp6 = self->priv->current_block, (_tmp6 == NULL) ? NULL : vala_basic_block_ref (_tmp6));
	_tmp7 = FALSE;
	if (last_true_block != NULL) {
		_tmp7 = TRUE;
	} else {
		_tmp7 = last_false_block != NULL;
	}
	/* reachable?*/
	if (_tmp7) {
		ValaBasicBlock* _tmp8;
		_tmp8 = NULL;
		self->priv->current_block = (_tmp8 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp8);
		if (last_true_block != NULL) {
			vala_basic_block_connect (last_true_block, self->priv->current_block);
		}
		if (last_false_block != NULL) {
			vala_basic_block_connect (last_false_block, self->priv->current_block);
		}
	}
	(last_block == NULL) ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL));
	(last_true_block == NULL) ? NULL : (last_true_block = (vala_basic_block_unref (last_true_block), NULL));
	(last_false_block == NULL) ? NULL : (last_false_block = (vala_basic_block_unref (last_false_block), NULL));
}


static void vala_flow_analyzer_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* after_switch_block;
	ValaFlowAnalyzerJumpTarget* _tmp0;
	ValaBasicBlock* _tmp1;
	ValaBasicBlock* condition_block;
	gboolean has_default_label;
	GeeList* _tmp7;
	gboolean _tmp8;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	after_switch_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp0 = vala_flow_analyzer_jump_target_new_break_target (after_switch_block));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_flow_analyzer_jump_target_unref (_tmp0), NULL));
	/* condition*/
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) vala_switch_statement_get_expression (stmt));
	_tmp1 = NULL;
	condition_block = (_tmp1 = self->priv->current_block, (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1));
	vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_switch_statement_get_expression (stmt));
	has_default_label = FALSE;
	{
		GeeList* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* section_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		section_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = vala_switch_statement_get_sections (stmt))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (section_it)) {
			ValaSwitchSection* section;
			ValaBasicBlock* _tmp4;
			section = (ValaSwitchSection*) gee_iterator_get (section_it);
			_tmp4 = NULL;
			self->priv->current_block = (_tmp4 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp4);
			vala_basic_block_connect (condition_block, self->priv->current_block);
			{
				GeeList* _tmp5;
				GeeIterator* _tmp6;
				GeeIterator* stmt_it;
				_tmp5 = NULL;
				_tmp6 = NULL;
				stmt_it = (_tmp6 = gee_iterable_iterator ((GeeIterable*) (_tmp5 = vala_block_get_statements ((ValaBlock*) section))), (_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL)), _tmp6);
				while (gee_iterator_next (stmt_it)) {
					ValaStatement* stmt;
					stmt = (ValaStatement*) gee_iterator_get (stmt_it);
					vala_code_node_accept ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
					(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
				}
				(stmt_it == NULL) ? NULL : (stmt_it = (gee_collection_object_unref (stmt_it), NULL));
			}
			if (vala_switch_section_has_default_label (section)) {
				has_default_label = TRUE;
			}
			if (self->priv->current_block != NULL) {
				/* end of switch section reachable
				 we don't allow fall-through*/
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) section), "missing break statement at end of switch section");
				vala_code_node_set_error ((ValaCodeNode*) section, TRUE);
				vala_basic_block_connect (self->priv->current_block, after_switch_block);
			}
			(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
		}
		(section_it == NULL) ? NULL : (section_it = (gee_collection_object_unref (section_it), NULL));
	}
	if (!has_default_label) {
		vala_basic_block_connect (condition_block, after_switch_block);
	}
	/* after switch
	 reachable?*/
	_tmp7 = NULL;
	if ((_tmp8 = gee_collection_get_size ((GeeCollection*) (_tmp7 = vala_basic_block_get_predecessors (after_switch_block))) > 0, (_tmp7 == NULL) ? NULL : (_tmp7 = (gee_collection_object_unref (_tmp7), NULL)), _tmp8)) {
		ValaBasicBlock* _tmp10;
		ValaBasicBlock* _tmp9;
		_tmp10 = NULL;
		_tmp9 = NULL;
		self->priv->current_block = (_tmp10 = (_tmp9 = after_switch_block, (_tmp9 == NULL) ? NULL : vala_basic_block_ref (_tmp9)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp10);
	} else {
		ValaBasicBlock* _tmp11;
		_tmp11 = NULL;
		self->priv->current_block = (_tmp11 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp11);
		self->priv->unreachable_reported = FALSE;
	}
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	(after_switch_block == NULL) ? NULL : (after_switch_block = (vala_basic_block_unref (after_switch_block), NULL));
	(condition_block == NULL) ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL));
}


static void vala_flow_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* condition_block;
	ValaFlowAnalyzerJumpTarget* _tmp0;
	ValaBasicBlock* after_loop_block;
	ValaFlowAnalyzerJumpTarget* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* last_block;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* _tmp3;
	gboolean _tmp7;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	condition_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp0 = vala_flow_analyzer_jump_target_new_continue_target (condition_block));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_flow_analyzer_jump_target_unref (_tmp0), NULL));
	after_loop_block = vala_basic_block_new ();
	_tmp1 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp1 = vala_flow_analyzer_jump_target_new_break_target (after_loop_block));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_flow_analyzer_jump_target_unref (_tmp1), NULL));
	/* condition*/
	_tmp2 = NULL;
	last_block = (_tmp2 = self->priv->current_block, (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2));
	vala_basic_block_connect (last_block, condition_block);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_block = (_tmp4 = (_tmp3 = condition_block, (_tmp3 == NULL) ? NULL : vala_basic_block_ref (_tmp3)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp4);
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) vala_while_statement_get_condition (stmt));
	vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_while_statement_get_condition (stmt));
	/* loop block*/
	if (vala_flow_analyzer_always_false (self, vala_while_statement_get_condition (stmt))) {
		ValaBasicBlock* _tmp5;
		_tmp5 = NULL;
		self->priv->current_block = (_tmp5 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp5);
		self->priv->unreachable_reported = FALSE;
	} else {
		ValaBasicBlock* _tmp6;
		_tmp6 = NULL;
		self->priv->current_block = (_tmp6 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp6);
		vala_basic_block_connect (condition_block, self->priv->current_block);
	}
	vala_code_node_accept ((ValaCodeNode*) vala_while_statement_get_body (stmt), (ValaCodeVisitor*) self);
	/* end of loop block reachable?*/
	if (self->priv->current_block != NULL) {
		vala_basic_block_connect (self->priv->current_block, condition_block);
	}
	_tmp7 = FALSE;
	if (vala_flow_analyzer_always_true (self, vala_while_statement_get_condition (stmt))) {
		GeeList* _tmp8;
		_tmp8 = NULL;
		_tmp7 = gee_collection_get_size ((GeeCollection*) (_tmp8 = vala_basic_block_get_predecessors (after_loop_block))) == 0;
		(_tmp8 == NULL) ? NULL : (_tmp8 = (gee_collection_object_unref (_tmp8), NULL));
	} else {
		_tmp7 = FALSE;
	}
	/* after loop
	 reachable?*/
	if (_tmp7) {
		ValaBasicBlock* _tmp9;
		_tmp9 = NULL;
		self->priv->current_block = (_tmp9 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp9);
		self->priv->unreachable_reported = FALSE;
	} else {
		ValaBasicBlock* _tmp11;
		ValaBasicBlock* _tmp10;
		vala_basic_block_connect (condition_block, after_loop_block);
		_tmp11 = NULL;
		_tmp10 = NULL;
		self->priv->current_block = (_tmp11 = (_tmp10 = after_loop_block, (_tmp10 == NULL) ? NULL : vala_basic_block_ref (_tmp10)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp11);
	}
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	(condition_block == NULL) ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL));
	(after_loop_block == NULL) ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL));
	(last_block == NULL) ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL));
}


static void vala_flow_analyzer_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* condition_block;
	ValaFlowAnalyzerJumpTarget* _tmp0;
	ValaBasicBlock* after_loop_block;
	ValaFlowAnalyzerJumpTarget* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* last_block;
	ValaBasicBlock* loop_block;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* _tmp3;
	gboolean _tmp5;
	gboolean _tmp11;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	condition_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp0 = vala_flow_analyzer_jump_target_new_continue_target (condition_block));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_flow_analyzer_jump_target_unref (_tmp0), NULL));
	after_loop_block = vala_basic_block_new ();
	_tmp1 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp1 = vala_flow_analyzer_jump_target_new_break_target (after_loop_block));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_flow_analyzer_jump_target_unref (_tmp1), NULL));
	/* loop block*/
	_tmp2 = NULL;
	last_block = (_tmp2 = self->priv->current_block, (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2));
	loop_block = vala_basic_block_new ();
	vala_basic_block_connect (last_block, loop_block);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_block = (_tmp4 = (_tmp3 = loop_block, (_tmp3 == NULL) ? NULL : vala_basic_block_ref (_tmp3)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp4);
	vala_code_node_accept ((ValaCodeNode*) vala_do_statement_get_body (stmt), (ValaCodeVisitor*) self);
	_tmp5 = FALSE;
	if (self->priv->current_block != NULL) {
		_tmp5 = TRUE;
	} else {
		GeeList* _tmp6;
		_tmp6 = NULL;
		_tmp5 = gee_collection_get_size ((GeeCollection*) (_tmp6 = vala_basic_block_get_predecessors (condition_block))) > 0;
		(_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL));
	}
	/* condition
	 reachable?*/
	if (_tmp5) {
		ValaBasicBlock* _tmp10;
		ValaBasicBlock* _tmp9;
		if (self->priv->current_block != NULL) {
			ValaBasicBlock* _tmp8;
			ValaBasicBlock* _tmp7;
			_tmp8 = NULL;
			_tmp7 = NULL;
			last_block = (_tmp8 = (_tmp7 = self->priv->current_block, (_tmp7 == NULL) ? NULL : vala_basic_block_ref (_tmp7)), (last_block == NULL) ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL)), _tmp8);
			vala_basic_block_connect (last_block, condition_block);
		}
		vala_basic_block_add_node (condition_block, (ValaCodeNode*) vala_do_statement_get_condition (stmt));
		vala_basic_block_connect (condition_block, loop_block);
		_tmp10 = NULL;
		_tmp9 = NULL;
		self->priv->current_block = (_tmp10 = (_tmp9 = condition_block, (_tmp9 == NULL) ? NULL : vala_basic_block_ref (_tmp9)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp10);
		vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_do_statement_get_condition (stmt));
	}
	_tmp11 = FALSE;
	if (self->priv->current_block != NULL) {
		_tmp11 = TRUE;
	} else {
		GeeList* _tmp12;
		_tmp12 = NULL;
		_tmp11 = gee_collection_get_size ((GeeCollection*) (_tmp12 = vala_basic_block_get_predecessors (after_loop_block))) > 0;
		(_tmp12 == NULL) ? NULL : (_tmp12 = (gee_collection_object_unref (_tmp12), NULL));
	}
	/* after loop
	 reachable?*/
	if (_tmp11) {
		ValaBasicBlock* _tmp16;
		ValaBasicBlock* _tmp15;
		if (self->priv->current_block != NULL) {
			ValaBasicBlock* _tmp14;
			ValaBasicBlock* _tmp13;
			_tmp14 = NULL;
			_tmp13 = NULL;
			last_block = (_tmp14 = (_tmp13 = self->priv->current_block, (_tmp13 == NULL) ? NULL : vala_basic_block_ref (_tmp13)), (last_block == NULL) ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL)), _tmp14);
			vala_basic_block_connect (last_block, after_loop_block);
		}
		_tmp16 = NULL;
		_tmp15 = NULL;
		self->priv->current_block = (_tmp16 = (_tmp15 = after_loop_block, (_tmp15 == NULL) ? NULL : vala_basic_block_ref (_tmp15)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp16);
	}
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	(condition_block == NULL) ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL));
	(after_loop_block == NULL) ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL));
	(last_block == NULL) ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL));
	(loop_block == NULL) ? NULL : (loop_block = (vala_basic_block_unref (loop_block), NULL));
}


static void vala_flow_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* iterator_block;
	ValaFlowAnalyzerJumpTarget* _tmp2;
	ValaBasicBlock* after_loop_block;
	ValaFlowAnalyzerJumpTarget* _tmp3;
	ValaBasicBlock* condition_block;
	ValaBasicBlock* _tmp5;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* _tmp6;
	gboolean _tmp7;
	ValaBasicBlock* _tmp14;
	ValaBasicBlock* _tmp13;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	/* initializer*/
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* init_expr_it;
		/* initializer*/
		_tmp0 = NULL;
		_tmp1 = NULL;
		init_expr_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_for_statement_get_initializer (stmt))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		/* initializer*/
		while (gee_iterator_next (init_expr_it)) {
			ValaExpression* init_expr;
			/* initializer*/
			init_expr = (ValaExpression*) gee_iterator_get (init_expr_it);
			vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) init_expr);
			vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) init_expr);
			(init_expr == NULL) ? NULL : (init_expr = (vala_code_node_unref (init_expr), NULL));
		}
		(init_expr_it == NULL) ? NULL : (init_expr_it = (gee_collection_object_unref (init_expr_it), NULL));
	}
	iterator_block = vala_basic_block_new ();
	_tmp2 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp2 = vala_flow_analyzer_jump_target_new_continue_target (iterator_block));
	(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_flow_analyzer_jump_target_unref (_tmp2), NULL));
	after_loop_block = vala_basic_block_new ();
	_tmp3 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp3 = vala_flow_analyzer_jump_target_new_break_target (after_loop_block));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_flow_analyzer_jump_target_unref (_tmp3), NULL));
	/* condition*/
	condition_block = vala_basic_block_new ();
	vala_basic_block_connect (self->priv->current_block, condition_block);
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->current_block = (_tmp5 = (_tmp4 = condition_block, (_tmp4 == NULL) ? NULL : vala_basic_block_ref (_tmp4)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp5);
	if (vala_for_statement_get_condition (stmt) != NULL) {
		vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) vala_for_statement_get_condition (stmt));
	}
	if (vala_for_statement_get_condition (stmt) != NULL) {
		vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_for_statement_get_condition (stmt));
	}
	/* loop block*/
	_tmp6 = NULL;
	self->priv->current_block = (_tmp6 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp6);
	vala_basic_block_connect (condition_block, self->priv->current_block);
	vala_code_node_accept ((ValaCodeNode*) vala_for_statement_get_body (stmt), (ValaCodeVisitor*) self);
	_tmp7 = FALSE;
	if (self->priv->current_block != NULL) {
		_tmp7 = TRUE;
	} else {
		GeeList* _tmp8;
		_tmp8 = NULL;
		_tmp7 = gee_collection_get_size ((GeeCollection*) (_tmp8 = vala_basic_block_get_predecessors (iterator_block))) > 0;
		(_tmp8 == NULL) ? NULL : (_tmp8 = (gee_collection_object_unref (_tmp8), NULL));
	}
	/* iterator
	 reachable?*/
	if (_tmp7) {
		ValaBasicBlock* _tmp10;
		ValaBasicBlock* _tmp9;
		if (self->priv->current_block != NULL) {
			vala_basic_block_connect (self->priv->current_block, iterator_block);
		}
		_tmp10 = NULL;
		_tmp9 = NULL;
		self->priv->current_block = (_tmp10 = (_tmp9 = iterator_block, (_tmp9 == NULL) ? NULL : vala_basic_block_ref (_tmp9)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp10);
		{
			GeeList* _tmp11;
			GeeIterator* _tmp12;
			GeeIterator* it_expr_it;
			_tmp11 = NULL;
			_tmp12 = NULL;
			it_expr_it = (_tmp12 = gee_iterable_iterator ((GeeIterable*) (_tmp11 = vala_for_statement_get_iterator (stmt))), (_tmp11 == NULL) ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL)), _tmp12);
			while (gee_iterator_next (it_expr_it)) {
				ValaExpression* it_expr;
				it_expr = (ValaExpression*) gee_iterator_get (it_expr_it);
				vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) it_expr);
				vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) it_expr);
				(it_expr == NULL) ? NULL : (it_expr = (vala_code_node_unref (it_expr), NULL));
			}
			(it_expr_it == NULL) ? NULL : (it_expr_it = (gee_collection_object_unref (it_expr_it), NULL));
		}
		vala_basic_block_connect (self->priv->current_block, condition_block);
	}
	/* after loop*/
	vala_basic_block_connect (condition_block, after_loop_block);
	_tmp14 = NULL;
	_tmp13 = NULL;
	self->priv->current_block = (_tmp14 = (_tmp13 = after_loop_block, (_tmp13 == NULL) ? NULL : vala_basic_block_ref (_tmp13)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp14);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	(iterator_block == NULL) ? NULL : (iterator_block = (vala_basic_block_unref (iterator_block), NULL));
	(after_loop_block == NULL) ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL));
	(condition_block == NULL) ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL));
}


static void vala_flow_analyzer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* loop_block;
	ValaFlowAnalyzerJumpTarget* _tmp0;
	ValaBasicBlock* after_loop_block;
	ValaFlowAnalyzerJumpTarget* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* last_block;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* _tmp3;
	ValaBasicBlock* _tmp6;
	ValaBasicBlock* _tmp5;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	/* collection*/
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) vala_foreach_statement_get_collection (stmt));
	vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_foreach_statement_get_collection (stmt));
	loop_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp0 = vala_flow_analyzer_jump_target_new_continue_target (loop_block));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_flow_analyzer_jump_target_unref (_tmp0), NULL));
	after_loop_block = vala_basic_block_new ();
	_tmp1 = NULL;
	gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp1 = vala_flow_analyzer_jump_target_new_break_target (after_loop_block));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_flow_analyzer_jump_target_unref (_tmp1), NULL));
	/* loop block*/
	_tmp2 = NULL;
	last_block = (_tmp2 = self->priv->current_block, (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2));
	vala_basic_block_connect (last_block, loop_block);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_block = (_tmp4 = (_tmp3 = loop_block, (_tmp3 == NULL) ? NULL : vala_basic_block_ref (_tmp3)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp4);
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) stmt);
	vala_code_node_accept ((ValaCodeNode*) vala_foreach_statement_get_body (stmt), (ValaCodeVisitor*) self);
	if (self->priv->current_block != NULL) {
		vala_basic_block_connect (self->priv->current_block, loop_block);
	}
	/* after loop*/
	vala_basic_block_connect (last_block, after_loop_block);
	if (self->priv->current_block != NULL) {
		vala_basic_block_connect (self->priv->current_block, after_loop_block);
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	self->priv->current_block = (_tmp6 = (_tmp5 = after_loop_block, (_tmp5 == NULL) ? NULL : vala_basic_block_ref (_tmp5)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp6);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	(loop_block == NULL) ? NULL : (loop_block = (vala_basic_block_unref (loop_block), NULL));
	(after_loop_block == NULL) ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL));
	(last_block == NULL) ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL));
}


static void vala_flow_analyzer_real_visit_break_statement (ValaCodeVisitor* base, ValaBreakStatement* stmt) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) stmt);
	{
		gint i;
		i = gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1;
		for (; i >= 0; i--) {
			ValaFlowAnalyzerJumpTarget* jump_target;
			jump_target = (ValaFlowAnalyzerJumpTarget*) gee_list_get ((GeeList*) self->priv->jump_stack, i);
			if (vala_flow_analyzer_jump_target_get_is_break_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
				return;
			} else {
				if (vala_flow_analyzer_jump_target_get_is_finally_clause (jump_target)) {
					ValaBasicBlock* _tmp2;
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
					_tmp2 = NULL;
					_tmp1 = NULL;
					self->priv->current_block = (_tmp2 = (_tmp1 = vala_flow_analyzer_jump_target_get_last_block (jump_target), (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp2);
				}
			}
			(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
		}
	}
	vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) stmt), "no enclosing loop or switch statement found");
	vala_code_node_set_error ((ValaCodeNode*) stmt, TRUE);
}


static void vala_flow_analyzer_real_visit_continue_statement (ValaCodeVisitor* base, ValaContinueStatement* stmt) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) stmt);
	{
		gint i;
		i = gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1;
		for (; i >= 0; i--) {
			ValaFlowAnalyzerJumpTarget* jump_target;
			jump_target = (ValaFlowAnalyzerJumpTarget*) gee_list_get ((GeeList*) self->priv->jump_stack, i);
			if (vala_flow_analyzer_jump_target_get_is_continue_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
				return;
			} else {
				if (vala_flow_analyzer_jump_target_get_is_finally_clause (jump_target)) {
					ValaBasicBlock* _tmp2;
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
					_tmp2 = NULL;
					_tmp1 = NULL;
					self->priv->current_block = (_tmp2 = (_tmp1 = vala_flow_analyzer_jump_target_get_last_block (jump_target), (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp2);
				}
			}
			(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
		}
	}
	vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) stmt), "no enclosing loop found");
	vala_code_node_set_error ((ValaCodeNode*) stmt, TRUE);
}


static void vala_flow_analyzer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) stmt);
	if (vala_return_statement_get_return_expression (stmt) != NULL) {
		vala_flow_analyzer_handle_errors (self, (ValaCodeNode*) vala_return_statement_get_return_expression (stmt));
	}
	{
		gint i;
		i = gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1;
		for (; i >= 0; i--) {
			ValaFlowAnalyzerJumpTarget* jump_target;
			jump_target = (ValaFlowAnalyzerJumpTarget*) gee_list_get ((GeeList*) self->priv->jump_stack, i);
			if (vala_flow_analyzer_jump_target_get_is_return_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
				return;
			} else {
				if (vala_flow_analyzer_jump_target_get_is_finally_clause (jump_target)) {
					ValaBasicBlock* _tmp2;
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
					_tmp2 = NULL;
					_tmp1 = NULL;
					self->priv->current_block = (_tmp2 = (_tmp1 = vala_flow_analyzer_jump_target_get_last_block (jump_target), (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp2);
				}
			}
			(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
		}
	}
	vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) stmt), "no enclosing loop found");
	vala_code_node_set_error ((ValaCodeNode*) stmt, TRUE);
}


static void vala_flow_analyzer_handle_errors (ValaFlowAnalyzer* self, ValaCodeNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	if (vala_code_node_get_tree_can_fail (node)) {
		ValaBasicBlock* _tmp0;
		ValaBasicBlock* last_block;
		ValaBasicBlock* _tmp5;
		_tmp0 = NULL;
		last_block = (_tmp0 = self->priv->current_block, (_tmp0 == NULL) ? NULL : vala_basic_block_ref (_tmp0));
		{
			gint i;
			/* exceptional control flow*/
			i = gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1;
			for (; i >= 0; i--) {
				ValaFlowAnalyzerJumpTarget* jump_target;
				jump_target = (ValaFlowAnalyzerJumpTarget*) gee_list_get ((GeeList*) self->priv->jump_stack, i);
				if (vala_flow_analyzer_jump_target_get_is_return_target (jump_target)) {
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
					_tmp1 = NULL;
					self->priv->current_block = (_tmp1 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp1);
					self->priv->unreachable_reported = FALSE;
					(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
					break;
				} else {
					if (vala_flow_analyzer_jump_target_get_is_error_target (jump_target)) {
						/* TODO check whether jump target catches node.error_type*/
						vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
						if (vala_flow_analyzer_jump_target_get_error_domain (jump_target) == NULL) {
							ValaBasicBlock* _tmp2;
							/* catch all clause*/
							_tmp2 = NULL;
							self->priv->current_block = (_tmp2 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp2);
							self->priv->unreachable_reported = FALSE;
							(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
							break;
						}
					} else {
						if (vala_flow_analyzer_jump_target_get_is_finally_clause (jump_target)) {
							ValaBasicBlock* _tmp4;
							ValaBasicBlock* _tmp3;
							vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
							_tmp4 = NULL;
							_tmp3 = NULL;
							self->priv->current_block = (_tmp4 = (_tmp3 = vala_flow_analyzer_jump_target_get_last_block (jump_target), (_tmp3 == NULL) ? NULL : vala_basic_block_ref (_tmp3)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp4);
						}
					}
				}
				(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
			}
		}
		/* normal control flow*/
		_tmp5 = NULL;
		self->priv->current_block = (_tmp5 = vala_basic_block_new (), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp5);
		vala_basic_block_connect (last_block, self->priv->current_block);
		(last_block == NULL) ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL));
	}
}


static void vala_flow_analyzer_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_flow_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) stmt);
	{
		gint i;
		i = gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1;
		for (; i >= 0; i--) {
			ValaFlowAnalyzerJumpTarget* jump_target;
			jump_target = (ValaFlowAnalyzerJumpTarget*) gee_list_get ((GeeList*) self->priv->jump_stack, i);
			if (vala_flow_analyzer_jump_target_get_is_return_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
				return;
			} else {
				if (vala_flow_analyzer_jump_target_get_is_error_target (jump_target)) {
					/* TODO check whether jump target catches stmt.error_type*/
					vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
					if (vala_flow_analyzer_jump_target_get_error_domain (jump_target) == NULL) {
						ValaBasicBlock* _tmp1;
						_tmp1 = NULL;
						self->priv->current_block = (_tmp1 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp1);
						self->priv->unreachable_reported = FALSE;
						(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
						return;
					}
				} else {
					if (vala_flow_analyzer_jump_target_get_is_finally_clause (jump_target)) {
						ValaBasicBlock* _tmp3;
						ValaBasicBlock* _tmp2;
						vala_basic_block_connect (self->priv->current_block, vala_flow_analyzer_jump_target_get_basic_block (jump_target));
						_tmp3 = NULL;
						_tmp2 = NULL;
						self->priv->current_block = (_tmp3 = (_tmp2 = vala_flow_analyzer_jump_target_get_last_block (jump_target), (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp3);
					}
				}
			}
			(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
		}
	}
	g_assert_not_reached ();
}


static void vala_flow_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaFlowAnalyzer * self;
	ValaBasicBlock* _tmp0;
	ValaBasicBlock* before_try_block;
	ValaBasicBlock* after_try_block;
	ValaBasicBlock* finally_block;
	gint finally_jump_stack_size;
	GeeList* catch_clauses;
	ValaBasicBlock* _tmp11;
	ValaBasicBlock* _tmp10;
	GeeList* catch_stack;
	GeeList* _tmp20;
	gboolean _tmp21;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	_tmp0 = NULL;
	before_try_block = (_tmp0 = self->priv->current_block, (_tmp0 == NULL) ? NULL : vala_basic_block_ref (_tmp0));
	after_try_block = vala_basic_block_new ();
	finally_block = NULL;
	if (vala_try_statement_get_finally_body (stmt) != NULL) {
		ValaBasicBlock* _tmp1;
		ValaBasicBlock* _tmp3;
		ValaBasicBlock* _tmp2;
		ValaFlowAnalyzerJumpTarget* _tmp4;
		_tmp1 = NULL;
		finally_block = (_tmp1 = vala_basic_block_new (), (finally_block == NULL) ? NULL : (finally_block = (vala_basic_block_unref (finally_block), NULL)), _tmp1);
		_tmp3 = NULL;
		_tmp2 = NULL;
		self->priv->current_block = (_tmp3 = (_tmp2 = finally_block, (_tmp2 == NULL) ? NULL : vala_basic_block_ref (_tmp2)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp3);
		vala_code_node_accept ((ValaCodeNode*) vala_try_statement_get_finally_body (stmt), (ValaCodeVisitor*) self);
		if (self->priv->current_block == NULL) {
			/* don't allow finally blocks with e.g. return statements*/
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) stmt), "end of finally block not reachable");
			vala_code_node_set_error ((ValaCodeNode*) stmt, TRUE);
			(before_try_block == NULL) ? NULL : (before_try_block = (vala_basic_block_unref (before_try_block), NULL));
			(after_try_block == NULL) ? NULL : (after_try_block = (vala_basic_block_unref (after_try_block), NULL));
			(finally_block == NULL) ? NULL : (finally_block = (vala_basic_block_unref (finally_block), NULL));
			return;
		}
		_tmp4 = NULL;
		gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp4 = vala_flow_analyzer_jump_target_new_finally_clause (finally_block, self->priv->current_block));
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_flow_analyzer_jump_target_unref (_tmp4), NULL));
	}
	finally_jump_stack_size = gee_collection_get_size ((GeeCollection*) self->priv->jump_stack);
	catch_clauses = vala_try_statement_get_catch_clauses (stmt);
	{
		gint i;
		i = gee_collection_get_size ((GeeCollection*) catch_clauses) - 1;
		for (; i >= 0; i--) {
			ValaCatchClause* catch_clause;
			catch_clause = (ValaCatchClause*) gee_list_get ((GeeList*) catch_clauses, i);
			if (vala_catch_clause_get_error_type (catch_clause) != NULL) {
				ValaFlowAnalyzerJumpTarget* _tmp7;
				ValaBasicBlock* _tmp5;
				ValaTypeSymbol* _tmp6;
				_tmp7 = NULL;
				_tmp5 = NULL;
				_tmp6 = NULL;
				gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp7 = vala_flow_analyzer_jump_target_new_error_target (_tmp5 = vala_basic_block_new (), catch_clause, (_tmp6 = vala_data_type_get_data_type (vala_catch_clause_get_error_type (catch_clause)), VALA_IS_ERROR_DOMAIN (_tmp6) ? ((ValaErrorDomain*) _tmp6) : NULL), NULL));
				(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_flow_analyzer_jump_target_unref (_tmp7), NULL));
				(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_basic_block_unref (_tmp5), NULL));
			} else {
				ValaFlowAnalyzerJumpTarget* _tmp9;
				ValaBasicBlock* _tmp8;
				_tmp9 = NULL;
				_tmp8 = NULL;
				gee_collection_add ((GeeCollection*) self->priv->jump_stack, _tmp9 = vala_flow_analyzer_jump_target_new_error_target (_tmp8 = vala_basic_block_new (), catch_clause, NULL, NULL));
				(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_flow_analyzer_jump_target_unref (_tmp9), NULL));
				(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_basic_block_unref (_tmp8), NULL));
			}
			(catch_clause == NULL) ? NULL : (catch_clause = (vala_code_node_unref (catch_clause), NULL));
		}
	}
	_tmp11 = NULL;
	_tmp10 = NULL;
	self->priv->current_block = (_tmp11 = (_tmp10 = before_try_block, (_tmp10 == NULL) ? NULL : vala_basic_block_ref (_tmp10)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp11);
	vala_code_node_accept ((ValaCodeNode*) vala_try_statement_get_body (stmt), (ValaCodeVisitor*) self);
	if (self->priv->current_block != NULL) {
		if (finally_block != NULL) {
			ValaBasicBlock* _tmp13;
			ValaBasicBlock* _tmp12;
			vala_basic_block_connect (self->priv->current_block, finally_block);
			_tmp13 = NULL;
			_tmp12 = NULL;
			self->priv->current_block = (_tmp13 = (_tmp12 = finally_block, (_tmp12 == NULL) ? NULL : vala_basic_block_ref (_tmp12)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp13);
		}
		vala_basic_block_connect (self->priv->current_block, after_try_block);
	}
	/* remove catch clauses from jump stack*/
	catch_stack = (GeeList*) gee_array_list_new (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, (GBoxedCopyFunc) vala_flow_analyzer_jump_target_ref, vala_flow_analyzer_jump_target_unref, g_direct_equal);
	{
		gint i;
		i = gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1;
		for (; i >= finally_jump_stack_size; i--) {
			ValaFlowAnalyzerJumpTarget* jump_target;
			jump_target = (ValaFlowAnalyzerJumpTarget*) gee_list_get ((GeeList*) self->priv->jump_stack, i);
			gee_collection_add ((GeeCollection*) catch_stack, jump_target);
			gee_list_remove_at (self->priv->jump_stack, i);
			(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
		}
	}
	{
		GeeIterator* jump_target_it;
		jump_target_it = gee_iterable_iterator ((GeeIterable*) catch_stack);
		while (gee_iterator_next (jump_target_it)) {
			ValaFlowAnalyzerJumpTarget* jump_target;
			GeeList* _tmp14;
			gboolean _tmp15;
			jump_target = (ValaFlowAnalyzerJumpTarget*) gee_iterator_get (jump_target_it);
			_tmp14 = NULL;
			if ((_tmp15 = gee_collection_get_size ((GeeCollection*) (_tmp14 = vala_basic_block_get_predecessors (vala_flow_analyzer_jump_target_get_basic_block (jump_target)))) == 0, (_tmp14 == NULL) ? NULL : (_tmp14 = (gee_collection_object_unref (_tmp14), NULL)), _tmp15)) {
				/* unreachable*/
				vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) vala_flow_analyzer_jump_target_get_catch_clause (jump_target)), "unreachable catch clause detected");
			} else {
				ValaBasicBlock* _tmp17;
				ValaBasicBlock* _tmp16;
				_tmp17 = NULL;
				_tmp16 = NULL;
				self->priv->current_block = (_tmp17 = (_tmp16 = vala_flow_analyzer_jump_target_get_basic_block (jump_target), (_tmp16 == NULL) ? NULL : vala_basic_block_ref (_tmp16)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp17);
				vala_basic_block_add_node (self->priv->current_block, (ValaCodeNode*) vala_flow_analyzer_jump_target_get_catch_clause (jump_target));
				vala_code_node_accept ((ValaCodeNode*) vala_catch_clause_get_body (vala_flow_analyzer_jump_target_get_catch_clause (jump_target)), (ValaCodeVisitor*) self);
				if (self->priv->current_block != NULL) {
					if (finally_block != NULL) {
						ValaBasicBlock* _tmp19;
						ValaBasicBlock* _tmp18;
						vala_basic_block_connect (self->priv->current_block, finally_block);
						_tmp19 = NULL;
						_tmp18 = NULL;
						self->priv->current_block = (_tmp19 = (_tmp18 = finally_block, (_tmp18 == NULL) ? NULL : vala_basic_block_ref (_tmp18)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp19);
					}
					vala_basic_block_connect (self->priv->current_block, after_try_block);
				}
			}
			(jump_target == NULL) ? NULL : (jump_target = (vala_flow_analyzer_jump_target_unref (jump_target), NULL));
		}
		(jump_target_it == NULL) ? NULL : (jump_target_it = (gee_collection_object_unref (jump_target_it), NULL));
	}
	if (finally_block != NULL) {
		gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size ((GeeCollection*) self->priv->jump_stack) - 1);
	}
	/* after try statement
	 reachable?*/
	_tmp20 = NULL;
	if ((_tmp21 = gee_collection_get_size ((GeeCollection*) (_tmp20 = vala_basic_block_get_predecessors (after_try_block))) > 0, (_tmp20 == NULL) ? NULL : (_tmp20 = (gee_collection_object_unref (_tmp20), NULL)), _tmp21)) {
		ValaBasicBlock* _tmp23;
		ValaBasicBlock* _tmp22;
		_tmp23 = NULL;
		_tmp22 = NULL;
		self->priv->current_block = (_tmp23 = (_tmp22 = after_try_block, (_tmp22 == NULL) ? NULL : vala_basic_block_ref (_tmp22)), (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp23);
	} else {
		ValaBasicBlock* _tmp24;
		_tmp24 = NULL;
		self->priv->current_block = (_tmp24 = NULL, (self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)), _tmp24);
		self->priv->unreachable_reported = FALSE;
	}
	(before_try_block == NULL) ? NULL : (before_try_block = (vala_basic_block_unref (before_try_block), NULL));
	(after_try_block == NULL) ? NULL : (after_try_block = (vala_basic_block_unref (after_try_block), NULL));
	(finally_block == NULL) ? NULL : (finally_block = (vala_basic_block_unref (finally_block), NULL));
	(catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL));
	(catch_stack == NULL) ? NULL : (catch_stack = (gee_collection_object_unref (catch_stack), NULL));
}


static void vala_flow_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt) {
	ValaFlowAnalyzer * self;
	self = (ValaFlowAnalyzer*) base;
	g_return_if_fail (stmt != NULL);
	if (vala_flow_analyzer_unreachable (self, (ValaCodeNode*) stmt)) {
		return;
	}
	vala_code_node_accept ((ValaCodeNode*) vala_lock_statement_get_body (stmt), (ValaCodeVisitor*) self);
}


static gboolean vala_flow_analyzer_unreachable (ValaFlowAnalyzer* self, ValaCodeNode* node) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (node != NULL, FALSE);
	if (self->priv->current_block == NULL) {
		if (!self->priv->unreachable_reported) {
			vala_report_warning (vala_code_node_get_source_reference (node), "unreachable code detected");
			self->priv->unreachable_reported = TRUE;
		}
		return TRUE;
	}
	return FALSE;
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_break_target (GType object_type, ValaBasicBlock* basic_block) {
	ValaFlowAnalyzerJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	self = (ValaFlowAnalyzerJumpTarget*) g_type_create_instance (object_type);
	vala_flow_analyzer_jump_target_set_basic_block (self, basic_block);
	vala_flow_analyzer_jump_target_set_is_break_target (self, TRUE);
	return self;
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_break_target (ValaBasicBlock* basic_block) {
	return vala_flow_analyzer_jump_target_construct_break_target (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, basic_block);
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_continue_target (GType object_type, ValaBasicBlock* basic_block) {
	ValaFlowAnalyzerJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	self = (ValaFlowAnalyzerJumpTarget*) g_type_create_instance (object_type);
	vala_flow_analyzer_jump_target_set_basic_block (self, basic_block);
	vala_flow_analyzer_jump_target_set_is_continue_target (self, TRUE);
	return self;
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_continue_target (ValaBasicBlock* basic_block) {
	return vala_flow_analyzer_jump_target_construct_continue_target (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, basic_block);
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_return_target (GType object_type, ValaBasicBlock* basic_block) {
	ValaFlowAnalyzerJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	self = (ValaFlowAnalyzerJumpTarget*) g_type_create_instance (object_type);
	vala_flow_analyzer_jump_target_set_basic_block (self, basic_block);
	vala_flow_analyzer_jump_target_set_is_return_target (self, TRUE);
	return self;
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_return_target (ValaBasicBlock* basic_block) {
	return vala_flow_analyzer_jump_target_construct_return_target (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, basic_block);
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_error_target (GType object_type, ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code) {
	ValaFlowAnalyzerJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	g_return_val_if_fail (catch_clause != NULL, NULL);
	self = (ValaFlowAnalyzerJumpTarget*) g_type_create_instance (object_type);
	vala_flow_analyzer_jump_target_set_basic_block (self, basic_block);
	vala_flow_analyzer_jump_target_set_catch_clause (self, catch_clause);
	vala_flow_analyzer_jump_target_set_error_domain (self, error_domain);
	vala_flow_analyzer_jump_target_set_error_code (self, error_code);
	vala_flow_analyzer_jump_target_set_is_error_target (self, TRUE);
	return self;
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_error_target (ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code) {
	return vala_flow_analyzer_jump_target_construct_error_target (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, basic_block, catch_clause, error_domain, error_code);
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct_finally_clause (GType object_type, ValaBasicBlock* basic_block, ValaBasicBlock* last_block) {
	ValaFlowAnalyzerJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	g_return_val_if_fail (last_block != NULL, NULL);
	self = (ValaFlowAnalyzerJumpTarget*) g_type_create_instance (object_type);
	vala_flow_analyzer_jump_target_set_basic_block (self, basic_block);
	vala_flow_analyzer_jump_target_set_last_block (self, last_block);
	vala_flow_analyzer_jump_target_set_is_finally_clause (self, TRUE);
	return self;
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new_finally_clause (ValaBasicBlock* basic_block, ValaBasicBlock* last_block) {
	return vala_flow_analyzer_jump_target_construct_finally_clause (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, basic_block, last_block);
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_construct (GType object_type) {
	ValaFlowAnalyzerJumpTarget* self;
	self = (ValaFlowAnalyzerJumpTarget*) g_type_create_instance (object_type);
	return self;
}


static ValaFlowAnalyzerJumpTarget* vala_flow_analyzer_jump_target_new (void) {
	return vala_flow_analyzer_jump_target_construct (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET);
}


static gboolean vala_flow_analyzer_jump_target_get_is_break_target (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_break_target;
}


static void vala_flow_analyzer_jump_target_set_is_break_target (ValaFlowAnalyzerJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_break_target = value;
}


static gboolean vala_flow_analyzer_jump_target_get_is_continue_target (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_continue_target;
}


static void vala_flow_analyzer_jump_target_set_is_continue_target (ValaFlowAnalyzerJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_continue_target = value;
}


static gboolean vala_flow_analyzer_jump_target_get_is_return_target (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_return_target;
}


static void vala_flow_analyzer_jump_target_set_is_return_target (ValaFlowAnalyzerJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_return_target = value;
}


static gboolean vala_flow_analyzer_jump_target_get_is_error_target (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_error_target;
}


static void vala_flow_analyzer_jump_target_set_is_error_target (ValaFlowAnalyzerJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_error_target = value;
}


static ValaErrorDomain* vala_flow_analyzer_jump_target_get_error_domain (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_error_domain;
}


static void vala_flow_analyzer_jump_target_set_error_domain (ValaFlowAnalyzerJumpTarget* self, ValaErrorDomain* value) {
	ValaErrorDomain* _tmp2;
	ValaErrorDomain* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_error_domain = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_error_domain == NULL) ? NULL : (self->priv->_error_domain = (vala_code_node_unref (self->priv->_error_domain), NULL)), _tmp2);
}


static ValaErrorCode* vala_flow_analyzer_jump_target_get_error_code (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_error_code;
}


static void vala_flow_analyzer_jump_target_set_error_code (ValaFlowAnalyzerJumpTarget* self, ValaErrorCode* value) {
	ValaErrorCode* _tmp2;
	ValaErrorCode* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_error_code = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_error_code == NULL) ? NULL : (self->priv->_error_code = (vala_code_node_unref (self->priv->_error_code), NULL)), _tmp2);
}


static gboolean vala_flow_analyzer_jump_target_get_is_finally_clause (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_finally_clause;
}


static void vala_flow_analyzer_jump_target_set_is_finally_clause (ValaFlowAnalyzerJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_finally_clause = value;
}


static ValaBasicBlock* vala_flow_analyzer_jump_target_get_basic_block (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_basic_block;
}


static void vala_flow_analyzer_jump_target_set_basic_block (ValaFlowAnalyzerJumpTarget* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_basic_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->_basic_block == NULL) ? NULL : (self->priv->_basic_block = (vala_basic_block_unref (self->priv->_basic_block), NULL)), _tmp2);
}


static ValaBasicBlock* vala_flow_analyzer_jump_target_get_last_block (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_last_block;
}


static void vala_flow_analyzer_jump_target_set_last_block (ValaFlowAnalyzerJumpTarget* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_last_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->_last_block == NULL) ? NULL : (self->priv->_last_block = (vala_basic_block_unref (self->priv->_last_block), NULL)), _tmp2);
}


static ValaCatchClause* vala_flow_analyzer_jump_target_get_catch_clause (ValaFlowAnalyzerJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_catch_clause;
}


static void vala_flow_analyzer_jump_target_set_catch_clause (ValaFlowAnalyzerJumpTarget* self, ValaCatchClause* value) {
	ValaCatchClause* _tmp2;
	ValaCatchClause* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_catch_clause = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_catch_clause == NULL) ? NULL : (self->priv->_catch_clause = (vala_code_node_unref (self->priv->_catch_clause), NULL)), _tmp2);
}


static void vala_flow_analyzer_value_jump_target_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_flow_analyzer_value_jump_target_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_flow_analyzer_jump_target_unref (value->data[0].v_pointer);
	}
}


static void vala_flow_analyzer_value_jump_target_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_flow_analyzer_jump_target_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_flow_analyzer_value_jump_target_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_flow_analyzer_value_jump_target_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaFlowAnalyzerJumpTarget* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_flow_analyzer_jump_target_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_flow_analyzer_value_jump_target_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaFlowAnalyzerJumpTarget** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_flow_analyzer_jump_target_ref (value->data[0].v_pointer);
	}
	return NULL;
}


static GParamSpec* vala_flow_analyzer_param_spec_jump_target (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaFlowAnalyzerParamSpecJumpTarget* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


static gpointer vala_flow_analyzer_value_get_jump_target (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET), NULL);
	return value->data[0].v_pointer;
}


static void vala_flow_analyzer_value_set_jump_target (GValue* value, gpointer v_object) {
	ValaFlowAnalyzerJumpTarget* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_flow_analyzer_jump_target_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_flow_analyzer_jump_target_unref (old);
	}
}


static void vala_flow_analyzer_jump_target_class_init (ValaFlowAnalyzerJumpTargetClass * klass) {
	vala_flow_analyzer_jump_target_parent_class = g_type_class_peek_parent (klass);
	VALA_FLOW_ANALYZER_JUMP_TARGET_CLASS (klass)->finalize = vala_flow_analyzer_jump_target_finalize;
	g_type_class_add_private (klass, sizeof (ValaFlowAnalyzerJumpTargetPrivate));
}


static void vala_flow_analyzer_jump_target_instance_init (ValaFlowAnalyzerJumpTarget * self) {
	self->priv = VALA_FLOW_ANALYZER_JUMP_TARGET_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_flow_analyzer_jump_target_finalize (ValaFlowAnalyzerJumpTarget* obj) {
	ValaFlowAnalyzerJumpTarget * self;
	self = VALA_FLOW_ANALYZER_JUMP_TARGET (obj);
	(self->priv->_error_domain == NULL) ? NULL : (self->priv->_error_domain = (vala_code_node_unref (self->priv->_error_domain), NULL));
	(self->priv->_error_code == NULL) ? NULL : (self->priv->_error_code = (vala_code_node_unref (self->priv->_error_code), NULL));
	(self->priv->_basic_block == NULL) ? NULL : (self->priv->_basic_block = (vala_basic_block_unref (self->priv->_basic_block), NULL));
	(self->priv->_last_block == NULL) ? NULL : (self->priv->_last_block = (vala_basic_block_unref (self->priv->_last_block), NULL));
	(self->priv->_catch_clause == NULL) ? NULL : (self->priv->_catch_clause = (vala_code_node_unref (self->priv->_catch_clause), NULL));
}


static GType vala_flow_analyzer_jump_target_get_type (void) {
	static GType vala_flow_analyzer_jump_target_type_id = 0;
	if (vala_flow_analyzer_jump_target_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_flow_analyzer_value_jump_target_init, vala_flow_analyzer_value_jump_target_free_value, vala_flow_analyzer_value_jump_target_copy_value, vala_flow_analyzer_value_jump_target_peek_pointer, "p", vala_flow_analyzer_value_jump_target_collect_value, "p", vala_flow_analyzer_value_jump_target_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaFlowAnalyzerJumpTargetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_flow_analyzer_jump_target_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaFlowAnalyzerJumpTarget), 0, (GInstanceInitFunc) vala_flow_analyzer_jump_target_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_flow_analyzer_jump_target_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaFlowAnalyzerJumpTarget", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_flow_analyzer_jump_target_type_id;
}


static gpointer vala_flow_analyzer_jump_target_ref (gpointer instance) {
	ValaFlowAnalyzerJumpTarget* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


static void vala_flow_analyzer_jump_target_unref (gpointer instance) {
	ValaFlowAnalyzerJumpTarget* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_FLOW_ANALYZER_JUMP_TARGET_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static void vala_flow_analyzer_class_init (ValaFlowAnalyzerClass * klass) {
	vala_flow_analyzer_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_flow_analyzer_finalize;
	g_type_class_add_private (klass, sizeof (ValaFlowAnalyzerPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_flow_analyzer_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_flow_analyzer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_flow_analyzer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_flow_analyzer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_flow_analyzer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_flow_analyzer_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_flow_analyzer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_flow_analyzer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_flow_analyzer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_flow_analyzer_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_flow_analyzer_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_declaration_statement = vala_flow_analyzer_real_visit_declaration_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_flow_analyzer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_flow_analyzer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_statement = vala_flow_analyzer_real_visit_switch_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_flow_analyzer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_flow_analyzer_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_flow_analyzer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_flow_analyzer_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_break_statement = vala_flow_analyzer_real_visit_break_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_continue_statement = vala_flow_analyzer_real_visit_continue_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_flow_analyzer_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_yield_statement = vala_flow_analyzer_real_visit_yield_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_flow_analyzer_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_flow_analyzer_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_flow_analyzer_real_visit_lock_statement;
}


static void vala_flow_analyzer_instance_init (ValaFlowAnalyzer * self) {
	self->priv = VALA_FLOW_ANALYZER_GET_PRIVATE (self);
	self->priv->jump_stack = (GeeList*) gee_array_list_new (VALA_FLOW_ANALYZER_TYPE_JUMP_TARGET, (GBoxedCopyFunc) vala_flow_analyzer_jump_target_ref, vala_flow_analyzer_jump_target_unref, g_direct_equal);
}


static void vala_flow_analyzer_finalize (ValaCodeVisitor* obj) {
	ValaFlowAnalyzer * self;
	self = VALA_FLOW_ANALYZER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->current_block == NULL) ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL));
	(self->priv->jump_stack == NULL) ? NULL : (self->priv->jump_stack = (gee_collection_object_unref (self->priv->jump_stack), NULL));
	(self->priv->var_map == NULL) ? NULL : (self->priv->var_map = (gee_collection_object_unref (self->priv->var_map), NULL));
	(self->priv->used_vars == NULL) ? NULL : (self->priv->used_vars = (gee_collection_object_unref (self->priv->used_vars), NULL));
	(self->priv->phi_functions == NULL) ? NULL : (self->priv->phi_functions = (gee_collection_object_unref (self->priv->phi_functions), NULL));
	VALA_CODE_VISITOR_CLASS (vala_flow_analyzer_parent_class)->finalize (obj);
}


GType vala_flow_analyzer_get_type (void) {
	static GType vala_flow_analyzer_type_id = 0;
	if (vala_flow_analyzer_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFlowAnalyzerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_flow_analyzer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaFlowAnalyzer), 0, (GInstanceInitFunc) vala_flow_analyzer_instance_init, NULL };
		vala_flow_analyzer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaFlowAnalyzer", &g_define_type_info, 0);
	}
	return vala_flow_analyzer_type_id;
}




