/* valablock.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valablock.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valalocalvariable.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaBlockPrivate {
	gboolean _contains_jump_statement;
	GeeList* statement_list;
	GeeList* local_variables;
};

#define VALA_BLOCK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_BLOCK, ValaBlockPrivate))
enum  {
	VALA_BLOCK_DUMMY_PROPERTY,
	VALA_BLOCK_CONTAINS_JUMP_STATEMENT
};
static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_block_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_block_parent_class = NULL;
static ValaStatementIface* vala_block_vala_statement_parent_iface = NULL;
static void vala_block_finalize (GObject * obj);



/**
 * Creates a new block.
 *
 * @param source reference to source code
 */
ValaBlock* vala_block_new (ValaSourceReference* source_reference) {
	ValaBlock * self;
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_BLOCK, 0, NULL);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Append a statement to this block.
 *
 * @param stmt a statement
 */
void vala_block_add_statement (ValaBlock* self, ValaStatement* stmt) {
	g_return_if_fail (VALA_IS_BLOCK (self));
	g_return_if_fail (VALA_IS_STATEMENT (stmt));
	gee_collection_add (GEE_COLLECTION (self->priv->statement_list), stmt);
}


/**
 * Returns a copy of the list of statements.
 *
 * @return statement list
 */
GeeList* vala_block_get_statements (ValaBlock* self) {
	g_return_val_if_fail (VALA_IS_BLOCK (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_STATEMENT, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->statement_list));
}


/**
 * Add a local variable to this block.
 *
 * @param decl a variable declarator
 */
void vala_block_add_local_variable (ValaBlock* self, ValaLocalVariable* local) {
	g_return_if_fail (VALA_IS_BLOCK (self));
	g_return_if_fail (VALA_IS_LOCAL_VARIABLE (local));
	gee_collection_add (GEE_COLLECTION (self->priv->local_variables), local);
}


/**
 * Returns a copy of the list of local variables.
 *
 * @return variable declarator list
 */
GeeList* vala_block_get_local_variables (ValaBlock* self) {
	g_return_val_if_fail (VALA_IS_BLOCK (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_LOCAL_VARIABLE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->local_variables));
}


static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBlock * self;
	self = VALA_BLOCK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_block (visitor, self);
}


static void vala_block_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBlock * self;
	self = VALA_BLOCK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* stmt_collection;
		int stmt_it;
		stmt_collection = self->priv->statement_list;
		for (stmt_it = 0; stmt_it < gee_collection_get_size (GEE_COLLECTION (stmt_collection)); stmt_it = stmt_it + 1) {
			ValaStatement* stmt;
			stmt = ((ValaStatement*) (gee_list_get (GEE_LIST (stmt_collection), stmt_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (stmt), visitor);
				(stmt == NULL ? NULL : (stmt = (g_object_unref (stmt), NULL)));
			}
		}
	}
}


gboolean vala_block_get_contains_jump_statement (ValaBlock* self) {
	g_return_val_if_fail (VALA_IS_BLOCK (self), FALSE);
	return self->priv->_contains_jump_statement;
}


void vala_block_set_contains_jump_statement (ValaBlock* self, gboolean value) {
	g_return_if_fail (VALA_IS_BLOCK (self));
	self->priv->_contains_jump_statement = value;
	g_object_notify (((GObject *) (self)), "contains-jump-statement");
}


static void vala_block_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaBlock * self;
	self = VALA_BLOCK (object);
	switch (property_id) {
		case VALA_BLOCK_CONTAINS_JUMP_STATEMENT:
		g_value_set_boolean (value, vala_block_get_contains_jump_statement (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_block_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaBlock * self;
	self = VALA_BLOCK (object);
	switch (property_id) {
		case VALA_BLOCK_CONTAINS_JUMP_STATEMENT:
		vala_block_set_contains_jump_statement (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_block_class_init (ValaBlockClass * klass) {
	vala_block_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaBlockPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_block_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_block_set_property;
	G_OBJECT_CLASS (klass)->finalize = vala_block_finalize;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_block_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_block_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_BLOCK_CONTAINS_JUMP_STATEMENT, g_param_spec_boolean ("contains-jump-statement", "contains-jump-statement", "contains-jump-statement", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_block_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_block_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_block_instance_init (ValaBlock * self) {
	self->priv = VALA_BLOCK_GET_PRIVATE (self);
	self->priv->statement_list = GEE_LIST (gee_array_list_new (VALA_TYPE_STATEMENT, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
	self->priv->local_variables = GEE_LIST (gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
}


static void vala_block_finalize (GObject * obj) {
	ValaBlock * self;
	self = VALA_BLOCK (obj);
	(self->priv->statement_list == NULL ? NULL : (self->priv->statement_list = (g_object_unref (self->priv->statement_list), NULL)));
	(self->priv->local_variables == NULL ? NULL : (self->priv->local_variables = (g_object_unref (self->priv->local_variables), NULL)));
	G_OBJECT_CLASS (vala_block_parent_class)->finalize (obj);
}


GType vala_block_get_type (void) {
	static GType vala_block_type_id = 0;
	if (vala_block_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBlockClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_block_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBlock), 0, (GInstanceInitFunc) vala_block_instance_init };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_block_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_block_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaBlock", &g_define_type_info, 0);
		g_type_add_interface_static (vala_block_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_block_type_id;
}




