/* valamemberaccess.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valamemberaccess.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaproperty.h>
#include <vala/valacodegenerator.h>




struct _ValaMemberAccessPrivate {
	char* _member_name;
	gboolean _pointer_member_access;
	gboolean _prototype_access;
	gboolean _creation_member;
	gboolean _qualified;
	ValaExpression* _inner;
	GeeList* type_argument_list;
};

#define VALA_MEMBER_ACCESS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessPrivate))
enum  {
	VALA_MEMBER_ACCESS_DUMMY_PROPERTY,
	VALA_MEMBER_ACCESS_INNER,
	VALA_MEMBER_ACCESS_MEMBER_NAME,
	VALA_MEMBER_ACCESS_POINTER_MEMBER_ACCESS,
	VALA_MEMBER_ACCESS_PROTOTYPE_ACCESS,
	VALA_MEMBER_ACCESS_CREATION_MEMBER,
	VALA_MEMBER_ACCESS_QUALIFIED
};
static void vala_member_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_member_access_real_to_string (ValaCodeNode* base);
static void vala_member_access_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_member_access_real_is_pure (ValaExpression* base);
static void vala_member_access_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static ValaCodeBinding* vala_member_access_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen);
static gpointer vala_member_access_parent_class = NULL;
static void vala_member_access_finalize (GObject * obj);



/**
 * Creates a new member access expression.
 *
 * @param inner            parent of the member
 * @param member_name      member name
 * @param source_reference reference to source code
 * @return                 newly created member access expression
 */
ValaMemberAccess* vala_member_access_new (ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess * self;
	g_return_val_if_fail (inner == NULL || VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_MEMBER_ACCESS, 0, NULL);
	vala_member_access_set_inner (self, inner);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaMemberAccess* vala_member_access_new_simple (const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess * self;
	g_return_val_if_fail (member_name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_MEMBER_ACCESS, 0, NULL);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaMemberAccess* vala_member_access_new_pointer (ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess * self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_MEMBER_ACCESS, 0, NULL);
	vala_member_access_set_inner (self, inner);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	vala_member_access_set_pointer_member_access (self, TRUE);
	return self;
}


/**
 * Appends the specified type as generic type argument.
 *
 * @param arg a type reference
 */
void vala_member_access_add_type_argument (ValaMemberAccess* self, ValaDataType* arg) {
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	g_return_if_fail (VALA_IS_DATA_TYPE (arg));
	gee_collection_add (GEE_COLLECTION (self->priv->type_argument_list), arg);
	vala_code_node_set_parent_node (VALA_CODE_NODE (arg), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the list of generic type arguments.
 *
 * @return type argument list
 */
GeeList* vala_member_access_get_type_arguments (ValaMemberAccess* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->type_argument_list));
}


static void vala_member_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_member_access_get_inner (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_member_access_get_inner (self)), visitor);
	}
	{
		GeeList* type_arg_collection;
		int type_arg_it;
		type_arg_collection = self->priv->type_argument_list;
		for (type_arg_it = 0; type_arg_it < gee_collection_get_size (GEE_COLLECTION (type_arg_collection)); type_arg_it = type_arg_it + 1) {
			ValaDataType* type_arg;
			type_arg = ((ValaDataType*) (gee_list_get (GEE_LIST (type_arg_collection), type_arg_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (type_arg), visitor);
				(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
			}
		}
	}
	vala_code_visitor_visit_member_access (visitor, self);
	vala_code_visitor_visit_expression (visitor, VALA_EXPRESSION (self));
}


static char* vala_member_access_real_to_string (ValaCodeNode* base) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (base);
	if (vala_member_access_get_inner (self) == NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->_member_name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	} else {
		char* _tmp2;
		char* _tmp3;
		_tmp2 = NULL;
		_tmp3 = NULL;
		return (_tmp3 = g_strdup_printf ("%s.%s", (_tmp2 = vala_code_node_to_string (VALA_CODE_NODE (vala_member_access_get_inner (self)))), self->priv->_member_name), (_tmp2 = (g_free (_tmp2), NULL)), _tmp3);
	}
}


static void vala_member_access_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (base);
	g_return_if_fail (VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (VALA_IS_EXPRESSION (new_node));
	if (vala_member_access_get_inner (self) == old_node) {
		vala_member_access_set_inner (self, new_node);
	}
}


static gboolean vala_member_access_real_is_pure (ValaExpression* base) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (base);
	/* accessing property could have side-effects*/
	return (vala_member_access_get_inner (self) == NULL || vala_expression_is_pure (vala_member_access_get_inner (self))) && !(VALA_IS_PROPERTY (vala_expression_get_symbol_reference (VALA_EXPRESSION (self))));
}


static void vala_member_access_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size (GEE_COLLECTION (self->priv->type_argument_list)); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = ((ValaDataType*) (gee_list_get (((GeeList*) (self->priv->type_argument_list)), i)))) == old_type, (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1)) {
				gee_list_set (((GeeList*) (self->priv->type_argument_list)), i, new_type);
				return;
			}
		}
	}
}


static ValaCodeBinding* vala_member_access_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (base);
	g_return_val_if_fail (VALA_IS_CODE_GENERATOR (codegen), NULL);
	return vala_code_generator_create_member_access_binding (codegen, self);
}


ValaExpression* vala_member_access_get_inner (ValaMemberAccess* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), NULL);
	return self->priv->_inner;
}


void vala_member_access_set_inner (ValaMemberAccess* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_inner = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_inner == NULL ? NULL : (self->priv->_inner = (g_object_unref (self->priv->_inner), NULL))), _tmp2);
	if (self->priv->_inner != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_inner), VALA_CODE_NODE (self));
	}
	g_object_notify (((GObject *) (self)), "inner");
}


const char* vala_member_access_get_member_name (ValaMemberAccess* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), NULL);
	return self->priv->_member_name;
}


void vala_member_access_set_member_name (ValaMemberAccess* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_member_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_member_name = (g_free (self->priv->_member_name), NULL)), _tmp2);
	g_object_notify (((GObject *) (self)), "member-name");
}


gboolean vala_member_access_get_pointer_member_access (ValaMemberAccess* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), FALSE);
	return self->priv->_pointer_member_access;
}


void vala_member_access_set_pointer_member_access (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	self->priv->_pointer_member_access = value;
	g_object_notify (((GObject *) (self)), "pointer-member-access");
}


gboolean vala_member_access_get_prototype_access (ValaMemberAccess* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), FALSE);
	return self->priv->_prototype_access;
}


void vala_member_access_set_prototype_access (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	self->priv->_prototype_access = value;
	g_object_notify (((GObject *) (self)), "prototype-access");
}


gboolean vala_member_access_get_creation_member (ValaMemberAccess* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), FALSE);
	return self->priv->_creation_member;
}


void vala_member_access_set_creation_member (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	self->priv->_creation_member = value;
	g_object_notify (((GObject *) (self)), "creation-member");
}


gboolean vala_member_access_get_qualified (ValaMemberAccess* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), FALSE);
	return self->priv->_qualified;
}


void vala_member_access_set_qualified (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	self->priv->_qualified = value;
	g_object_notify (((GObject *) (self)), "qualified");
}


static void vala_member_access_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (object);
	switch (property_id) {
		case VALA_MEMBER_ACCESS_INNER:
		g_value_set_object (value, vala_member_access_get_inner (self));
		break;
		case VALA_MEMBER_ACCESS_MEMBER_NAME:
		g_value_set_string (value, vala_member_access_get_member_name (self));
		break;
		case VALA_MEMBER_ACCESS_POINTER_MEMBER_ACCESS:
		g_value_set_boolean (value, vala_member_access_get_pointer_member_access (self));
		break;
		case VALA_MEMBER_ACCESS_PROTOTYPE_ACCESS:
		g_value_set_boolean (value, vala_member_access_get_prototype_access (self));
		break;
		case VALA_MEMBER_ACCESS_CREATION_MEMBER:
		g_value_set_boolean (value, vala_member_access_get_creation_member (self));
		break;
		case VALA_MEMBER_ACCESS_QUALIFIED:
		g_value_set_boolean (value, vala_member_access_get_qualified (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_member_access_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (object);
	switch (property_id) {
		case VALA_MEMBER_ACCESS_INNER:
		vala_member_access_set_inner (self, g_value_get_object (value));
		break;
		case VALA_MEMBER_ACCESS_MEMBER_NAME:
		vala_member_access_set_member_name (self, g_value_get_string (value));
		break;
		case VALA_MEMBER_ACCESS_POINTER_MEMBER_ACCESS:
		vala_member_access_set_pointer_member_access (self, g_value_get_boolean (value));
		break;
		case VALA_MEMBER_ACCESS_PROTOTYPE_ACCESS:
		vala_member_access_set_prototype_access (self, g_value_get_boolean (value));
		break;
		case VALA_MEMBER_ACCESS_CREATION_MEMBER:
		vala_member_access_set_creation_member (self, g_value_get_boolean (value));
		break;
		case VALA_MEMBER_ACCESS_QUALIFIED:
		vala_member_access_set_qualified (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_member_access_class_init (ValaMemberAccessClass * klass) {
	vala_member_access_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMemberAccessPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_member_access_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_member_access_set_property;
	G_OBJECT_CLASS (klass)->finalize = vala_member_access_finalize;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_member_access_real_accept;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_member_access_real_to_string;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_member_access_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_member_access_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_member_access_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->create_code_binding = vala_member_access_real_create_code_binding;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_INNER, g_param_spec_object ("inner", "inner", "inner", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_MEMBER_NAME, g_param_spec_string ("member-name", "member-name", "member-name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_POINTER_MEMBER_ACCESS, g_param_spec_boolean ("pointer-member-access", "pointer-member-access", "pointer-member-access", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_PROTOTYPE_ACCESS, g_param_spec_boolean ("prototype-access", "prototype-access", "prototype-access", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_CREATION_MEMBER, g_param_spec_boolean ("creation-member", "creation-member", "creation-member", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_QUALIFIED, g_param_spec_boolean ("qualified", "qualified", "qualified", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_member_access_instance_init (ValaMemberAccess * self) {
	self->priv = VALA_MEMBER_ACCESS_GET_PRIVATE (self);
	self->priv->type_argument_list = GEE_LIST (gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
}


static void vala_member_access_finalize (GObject * obj) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (obj);
	self->priv->_member_name = (g_free (self->priv->_member_name), NULL);
	(self->priv->_inner == NULL ? NULL : (self->priv->_inner = (g_object_unref (self->priv->_inner), NULL)));
	(self->priv->type_argument_list == NULL ? NULL : (self->priv->type_argument_list = (g_object_unref (self->priv->type_argument_list), NULL)));
	G_OBJECT_CLASS (vala_member_access_parent_class)->finalize (obj);
}


GType vala_member_access_get_type (void) {
	static GType vala_member_access_type_id = 0;
	if (vala_member_access_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMemberAccessClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_member_access_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMemberAccess), 0, (GInstanceInitFunc) vala_member_access_instance_init };
		vala_member_access_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaMemberAccess", &g_define_type_info, 0);
	}
	return vala_member_access_type_id;
}




