/* valainitializerlist.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valainitializerlist.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaInitializerListPrivate {
	GeeList* initializers;
};

#define VALA_INITIALIZER_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListPrivate))
enum  {
	VALA_INITIALIZER_LIST_DUMMY_PROPERTY,
	VALA_INITIALIZER_LIST_SIZE
};
static void vala_initializer_list_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_initializer_list_real_is_pure (ValaExpression* base);
static void vala_initializer_list_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gpointer vala_initializer_list_parent_class = NULL;
static void vala_initializer_list_dispose (GObject * obj);



/**
 * Appends the specified expression to this initializer list.
 *
 * @param expr an expression
 */
void vala_initializer_list_append (ValaInitializerList* self, ValaExpression* expr) {
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (self));
	g_return_if_fail (VALA_IS_EXPRESSION (expr));
	gee_collection_add (GEE_COLLECTION (self->priv->initializers), expr);
	vala_code_node_set_parent_node (VALA_CODE_NODE (expr), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the expression list.
 *
 * @return expression list
 */
GeeList* vala_initializer_list_get_initializers (ValaInitializerList* self) {
	g_return_val_if_fail (VALA_IS_INITIALIZER_LIST (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->initializers));
}


/**
 * Creates a new initializer list.
 *
 * @param source_reference reference to source code
 * @return                 newly created initializer list
 */
ValaInitializerList* vala_initializer_list_new (ValaSourceReference* source_reference) {
	ValaInitializerList * self;
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_INITIALIZER_LIST, 0, NULL);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_initializer_list_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* expr_collection;
		int expr_it;
		expr_collection = self->priv->initializers;
		for (expr_it = 0; expr_it < gee_collection_get_size (GEE_COLLECTION (expr_collection)); expr_it = expr_it + 1) {
			ValaExpression* expr;
			expr = ((ValaExpression*) (gee_list_get (GEE_LIST (expr_collection), expr_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (expr), visitor);
				(expr == NULL ? NULL : (expr = (g_object_unref (expr), NULL)));
			}
		}
	}
}


static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_initializer_list (visitor, self);
}


static gboolean vala_initializer_list_real_is_pure (ValaExpression* base) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (base);
	{
		GeeList* initializer_collection;
		int initializer_it;
		initializer_collection = self->priv->initializers;
		for (initializer_it = 0; initializer_it < gee_collection_get_size (GEE_COLLECTION (initializer_collection)); initializer_it = initializer_it + 1) {
			ValaExpression* initializer;
			initializer = ((ValaExpression*) (gee_list_get (GEE_LIST (initializer_collection), initializer_it)));
			{
				if (!vala_expression_is_pure (initializer)) {
					gboolean _tmp0;
					return (_tmp0 = FALSE, (initializer == NULL ? NULL : (initializer = (g_object_unref (initializer), NULL))), _tmp0);
				}
				(initializer == NULL ? NULL : (initializer = (g_object_unref (initializer), NULL)));
			}
		}
	}
	return TRUE;
}


static void vala_initializer_list_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (base);
	g_return_if_fail (VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (VALA_IS_EXPRESSION (new_node));
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size (GEE_COLLECTION (self->priv->initializers)); i++) {
			ValaExpression* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = ((ValaExpression*) (gee_list_get (((GeeList*) (self->priv->initializers)), i)))) == old_node, (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1)) {
				gee_list_set (((GeeList*) (self->priv->initializers)), i, new_node);
			}
		}
	}
}


gint vala_initializer_list_get_size (ValaInitializerList* self) {
	g_return_val_if_fail (VALA_IS_INITIALIZER_LIST (self), 0);
	return gee_collection_get_size (GEE_COLLECTION (self->priv->initializers));
}


static void vala_initializer_list_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (object);
	switch (property_id) {
		case VALA_INITIALIZER_LIST_SIZE:
		g_value_set_int (value, vala_initializer_list_get_size (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_initializer_list_class_init (ValaInitializerListClass * klass) {
	vala_initializer_list_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInitializerListPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_initializer_list_get_property;
	G_OBJECT_CLASS (klass)->dispose = vala_initializer_list_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_initializer_list_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_initializer_list_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_initializer_list_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_initializer_list_real_replace_expression;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INITIALIZER_LIST_SIZE, g_param_spec_int ("size", "size", "size", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
}


static void vala_initializer_list_instance_init (ValaInitializerList * self) {
	self->priv = VALA_INITIALIZER_LIST_GET_PRIVATE (self);
	self->priv->initializers = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
}


static void vala_initializer_list_dispose (GObject * obj) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (obj);
	(self->priv->initializers == NULL ? NULL : (self->priv->initializers = (g_object_unref (self->priv->initializers), NULL)));
	G_OBJECT_CLASS (vala_initializer_list_parent_class)->dispose (obj);
}


GType vala_initializer_list_get_type (void) {
	static GType vala_initializer_list_type_id = 0;
	if (G_UNLIKELY (vala_initializer_list_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInitializerListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_initializer_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInitializerList), 0, (GInstanceInitFunc) vala_initializer_list_instance_init };
		vala_initializer_list_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaInitializerList", &g_define_type_info, 0);
	}
	return vala_initializer_list_type_id;
}




